;; (c) 2008,2009 Lau B. Jensen <lau.jensen {at} bestinclass.dk
;;                         Meikel Brandmeyer <mb {at} kotka.de>
;; All rights reserved.
;;
;; The use and distribution terms for this software are covered by the
;; Eclipse Public License 1.0 (http://opensource.org/licenses/eclipse-1.0.php)
;; which can be found in the file LICENSE.txt at the root of this distribution.
;; By using this software in any fashion, you are agreeing to be bound by the
;; terms of this license. You must not remove this notice, or any other, from
;; this software.

(clojure.core/in-ns 'clojureql)

;; CONNECTION ==============================================

(defstruct connection-info
  :jdbc-url :username :password)

(defn make-connection-info
  "Given the arguments, returns a hash-map which serves as your connection
  information for e.g.: with-connection."
  ([protocol host username password]
     (struct connection-info (format "jdbc:%s:%s" protocol host) username password))
  ([protocol host]
     (struct connection-info (format "jdbc:%s:%s" protocol host) nil nil)))

;; SET-ENV =================================================

(defmulti set-env-value
  "Set the given value in the PreparedStatement. Standard methods are
  provided. Custom methods may be defined to provide custom mappings
  between values and their representation in the database. Dispatches
  on the type of value."
  {:arglists '([stmt cnt value])}
  (fn set-env-value-dispatch [_ _ value] (type value)))

(defn set-env
  [#^PreparedStatement stmt env]
  (loop [env (seq env)
         cnt 1]
    (when env
      (let [value (first env)]
        (if-not (nil? value)
          (do
            (set-env-value stmt cnt value)
            (recur (next env) (inc cnt)))
          (recur (next env) cnt))))))

(defmethod set-env-value String
  set-env-value-string
  [#^PreparedStatement stmt cnt #^String value]
  (.setString stmt cnt value))

(defmethod set-env-value Float
  set-env-value-float
  [#^PreparedStatement stmt cnt #^Float value]
  (.setFloat stmt cnt value))

(defmethod set-env-value BigDecimal
  set-env-value-bigdecimal
  [#^PreparedStatement stmt cnt #^BigDecimal value]
  (.setBigDecimal stmt cnt value))

(defmethod set-env-value Double
  set-env-value-double
  [#^PreparedStatement stmt cnt #^Double value]
  (.setDouble stmt cnt value))

(defmethod set-env-value Long
  set-env-value-long
  [#^PreparedStatement stmt cnt #^Long value]
  (.setLong stmt cnt value))

(defmethod set-env-value Integer
  set-env-value-integer
  [#^PreparedStatement stmt cnt #^Integer value]
  (.setInt stmt cnt value))

(defmethod set-env-value Short
  set-env-value-short
  [#^PreparedStatement stmt cnt #^Short value]
  (.setShort stmt cnt value))

(defmethod set-env-value java.net.URL
  set-env-value-url
  [#^PreparedStatement stmt cnt #^java.net.URL value]
  (.setURL stmt cnt value))

(defmethod set-env-value java.sql.Date
  set-env-value-date
  [#^PreparedStatement stmt cnt #^java.sql.Date value]
  (.setDate stmt cnt value))

(defmethod set-env-value java.util.Date
  set-env-value-date
  [stmt cnt #^java.util.Date value]
  (set-env-value stmt cnt (java.sql.Date. (.getTime value))))

(defmethod set-env-value java.sql.Time
  set-env-value-time
  [#^PreparedStatement stmt cnt #^java.sql.Time value]
  (.setTime stmt cnt value))

(defmethod set-env-value java.sql.Timestamp
  set-env-value-timestamp
  [#^PreparedStatement stmt cnt #^java.sql.Timestamp value]
  (.setTimestamp stmt cnt value))

;; DRIVER ==================================================

(defn load-driver
  "Load the named JDBC driver. Has to be called once before accessing
  the database."
  [driver]
  (try
   (clojure.lang.RT/classForName driver)
   (catch Exception e
     (throw
      (Exception. "The driver could not be loaded, please verify thats its found on the classpath")))))

;; INVERT ==================================================

(defmulti invert
  "Turn the given form into its complement."
  {:arglists '([form])}
  (fn invert-dispatch [form] (-> form first ->string)))

(defmethod invert "and"
  invert-and
  [form]
  (cons 'or (map invert (next form))))

(defmethod invert "or"
  invert-or
  [form]
  (cons 'and (map invert (next form))))

(defmethod invert "not"
  invert-not
  [form]
  (second form))

(def #^{:doc "Map of predicates to their complement."} invert-complement
  (atom {"="  "<>" "<>" "="
         "<=" ">"  ">"  "<="
         ">=" "<"  "<"  ">="
         "nil?" "not-nil?" "not-nil?" "nil?"}))

(defmethod invert :default
  invert-default
  [form]
  (let [pred (-> form first ->string)]
    (if-let [comp-pred (@invert-complement pred)]
      (cons comp-pred (next form))
      (throw
        (Exception. (str "Don't know how to complement predicate: " pred))))))

;; COMPILE FUNCTION ========================================

(def #^{:doc "A map of SQL function names to their type."} sql-function-type
  (atom {"+"    ::Infix "-"  ::Infix "*"   ::Infix "/" ::Infix "="  ::Infix
         "<="   ::Infix ">=" ::Infix "<"   ::Infix ">" ::Infix "<>" ::Infix
         "like" ::Infix "or" ::Infix "and" ::Infix
         "nil?" ::Nil?  "not-nil?" ::Nil? "not" ::Not}))

(defmulti compile-function
  "Compile a function specification into a string.
  (compile-function '(= 25 (sum count))) => (25 = sum(count))"
  {:arglists '([form])}
  (fn compile-function-dispatch [form]
    (if (or (seq? form) (vector? form))
      (get @sql-function-type (-> form first ->string) ::Funcall)
      ::Identity)))

(defmethod compile-function ::Infix
  [form]
  (str "(" (str-cat " " (interpose (->string (first form))
                                   (map compile-function (rest form))))
       ")"))

(defmethod compile-function ::Funcall
  [form]
  (str (first form) "(" (str-cat "," (cons (->string (second form))
                                           (nnext form))) ")"))

(defmethod compile-function ::Nil?
  [form]
  (str (->string (second form))
       (if (= "nil?" (->string (first form)))
         " IS NULL"
         " IS NOT NULL")))

(defmethod compile-function ::Not
  [form]
  (let [form (invert (second form))]
    (compile-function form)))

(defmethod compile-function ::Identity
  [form]
  form)

;; COMPILE ALIAS ===========================================

(defn compile-alias
  "Checks whether the given column has an alias in the aliases map. If so
  it is converted to a SQL alias of the form „column AS alias“."
  [col-or-table-spec col-or-table aliases]
  (if-let [aka (aliases col-or-table)]
    (str (->string col-or-table-spec) " AS " (->string aka))
    col-or-table-spec))

;; OTHERS ==================================================

(defn identifiers-list
  "Transforms a sequence into a list of delimited identifiers."
  [identifiers]
  (str "(" (str-cat "," identifiers) ")"))

(defn list-constraint
  "Returns a constraint having a list of delimited identifiers as argument."
  [name identifiers]
  (when-not (empty? identifiers)
    (str name " " (identifiers-list identifiers))))
