;; (c) 2008,2009 Lau B. Jensen <lau.jensen {at} bestinclass.dk
;;                         Meikel Brandmeyer <mb {at} kotka.de>
;; All rights reserved.
;;
;; The use and distribution terms for this software are covered by the
;; Eclipse Public License 1.0 (http://opensource.org/licenses/eclipse-1.0.php)
;; which can be found in the file LICENSE.txt at the root of this distribution.
;; By using this software in any fashion, you are agreeing to be bound by the
;; terms of this license. You must not remove this notice, or any other, from
;; this software.

(clojure.core/in-ns 'clojureql)

(defn compile-column-spec
  "Returns a list of column specifications with optional aliases."
  [columns aliases]
  (str-cat ","
           (map (fn [spec]
                  (let [col (column-from-spec spec)]
                    (-> spec
                      compile-function
                      (compile-alias col aliases)
                      ->string)))
                columns)))

(defn compile-table-spec
  "Returns a list of table specifications with optional aliases."
  [tables aliases]
  (str-cat ","
           (map (fn [spec]
                  (let [table (table-from-spec spec)]
                    (-> spec
                      (compile-alias table aliases)
                      ->string)))
                tables)))

(defmethod compile-sql [::Select ::Generic]
  [stmt _]
  (let [{:keys [columns tables predicates column-aliases table-aliases]} stmt
        cols  (compile-column-spec columns column-aliases)
        tabls (compile-table-spec tables table-aliases)
        stmnt (list* "SELECT" cols
                     "FROM"   tabls
                     (when predicates
                       (list "WHERE" (compile-function predicates))))]
    (str-cat " " stmnt)))

(defmethod compile-sql [::Join ::Generic]
  [stmt _]
  (let [join-types {::InnerJoin "INNER"
                    ::LeftJoin  "LEFT"
                    ::RightJoin "RIGHT"
                    ::FullJoin  "FULL"}
        {:keys [query on]} stmt
        {:keys [columns tables predicates column-aliases table-aliases]} query
        cols  (compile-column-spec columns column-aliases)
        left  (compile-table-spec [(first tables)] table-aliases)
        right (compile-table-spec [(second tables)] table-aliases)
        stmnt (list* "SELECT" cols
                     "FROM"   left
                     (join-types (type stmt))
                     "JOIN"   right
                     "ON"     (compile-function on)
                     (when predicates
                       ["WHERE" (compile-function predicates)]))]
    (str-cat " " stmnt)))

(defmethod compile-sql [::FullJoin ::EmulateFullJoin]
  [stmt db]
  (let [{:keys [query on]} stmt
        {:keys [columns tables predicates column-aliases table-aliases]} query
        cols  (compile-column-spec columns column-aliases)
        left  (compile-table-spec [(first tables)] table-aliases)
        right (compile-table-spec [(second tables)] table-aliases)
        stmnt (concat ["SELECT" cols
                       "FROM"   left
                       "LEFT JOIN" right
                       "ON"     (compile-function on)]
                       (when predicates
                         ["WHERE" (compile-function predicates)])
                       ["UNION ALL"
                       "SELECT" cols
                       "FROM"   right
                       "LEFT JOIN" left
                       "ON"     (compile-function on)
                       "WHERE"
                       (if predicates
                         (compile-function
                           (quasiquote (and ~predicates (nil? ~(nth on 2)))))
                         (compile-function (quasiquote (nil? ~(nth on 2)))))])]
    (str-cat " " stmnt)))

(prefer-method compile-sql
               [::FullJoin ::EmulateFullJoin]
               [::Select ::Generic])

(prefer-method compile-sql
               [::FullJoin ::EmulateFullJoin]
               [::Join ::Generic])

(defmethod compile-sql [::OrderedSelect ::Generic]
  [stmt db]
  (let [{:keys [query order columns]} stmt]
    (str-cat " " [(compile-sql query db)
                  "ORDER BY"
                  (str-cat "," (map ->string columns))
                  (condp = order
                    :ascending  "ASC"
                    :descending "DESC")])))

(defmethod compile-sql [::GroupedSelect ::Generic]
  [stmt db]
  (let [{:keys [query columns]} stmt]
    (str-cat " " [(compile-sql query db)
                  "GROUP BY"
                  (str-cat "," (map ->string columns))])))

(defmethod compile-sql [::HavingSelect ::Generic]
  [stmt db]
  (let [{:keys [query predicates]} stmt]
    (str-cat " " [(compile-sql query db)
                  "HAVING"
                  (compile-function predicates)])))

(defmethod compile-sql [::DistinctSelect ::Generic]
  [stmt db]
  (apply str "SELECT DISTINCT" (drop 6 (-> stmt :query (compile-sql db)))))

(defmethod compile-sql [::Union ::Generic]
  [stmt db]
  (let [{:keys [queries all]} stmt]
    (str "(" (str-cat " " (interpose (if all
                                       ") UNION ALL ("
                                       ") UNION (")
                                     (map #(compile-sql % db) queries)))
         ")")))

(defmethod compile-sql [::Intersect ::Generic]
  [stmt db]
  (let [{:keys [queries]} stmt]
    (str "(" (str-cat " " (interpose ") INTERSECT ("
                                     (map #(compile-sql % db) queries)))
         ")")))

(defmethod compile-sql [::Difference ::Generic]
  [stmt db]
  (let [{:keys [queries]} stmt]
    (str "(" (str-cat " " (interpose ") MINUS ("
                                     (map #(compile-sql % db) queries)))
         ")")))

(defmethod compile-sql [::InsertQuery ::Generic]
  [stmt conn]
  (let [{:keys [table query]} stmt]
    (str-cat " " ["INSERT INTO" table
                  (compile-sql query conn)])))

(defmethod compile-sql [::InsertValues ::Generic]
  [stmt _]
  (let [{:keys [table columns env]} stmt]
    (str-cat " " ["INSERT INTO" table "("
                  (str-cat "," (map ->string columns))
                  ") VALUES ("
                  (str-cat "," (map #(if (nil? %) "NULL" "?") env))
                  ")"])))

(defmethod compile-sql [::Update ::Generic]
  [stmt _]
  (let [{:keys [table columns predicates]} stmt]
    (str-cat " " ["UPDATE" table
                  "SET"    (str-cat "," (map (comp
                                               #(str % " = ?")
                                               ->string)
                                             columns))
                  "WHERE"  (compile-function predicates)])))

(defmethod compile-sql [::Delete ::Generic]
  [stmt _]
  (let [{:keys [table predicates]} stmt]
    (str-cat " " (list* "DELETE FROM" table
                        (when predicates
                          ["WHERE" (compile-function predicates)])))))
