;; Copyright (c) 2008,2009 Lau B. Jensen <lau.jensen {at} bestinclass.dk
;;                         Meikel Brandmeyer <mb {at} kotka.de>
;; All rights reserved.
;;
;; The use and distribution terms for this software are covered by the
;; Eclipse Public License 1.0 (http://opensource.org/licenses/eclipse-1.0.php)
;; which can be found in the file LICENSE.txt at the root of this distribution.
;; By using this software in any fashion, you are agreeing to be bound by the
;; terms of this license. You must not remove this notice, or any other, from
;; this software.

(clojure.core/ns clojureql.backend.mysql
  (:require
     [clojureql :as cql]
     [clojureql.util :as util]
     [clojureql.schema :as schema]))

; Register for emulation of full join.
(swap! cql/sql-hierarchy derive com.mysql.jdbc.Connection ::cql/EmulateFullJoin)

(defmethod cql/compile-sql
  [::cql/CreateTable com.mysql.jdbc.Connection]
  [stmt db]
  (cql/with-table-options stmt
    (str "CREATE TABLE " table " ("
      (util/str-cat ","
        (concat
          (map (fn [[column col-type]]
                 (str column " "
                      (cql/standard-column col-type
                        :not-null (contains? non-nulls column)
                        :default  (get defaults column)
                        :others   (cql/column-option "AUTO_INCREMENT" column auto-inc))))
               columns)
          [(cql/list-constraint "PRIMARY KEY" primary-key)]
          (map cql/standard-unique uniques)
          (map #(apply cql/standard-foreign-key %) foreign-key)))
      ")")))

(defn mysql-get-column-definition
  "Returns a string containing all of given column definition."
  [db table column]
  (let [col-info (first (schema/columns db (str table) (str column)))]
    (cql/standard-column (schema/data-type col-info)
      :not-null (= (:is_nullable col-info) "NO")
      :default  nil
      :others   (when (= (:is_autoincrement col-info) "YES")
                  " AUTO_INCREMENT"))))

(defmethod cql/compile-sql-alter
  [::cql/Rename com.mysql.jdbc.Connection]
  [stmt db]
  (let [{:keys [options table]} stmt]
    (util/str-cat " "
      ["ALTER TABLE" table
        (if (= 'column (first options))
          (util/str-cat " " ["CHANGE" (options 1) (options 2)
                             (mysql-get-column-definition db table (options 1))])
          (str "RENAME " (first options)))])))
