;; (c) 2008,2009 Lau B. Jensen <lau.jensen {at} bestinclass.dk
;;                         Meikel Brandmeyer <mb {at} kotka.de>
;; All rights reserved.
;;
;; The use and distribution terms for this software are covered by the
;; Eclipse Public License 1.0 (http://opensource.org/licenses/eclipse-1.0.php)
;; which can be found in the file LICENSE.txt at the root of this distribution.
;; By using this software in any fashion, you are agreeing to be bound by the
;; terms of this license. You must not remove this notice, or any other, from
;; this software.

(clojure.core/in-ns 'clojureql)

(load "clojureql/backend/util")

;; SQL COMPILATION ==========================================

(def sql-hierarchy
  (atom (-> (make-hierarchy)
          (derive java.lang.Object    ::Generic)
          (derive java.sql.Connection ::Generic)
          (derive ::Raw               ::Generic)
          (derive ::Select            ::ExecuteQuery)
          (derive ::Join              ::Select)
          (derive ::InnerJoin         ::Join)
          (derive ::LeftJoin          ::Join)
          (derive ::RightJoin         ::Join)
          (derive ::FullJoin          ::Join)
          (derive ::OrderedSelect     ::Select)
          (derive ::GroupedSelect     ::Select)
          (derive ::DistinctSelect    ::Select)
          (derive ::HavingSelect      ::Select)
          (derive ::Union             ::ExecuteQuery)
          (derive ::Intersect         ::ExecuteQuery)
          (derive ::Difference        ::ExecuteQuery)
          (derive ::Update            ::ExecuteUpdate)
          (derive ::Insert            ::ExecuteUpdate)
          (derive ::InsertQuery       ::Insert)
          (derive ::InsertValues      ::Insert)
          (derive ::Delete            ::ExecuteUpdate))))

(defmulti compile-sql
  "Compile the given SQL statement for the given database."
  {:arglists '([stmt db])}
  (fn [stmt db] [(type stmt) (class db)])
  :hierarchy sql-hierarchy)

(load "clojureql/backend/query")
(load "clojureql/backend/sql")

(defmethod compile-sql [::Raw ::Generic]
  [stmt _]
  (:statement stmt))

;; SQL EXECUTION ============================================

(defn prepare-statement
  "Return a prepared statement for the given SQL statement in the
  context of the given connection."
  {:tag PreparedStatement}
  [sql-stmt #^Connection conn]
  (let [compiled-stmt (compile-sql sql-stmt conn)]
    (when compiled-stmt
      (doto (.prepareStatement conn compiled-stmt)
        (set-env (sql-stmt :env))))))

(defn in-transaction*
  "Execute thunk wrapped into a savepoint transaction."
  [#^Connection conn thunk]
  (let [auto-commit-state (.getAutoCommit conn)]
    (try
      (.setAutoCommit conn false)
      (let [savepoint (.setSavepoint conn)]
        (try
          (let [result (thunk)]
            (.releaseSavepoint conn savepoint)
            result)
          (catch Exception e
            (.rollback conn savepoint)
            (throw e))))
      (finally
        (.setAutoCommit conn auto-commit-state)))))

(defmacro in-transaction
  "Execute body wrapped into a savepoint transaction."
  [conn & body]
  `(in-transaction* ~conn (fn [] ~@body)))

(defmulti execute-sql
  "Execute the given SQL statement in the context of the given connection
  as obtained by with-connection."
  {:arglists '([sql-stmt conn])}
  (fn [sql-stmt conn] (type sql-stmt))
  :default  ::Execute
  :hierarchy sql-hierarchy)

(defmethod execute-sql ::Execute
  [sql-stmt conn]
  (let [prepd-stmt (prepare-statement sql-stmt conn)]
    (when prepd-stmt
      (.execute prepd-stmt))
    prepd-stmt))

(defmethod execute-sql ::ExecuteQuery
  [sql-stmt conn]
  (let [stmt       ((get-method execute-sql ::Execute) sql-stmt conn)
        result-set (.getResultSet #^PreparedStatement stmt)]
    (resultset-seq result-set)))

(defmethod execute-sql ::ExecuteUpdate
  [sql-stmt conn]
  (let [stmt ((get-method execute-sql ::Execute) sql-stmt conn)]
    (.getUpdateCount #^PreparedStatement stmt)))

(defmethod execute-sql ::LetQuery
  [sql-stmt conn]
  ((sql-stmt :fn) conn))

(defmethod execute-sql ::Batch
  [sql-stmt conn]
  (in-transaction conn
    (doall (map #(execute-sql % conn) (sql-stmt :statements)))))

(defmethod execute-sql ::Raw
  [sql-stmt conn]
  (let [stmt       ((get-method execute-sql ::Execute) sql-stmt conn)
        result-set (.getResultSet #^PreparedStatement stmt)]
    (resultset-seq result-set)))

;; INTERFACE ================================================

; All global conections are stored using :name connection-object
(def global-connections
     (atom {}))

(defn with-connection*
  "Open the given database connection and calls thunk with the connection.
  Takes care that the connection is closed after thunk returns."
  [conn-info thunk]
  (io! "Database interaction cannot happen in a transaction"
       (with-open [conn (java.sql.DriverManager/getConnection
                          (:jdbc-url conn-info)
                          (:username conn-info)
                          (:password conn-info))]
         (thunk conn))))

(defmacro with-connection
  "Open the database described by the given connection-info and bind
  it to connection. Then execute body."
  [connections & body]
  (let [conn-vars  (take-nth 2 connections)
	conn-infos (take-nth 2 (next connections))
	conns      (map (fn [info]
			  `(let [info# ~info]
			     (java.sql.DriverManager/getConnection
			      (:jdbc-url info#)
			      (:username info#)
			      (:password info#))))
			conn-infos)]
    `(io! "Database interaction cannot happen in a transaction"
	    (with-open ~(vec (interleave conn-vars conns))
	      ~@body))))

(defn open-global
  "Opens a connection using conn-info and stores it as a global named conn-name.

    e.g.: (make-global :mysql1 (make-connection-info....))"
  [conn-name conn-info]
  (swap! global-connections assoc conn-name
	 (java.sql.DriverManager/getConnection
	  (:jdbc-url conn-info)
	  (:username conn-info)
	  (:password conn-info))))

(defn close-global
  "Supplied with a keyword identifying a global connection, that connection
  is closed and the reference dropped."
  [conn-name]
  (if-let [conn (conn-name @global-connections)]
    (do
      (.close conn)
      (swap! global-connections dissoc conn-name))
    (throw
     (Exception. (format "No global connection by that name is open (%s)" conn-name)))))

(defn run-global
  "Executes statements via a global connection."
  ([conn-name ast]
     (if-let [open-conn (conn-name @global-connections)]
       (execute-sql ast open-conn)
       (throw
	(Exception. (format "No global connection by that name is open (%s)" conn-name)))))
  ([conn-name ast wfunc]
     (if-let [open-conn (conn-name @global-connections)]
       (wfunc (execute-sql ast open-conn))
       (throw
	(Exception. (format "No global connection by that name is open (%s)" conn-name))))))

(defn run*
  "Driver for run, don't call directly."
  [conn-info ast wfunc]
  (with-connection [open-conn conn-info]
    (wfunc (execute-sql ast open-conn))))

(defmacro run
  "Executes the AST (statement) and optionally exposes the results

   Examples:
      #1: Using global connection named mysql1
         (run :mysql1 results (query table1 *)
         (doseq [r results]
          (prn r)))
      #2: Using named connection
        (run (make-connection-info...) results (query....))
      #3: Not needing to work with the exposed result
        (run :mysql1 (insert-into...)) "
  ([connection-info results ast & body]
     `(if-not (keyword? ~connection-info)
	(run* ~connection-info ~ast (fn [~results] ~@body))
	(run-global ~connection-info ~ast (fn [~results] ~@body))))
  ([connection-info ast]
     `(if-not (keyword? ~connection-info)
       (with-connection [open-connection# ~connection-info]
	  (execute-sql ~ast open-connection#))
       (run-global ~connection-info ~ast))))
