(ns scicloj.clay.v2.page
  (:require
   [babashka.fs :as fs]
   [clj-yaml.core :as yaml]
   [clojure.java.io :as io]
   [clojure.java.shell :as shell]
   [clojure.string :as str]
   [hiccup.core :as hiccup]
   [hiccup.page]
   [scicloj.clay.v2.files :as files]
   [scicloj.clay.v2.item :as item]
   [scicloj.clay.v2.prepare :as prepare]
   [scicloj.clay.v2.styles :as styles]
   [scicloj.clay.v2.util.resource :as resource]))

(def special-lib-resources
  {:vega {:js {:from-the-web
               ["https://cdn.jsdelivr.net/npm/vega@5.25.0"
                "https://cdn.jsdelivr.net/npm/vega-lite@5.16.3"
                "https://cdn.jsdelivr.net/npm/vega-embed@6.22.2"]}}
   :datatables {:js {:from-the-web
                     ["https://cdn.datatables.net/1.13.6/js/jquery.dataTables.min.js"]}
                :css {:from-the-web
                      ["https://cdn.datatables.net/1.13.6/css/jquery.dataTables.min.css"]}}
   :mermaid {:js {:from-the-web
                  ["https://cdn.jsdelivr.net/npm/mermaid@11.10.1/dist/mermaid.min.js"]}}
   :graphviz {:js {:from-the-web
                   ["https://cdn.jsdelivr.net/npm/@viz-js/viz@3.17.0/dist/viz-global.min.js"]}}
   :echarts {:js {:from-the-web
                  ["https://cdn.jsdelivr.net/npm/echarts@5.4.1/dist/echarts.min.js"]}}
   :cytoscape {:js {:from-the-web
                    ["https://cdnjs.cloudflare.com/ajax/libs/cytoscape/3.23.0/cytoscape.min.js"]}}
   :plotly {:js {:from-the-web
                 ["https://cdnjs.cloudflare.com/ajax/libs/plotly.js/2.20.0/plotly.min.js"]}}
   :katex {:js {:from-the-web
                ["https://cdn.jsdelivr.net/npm/katex@0.16.10/dist/katex.min.js"]}
           :css {:from-the-web
                 ;; fetching the KaTeX css from the web
                 ;; to avoid fetching the fonts locally,
                 ;; which would need a bit more care
                 ;; (see https://katex.org/docs/font.html)
                 ["https://cdn.jsdelivr.net/npm/katex@0.16.10/dist/katex.min.css"]}}
   :three-d-mol {:js {:from-the-web
                      ["https://cdnjs.cloudflare.com/ajax/libs/3Dmol/1.5.3/3Dmol.min.js"]}}
   :leaflet {;; fetching Leaflet from the web
             ;; to avoid fetching the images locally,
             ;; which would need a bit more care.
             :js {:from-the-web
                  ["https://cdnjs.cloudflare.com/ajax/libs/leaflet/1.9.4/leaflet.js"
                   "https://cdn.jsdelivr.net/npm/leaflet-providers@2.0.0/leaflet-providers.min.js"]}
             :css {:from-the-web
                   ["https://cdnjs.cloudflare.com/ajax/libs/leaflet/1.9.4/leaflet.css"]}}
   :reagent {:js {:from-the-web
                  ["https://cdn.jsdelivr.net/npm/react@18/umd/react.production.min.js"
                   "https://cdn.jsdelivr.net/npm/react-dom@18/umd/react-dom.production.min.js"
                   "https://daslu.github.io/scittle/js/scittle.js"
                   "https://daslu.github.io/scittle/js/scittle.cljs-ajax.js"
                   "https://daslu.github.io/scittle/js/scittle.reagent.js"
                   "https://cdn.jsdelivr.net/npm/d3-require@1"]}}
   ;; :tmdjs {:js {:from-the-web
   ;;              ["https://daslu.github.io/scittle/js/scittle.tmdjs.js"]}}
   :emmy {:js {:from-the-web
               ["https://daslu.github.io/scittle/js/scittle.emmy.js"]}}
   :emmy-viewers {:js {:from-the-web
                       ["https://daslu.github.io/scittle/js/scittle.emmy.js"
                        "https://daslu.github.io/scittle/js/scittle.emmy-viewers.js"]}
                  :css {:from-the-web
                        ["https://unpkg.com/mafs@0.18.8/core.css"
                         "https://unpkg.com/mafs@0.18.8/font.css"
                         "https://unpkg.com/mathbox@2.3.1/build/mathbox.css"
                         "https://unpkg.com/mathlive@0.85.1/dist/mathlive-static.css"
                         "https://unpkg.com/mathlive@0.85.1/dist/mathlive-fonts.css"]}}
   ;; :mathbox {:js {:from-the-web
   ;;                ["https://daslu.github.io/scittle/js/scittle.mathbox.js"]}}
   :portal {:js {:from-the-web ["https://djblue.github.io/portal/main.js"]}}
   :d3 {:js {:from-the-web
             ["https://cdn.jsdelivr.net/npm/d3@7"]}}
   :html-default {:js {:from-the-web
                       ["https://code.jquery.com/jquery-3.6.0.min.js"
                        "https://code.jquery.com/ui/1.13.1/jquery-ui.min.js"]}}
   :md-default {:js {:from-the-web
                     ["https://code.jquery.com/jquery-3.6.0.min.js"
                      "https://code.jquery.com/ui/1.13.1/jquery-ui.min.js"]}}
   :htmlwidgets-ggplotly {:js {:from-the-web
                               ["https://raw.githubusercontent.com/scicloj/ggplotly-deps/refs/heads/main/lib/htmlwidgets-1.6.2/htmlwidgets.js"
                                "https://raw.githubusercontent.com/scicloj/ggplotly-deps/refs/heads/main/lib/plotly-binding-4.10.4.9000/plotly.js"
                                "https://raw.githubusercontent.com/scicloj/ggplotly-deps/refs/heads/main/lib/typedarray-0.1/typedarray.min.js"
                                "https://raw.githubusercontent.com/scicloj/ggplotly-deps/refs/heads/main/lib/jquery-3.5.1/jquery.min.js"
                                "https://raw.githubusercontent.com/scicloj/ggplotly-deps/refs/heads/main/lib/crosstalk-1.2.1/js/crosstalk.min.js"
                                "https://raw.githubusercontent.com/scicloj/ggplotly-deps/refs/heads/main/lib/plotly-main-2.11.1/plotly-latest.min.js"]}
                          :css {:from-the-web
                                ["https://raw.githubusercontent.com/scicloj/ggplotly-deps/refs/heads/main/lib/crosstalk-1.2.1/css/crosstalk.min.css"
                                 "https://raw.githubusercontent.com/scicloj/ggplotly-deps/refs/heads/main/lib/plotly-htmlwidgets-css-2.11.1/plotly-htmlwidgets.css"]}}
   :highcharts {:js {:from-the-web ["https://code.highcharts.com/highcharts.js"]}}})

(def include
  {:js hiccup.page/include-js
   :css hiccup.page/include-css})

(defn include-inline [js-or-css]
  (fn [url]
    (->> url
         ((include js-or-css))
         (map (fn [script-tag]
                (let [{:keys [src]} (second script-tag)]
                  (-> script-tag
                      (conj (slurp src))
                      (update 1 dissoc :src))))))))

(defn include-from-a-local-file [url custom-name js-or-css context]
  (let [[path relative-path]
        (files/next-file! context custom-name url (str "." (name js-or-css)))]
    (io/make-parents path)
    (->> url (resource/get) (spit path))
    ((include js-or-css) relative-path)))

(defn clone-repo-if-needed! [gh-repo]
  (let [target-path (str "/tmp/.clay/clones/" gh-repo)]
    (io/make-parents target-path)
    (when-not
        (.exists (io/file target-path))
      (let [repo-url (str "https://github.com/" gh-repo)]
        (prn [:cloning repo-url])
        (shell/sh "git" "clone" repo-url target-path)))
    target-path))

(defn include-from-a-local-copy-of-repo [{:as details
                                          :keys [gh-repo relative-path paths]}
                                         lib
                                         js-or-css
                                         {:keys [base-target-path]}]
  (let [repo-path (clone-repo-if-needed! gh-repo)
        target-repo-path (str (name lib) "/gh-repos/" gh-repo)
        target-copy-path (str base-target-path "/" target-repo-path)]
    (when-not (.exists (io/file target-copy-path))
      (babashka.fs/copy-tree (str repo-path "/" relative-path)
                             target-copy-path))
    (->> paths
         (map (fn [path]
                (->> path
                     (str target-repo-path "/")
                     ((include js-or-css))))))))

(defn resolve-deps [lib js-or-css]
  (cond (keyword? lib) (->> lib
                            special-lib-resources
                            js-or-css)
        (map? lib) (some->> lib
                            js-or-css
                            (hash-map :from-the-web))))

(defn include-libs-hiccup [{:as spec :keys [inline-js-and-css]}
                           deps-types libs]
  (->> deps-types
       (mapcat (fn [js-or-css]
                 (->> libs
                      (mapcat
                       (fn [lib]
                         (let [deps (resolve-deps lib js-or-css)
                               {:keys [from-the-web
                                       from-local-copy
                                       from-local-copy-of-repo]} (resolve-deps lib
                                                                               js-or-css)]
                           (if inline-js-and-css
                             (->> (concat from-the-web
                                          from-local-copy
                                          from-local-copy-of-repo)
                                  (map (include-inline js-or-css)))
                             ;; else
                             (concat
                              (some->> from-the-web
                                       (apply (include js-or-css))
                                       vector)
                              (some->> from-local-copy
                                       (map (fn [url]
                                              (include-from-a-local-file
                                               url
                                               lib
                                               js-or-css
                                               spec))))
                              (some->> from-local-copy-of-repo
                                       (map (fn [details]
                                              (include-from-a-local-copy-of-repo
                                               details
                                               lib
                                               js-or-css
                                               spec))))))))))))
       (apply concat)))

(defn include-libs [spec deps-types libs]
  (->> libs
       (include-libs-hiccup spec deps-types)
       hiccup/html
       (format "\n%s\n")))

(def font-links
  " <link rel=\"preconnect\" href=\"https://fonts.googleapis.com\">
    <link rel=\"preconnect\" href=\"https://fonts.gstatic.com\" crossorigin>
    <link href=\"https://fonts.googleapis.com/css2?family=Roboto&display=swap\" rel=\"stylesheet\">
")

(defn items->deps [items]
  (->> items
       (mapcat :deps)
       distinct))

(defn html [{:as spec
             :keys [items title toc? favicon exception]}]
  (let [deps (items->deps items)
        special-libs (concat [:html-default :katex] deps)
        head [:head
              [:meta {:charset "UTF-8"}]
              [:meta {:name "viewport" :content "width=device-width, initial-scale=1"}]
              (when favicon
                [:link {:rel "icon" :href favicon}])
              [:style (styles/main :table)]
              [:style (styles/main :loader)]
              [:style (styles/main :code)]
              [:style (styles/main :bootswatch-cosmo-bootstrap.min)]
              [:style (styles/main :bootstrap-generated-by-quarto.min)]
              [:style (styles/highlight :panda-syntax-light)]
              [:style (styles/main :main)]
              (when toc?
                (include-from-a-local-file
                 "https://cdn.rawgit.com/afeld/bootstrap-toc/v1.0.1/dist/bootstrap-toc.min.css"
                 "bootstrap-toc"
                 :css
                 spec))
              (when toc?
                [:style (styles/main :bootstrap-toc-customization)])
              (include-libs spec [:css] special-libs)
              [:title (or title "Clay")]]
        body [:body  {:style {:margin "auto"
                              :border (when exception "2px solid red")}
                      :data-spy "scroll"
                      :data-target "#toc"}
              (when toc?
                (include-from-a-local-file
                 "https://cdn.rawgit.com/afeld/bootstrap-toc/v1.0.1/dist/bootstrap-toc.min.js"
                 "bootstrap-toc"
                 :js
                 spec))
              (include-libs spec [:js] special-libs)
              (when (some #{:scittle :reagent} deps)
                (item/scittle-tag
                 item/scittle-header-form))
              [:div.container
               [:div.row
                (when toc?
                  [:div.col-sm-3
                   [:nav.sticky-top {:id "toc"
                                     :data-toggle "toc"}]])
                [:div {:class (if toc?
                                "col-sm-9"
                                "col-sm-12")}
                 [:div
                  (->> items
                       (map-indexed
                        (fn [i item]
                          [:div {:style {:margin "15px"}}
                           (prepare/item->hiccup item spec)]))
                       (into [:div]))]]]]
              [:script {:type "text/javascript"}
               (-> "highlight/highlight.min.js"
                   io/resource
                   slurp)]
              [:script {:type "text/javascript"}
               "hljs.highlightAll();"]]]
    (hiccup.page/html5 head body)))

(defn front-matter
  "Returns document metadata suitable for inclusion as front-matter to a Markdown document."
  [{:as spec
    :keys [title favicon quarto format]}]
  (let [quarto-target (or (second format) :html)]
    (cond-> quarto
            ;; Users may provide non-quarto specific configuration (see also html),
            ;; if so this will be added to the quarto front-matter to make them behave the same way
            title (assoc-in [:format quarto-target :title] title)
            favicon (update-in [:format quarto-target :include-in-header :text]
                               str "<link rel = \"icon\" href = \"" favicon "\" />"))))

(defn md [{:as spec
           :keys [items]}]
  (let [deps (items->deps items)]
    (str
     "---\n"
     (yaml/generate-string (front-matter spec))
     "\n---\n"
     (hiccup/html
      [:style (styles/main :table)]
      [:style (styles/main :md-main)]
      [:style (styles/main :main)])
     (->> deps
          (cons :md-default)
          (include-libs spec [:js :css]))
     (when (some #{:scittle :reagent} deps)
       (hiccup/html
        (item/scittle-tag
         item/scittle-header-form)))
     (->> items
          (map-indexed
           (fn [i item]
             (prepare/item->md item)))
          (str/join "\n\n")))))


(defn gfm [{:as spec
            :keys [items]}]
  (->> items
       prepare/items->gfm
       (remove nil?)
       (mapcat #(str/split % #"\n"))
       (remove #(re-matches #":::.*" %))
       (str/join "\n")))

(defn hiccup [{:as spec
               :keys [items title quarto]}]
  (let [deps (items->deps items)]
    (vec (concat (->> deps
                      (include-libs-hiccup spec [:js :css]))
                 (when (some #{:scittle :reagent} deps)
                   (item/scittle-tag
                    item/scittle-header-form))
                 (->> items
                      (map #(prepare/item->hiccup % spec)))))))
