(ns ifast.core
  (:require #?(:clj [clojure.spec.alpha :as s]
               :cljs [cljs.spec.alpha :as s])
            [ifast.meal-record :as record]
            [ifast.fasting :as fasting]))

(defn empty-app-data
  "Create a new application."
  []
  {::meal-record   (record/empty-record)
   ::fasting-style fasting/sixteen-eight})

(defn record-a-meal
  "Record a meal with the given timestamp."
  [app-data timestamp]
  (update-in app-data [::meal-record] record/record-meal-at timestamp))

(def get-meal-record ::meal-record)

;; TODO move this and above fns to external module
(defn fasting-over?
  "Determines if, with the chosen fasting style, the fasting period is already over."
  [{:keys [::meal-record ::fasting-style]} time]
  (let [time-of-last-meal (record/last-meal meal-record)]
    (if-not (nil? time-of-last-meal)
      (fasting/can-eat? fasting-style time-of-last-meal time)
      false)))

(defprotocol IDataPersistance
  "Store and save app data."

  (load-data! [this] "Load app data from somewhere. Ought to return ::app-data.")

  (persist-data! [this app-data] "Store app-data somewhere."))

(s/def ::storage #(satisfies? IDataPersistance %))

(s/def ::app (s/keys :req [::app-data ::storage]))

(s/fdef init-app!
        :args (s/cat :storage ::storage)
        :ret ::app)
(defn init-app!
  "Create a new application or load an exising."
  [storage]
  (let [previous-data (load-data! storage)]
    {::app-data (or previous-data (empty-app-data))
     ::storage  storage}))

(s/fdef record-meal!
        :args (s/cat :app ::app :time ::record/timestamp)
        :ret ::app)
(defn record-meal!
  "Record a meal and persist it."
  [app timestamp]
  (let [app* (update-in app [::app-data] record-a-meal timestamp)]
    (persist-data! (::storage app)
                   (::app-data app*))
    app*))

(s/fdef done-fasting?
        :args (s/cat :app ::app :time ::record/timestamp)
        :ret boolean?)
(defn done-fasting?
  "Check the data in `app` if fasting is over @ `current-time`."
  [app current-time]
  (fasting-over? (::app-data app) current-time))
