(ns scrum.devtools.ui
  (:require [rum.core :as rum]
            [clojure.data :as cd]
            [cljs.pprint :as pprint]
            [scrum.devtools.state-tree :as state-tree]
            [scrum.devtools.styles :as styles]))

(when ^boolean goog.DEBUG
  (enable-console-print!))

;; history mixin
(def history-mixin
  {:did-mount
   (fn [{[_ state] :rum/args
         local ::local
         :as st}]
     (add-watch state ::history
       (fn [_ _ old-s new-s]
         (let [{cidx :curr-state
                pidx :prev-state
                history :history}
               @local]
           (cond
             (nil? cidx)
             (swap! local (fn [st]
                            (-> st
                              (update :history conj {:time (js/Date.now) :state new-s :prev old-s})
                              (assoc :prev-state cidx :curr-state (count history)))))

             (not= (->> cidx (nth history) :state) new-s)
             (swap! local (fn [st]
                            (-> st
                                (update :history conj {:time (js/Date.now) :state new-s :prev old-s})
                                (assoc :prev-state cidx :curr-state (count history)))))

             :else nil))))
     st)
   :will-unmount
   (fn [{[_ state] :rum/args
         :as st}]
     (remove-watch state ::history)
     st)})


;; ui
(rum/defc Button
  [on-click text]
  [:button {:style (styles/button)
            :on-click on-click}
   text])

(rum/defc Menu [on-choose]
  [:div {:style (styles/menu-container)}
   [:div {:style (styles/menu-item)}
    (Button #(on-choose :history) "State history")]
   [:div {:style (styles/menu-item)}
    (Button #(on-choose :visual) "State visualization")]])

(rum/defc Header [screen on-nav]
  [:div {:style (styles/header)}
   (Button #(on-nav :menu) "<")
   [:div {:style (styles/header-title)}
    (case screen
      :history "State history"
      :visual "State visualization"
      nil)]])

(rum/defc Visualization
  [screen on-nav state]
  [:div
   (Header screen on-nav)
   [:div {:style (styles/screen-container)}
    (state-tree/render state)]])

(rum/defcs HistoryItem <
  (rum/local false ::visible?)
  [{visible? ::visible?} on-select active? {:keys [time state prev]}]
  [:div {:style (styles/history-item active?)}
   [:div {:style {:display "flex"
                  :justifyContent "space-between"}}
    [:div {:style {:alignSelf "center"}}
     time]
    [:div
     (Button #(swap! visible? not) (if @visible? "hide diff" "show diff"))
     [:span " "]
     (Button on-select "apply state")]]
   (when @visible?
     (let [[before after] (cd/diff prev state)]
       [:div
        [:pre
         [:code "Before:\n"]
         [:code (with-out-str (pprint/pprint before))]
         [:code "\n"]
         [:code "After:\n"]
         [:code (with-out-str (pprint/pprint after))]]]))])


(rum/defc History
  [screen on-nav on-select history cidx]
  [:div
   (Header screen on-nav)
   [:div {:style (styles/screen-container)}
    (->> history
      (sort-by :time)
      (map-indexed #(HistoryItem (partial on-select %1) (= cidx %1) %2)))]])

(rum/defcs UI <
  history-mixin
  rum/static
  rum/reactive
  (rum/local
    {:screen :menu
     :history []
     :curr-state nil
     :prev-state nil}
    ::local)
  [{local ::local}
   {:keys [position]}
   state]
  (let [_ (rum/react state)
        {:keys [screen history curr-state]} @local
        on-nav #(swap! local assoc :screen %)
        on-select-state
        #(do
           (swap! local assoc :prev-state curr-state :curr-state %)
           (reset! state (->> % (nth history) :state)))]
    [:div {:style (styles/container position)}
     (case screen
       :menu (Menu on-nav)
       :history (History screen on-nav on-select-state history curr-state)
       :visual (Visualization screen on-nav @state)
       nil)]))


;; lifecycle
(defn mount!
  ([state node]
   (mount! state node {}))
  ([state node opts]
   (rum/mount (UI opts state) node)))

(defn unmount! [node]
  (rum/unmount node))
