; This Source Code Form is subject to the terms of the Mozilla Public License,
; v. 2.0. If a copy of the MPL was not distributed with this file, You can
; obtain one at http://mozilla.org/MPL/2.0/.

(ns cemerick.splice.types
  (:require [quilt.sedan :as sedan]
            [quilt.sedan.impl :as sedan.impl]
                   cljs.reader
                                           
                                      
                                )
  (                      :require-macros
    [quilt.sedan.macros :refer (define-partition! !0)]))

; TODO probably should move this to splice.reference; I really don't ever want
; to implement another type like this...

(defprotocol IEpoch
  (as-of [x])
  (unbounded? [x]))

(defn- reference-components
  [reference]
  (if (unbounded? reference)
    @reference
    [@reference (as-of reference)]))

(deftype Reference [referent epoch]
                                   IDeref
  (                   -deref [_] referent)
  IEpoch
  (as-of [x] epoch)
  (unbounded? [x] (= epoch sedan/top))
                          IComparable
  (                       -compare [_ other]
    (!0 (sedan.impl/default-compare referent (.-referent other))
      (sedan.impl/default-compare epoch (.-epoch other))))
              
       
                                 
         IHash
  (                      -hash [this]
    (inc (hash (reference-components this))))
         IEquiv
  (                    -equiv [this other]
    (and (instance? Reference other)
      (= referent (.-referent other))
      (= epoch (.-epoch other)))))

(declare reference)

(defn- reference-tagged-reader-fn
  [x]
  (if (vector? x)
    (apply reference x)
    (reference x)))

     
   
                                                            
                                          

                                                           
                      
                                      
                        
                                                   
                                      
                       
                              
                                  
             
                                           

                                          
                    
                                                    
                              
                              
                       
                                          
                       

      
(do
  (cljs.reader/register-tag-parser! 'ref reference-tagged-reader-fn)
  (extend-type Reference
    IPrintWithWriter
    (-pr-writer [this w opts]
      (-write w "#ref ")
      (pr-writer (if (unbounded? this)
                    @this
                    (reference-components this))
        w opts))))

(defn reference?
  "Returns true iff [x] is a Reference."
  [x]
  (instance? Reference x))

(defn reference
  "Creates a new reference given a [referent], and an optional value indicating
  the endpoint of the epoch after which changes to the [referent] should be
  disregarded.

When called with a Reference, that Reference will be returned unmodified.  When
called with a Reference and an epoch endpoint, will return a Reference that
contains the same referent, but with the new epoch endpoint."
  ([referent]
     (if (instance? Reference referent)
       referent
       (reference referent sedan/top)))
  ([referent epoch]
     (if (instance? Reference referent)
       (Reference. (.-referent referent) epoch)
       (Reference. referent epoch))))

(define-partition! 0x50 :reference Reference
  (fn decode-reference [tag ^String s]
    (sedan.impl/decode-sequence reference (sedan.impl/without-tag s) false))
  (encode* [^Reference x buffer]
    (sedan.impl/encode-sequence [(.-referent x) (.-epoch x)] buffer false))
  (compare* [a b]
    (sedan.impl/default-compare a b)))


;;; ********************* (partially) ordered attributes

; TODO would like to deref to get the attr, but the rank is hardly "just"
; metadata like reference epochs (mostly?) are
(defprotocol IOrderedAttribute
  (attr [x])
  (rank [x]))

(defn- oattr-components
  [oattr]
  [(attr oattr) (rank oattr)]) 

(deftype OrderedAttribute [attr rank]
  IOrderedAttribute
  (attr [_] attr)
  (rank [_] rank)
                          IComparable
  (                       -compare [_ other]
    (!0 (sedan.impl/default-compare attr (.-attr other))
      (sedan.impl/default-compare rank (.-rank other))))
              
       
                                 
         IHash
  (                      -hash [this]
    (inc (hash (oattr-components this))))
         IEquiv
  (                    -equiv [this other]
    (and (instance? OrderedAttribute other)
      (= attr (.-attr other))
      (= rank (.-rank other)))))

(declare oattr)

(defn- oattr-tagged-reader-fn
  [x]
  (assert (and (vector? x) (== 2 (count x)))
    "oattr must be represented by a vector of [attribute, rank]")
  (apply oattr x))

; TODO could print rank strings (if they _are_ strings) in hex to make REPL debugging easier

     
   
                                                            
                                     

                                                                    
                     
                                              
                                             
                       
                              
                                  
                     
                                           

                                                  
                             
                                                
                                        
                     
                                        
                       

      
(do
  (cljs.reader/register-tag-parser! 'oa oattr-tagged-reader-fn)
  (extend-type OrderedAttribute 
    IPrintWithWriter
    (-pr-writer [this w opts]
      (-write w "#oa ")
      (pr-writer (oattr-components this) w opts))))

(defn oattr?
  "Returns true iff [x] is a partially-ordered attribute."
  [x]
  (instance? OrderedAttribute x))

(defn oattr
  "Creates a new partially-ordered attribute given a base [attribute] (which may
  be any splice-compatible value _other than_ a oattr), and a [rank] value
  indicating the ordering of that attribute.

When called with a OrderedAttribute, it will be returned unmodified.  When
called with a OrderedAttribute and a new rank value, will return a OrderedAttribute that
contains the same base attribute, but with the new rank."
  [attr rank]
  (if (instance? OrderedAttribute attr)
    (OrderedAttribute. (.-attr attr) rank)
    (OrderedAttribute. attr rank)))

(define-partition! 0x51 :oattr OrderedAttribute 
  (fn decode-oattr [tag ^String s]
    (sedan.impl/decode-sequence oattr (sedan.impl/without-tag s) false))
  (encode* [^OrderedAttribute x buffer]
    (sedan.impl/encode-sequence [(.-attr x) (.-rank x)] buffer false))
  (compare* [a b]
    (sedan.impl/default-compare a b)))


;;;;;;;;;;;; This file autogenerated from src/cljx/cemerick/splice/types.cljx
