; This Source Code Form is subject to the terms of the Mozilla Public License,
; v. 2.0. If a copy of the MPL was not distributed with this file, You can
; obtain one at http://mozilla.org/MPL/2.0/.

(ns cemerick.sedan.numbers
  (:require [cemerick.sedan.sbuffers :as sbuf]
                                                     
            [clojure.string :as str])
         (:require-macros [cemerick.sedan.macros :refer (!0)])
  (:refer-clojure :exclude (integer? float? number? zero? neg?
                                                 
                                                 
                                                   )))

(def INF        js/Infinity                               )
(def -INF (- INF))

(def ^:private zero-code 48) ; \0
; no idea why \a is the origin...
(def ^:private bcd-zero-exponent 97) ; \a

; equivalent to (and ported from) BCDUtils.base10toBase10kSortableInt, except:
; * the exponent is kept separate as the head of a separate return vector
; * everything is an int, not a char, so the exponent won't roll over from 0 to 
;   65535
; NOTE ... BCDUtils calls this an exponent, but it's really not...it's the
; number of *encoded* digits, unrelated to the magnitude of the number being
; encoded
; kept here only for reference purposes, no external code should ever call it
; TODO move to the test ns then?
(defn- bcd-base-10kint
  [n]
  (if (clojure.core/zero? n)
    [bcd-zero-exponent]
    (let [^String s (str n)
        neg (= \- (.charAt s 0))
        enc (->> (if neg (rest s) s)
                 reverse
                 (partition-all 4)
                 reverse
                 (map #(loop [vals %
                              enc 0
                              factor 1]
                         (if-let [v (first vals)]
                           (recur (rest vals)
                                  (+ enc (* factor (- (int v) zero-code)))
                                  (* factor 10))
                           enc))))]
    (if neg
      [(- bcd-zero-exponent (count enc)) (map (fn [v] (- 9999 v)) enc)]
      [(+ bcd-zero-exponent (count enc)) enc]))))

(defn ==-0
  "Returns true if [x] is -0.0."
  [x]
  ; WTF am I required to hint .equals calls?
                                
         (and (identical? x 0)
              (identical? -INF (/ 1 x))))

; start of big support for CLJS
; so much of this is cruft due to using opaque, non-compuatble containers for
; bigs; once we're using bignumber.js or similar, we can actually do math and
; not F around with strings so gd much
(defprotocol INumber
  (float? [x])
  (big? [x])
  (zero? [x])
  (neg? [x]))

(defprotocol IDecimal
  (scale [x])
  (unscaled-value [x]))

(defn number? [x] (satisfies? INumber x))
(defn decimal? [x] (satisfies? IDecimal x))
(defn integer? [x] (not (or (decimal? x) (float? x))))
(defn finite?
  [x]
                                                                 
         (or (big? x) (js/isFinite x)))

(defn- number-components
  "Returns a tuple of [sign mantissa exponent] of the given number.  The sign
  may be '-' or nil.  Mantissas and exponents are normalized,
  e.g. -456.789e0 yields [\"-\" \"456789\" 2]."
  [number]
  (let [[_ s m e] (re-matches #"([\+\-])?([\d\.]+)[eE]?([\-\+]?\d+)?" (str number))
        [i f] (str/split m #"\.")
        f (when f (if (decimal? number) f (str/replace f #"0+$" "")))
        m (str i f)
        [m shift] (if (re-matches #"0+" m)
                    ["0" (if (decimal? number)
                           (- (dec (count m)))
                           0)]
                    (let [m' (str/replace m #"^0+" "")]
                      [m' (- (count m') (count m))]))
        e (+ (count i)
             shift
             -1
             (if e (                            js/parseInt e 10) 0))]
    [(when (= s "-") s) m e]))


     
                        
        
                   
                  
                                    
                                  
      
                    
                  
                                    
                                  
         
                    
                  
                                    
                                  
                     
                    
                 
                                    
                                  
            
                    
                 
                                    
                                  
            
                    
                                    
                                  
                  

      
(extend-protocol INumber
  number
  (neg? [x] (cljs.core/neg? x))
  (float? [x]
    ;; fucking javascript fucking javascript
    ; Number.isInteger exists in firefox, but nowhere else
    ; The polyfill suggested in the ecma harmony wiki is dubious (unkind
    ; comments from Eich and Crockford, though that didn't stop the same
    ; polyfill from landing on the MDN page for isInteger). Even so, that
    ; polyfill doesn't distinguish -0.0.
    ; This is slow, but reliable; TODO find better solution later
    (and (cljs.core/number? x)
         (or (==-0 x)
             (re-find #"[eE]|\.\d" x))))
  (big? [x] false)
  (zero? [x] (cljs.core/zero? x)))

     
                         
            
                                         
                         

      
(deftype BigDecimal [n]
  Object
  (toString [x] n)
  IEquiv
  (-equiv [x y]
    (and (instance? BigDecimal y)
         (= (number-components x) (number-components y))))
  IPrintWithWriter
  (-pr-writer [x writer _]
    (-write writer n)
    (-write writer "M"))
  IDecimal
  (unscaled-value [x]
    (second (number-components x)))
  (scale [x]
    (let [[s m e] (number-components x)]
      (- (dec (count m)) e)))
  INumber
  (neg? [x] (= "-" (.charAt n 0)))
  (float? [x] false)
  (big? [x] true)
  (zero? [x] (boolean (re-matches #"[+-]?0(\.0+)?([eE][+-]?\d+)?" n))))


(defn bigdec
  "Creates a shim BigDecimal, containing the given string representation of a
number.  Always use this, never directly create an instance of BigDecimal."
  [string]
  (BigDecimal. (str/replace (str string) #"M$"  "")))

      
(deftype BigInteger [n]
  Object
  (toString [x] n)
  IEquiv
  (-equiv [x y]
    (and (instance? BigInteger y)
         (= n (.-n y))))
  IPrintWithWriter
  (-pr-writer [x writer _]
    (-write writer n)
    (-write writer "N"))
  INumber
  (neg? [x] (= "-" (.charAt n 0)))
  (decimal? [x] false)
  (float? [x] false)
  (big? [x] true)
  (zero? [x] (boolean (re-matches #"0+" n))))

(defn bigint
  "Creates a shim BigInteger, containing the given string representation of a
number.  Always use this, never directly create an instance of BigInteger."
  [string]
  ; we wouldn't need this if decode-base10k didn't produce strings with leading zeros
  (BigInteger. (str/replace (str string) #"N$" "")))

                            
       (defn ncompare
         [a b]
         ; the intermediate comparison dispatch swaps these args
         (.__ncompare b a))

(defn- compare-components
  [this other]
  (let [[s m e] (number-components this)
        [s' m' e'] (number-components other)
        neg? (= s "-")
        ; significant zeros don't figure into numerical comparisons/equality
        m (str/replace-first m #"0+$" "")
        m' (str/replace-first m' #"0+$" "")]
    (cond
     (= m m' "") 0
     (= m "") (if (= "-" s') 1 -1)
     (= m' "") (if neg? -1 1)
     :else (if (= s s')
             ((if neg? - +)
              (!0 (ncompare e e')
                  (compare m m')))
             (if neg? -1 1)))))

; all places where `this` escapes need `(.valueOf this)`
; see https://groups.google.com/forum/#!topic/clojurescript/q3jVGxkGTIw

       (defn- compare-components-method
         [other]
         (this-as this (compare-components (.valueOf this) other)))

       (defn- compare-components-number-method
         [other]
         (this-as this
           (if (js/isFinite other)
             (compare-components (.valueOf this) other)
             (if (pos? other) -1 1))))

      
(set! (.-__ncompare js/cemerick.sedan.numbers.BigInteger.prototype)
      (fn [other]
        (this-as this (.__ncompare_bigint other (.valueOf this)))))

      
(set! (.-__ncompare_bigint js/cemerick.sedan.numbers.BigInteger.prototype)
      compare-components-method)

      
(set! (.-__ncompare_bigdec js/cemerick.sedan.numbers.BigInteger.prototype)
      compare-components-method)

      
(set! (.-__ncompare_number js/cemerick.sedan.numbers.BigInteger.prototype)
      compare-components-number-method)

      
(set! (.-__ncompare js/cemerick.sedan.numbers.BigDecimal.prototype)
      (fn [other]
        (this-as this (.__ncompare_bigdec other (.valueOf this)))))

      
(set! (.-__ncompare_bigint js/cemerick.sedan.numbers.BigDecimal.prototype)
      compare-components-method)

      
(set! (.-__ncompare_bigdec js/cemerick.sedan.numbers.BigDecimal.prototype)
      compare-components-method)

      
(set! (.-__ncompare_number js/cemerick.sedan.numbers.BigDecimal.prototype)
      compare-components-number-method)

      
(set! (.-__ncompare js/Number.prototype)
      (fn [other]
        (this-as this (.__ncompare_number other (.valueOf this)))))

      
(set! (.-__ncompare_number js/Number.prototype)
      (fn [other]
        (this-as this (goog.array.defaultCompare (.valueOf this) other))))

      
(defn- number-compare-components-method
  [other]
  (this-as this
           (if (js/isFinite this)
             (compare-components (.valueOf this) other)
             (if (pos? this) 1 -1))))

      
(set! (.-__ncompare_bigint js/Number.prototype)
      number-compare-components-method)

      
(set! (.-__ncompare_bigdec js/Number.prototype)
      number-compare-components-method)

;;;;;;;;;;;;;;;;; end JS bignum "support"

(defn- seq->string
  [char-points]
                                                                            
         (let [buf (sbuf/create)]
           (doseq [c (flatten char-points)]
             (sbuf/write buf (js/String.fromCharCode c)))
           (str buf)))

(defn- partition-string
  [max-count s]
  (let [len (count s)]
    (for [start (range 0 (count s) max-count)]
      (subs s start (min len (+ start max-count))))))

(defn- partition-int
  [string]
  (->> string
       str/reverse
       (partition-string 4)
       (map str/reverse)
       reverse))

(defn- suffix-pad
  [s]
  (str s (case (mod (count s) 4)
           1 "000"
           2 "00"
           3 "0"
           nil)))

(defn- prefix-pad
  [s]
  (str (case (mod (count s) 4)
           1 "000"
           2 "00"
           3 "0"
           nil)
       s))

; 0 - negative sign, large scale follows
; [1,62] - negative sign, value follows
; 63 - zero scale (and value) (midpoint of [1, 126])
; [64,126] - positive sign, value follows
; 127 - positive sign, large scale follows
; (using 0-127 so that sign+scale can fit in one byte for (most) numbers when
; UTF-8 encoded)
(def ^:private zero-scale 63)

; TODO this *always* uses partition-int, can remove the parameterization of partition-fn
(defn- base-10kint
  ([n] (base-10kint (str n) false true partition-int))
  ([^String s neg allow-10k-scale partition-fn]
   (if (= "0" s)
     [zero-scale]
     (let [neg-sign (= \- (.charAt s 0))
           neg (or neg neg-sign)
           enc (->> (if neg-sign (subs s 1) s)
                    partition-fn
                    ; prior failure #2
                    (map #(                              js/parseInt % 10)))
           enc (if neg (map (fn [v] (- 9999 v)) enc) enc)
           scale (count enc)
           op (if neg - +)]
       (cond
         (< scale zero-scale) [(op zero-scale scale) enc]
         allow-10k-scale [(if neg 0 127)
                          (base-10kint (str (op scale)) false false partition-fn)
                          enc]
         ; TODO TODO TODO need to specify (maybe enforce?) implementation limits
         ; on values how big is this? (a number that, when encoded in base-10000
         ; has a number of digits S, such that when S is encoded in base-10000,
         ; has >= 62 digits)
         :default (throw (ex-info "Cannot encode scale, too large" {:scale scale})))))))

(defn- char-code [^String s idx] (                          .charCodeAt s idx))
(defn- char-code-seq [s] (map #(char-code s %) (range (count s))))

(defn- trim-leading-zeros
  [s]
  ; adding |\0. to the lookahead assertion doesn't work?
  (str/replace-first s #"^0+(?=[^0])" ""))

(defn- decode-base10k
  "Turns a base-10k string into a base-10 string. [neg?] must be true if the
numeric value the string represents is negative."
  [s neg?]
  (->> (char-code-seq s)
       (map (if neg? #(- 9999 %) identity))
       (map (comp prefix-pad str))
       (apply str)
       ; only necessarily while we're not using a real big number type on JS, leading
       ; zeros cause difficulties since we're just shuffling strings aroung
       trim-leading-zeros))

(defn- parseint
  [base-10]
       
      
                               
                                  
                             
        
  (let [v (js/parseInt base-10 10)]
    ; this is hell
    (if (neg? (.indexOf base-10 (str v)))
      (bigint base-10)
      v)))

(defn- decode*
  [s decode-fn]
  ; zero and indication of negation only occur at the first byte of a number
  (let [head (char-code s 0)]
    (if (== head zero-scale)
      [0 (subs s 1)]
      (let [neg? (< head zero-scale)
            decoding (decode-fn s neg?)]
        (if neg?
          (update-in decoding [0]
                            
                     ; this is the most hilarious code I've ever written
                            (fn [x]
                              (if (cljs.core/number? x)
                                (- x)
                                (let [s (.-n x)
                                      s (if (= "-" (.charAt s 0))
                                          (subs s 1)
                                          (str "-" s))]
                                  (if (decimal? x)
                                    (bigdec s)
                                    (bigint s))))))
          decoding)))))

(defn- decode-int*
  [^String s neg?]
  (let [head (char-code s 0)
        [scale int-body] (if (or (== 0 head) (== 127 head))
                           (decode-int* (subs s 1) neg?)
                           [(Math/abs (- zero-scale head)) (subs s 1)])
        decoded-int (decode-base10k (subs int-body 0 scale) neg?)]
    [(parseint decoded-int)
     (subs int-body scale)
     decoded-int]))

(defn decode-int
  "Decodes strings produced by `encode-int` into integers.  An error will be
thrown if the underlying runtime cannot represent the encoded number."
  [s]
  (nth (decode* s decode-int*) 0))

(defn encode-int
  "Encodes the given _integer_ [n]."
  [n]
  {:pre [(integer? n)]}
  (-> n base-10kint seq->string))

; <n:encoded-int>:
; <sign><scale:encoded-int><n:encoded-int>
; decimals:
; <significand-sign><exp:encoded-int><significand:encoded-int>

(defn- invert-mantissa
  [mstring]
  (->> (char-code-seq mstring)
       (map #(- 57 %))))

(defn- revert-mantissa
  [mstring]
  (->> (invert-mantissa mstring)
       (map                   js/String.fromCharCode)
       (apply str)))


; numbers divided into 5+3 partitions (+1 to account for -0.0, +1 for -Inf, +1 for
; +Inf) per http://tools.ietf.org/html/draft-wood-ldapext-float-00
; The partition bytes are reused for "top-level" sedan tag bytes, thus the
; deviation between the zero-scale used for strictly-integer encoding (which
; does not correspond to a sedan partition

(def ^:private partition-zero 0x14)
(def ^:private partition-neg-zero (dec partition-zero))
(def ^:private partition-2 (dec partition-neg-zero))
(def ^:private partition-1 (dec partition-2))
(def ^:private partition-neg-inf (dec partition-1))
(def ^:private partition-4 (inc partition-zero))
(def ^:private partition-5 (inc partition-4))
(def ^:private partition-pos-inf (inc partition-5))

(defn- encode-number
  "Encodes the given number [f], which can be of any type, magnitude, and
precision.  The return is a nested sequence of _character codes_. Use
`seq->string` to produce a string from this."
  [f]
  (cond
   (zero? f) (if (==-0 f)
               ; specialized partition just for -0.0
               [partition-neg-zero]
               ; case/partition 3
               [partition-zero
                (let [[_ m e] (number-components f)]
                  (base-10kint (str e) false true partition-int))])
   
   (not (finite? f))
   (if (pos? f)
     [partition-pos-inf]
     [partition-neg-inf])

                         
          (and (cljs.core/number? f) (js/isNaN f)) ; no Number.isNaN on phantom
   (throw (ex-info "Cannot encode NaN" {:number f}))

   :default
   ; as janky as it is, using a regex is the most straightforward way to make
   ; this portable, given there's no way to get exponent and mantissa out of a
   ; js float
   ; TODO maybe this will work: http://stackoverflow.com/a/17156580/11809
   ; JVM-specific impl eventually maybe whatever
   (let [[s m e] (number-components f)
         neg-m (= s "-")
         neg-e (< e 0)]
     ;(prn int frac shift m e neg-m)
     ; other 4 cases
     (if neg-m
       (if (not neg-e)
         [partition-1
          (base-10kint (str (- e)) false true partition-int)
          (invert-mantissa m)]
         [partition-2
          (base-10kint (str (- e)) false true partition-int)
          (invert-mantissa m)])
       (if neg-e
         [partition-4
          (base-10kint (str e) false true partition-int)
          (char-code-seq m)]
         [partition-5
          (base-10kint (str e) false true partition-int)
          (char-code-seq m)])))))

; if we did this properly, then tag-for could be used by encode-number, not the
; other way 'round. Tension caused by sedan impl ns encoding by writing the tag,
; then calling out to "type"-specific encoding fns. #perf
(defn tag-for
  "Returns the tag byte for the given number."
  [n]
  (first (encode-number n)))

(defn decode
  "Decodes strings produced by `encode` into numbers.  An error will be thrown
if the underlying runtime cannot represent the encoded number with the intended
precision/scale."
  [s]
  (let [head (char-code s 0)
        special-value (condp == head
                        partition-neg-zero -0.0
                        partition-neg-inf -INF
                        partition-pos-inf INF
                        nil)]
    (or special-value
        (when (== head partition-zero)
          (let [e (decode-int (subs s 1))]
            (if (zero? e)
              0
              (BigDecimal. (str "0e" e)))))
        (let [neg? (< head partition-zero)
              [e sig-string] (decode* (subs s 1) decode-int*)
              [sig-string e] (condp == head
                               partition-1 [(revert-mantissa sig-string) (- e)]
                               partition-2 [(revert-mantissa sig-string) (- e)]
                               partition-4 [sig-string e]
                               partition-5 [sig-string e])
              sig-len (count sig-string)]
          ;; TODO surely there's a saner way to emit "correct" types
          ;; (non-big if not necessary to retain the represented precision)

          ;; TODO can we use >= here and emit integers for e.g. 1e7?
          (if (== e (dec sig-len))
            ; TODO If eventually we can do actual math on bigs in JS :-O,
            ; negating a parsed number as necessary would be better than consing
            ; another string
            (parseint (str (when neg? \-) sig-string))
            (let [vs (str (first sig-string)
                          \.
                          (subs sig-string 1)
                          "e" e)]
              ; do we have zero digits we need to preserve?
              (if (or
                   ; prevent 1e1M turning into 10.0
                   (and (== 1 (count sig-string)) (pos? e))
                   (re-find #"0$" sig-string))
                (BigDecimal. (str (when neg? \-) vs))
                (let [vs (str (when neg? \-) vs)
                      v (                                js/parseFloat vs)]
                  (if (or (not (finite? v))
                          ; effectively the same as prior failure #1:
                          ; compensating for infintesimal doubles that hit zero eventually,
                          ; which Double parses "incorrectly"
                          (and (zero? v)
                               (re-find #"[^0]" sig-string))
                          ; prior failure #4
                          ; ensuring that the represented number didn't lose
                          ; precision upon parsing; if it did, then the value was a
                          ; big to begin with
                          (and (>= (count sig-string) 15)
                               (let [[s m e] (number-components v)]
                                 (not= sig-string m))))
                    (BigDecimal. vs)
                    v)))))))))

(defn encode
  "Encodes the given number [f], which can be of any type, magnitude, and
precision, returning a string.  The string _will_ need sane encoding, like UTF-8."
  [n]
  (-> n encode-number seq->string))

; <n:encoded-int>:
; <sign><scale:encoded-int><n:encoded-int>
; decimals:
; <significand-sign><exp:encoded-int><significand:base-10>

;;;;;;;;;;;; This file autogenerated from src/cljx/cemerick/sedan/numbers.cljx
