(ns org.purefn.sqlium.import
  (:require [clojure.java.jdbc :as jdbc]
            [clojure.set :as set]
            [clojure.string :as str]
            [clojure.tools.logging :as log]
            [clj-time.core :as time]
            [clj-time.coerce :as tc]
            [org.purefn.sqlium.dsl :as dsl]
            [org.purefn.sqlium.sql :as sql])
  (:import java.util.ArrayList
           java.sql.ResultSet))

(def default-batch-size 10000)

(defn table-id
  "Returns keyword representing a table id field, eg :table/field."
  [table]
  (keyword (:name table) (:id table)))

(defn join-results
  "Takes a relationship map for a many relationship, table data, and
   the relationship's data, and joins the relationship data into the
   table data."
  [rel data rel-data]
  (let [{:keys [source-table target column]} rel
        rel-data-lookup (group-by #(get % column) rel-data)
        id-col (dsl/id-column source-table)]
    (map (fn [rec]
           (let [rec-id (get rec id-col)
                 rec-data (get rel-data-lookup rec-id)]
             (cond-> rec
               rec-data (assoc-in [:many-relationships column] rec-data))))
         data)))

(defn fetch-results
  "Takes the [sql col-aliases] tuple as returned by `sql/select` and
   fetches the results, renaming the keys in the returned records
   using col-aliases."
  [db sql-with-aliases]
  (let [[sql col-aliases] sql-with-aliases]
    ;; TODO: fix jdbc call
    (->> (jdbc/query db [sql])
         (map #(set/rename-keys % col-aliases)))))

(defn- result-set-column-list
  "Extracts a single column by name from each row of a ResultSet,
   adding it to an ArrayList."
  [^ResultSet rs colname]
  (let [alist (ArrayList. 1000)]
    (while (.next rs)
      (.add alist (.getObject rs colname)))
    alist))

(defn fetch-column
  "Efficient, low-level query function to return a collection of a
   single column. Takes a sql query, the column name to fetch, and
   returns an ArrayList with the values of that column."
  [db query column]
  (jdbc/db-query-with-resultset db [query] #(result-set-column-list % column)))

(defn import-many-relationship
  "Takes a db, a many relationship map and collection of source table data,
   retrieves the related data and merges it into the table data."
  [db rel data]
  (when-let [sql-with-aliases (sql/many-relationship-select rel data)]
    (log/debug :fn "import-many-relationship"
               :query (first sql-with-aliases))
    (let [many-data (fetch-results db sql-with-aliases)
          many-rels (get-in rel [:target :relationships :many])]
      (reduce (fn [data rel]
                (let [rel-data (import-many-relationship db rel data)]
                  (join-results rel data rel-data)))
              many-data
              many-rels))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; fetching ids
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defn table-paths
  "Returns a map, keyed by table name, with all paths to get to that
   table in given compiled spec.

   A path is a vector of [table-with-rel* table-without-rel]

   Returns  `{table-name #{path}}`"
  ([table-spec]
   (table-paths {} [] table-spec))
  ([paths base-path table-spec]
   (let [{:keys [name fields]} table-spec
         rels (filter dsl/relationship? fields)
         base-table (dissoc table-spec :fields)
         table-path (conj base-path base-table)
         next-paths (update paths name (fnil conj #{}) table-path)]
     (reduce (fn [ps rel]
               (let [target (:target rel)
                     base-rel (dissoc rel :target)]
                 (table-paths ps (conj base-path
                                       (assoc base-table :relationship base-rel))
                              target)))
             next-paths
             rels))))

(defn join-path
  "Takes a path (as from table-paths) and returns collection of maps
   describing each join in the path, containing:

   `{:base {:name \"table-name\"
            :alias \"optional-alias\"
            :join-col \"col-name\"}
     :target {:name \"table-name\"
              :alias \"table-alias\"
              :join-col \"col-name\"}`

   Table aliases are used so that tables can be repeated in a
   relationship chain. The aliases are generated by appending an
   incrementing number to the table name."
  [path]
  (->> path
       (partition 2 1)
       (map-indexed
        (fn [idx [t1 t2]]
          (let [{t1-id :id t1-name :name} t1
                {t2-id :id t2-name :name} t2
                join-col (get-in t1 [:relationship :column])
                many-rel? (when join-col
                            (= t2-name (namespace join-col)))
                t1-join-col (if many-rel?
                              t1-id
                              (name join-col))
                t2-join-col (if many-rel?
                              (name join-col)
                              t2-id)
                t1-alias (when (pos? idx)
                           (str t1-name (dec idx)))
                t2-alias (str t2-name idx)]
            {:base {:name t1-name
                    :alias t1-alias
                    :join-col t1-join-col}
             :target {:name t2-name
                      :alias t2-alias
                      :join-col t2-join-col}})))))

(defn path-query
  "Takes a path as returned by `table-paths`, and a collection of
   conditions, and returns a query to retrieve ids from the base table
   in the path.

   Each condition is a map of:

   `{:column :table/col
     :comparator \"comparator-sym\"
     :value some-val}`"
  [path conditions]
  (let [{:keys [id name] :as base-table} (first path)
        base-query (format "SELECT %s AS id FROM %s"
                           (str name "." id)
                           name)
        joins? (> (count path) 1)
        jp (when joins? (join-path path))
        target-table (if joins?
                       (:target (last jp))
                       {:name name
                        :alias name})
        applicable-conds (filter #(= (:name target-table)
                                     (namespace (:column %)))
                                 conditions)]
    (str base-query
         (when joins?
           (str " "
                (->> jp
                     (map sql/inner-join-sql)
                     (str/join " "))))
         (when (seq applicable-conds)
           (str " WHERE "
                (->> applicable-conds
                         (map #(sql/condition-sql % (:alias target-table)))
                         (str/join " OR ")))))))

(defn expiry-condition
  "Returns condition map for an expiry."
  [expiry]
  (let [{:keys [field age]} expiry
        ;; picking an arbitrary cutoff number to distinguish #
        ;; of days from millisecond timestamp
        cutoff (if (and (number? age) (< age 100000))
                 (-> (time/now)
                     (time/to-time-zone (time/default-time-zone))
                     (time/minus (time/days age))
                     (time/with-time-at-start-of-day))
                 age)]
    {:column field
     :comparator ">"
     :value (sql/mysql-date-string (tc/to-date-time cutoff))}))

;; TODO: push this time stuff to the client

(defn delta-condition
  "Returns condition map for a delta field."
  [field date]
  {:column field
   :comparator ">"
   :value (sql/mysql-date-string (tc/to-date-time date))})

(defn condition-table
  "Returns the table used in a condition's column."
  [condition]
  (namespace (:column condition)))

(defn all-ids-query
  "Returns a query that will select all ids in the base table of
   spec."
  [spec]
  (let [{:keys [id name]} spec]
    (format "SELECT %s AS id FROM %s"
            id name)))

(defn update-table-id-query
  "Takes a map describing update table, and returns a query to
   retrieve ids to update. Update map has keys:

     * :table     Name of the update table
     * :id        Name of column with entity ids
     * :updated   Name of column with entity update datetimes.
     * :since     Datetime to return updates since."
  [{:keys [table id updated since] :as update-map}]
  (format "SELECT %s FROM %s WHERE %s"
          id table
          (sql/condition-sql {:column (keyword table updated)
                              :comparator ">"
                              :value (sql/mysql-date-string (tc/to-date-time since))})))

(defn id-query
  "Takes parsed spec (not grouped) and returns query to retrieve the
   ids, controlled by opts map."
  [spec {:keys [limit offset delta expiry] :as opts}]
  (let [tables (table-paths spec)
        conds (cond->
                  (when delta
                    (map #(delta-condition % (:date delta))
                         (:fields delta)))
                expiry (conj (expiry-condition expiry)))
        cond-tables (set (map condition-table conds))
        paths (mapcat (partial get tables) cond-tables)
        queries (map #(path-query % conds) paths)]
    (if (seq queries)
      (str/join " UNION " queries)
      (all-ids-query spec))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; main API
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;;; How this works

;; - queries for all the ids to fetch as part of the import
;;   - this query uses all the conditions/options passed in
;;   - joins in the full depth of the data spec, so we can have delta
;;     and expiry fields anywhere in the entity
;; - returns a lazy seq, which will fetch the ids in batches of
;;   batch option or `default-batch-size`

;; TODO: take db
(defn import-table
  "Takes an analyzed DSL spec with a table at the top level, and opts
   map, and performs the queries to fetch the table data and its
   associated relationships. opts map has keys:

    * :limit   Number of records to retrieve [unlimited]
    * :offset  Starting record [0]
    * :delta   Turns the import into a deltas import. A map with
               parameters for retrieving deltas.
    * :update-table  Also turns into ino a deltas import. Preferred
               over :delta, if present. A map describing the table
               that contains entity ids to update.
    * :expiry  Spec for filter for records older than a specified
               number of days.
    * :batch   Batch size.

   If present, the delta map should have keys:
    * :fields  Collection of :table/column (datetime) fields that will
               be used for update detection.
    * :date    Date to be used for comparison. Anything that clj-time can
               coerce to DateTime.

   If present, the update-table map should have keys:
    * :table     Name of the update table
    * :id        Name of column with entity ids
    * :updated   Name of column with entity update datetimes.
    * :since     Datetime to return updates since.

   If present, the expiry map should have keys:
    * :field   The :table/column (datetime) field that will be used to
               determine record age.
    * :age     DateTime of the cutoff.

   The resulting records have keys in the form of :table/column for
   each group, and many-relationships
   in [:many-relationships :foreign-table/column] keys.

   Returns a lazy-seq of results."
  ([db table-spec]
   (import-table db table-spec {}))
  ([db table-spec {:keys [limit offset delta expiry batch update-table]
                   :or {batch default-batch-size} :as opts}]
   (let [{:keys [grouped spec]} table-spec
         col-aliases (sql/group-column-mappings grouped)
         id-col (table-id grouped)
         ids-q (if update-table
                 (update-table-id-query update-table)
                 (id-query spec opts))
         _ (log/debug :fn "import-table"
                      :msg "Fetching ids"
                      :query ids-q)
         ids (fetch-column db ids-q "id")
         cnt (count ids)
         _ (log/info :fn "import-table"
                     :msg (str "Fetched " cnt " ids"))
         data-query (str "SELECT " (sql/aliased-fields-statement col-aliases)
                         " " (sql/from-statement grouped false))
         many-rels (get-in grouped [:relationships :many])]
     (with-meta
       ((fn next-batch
          ([batches]
           (next-batch batches 0))
          ([batches position]
           (let [cur (first batches)]
             (when (seq cur)
               (lazy-seq
                (let [cur-query (str data-query " WHERE "
                                     (sql/in-statement  {:field id-col
                                                         :vals cur}))
                      _ (log/debug :fn "import-table"
                                   :msg "Fetching next chunk of ids"
                                   :query cur-query
                                   :position position
                                   :total cnt)
                      table-data (fetch-results db [cur-query col-aliases])
                      entity-data
                      (reduce (fn [data rel]
                                (if-let [rel-data (import-many-relationship db rel data)]
                                  (join-results rel data rel-data)
                                  data))
                              table-data
                              many-rels)]
                  (concat entity-data (next-batch (rest batches)
                                                  (+ position batch)))))))))
        (partition-all batch ids))
       {:total cnt}))))

(defn import-record
  "Like import-table, but returns single record from given db."
  [db table-spec entid]
  (let [{:keys [spec grouped]} table-spec
        col-aliases (sql/group-column-mappings grouped)
        id-col (table-id grouped)
        ent-query (str "SELECT " (sql/aliased-fields-statement col-aliases)
                       " " (sql/from-statement grouped false)
                       " WHERE " (sql/condition-sql {:column id-col
                                                     :value entid}))
        many-rels (get-in grouped [:relationships :many])
        ent-data (fetch-results db [ent-query col-aliases])]
    (first
     (reduce (fn [data rel]
               (if-let [rel-data (import-many-relationship db rel data)]
                 (join-results rel data rel-data)
                 data))
             ent-data
             many-rels))))
