(ns jdbc.util
  (:require [honeysql.helpers :as helpers]))

(defn- criteria-dispatch [_ c]
  (class c))

(defmulti criteria #'criteria-dispatch)

(defmethod criteria clojure.lang.APersistentMap [m c]
  (reduce criteria m c))

(defmethod criteria clojure.lang.APersistentVector [m c]
  (if (= (count c) 2)
    (helpers/merge-where m (into [:=] c))
    (helpers/merge-where m c)))

(defmethod criteria :default [_ c]
  (throw (IllegalStateException. "Ill formed criteria:  [%s]" c)))

(defn query-map
  "build a honeysql map to select * from table reducing criteria to a where clause"
  [table selection & c]
  (reduce criteria
          (-> (helpers/select selection)
              (helpers/from   table))
          c))

(defn select-map
  "build a honeysql map to select * from table reducing criteria to a where clause"
  [table & c]
  (apply query-map table :* c))

(defn exists-map
  "build a honeysql map to select * from table reducing criteria to a where clause"
  [table & c]
  (apply query-map table [:%count.* :n] c))

(defn update-map
  "build a honeysql map to update table setting m values reducing criteria to a where clause"
  [table m & c]
  (reduce criteria
          (-> (helpers/update table)
              (helpers/sset   m))
          c))

(defn delete-map
  "build a honeysql map to delete from table reducing criteria to a where clause"
  [table & c]
  (reduce criteria (helpers/delete-from table) c))
