
(ns pinkgorilla.notebook-ui.settings.events
  "events related to the settings dialog"
  (:require
   [taoensso.timbre :refer-macros [info]]
   [cljs.reader :as rd]
   [re-frame.core :refer [reg-event-db reg-event-fx dispatch reg-sub]]
   [pinkgorilla.notebook-ui.settings.localstorage :refer [ls-get ls-set!]]
   [pinkgorilla.notebook-ui.settings.default :refer [default-settings]]
   [pinkgorilla.notebook-ui.settings.component :refer [settings-dialog]]))

; init

(reg-event-db
 :settings/init
 (fn [db [_]]
   (info "Notebook Settings: Loading from Localstorage ..")
   (let [db (or db {})
         settings-ls (->> (ls-get :notebook-settings)
                          (merge default-settings)) ; localstorage config might lack some keys (version update)
         settings (or settings-ls default-settings)]
     (if (nil? settings-ls)
       (info "Notebook Settings: localstorage empty! Using default settings")
       (info "Notebook Settings: successfully loaded from localstorage: " (keys settings)))
     (assoc-in db [:settings] settings))))

; subscriptions

(reg-sub
 :settings
 (fn [db _]
   (get-in db [:settings])))

; dialog visibility

(reg-event-db
 :settings/show
 (fn [db [_]]
   (info "showing settings dialog")
   (dispatch [:modal/open [settings-dialog] :medium [:settings/localstorage-save]])
   db))

;; Settings Change

(reg-event-db
 :settings/localstorage-save
 (fn [db [_]]
   (let [settings (:settings db)]
     (info "saving settings to localstorage: " settings)
     (ls-set! :notebook-settings settings)
     db)))

(reg-event-db
 :settings/set
 (fn [db [_ k v]]
   (info "changing setting " k " to: " v)
   (dispatch [:settings/localstorage-save])
   (assoc-in db [:settings k] v)))

;; secrets

(reg-event-fx
 :set-clj-secrets
 (fn [{:keys [db]}]
   (let [secrets (get-in db [:settings :secrets])
         secrets (if (nil? secrets) {} secrets)]
     (info "setting clj repl secrets..")
     ;(clj-eval-ignore-result "pinkgorilla.notebook.secret/set-secrets!" secrets)
     {})))

(reg-event-db
 :secret/add
 (fn [db [_ s]]
   (info "adding secret:" (:name s))
   (dispatch [:set-clj-secrets]) ; push new secrets to clj
   (assoc-in db [:settings :secrets (keyword (:name s))] (:secret s))))

(reg-event-db
 :secret/remove
 (fn [db [_ n]]
   (info "removing secret:" n)
   (dispatch [:set-clj-secrets]) ; push new secrets to clj
   (update-in db [:settings :secrets] dissoc n)))

(reg-event-db
 :secret/import
 (fn [db [_ s]]
   (info "importing secrets:" s)
   (dispatch [:set-clj-secrets]) ; push new secrets to clj
   (assoc-in db [:settings :secrets] (cljs.reader/read-string s))))
