(ns pinkgorilla.notebook-ui.notebook.segment
  (:require
   [taoensso.timbre :as timbre :refer [debug info warn error]]
   [re-frame.core :refer [subscribe dispatch]]
   [pinkgorilla.notebook-ui.eval-result.eval-result :refer [eval-result-view]]
   [pinkgorilla.notebook-ui.eval-result.code-viewer :refer [code-view]]
   [pinkgorilla.notebook-ui.eval-result.code :refer [code-edit]]
   [pinkgorilla.notebook-ui.notebook.codemirror :refer [codemirror-edit]]
   [pinkgorilla.notebook-ui.notebook.md :refer [md-segment-edit md-segment-view]]
   [pinkgorilla.notebook-ui.completion.component :refer [completion-component]]
   [pinkgorilla.notebook-ui.notebook.segment-nav :refer [segment-nav]]))

;; input

(defn segment-input-md [{:keys [active? md-editor] :as opt} eval-result]
  (when active?
    (case md-editor
      :codemirror [codemirror-edit eval-result opt]
      :text [md-segment-edit eval-result]
      [md-segment-edit eval-result])
    ;[md-segment-view eval-result]
    ))

(defn segment-input-code [{:keys [active? code-editor code-viewer] :as opt} eval-result]
  [:div {:style {:position "relative"}} ; container for kernel absolute position
   ; kernel - one color for both dark and light themes.
   (when-let [kernel (:kernel eval-result)] ; snippets might not define kernel
     [:span.pr-2.text-right.text-blue-600.tracking-wide.font-bold.border.border-blue-300.rounded
      {:on-click #(dispatch [:segment/kernel-toggle])
       :style {:position "absolute"
               :z-index 200 ; dialog is 1040 (we have to be lower)
               :top "2px" ; not too big, so it works for single-row code segments
               :right "10px"
               :width "50px"
               :height "22px"}} kernel])
   (if active?
     (case code-editor
       :codemirror  [codemirror-edit eval-result opt]
       :text        [code-edit eval-result]
       [code-edit eval-result])
     (case code-viewer
       :codemirror   [codemirror-edit eval-result opt]
       :highlight-js [code-view eval-result]
       [code-view eval-result]))])

(defn segment-input [{:keys [id code md] :as eval-result}]
  (let [settings (subscribe [:settings])
        segment-active (subscribe [:notebook/segment-active])
        segment-queued (subscribe [:segment/queued? id])
        edit? (subscribe [:notebook/edit?])]
    (fn [{:keys [id code md] :as eval-result}]
      (let [{:keys [code-editor code-viewer md-editor]
             :or {code-editor :codemirror
                  code-viewer :highlight-js
                  md-editor :text}}  @settings
            active? (= (:id @segment-active) id)
            edit? @edit?
            queued? @segment-queued
            options  {:active? active?
                      :edit? edit?
                      :md-editor md-editor
                      :code-editor code-editor
                      :code-viewer code-viewer}
            full? (= (:layout @settings) :single)]
        [:div.text-left.bg-gray-100 ; .border-solid
         {:id id
          :on-click #(do
                       (dispatch [:notebook/move :to id])
                       (dispatch [:notebook/set-edit? true]))
          :class (str (if queued?
                        "border border-solid border-blue-600"
                        (if active?
                          (if edit? "border border-solid border-red-600"
                              "border border-solid border-gray-600")
                          ""))
                      (if full? " h-full" ""))}

         (cond
           code [segment-input-code options eval-result]
           md   [segment-input-md options eval-result])]))))

;; output

(defn segment-output [{:keys [code md] :as eval-result}]
  [:div
   (when md
     [md-segment-view eval-result])
   (when code
     [eval-result-view eval-result])])

;; layout

(defn layout-vertical [eval-result]
  [:div.w-full ; .mt-5.p-2
   [segment-input eval-result]
   [segment-output eval-result]])

(defn layout-horizontal [eval-result]
  [:div.w-full.flex.flex-row ; .mt-5.p-2
   [:div {:class "w-1/2"}
    [segment-input eval-result]]
   [:div {:class "w-1/2"}
    [segment-output eval-result]]])

(defn layout-single [{:keys [id] :as eval-result}]
  [:div.flex.flex-row.w-full.h-full.min-h-full.bg-yellow-400.items-stretch
    ; LEFT: code-mirror / code-completion
   [:div.bg-orange-500.h-full.flex.flex-col {:class "w-1/2"}
    [segment-nav]
    [:div.w-full.bg-red-300.flex-grow
      ;{:style {:height "600px"}}
     [segment-input eval-result]]
    [:div.h-40.w-full.bg-teal-300
     [completion-component]]]
      ; RIGHT: error / console / result
   [:div.bg-blue-100.h-full.flex.flex-col
    {:class "w-1/2"
     :overflow-y "auto"}
    [:div.flex-grow  ; flex-grow scales the element to remaining space
     [segment-output eval-result]]]])

; segment

(defn segment-view [{:keys [id] :as eval-result}]
  (let [settings (subscribe [:settings])]
    (fn [{:keys [id] :as eval-result}]
      (let [layout (or (:layout @settings) :vertical)]
        (case layout
          :vertical   [layout-vertical eval-result]
          :horizontal [layout-horizontal eval-result]
          :single     [layout-single eval-result]
          [layout-vertical eval-result])))))




