(ns pinkgorilla.notebook-ui.codemirror.editor
  (:require
   [reagent.core :as r]
   [reagent.dom :as rd]
   ["codemirror" :as CodeMirror]

   ["codemirror/addon/edit/closebrackets"]
   ["codemirror/addon/edit/matchbrackets"]
   ["codemirror/addon/hint/show-hint"]
   ["codemirror/addon/runmode/runmode"]
   ["codemirror/addon/runmode/colorize"]

   ["codemirror/mode/clojure/clojure"]
   ["codemirror/mode/markdown/markdown"]
   ; [cljsjs.codemirror.mode.xml]
   ; [cljsjs.codemirror.mode.javascript]
   ; ["parinfer-codemirror"]
   ; [cljsjs.codemirror.mode.clojure-parinfer]

   ;["codemirror/keymap/vim"]
   [pinkgorilla.ui.config :refer [link-css]]

   [pinkgorilla.notebook-ui.codemirror.eval :refer [should-eval run-eval]]
   [pinkgorilla.notebook-ui.codemirror.completion
    :refer [remove-completions start-completions completion-show-one completion-cycle completion-show-all]]
   [pinkgorilla.notebook-ui.codemirror.highlight]
   [pinkgorilla.notebook-ui.codemirror.buffer :refer [value-atom on-change should-go-up should-go-down go-up go-down]]
   [pinkgorilla.notebook-ui.codemirror.parinfer :refer [on-cm-init]]
   [pinkgorilla.notebook-ui.codemirror.options :refer [settings cm-default-opts]]))

;; http://gcctech.org/csc/javascript/javascript_keycodes.htm

(def tab           9)
(def enter        13)
(def escape       27)

(def arrow-left   37)
(def arrow-up 	  38)
(def arrow-right  39)
(def arrow-down   40)

(def shift 	      16)
(def ctrl 	      17)
(def alt 	        18)
(def pause-break  19)
(def caps-lock 	  20)
(def window-left  91)
(def window-right 92)
(def select       93)

;; On Escape, should we revert to the pre-completion-text?
(def cancel-keys #{enter escape})
(def cmp-ignore #{tab shift ctrl alt window-left select})
(def cmp-show #{ctrl alt window-left select})

(defn code-mirror-impl
  "Create a code-mirror editor that knows a fair amount about being a good
  repl. The parameters:
  :style (reagent style map) will be applied to the container element"
  [{:keys [style cm-opts]}]
  (let [cm (atom nil)]
    (r/create-class
     {:component-did-mount
      (fn [this]
        (let [el (rd/dom-node this)
              inst (CodeMirror.
                    el
                    (clj->js
                     (merge
                      cm-default-opts
                      cm-opts
                      {:value @value-atom})))]

          (reset! cm inst)

          (.on inst "change"
               (fn []
                 (let [value (.getValue inst)]
                   (when-not (= value @value-atom)
                     (on-change value)))))

          (.on inst "keyup"
               (fn [inst evt]
                 (if (cancel-keys (.-keyCode evt))
                   (remove-completions inst evt)
                   (if (cmp-show (.-keyCode evt))
                     (completion-show-one inst evt)
                     (when-not (cmp-ignore (.-keyCode evt))
                       (start-completions inst evt))))))

          (.on inst "keydown"
               (fn [inst evt]
                 (case (.-keyCode evt)

                   (ctrl alt window-left select)
                   (completion-show-all inst evt)

                   tab
                   (completion-cycle inst evt)

                   enter
                   (let [source (.getValue inst)]
                     (when (should-eval source inst evt)
                       (.preventDefault evt)
                       (run-eval source)))

                   up
                   (let [source (.getValue inst)]
                     (when (and (not (.-shiftKey evt))
                                (should-go-up source inst))
                       (.preventDefault evt)
                       (go-up)))

                   down
                   (let [source (.getValue inst)]
                     (when (and (not (.-shiftKey evt))
                                (should-go-down source inst))
                       (.preventDefault evt)
                       (go-down)))

                   :none)))

          (when on-cm-init
            (on-cm-init inst))))

      :component-did-update
      (fn [this old-argv]
        (when-not (= @value-atom (.getValue @cm))
          (.setValue @cm @value-atom)
          (let [last-line (.lastLine @cm)
                last-ch (count (.getLine @cm last-line))]
            (.setCursor @cm last-line last-ch))))

      :reagent-render
      (fn [_ _ _]
        @value-atom
        [:div {:style style}])})))

(defn code-mirror [options]
  [:div ; if this is :<> then it can fuck up flexbox styling
   [link-css "codemirror/lib/codemirror.css"]
   [code-mirror-impl options]])

