(ns pinkgorilla.notebook-ui.codemirror.key-binding
  (:require
   [taoensso.timbre :refer-macros [debug info infof error]]
   [re-frame.core :refer [dispatch]]
   [keybind.core :as key]
   [pinkgorilla.notebook-ui.codemirror.extension :refer [run-extension run-extension-no-context]]
   [pinkgorilla.notebook-ui.codemirror.options :refer [code-mirror-key-down code-mirror-key-up]]))

(defn bind!
  "Binds a sequence of button presses, specified by `spec`, to `cb` when
  pressed. Keys must be unique per `spec`, and can be used to remove keybinding
  with `unbind!`.
  `spec` format is emacs-like strings a-la \"ctrl-c k\", \"meta-shift-k\", etc."
  [BINDINGS {:keys [scope key spec handler]}]
  (infof "code-mirror key: [%s] handler: %s" spec handler)
  (swap! BINDINGS
         key/bind spec key (fn [context]
                             (info "fire! context: " context)
                             (case scope
                               :global (dispatch handler)
                               :codemirror-no-context (run-extension-no-context context handler)
                               :codemirror (run-extension context handler)
                               (error "keybinding scope unknown: " scope))
                             #_(.preventDefault e) ; eventhandlers might chose to do nothing
                             )))

(defn init-keybindings! [BINDINGS keybindings]
  (info "binding... " (count keybindings))
  (doall (for [b keybindings]
           (bind! BINDINGS b)))
  (info "binding done."))

(def DOWN (atom {}))
(def UP (atom {}))

(info "codemirror-down")
(init-keybindings! DOWN code-mirror-key-down)

;(info "codemirror-up")
;(init-keybindings! UP code-mirror-key-up)

(defn my-e->chord [{:keys [e]}]
  ;(info "my-e->chord")
  (into {} (for [[key attr] key/KEYATTRS]
             [key (aget e attr)])))

(defn dispatch! [bindings {:keys [e] :as context}]
  (when (get key/KNOWN-KEYS (.-keyCode e))
    (info "dispatching key: " (.-keyCode e))
    (with-redefs [key/e->chord my-e->chord]
      (key/dispatch context @bindings))))

; on-key-down in codemirror gets partially applied with cm-opts
; code-mirror raises normal js events [sender evt]
(defn on-key-down [cm-opts cm evt]
  (let [context {:cm-opts cm-opts :cm cm :e evt}]
    (info "key-down!! " context "key: " (.-keyCode evt))
    (dispatch! DOWN context)))

(defn on-key-up [cm-opts cm evt]
  (let [context {:cm-opts cm-opts :cm cm :e evt}]
    (info "key-up!! " context "key: " (.-keyCode evt))
    (dispatch! UP context)))

(comment
;; http://gcctech.org/csc/javascript/javascript_keycodes.htm


  (def tab           9)
  (def enter        13)
  (def escape       27)

  (def arrow-left   37)
  (def arrow-up 	  38)
  (def arrow-right  39)
  (def arrow-down   40)

  (def shift 	      16)
  (def ctrl 	      17)
  (def alt 	      18)
  (def pause-break  19)
  (def caps-lock 	  20)
  (def window-left  91)
  (def window-right 92)
  (def select       93))

#_(defn on-key-down [cm-opts cm evt]
    (info "key-down" (.-keyCode evt))
    (case (.-keyCode evt)

      enter (when (and (not (.-shiftKey evt))
                       (.-metaKey evt))
              (when-let [source (.getValue cm)]
                (when (should-eval cm evt source)
                  (.preventDefault evt)
                  (run-eval source))))

      #'arrow-up

      ~arrow-down

      :none))

