# lein-git-version

This plugin allows you to manipulate the version used in lein projects to suit your needs.  The plugin is generally used to pull version information from git, but any shell command that grabs scm version information can be configured.

Additionaly, the plugin creates `version.clj` in your src tree containing a def for the version.  This allows you to programatically access the version information generated by this plugin.

I recommed you add `version.clj` to your `.gitignore` file due to the transient nature of auto-generated code and version numbers.

## Usage

Put `[org.ozias.plugins/lein-git-version "1.0.2"]` into the `:plugins` vector of your `:dev` profile in your project.clj.

```clojure
(defproject some.cool/proj "0.1.0-SNAPSHOT"
  ...
  :profiles {:dev {:plugins [...
                             [org.ozias.plugins/lein-git-version "1.0.2"]
                             ...]}}
  ...)
```

## Configuration
By default, with no configuration in the project file, the plugin will only generate the `version.clj` file with the current project version.  In order to customize the behavior of the plugin, use the `:git-version` key in `project.clj` as described below: 

The following are the defaults:

```clojure
(defproject nifty "blah blah" 
  ...
  :git-version {:append? false
                :version-command []
                :version-file-command []
                :assoc-in-keys [[:version]]
                :filepath "src/group-id/artifact-id"
                :filename "version.clj"}
  ...)
```

`append?` is used to tell the plugin to append the result of the version command to the original version or you use the result of the version command as the entire version.  The append happens in accordance with http://semver.org build metadata standards (i.e. 1.0.0+buildmeta1).

`version-command` is a vector representing the shell command used to grab the version information.

`version-file-command` is a vector representing the shell command used to grab the version information for the `version.clj` file.  Use this if you wish to put a different version in the file than elsewhere.  For example, I tend to append the short representation of the git SHA for my snapshot jar files (i.e. 0.1.0-SNAPSHOT+ab232fa).  In the file, I like to place the long (40-character) version of the git SHA.

`assoc-in-keys` is a vector of vectors use to place the new version string into the project map.  The default is placing the version string at the `:version` key in the map.  If you would like to include it elsewhere (in the `:manifest` key
for example) include additional entries as below:

```clojure
:assoc-in-keys [[:version]
                [:manifest "Implementation-Version"]
                [:env :version]]
```

`filepath` the path you wish to place `version.clj` at.  The default is to place the file at `src/group-id/artifact-id/version.clj`.  If you have a different directory structure, use this to control where the version file is generated.

`filename` is the name of the file that is created that allows you to access
the version string programatically.

## Examples
These are my most common use cases:

### Snapshots
*project.clj*

Note I've used the `:filepath` key here because my plugin source is not in the standard location due to the manner in which lein loads plugins.  Normally `version.clj` would be created at `src/org/ozias/plugins/` in this case.

```clojure
(defproject org.ozias.plugins/lein-git-version "1.0.2-SNAPSHOT"
  ...
  :profiles {:dev {:plugins [[org.ozias.plugins/lein-git-version "1.0.2"]]}}
  ...
  :manifest {"Implementation-Version" "1.0.2-SNAPSHOT"}
  :git-version {:append? true
                :version-command ["git" "rev-parse" "--short" "HEAD"]
                :version-file-command ["git" "rev-parse" "HEAD"]
                :filepath "lein_git_version"
                :assoc-in-keys [[:version] [:manifest "Implementation-Version"]]}
  ...)
```

*Output*

```
$ lein git-version
1.0.2-SNAPSHOT+90a29f8
$ lein install
Created lein-git-version/target/lein-git-version-1.0.2-SNAPSHOT+90a29f8.jar
Wrote lein-git-version/pom.xml
$ cat src/lein_git_version/version.clj
;; Do not edit.  Generated by lein-git-version plugin.
(ns lein_git_version.version)

(def version "1.0.2-SNAPSHOT+90a29f846ef5206d851f395f744b5563324a2fc3")
```

### Releases
*project.clj*

Note that I name my release branches after the version, so `git rev-parse --abbrev-ref HEAD` gives me the version number I want.

```clojure
(defproject org.ozias.plugins/lein-git-version "1.0.2"
  ...
  :profiles {:dev {:plugins [[org.ozias.plugins/lein-git-version "1.0.2"]]}}
  ...
  :manifest {"Implementation-Version" "1.0.2"}
  :git-version {:version-command ["git" "rev-parse" "--abbrev-ref" "HEAD"]
                :version-file-command ["git" "rev-parse" "HEAD"]
                :filepath "lein_git_version"
                :assoc-in-keys [[:version] [:manifest "Implementation-Version"]]}
  ...)
```

*Output*

```
$ lein git-version
1.0.2
$ lein install
Created lein-git-version/target/lein-git-version-1.0.2.jar
Wrote lein-git-version/pom.xml
$ cat src/lein_git_version/version.clj
;; Do not edit.  Generated by lein-git-version plugin.
(ns lein_git_version.version)

(def version "90a29f846ef5206d851f395f744b5563324a2fc3")
```

## License

Copyright © 2012 Colin Steele

Distributed under the Eclipse Public License, the same as Clojure.
