(ns leiningen.archaic.utils.repo
  (:require [clojure.java.io :as io :only (writer)]
            [clojure.string :as str :only (join replace)]
            [clojure.tools.reader.edn :as r :only (read-string)]
            [leiningen.core.project :as proj :only (defaults)]
            [leiningen.core.user :as uu :only [profile-auth resolve-credentials]]
            [org.ozias.cljlibs.logging.logging :refer (infoc warnc)]
            [org.ozias.cljlibs.mvnmeta.mvnmeta :as mvnmeta :only (artifact-metadata)]
            (org.ozias.cljlibs.mvnmeta.protocol [file]
                                                [httpx])
            [org.ozias.cljlibs.semver.semver :as sv]
            [rewrite-clj.parser :as parser]
            [rewrite-clj.printer :as prn]
            [rewrite-clj.zip :as z]))

(def default-repositories [["clojars" {:url "http://clojars.org/repo"}]
                           ["central" {:url "http://repo1.maven.org/maven2"}]
                           ["local" {:url (str "file://" (System/getProperty "user.home") "/.m2/repository")}]])

(defn- merge-credentials [[name settings]]
  (let [settings (if (string? settings)
                  (assoc {} :url settings)
                  settings)]
    (->> settings uu/profile-auth uu/resolve-credentials)))

(defn resolve-repos [project-map]
  (for [repo (:repositories project-map default-repositories)]
    (merge-credentials repo)))

(defn- snapshot? [version]
  (re-matches #".*-SNAPSHOT(\+[A-Za-z0-9.-]+)?" version))

(defn- qualified? [version]
  (and (not (snapshot? version)) (re-matches #".*-\w+" version)))

(defn- fetch-artifact-meta [{:keys [url]} artifact {:keys [release snapshot qualified any]}]
  (let [aid (str/trim (clojure.core/name artifact))
        gid (or (namespace artifact) aid)
        artmap {:url url :group-id gid :artifact-id aid}
        artmap (if (.startsWith url "file://")
                 (assoc artmap :filename "maven-metadata-local.xml") artmap)
        metamap (mvnmeta/artifact-metadata artmap)
        metadata ((keyword (str/join "/" [gid aid])) ((keyword url) metamap))]
    (if-not (contains? metadata :exception)
      (let [rel (:release metadata)
            versions (:versions metadata)
            latest (:latest metadata)
            lud (:lastUpdated metadata)]
        (cond
         any (into rel (filter sv/semver? versions))
         snapshot (into rel (filter #(and (sv/semver? %) (not (qualified? %))) versions))
         qualified (into rel (filter #(and (sv/semver? %) (not (snapshot? %))) versions))
         release rel)))))

(def f-a-m-memo (memoize fetch-artifact-meta))

(defn- prompt [& msg]
  (loop [i 3]
    (when (pos? i)
      (warnc :green (apply str msg) :reset " " :green :bold "[y/n]" :green "?")
      (print "    ")
      (.flush *out*)
      (let [r (or (read-line) "")
            r (.toLowerCase r)]
        (condp = r
          "y" true
          "n" false
          (recur (dec i)))))))

(defn- change-artifact-version [artvec artifact newversion]
  (->> (for [[k v] artvec]
         (if (= k artifact)
           [k newversion]
           [k v]))
       (into [])))

(defn- set-version-token [zip ks newver]
  (-> (reduce #(z/find-value %1 z/next %2) zip ks) z/next (z/replace newver)))

(defn- upgrade-in-file [print file ks kw k highest]
  (let [root (z/of-string (slurp file))
        profmap (-> root z/down)
        ks (conj ks k)
        c (set-version-token profmap ks highest)]
    (warnc :green (str "    Upgrading " k " to " highest " in " file))
    (if print
      (z/print-root c)
      (binding [*out* (io/writer file)]
        (z/print-root c)
        (.flush *out*)))))

(defn fetch-artifacts [file repos {:keys [force upgrade print] :as options}]
  (fn [artvec kw ks]
    (let [vecname (clojure.core/name kw)]
      (infoc :cyan (str "  " vecname ":"))
      (if (seq artvec)
        (for [[k v] artvec]
          (if-not (and (:no-clojure options)(= (str k) "org.clojure/clojure"))
            (do
              (infoc :green (str "    " k))
              (infoc :green "      checking:" :reset " " :magenta (str v))
              (let [res (->> (for [repo repos]
                               (f-a-m-memo repo k options))
                             (remove nil?)
                             (reduce into)
                             set
                             (sort sv/compare-versions)
                             reverse)]
                (infoc :green "      against:"
                       :reset "  "
                       :magenta (str/join " " res))
                (let [highest (first res)
                      cmp (sv/compare-versions (str v) highest)]
                  (when (> 0 cmp)
                    (warnc :green "    ["
                           :green :bold (str k " ")
                           :cyan :bold (str "\"" highest "\"")
                           :green "] is available but you are using "
                           :yellow :bold (str "\"" v "\""))
                    (cond (and upgrade force)
                          (upgrade-in-file print file ks kw k highest)
                          upgrade (if (prompt "    Do you wish to upgrade this dependency")
                                    (upgrade-in-file print file ks kw k highest)))
                    [k v highest]))))))
        (infoc :yellow (str "    no " vecname " found"))))))
