(ns leiningen.archaic.proj
  (:require [clojure.java.io :as io :only (file)]
            [clojure.string :as str :only (join)]
            (leiningen.archaic.utils [io :refer :all]
                                     [repo :refer (fetch-artifacts resolve-repos)])
            [leiningen.core.main :as main :only (abort)]
            [leiningen.core.project :as p :only (unmerge-profiles)]
            [org.ozias.cljlibs.logging.logging :refer (infoc tracec warnc)]))

(defn- projectclj? [path]
  (and path (.isFile path) (= (.getName path) "project.clj")))

(defn- parse-project
  [file project {:keys [options]}]
  (warnc :green "Checking project:" :reset " " :green (str file))
  (let [repos (resolve-repos project)
        fetcher (fetch-artifacts file repos options)
        {:keys [plugins dependencies]} (p/unmerge-profiles project [:system :user :provieded :dev])]
    (infoc :green "project:" :reset " " :yellow (:name project))
    (tracec :green "repositories:"
            :reset " "
            :yellow (str/join ", " (for [repo repos] (:url repo))))
    (if (or (:all options)(:plugins options))
      (if-let [pres (filter (complement nil?) (fetcher plugins :plugins [:plugins]))]
        (if (empty? pres)
          (infoc :green"    All plugin versions current")
          (doall pres))))
    (if (or (:all options)(:dependencies options))
      (if-let [dres (filter (complement nil?) (fetcher dependencies :dependencies [:dependencies]))]
        (if (empty? dres)
          (infoc :green "    All dependency versions current")
          (dorun dres))))
    (infoc "")))

(defn- parse-project-paths [paths project opts]
  (cond
   (seq? paths) (doall (for [path paths]
                         (parse-project path (read-project-map path opts) opts)))
   paths (parse-project paths project opts)
   :else (-> (->> ["Invalid path: No 'project.clj' files found!"
                   "Please check your supplied path(s) or supply the recursive flag."]
                  (str/join "\n"))
             (arcabort opts))))

(defn- find-projectclj [path]
  (let [path (io/file path "project.clj")]
    (if (projectclj? path) path)))

(defn- find-all-projectclj [path]
  (let [dirs (filter directory? (file-seq-nh path))]
    (->> (for [dir dirs]
              (if-let [file (find-projectclj dir)]
                file))
         (remove nil?))))

(defn- args-paths [arguments]
  (->> (for [arg arguments]
         (if-let [file (exists? arg)]
           file))
       (remove nil?)))

(defn- check-paths [project {{:keys [recursive]} :options :as opts}]
  (fn [paths]
    (->> (for [path paths]
           (if-let [path (exists? path)]
             (-> (cond (projectclj? path) path
                       (.isDirectory path) (if recursive
                                             (find-all-projectclj path)
                                             (find-projectclj path))
                       :else (-> "Can only check 'project.clj' files!"
                                 (arcabort opts)))
                 (parse-project-paths project opts))
             (-> "File or directory does not exist!"
                 (arcabort opts))))
         (remove nil?))))

(defn run-task [{:keys [root] :as project} {:keys [arguments] :as opts}]
  (let [paths (args-paths arguments)
        checkp (check-paths project opts)]
    (or (seq (checkp paths)) (seq (checkp (list (or root ".")))))))
