(ns leiningen.archaic.profiles
  (:require [clojure.string :as str :only (join)]
            (leiningen.archaic.utils [io :refer :all]
                                     [repo :refer (fetch-artifacts resolve-repos)])
            [org.ozias.cljlibs.logging.logging :refer (infoc tracec warnc)]))

(def profile-paths
  [(str (System/getProperty "user.home") "/.lein/profiles.clj")
   (str (System/getProperty "user.home") "/.lein/profiles.d")
   "/etc/leiningen/profiles.clj"])

(defn- parse-profile [{:keys [plugins dependencies] :as profmap} profile {:keys [options]}]
  (let [repos (resolve-repos profmap)
        fetcher (fetch-artifacts repos options)]
    (infoc :green "profile:" :reset " " :yellow profile)
    (tracec :green "repositories:"
            :reset " "
            :yellow (str/join ", " (for [repo repos] (:url repo))))
    (if (or (:all options)(:plugins options))
      (if-let [pres (filter (complement nil?) (fetcher plugins :plugins))]
        (if (empty? pres)
          (infoc :green"    All plugin versions current")
          (doall pres))))
    (if (or (:all options)(:dependencies options))
      (if-let [dres (filter (complement nil?) (fetcher dependencies :dependencies))]
        (if (empty? dres)
          (infoc :green "    All dependency versions current")
          (dorun dres))))))

(defn- parse-profiles [file args]
  (warnc :green (str "Checking profile(s) (" file ")"))
  (doseq [[profile profmap] (read-profiles-map file args)]
    (parse-profile profmap (name profile) args))
  (infoc ""))

(defn- parse-profiled [file args]
  (warnc :green (str "Checking profile ("  file ")"))
  (-> (read-profiles-map file args)
      (parse-profile (apply str (take-while #(not (= \. %)) (.getName file))) args))
  (infoc ""))

(defn- parse-project-profiles [profiles args]
  (warnc :green "Checking profile(s) (project.clj)")
  (doseq [profile profiles]
    (parse-profile (val profile) (clojure.core/name (key profile)) args))
  (infoc ""))

(defn- check-path [path args]
  (if-let [file (exists? path)]
    (cond (.isFile file) (parse-profiles file args)
          (.isDirectory file) (doseq [file (filter (comp not directory?) (file-seq-nh file))]
                                (parse-profiled file args)))
    (infoc :yellow (str "No profile file found at " path "\n"))))

(defn run-task [{:keys [root profiles]} args]
  (let [profile-paths (if (and root (exists? root))
                        (conj profile-paths (str root "/profiles.clj"))
                        profile-paths)]
    (doseq [path profile-paths] (check-path path args))
    (parse-project-profiles profiles args)))

