(ns org.ozias.cljlibs.utils.core
  (:require [clojure.java.io :refer [as-file]]
            [clojure.string :refer [join trim]]
            [environ.core :refer :all]
            [me.raynes.conch :refer [with-programs]])
  (:import (java.io File)
           (java.util.regex Matcher Pattern)))

(defn os?
  "true if the os.name environment variable
  starts with the given name"
  [name]
  (.startsWith ^String (env :os-name) name))

(defn linux?
  "true if the os.name environment variable
  starts with 'Linux'"
  []
  (os? "Linux"))

(defn mac?
  "true if the os.name environment variable
  starts with 'Mac'"
  []
  (os? "Mac"))

(defn windows?
  "true if the os.name environment variable
  starts with 'Windows'"
  []
  (os? "Windows"))

(defn localhost?
  "Is the given target localhost?

  Matches \"localhost\" or \"user@localhost\""
  [target]
  (re-matches #"(\w+@)?localhost$" target))

(defn tmpdir
  "Evaluates to the path for the directory to store temporary files"
  []
  (if (or (linux?) (mac?))
    (or (env :tmp) (env :temp) (env :tmpdir) "/tmp")
    (if (windows?)
      (env :tmp))))

(defn- seq->path
  "Evaluates to a path string properly separated by the given file separator.

  If you wish to prefix the path with a separator, place nil in the first
  position in the path sequence"
  [pathseq sep]
  (if (every? string? (rest pathseq))
    (join sep pathseq)))

(defn- to-path-dispatcher
  "Dispatcher for the to-path multimethod"
  [& dv]
  (let [fdv (first dv)]
    (if (map? fdv) (:os fdv) dv)))

(defmulti to-path "Convert a sequence to a path
  Accepts a map with the following key/value pairs:

  {:os - :windows, :linux, or :macosx,
   :ps - A sequence of strings to convert to a path}

  (to-path {:os :windows '(\"C:\" \"Users\" \"testuser\")}) =>
  \"C:\\Users\\testuser\"

  or variadic string arguments that will be separated by the system
  specific file separator (use nil in the first position to prefix
  the result with a separator)

  (to-path nil \"usr\" \"local\" \"bin\") => \"/usr/local/bin\"
  (to-path \"usr\" \"local\" \"bin\") => \"usr/local/bin\""
          to-path-dispatcher)

(defmethod to-path :windows [{:keys [ps]}]
  (seq->path ps "\\"))
(defmethod to-path :linux [{:keys [ps]}]
  (seq->path ps "/"))
(defmethod to-path :macosx [{:keys [ps]}]
  (seq->path ps "/"))
(defmethod to-path :default [& args]
  (seq->path args (env :file-separator)))

(defn- os-key-dispatcher
  "Used to generate a key based on os type to
  be used as dispatching function."
  []
  (cond
    (linux?) :linux
    (mac?) :mac
    (windows?) :windows))

(defn- nix-cpu-arch
  "check the cpu arch on *nix oses"
  []
  (with-programs [uname]
                 (let [arch (trim (uname "-m"))]
                   (if (= "x86_64" arch)
                     64
                     (if (= "i686" arch)
                       32)))))

(defmulti cpu-arch
          "evaluates to the number of bits for the current cpu architecture"
          os-key-dispatcher)

(defmethod cpu-arch :windows []
  (let [arch (env :processor-architecture)]
    (if (= "AMD64" arch)
      64
      (if (= "x86" arch)
        32))))

(defmethod cpu-arch :linux []
  (nix-cpu-arch))

(defmethod cpu-arch :max []
  (nix-cpu-arch))

(defn cpu-arch-64?
  "true if the cpu architecture is 64-bits"
  []
  (= 64 (cpu-arch)))

(defn cpu-arch-32?
  "true if the cpu architecture is 32-bits"
  []
  (= 32 (cpu-arch)))

(defn- nix-cores
  "Determine the number of cores on a *nix system"
  []
  (with-programs [nproc]
                 (-> (nproc) trim Integer/parseInt)))

(defmulti cores
          "evalutes to the number of processor cores"
          os-key-dispatcher)

(defmethod cores :windows []
  (Integer/parseInt (env :number-of-processors)))

(defmethod cores :linux []
  (nix-cores))

(defmethod cores :mac []
  (nix-cores))

(defn assoc-if
  "Same as assoc, but skip the assoc if v is nil"
  [m & kvs]
  (->> kvs (partition 2) (filter second) (map vec) (into m)))

(defn successful?
  "Was the shell command successful?

  Checks for an exit code of zero in a conch process map.

  Evaluates to the output if it is 0, nil otherwise."
  [omap]
  (zero? @(:exit-code omap)))

(defn prseq
  "println every element in a sequence"
  [s]
  (doseq [line s]
    (println line)))

(defn in?
  "true if seq contains elm"
  [seq elm]
  (some #(= elm %) seq))

(defn repeat-str
  "Repeat the given string n times"
  [s n]
  (->> (repeat s) (take n) (reduce str)))

(defn wrap
  "wrap the given string at n characters"
  ([s n]
   (mapv join (partition-all n s)))
  ([s]
   (wrap s 80)))

(defn exists?
  "does the given path exist"
  [path]
  (.exists ^File (as-file path)))

(defn fmapv
  "Map a function across the values in a map"
  [f m]
  (into {} (for [[k v] m] [k (f v)])))

(defn fmapk
  "Map a function across the keys in a map"
  [f m]
  (into {} (for [[k v] m] [(f k) v])))

(defn flatten-map
  "flatten a given map, generating new keys separated by the string sep
  and optionally prefixed by the string in pre.

  (flatten-map {:a {:b {:c 1}}} \"-\") => {:a-b-c 1}
  (flatten-map {:a {:b {:c 1}}} \"-\" \"pre\") => {:pre-a-b-c 1}"
  ([m sep]
   (into {} (flatten-map m sep nil)))
  ([m sep pre]
   (mapcat (fn [[k v]]
             (let [prefix (if pre (str pre sep (name k)) (name k))]
               (if (map? v)
                 (flatten-map v sep prefix)
                 [[(keyword prefix) v]])))
           m)))

(defn strpart
  "Splits the string into a lazy sequence of substrings, alternating
  between substrings that match the pattern and the substrings
  between the matches.  The sequence always starts with the substring
  before the first match, or an empty string if the beginning of the
  string matches.

  For example: (partition #\"[a-z]+\" \"abc123def\")
  returns: (\"\" \"abc\" \"123\" \"def\")"
  [^Pattern re ^String s]
  (let [m ^Matcher (re-matcher re s)]
    ((fn step [prevend]
       (lazy-seq
         (if (.find m)
           (cons (.subSequence s prevend (.start m))
                 (cons (re-groups m)
                       (step (+ (.start m) (count (.group m))))))
           (when (< prevend (.length s))
             (list (.subSequence s prevend (.length s)))))))
     0)))