(ns org.ozias.cljlibs.shell.shell
  (:require [me.raynes.conch :refer (let-programs)]))

(defn localhost? 
  "Is the given target localhost?
 
  Matches \"localhost\" or \"user@localhost\""
  [target]
  (re-matches #"(\w+@)?localhost$" target))

(defn successful?
  "Was the shell command successful?

  Checks for an exit code of zero in a conch process map.

  Evaluates to the output if it is 0, nil otherwise."
  [output]
  (if (= 0 @(:exit-code output)) output))

(defn cmd->sshcmd 
  "Given a target, evaluates to a function that takes a command vector 
  as an argument.

  If the target is not a localhost target, [\"ssh\" target] will be
  prepended to the command vector."
  [target]
  (if (localhost? target)
    (fn [cmd] cmd)
    (fn [cmd] (into ["ssh" target] cmd))))

(defn shell-cmd 
  "Execute the given shell command.

  cmdv should be a vector of strings representing the command you wish to execute.
  [\"ls\" \"-al\"]

  If basedir is supplied, the command will be run in that directory."
  [cmdv & {:keys [basedir outfn errfn]}]
  (if (and (seq cmdv)(vector? cmdv)(every? string? cmdv))
    (let [process-map {:seq true :verbose true}
          process-map (if (seq basedir) (assoc process-map :dir basedir) process-map)
          process-map (if (fn? outfn) (assoc process-map :out outfn) process-map)
          process-map (if (fn? errfn) (assoc process-map :err errfn) process-map)]
      (let-programs [cmd (first cmdv)]
                    (apply cmd (conj (subvec cmdv 1) process-map))))))
