(ns org.ozias.cljlibs.semver.semver)

(def ^:private semver-re
  #"(\d+\.\d+\.\d+)(-([A-Za-z0-9-.]+))?(\+([A-Za-z0-9-.]+))?")

(defn semver?
  "Checks if the given string is a valid semantic version (See http://semver.org).

  Evaluates to a vector of three values if the semantic version is valid:

  1. The major.minor.patch version (i.e. 1.0.0)
  2. The pre-release information (i.e alpha)
  3. The build metadata (i.e. 20140203111111)

  Evaluates to nil otherwise"
  [version]
  (if-let [match (and (seq version)(re-matches semver-re version))]
    (->> match
         (partition 2)
         (map last)
         (into []))))

(defn svvec->svmap
  "Convert a semver vector version to a map.
  \"1.0.0\" -> {:major 1 :minor 0 :patch 0}

  Evaluates to a vector
  [{:major 1 :minor 0 :patch 0} pre-release buildmeta]"
  [[mmp pr bm]]
  [(zipmap [:major :minor :patch] (clojure.string/split mmp #"\.")) pr bm])

(defn svvec->str
  "Convert a semver vector to a string."
  [[{:keys [major minor patch]} pr bm]]
  (str major "." minor "." patch 
       (if (seq pr) (str "-" pr))
       (if (seq bm) (str "+" bm))))

(defn strpart
  "Splits the string into a lazy sequence of substrings, alternating
  between substrings that match the patthern and the substrings
  between the matches.  The sequence always starts with the substring
  before the first match, or an empty string if the beginning of the
  string matches.

  For example: (partition #\"[a-z]+\" \"abc123def\")
  returns: (\"\" \"abc\" \"123\" \"def\")"
  [^java.util.regex.Pattern re ^String s]
  (let [m (re-matcher re s)]
    ((fn step [prevend]
       (lazy-seq
        (if (.find m)
          (cons (.subSequence s prevend (.start m))
                (cons (re-groups m)
                      (step (+ (.start m) (count (.group m))))))
          (when (< prevend (.length s))
            (list (.subSequence s prevend (.length s)))))))
     0)))

(defn- parse-int [s]
  (Integer/parseInt (re-find #"\A-?\d+" s)))

(defn- nil-fill [v l]
  (loop [vec v]
    (if (= (count vec) l)
      vec
      (recur (conj vec nil)))))

(defn- char->str [c]
  (if (instance? java.lang.Character c)
    (str c)
    c))

(defn- numeric? [s]
  (string? (re-matches #"\d+" (char->str s))))

(defn- alpha? [s]
  (string? (re-matches #"[A-Za-z]+" (char->str s))))

(defn- identifier? [s]
  (string? (re-matches #"[0-9A-Za-z-]+" s)))

(defn- compare-numeric [x y]
  (let [x (parse-int (char->str x))
        y (parse-int (char->str y))]
    (cond
     (> x y) 1
     (= x y) 0
     (< x y) -1)))

(defn- compare-char [x y]
  (cond
   (and (numeric? x)(numeric? y)) (compare-numeric x y)
   (and (alpha? x)(alpha? y)) (compare x y)))

(defn- compare-alphanumeric [x y]
  (let [xd (map #(.toLowerCase %) (strpart #"\d+" x))
        yd (map #(.toLowerCase %) (strpart #"\d+" y))
        res (first (filter (complement zero?) (map compare-char xd yd)))]
    (if (nil? res) 0 res)))

(defn- identifier-compare [x y]
  (cond
   ; A SNAPSHOT qualifier always has lower precedence
   (= x "SNAPSHOT") -1
   (= y "SNAPSHOT") 1
   ; A nil qualifier always has lower precedence
   ; i.e. A longer dotted qualifier with equal predecessors
   ; has greater precedence.
   (and (nil? x)(seq y)) -1
   (and (seq x)(nil? y)) 1
   ; Two numeric identifiers are compared numerically
   (and (numeric? x)(numeric? y)) (compare-numeric x y)
   ; Otherwise a numeric identifier has lower precedence
   ; than an alphanumeric identifier.
   (and (numeric? x)(identifier? y)) -1
   (and (identifier? x)(numeric? y)) 1
   ; Else compare the identifers alphanumerically
   :else (compare-alphanumeric x y)))

(defn- identifier-precedence
  "Split the identifiers on '.', fill each vector with
  nil to the max of both, and the compare each identifier.
  The equal identifiers are filtered out and the first non-zero
  result is grabbed.

  If the result is nil the identifiers have equal precedence (0).
  Otherwise the result is (< 0 res)."
  [x y]
  (let [xv (clojure.string/split x #"\.")
        xy (clojure.string/split y #"\.")
        m (max (count xv) (count xy))
        xv (nil-fill xv m)
        xy (nil-fill xy m)
        res (first (filter (complement zero?) (map identifier-compare xv xy)))]
    (if (nil? res) 0 res)))

(defn- nil-precedence
  "A nil qualifier has precedence over
  any non-empty qualifier (i.e. 1.0.0 > 1.0.0-alpha)"
  [x y]
  (cond
   (and (nil? x)(nil? y)) 0
   (and (nil? x)(seq y)) 1
   (and (seq x)(nil? y)) -1
   :else (identifier-precedence x y)))

; Compare major.minor.patch-<qual> versions.
; major minor and patch are compared numerically.
; qualifiers are compared more extensively if
; the comparison is tied after major.minor.patch.
; 
; 1.0.0 < 1.0.1 < 1.1.0 < 2.0.1
(defn compare-semver [x y]
  (let [svx (first x)
        svy (first y)
        majx (parse-int (:major svx))
        majy (parse-int (:major svy))
        minx (parse-int (:minor svx))
        miny (parse-int (:minor svy))
        px (parse-int (:patch svx))
        py (parse-int (:patch svy))]
    (if (= majx majy)
      (if (= minx miny)
        (if (= px py)
          (nil-precedence (second x) (second y))
          (compare px py))
        (compare minx miny))
      (compare majx majy))))
