(ns org.ozias.cljlibs.mvnmeta.mvnmeta
  (:require [clojure.string :as str]))

(defn- url->fullurl
  "turn a base url into a full url"
  [{:keys [url group-id artifact-id filename]}]
  (let [group-id (if (nil? group-id) artifact-id group-id)]
    (str/join "/" [(if (= (last url) \/) (str/join (butlast url)) url)
                   (if group-id (str/replace group-id "." "/") artifact-id)
                   artifact-id
                   (or filename "maven-metadata.xml")])))

(defn- parse-xml
  "Grab the content from xml at kw"
  [kw]
  (fn [xmlseq]
    (->> (for [element xmlseq]
           (if (= (:tag element) kw)
             (first (:content element))))
         (remove nil?)
         (into []))))

(defn- gidaid->kw
  "turns a group id, artifact id combo into a keyword"
  [{:keys [group-id artifact-id]}]
  (if (nil? artifact-id)
    (Exception. "Artifact ID cannot be nil")
    (if (nil? group-id)
      (keyword artifact-id)
      (keyword (str/join "/" [group-id artifact-id])))))

(defn- url->kw
  "turns a url into a keyword"
  [{:keys [url]}]
  (keyword url))

(defn fetch-artifact-metadata
  "Fecth artifact metadata"
  [fetchfn mvnmetamap]
    (let [meta (fetchfn (assoc mvnmetamap :url (url->fullurl mvnmetamap)))
          coord (gidaid->kw mvnmetamap)
          urlkw (url->kw mvnmetamap)
          get-versions (parse-xml :version)
          get-release (parse-xml :release)
          get-latest (parse-xml :latest)
          get-last-updated (parse-xml :lastUpdated)]
      (->> (if (instance? Exception meta)
             (assoc {} :exception meta)
             (assoc {}
               :versions (get-versions meta)
               :release (get-release meta)
               :latest (get-latest meta)
               :lastUpdated (get-last-updated meta)))
           (assoc {} (if (instance? Exception coord) (keyword "unk") coord))
           (assoc {} urlkw))))

(defn- add-value
  "add value from m2 into m1 if it exists"
  [m1 m2 k]
  (if-let [v (get m2 k)]
    (assoc m1 k v)
    m1))

(defn- protocol-dispatch-fn
  "protocol dispatch function for artifact-metadata defmulti"
  [{:keys [url artifacts]}]
  (if (seq artifacts)
    "multi"
    (if (seq url)
      (->> url
           (take-while #(not= \: %))
           (apply str)))))

(defmulti artifact-metadata
  "Dispatch on protocol string given a map with a :url key"
  protocol-dispatch-fn)

(defmethod artifact-metadata :default [mvnmetamap]
  (let [protocolstr (protocol-dispatch-fn mvnmetamap)]
    (throw (Exception. (str "Unknown protocol"
                            (if-not protocolstr "" (str ": " protocolstr)))))))

(defmethod artifact-metadata "multi"
  [{:keys [url artifacts username
           password passphrase filename type] :as mvnmetamap}]
  (reduce (partial merge-with merge)
   (for [artifact artifacts]
     (-> (assoc mvnmetamap :url (or (get artifact :url) url))
         (assoc :artifact-id (get artifact :artifact-id))
         (add-value artifact :group-id)
         (add-value artifact :username)
         (add-value artifact :password)
         (add-value artifact :passphrase)
         (add-value artifact :filename)
         (add-value artifact :type)
         (dissoc :artifacts)
         artifact-metadata))))
