(ns org.ozias.cljlibs.logging.logging
  (:require [colorize.core :as col]
            [clojure.string :as str]
            [taoensso.timbre :as timbre]))

(def ^:private nc (atom false))

(defn- fmt-output-fn
  [{:keys [level throwable message timestamp hostname ns]} & appender-fmt-output-opts]
  (format "%s %s %-6s [%s] - %s%s"
          timestamp hostname (-> level name str/upper-case) ns (or message "")
          (or (timbre/stacktrace throwable "\n") "")))

(defn seq->coloredstr [[tags strings]]
  ((->> tags (map #(partial col/color %)) (reduce comp))
   (->> strings (apply str))))

(defn colored [args]
  (if @nc
    (->> args (filter string?) (interpose " ") (apply str))
    (cond (some keyword? args)
          (let [grouped (->> args (partition-by string?) (partition 2))]
            (->> (for [group grouped] (seq->coloredstr group))
                 doall
                 (apply str)))
          (every? string? args) (apply str (interpose " " args)))))

(defn reportc [& message]
  (if (seq message)
    (let [fst (first message)]
      (if (instance? java.lang.Exception fst)
        (timbre/report fst (colored (rest message)))
        (timbre/report (colored message))))))

(defn fatalc [& message]
  (if (seq message)
    (let [fst (first message)]
      (if (instance? java.lang.Exception fst)
        (timbre/fatal fst (colored (rest message)))
        (timbre/fatal (colored message))))))

(defn errorc [& message]
  (if (seq message)
    (let [fst (first message)]
      (if (instance? java.lang.Exception fst)
        (timbre/error fst (colored (rest message)))
        (timbre/error (colored message))))))

(defn warnc [& message]
  (if (seq message)
    (let [fst (first message)]
      (if (instance? java.lang.Exception fst)
        (timbre/warn fst (colored (rest message)))
        (timbre/warn (colored message))))))

(defn infoc [& message]
  (if (seq message)
    (let [fst (first message)]
      (if (instance? java.lang.Exception fst)
        (timbre/info fst (colored (rest message)))
        (timbre/info (colored message))))))

(defn debugc [& message]
  (if (seq message)
    (let [fst (first message)]
      (if (instance? java.lang.Exception fst)
        (timbre/debug fst (colored (rest message)))
        (timbre/debug (colored message))))))

(defn tracec [& message]
  (if (seq message)
    (let [fst (first message)]
      (if (instance? java.lang.Exception fst)
        (timbre/trace fst (colored (rest message)))
        (timbre/trace (colored message))))))

(defn configure-logging
  "Configure timbre logging framework.

  The argument is a map of the format:

  {:options
   {... 
    :logfile \"/path/to/log\"  ;fully qualified path to logfile 
    :verbosity 0-3           ;output level: 0 - warn, 1 - info, 2 - debug, 3 - trace
    :no-color true/false     ;color support disabled/enabled
    :stdout true/false       ;log to stdout (false is default)
    :formatter formatterfn   ;formatting function (set timbre for example)
    ...}}

  All the keys are optional.  The default configuration will log info and above messages 
  to stdout. Note that if no logfile is supplied stdout true regardless of the map
  value.

  Evaluates to the passed in options map.  Useful for chaining with parse-opts from 
  tools.cli."
  [{{:keys [logfile verbosity no-color stdout formatter] 
     :or {verbosity 1 no-color false stdout false formatter fmt-output-fn}} :options :as options}]
  (reset! nc no-color)
  (let [stdout (if (nil? logfile) true stdout)
        options (if (contains? options :options)
                  (update-in options [:options] assoc 
                             :verbosity verbosity
                             :no-color no-color
                             :stdout stdout
                             :formatter formatter)
                  options)]
    (condp = verbosity
      0 (timbre/set-level! :warn)
      1 (timbre/set-level! :info)
      2 (timbre/set-level! :debug)
      3 (timbre/set-level! :trace))
    (timbre/set-config! [:fmt-output-fn] formatter)
    (timbre/set-config! [:appenders :standard-out :enabled?] stdout)
    (timbre/set-config! [:appenders :spit :enabled?] false)
    (timbre/set-config! [:shared-appender-config :spit-filename] logfile)
    (timbre/set-config! [:appenders :my-spit]
                        {:doc "Spit appender with newline"
                         :min-level nil
                         :enabled? true
                         :async? false
                         :rate-limit nil
                         :fn (fn [{:keys [ap-config output]}]
                               (when-let [filename (:spit-filename ap-config)]
                                 (try (spit filename (str output "\n") :append true)
                                      (catch java.io.IOException _))))})
    options))
