(ns org.ozias.cljlibs.logging.logging
  (:require [clojure.string :as str]
            [taoensso.timbre :as timbre]))

(defn- fmt-output-fn
  [{:keys [level throwable message timestamp hostname ns]}
   ;; Any extra appender-specific opts:
   & [{:keys [nofonts?] :as appender-fmt-output-opts}]]
  ;; <timestamp> <hostname> <LEVEL> [<ns>] - <message> <throwable>
  (format "%s %s %-5s [%s] - %s%s"
    timestamp hostname (-> level name str/upper-case) ns (or message "")
    (or (timbre/stacktrace throwable "\n" (when nofonts? {})) "")))

(defn configure-logging
  "Configure timbre logging framework.

  The argument is a map of the format:

  {:options
   {... 
    :logfile \"/path/to/log\"  ;fully qualified path to logfile 
    :verbosity 0-3           ;output level: 0 - warn, 1 - info, 2 - debug, 3 - trace
    :stdout true/false       ;log to stdout (false is default)
    :formatter formatterfn   ;formatting function (set timbre for example)
    ...}}

  All the keys are optional.  The default configuration will log info and above messages 
  to stdout. Note that if no logfile is supplied stdout true regardless of the map
  value.

  Evaluates to the passed in options map.  Useful for chaining with parse-opts from 
  tools.cli."
  [{{:keys [logfile verbosity stdout formatter] 
     :or {verbosity 1 stdout false formatter fmt-output-fn}} :options :as options}]
  (let [stdout (if (nil? logfile) true stdout)]
    (condp = verbosity
      0 (timbre/set-level! :warn)
      1 (timbre/set-level! :info)
      2 (timbre/set-level! :debug)
      3 (timbre/set-level! :trace))
    (timbre/set-config! [:fmt-output-fn] formatter)
    (timbre/set-config! [:appenders :standard-out :enabled?] stdout)
    (timbre/set-config! [:appenders :spit :enabled?] false)
    (timbre/set-config! [:shared-appender-config :spit-filename] logfile)
    (timbre/set-config! [:appenders :my-spit]
                        {:doc "Spit appender with newline"
                         :min-level nil
                         :enabled? true
                         :async? false
                         :rate-limit nil
                         :fn (fn [{:keys [ap-config output]}]
                               (when-let [filename (:spit-filename ap-config)]
                                 (try (spit filename (str output "\n") :append true)
                                      (catch java.io.IOException _))))}))
    options)
