(ns org.ozias.cljlibs.logging.logging
  (:require [clojure.string :as str]
            [colorize.core :as col]
            [org.ozias.cljlibs.utils.core :refer [windows?]]
            [taoensso.timbre :as timbre]))

(def ^{:private true :doc "no color atom"} nc (atom false))

(defn- fmt-output-fn
  "The default format output function"
  [{:keys [level throwable message timestamp hostname ns]}
   & appender-fmt-output-opts]
  (format "%s %s %-6s [%s] - %s%s"
          timestamp hostname (-> level name str/upper-case) ns (or message "")
          (or (timbre/stacktrace throwable "\n") "")))

(defn- seq->coloredstr
  "Convert a vector of tags and strings to a colored string"
  [[tags strings]]
  ((->> tags (map #(partial col/color %)) (reduce comp))
   (str/join strings)))

(defn- colored
  "color the args"
  [args]
  (if (or @nc (windows?))
    (->> args (filter (complement keyword?)) (apply str))
    (cond (some keyword? args)
          (let [grouped (->> args (partition-by (complement keyword?))
                             (partition 2))]
            (->> (for [group grouped] (seq->coloredstr group))
                 doall
                 (apply str)))
          :else (str/join args))))

(defn- logfn
  "base logging function"
  [tfn message]
  (if (seq message)
    (let [fst (first message)]
      (cond (keyword? fst) (tfn (colored message))
            (or (nil? fst) (empty? fst)) nil
            (instance? Exception fst) (tfn fst
                                           (colored
                                             (rest message)))
            :else (tfn (colored message))))))

(defn reportc
  "log a colored report message"
  [& message]
  (logfn #(timbre/report %) message))

(defn fatalc
  "log a colored fatal message"
  [& message]
  (logfn #(timbre/fatal %) message))

(defn errorc
  "log a colored error message"
  [& message]
  (logfn #(timbre/error %) message))

(defn warnc
  "log a colored warn message"
  [& message]
  (logfn #(timbre/warn %) message))

(defn infoc
  "log a colored info message"
  [& message]
  (logfn #(timbre/info %) message))

(defn debugc
  "log a colored debug message"
  [& message]
  (logfn #(timbre/debug %) message))

(defn tracec
  "log a colored trace message"
  [& message]
  (logfn #(timbre/trace %) message))

(defn configure-logging
  "Configure timbre logging framework.

  The argument is a map of the format:

  {:options
   {...
    :logfile \"/path/to/log\"  ;fully qualified path to logfile
    :verbosity 0-3           ;output level: 0 - warn, 1 - info,
                             ;2 - debug, 3 - trace
    :no-color true/false     ;color support disabled/enabled
    :stdout true/false       ;log to stdout (false is default)
    :formatter formatterfn   ;formatting function (set timbre for example)
    ...}}

  All the keys are optional.  The default configuration will log info and above
  messages to stdout. Note that if no logfile is supplied stdout true regardless
  of the map value.

  Evaluates to the passed in options map.  Useful for chaining with parse-opts
  from tools.cli."
  [{{:keys [logfile verbosity no-color stdout formatter]
     :or   {verbosity 1 no-color false stdout
                      false formatter fmt-output-fn}} :options :as options}]
  (reset! nc no-color)
  (let [stdout (if (nil? logfile) true stdout)
        options (if (contains? options :options)
                  (update-in options [:options] assoc
                             :verbosity verbosity
                             :no-color no-color
                             :stdout stdout
                             :formatter formatter)
                  options)]
    (condp = verbosity
      0 (timbre/set-level! :warn)
      1 (timbre/set-level! :info)
      2 (timbre/set-level! :debug)
      3 (timbre/set-level! :trace))
    (timbre/set-config! [:fmt-output-fn] formatter)
    (timbre/set-config! [:appenders :standard-out :enabled?] stdout)
    (timbre/set-config! [:appenders :spit :enabled?] false)
    (timbre/set-config! [:shared-appender-config :spit-filename] logfile)
    (timbre/set-config!
      [:appenders :my-spit]
      {:doc        "Spit appender with newline"
       :min-level  nil
       :enabled?   true
       :async?     false
       :rate-limit nil
       :fn         (fn [{:keys [ap-config output]}]
                     (when-let [filename (:spit-filename ap-config)]
                       (try
                         (spit filename
                               (str output "\n") :append true)
                         (catch java.io.IOException _))))})
    options))
