(ns org.ozias.cljlibs.bootstrap.bootstrap
  (:require [clojure.java.io :as io]
            [clojure.string :as str]
            (org.ozias.cljlibs.bootstrap [version :as ver]
                                         [wgetjava :as wj])
            [org.ozias.cljlibs.shell.shell :refer :all]
            [taoensso.timbre :refer (info)]))

(def ^:private jremap
  {:linux   {:i686   [:type :jre :arch :32]
             :x86_64 [:type :jre]}
   :macosx  {:x86_64 [:type :jre :os :mac]}})

(def ^:private teststrings
  {:isnotdir "! -d "
   :isnotfile "! -f "})

(def ^:private basecmds
  {:uname ["uname" "-sm"]
   :test  ["test"]
   :mkdir ["mkdir" "-p"]
   :cp    ["cp"]
   :scp   ["scp"]
   :tar   ["tar" "xf"]
   :rm    ["rm" "-rf"]})

(defn version
  "Generate the bootstrap version string."
  []
  (str "bootstrap (git version: " ver/version ")"))
      
(defn- loadjre [target path args]
  (let [ssh (cmd->sshcmd target)
        tgz (apply wj/download args)
        srcpath (str "/tmp/" tgz)
        cp (-> basecmds :cp (conj srcpath tgz))
        scp (-> basecmds :scp (conj srcpath (str target ":" tgz)))
        tar (-> basecmds :tar (conj tgz "-C" path) ssh)
        rmtmp (-> basecmds :rm (conj srcpath))
        rmtgz (-> basecmds :rm (conj tgz) ssh)]
    (and (if (localhost? target)
           (-> cp shell-cmd successful?)
           (-> scp shell-cmd successful?))
         (-> tar shell-cmd successful?)
         (-> rmtmp shell-cmd successful?)
         (-> rmtgz shell-cmd successful?))))

(defn bootstrap
  "Bootstrap a jar at user@host via ssh/scp."
  [&{:keys [user host path jar options]}]
  (let [target (if (seq user) (str user "@" host) host)
        jrepath (str path "/jre1.7.0_51")
        java (str jrepath "/bin/java")
        jarpath (str path "/" (-> jar .toString (str/split #"/") last))
        ssh (cmd->sshcmd target)
        uname (-> basecmds :uname ssh)
        basedir (-> basecmds :test (conj (str (:isnotdir teststrings) path)) ssh)
        mkbasedir (-> basecmds :mkdir (conj path) ssh)
        jredir (-> basecmds :test (conj (str (:isnotdir teststrings) jrepath)) ssh)
        jarfile (-> basecmds :test (conj (str (:isnotfile teststrings) jarpath)) ssh)
        cp (-> basecmds :cp (conj (-> jar .toString) path))
        scp (-> basecmds :scp (conj (-> jar .toString) (str target ":" path)))
        jarcmd (-> [(str "nohup sh -c \"( ( " java " -jar " jarpath " " options " &>/dev/null ) & )\"")] ssh)]
    (when-let [args (->> (-> uname shell-cmd successful? :proc :out first 
                            str/lower-case (str/split #" "))
                        (mapv keyword)
                        (get-in jremap))]
      (and (-> basedir shell-cmd successful?)
           (-> mkbasedir shell-cmd successful?))
      (and (-> jredir shell-cmd successful?)
           (loadjre target path args))
      (if (-> jarfile shell-cmd successful?)
        (if (localhost? target)
          (-> cp shell-cmd successful?)
          (-> scp shell-cmd successful?)))
      (-> jarcmd shell-cmd successful?))))
