(ns org.ozias.cljlibs.bootstrap.bootstrap
  (:require [clojure.string :as str]
            [org.ozias.cljlibs.bootstrap.version :as ver]
            [org.ozias.cljlibs.shell.shell :refer :all]
            [org.ozias.cljlibs.utils.core :refer :all]
            [wgetjava.handler :refer [download]])
  (:import (java.io IOException)))

(def ^{:private true :doc "A map of os arch to wgetjava options"}
  wgetopts {:cygwin_nt-6.1  {:i686   [:type :jre :os :win :arch :32]
                             :x86_64 [:type :jre :os :win]}
            :darwin         {:i386   [:type :jre :os :mac :arch :32]
                             :x86_64 [:type :jre :os :mac]}
            :linux          {:i686   [:type :jre :arch :32]
                             :x86_64 [:type :jre]}
            :mingw32_nt-6.1 {:i686 [:type :jre :os :win :arch :32]}})

(def ^{:private true :doc "bash test strings"}
  teststrings {:isnotdir  "! -d "
               :isnotfile "! -f "})

(def ^{:private true :doc "A map of base command vectors"}
  basecmds {:uname ["uname" "-sm"]
            :ver   ["cmd" "/c" "ver"]
            :warch ["cmd" "/c" "echo" "%PROCESSOR_ARCHITECTURE%"]
            :test  ["test"]
            :mkdir ["mkdir" "-p"]
            :cp    ["cp"]
            :scp   ["scp" "-v"]
            :rsync ["rsync" "-vP"]
            :tar   ["tar" "xf"]
            :rm    ["rm" "-rf"]})

(defn version
  "Generate the bootstrap version string."
  []
  (str "bootstrap (git version: " ver/version ")"))

(defn- out
  "print output"
  [line _]
  (println line))

(defn loadjre
  "load a jre on the given target at the given path.
 The args are valid kv pairs to the wgetjava download function."
  [target path args]
  (let [ssh (cmd->sshcmd target)
        tgz (last (str/split (last (last (apply download args))) #"/"))
        srcpath (str "/tmp/" tgz)
        cp (-> basecmds :cp (conj srcpath tgz))
        scp (-> basecmds :scp (conj srcpath (str target ":" tgz)))
        tar (-> basecmds :tar (conj tgz "-C" path) ssh)
        rmtmp (-> basecmds :rm (conj srcpath))
        rmtgz (-> basecmds :rm (conj tgz) ssh)]
    (and (if (localhost? target)
           (-> cp (shell-cmd :out out :err out) successful?)
           (-> scp (shell-cmd :out out :err out) successful?))
         (-> tar (shell-cmd :out out :err out) successful?)
         (-> rmtmp (shell-cmd :out out :err out) successful?)
         (-> rmtgz (shell-cmd :out out :err out) successful?))))

(defn- future-monitor
  "Given a deref'd future (potentially blocking), call the callback function
  with the result as the first argument and args as the rest."
  [derefd callback & args]
  (future (apply callback derefd args)))

(defn fmon
  "Monitor (block on) the given future.  Call the callback function with the
  result as the first argument and args as the rest when the future completes."
  [f callback & args]
  (apply future-monitor (deref f) callback args))

(defn timed-fmon
  "Monitor (block on) the given future until a given timeout period has passed
  or the future completes.

  If the timeout period has expired the callback function is called with the
  given timeout value as the first argument and args as the rest.

  If the future completed the callback function is called with the result as the
  first argument and args as the rest."
  [f callback timeout-ms timeout-result & args]
  (apply future-monitor (deref f timeout-ms timeout-result) callback args))

(defn timed-fmon-loop
  "Run a future monitor in a timed loop.  If the loop expires and the future
  isn't complete the future is cancelled.  Otherwise the result of the future is
  returned"
  [f callback period-ms step]
  (loop [remaining (- period-ms step)]
    (let [m (timed-fmon f callback step
                        :tc :per period-ms :rem remaining)]
      (if (or (neg? remaining) (zero? remaining))
        (future-cancel f)
        (do @m
            (if ((complement realized?) f)
              (recur (- remaining step))
              @f))))))

(comment
  (doseq [buf (shell-cmd [] :verbose false :seq true :buffer 80)]
    (print buf)
    (flush)))

; ubuntu@ec2-174-129-241-245.compute-1.amazonaws.com
; ubuntu@ec2-23-21-46-9.compute-1.amazonaws.com
(defn- target
  "Generate a target from a user and host string"
  [user host]
  (let [host (if (string? host) host "localhost")
        user (if (string? user) (str user "@") "")]
    (str user host)))

(defn target-os-arch
  "Determine the os and arch combination for user@host

  Evaluates to a sequence of the form (os arch)"
  [{:keys [user host sshopts] :or {sshopts (list)}}]
  (let [ssh (apply cmd->sshcmd (target user host) sshopts)]
    (try
      (let [uname (-> basecmds :uname ssh shell-cmd)]
        (if (successful? uname)
          (-> uname :proc :out first (str/split #" "))))
      (catch IOException _
        (let [ver (-> basecmds :ver ssh shell-cmd)
              arch (-> basecmds :warch ssh shell-cmd)]
          (if (and (successful? ver) (successful? arch))
            (list (->> (-> ver :proc :out rest first (str/split #" "))
                       (take 2)
                       (rest)
                       (str/join " "))
                  (-> arch :proc :out first))))))))

(defn target-os-k
  "Generate an OS key"
  [config]
  (let [[os _] (target-os-arch config)]
    (condp = os
      "CYGWIN_NT-6.1" :windows
      "Darwin" :macosx
      "Linux" :linux
      "MINGW32_NT-6.1" :windows
      "Windows" :windows)))

(defn get-wgetopts
  "Get the wgetjava options from the wgetopts map"
  [oaks]
  (->> oaks
       (map #(.toLowerCase ^String %))
       (mapv keyword)
       (get-in wgetopts)))

(defn bootstrap2
  "bootstrap a jar at user@host via ssh (or not if localhost"
  [config]
  (let [[os arch] (target-os-arch config)]
    (println "Target OS/Arch =>" os arch)))


;(if-let [target (target user host)]
;  (let [jrepath (to-path {:os :windows
;                          :ps (list
;                                (if (string? path) path ".") "jre1.8.0_05")})
;        javaexe (to-path {:os :windows :ps (list jrepath "bin" "java")})]
;    (println "Target   =>" target)
;    (println "JRE Path =>" jrepath)
;    (println "Java EXE =>" javaexe)
;    (println "Bootstrapped"))))

;(defn bootstrap
;  "Bootstrap a jar at user@host via ssh/scp."
;  [& {:keys [user host path jar options]}]
;  (let [jrepath (str path "/jre1.8.0_5")
;        java (str jrepath "/bin/java")
;        jarpath (str path "/" (-> jar str (str/split #"/") last))
;        ssh (cmd->sshcmd (target user host))
;        uname (-> basecmds :uname ssh)
;        basedir (-> basecmds :test
;                    (conj (str (:isnotdir teststrings) path)) ssh)
;        mkbasedir (-> basecmds :mkdir (conj path) ssh)
;        jredir (-> basecmds :test
;                   (conj (str (:isnotdir teststrings) jrepath)) ssh)
;        jarfile (-> basecmds :test
;                    (conj (str (:isnotfile teststrings) jarpath)) ssh)
;        cp (-> basecmds :cp (conj (str jar) path))
;        scp (-> basecmds :scp (conj (str jar) (str target ":" path)))
;       jarcmd (ssh [(str "nohup sh -c \"( ( " java " -jar " jarpath " " options
;                          " &>/dev/null ) & )\"")])]
;    (when-let [args (->> (-> uname shell-cmd successful? :proc :out first
;                             str/lower-case (str/split #" "))
;                         (mapv keyword)
;                         (get-in jremap))]
;      (and (-> basedir shell-cmd successful?)
;           (-> mkbasedir shell-cmd successful?))
;      (and (-> jredir shell-cmd successful?)
;           (loadjre target path args))
;      (if (-> jarfile shell-cmd successful?)
;        (if (localhost? target)
;          (-> cp shell-cmd successful?)
;          (-> scp shell-cmd successful?)))
;      (-> jarcmd shell-cmd successful?))))