(ns onyx.state.lmdb
  "Implementation of a LMDB backed state store. Currently this is alpha level quality."
  (:require [onyx.state.protocol.db :as db]
            [onyx.state.serializers.utils :as u]
            [onyx.state.serializers.windowing-key-encoder :as enc :refer [encode-key]]
            [onyx.state.serializers.windowing-key-decoder :as dec]
            [onyx.state.serializers.state-entry-key-encoder :as senc]
            [onyx.state.serializers.state-entry-key-decoder :as sdec]
            [onyx.state.serializers.checkpoint :as cp]
            [onyx.compression.nippy :refer [statedb-compress statedb-decompress]])
  (:import [org.fusesource.lmdbjni Database Env Transaction Entry Constants]
           [org.agrona MutableDirectBuffer]
           [org.agrona.concurrent UnsafeBuffer]))

(defn ^Transaction read-txn [^Env env]
   (.createReadTransaction env))

(defn items
  [^Database db txn]
  (iterator-seq (.iterate db txn)))

(defn seek-items
  [^Database db txn ^bytes k]
  (iterator-seq (.seek db txn k)))

(defn db-empty? [db env]
  (let [txn (read-txn env)]
    (try 
     (zero? (count (items db txn)))
     (finally
      (.abort txn)))))

;; TODO: export metrics
(defn stat [^Database db]
  (let [stat (.stat db)]
    {:ms-entries (.ms_entries stat)
     :ms-psize (.ms_psize stat)
     :ms-overflow-pages (.ms_overflow_pages stat)
     :ms-depth (.ms_depth stat)
     :ms-leaf-pages (.ms_leaf_pages stat)}))

(defn get-state-idx [^bytes bs]
  ;; remove extra allocation
  (.getShort (UnsafeBuffer. bs) 0))

(defn set-state-idx [^bytes bs idx]
  ;; remove extra allocation
  (.putShort (UnsafeBuffer. bs) 0 idx))

;; FIXME, just use idx+1 and skip over idx on rest?
(defn to-history-idx [idx] (- -1 idx))

(def ^:const min-extent 0)

(deftype StateBackend [groups group-counter entry-counter ^Database db ^String name ^Env env 
                       serialize-fn deserialize-fn window-encoders 
                       window-decoders log-entry-encoders log-entry-decoders 
                       trigger-encoders trigger-decoders]
  db/State
  ;; FIXME rename to time-entry
  (put-extent-entry! [this window-id group time v]
    (let [;b (UnsafeBuffer. (byte-array 26))
          enc (get log-entry-encoders window-id)
          idx (to-history-idx window-id)
          ;; FIXME, SHOULD BE DONE IN DB
          group (some-> group deserialize-fn)
          group-id (or (get @groups group)
                       (do (swap! group-counter inc)
                           (swap! groups assoc group @group-counter)
                           @group-counter))]
      (senc/set-idx enc idx)
      (senc/set-group enc group-id)
      (senc/set-time enc time)
      (senc/set-offset enc (long (swap! entry-counter inc)))
      (.put db 
            ^bytes (senc/get-bytes enc) 
            ^bytes (serialize-fn v))))
  (delete-extent-entries! [this window-id group start end]
    (let [enc (get log-entry-encoders window-id)
          dec (get log-entry-decoders window-id)
          idx (to-history-idx window-id)
          ;; FIXME, SHOULD BE DONE IN DB
          group-id (some->> group
                            deserialize-fn
                            (get @groups))]
      (when (or group-id (nil? group))
        (senc/set-idx enc idx)
        (senc/set-group enc group-id)
        (senc/set-time enc start)
        (senc/set-offset enc 0)
        (let [seek-key (senc/get-bytes enc) 
              txn (read-txn env)]
          (try 
           (let [iterator (.seek db txn seek-key)] 
             (loop []
               (if (.hasNext iterator)
                 (let [entry ^Entry (.next iterator)
                       key-bs (.getKey entry)
                       _ (sdec/wrap-impl dec key-bs)
                       ts (sdec/get-time dec)]
                   (when (and (= idx (sdec/get-idx dec))
                              (= group-id (sdec/get-group-id dec))
                              (<= ts end))
                     (.delete db ^bytes key-bs)
                     (recur))))))
           (finally 
            (.abort txn)))))))
  (get-extent-entries [this window-id group start end]
    (let [enc (get log-entry-encoders window-id)
          dec (get log-entry-decoders window-id)
          idx (to-history-idx window-id)
          group-id (some->> group
                            deserialize-fn
                            (get @groups))
          vs (transient [])]
      (when (or (nil? group) ;; hack
                group-id)
        (senc/set-idx enc idx)
        (senc/set-group enc group-id)
        (senc/set-time enc start)
        (senc/set-offset enc 0)
        (let [seek-key (senc/get-bytes enc) 
              txn (read-txn env)]
          (try 
           (let [iterator (.seek db txn seek-key)] 
             (loop []
               (if (.hasNext iterator)
                 (let [entry ^Entry (.next iterator)
                       _ (sdec/wrap-impl dec (.getKey entry))]
                   (when (and (= idx (sdec/get-idx dec))
                              (= group-id (sdec/get-group-id dec))
                              (<= (sdec/get-time dec) end))
                     (conj! vs (deserialize-fn (.getValue entry)))
                     (recur))))))
           (finally 
            (.abort txn)))))
      (persistent! vs)))
  (put-extent! [this window-id group-id extent v]
    (let [enc (get window-encoders window-id)]
      (.put db 
            ^bytes (encode-key enc window-id group-id extent)
            ^bytes (serialize-fn v))))
  (get-extent [this window-id group-id extent]
    (let [enc (get window-encoders window-id)]
      (some-> (.get db ^bytes (encode-key enc window-id group-id extent))
              (deserialize-fn))))
  (delete-extent! [this window-id group-id extent]
    (let [enc (get window-encoders window-id)]
      (.delete db ^bytes (encode-key enc window-id group-id extent))))
  (put-trigger! [this trigger-id group-id v]
    (let [enc (get trigger-encoders trigger-id)]
      (.put db 
            ^bytes (encode-key enc trigger-id group-id nil)
            ^bytes (serialize-fn v))))
  (get-trigger [this trigger-id group-id]
    (if-let [enc (get trigger-encoders trigger-id)] 
      (if-let [value (.get db ^bytes (encode-key enc trigger-id group-id nil))]
        (deserialize-fn value)
        :not-found)
      :not-found))
  (group-id [this group-key]
    (serialize-fn group-key))
  (group-key [this group-id]
    ;; nil for ungrouped windows
    ;; improve the way this works?
    (some-> group-id deserialize-fn))
  (groups [this state-idx]
    (when-let [enc (or (get window-encoders state-idx) 
                       (get trigger-encoders state-idx))]
      (let [_ (enc/set-state-idx enc state-idx)
            _ (enc/set-group enc (byte-array 0))
            k (enc/get-bytes enc) 
            txn (read-txn env)]
        (try 
         (let [iterator (.seek db txn k)
               decoder (or (get window-decoders state-idx)
                           (get trigger-decoders state-idx))
               vs (transient [])] 
           (loop [prev-group (byte-array 0)]
             (if (.hasNext iterator)
               (let [entry ^Entry (.next iterator)] 
                 (dec/wrap-impl decoder (.getKey entry))
                 (when (= state-idx (dec/get-state-idx decoder))
                   (when-let [bs (dec/get-group decoder)]
                     (when-not (u/equals prev-group bs) 
                       (conj! vs (deserialize-fn bs)))
                     (recur bs))))))
           (persistent! vs))
         (finally 
          (.abort txn))))))
  (trigger-keys [this trigger-idx]
    (when-let [decoder (get trigger-decoders trigger-idx)]
      (let [txn (read-txn env)]
        (try 
         (let [iterator (.iterate db txn)
               vs (transient [])] 
           (loop []
             (if (.hasNext iterator)
               (let [entry ^Entry (.next iterator)]
                 (when (= trigger-idx (get-state-idx (.getKey entry)))
                   (dec/wrap-impl decoder (.getKey entry))  
                   (conj! vs [(dec/get-group decoder)
                              (db/group-key this (dec/get-group decoder))]))
                 (recur))))
           (persistent! vs))
         (finally 
          (.abort txn))))))
  (group-extents [this window-idx group-id]
     (let [ungrouped? (nil? group-id)
           enc (get window-encoders window-idx)
           _ (enc/set-state-idx enc window-idx)
           _ (enc/set-group enc group-id)
           _ (enc/set-extent enc min-extent)
           k (enc/get-bytes enc) 
           txn (read-txn env)]
       (try 
        (let [iterator (.seek db txn k)
              decoder (get window-decoders window-idx)
              vs (transient [])] 
          (loop []
            (if (.hasNext iterator)
              (let [entry ^Entry (.next iterator)]
                (dec/wrap-impl decoder (.getKey entry))
                (when (and (= window-idx (dec/get-state-idx decoder))
                           (or ungrouped? (u/equals (dec/get-group decoder) group-id)))
                  (conj! vs (dec/get-extent decoder))
                  (recur)))))
          (persistent! vs))
        (finally 
         (.abort txn)))))
  (drop! [this]
    ;; FIXME, does not actually delete the DB files
    (.drop db true))
  (close! [this]
    (.close env)
    (.close db))
  (export-reader [this]
    nil)
  (export [this state-encoder]
    (let [txn (read-txn env)]
      (try 
       (cp/set-next-bytes state-encoder (serialize-fn {}))
       (->> txn
            (items db)
            (run! (fn [^Entry entry]
                    (cp/set-next-bytes state-encoder (.getKey entry))
                    (cp/set-next-bytes state-encoder (.getValue entry))))) 
       (finally
        (.abort txn)))))
  (restore! [this state-decoder mapping]
    (when-not (db-empty? db env)
      (throw (Exception. "LMDB DB is not empty. This should never happen.")))
    ;; FIXME throw away items currently stored in memroy DB
    (cp/get-next-bytes state-decoder)
    (loop []
      (let [k ^bytes (cp/get-next-bytes state-decoder)
            v ^bytes (cp/get-next-bytes state-decoder)]
        (when k
          (assert v)
          (let [serialized-state-index (get-state-idx k)] 
            ;; re-index state index
            (when-let [new-idx (mapping serialized-state-index)]
              (set-state-idx k new-idx)
              (.put db k v)))
          (recur))))))

(defmethod db/open-db-reader :lmdb
  [peer-config 
   definition
   {:keys [window-encoders 
           window-decoders 
           trigger-encoders 
           trigger-decoders]}]
  ;; Not implemented yet.
  nil)

(defmethod db/create-db 
  :lmdb
  [peer-config 
   db-name 
   {:keys [window-encoders 
           window-decoders 
           log-entry-encoders
           log-entry-decoders
           trigger-encoders 
           trigger-decoders]}]
  (let [max-size 1024000
        path (str (System/getProperty "java.io.tmpdir") "/onyx/" (java.util.UUID/randomUUID) "/")
        _ (.mkdirs (java.io.File. path))
        env (doto (Env. path)
              ;(.addFlags (reduce bit-or [Constants/NOSYNC Constants/MAPASYNC]))
              (.setMapSize max-size))
        db (.openDatabase env db-name)]
    (->StateBackend (atom {})
                    (atom (long -1))
                    (atom (long -1))
                    db name env 
                    statedb-compress 
                    statedb-decompress 
                    window-encoders 
                    window-decoders 
                    log-entry-encoders
                    log-entry-decoders
                    trigger-encoders 
                    trigger-decoders)))
