(ns eywa.client.files
  "EYWA File Operations for Babashka/Clojure

  This namespace provides comprehensive file upload/download capabilities
  for the EYWA Clojure client, handling the complete file lifecycle:
  - Upload files with automatic URL generation and S3 upload
  - Download files with automatic URL generation
  - List and manage uploaded files with filtering
  - Progress tracking for uploads and downloads"
  (:require
   [clojure.core.async :as async :refer [go <!]]
   [clojure.java.io :as io]
   [clojure.string :as str])
  (:import
   [java.io File FileInputStream FileOutputStream]
   [java.net URI URL HttpURLConnection]
   [java.nio.file Files Paths]
   [java.security MessageDigest]
   [java.util UUID]))

;; Forward declarations for functions provided by eywa.client
;; This breaks the circular dependency
(declare graphql info debug error warn)

;; Forward declarations for functions defined later in this namespace
(declare get-file-by-name list-files download-stream)

;; Exception types for file operations

(defn file-upload-error [msg & {:keys [type code] :or {type :upload-error}}]
  (ex-info (str "File upload error: " msg)
           (cond-> {:type type}
             code (assoc :http-code code))))

(defn file-download-error [msg & {:keys [type code] :or {type :download-error}}]
  (ex-info (str "File download error: " msg)
           (cond-> {:type type}
             code (assoc :http-code code))))

;; Retry helper

(defn- with-retry
  "Execute a function with retry logic and exponential backoff.
  
  Args:
    f - Function to execute (should return {:status :success/:error})
    opts - Map with:
      :retries - Number of retries (default: 3)
      :retry-delay-ms - Initial delay between retries (default: 1000)
      :backoff-multiplier - Multiply delay by this each retry (default: 2)
      :retryable-codes - Set of HTTP codes to retry on (default: #{408 429 500 502 503 504})
  
  Returns:
    Result of f, or throws after exhausting retries"
  [f & {:keys [retries retry-delay-ms backoff-multiplier retryable-codes]
        :or {retries 3
             retry-delay-ms 1000
             backoff-multiplier 2
             retryable-codes #{408 429 500 502 503 504}}}]
  (loop [attempt 0
         delay retry-delay-ms]
    (let [result (try
                   (f)
                   (catch Exception e
                     {:status :error :exception e}))]
      (cond
        ;; Success - return result
        (= :success (:status result))
        result

        ;; No more retries - return/throw error
        (>= attempt retries)
        (if (:exception result)
          (throw (:exception result))
          result)

        ;; Retryable error - retry with backoff
        (and (= :error (:status result))
             (or (:exception result)
                 (retryable-codes (:code result))))
        (do
          (debug (str "Retry attempt " (inc attempt) "/" retries
                             " after " delay "ms"
                             (when-let [code (:code result)]
                               (str " (HTTP " code ")"))))
          (Thread/sleep delay)
          (recur (inc attempt) (* delay backoff-multiplier)))

        ;; Non-retryable error - return immediately
        :else
        result))))

;; Utility functions

(defn- mime-type
  "Detect MIME type from file extension"
  [filename]
  (let [ext (-> filename
                (str/split #"\.")
                last
                str/lower-case)]
    (case ext
      "txt" "text/plain"
      "html" "text/html"
      "css" "text/css"
      "js" "application/javascript"
      "json" "application/json"
      "xml" "application/xml"
      "pdf" "application/pdf"
      "png" "image/png"
      "jpg" "image/jpeg"
      "jpeg" "image/jpeg"
      "gif" "image/gif"
      "svg" "image/svg+xml"
      "zip" "application/zip"
      "csv" "text/csv"
      "doc" "application/msword"
      "docx" "application/vnd.openxmlformats-officedocument.wordprocessingml.document"
      "xls" "application/vnd.ms-excel"
      "xlsx" "application/vnd.openxmlformats-officedocument.spreadsheetml.sheet"
      "ppt" "application/vnd.ms-powerpoint"
      "pptx" "application/vnd.openxmlformats-officedocument.presentationml.presentation"
      "application/octet-stream")))

(defn- http-put!
  "Perform HTTP PUT request to upload data"
  [url data content-type progress-fn]
  (let [connection (.openConnection (URL. url))
        _ (.setRequestMethod connection "PUT")
        _ (.setDoOutput connection true)
        _ (.setRequestProperty connection "Content-Type" content-type)
        content-length (if (string? data) (.length (.getBytes data "UTF-8")) (count data))
        _ (.setRequestProperty connection "Content-Length" (str content-length))]

    (when progress-fn (progress-fn 0 content-length))

    (with-open [out (.getOutputStream connection)]
      (if (string? data)
        (.write out (.getBytes data "UTF-8"))
        (.write out data)))

    (when progress-fn (progress-fn content-length content-length))

    (let [response-code (.getResponseCode connection)]
      (if (= 200 response-code)
        {:status :success :code response-code}
        (let [error-stream (.getErrorStream connection)
              error-msg (when error-stream
                          (slurp error-stream))]
          {:status :error :code response-code :message error-msg})))))

(defn- http-put-stream!
  "Perform HTTP PUT request streaming from an InputStream.
  
  Streams the data in chunks to avoid loading entire file into memory.
  
  Args:
    url - URL to PUT to
    input-stream - InputStream to read from (caller must close!)
    content-length - Total bytes to upload
    content-type - MIME type
    progress-fn - Optional function called with (bytes-uploaded, total-bytes)
    opts - Optional map with :connect-timeout-ms and :read-timeout-ms
  
  Returns:
    Map with :status :success or :error, :code response-code"
  [url input-stream content-length content-type progress-fn & [{:keys [connect-timeout-ms read-timeout-ms]
                                                                :or {connect-timeout-ms 10000
                                                                     read-timeout-ms 300000}}]]
  (let [connection (.openConnection (URL. url))
        _ (.setConnectTimeout connection connect-timeout-ms)
        _ (.setReadTimeout connection read-timeout-ms)
        _ (.setRequestMethod connection "PUT")
        _ (.setDoOutput connection true)
        _ (.setRequestProperty connection "Content-Type" content-type)
        _ (.setRequestProperty connection "Content-Length" (str content-length))]

    (when progress-fn (progress-fn 0 content-length))

    (with-open [out (.getOutputStream connection)]
      (let [buffer (byte-array 8192)
            total-written (atom 0)]
        (loop []
          (let [bytes-read (.read input-stream buffer)]
            (when (> bytes-read 0)
              (.write out buffer 0 bytes-read)
              (swap! total-written + bytes-read)
              (when progress-fn
                (progress-fn @total-written content-length))
              (recur))))))

    (let [response-code (.getResponseCode connection)]
      (if (= 200 response-code)
        {:status :success :code response-code}
        (let [error-stream (.getErrorStream connection)
              error-msg (when error-stream
                          (slurp error-stream))]
          {:status :error :code response-code :message error-msg})))))

(defn- http-get!
  "Perform HTTP GET request to download data"
  [url progress-fn]
  (let [connection (.openConnection (URL. url))
        _ (.setRequestMethod connection "GET")
        response-code (.getResponseCode connection)]

    (if (= 200 response-code)
      (let [content-length (try
                             (Long/parseLong (.getHeaderField connection "Content-Length"))
                             (catch Exception _ 0))
            input-stream (.getInputStream connection)]

        (when (and progress-fn (> content-length 0))
          (progress-fn 0 content-length))

        (let [baos (java.io.ByteArrayOutputStream.)
              buffer (byte-array 8192)
              total-read (atom 0)]

          (loop []
            (let [bytes-read (.read input-stream buffer)]
              (when (> bytes-read 0)
                (.write baos buffer 0 bytes-read)
                (swap! total-read + bytes-read)
                (when (and progress-fn (> content-length 0))
                  (progress-fn @total-read content-length))
                (recur))))

          (.close input-stream)
          {:status :success :data (.toByteArray baos)}))
      {:status :error :code response-code})))

(defn- http-get-stream!
  "Perform HTTP GET request and return InputStream for streaming.
  
  Returns map with:
  - :status :success or :error
  - :stream InputStream (caller must close!)
  - :content-length Long (0 if unknown)
  - :code HTTP response code (on error)
  
  Args:
    url - URL to GET from
    opts - Optional map with :connect-timeout-ms and :read-timeout-ms"
  [url & [{:keys [connect-timeout-ms read-timeout-ms]
           :or {connect-timeout-ms 10000
                read-timeout-ms 300000}}]]
  (let [connection (.openConnection (URL. url))
        _ (.setConnectTimeout connection connect-timeout-ms)
        _ (.setReadTimeout connection read-timeout-ms)
        _ (.setRequestMethod connection "GET")
        response-code (.getResponseCode connection)]

    (if (= 200 response-code)
      (let [content-length (try
                             (Long/parseLong (.getHeaderField connection "Content-Length"))
                             (catch Exception _ 0))
            input-stream (.getInputStream connection)]
        {:status :success
         :stream input-stream
         :content-length content-length})
      {:status :error :code response-code})))

(defn- calculate-hash
  "Calculate hash of file or data"
  [file-or-data algorithm]
  (let [md (MessageDigest/getInstance algorithm)
        data (if (instance? File file-or-data)
               (Files/readAllBytes (.toPath file-or-data))
               (if (string? file-or-data)
                 (.getBytes file-or-data "UTF-8")
                 file-or-data))]
    (.update md data)
    (let [digest (.digest md)]
      (apply str (map #(format "%02x" (bit-and % 0xff)) digest)))))

;; Core file operations

(defn upload-file
  "Upload a file to EYWA file service using streaming (memory efficient).
  
  Args:
    filepath - Path to the file to upload (string or File)
    options - Map of options:
      :name - Custom filename (defaults to file basename)
      :content-type - MIME type (auto-detected if not provided)
      :folder-uuid - UUID of parent folder (optional)
      :progress-fn - Function called with (bytes-uploaded, total-bytes)
  
  Returns:
    Core.async channel that will contain file information map or exception"
  [filepath & {:keys [name content-type folder-uuid progress-fn]}]
  (go
    (try
      (let [file (if (instance? File filepath) filepath (File. filepath))
            _ (when-not (.exists file)
                (throw (file-upload-error (str "File not found: " filepath))))
            _ (when-not (.isFile file)
                (throw (file-upload-error (str "Path is not a file: " filepath))))

            file-size (.length file)
            file-name (or name (.getName file))
            detected-content-type (or content-type (mime-type file-name))

            _ (info (str "Starting streaming upload: " file-name " (" file-size " bytes)"))

            ;; Step 1: Request upload URL
            upload-query "mutation RequestUpload($file: FileInput!) {
                           requestUploadURL(file: $file)
                         }"

            variables (cond-> {:file {:name file-name
                                      :content_type detected-content-type
                                      :size file-size}}
                        folder-uuid (assoc-in [:file :folder] {:euuid folder-uuid}))

            {:keys [error result]} (<! (graphql upload-query variables))

            _ (when error (throw (file-upload-error (str "Failed to get upload URL: " error))))

            upload-url (get-in result [:data :requestUploadURL])
            _ (when-not upload-url (throw (file-upload-error "No upload URL in response")))

            _ (debug (str "Upload URL received: " (subs upload-url 0 (min 50 (count upload-url))) "..."))

            ;; Step 2: Stream file to S3
            upload-result (with-open [fis (FileInputStream. file)]
                            (http-put-stream! upload-url fis file-size detected-content-type progress-fn))

            _ (when (= :error (:status upload-result))
                (throw (file-upload-error (str "S3 upload failed (" (:code upload-result) "): " (:message upload-result)))))

            _ (debug "File uploaded to S3 successfully")

            ;; Step 3: Confirm upload
            confirm-query "mutation ConfirmUpload($url: String!) {
                            confirmFileUpload(url: $url)
                          }"

            {:keys [error result]} (<! (graphql confirm-query {:url upload-url}))

            _ (when error (throw (file-upload-error (str "Upload confirmation failed: " error))))

            confirmed? (get-in result [:data :confirmFileUpload])
            _ (when-not confirmed? (throw (file-upload-error "Upload confirmation returned false")))

            _ (debug "Upload confirmed")

            ;; Step 4: Get file information
            file-info (<! (get-file-by-name file-name))
            _ (when-not file-info (throw (file-upload-error "Could not retrieve uploaded file information")))]

        (info (str "Upload completed: " file-name " -> " (:euuid file-info)))
        file-info)

      (catch Exception e
        (error (str "Upload failed: " (.getMessage e)))
        e))))

(defn upload-stream
  "Upload data from an InputStream to EYWA file service.
  
  This allows uploading from any source that provides an InputStream.
  Useful for piping data, network streams, or generated content.
  
  Args:
    input-stream - InputStream to upload from (will be closed by this function)
    file-name - Name for the uploaded file
    content-length - Total bytes to upload (must be known upfront)
    options - Map of options:
      :content-type - MIME type (default: application/octet-stream)
      :folder-uuid - UUID of parent folder (optional)
      :progress-fn - Function called with (bytes-uploaded, total-bytes)
  
  Returns:
    Core.async channel that will contain file information map or exception
  
  Example:
    (with-open [is (io/input-stream source)]
      (<! (upload-stream is \"data.bin\" file-size :content-type \"application/octet-stream\")))"
  [input-stream file-name content-length & {:keys [content-type folder-uuid progress-fn]
                                            :or {content-type "application/octet-stream"}}]
  (go
    (try
      (let [_ (info (str "Starting stream upload: " file-name " (" content-length " bytes)"))

            ;; Step 1: Request upload URL
            upload-query "mutation RequestUpload($file: FileInput!) {
                           requestUploadURL(file: $file)
                         }"

            variables (cond-> {:file {:name file-name
                                      :content_type content-type
                                      :size content-length}}
                        folder-uuid (assoc-in [:file :folder] {:euuid folder-uuid}))

            {:keys [error result]} (<! (graphql upload-query variables))

            _ (when error (throw (file-upload-error (str "Failed to get upload URL: " error))))

            upload-url (get-in result [:data :requestUploadURL])
            _ (when-not upload-url (throw (file-upload-error "No upload URL in response")))

            _ (debug (str "Upload URL received: " (subs upload-url 0 (min 50 (count upload-url))) "..."))

            ;; Step 2: Stream to S3
            upload-result (try
                            (http-put-stream! upload-url input-stream content-length content-type progress-fn)
                            (finally
                              (try (.close input-stream) (catch Exception _))))

            _ (when (= :error (:status upload-result))
                (throw (file-upload-error (str "S3 upload failed (" (:code upload-result) "): " (:message upload-result)))))

            _ (debug "Stream uploaded to S3 successfully")

            ;; Step 3: Confirm upload
            confirm-query "mutation ConfirmUpload($url: String!) {
                            confirmFileUpload(url: $url)
                          }"

            {:keys [error result]} (<! (graphql confirm-query {:url upload-url}))

            _ (when error (throw (file-upload-error (str "Upload confirmation failed: " error))))

            confirmed? (get-in result [:data :confirmFileUpload])
            _ (when-not confirmed? (throw (file-upload-error "Upload confirmation returned false")))

            _ (debug "Upload confirmed")

            ;; Step 4: Get file information
            file-info (<! (get-file-by-name file-name))
            _ (when-not file-info (throw (file-upload-error "Could not retrieve uploaded file information")))]

        (info (str "Stream upload completed: " file-name " -> " (:euuid file-info)))
        file-info)

      (catch Exception e
        (error (str "Stream upload failed: " (.getMessage e)))
        e))))

(defn upload-content
  "Upload content directly from memory.
  
  Args:
    content - String or byte array content to upload
    name - Filename for the content
    options - Map of options:
      :content-type - MIME type (default: 'text/plain')
      :folder-uuid - UUID of parent folder (optional)
      :progress-fn - Function called with (bytes-uploaded, total-bytes)
  
  Returns:
    Core.async channel that will contain file information map or exception"
  [content name & {:keys [content-type folder-uuid progress-fn]
                   :or {content-type "text/plain"}}]
  (go
    (try
      (let [content-bytes (if (string? content)
                            (.getBytes content "UTF-8")
                            content)
            file-size (count content-bytes)

            _ (info (str "Starting content upload: " name " (" file-size " bytes)"))

            ;; Step 1: Request upload URL
            upload-query "mutation RequestUpload($file: FileInput!) {
                           requestUploadURL(file: $file)
                         }"

            variables (cond-> {:file {:name name
                                      :content_type content-type
                                      :size file-size}}
                        folder-uuid (assoc-in [:file :folder] {:euuid folder-uuid}))

            {:keys [error result]} (<! (graphql upload-query variables))

            _ (when error (throw (file-upload-error (str "Failed to get upload URL: " error))))

            upload-url (get-in result [:data :requestUploadURL])

            ;; Step 2: Upload content to S3
            upload-result (http-put! upload-url content-bytes content-type progress-fn)

            _ (when (= :error (:status upload-result))
                (throw (file-upload-error (str "S3 upload failed (" (:code upload-result) "): " (:message upload-result)))))

            ;; Step 3: Confirm upload
            confirm-query "mutation ConfirmUpload($url: String!) {
                            confirmFileUpload(url: $url)
                          }"

            {:keys [error result]} (<! (graphql confirm-query {:url upload-url}))

            _ (when error (throw (file-upload-error (str "Upload confirmation failed: " error))))

            confirmed? (get-in result [:data :confirmFileUpload])
            _ (when-not confirmed? (throw (file-upload-error "Upload confirmation returned false")))

            ;; Step 4: Get file information
            file-info (<! (get-file-by-name name))
            _ (when-not file-info (throw (file-upload-error "Could not retrieve uploaded file information")))]

        (info (str "Content upload completed: " name " -> " (:euuid file-info)))
        file-info)

      (catch Exception e
        (error (str "Content upload failed: " (.getMessage e)))
        e))))

(defn download-stream
  "Download a file from EYWA and return an InputStream for streaming.
  
  This is the recommended approach for large files as it doesn't buffer
  the entire file in memory. The returned InputStream can be used with
  clojure.java.io functions.
  
  Args:
    file-uuid - UUID of the file to download
  
  Returns:
    Core.async channel containing:
    - On success: map with :stream (InputStream - MUST be closed by caller!)
                  and :content-length (bytes, 0 if unknown)
    - On error: Exception
  
  Example:
    (async/go
      (let [result (async/<! (download-stream file-uuid))]
        (if (instance? Exception result)
          (handle-error result)
          (with-open [stream (:stream result)]
            (io/copy stream output-stream)))))"
  [file-uuid]
  (go
    (try
      (let [_ (info (str "Starting streaming download: " file-uuid))

            ;; Step 1: Request download URL
            download-query "query RequestDownload($file: FileInput!) {
                             requestDownloadURL(file: $file)
                           }"

            {:keys [error result]} (<! (graphql download-query {:file {:euuid file-uuid}}))

            _ (when error (throw (file-download-error (str "Failed to get download URL: " error))))

            download-url (get-in result [:data :requestDownloadURL])
            _ (when-not download-url (throw (file-download-error "No download URL in response")))

            _ (debug (str "Download URL received: " (subs download-url 0 (min 50 (count download-url))) "..."))

            ;; Step 2: Get InputStream from S3
            stream-result (http-get-stream! download-url)

            _ (when (= :error (:status stream-result))
                (throw (file-download-error (str "Download failed (" (:code stream-result) ")"))))]

        (info (str "Stream ready for: " file-uuid " (" (:content-length stream-result) " bytes)"))
        {:stream (:stream stream-result)
         :content-length (:content-length stream-result)})

      (catch Exception e
        (error (str "Stream download failed: " (.getMessage e)))
        e))))

(defn download-file
  "Download a file from EYWA file service (convenience function for small files).
  
  For large files, use download-stream instead to avoid buffering entire file in memory.
  
  Args:
    file-uuid - UUID of the file to download
    save-path - Path to save the file (if nil, returns content as bytes)
    progress-fn - Function called with (bytes-downloaded, total-bytes)
  
  Returns:
    Core.async channel that will contain saved path or content bytes, or exception"
  [file-uuid & {:keys [save-path progress-fn]}]
  (go
    (try
      (let [stream-result (<! (download-stream file-uuid))]
        (if (instance? Exception stream-result)
          stream-result
          (let [{:keys [stream content-length]} stream-result]
            (try
              (when progress-fn (progress-fn 0 content-length))

              (if save-path
                ;; Save to file
                (let [file (File. save-path)
                      parent-dir (.getParentFile file)]
                  (when (and parent-dir (not (.exists parent-dir)))
                    (.mkdirs parent-dir))

                  (with-open [is stream
                              fos (FileOutputStream. file)]
                    (let [buffer (byte-array 8192)
                          total-read (atom 0)]
                      (loop []
                        (let [bytes-read (.read is buffer)]
                          (when (> bytes-read 0)
                            (.write fos buffer 0 bytes-read)
                            (swap! total-read + bytes-read)
                            (when (and progress-fn (> content-length 0))
                              (progress-fn @total-read content-length))
                            (recur))))))

                  (info (str "Download completed: " file-uuid " -> " save-path))
                  save-path)

                ;; Return content as bytes
                (let [baos (java.io.ByteArrayOutputStream.)
                      buffer (byte-array 8192)
                      total-read (atom 0)]
                  (with-open [is stream]
                    (loop []
                      (let [bytes-read (.read is buffer)]
                        (when (> bytes-read 0)
                          (.write baos buffer 0 bytes-read)
                          (swap! total-read + bytes-read)
                          (when (and progress-fn (> content-length 0))
                            (progress-fn @total-read content-length))
                          (recur)))))

                  (let [content (.toByteArray baos)]
                    (info (str "Download completed: " file-uuid " (" (count content) " bytes)"))
                    content)))

              (catch Exception e
                (try (.close stream) (catch Exception _))
                (throw e))))))

      (catch Exception e
        (error (str "Download failed: " (.getMessage e)))
        e))))

(defn list-files
  "List files in EYWA file service.
  
  Args:
    options - Map of filter options:
      :limit - Maximum number of files to return
      :status - Filter by status (PENDING, UPLOADED, etc.)
      :name-pattern - Filter by name pattern (SQL LIKE)
      :folder-uuid - Filter by folder UUID
  
  Returns:
    Core.async channel that will contain list of file maps or exception"
  [& {:keys [limit status name-pattern folder-uuid]}]
  (go
    (try
      (let [_ (debug (str "Listing files (limit=" limit ", status=" status ")"))

            query "query ListFiles($limit: Int, $where: FileWhereInput) {
                     searchFile(_limit: $limit, _where: $where, _order_by: {uploaded_at: desc}) {
                       euuid
                       name
                       status
                       content_type
                       size
                       uploaded_at
                       uploaded_by {
                         name
                       }
                       folder {
                         euuid
                         name
                       }
                     }
                   }"

            where-conditions (cond-> []
                               status (conj {:status {:_eq status}})
                               name-pattern (conj {:name {:_ilike (str "%" name-pattern "%")}})
                               folder-uuid (conj {:folder {:euuid {:_eq folder-uuid}}}))

            variables (cond-> {}
                        limit (assoc :limit limit)
                        (seq where-conditions) (assoc :where
                                                      (if (= 1 (count where-conditions))
                                                        (first where-conditions)
                                                        {:_and where-conditions})))

            {:keys [error result]} (<! (graphql query variables))

            _ (when error (throw (ex-info (str "Failed to list files: " error) {:error error})))

            files (get-in result [:data :searchFile])]

        (debug (str "Found " (count files) " files"))
        files)

      (catch Exception e
        (error (str "Failed to list files: " (.getMessage e)))
        e))))

(defn get-file-info
  "Get information about a specific file.
  
  Args:
    file-uuid - UUID of the file
  
  Returns:
    Core.async channel that will contain file information map or nil if not found"
  [file-uuid]
  (go
    (try
      (let [query "query GetFile($uuid: UUID!) {
                     getFile(euuid: $uuid) {
                       euuid
                       name
                       status
                       content_type
                       size
                       uploaded_at
                       uploaded_by {
                         name
                       }
                       folder {
                         euuid
                         name
                       }
                     }
                   }"

            {:keys [error result]} (<! (graphql query {:uuid file-uuid}))]

        (if error
          (do
            (debug (str "File not found or error: " error))
            nil)
          (get-in result [:data :getFile])))

      (catch Exception e
        (debug (str "File not found or error: " (.getMessage e)))
        nil))))

(defn get-file-by-name
  "Get file information by name (returns most recent if multiple).
  
  Args:
    name - File name to search for
  
  Returns:
    Core.async channel that will contain file information map or nil if not found"
  [name]
  (go
    (let [files (<! (list-files :limit 1 :name-pattern name))]
      (if (instance? Exception files)
        files
        (first files)))))

(defn delete-file
  "Delete a file from EYWA file service.
  
  Args:
    file-uuid - UUID of the file to delete
  
  Returns:
    Core.async channel that will contain true if deletion successful, false otherwise"
  [file-uuid]
  (go
    (try
      (let [query "mutation DeleteFile($uuid: UUID!) {
                     deleteFile(euuid: $uuid)
                   }"

            {:keys [error result]} (<! (graphql query {:uuid file-uuid}))

            _ (when error (throw (ex-info (str "Failed to delete file: " error) {:error error})))

            success? (get-in result [:data :deleteFile])]

        (if success?
          (info (str "File deleted: " file-uuid))
          (warn (str "File deletion failed: " file-uuid)))

        success?)

      (catch Exception e
        (error (str "Failed to delete file: " (.getMessage e)))
        false))))

(defn calculate-file-hash
  "Calculate hash of a file for integrity verification.
  
  Args:
    filepath - Path to the file
    algorithm - Hash algorithm ('MD5', 'SHA-1', 'SHA-256', etc.)
  
  Returns:
    Hex digest of the file hash"
  [filepath & {:keys [algorithm] :or {algorithm "SHA-256"}}]
  (calculate-hash (File. filepath) algorithm))

;; Folder operations

(defn create-folder
  "Create a new folder in EYWA file service.
  
  Args:
    name - Folder name
    parent-folder-uuid - UUID of parent folder (optional, nil for root)
  
  Returns:
    Core.async channel containing folder information map or exception"
  [name & {:keys [parent-folder-uuid]}]
  (go
    (try
      (let [_ (info (str "Creating folder: " name))

            mutation "mutation CreateFolder($folder: FolderInput!) {
                       syncFolder(data: $folder) {
                         euuid
                         name
                         created_at
                         parent {
                           euuid
                           name
                         }
                       }
                     }"

            variables (cond-> {:folder {:name name}}
                        parent-folder-uuid (assoc-in [:folder :parent :euuid] parent-folder-uuid))

            {:keys [error result]} (<! (graphql mutation variables))

            _ (when error (throw (ex-info (str "Failed to create folder: " error) {:error error})))

            folder (get-in result [:data :syncFolder])]

        (info (str "Folder created: " name " -> " (:euuid folder)))
        folder)

      (catch Exception e
        (error (str "Failed to create folder: " (.getMessage e)))
        e))))

(defn list-folders
  "List folders in EYWA file service.
  
  Args:
    options - Map of filter options:
      :limit - Maximum number of folders to return
      :name-pattern - Filter by name pattern (SQL LIKE)
      :parent-folder-uuid - Filter by parent folder UUID (nil for root folders)
  
  Returns:
    Core.async channel containing list of folder maps or exception"
  [& {:keys [limit name-pattern parent-folder-uuid]}]
  (go
    (try
      (let [_ (debug (str "Listing folders (limit=" limit ")"))

            query "query ListFolders($limit: Int, $where: FolderWhereInput) {
                     searchFolder(_limit: $limit, _where: $where, _order_by: {name: asc}) {
                       euuid
                       name
                       created_at
                       parent {
                         euuid
                         name
                       }
                     }
                   }"

            where-conditions (cond-> []
                               name-pattern (conj {:name {:_ilike (str "%" name-pattern "%")}})
                               (some? parent-folder-uuid)
                               (conj (if parent-folder-uuid
                                       {:parent {:euuid {:_eq parent-folder-uuid}}}
                                       {:parent {:_is_null true}})))

            variables (cond-> {}
                        limit (assoc :limit limit)
                        (seq where-conditions) (assoc :where
                                                      (if (= 1 (count where-conditions))
                                                        (first where-conditions)
                                                        {:_and where-conditions})))

            {:keys [error result]} (<! (graphql query variables))

            _ (when error (throw (ex-info (str "Failed to list folders: " error) {:error error})))

            folders (get-in result [:data :searchFolder])]

        (debug (str "Found " (count folders) " folders"))
        folders)

      (catch Exception e
        (error (str "Failed to list folders: " (.getMessage e)))
        e))))

(defn get-folder-info
  "Get information about a specific folder.
  
  Args:
    folder-uuid - UUID of the folder
  
  Returns:
    Core.async channel containing folder information map or nil if not found"
  [folder-uuid]
  (go
    (try
      (let [query "query GetFolder($uuid: UUID!) {
                     getFolder(euuid: $uuid) {
                       euuid
                       name
                       created_at
                       parent {
                         euuid
                         name
                       }
                     }
                   }"

            {:keys [error result]} (<! (graphql query {:uuid folder-uuid}))]

        (if error
          (do
            (debug (str "Folder not found or error: " error))
            nil)
          (get-in result [:data :getFolder])))

      (catch Exception e
        (debug (str "Folder not found or error: " (.getMessage e)))
        nil))))

(defn delete-folder
  "Delete a folder from EYWA file service.
  
  Note: Folder must be empty (no files or subfolders) to be deleted.
  
  Args:
    folder-uuid - UUID of the folder to delete
  
  Returns:
    Core.async channel containing true if deletion successful, false otherwise"
  [folder-uuid]
  (go
    (try
      (let [mutation "mutation DeleteFolder($uuid: UUID!) {
                       deleteFolder(euuid: $uuid)
                     }"

            {:keys [error result]} (<! (graphql mutation {:uuid folder-uuid}))

            _ (when error (throw (ex-info (str "Failed to delete folder: " error) {:error error})))

            success? (get-in result [:data :deleteFolder])]

        (if success?
          (info (str "Folder deleted: " folder-uuid))
          (warn (str "Folder deletion failed: " folder-uuid)))

        success?)

      (catch Exception e
        (error (str "Failed to delete folder: " (.getMessage e)))
        false))))

;; Convenience functions

;; Convenience functions removed - users should compose core functions themselves
;; See REFACTORING_PLAN.md for migration guide

;; removed - see migration guide above

;; Data processing helpers

;; removed - see migration guide above

;; removed - see migration guide above

;; removed - see migration guide above

;; removed - see migration guide above
