(ns emmy.portal.deps
  "Functions for resolving npm dependencies declared in `deps.cljs` files on the
  classpath.

  This code is a simplified version of code in
  thheller's [shadow-cljs](https://github.com/thheller/shadow-cljs/blob/master/src/main/shadow/cljs/devtools/server/npm_deps.clj)
  library and
  the [clerk-utils](https://github.com/mentat-collective/Clerk-Utils/blob/main/src/mentat/clerk_utils/build/shadow.clj)
  library."
  (:require [clojure.data.json :as json]
            [clojure.edn :as edn]
            [clojure.java.shell :refer [sh]]
            [clojure.string]
            [clojure.java.io :as io]))

(def ^:private windows?
  (clojure.string/starts-with?
   (System/getProperty "os.name")
   "Windows"))

(def npm-cmd
  "System-specific NPM command, tuned for Windows or non-Windows."
  (if windows? "npm.cmd" "npm"))

(defn dep->str
  "Given a dependency name (specified as a key in the `:npm-deps` map of
  `deps.cljs`), returns an npm-compatible (string) name for the dependency."
  [dep-id]
  (cond (keyword? dep-id)

        ;; :some/foo? :react :@some/scoped?
        (subs (str dep-id) 1)

        (symbol? dep-id)
        (str dep-id)

        (string? dep-id)
        dep-id

        :else
        (throw
         (ex-info (format "invalid dependency id %s" dep-id) {}))))

(defn get-deps-from-classpath
  "Returns a sequence of maps of the form {:id <str> :version <str> :url
  <url-of-this-deps.cljs>} for every dependency specified via the `:npm-deps`
  key of each `deps.cljs` file on the classpath."
  []
  (let [xform (map
               (fn [url]
                 (-> (slurp url)
                     (edn/read-string)
                     (select-keys [:npm-deps])
                     (assoc :url url))))
        files (-> (Thread/currentThread)
                  (.getContextClassLoader)
                  (.getResources "deps.cljs")
                  (enumeration-seq))
        deps  (into [] xform files)]
    (for [{:keys [url npm-deps]} deps
          [dep-id dep-version] npm-deps]
      {:id (dep->str dep-id)
       :version dep-version
       :url url})))

(defn resolve-conflicts
  "Given a sequence of dependencies, returns a sequence of dependencies distinct
  by ID generated by picking the first one on the classpath.

  Not ideal, but this is how shadow currently does it!"
  [deps]
  (vals
   (reduce
    (fn [acc {:keys [id] :as dep}]
      (update acc id #(or % dep)))
    {}
    deps)))

(defn read-package-json
  "Returns a map with the contents of package.json if it exists in
  `install-dir` (default \".\"), {} otherwise."
  ([] (read-package-json "."))
  ([install-dir]
   (let [package-json-file (io/file install-dir "package.json")]
     (if-not (.exists package-json-file)
       {}
       (-> (slurp package-json-file)
           (json/read-str))))))

(defn is-installed?
  "Returns true if the supplied dependency is already present in some entry in the
  parsed `package-json`, false otherwise."
  [{:keys [id]} package-json]
  (or (get-in package-json ["dependencies" id])
      (get-in package-json ["devDependencies" id])
      (get-in package-json ["peerDependencies" id])))

(defn sh-print
  "Takes a vector of command pieces, prints the command and prints the results."
  [command]
  (println (str "running: " (clojure.string/join " " command)))
  (println
   (:out
    (apply sh command))))

(defn ^:no-doc install-deps
  "Given a sequence of dependencies, runs `npm install --no-save` for each
  id/version combination."
  [deps]
  (when (seq deps)
    (sh-print
     (into [npm-cmd "install" "--no-save"]
           (for [{:keys [id version]} deps]
             (str id "@" version))))))

(defn npm-install!
  "This command:

  - Triggers an `npm install` (if ./package.json is present)
  - Installs any dependency referenced by a `deps.cljs` file on the classpath

  The `npm install` makes sure that if some dependency specified in a
  `deps.cljs` is already present in `package.json` that it actually ends up in
  `node_modules`."
  []
  (let [package-json (read-package-json)
        deps         (->> (get-deps-from-classpath)
                          (resolve-conflicts)
                          (remove #(is-installed? % package-json)))]
    (when (seq package-json)
      (sh-print [npm-cmd "install"]))
    (install-deps deps)))
