package org.logolith.kzgo;

import com.sun.jna.Pointer;
import com.sun.jna.Memory;
import com.sun.jna.ptr.IntByReference;
import java.util.Objects;
import java.util.Arrays;
import java.util.List;

/**
 * Public Java wrapper for the KZG Bridge native library.
 * Provides static methods for interacting with the KZG functions.
 */
public class KZG {

    private static final int SUCCESS = 0;
    // Define other error constants if needed for specific exception messages
    private static final int ERR_INVALID_INPUT = 1;
    private static final int ERR_COMMIT_FAILED = 2;
    private static final int ERR_OPEN_FAILED = 3;
    private static final int ERR_VERIFY_FAILED = 4;
    private static final int ERR_INTERNAL = 5;
    private static final int ERR_DESERIALIZE = 6;
    private static final int ERR_SRS_MISMATCH = 7; // Added based on Go changes
    private static final int ERR_HASH_TO_FIELD_FAILED = 8; // Added based on Go changes

    // Helper to create input CBuffer from byte array, returning ByValue for direct use
    private static CBuffer.ByValue createInputBufferByValue(byte[] data) {
        if (data == null || data.length == 0) {
            return new CBuffer.ByValue(); // Represents {data: null, len: 0}
        }
        Memory mem = new Memory(data.length);
        mem.write(0, data, 0, data.length);
        CBuffer.ByValue buf = new CBuffer.ByValue(); // Create ByValue instance
        buf.data = mem;
        buf.len = data.length;
        return buf;
    }

    /**
     * Computes the KZG commitment for a polynomial defined by its coefficients.
     *
     * @param polyCoeffsBytes A single byte array containing the concatenated 32-byte Fr elements (coefficients),
     *                        MUST match the size defined by the loaded SRS.
     * @return The KZG commitment as a byte array.
     * @throws KZGException If the native call fails or if the number of coefficients doesn't match the SRS size.
     */
    public static byte[] commit(byte[] polyCoeffsBytes) throws KZGException {
        CBuffer.ByValue inputBuf = createInputBufferByValue(polyCoeffsBytes);
        CBuffer resultBuf = new CBuffer(); // Output struct passed by ref
        IntByReference errorCodeRef = new IntByReference(); // Output error code passed by ref
        int overallStatus = -1;
        try {
            overallStatus = KZGBridgeNative.INSTANCE.commit_to_poly(inputBuf, resultBuf, errorCodeRef);

            if (overallStatus != SUCCESS || errorCodeRef.getValue() != SUCCESS) {
                 int finalErrorCode = (overallStatus != SUCCESS) ? overallStatus : errorCodeRef.getValue();
                 // Free potentially allocated buffer if Go failed after allocation
                 if (resultBuf.data != null) {
                     KZGBridgeNative.INSTANCE.free_buffer(resultBuf.asByValue());
                 }
                 throw new KZGException("commit_to_poly failed", finalErrorCode);
             }
            // Success: Read bytes from the result buffer populated by Go
            byte[] resultBytes = resultBuf.readBytes();
            return resultBytes;

        } finally {
            // Free the C buffer returned by the native call
             // Free only on SUCCESS and if data is not null. Error cases handled above.
             if (overallStatus == SUCCESS && errorCodeRef.getValue() == SUCCESS && resultBuf.data != null) {
                 KZGBridgeNative.INSTANCE.free_buffer(resultBuf.asByValue());
             }
            // Input buffer memory (allocated in createInputBufferByValue) is managed by JNA GC
        }
    }

    /**
     * Computes the KZG commitment by performing interpolation on evaluations within the bridge.
     *
     * @param evaluationsBytes A single byte array containing the concatenated 32-byte Fr elements (evaluations/y-coordinates).
     *                         The number of evaluations determines the polynomial degree, which must be less than the SRS size.
     * @return The KZG commitment as a byte array.
     * @throws KZGException If the native call fails, input is invalid, or the interpolated polynomial degree exceeds the SRS size.
     */
    public static byte[] commitFromEvaluations(byte[] evaluationsBytes) throws KZGException {
        // Optional but good practice: Basic Java-side validation
        if (evaluationsBytes == null || evaluationsBytes.length == 0) {
            throw new KZGException("Evaluations cannot be null or empty", ERR_INVALID_INPUT);
        }
        if (evaluationsBytes.length % 32 != 0) { // Assuming frElementBytes is 32
            throw new KZGException("Evaluations byte length must be multiple of 32", ERR_INVALID_INPUT);
        }

        CBuffer.ByValue inputBuf = createInputBufferByValue(evaluationsBytes);
        CBuffer resultBuf = new CBuffer(); // Output struct passed by ref
        IntByReference errorCodeRef = new IntByReference(); // Output error code passed by ref
        int overallStatus = -1;

        try {
            // Call the new native function
            overallStatus = KZGBridgeNative.INSTANCE.commit_from_evaluations(inputBuf, resultBuf, errorCodeRef);

            if (overallStatus != SUCCESS || errorCodeRef.getValue() != SUCCESS) {
                 int finalErrorCode = (overallStatus != SUCCESS) ? overallStatus : errorCodeRef.getValue();
                 // Free potentially allocated buffer if Go failed after allocation
                 if (resultBuf.data != null) {
                     KZGBridgeNative.INSTANCE.free_buffer(resultBuf.asByValue());
                 }
                 throw new KZGException("commit_from_evaluations failed", finalErrorCode);
             }
            // Success: Read bytes from the result buffer populated by Go
            byte[] resultBytes = resultBuf.readBytes();
            return resultBytes;

        } finally {
            // Free the C buffer returned by the native call
             if (overallStatus == SUCCESS && errorCodeRef.getValue() == SUCCESS && resultBuf.data != null) {
                 KZGBridgeNative.INSTANCE.free_buffer(resultBuf.asByValue());
             }
            // Input buffer memory (allocated in createInputBufferByValue) is managed by JNA GC
        }
    }

    /**
     * Computes the KZG opening proof for a polynomial at a given point.
     *
     * @param polyCoeffsBytes A single byte array containing the concatenated 32-byte Fr elements (coefficients).
     *                        The number of coefficients MUST match the SRS size.
     * @param pointZBytes The 32-byte representation of the evaluation point Fr element.
     * @return An OpenResultData object containing the proof (H point) and claimed value bytes.
     * @throws KZGException If the native call fails.
     */
    public static OpenResultData open(byte[] polyCoeffsBytes, byte[] pointZBytes) throws KZGException {
        CBuffer.ByValue coeffsBuf = createInputBufferByValue(polyCoeffsBytes);
        CBuffer.ByValue pointBuf = createInputBufferByValue(pointZBytes);
        CBuffer proofHBuf = new CBuffer(); // Output
        CBuffer claimedValueBuf = new CBuffer(); // Output
        IntByReference errorCodeRef = new IntByReference(); // Output
        int overallStatus = -1;

        try {
            overallStatus = KZGBridgeNative.INSTANCE.open_poly(coeffsBuf, pointBuf, proofHBuf, claimedValueBuf, errorCodeRef);

            if (overallStatus != SUCCESS || errorCodeRef.getValue() != SUCCESS) {
                int finalErrorCode = (overallStatus != SUCCESS) ? overallStatus : errorCodeRef.getValue();
                // Free potentially allocated buffers on error
                if (proofHBuf.data != null) KZGBridgeNative.INSTANCE.free_buffer(proofHBuf.asByValue());
                if (claimedValueBuf.data != null) KZGBridgeNative.INSTANCE.free_buffer(claimedValueBuf.asByValue());
                throw new KZGException("open_poly failed", finalErrorCode);
            }

            // Read data before freeing buffers
            byte[] proofH = proofHBuf.readBytes();
            byte[] claimedValue = claimedValueBuf.readBytes();
            return new OpenResultData(proofH, claimedValue);
        } finally {
            // Free BOTH C buffers returned by the native call
            // Free only on SUCCESS and if data is not null. Error cases handled above.
            if (overallStatus == SUCCESS && errorCodeRef.getValue() == SUCCESS) {
                 if (proofHBuf.data != null) {
                     KZGBridgeNative.INSTANCE.free_buffer(proofHBuf.asByValue());
                 }
                 if (claimedValueBuf.data != null) {
                    KZGBridgeNative.INSTANCE.free_buffer(claimedValueBuf.asByValue());
                 }
            }
            // Input buffer memory is managed by JNA GC
        }
    }

    /**
     * Verifies a KZG opening proof using the globally loaded VK (Server-side).
     *
     * @param commitmentBytes The commitment bytes.
     * @param proofHBytes The proof bytes (H point).
     * @param pointZBytes The evaluation point bytes.
     * @param claimedValueBytes The claimed value bytes.
     * @return true if the proof is valid, false otherwise.
     * @throws KZGException If the native call encounters an error other than verification failure.
     */
    public static boolean verify(byte[] commitmentBytes, byte[] proofHBytes, byte[] pointZBytes, byte[] claimedValueBytes) throws KZGException {
        CBuffer.ByValue commitmentBuf = createInputBufferByValue(commitmentBytes);
        CBuffer.ByValue proofHBuf = createInputBufferByValue(proofHBytes);
        CBuffer.ByValue pointZBuf = createInputBufferByValue(pointZBytes);
        CBuffer.ByValue claimedValueBuf = createInputBufferByValue(claimedValueBytes);

        int result = KZGBridgeNative.INSTANCE.verify_poly_proof(commitmentBuf, proofHBuf, pointZBuf, claimedValueBuf);

        if (result == SUCCESS) {
            return true;
        } else if (result == ERR_VERIFY_FAILED) {
            return false;
        } else {
            // Throw for other errors (deserialization, input, etc.)
            throw new KZGException("verify_poly_proof failed", result);
        }
        // Input buffer memory is managed by JNA GC
    }

    /**
     * Verifies a KZG opening proof using a provided Verifying Key (Client-side).
     *
     * @param commitmentBytes The commitment bytes.
     * @param proofHBytes The proof bytes (H point).
     * @param pointZBytes The evaluation point bytes.
     * @param claimedValueBytes The claimed value bytes.
     * @param vkBytes The serialized Verifying Key bytes.
     * @return true if the proof is valid, false otherwise.
     * @throws KZGException If the native call encounters an error other than verification failure.
     */
    public static boolean verifyWithVk(byte[] commitmentBytes, byte[] proofHBytes, byte[] pointZBytes, byte[] claimedValueBytes, byte[] vkBytes) throws KZGException {
        CBuffer.ByValue commitmentBuf = createInputBufferByValue(commitmentBytes);
        CBuffer.ByValue proofHBuf = createInputBufferByValue(proofHBytes);
        CBuffer.ByValue pointZBuf = createInputBufferByValue(pointZBytes);
        CBuffer.ByValue claimedValueBuf = createInputBufferByValue(claimedValueBytes);
        CBuffer.ByValue vkBuf = createInputBufferByValue(vkBytes);

        int result = KZGBridgeNative.INSTANCE.verify_poly_proof_with_vk(commitmentBuf, proofHBuf, pointZBuf, claimedValueBuf, vkBuf);

        if (result == SUCCESS) {
            return true;
        } else if (result == ERR_VERIFY_FAILED) {
            return false;
        } else {
            // Throw for other errors (deserialization, input, etc.)
            throw new KZGException("verify_poly_proof_with_vk failed", result);
        }
        // Input buffer memory is managed by JNA GC
    }

    /**
     * Performs polynomial interpolation given values (y-coordinates) corresponding to domain 0, 1, 2, ...
     *
     * @param valuesBytes A single byte array containing the concatenated 32-byte Fr elements (y-coordinates).
     * @return The coefficients of the interpolated polynomial as a byte array.
     * @throws KZGException If the native call fails or if input is invalid.
     */
    public static byte[] interpolate(byte[] valuesBytes) throws KZGException {
        if (valuesBytes == null || valuesBytes.length == 0) {
            throw new KZGException("Interpolation values cannot be null or empty", ERR_INVALID_INPUT);
        }

        CBuffer.ByValue valuesBuf = createInputBufferByValue(valuesBytes);
        // Prepare output parameters (passed by reference)
        CBuffer resultCoeffsBuf = new CBuffer(); // JNA will pass a pointer to this
        IntByReference errorCodeRef = new IntByReference(); // JNA will pass a pointer to this

        int overallStatus = -1; // Default to an error status

        try {
            overallStatus = KZGBridgeNative.INSTANCE.interpolate_poly(valuesBuf, resultCoeffsBuf, errorCodeRef);

            // Check both the function's return status and the output error code
            if (overallStatus != SUCCESS || errorCodeRef.getValue() != SUCCESS) {
                int finalErrorCode = (overallStatus != SUCCESS) ? overallStatus : errorCodeRef.getValue();
                // Important: Check if Go allocated the buffer before failing. If so, we still need to free it.
                if (resultCoeffsBuf.data != null) {
                    // free_buffer expects ByValue, so create one from the output buffer
                    CBuffer.ByValue bufferToFree = new CBuffer.ByValue();
                    bufferToFree.data = resultCoeffsBuf.data;
                    bufferToFree.len = resultCoeffsBuf.len;
                    KZGBridgeNative.INSTANCE.free_buffer(bufferToFree);
                }
                throw new KZGException("interpolate_poly failed", finalErrorCode);
            }

            // Success: Read bytes from the result buffer populated by Go
            byte[] resultBytes = resultCoeffsBuf.readBytes();
            return resultBytes;

        } finally {
            // Free the C buffer *only if* the call was successful (overallStatus=SUCCESS and errorCodeRef=SUCCESS)
            // and the data pointer is not null.
            if (overallStatus == SUCCESS && errorCodeRef.getValue() == SUCCESS && resultCoeffsBuf != null && resultCoeffsBuf.data != null) {
                 // free_buffer expects ByValue, create one from the output buffer
                 CBuffer.ByValue bufferToFree = new CBuffer.ByValue();
                 bufferToFree.data = resultCoeffsBuf.data;
                 bufferToFree.len = resultCoeffsBuf.len;
                 KZGBridgeNative.INSTANCE.free_buffer(bufferToFree);
            } else {
            }
            // Input buffer memory (allocated in createInputBufferByValue) is managed by JNA GC
        }
    }

    /**
     * Hashes arbitrary message bytes to a single Fr field element using a domain separation tag.
     *
     * @param msgBytes The message bytes to hash.
     * @param domainBytes The domain separation tag bytes.
     * @return The resulting Fr element as a 32-byte array.
     * @throws KZGException If the native hash-to-field call fails.
     */
    public static byte[] hashToFr(byte[] msgBytes, byte[] domainBytes) throws KZGException {
        CBuffer.ByValue msgBuf = createInputBufferByValue(msgBytes);
        CBuffer.ByValue domainBuf = createInputBufferByValue(domainBytes);
        CBuffer resultBuf = new CBuffer(); // Output
        IntByReference errorCodeRef = new IntByReference(); // Output
        int overallStatus = -1;
        try {
            overallStatus = KZGBridgeNative.INSTANCE.hash_to_fr(msgBuf, domainBuf, resultBuf, errorCodeRef);

            if (overallStatus != SUCCESS || errorCodeRef.getValue() != SUCCESS) {
                 int finalErrorCode = (overallStatus != SUCCESS) ? overallStatus : errorCodeRef.getValue();
                 // Free potentially allocated buffer on error
                 if (resultBuf.data != null) {
                    KZGBridgeNative.INSTANCE.free_buffer(resultBuf.asByValue());
                 }
                 throw new KZGException("hash_to_fr failed", finalErrorCode);
             }
            // Use the CBuffer method to read bytes
            byte[] frBytes = resultBuf.readBytes();
            return frBytes;
        } finally {
            // Free the C buffer returned by the native call
             // Free only on SUCCESS and if data is not null. Error cases handled above.
             if (overallStatus == SUCCESS && errorCodeRef.getValue() == SUCCESS && resultBuf.data != null) {
                 KZGBridgeNative.INSTANCE.free_buffer(resultBuf.asByValue());
             }
            // Input buffer memory (allocated in createInputBufferByValue) is managed by JNA GC
        }
    }

    /** Simple data class to hold the results from the open function. */
    public static class OpenResultData {
        public final byte[] proofH;
        public final byte[] claimedValue;

        public OpenResultData(byte[] proofH, byte[] claimedValue) {
            this.proofH = proofH;
            this.claimedValue = claimedValue;
        }
    }

    /** Custom exception class for KZG errors. */
    public static class KZGException extends Exception {
        public final int errorCode;

        public KZGException(String message, int errorCode) {
            super(message + " (code: " + errorCode + ")");
            this.errorCode = errorCode;
        }
    }

    /**
     * Computes the KZG commitment directly from serialized raw node data.
     * This performs hashing, evaluation generation, interpolation, padding, and commitment within Go.
     *
     * @param nodeDataBytes Serialized node data (type, count, keys, values/commits) according to the agreed format.
     * @return The KZG commitment as a byte array.
     * @throws KZGException If the native call fails.
     */
    public static byte[] commitNode(byte[] nodeDataBytes) throws KZGException {
        CBuffer.ByValue inputBuf = createInputBufferByValue(nodeDataBytes);
        CBuffer resultBuf = new CBuffer();
        IntByReference errorCodeRef = new IntByReference();
        int overallStatus = -1;
        try {
            overallStatus = KZGBridgeNative.INSTANCE.commit_node(inputBuf, resultBuf, errorCodeRef);
            if (overallStatus != SUCCESS || errorCodeRef.getValue() != SUCCESS) {
                int finalErrorCode = (overallStatus != SUCCESS) ? overallStatus : errorCodeRef.getValue();
                if (resultBuf.data != null) {
                    KZGBridgeNative.INSTANCE.free_buffer(resultBuf.asByValue());
                }
                throw new KZGException("commit_node failed", finalErrorCode);
            }
            byte[] resultBytes = resultBuf.readBytes();
            return resultBytes;
        } finally {
            if (overallStatus == SUCCESS && errorCodeRef.getValue() == SUCCESS && resultBuf.data != null) {
                KZGBridgeNative.INSTANCE.free_buffer(resultBuf.asByValue());
            }
            // Input buffer memory managed by JNA GC
        }
    }

    /**
     * Parses node data, computes polynomial evaluations, interpolates, pads to SRS size,
     * and returns the serialized padded polynomial coefficients.
     *
     * @param nodeDataBytes The serialized node data (following the expected format).
     * @return The byte array of the serialized padded polynomial coefficients.
     * @throws KZGException if the underlying Go call fails.
     */
    public static synchronized byte[] interpolateNode(byte[] nodeDataBytes) throws KZGException {
        Objects.requireNonNull(nodeDataBytes, "Node data bytes cannot be null");

        // Prepare input buffer for JNA
        CBuffer.ByValue inputBuf = createInputBufferByValue(nodeDataBytes);

        // Prepare output pointers for JNA
        CBuffer resultBuf = new CBuffer(); // Passed by reference to native code
        IntByReference errorCodeRef = new IntByReference();

        int overallStatus = -1;
        try {
            // Call the native Go function via the JNA interface
            overallStatus = KZGBridgeNative.INSTANCE.interpolate_node(inputBuf, resultBuf, errorCodeRef);

            int errorCode = errorCodeRef.getValue();

            // Check error code BEFORE accessing result buffer
            if (errorCode != SUCCESS) {
                // No need to free resultBuf.data here, Go should return nil data on error
                throw new KZGException("interpolate_node failed", errorCode);
            }
            if (overallStatus != SUCCESS){
                 // Should ideally match errorCode, but check just in case
                 throw new KZGException("interpolate_node status check failed", overallStatus);
            }

            // --- Process successful result ---
            byte[] coeffBytes = resultBuf.readBytes(); // Use helper method if available

            return coeffBytes;

        } finally {
            // Free the C buffer returned by the native call if successful
            if (overallStatus == SUCCESS && errorCodeRef.getValue() == SUCCESS && resultBuf.data != null) {
                KZGBridgeNative.INSTANCE.free_buffer(resultBuf.asByValue());
            }
            // Input buffer memory (if allocated with Memory) is managed by JNA GC
        }
    }
} 
