if(typeof Math.imul == "undefined" || (Math.imul(0xffffffff,5) == 0)) {
    Math.imul = function (a, b) {
        var ah  = (a >>> 16) & 0xffff;
        var al = a & 0xffff;
        var bh  = (b >>> 16) & 0xffff;
        var bl = b & 0xffff;
        // the shift by 0 fixes the sign on the high part
        // the final |0 converts the unsigned value into a signed value
        return ((al * bl) + (((ah * bl + al * bh) << 16) >>> 0)|0);
    }
}

// CodeMirror, copyright (c) by Marijn Haverbeke and others
// Distributed under an MIT license: http://codemirror.net/LICENSE

// This is CodeMirror (http://codemirror.net), a code editor
// implemented in JavaScript on top of the browser's DOM.
//
// You can find some technical background for some of the code below
// at http://marijnhaverbeke.nl/blog/#cm-internals .

(function (global, factory) {
  typeof exports === 'object' && typeof module !== 'undefined' ? module.exports = factory() :
  typeof define === 'function' && define.amd ? define(factory) :
  (global.CodeMirror = factory());
}(this, (function () { 'use strict';

// Kludges for bugs and behavior differences that can't be feature
// detected are enabled based on userAgent etc sniffing.
var userAgent = navigator.userAgent
var platform = navigator.platform

var gecko = /gecko\/\d/i.test(userAgent)
var ie_upto10 = /MSIE \d/.test(userAgent)
var ie_11up = /Trident\/(?:[7-9]|\d{2,})\..*rv:(\d+)/.exec(userAgent)
var edge = /Edge\/(\d+)/.exec(userAgent)
var ie = ie_upto10 || ie_11up || edge
var ie_version = ie && (ie_upto10 ? document.documentMode || 6 : +(edge || ie_11up)[1])
var webkit = !edge && /WebKit\//.test(userAgent)
var qtwebkit = webkit && /Qt\/\d+\.\d+/.test(userAgent)
var chrome = !edge && /Chrome\//.test(userAgent)
var presto = /Opera\//.test(userAgent)
var safari = /Apple Computer/.test(navigator.vendor)
var mac_geMountainLion = /Mac OS X 1\d\D([8-9]|\d\d)\D/.test(userAgent)
var phantom = /PhantomJS/.test(userAgent)

var ios = !edge && /AppleWebKit/.test(userAgent) && /Mobile\/\w+/.test(userAgent)
// This is woefully incomplete. Suggestions for alternative methods welcome.
var mobile = ios || /Android|webOS|BlackBerry|Opera Mini|Opera Mobi|IEMobile/i.test(userAgent)
var mac = ios || /Mac/.test(platform)
var chromeOS = /\bCrOS\b/.test(userAgent)
var windows = /win/i.test(platform)

var presto_version = presto && userAgent.match(/Version\/(\d*\.\d*)/)
if (presto_version) { presto_version = Number(presto_version[1]) }
if (presto_version && presto_version >= 15) { presto = false; webkit = true }
// Some browsers use the wrong event properties to signal cmd/ctrl on OS X
var flipCtrlCmd = mac && (qtwebkit || presto && (presto_version == null || presto_version < 12.11))
var captureRightClick = gecko || (ie && ie_version >= 9)

function classTest(cls) { return new RegExp("(^|\\s)" + cls + "(?:$|\\s)\\s*") }

var rmClass = function(node, cls) {
  var current = node.className
  var match = classTest(cls).exec(current)
  if (match) {
    var after = current.slice(match.index + match[0].length)
    node.className = current.slice(0, match.index) + (after ? match[1] + after : "")
  }
}

function removeChildren(e) {
  for (var count = e.childNodes.length; count > 0; --count)
    { e.removeChild(e.firstChild) }
  return e
}

function removeChildrenAndAdd(parent, e) {
  return removeChildren(parent).appendChild(e)
}

function elt(tag, content, className, style) {
  var e = document.createElement(tag)
  if (className) { e.className = className }
  if (style) { e.style.cssText = style }
  if (typeof content == "string") { e.appendChild(document.createTextNode(content)) }
  else if (content) { for (var i = 0; i < content.length; ++i) { e.appendChild(content[i]) } }
  return e
}

var range
if (document.createRange) { range = function(node, start, end, endNode) {
  var r = document.createRange()
  r.setEnd(endNode || node, end)
  r.setStart(node, start)
  return r
} }
else { range = function(node, start, end) {
  var r = document.body.createTextRange()
  try { r.moveToElementText(node.parentNode) }
  catch(e) { return r }
  r.collapse(true)
  r.moveEnd("character", end)
  r.moveStart("character", start)
  return r
} }

function contains(parent, child) {
  if (child.nodeType == 3) // Android browser always returns false when child is a textnode
    { child = child.parentNode }
  if (parent.contains)
    { return parent.contains(child) }
  do {
    if (child.nodeType == 11) { child = child.host }
    if (child == parent) { return true }
  } while (child = child.parentNode)
}

function activeElt() {
  // IE and Edge may throw an "Unspecified Error" when accessing document.activeElement.
  // IE < 10 will throw when accessed while the page is loading or in an iframe.
  // IE > 9 and Edge will throw when accessed in an iframe if document.body is unavailable.
  var activeElement
  try {
    activeElement = document.activeElement
  } catch(e) {
    activeElement = document.body || null
  }
  while (activeElement && activeElement.root && activeElement.root.activeElement)
    { activeElement = activeElement.root.activeElement }
  return activeElement
}

function addClass(node, cls) {
  var current = node.className
  if (!classTest(cls).test(current)) { node.className += (current ? " " : "") + cls }
}
function joinClasses(a, b) {
  var as = a.split(" ")
  for (var i = 0; i < as.length; i++)
    { if (as[i] && !classTest(as[i]).test(b)) { b += " " + as[i] } }
  return b
}

var selectInput = function(node) { node.select() }
if (ios) // Mobile Safari apparently has a bug where select() is broken.
  { selectInput = function(node) { node.selectionStart = 0; node.selectionEnd = node.value.length } }
else if (ie) // Suppress mysterious IE10 errors
  { selectInput = function(node) { try { node.select() } catch(_e) {} } }

function bind(f) {
  var args = Array.prototype.slice.call(arguments, 1)
  return function(){return f.apply(null, args)}
}

function copyObj(obj, target, overwrite) {
  if (!target) { target = {} }
  for (var prop in obj)
    { if (obj.hasOwnProperty(prop) && (overwrite !== false || !target.hasOwnProperty(prop)))
      { target[prop] = obj[prop] } }
  return target
}

// Counts the column offset in a string, taking tabs into account.
// Used mostly to find indentation.
function countColumn(string, end, tabSize, startIndex, startValue) {
  if (end == null) {
    end = string.search(/[^\s\u00a0]/)
    if (end == -1) { end = string.length }
  }
  for (var i = startIndex || 0, n = startValue || 0;;) {
    var nextTab = string.indexOf("\t", i)
    if (nextTab < 0 || nextTab >= end)
      { return n + (end - i) }
    n += nextTab - i
    n += tabSize - (n % tabSize)
    i = nextTab + 1
  }
}

var Delayed = function() {this.id = null};
Delayed.prototype.set = function (ms, f) {
  clearTimeout(this.id)
  this.id = setTimeout(f, ms)
};

function indexOf(array, elt) {
  for (var i = 0; i < array.length; ++i)
    { if (array[i] == elt) { return i } }
  return -1
}

// Number of pixels added to scroller and sizer to hide scrollbar
var scrollerGap = 30

// Returned or thrown by various protocols to signal 'I'm not
// handling this'.
var Pass = {toString: function(){return "CodeMirror.Pass"}}

// Reused option objects for setSelection & friends
var sel_dontScroll = {scroll: false};
var sel_mouse = {origin: "*mouse"};
var sel_move = {origin: "+move"};
// The inverse of countColumn -- find the offset that corresponds to
// a particular column.
function findColumn(string, goal, tabSize) {
  for (var pos = 0, col = 0;;) {
    var nextTab = string.indexOf("\t", pos)
    if (nextTab == -1) { nextTab = string.length }
    var skipped = nextTab - pos
    if (nextTab == string.length || col + skipped >= goal)
      { return pos + Math.min(skipped, goal - col) }
    col += nextTab - pos
    col += tabSize - (col % tabSize)
    pos = nextTab + 1
    if (col >= goal) { return pos }
  }
}

var spaceStrs = [""]
function spaceStr(n) {
  while (spaceStrs.length <= n)
    { spaceStrs.push(lst(spaceStrs) + " ") }
  return spaceStrs[n]
}

function lst(arr) { return arr[arr.length-1] }

function map(array, f) {
  var out = []
  for (var i = 0; i < array.length; i++) { out[i] = f(array[i], i) }
  return out
}

function insertSorted(array, value, score) {
  var pos = 0, priority = score(value)
  while (pos < array.length && score(array[pos]) <= priority) { pos++ }
  array.splice(pos, 0, value)
}

function nothing() {}

function createObj(base, props) {
  var inst
  if (Object.create) {
    inst = Object.create(base)
  } else {
    nothing.prototype = base
    inst = new nothing()
  }
  if (props) { copyObj(props, inst) }
  return inst
}

var nonASCIISingleCaseWordChar = /[\u00df\u0587\u0590-\u05f4\u0600-\u06ff\u3040-\u309f\u30a0-\u30ff\u3400-\u4db5\u4e00-\u9fcc\uac00-\ud7af]/
function isWordCharBasic(ch) {
  return /\w/.test(ch) || ch > "\x80" &&
    (ch.toUpperCase() != ch.toLowerCase() || nonASCIISingleCaseWordChar.test(ch))
}
function isWordChar(ch, helper) {
  if (!helper) { return isWordCharBasic(ch) }
  if (helper.source.indexOf("\\w") > -1 && isWordCharBasic(ch)) { return true }
  return helper.test(ch)
}

function isEmpty(obj) {
  for (var n in obj) { if (obj.hasOwnProperty(n) && obj[n]) { return false } }
  return true
}

// Extending unicode characters. A series of a non-extending char +
// any number of extending chars is treated as a single unit as far
// as editing and measuring is concerned. This is not fully correct,
// since some scripts/fonts/browsers also treat other configurations
// of code points as a group.
var extendingChars = /[\u0300-\u036f\u0483-\u0489\u0591-\u05bd\u05bf\u05c1\u05c2\u05c4\u05c5\u05c7\u0610-\u061a\u064b-\u065e\u0670\u06d6-\u06dc\u06de-\u06e4\u06e7\u06e8\u06ea-\u06ed\u0711\u0730-\u074a\u07a6-\u07b0\u07eb-\u07f3\u0816-\u0819\u081b-\u0823\u0825-\u0827\u0829-\u082d\u0900-\u0902\u093c\u0941-\u0948\u094d\u0951-\u0955\u0962\u0963\u0981\u09bc\u09be\u09c1-\u09c4\u09cd\u09d7\u09e2\u09e3\u0a01\u0a02\u0a3c\u0a41\u0a42\u0a47\u0a48\u0a4b-\u0a4d\u0a51\u0a70\u0a71\u0a75\u0a81\u0a82\u0abc\u0ac1-\u0ac5\u0ac7\u0ac8\u0acd\u0ae2\u0ae3\u0b01\u0b3c\u0b3e\u0b3f\u0b41-\u0b44\u0b4d\u0b56\u0b57\u0b62\u0b63\u0b82\u0bbe\u0bc0\u0bcd\u0bd7\u0c3e-\u0c40\u0c46-\u0c48\u0c4a-\u0c4d\u0c55\u0c56\u0c62\u0c63\u0cbc\u0cbf\u0cc2\u0cc6\u0ccc\u0ccd\u0cd5\u0cd6\u0ce2\u0ce3\u0d3e\u0d41-\u0d44\u0d4d\u0d57\u0d62\u0d63\u0dca\u0dcf\u0dd2-\u0dd4\u0dd6\u0ddf\u0e31\u0e34-\u0e3a\u0e47-\u0e4e\u0eb1\u0eb4-\u0eb9\u0ebb\u0ebc\u0ec8-\u0ecd\u0f18\u0f19\u0f35\u0f37\u0f39\u0f71-\u0f7e\u0f80-\u0f84\u0f86\u0f87\u0f90-\u0f97\u0f99-\u0fbc\u0fc6\u102d-\u1030\u1032-\u1037\u1039\u103a\u103d\u103e\u1058\u1059\u105e-\u1060\u1071-\u1074\u1082\u1085\u1086\u108d\u109d\u135f\u1712-\u1714\u1732-\u1734\u1752\u1753\u1772\u1773\u17b7-\u17bd\u17c6\u17c9-\u17d3\u17dd\u180b-\u180d\u18a9\u1920-\u1922\u1927\u1928\u1932\u1939-\u193b\u1a17\u1a18\u1a56\u1a58-\u1a5e\u1a60\u1a62\u1a65-\u1a6c\u1a73-\u1a7c\u1a7f\u1b00-\u1b03\u1b34\u1b36-\u1b3a\u1b3c\u1b42\u1b6b-\u1b73\u1b80\u1b81\u1ba2-\u1ba5\u1ba8\u1ba9\u1c2c-\u1c33\u1c36\u1c37\u1cd0-\u1cd2\u1cd4-\u1ce0\u1ce2-\u1ce8\u1ced\u1dc0-\u1de6\u1dfd-\u1dff\u200c\u200d\u20d0-\u20f0\u2cef-\u2cf1\u2de0-\u2dff\u302a-\u302f\u3099\u309a\ua66f-\ua672\ua67c\ua67d\ua6f0\ua6f1\ua802\ua806\ua80b\ua825\ua826\ua8c4\ua8e0-\ua8f1\ua926-\ua92d\ua947-\ua951\ua980-\ua982\ua9b3\ua9b6-\ua9b9\ua9bc\uaa29-\uaa2e\uaa31\uaa32\uaa35\uaa36\uaa43\uaa4c\uaab0\uaab2-\uaab4\uaab7\uaab8\uaabe\uaabf\uaac1\uabe5\uabe8\uabed\udc00-\udfff\ufb1e\ufe00-\ufe0f\ufe20-\ufe26\uff9e\uff9f]/
function isExtendingChar(ch) { return ch.charCodeAt(0) >= 768 && extendingChars.test(ch) }

// Returns a number from the range [`0`; `str.length`] unless `pos` is outside that range.
function skipExtendingChars(str, pos, dir) {
  while ((dir < 0 ? pos > 0 : pos < str.length) && isExtendingChar(str.charAt(pos))) { pos += dir }
  return pos
}

// Returns the value from the range [`from`; `to`] that satisfies
// `pred` and is closest to `from`. Assumes that at least `to` satisfies `pred`.
function findFirst(pred, from, to) {
  for (;;) {
    if (Math.abs(from - to) <= 1) { return pred(from) ? from : to }
    var mid = Math.floor((from + to) / 2)
    if (pred(mid)) { to = mid }
    else { from = mid }
  }
}

// The display handles the DOM integration, both for input reading
// and content drawing. It holds references to DOM nodes and
// display-related state.

function Display(place, doc, input) {
  var d = this
  this.input = input

  // Covers bottom-right square when both scrollbars are present.
  d.scrollbarFiller = elt("div", null, "CodeMirror-scrollbar-filler")
  d.scrollbarFiller.setAttribute("cm-not-content", "true")
  // Covers bottom of gutter when coverGutterNextToScrollbar is on
  // and h scrollbar is present.
  d.gutterFiller = elt("div", null, "CodeMirror-gutter-filler")
  d.gutterFiller.setAttribute("cm-not-content", "true")
  // Will contain the actual code, positioned to cover the viewport.
  d.lineDiv = elt("div", null, "CodeMirror-code")
  // Elements are added to these to represent selection and cursors.
  d.selectionDiv = elt("div", null, null, "position: relative; z-index: 1")
  d.cursorDiv = elt("div", null, "CodeMirror-cursors")
  // A visibility: hidden element used to find the size of things.
  d.measure = elt("div", null, "CodeMirror-measure")
  // When lines outside of the viewport are measured, they are drawn in this.
  d.lineMeasure = elt("div", null, "CodeMirror-measure")
  // Wraps everything that needs to exist inside the vertically-padded coordinate system
  d.lineSpace = elt("div", [d.measure, d.lineMeasure, d.selectionDiv, d.cursorDiv, d.lineDiv],
                    null, "position: relative; outline: none")
  // Moved around its parent to cover visible view.
  d.mover = elt("div", [elt("div", [d.lineSpace], "CodeMirror-lines")], null, "position: relative")
  // Set to the height of the document, allowing scrolling.
  d.sizer = elt("div", [d.mover], "CodeMirror-sizer")
  d.sizerWidth = null
  // Behavior of elts with overflow: auto and padding is
  // inconsistent across browsers. This is used to ensure the
  // scrollable area is big enough.
  d.heightForcer = elt("div", null, null, "position: absolute; height: " + scrollerGap + "px; width: 1px;")
  // Will contain the gutters, if any.
  d.gutters = elt("div", null, "CodeMirror-gutters")
  d.lineGutter = null
  // Actual scrollable element.
  d.scroller = elt("div", [d.sizer, d.heightForcer, d.gutters], "CodeMirror-scroll")
  d.scroller.setAttribute("tabIndex", "-1")
  // The element in which the editor lives.
  d.wrapper = elt("div", [d.scrollbarFiller, d.gutterFiller, d.scroller], "CodeMirror")

  // Work around IE7 z-index bug (not perfect, hence IE7 not really being supported)
  if (ie && ie_version < 8) { d.gutters.style.zIndex = -1; d.scroller.style.paddingRight = 0 }
  if (!webkit && !(gecko && mobile)) { d.scroller.draggable = true }

  if (place) {
    if (place.appendChild) { place.appendChild(d.wrapper) }
    else { place(d.wrapper) }
  }

  // Current rendered range (may be bigger than the view window).
  d.viewFrom = d.viewTo = doc.first
  d.reportedViewFrom = d.reportedViewTo = doc.first
  // Information about the rendered lines.
  d.view = []
  d.renderedView = null
  // Holds info about a single rendered line when it was rendered
  // for measurement, while not in view.
  d.externalMeasured = null
  // Empty space (in pixels) above the view
  d.viewOffset = 0
  d.lastWrapHeight = d.lastWrapWidth = 0
  d.updateLineNumbers = null

  d.nativeBarWidth = d.barHeight = d.barWidth = 0
  d.scrollbarsClipped = false

  // Used to only resize the line number gutter when necessary (when
  // the amount of lines crosses a boundary that makes its width change)
  d.lineNumWidth = d.lineNumInnerWidth = d.lineNumChars = null
  // Set to true when a non-horizontal-scrolling line widget is
  // added. As an optimization, line widget aligning is skipped when
  // this is false.
  d.alignWidgets = false

  d.cachedCharWidth = d.cachedTextHeight = d.cachedPaddingH = null

  // Tracks the maximum line length so that the horizontal scrollbar
  // can be kept static when scrolling.
  d.maxLine = null
  d.maxLineLength = 0
  d.maxLineChanged = false

  // Used for measuring wheel scrolling granularity
  d.wheelDX = d.wheelDY = d.wheelStartX = d.wheelStartY = null

  // True when shift is held down.
  d.shift = false

  // Used to track whether anything happened since the context menu
  // was opened.
  d.selForContextMenu = null

  d.activeTouch = null

  input.init(d)
}

// Find the line object corresponding to the given line number.
function getLine(doc, n) {
  n -= doc.first
  if (n < 0 || n >= doc.size) { throw new Error("There is no line " + (n + doc.first) + " in the document.") }
  var chunk = doc
  while (!chunk.lines) {
    for (var i = 0;; ++i) {
      var child = chunk.children[i], sz = child.chunkSize()
      if (n < sz) { chunk = child; break }
      n -= sz
    }
  }
  return chunk.lines[n]
}

// Get the part of a document between two positions, as an array of
// strings.
function getBetween(doc, start, end) {
  var out = [], n = start.line
  doc.iter(start.line, end.line + 1, function (line) {
    var text = line.text
    if (n == end.line) { text = text.slice(0, end.ch) }
    if (n == start.line) { text = text.slice(start.ch) }
    out.push(text)
    ++n
  })
  return out
}
// Get the lines between from and to, as array of strings.
function getLines(doc, from, to) {
  var out = []
  doc.iter(from, to, function (line) { out.push(line.text) }) // iter aborts when callback returns truthy value
  return out
}

// Update the height of a line, propagating the height change
// upwards to parent nodes.
function updateLineHeight(line, height) {
  var diff = height - line.height
  if (diff) { for (var n = line; n; n = n.parent) { n.height += diff } }
}

// Given a line object, find its line number by walking up through
// its parent links.
function lineNo(line) {
  if (line.parent == null) { return null }
  var cur = line.parent, no = indexOf(cur.lines, line)
  for (var chunk = cur.parent; chunk; cur = chunk, chunk = chunk.parent) {
    for (var i = 0;; ++i) {
      if (chunk.children[i] == cur) { break }
      no += chunk.children[i].chunkSize()
    }
  }
  return no + cur.first
}

// Find the line at the given vertical position, using the height
// information in the document tree.
function lineAtHeight(chunk, h) {
  var n = chunk.first
  outer: do {
    for (var i$1 = 0; i$1 < chunk.children.length; ++i$1) {
      var child = chunk.children[i$1], ch = child.height
      if (h < ch) { chunk = child; continue outer }
      h -= ch
      n += child.chunkSize()
    }
    return n
  } while (!chunk.lines)
  var i = 0
  for (; i < chunk.lines.length; ++i) {
    var line = chunk.lines[i], lh = line.height
    if (h < lh) { break }
    h -= lh
  }
  return n + i
}

function isLine(doc, l) {return l >= doc.first && l < doc.first + doc.size}

function lineNumberFor(options, i) {
  return String(options.lineNumberFormatter(i + options.firstLineNumber))
}

// A Pos instance represents a position within the text.
function Pos(line, ch, sticky) {
  if ( sticky === void 0 ) sticky = null;

  if (!(this instanceof Pos)) { return new Pos(line, ch, sticky) }
  this.line = line
  this.ch = ch
  this.sticky = sticky
}

// Compare two positions, return 0 if they are the same, a negative
// number when a is less, and a positive number otherwise.
function cmp(a, b) { return a.line - b.line || a.ch - b.ch }

function equalCursorPos(a, b) { return a.sticky == b.sticky && cmp(a, b) == 0 }

function copyPos(x) {return Pos(x.line, x.ch)}
function maxPos(a, b) { return cmp(a, b) < 0 ? b : a }
function minPos(a, b) { return cmp(a, b) < 0 ? a : b }

// Most of the external API clips given positions to make sure they
// actually exist within the document.
function clipLine(doc, n) {return Math.max(doc.first, Math.min(n, doc.first + doc.size - 1))}
function clipPos(doc, pos) {
  if (pos.line < doc.first) { return Pos(doc.first, 0) }
  var last = doc.first + doc.size - 1
  if (pos.line > last) { return Pos(last, getLine(doc, last).text.length) }
  return clipToLen(pos, getLine(doc, pos.line).text.length)
}
function clipToLen(pos, linelen) {
  var ch = pos.ch
  if (ch == null || ch > linelen) { return Pos(pos.line, linelen) }
  else if (ch < 0) { return Pos(pos.line, 0) }
  else { return pos }
}
function clipPosArray(doc, array) {
  var out = []
  for (var i = 0; i < array.length; i++) { out[i] = clipPos(doc, array[i]) }
  return out
}

// Optimize some code when these features are not used.
var sawReadOnlySpans = false;
var sawCollapsedSpans = false;
function seeReadOnlySpans() {
  sawReadOnlySpans = true
}

function seeCollapsedSpans() {
  sawCollapsedSpans = true
}

// TEXTMARKER SPANS

function MarkedSpan(marker, from, to) {
  this.marker = marker
  this.from = from; this.to = to
}

// Search an array of spans for a span matching the given marker.
function getMarkedSpanFor(spans, marker) {
  if (spans) { for (var i = 0; i < spans.length; ++i) {
    var span = spans[i]
    if (span.marker == marker) { return span }
  } }
}
// Remove a span from an array, returning undefined if no spans are
// left (we don't store arrays for lines without spans).
function removeMarkedSpan(spans, span) {
  var r
  for (var i = 0; i < spans.length; ++i)
    { if (spans[i] != span) { (r || (r = [])).push(spans[i]) } }
  return r
}
// Add a span to a line.
function addMarkedSpan(line, span) {
  line.markedSpans = line.markedSpans ? line.markedSpans.concat([span]) : [span]
  span.marker.attachLine(line)
}

// Used for the algorithm that adjusts markers for a change in the
// document. These functions cut an array of spans at a given
// character position, returning an array of remaining chunks (or
// undefined if nothing remains).
function markedSpansBefore(old, startCh, isInsert) {
  var nw
  if (old) { for (var i = 0; i < old.length; ++i) {
    var span = old[i], marker = span.marker
    var startsBefore = span.from == null || (marker.inclusiveLeft ? span.from <= startCh : span.from < startCh)
    if (startsBefore || span.from == startCh && marker.type == "bookmark" && (!isInsert || !span.marker.insertLeft)) {
      var endsAfter = span.to == null || (marker.inclusiveRight ? span.to >= startCh : span.to > startCh)
      ;(nw || (nw = [])).push(new MarkedSpan(marker, span.from, endsAfter ? null : span.to))
    }
  } }
  return nw
}
function markedSpansAfter(old, endCh, isInsert) {
  var nw
  if (old) { for (var i = 0; i < old.length; ++i) {
    var span = old[i], marker = span.marker
    var endsAfter = span.to == null || (marker.inclusiveRight ? span.to >= endCh : span.to > endCh)
    if (endsAfter || span.from == endCh && marker.type == "bookmark" && (!isInsert || span.marker.insertLeft)) {
      var startsBefore = span.from == null || (marker.inclusiveLeft ? span.from <= endCh : span.from < endCh)
      ;(nw || (nw = [])).push(new MarkedSpan(marker, startsBefore ? null : span.from - endCh,
                                            span.to == null ? null : span.to - endCh))
    }
  } }
  return nw
}

// Given a change object, compute the new set of marker spans that
// cover the line in which the change took place. Removes spans
// entirely within the change, reconnects spans belonging to the
// same marker that appear on both sides of the change, and cuts off
// spans partially within the change. Returns an array of span
// arrays with one element for each line in (after) the change.
function stretchSpansOverChange(doc, change) {
  if (change.full) { return null }
  var oldFirst = isLine(doc, change.from.line) && getLine(doc, change.from.line).markedSpans
  var oldLast = isLine(doc, change.to.line) && getLine(doc, change.to.line).markedSpans
  if (!oldFirst && !oldLast) { return null }

  var startCh = change.from.ch, endCh = change.to.ch, isInsert = cmp(change.from, change.to) == 0
  // Get the spans that 'stick out' on both sides
  var first = markedSpansBefore(oldFirst, startCh, isInsert)
  var last = markedSpansAfter(oldLast, endCh, isInsert)

  // Next, merge those two ends
  var sameLine = change.text.length == 1, offset = lst(change.text).length + (sameLine ? startCh : 0)
  if (first) {
    // Fix up .to properties of first
    for (var i = 0; i < first.length; ++i) {
      var span = first[i]
      if (span.to == null) {
        var found = getMarkedSpanFor(last, span.marker)
        if (!found) { span.to = startCh }
        else if (sameLine) { span.to = found.to == null ? null : found.to + offset }
      }
    }
  }
  if (last) {
    // Fix up .from in last (or move them into first in case of sameLine)
    for (var i$1 = 0; i$1 < last.length; ++i$1) {
      var span$1 = last[i$1]
      if (span$1.to != null) { span$1.to += offset }
      if (span$1.from == null) {
        var found$1 = getMarkedSpanFor(first, span$1.marker)
        if (!found$1) {
          span$1.from = offset
          if (sameLine) { (first || (first = [])).push(span$1) }
        }
      } else {
        span$1.from += offset
        if (sameLine) { (first || (first = [])).push(span$1) }
      }
    }
  }
  // Make sure we didn't create any zero-length spans
  if (first) { first = clearEmptySpans(first) }
  if (last && last != first) { last = clearEmptySpans(last) }

  var newMarkers = [first]
  if (!sameLine) {
    // Fill gap with whole-line-spans
    var gap = change.text.length - 2, gapMarkers
    if (gap > 0 && first)
      { for (var i$2 = 0; i$2 < first.length; ++i$2)
        { if (first[i$2].to == null)
          { (gapMarkers || (gapMarkers = [])).push(new MarkedSpan(first[i$2].marker, null, null)) } } }
    for (var i$3 = 0; i$3 < gap; ++i$3)
      { newMarkers.push(gapMarkers) }
    newMarkers.push(last)
  }
  return newMarkers
}

// Remove spans that are empty and don't have a clearWhenEmpty
// option of false.
function clearEmptySpans(spans) {
  for (var i = 0; i < spans.length; ++i) {
    var span = spans[i]
    if (span.from != null && span.from == span.to && span.marker.clearWhenEmpty !== false)
      { spans.splice(i--, 1) }
  }
  if (!spans.length) { return null }
  return spans
}

// Used to 'clip' out readOnly ranges when making a change.
function removeReadOnlyRanges(doc, from, to) {
  var markers = null
  doc.iter(from.line, to.line + 1, function (line) {
    if (line.markedSpans) { for (var i = 0; i < line.markedSpans.length; ++i) {
      var mark = line.markedSpans[i].marker
      if (mark.readOnly && (!markers || indexOf(markers, mark) == -1))
        { (markers || (markers = [])).push(mark) }
    } }
  })
  if (!markers) { return null }
  var parts = [{from: from, to: to}]
  for (var i = 0; i < markers.length; ++i) {
    var mk = markers[i], m = mk.find(0)
    for (var j = 0; j < parts.length; ++j) {
      var p = parts[j]
      if (cmp(p.to, m.from) < 0 || cmp(p.from, m.to) > 0) { continue }
      var newParts = [j, 1], dfrom = cmp(p.from, m.from), dto = cmp(p.to, m.to)
      if (dfrom < 0 || !mk.inclusiveLeft && !dfrom)
        { newParts.push({from: p.from, to: m.from}) }
      if (dto > 0 || !mk.inclusiveRight && !dto)
        { newParts.push({from: m.to, to: p.to}) }
      parts.splice.apply(parts, newParts)
      j += newParts.length - 3
    }
  }
  return parts
}

// Connect or disconnect spans from a line.
function detachMarkedSpans(line) {
  var spans = line.markedSpans
  if (!spans) { return }
  for (var i = 0; i < spans.length; ++i)
    { spans[i].marker.detachLine(line) }
  line.markedSpans = null
}
function attachMarkedSpans(line, spans) {
  if (!spans) { return }
  for (var i = 0; i < spans.length; ++i)
    { spans[i].marker.attachLine(line) }
  line.markedSpans = spans
}

// Helpers used when computing which overlapping collapsed span
// counts as the larger one.
function extraLeft(marker) { return marker.inclusiveLeft ? -1 : 0 }
function extraRight(marker) { return marker.inclusiveRight ? 1 : 0 }

// Returns a number indicating which of two overlapping collapsed
// spans is larger (and thus includes the other). Falls back to
// comparing ids when the spans cover exactly the same range.
function compareCollapsedMarkers(a, b) {
  var lenDiff = a.lines.length - b.lines.length
  if (lenDiff != 0) { return lenDiff }
  var aPos = a.find(), bPos = b.find()
  var fromCmp = cmp(aPos.from, bPos.from) || extraLeft(a) - extraLeft(b)
  if (fromCmp) { return -fromCmp }
  var toCmp = cmp(aPos.to, bPos.to) || extraRight(a) - extraRight(b)
  if (toCmp) { return toCmp }
  return b.id - a.id
}

// Find out whether a line ends or starts in a collapsed span. If
// so, return the marker for that span.
function collapsedSpanAtSide(line, start) {
  var sps = sawCollapsedSpans && line.markedSpans, found
  if (sps) { for (var sp = (void 0), i = 0; i < sps.length; ++i) {
    sp = sps[i]
    if (sp.marker.collapsed && (start ? sp.from : sp.to) == null &&
        (!found || compareCollapsedMarkers(found, sp.marker) < 0))
      { found = sp.marker }
  } }
  return found
}
function collapsedSpanAtStart(line) { return collapsedSpanAtSide(line, true) }
function collapsedSpanAtEnd(line) { return collapsedSpanAtSide(line, false) }

// Test whether there exists a collapsed span that partially
// overlaps (covers the start or end, but not both) of a new span.
// Such overlap is not allowed.
function conflictingCollapsedRange(doc, lineNo, from, to, marker) {
  var line = getLine(doc, lineNo)
  var sps = sawCollapsedSpans && line.markedSpans
  if (sps) { for (var i = 0; i < sps.length; ++i) {
    var sp = sps[i]
    if (!sp.marker.collapsed) { continue }
    var found = sp.marker.find(0)
    var fromCmp = cmp(found.from, from) || extraLeft(sp.marker) - extraLeft(marker)
    var toCmp = cmp(found.to, to) || extraRight(sp.marker) - extraRight(marker)
    if (fromCmp >= 0 && toCmp <= 0 || fromCmp <= 0 && toCmp >= 0) { continue }
    if (fromCmp <= 0 && (sp.marker.inclusiveRight && marker.inclusiveLeft ? cmp(found.to, from) >= 0 : cmp(found.to, from) > 0) ||
        fromCmp >= 0 && (sp.marker.inclusiveRight && marker.inclusiveLeft ? cmp(found.from, to) <= 0 : cmp(found.from, to) < 0))
      { return true }
  } }
}

// A visual line is a line as drawn on the screen. Folding, for
// example, can cause multiple logical lines to appear on the same
// visual line. This finds the start of the visual line that the
// given line is part of (usually that is the line itself).
function visualLine(line) {
  var merged
  while (merged = collapsedSpanAtStart(line))
    { line = merged.find(-1, true).line }
  return line
}

function visualLineEnd(line) {
  var merged
  while (merged = collapsedSpanAtEnd(line))
    { line = merged.find(1, true).line }
  return line
}

// Returns an array of logical lines that continue the visual line
// started by the argument, or undefined if there are no such lines.
function visualLineContinued(line) {
  var merged, lines
  while (merged = collapsedSpanAtEnd(line)) {
    line = merged.find(1, true).line
    ;(lines || (lines = [])).push(line)
  }
  return lines
}

// Get the line number of the start of the visual line that the
// given line number is part of.
function visualLineNo(doc, lineN) {
  var line = getLine(doc, lineN), vis = visualLine(line)
  if (line == vis) { return lineN }
  return lineNo(vis)
}

// Get the line number of the start of the next visual line after
// the given line.
function visualLineEndNo(doc, lineN) {
  if (lineN > doc.lastLine()) { return lineN }
  var line = getLine(doc, lineN), merged
  if (!lineIsHidden(doc, line)) { return lineN }
  while (merged = collapsedSpanAtEnd(line))
    { line = merged.find(1, true).line }
  return lineNo(line) + 1
}

// Compute whether a line is hidden. Lines count as hidden when they
// are part of a visual line that starts with another line, or when
// they are entirely covered by collapsed, non-widget span.
function lineIsHidden(doc, line) {
  var sps = sawCollapsedSpans && line.markedSpans
  if (sps) { for (var sp = (void 0), i = 0; i < sps.length; ++i) {
    sp = sps[i]
    if (!sp.marker.collapsed) { continue }
    if (sp.from == null) { return true }
    if (sp.marker.widgetNode) { continue }
    if (sp.from == 0 && sp.marker.inclusiveLeft && lineIsHiddenInner(doc, line, sp))
      { return true }
  } }
}
function lineIsHiddenInner(doc, line, span) {
  if (span.to == null) {
    var end = span.marker.find(1, true)
    return lineIsHiddenInner(doc, end.line, getMarkedSpanFor(end.line.markedSpans, span.marker))
  }
  if (span.marker.inclusiveRight && span.to == line.text.length)
    { return true }
  for (var sp = (void 0), i = 0; i < line.markedSpans.length; ++i) {
    sp = line.markedSpans[i]
    if (sp.marker.collapsed && !sp.marker.widgetNode && sp.from == span.to &&
        (sp.to == null || sp.to != span.from) &&
        (sp.marker.inclusiveLeft || span.marker.inclusiveRight) &&
        lineIsHiddenInner(doc, line, sp)) { return true }
  }
}

// Find the height above the given line.
function heightAtLine(lineObj) {
  lineObj = visualLine(lineObj)

  var h = 0, chunk = lineObj.parent
  for (var i = 0; i < chunk.lines.length; ++i) {
    var line = chunk.lines[i]
    if (line == lineObj) { break }
    else { h += line.height }
  }
  for (var p = chunk.parent; p; chunk = p, p = chunk.parent) {
    for (var i$1 = 0; i$1 < p.children.length; ++i$1) {
      var cur = p.children[i$1]
      if (cur == chunk) { break }
      else { h += cur.height }
    }
  }
  return h
}

// Compute the character length of a line, taking into account
// collapsed ranges (see markText) that might hide parts, and join
// other lines onto it.
function lineLength(line) {
  if (line.height == 0) { return 0 }
  var len = line.text.length, merged, cur = line
  while (merged = collapsedSpanAtStart(cur)) {
    var found = merged.find(0, true)
    cur = found.from.line
    len += found.from.ch - found.to.ch
  }
  cur = line
  while (merged = collapsedSpanAtEnd(cur)) {
    var found$1 = merged.find(0, true)
    len -= cur.text.length - found$1.from.ch
    cur = found$1.to.line
    len += cur.text.length - found$1.to.ch
  }
  return len
}

// Find the longest line in the document.
function findMaxLine(cm) {
  var d = cm.display, doc = cm.doc
  d.maxLine = getLine(doc, doc.first)
  d.maxLineLength = lineLength(d.maxLine)
  d.maxLineChanged = true
  doc.iter(function (line) {
    var len = lineLength(line)
    if (len > d.maxLineLength) {
      d.maxLineLength = len
      d.maxLine = line
    }
  })
}

// BIDI HELPERS

function iterateBidiSections(order, from, to, f) {
  if (!order) { return f(from, to, "ltr") }
  var found = false
  for (var i = 0; i < order.length; ++i) {
    var part = order[i]
    if (part.from < to && part.to > from || from == to && part.to == from) {
      f(Math.max(part.from, from), Math.min(part.to, to), part.level == 1 ? "rtl" : "ltr")
      found = true
    }
  }
  if (!found) { f(from, to, "ltr") }
}

var bidiOther = null
function getBidiPartAt(order, ch, sticky) {
  var found
  bidiOther = null
  for (var i = 0; i < order.length; ++i) {
    var cur = order[i]
    if (cur.from < ch && cur.to > ch) { return i }
    if (cur.to == ch) {
      if (cur.from != cur.to && sticky == "before") { found = i }
      else { bidiOther = i }
    }
    if (cur.from == ch) {
      if (cur.from != cur.to && sticky != "before") { found = i }
      else { bidiOther = i }
    }
  }
  return found != null ? found : bidiOther
}

// Bidirectional ordering algorithm
// See http://unicode.org/reports/tr9/tr9-13.html for the algorithm
// that this (partially) implements.

// One-char codes used for character types:
// L (L):   Left-to-Right
// R (R):   Right-to-Left
// r (AL):  Right-to-Left Arabic
// 1 (EN):  European Number
// + (ES):  European Number Separator
// % (ET):  European Number Terminator
// n (AN):  Arabic Number
// , (CS):  Common Number Separator
// m (NSM): Non-Spacing Mark
// b (BN):  Boundary Neutral
// s (B):   Paragraph Separator
// t (S):   Segment Separator
// w (WS):  Whitespace
// N (ON):  Other Neutrals

// Returns null if characters are ordered as they appear
// (left-to-right), or an array of sections ({from, to, level}
// objects) in the order in which they occur visually.
var bidiOrdering = (function() {
  // Character types for codepoints 0 to 0xff
  var lowTypes = "bbbbbbbbbtstwsbbbbbbbbbbbbbbssstwNN%%%NNNNNN,N,N1111111111NNNNNNNLLLLLLLLLLLLLLLLLLLLLLLLLLNNNNNNLLLLLLLLLLLLLLLLLLLLLLLLLLNNNNbbbbbbsbbbbbbbbbbbbbbbbbbbbbbbbbb,N%%%%NNNNLNNNNN%%11NLNNN1LNNNNNLLLLLLLLLLLLLLLLLLLLLLLNLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLN"
  // Character types for codepoints 0x600 to 0x6f9
  var arabicTypes = "nnnnnnNNr%%r,rNNmmmmmmmmmmmrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrmmmmmmmmmmmmmmmmmmmmmnnnnnnnnnn%nnrrrmrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrrmmmmmmmnNmmmmmmrrmmNmmmmrr1111111111"
  function charType(code) {
    if (code <= 0xf7) { return lowTypes.charAt(code) }
    else if (0x590 <= code && code <= 0x5f4) { return "R" }
    else if (0x600 <= code && code <= 0x6f9) { return arabicTypes.charAt(code - 0x600) }
    else if (0x6ee <= code && code <= 0x8ac) { return "r" }
    else if (0x2000 <= code && code <= 0x200b) { return "w" }
    else if (code == 0x200c) { return "b" }
    else { return "L" }
  }

  var bidiRE = /[\u0590-\u05f4\u0600-\u06ff\u0700-\u08ac]/
  var isNeutral = /[stwN]/, isStrong = /[LRr]/, countsAsLeft = /[Lb1n]/, countsAsNum = /[1n]/
  // Browsers seem to always treat the boundaries of block elements as being L.
  var outerType = "L"

  function BidiSpan(level, from, to) {
    this.level = level
    this.from = from; this.to = to
  }

  return function(str) {
    if (!bidiRE.test(str)) { return false }
    var len = str.length, types = []
    for (var i = 0; i < len; ++i)
      { types.push(charType(str.charCodeAt(i))) }

    // W1. Examine each non-spacing mark (NSM) in the level run, and
    // change the type of the NSM to the type of the previous
    // character. If the NSM is at the start of the level run, it will
    // get the type of sor.
    for (var i$1 = 0, prev = outerType; i$1 < len; ++i$1) {
      var type = types[i$1]
      if (type == "m") { types[i$1] = prev }
      else { prev = type }
    }

    // W2. Search backwards from each instance of a European number
    // until the first strong type (R, L, AL, or sor) is found. If an
    // AL is found, change the type of the European number to Arabic
    // number.
    // W3. Change all ALs to R.
    for (var i$2 = 0, cur = outerType; i$2 < len; ++i$2) {
      var type$1 = types[i$2]
      if (type$1 == "1" && cur == "r") { types[i$2] = "n" }
      else if (isStrong.test(type$1)) { cur = type$1; if (type$1 == "r") { types[i$2] = "R" } }
    }

    // W4. A single European separator between two European numbers
    // changes to a European number. A single common separator between
    // two numbers of the same type changes to that type.
    for (var i$3 = 1, prev$1 = types[0]; i$3 < len - 1; ++i$3) {
      var type$2 = types[i$3]
      if (type$2 == "+" && prev$1 == "1" && types[i$3+1] == "1") { types[i$3] = "1" }
      else if (type$2 == "," && prev$1 == types[i$3+1] &&
               (prev$1 == "1" || prev$1 == "n")) { types[i$3] = prev$1 }
      prev$1 = type$2
    }

    // W5. A sequence of European terminators adjacent to European
    // numbers changes to all European numbers.
    // W6. Otherwise, separators and terminators change to Other
    // Neutral.
    for (var i$4 = 0; i$4 < len; ++i$4) {
      var type$3 = types[i$4]
      if (type$3 == ",") { types[i$4] = "N" }
      else if (type$3 == "%") {
        var end = (void 0)
        for (end = i$4 + 1; end < len && types[end] == "%"; ++end) {}
        var replace = (i$4 && types[i$4-1] == "!") || (end < len && types[end] == "1") ? "1" : "N"
        for (var j = i$4; j < end; ++j) { types[j] = replace }
        i$4 = end - 1
      }
    }

    // W7. Search backwards from each instance of a European number
    // until the first strong type (R, L, or sor) is found. If an L is
    // found, then change the type of the European number to L.
    for (var i$5 = 0, cur$1 = outerType; i$5 < len; ++i$5) {
      var type$4 = types[i$5]
      if (cur$1 == "L" && type$4 == "1") { types[i$5] = "L" }
      else if (isStrong.test(type$4)) { cur$1 = type$4 }
    }

    // N1. A sequence of neutrals takes the direction of the
    // surrounding strong text if the text on both sides has the same
    // direction. European and Arabic numbers act as if they were R in
    // terms of their influence on neutrals. Start-of-level-run (sor)
    // and end-of-level-run (eor) are used at level run boundaries.
    // N2. Any remaining neutrals take the embedding direction.
    for (var i$6 = 0; i$6 < len; ++i$6) {
      if (isNeutral.test(types[i$6])) {
        var end$1 = (void 0)
        for (end$1 = i$6 + 1; end$1 < len && isNeutral.test(types[end$1]); ++end$1) {}
        var before = (i$6 ? types[i$6-1] : outerType) == "L"
        var after = (end$1 < len ? types[end$1] : outerType) == "L"
        var replace$1 = before || after ? "L" : "R"
        for (var j$1 = i$6; j$1 < end$1; ++j$1) { types[j$1] = replace$1 }
        i$6 = end$1 - 1
      }
    }

    // Here we depart from the documented algorithm, in order to avoid
    // building up an actual levels array. Since there are only three
    // levels (0, 1, 2) in an implementation that doesn't take
    // explicit embedding into account, we can build up the order on
    // the fly, without following the level-based algorithm.
    var order = [], m
    for (var i$7 = 0; i$7 < len;) {
      if (countsAsLeft.test(types[i$7])) {
        var start = i$7
        for (++i$7; i$7 < len && countsAsLeft.test(types[i$7]); ++i$7) {}
        order.push(new BidiSpan(0, start, i$7))
      } else {
        var pos = i$7, at = order.length
        for (++i$7; i$7 < len && types[i$7] != "L"; ++i$7) {}
        for (var j$2 = pos; j$2 < i$7;) {
          if (countsAsNum.test(types[j$2])) {
            if (pos < j$2) { order.splice(at, 0, new BidiSpan(1, pos, j$2)) }
            var nstart = j$2
            for (++j$2; j$2 < i$7 && countsAsNum.test(types[j$2]); ++j$2) {}
            order.splice(at, 0, new BidiSpan(2, nstart, j$2))
            pos = j$2
          } else { ++j$2 }
        }
        if (pos < i$7) { order.splice(at, 0, new BidiSpan(1, pos, i$7)) }
      }
    }
    if (order[0].level == 1 && (m = str.match(/^\s+/))) {
      order[0].from = m[0].length
      order.unshift(new BidiSpan(0, 0, m[0].length))
    }
    if (lst(order).level == 1 && (m = str.match(/\s+$/))) {
      lst(order).to -= m[0].length
      order.push(new BidiSpan(0, len - m[0].length, len))
    }

    return order
  }
})()

// Get the bidi ordering for the given line (and cache it). Returns
// false for lines that are fully left-to-right, and an array of
// BidiSpan objects otherwise.
function getOrder(line) {
  var order = line.order
  if (order == null) { order = line.order = bidiOrdering(line.text) }
  return order
}

function moveCharLogically(line, ch, dir) {
  var target = skipExtendingChars(line.text, ch + dir, dir)
  return target < 0 || target > line.text.length ? null : target
}

function moveLogically(line, start, dir) {
  var ch = moveCharLogically(line, start.ch, dir)
  return ch == null ? null : new Pos(start.line, ch, dir < 0 ? "after" : "before")
}

function endOfLine(visually, cm, lineObj, lineNo, dir) {
  if (visually) {
    var order = getOrder(lineObj)
    if (order) {
      var part = dir < 0 ? lst(order) : order[0]
      var moveInStorageOrder = (dir < 0) == (part.level == 1)
      var sticky = moveInStorageOrder ? "after" : "before"
      var ch
      // With a wrapped rtl chunk (possibly spanning multiple bidi parts),
      // it could be that the last bidi part is not on the last visual line,
      // since visual lines contain content order-consecutive chunks.
      // Thus, in rtl, we are looking for the first (content-order) character
      // in the rtl chunk that is on the last line (that is, the same line
      // as the last (content-order) character).
      if (part.level > 0) {
        var prep = prepareMeasureForLine(cm, lineObj)
        ch = dir < 0 ? lineObj.text.length - 1 : 0
        var targetTop = measureCharPrepared(cm, prep, ch).top
        ch = findFirst(function (ch) { return measureCharPrepared(cm, prep, ch).top == targetTop; }, (dir < 0) == (part.level == 1) ? part.from : part.to - 1, ch)
        if (sticky == "before") { ch = moveCharLogically(lineObj, ch, 1, true) }
      } else { ch = dir < 0 ? part.to : part.from }
      return new Pos(lineNo, ch, sticky)
    }
  }
  return new Pos(lineNo, dir < 0 ? lineObj.text.length : 0, dir < 0 ? "before" : "after")
}

function moveVisually(cm, line, start, dir) {
  var bidi = getOrder(line)
  if (!bidi) { return moveLogically(line, start, dir) }
  if (start.ch >= line.text.length) {
    start.ch = line.text.length
    start.sticky = "before"
  } else if (start.ch <= 0) {
    start.ch = 0
    start.sticky = "after"
  }
  var partPos = getBidiPartAt(bidi, start.ch, start.sticky), part = bidi[partPos]
  if (part.level % 2 == 0 && (dir > 0 ? part.to > start.ch : part.from < start.ch)) {
    // Case 1: We move within an ltr part. Even with wrapped lines,
    // nothing interesting happens.
    return moveLogically(line, start, dir)
  }

  var mv = function (pos, dir) { return moveCharLogically(line, pos instanceof Pos ? pos.ch : pos, dir); }
  var prep
  var getWrappedLineExtent = function (ch) {
    if (!cm.options.lineWrapping) { return {begin: 0, end: line.text.length} }
    prep = prep || prepareMeasureForLine(cm, line)
    return wrappedLineExtentChar(cm, line, prep, ch)
  }
  var wrappedLineExtent = getWrappedLineExtent(start.sticky == "before" ? mv(start, -1) : start.ch)

  if (part.level % 2 == 1) {
    var ch = mv(start, -dir)
    if (ch != null && (dir > 0 ? ch >= part.from && ch >= wrappedLineExtent.begin : ch <= part.to && ch <= wrappedLineExtent.end)) {
      // Case 2: We move within an rtl part on the same visual line
      var sticky = dir < 0 ? "before" : "after"
      return new Pos(start.line, ch, sticky)
    }
  }

  // Case 3: Could not move within this bidi part in this visual line, so leave
  // the current bidi part

  var searchInVisualLine = function (partPos, dir, wrappedLineExtent) {
    var getRes = function (ch, moveInStorageOrder) { return moveInStorageOrder
      ? new Pos(start.line, mv(ch, 1), "before")
      : new Pos(start.line, ch, "after"); }

    for (; partPos >= 0 && partPos < bidi.length; partPos += dir) {
      var part = bidi[partPos]
      var moveInStorageOrder = (dir > 0) == (part.level != 1)
      var ch = moveInStorageOrder ? wrappedLineExtent.begin : mv(wrappedLineExtent.end, -1)
      if (part.from <= ch && ch < part.to) { return getRes(ch, moveInStorageOrder) }
      ch = moveInStorageOrder ? part.from : mv(part.to, -1)
      if (wrappedLineExtent.begin <= ch && ch < wrappedLineExtent.end) { return getRes(ch, moveInStorageOrder) }
    }
  }

  // Case 3a: Look for other bidi parts on the same visual line
  var res = searchInVisualLine(partPos + dir, dir, wrappedLineExtent)
  if (res) { return res }

  // Case 3b: Look for other bidi parts on the next visual line
  var nextCh = dir > 0 ? wrappedLineExtent.end : mv(wrappedLineExtent.begin, -1)
  if (nextCh != null && !(dir > 0 && nextCh == line.text.length)) {
    res = searchInVisualLine(dir > 0 ? 0 : bidi.length - 1, dir, getWrappedLineExtent(nextCh))
    if (res) { return res }
  }

  // Case 4: Nowhere to move
  return null
}

// EVENT HANDLING

// Lightweight event framework. on/off also work on DOM nodes,
// registering native DOM handlers.

var noHandlers = []

var on = function(emitter, type, f) {
  if (emitter.addEventListener) {
    emitter.addEventListener(type, f, false)
  } else if (emitter.attachEvent) {
    emitter.attachEvent("on" + type, f)
  } else {
    var map = emitter._handlers || (emitter._handlers = {})
    map[type] = (map[type] || noHandlers).concat(f)
  }
}

function getHandlers(emitter, type) {
  return emitter._handlers && emitter._handlers[type] || noHandlers
}

function off(emitter, type, f) {
  if (emitter.removeEventListener) {
    emitter.removeEventListener(type, f, false)
  } else if (emitter.detachEvent) {
    emitter.detachEvent("on" + type, f)
  } else {
    var map = emitter._handlers, arr = map && map[type]
    if (arr) {
      var index = indexOf(arr, f)
      if (index > -1)
        { map[type] = arr.slice(0, index).concat(arr.slice(index + 1)) }
    }
  }
}

function signal(emitter, type /*, values...*/) {
  var handlers = getHandlers(emitter, type)
  if (!handlers.length) { return }
  var args = Array.prototype.slice.call(arguments, 2)
  for (var i = 0; i < handlers.length; ++i) { handlers[i].apply(null, args) }
}

// The DOM events that CodeMirror handles can be overridden by
// registering a (non-DOM) handler on the editor for the event name,
// and preventDefault-ing the event in that handler.
function signalDOMEvent(cm, e, override) {
  if (typeof e == "string")
    { e = {type: e, preventDefault: function() { this.defaultPrevented = true }} }
  signal(cm, override || e.type, cm, e)
  return e_defaultPrevented(e) || e.codemirrorIgnore
}

function signalCursorActivity(cm) {
  var arr = cm._handlers && cm._handlers.cursorActivity
  if (!arr) { return }
  var set = cm.curOp.cursorActivityHandlers || (cm.curOp.cursorActivityHandlers = [])
  for (var i = 0; i < arr.length; ++i) { if (indexOf(set, arr[i]) == -1)
    { set.push(arr[i]) } }
}

function hasHandler(emitter, type) {
  return getHandlers(emitter, type).length > 0
}

// Add on and off methods to a constructor's prototype, to make
// registering events on such objects more convenient.
function eventMixin(ctor) {
  ctor.prototype.on = function(type, f) {on(this, type, f)}
  ctor.prototype.off = function(type, f) {off(this, type, f)}
}

// Due to the fact that we still support jurassic IE versions, some
// compatibility wrappers are needed.

function e_preventDefault(e) {
  if (e.preventDefault) { e.preventDefault() }
  else { e.returnValue = false }
}
function e_stopPropagation(e) {
  if (e.stopPropagation) { e.stopPropagation() }
  else { e.cancelBubble = true }
}
function e_defaultPrevented(e) {
  return e.defaultPrevented != null ? e.defaultPrevented : e.returnValue == false
}
function e_stop(e) {e_preventDefault(e); e_stopPropagation(e)}

function e_target(e) {return e.target || e.srcElement}
function e_button(e) {
  var b = e.which
  if (b == null) {
    if (e.button & 1) { b = 1 }
    else if (e.button & 2) { b = 3 }
    else if (e.button & 4) { b = 2 }
  }
  if (mac && e.ctrlKey && b == 1) { b = 3 }
  return b
}

// Detect drag-and-drop
var dragAndDrop = function() {
  // There is *some* kind of drag-and-drop support in IE6-8, but I
  // couldn't get it to work yet.
  if (ie && ie_version < 9) { return false }
  var div = elt('div')
  return "draggable" in div || "dragDrop" in div
}()

var zwspSupported
function zeroWidthElement(measure) {
  if (zwspSupported == null) {
    var test = elt("span", "\u200b")
    removeChildrenAndAdd(measure, elt("span", [test, document.createTextNode("x")]))
    if (measure.firstChild.offsetHeight != 0)
      { zwspSupported = test.offsetWidth <= 1 && test.offsetHeight > 2 && !(ie && ie_version < 8) }
  }
  var node = zwspSupported ? elt("span", "\u200b") :
    elt("span", "\u00a0", null, "display: inline-block; width: 1px; margin-right: -1px")
  node.setAttribute("cm-text", "")
  return node
}

// Feature-detect IE's crummy client rect reporting for bidi text
var badBidiRects
function hasBadBidiRects(measure) {
  if (badBidiRects != null) { return badBidiRects }
  var txt = removeChildrenAndAdd(measure, document.createTextNode("A\u062eA"))
  var r0 = range(txt, 0, 1).getBoundingClientRect()
  var r1 = range(txt, 1, 2).getBoundingClientRect()
  removeChildren(measure)
  if (!r0 || r0.left == r0.right) { return false } // Safari returns null in some cases (#2780)
  return badBidiRects = (r1.right - r0.right < 3)
}

// See if "".split is the broken IE version, if so, provide an
// alternative way to split lines.
var splitLinesAuto = "\n\nb".split(/\n/).length != 3 ? function (string) {
  var pos = 0, result = [], l = string.length
  while (pos <= l) {
    var nl = string.indexOf("\n", pos)
    if (nl == -1) { nl = string.length }
    var line = string.slice(pos, string.charAt(nl - 1) == "\r" ? nl - 1 : nl)
    var rt = line.indexOf("\r")
    if (rt != -1) {
      result.push(line.slice(0, rt))
      pos += rt + 1
    } else {
      result.push(line)
      pos = nl + 1
    }
  }
  return result
} : function (string) { return string.split(/\r\n?|\n/); }

var hasSelection = window.getSelection ? function (te) {
  try { return te.selectionStart != te.selectionEnd }
  catch(e) { return false }
} : function (te) {
  var range
  try {range = te.ownerDocument.selection.createRange()}
  catch(e) {}
  if (!range || range.parentElement() != te) { return false }
  return range.compareEndPoints("StartToEnd", range) != 0
}

var hasCopyEvent = (function () {
  var e = elt("div")
  if ("oncopy" in e) { return true }
  e.setAttribute("oncopy", "return;")
  return typeof e.oncopy == "function"
})()

var badZoomedRects = null
function hasBadZoomedRects(measure) {
  if (badZoomedRects != null) { return badZoomedRects }
  var node = removeChildrenAndAdd(measure, elt("span", "x"))
  var normal = node.getBoundingClientRect()
  var fromRange = range(node, 0, 1).getBoundingClientRect()
  return badZoomedRects = Math.abs(normal.left - fromRange.left) > 1
}

var modes = {};
var mimeModes = {};
// Extra arguments are stored as the mode's dependencies, which is
// used by (legacy) mechanisms like loadmode.js to automatically
// load a mode. (Preferred mechanism is the require/define calls.)
function defineMode(name, mode) {
  if (arguments.length > 2)
    { mode.dependencies = Array.prototype.slice.call(arguments, 2) }
  modes[name] = mode
}

function defineMIME(mime, spec) {
  mimeModes[mime] = spec
}

// Given a MIME type, a {name, ...options} config object, or a name
// string, return a mode config object.
function resolveMode(spec) {
  if (typeof spec == "string" && mimeModes.hasOwnProperty(spec)) {
    spec = mimeModes[spec]
  } else if (spec && typeof spec.name == "string" && mimeModes.hasOwnProperty(spec.name)) {
    var found = mimeModes[spec.name]
    if (typeof found == "string") { found = {name: found} }
    spec = createObj(found, spec)
    spec.name = found.name
  } else if (typeof spec == "string" && /^[\w\-]+\/[\w\-]+\+xml$/.test(spec)) {
    return resolveMode("application/xml")
  } else if (typeof spec == "string" && /^[\w\-]+\/[\w\-]+\+json$/.test(spec)) {
    return resolveMode("application/json")
  }
  if (typeof spec == "string") { return {name: spec} }
  else { return spec || {name: "null"} }
}

// Given a mode spec (anything that resolveMode accepts), find and
// initialize an actual mode object.
function getMode(options, spec) {
  spec = resolveMode(spec)
  var mfactory = modes[spec.name]
  if (!mfactory) { return getMode(options, "text/plain") }
  var modeObj = mfactory(options, spec)
  if (modeExtensions.hasOwnProperty(spec.name)) {
    var exts = modeExtensions[spec.name]
    for (var prop in exts) {
      if (!exts.hasOwnProperty(prop)) { continue }
      if (modeObj.hasOwnProperty(prop)) { modeObj["_" + prop] = modeObj[prop] }
      modeObj[prop] = exts[prop]
    }
  }
  modeObj.name = spec.name
  if (spec.helperType) { modeObj.helperType = spec.helperType }
  if (spec.modeProps) { for (var prop$1 in spec.modeProps)
    { modeObj[prop$1] = spec.modeProps[prop$1] } }

  return modeObj
}

// This can be used to attach properties to mode objects from
// outside the actual mode definition.
var modeExtensions = {}
function extendMode(mode, properties) {
  var exts = modeExtensions.hasOwnProperty(mode) ? modeExtensions[mode] : (modeExtensions[mode] = {})
  copyObj(properties, exts)
}

function copyState(mode, state) {
  if (state === true) { return state }
  if (mode.copyState) { return mode.copyState(state) }
  var nstate = {}
  for (var n in state) {
    var val = state[n]
    if (val instanceof Array) { val = val.concat([]) }
    nstate[n] = val
  }
  return nstate
}

// Given a mode and a state (for that mode), find the inner mode and
// state at the position that the state refers to.
function innerMode(mode, state) {
  var info
  while (mode.innerMode) {
    info = mode.innerMode(state)
    if (!info || info.mode == mode) { break }
    state = info.state
    mode = info.mode
  }
  return info || {mode: mode, state: state}
}

function startState(mode, a1, a2) {
  return mode.startState ? mode.startState(a1, a2) : true
}

// STRING STREAM

// Fed to the mode parsers, provides helper functions to make
// parsers more succinct.

var StringStream = function(string, tabSize) {
  this.pos = this.start = 0
  this.string = string
  this.tabSize = tabSize || 8
  this.lastColumnPos = this.lastColumnValue = 0
  this.lineStart = 0
};

StringStream.prototype.eol = function () {return this.pos >= this.string.length};
StringStream.prototype.sol = function () {return this.pos == this.lineStart};
StringStream.prototype.peek = function () {return this.string.charAt(this.pos) || undefined};
StringStream.prototype.next = function () {
  if (this.pos < this.string.length)
    { return this.string.charAt(this.pos++) }
};
StringStream.prototype.eat = function (match) {
  var ch = this.string.charAt(this.pos)
  var ok
  if (typeof match == "string") { ok = ch == match }
  else { ok = ch && (match.test ? match.test(ch) : match(ch)) }
  if (ok) {++this.pos; return ch}
};
StringStream.prototype.eatWhile = function (match) {
  var start = this.pos
  while (this.eat(match)){}
  return this.pos > start
};
StringStream.prototype.eatSpace = function () {
    var this$1 = this;

  var start = this.pos
  while (/[\s\u00a0]/.test(this.string.charAt(this.pos))) { ++this$1.pos }
  return this.pos > start
};
StringStream.prototype.skipToEnd = function () {this.pos = this.string.length};
StringStream.prototype.skipTo = function (ch) {
  var found = this.string.indexOf(ch, this.pos)
  if (found > -1) {this.pos = found; return true}
};
StringStream.prototype.backUp = function (n) {this.pos -= n};
StringStream.prototype.column = function () {
  if (this.lastColumnPos < this.start) {
    this.lastColumnValue = countColumn(this.string, this.start, this.tabSize, this.lastColumnPos, this.lastColumnValue)
    this.lastColumnPos = this.start
  }
  return this.lastColumnValue - (this.lineStart ? countColumn(this.string, this.lineStart, this.tabSize) : 0)
};
StringStream.prototype.indentation = function () {
  return countColumn(this.string, null, this.tabSize) -
    (this.lineStart ? countColumn(this.string, this.lineStart, this.tabSize) : 0)
};
StringStream.prototype.match = function (pattern, consume, caseInsensitive) {
  if (typeof pattern == "string") {
    var cased = function (str) { return caseInsensitive ? str.toLowerCase() : str; }
    var substr = this.string.substr(this.pos, pattern.length)
    if (cased(substr) == cased(pattern)) {
      if (consume !== false) { this.pos += pattern.length }
      return true
    }
  } else {
    var match = this.string.slice(this.pos).match(pattern)
    if (match && match.index > 0) { return null }
    if (match && consume !== false) { this.pos += match[0].length }
    return match
  }
};
StringStream.prototype.current = function (){return this.string.slice(this.start, this.pos)};
StringStream.prototype.hideFirstChars = function (n, inner) {
  this.lineStart += n
  try { return inner() }
  finally { this.lineStart -= n }
};

// Compute a style array (an array starting with a mode generation
// -- for invalidation -- followed by pairs of end positions and
// style strings), which is used to highlight the tokens on the
// line.
function highlightLine(cm, line, state, forceToEnd) {
  // A styles array always starts with a number identifying the
  // mode/overlays that it is based on (for easy invalidation).
  var st = [cm.state.modeGen], lineClasses = {}
  // Compute the base array of styles
  runMode(cm, line.text, cm.doc.mode, state, function (end, style) { return st.push(end, style); },
    lineClasses, forceToEnd)

  // Run overlays, adjust style array.
  var loop = function ( o ) {
    var overlay = cm.state.overlays[o], i = 1, at = 0
    runMode(cm, line.text, overlay.mode, true, function (end, style) {
      var start = i
      // Ensure there's a token end at the current position, and that i points at it
      while (at < end) {
        var i_end = st[i]
        if (i_end > end)
          { st.splice(i, 1, end, st[i+1], i_end) }
        i += 2
        at = Math.min(end, i_end)
      }
      if (!style) { return }
      if (overlay.opaque) {
        st.splice(start, i - start, end, "overlay " + style)
        i = start + 2
      } else {
        for (; start < i; start += 2) {
          var cur = st[start+1]
          st[start+1] = (cur ? cur + " " : "") + "overlay " + style
        }
      }
    }, lineClasses)
  };

  for (var o = 0; o < cm.state.overlays.length; ++o) loop( o );

  return {styles: st, classes: lineClasses.bgClass || lineClasses.textClass ? lineClasses : null}
}

function getLineStyles(cm, line, updateFrontier) {
  if (!line.styles || line.styles[0] != cm.state.modeGen) {
    var state = getStateBefore(cm, lineNo(line))
    var result = highlightLine(cm, line, line.text.length > cm.options.maxHighlightLength ? copyState(cm.doc.mode, state) : state)
    line.stateAfter = state
    line.styles = result.styles
    if (result.classes) { line.styleClasses = result.classes }
    else if (line.styleClasses) { line.styleClasses = null }
    if (updateFrontier === cm.doc.frontier) { cm.doc.frontier++ }
  }
  return line.styles
}

function getStateBefore(cm, n, precise) {
  var doc = cm.doc, display = cm.display
  if (!doc.mode.startState) { return true }
  var pos = findStartLine(cm, n, precise), state = pos > doc.first && getLine(doc, pos-1).stateAfter
  if (!state) { state = startState(doc.mode) }
  else { state = copyState(doc.mode, state) }
  doc.iter(pos, n, function (line) {
    processLine(cm, line.text, state)
    var save = pos == n - 1 || pos % 5 == 0 || pos >= display.viewFrom && pos < display.viewTo
    line.stateAfter = save ? copyState(doc.mode, state) : null
    ++pos
  })
  if (precise) { doc.frontier = pos }
  return state
}

// Lightweight form of highlight -- proceed over this line and
// update state, but don't save a style array. Used for lines that
// aren't currently visible.
function processLine(cm, text, state, startAt) {
  var mode = cm.doc.mode
  var stream = new StringStream(text, cm.options.tabSize)
  stream.start = stream.pos = startAt || 0
  if (text == "") { callBlankLine(mode, state) }
  while (!stream.eol()) {
    readToken(mode, stream, state)
    stream.start = stream.pos
  }
}

function callBlankLine(mode, state) {
  if (mode.blankLine) { return mode.blankLine(state) }
  if (!mode.innerMode) { return }
  var inner = innerMode(mode, state)
  if (inner.mode.blankLine) { return inner.mode.blankLine(inner.state) }
}

function readToken(mode, stream, state, inner) {
  for (var i = 0; i < 10; i++) {
    if (inner) { inner[0] = innerMode(mode, state).mode }
    var style = mode.token(stream, state)
    if (stream.pos > stream.start) { return style }
  }
  throw new Error("Mode " + mode.name + " failed to advance stream.")
}

// Utility for getTokenAt and getLineTokens
function takeToken(cm, pos, precise, asArray) {
  var getObj = function (copy) { return ({
    start: stream.start, end: stream.pos,
    string: stream.current(),
    type: style || null,
    state: copy ? copyState(doc.mode, state) : state
  }); }

  var doc = cm.doc, mode = doc.mode, style
  pos = clipPos(doc, pos)
  var line = getLine(doc, pos.line), state = getStateBefore(cm, pos.line, precise)
  var stream = new StringStream(line.text, cm.options.tabSize), tokens
  if (asArray) { tokens = [] }
  while ((asArray || stream.pos < pos.ch) && !stream.eol()) {
    stream.start = stream.pos
    style = readToken(mode, stream, state)
    if (asArray) { tokens.push(getObj(true)) }
  }
  return asArray ? tokens : getObj()
}

function extractLineClasses(type, output) {
  if (type) { for (;;) {
    var lineClass = type.match(/(?:^|\s+)line-(background-)?(\S+)/)
    if (!lineClass) { break }
    type = type.slice(0, lineClass.index) + type.slice(lineClass.index + lineClass[0].length)
    var prop = lineClass[1] ? "bgClass" : "textClass"
    if (output[prop] == null)
      { output[prop] = lineClass[2] }
    else if (!(new RegExp("(?:^|\s)" + lineClass[2] + "(?:$|\s)")).test(output[prop]))
      { output[prop] += " " + lineClass[2] }
  } }
  return type
}

// Run the given mode's parser over a line, calling f for each token.
function runMode(cm, text, mode, state, f, lineClasses, forceToEnd) {
  var flattenSpans = mode.flattenSpans
  if (flattenSpans == null) { flattenSpans = cm.options.flattenSpans }
  var curStart = 0, curStyle = null
  var stream = new StringStream(text, cm.options.tabSize), style
  var inner = cm.options.addModeClass && [null]
  if (text == "") { extractLineClasses(callBlankLine(mode, state), lineClasses) }
  while (!stream.eol()) {
    if (stream.pos > cm.options.maxHighlightLength) {
      flattenSpans = false
      if (forceToEnd) { processLine(cm, text, state, stream.pos) }
      stream.pos = text.length
      style = null
    } else {
      style = extractLineClasses(readToken(mode, stream, state, inner), lineClasses)
    }
    if (inner) {
      var mName = inner[0].name
      if (mName) { style = "m-" + (style ? mName + " " + style : mName) }
    }
    if (!flattenSpans || curStyle != style) {
      while (curStart < stream.start) {
        curStart = Math.min(stream.start, curStart + 5000)
        f(curStart, curStyle)
      }
      curStyle = style
    }
    stream.start = stream.pos
  }
  while (curStart < stream.pos) {
    // Webkit seems to refuse to render text nodes longer than 57444
    // characters, and returns inaccurate measurements in nodes
    // starting around 5000 chars.
    var pos = Math.min(stream.pos, curStart + 5000)
    f(pos, curStyle)
    curStart = pos
  }
}

// Finds the line to start with when starting a parse. Tries to
// find a line with a stateAfter, so that it can start with a
// valid state. If that fails, it returns the line with the
// smallest indentation, which tends to need the least context to
// parse correctly.
function findStartLine(cm, n, precise) {
  var minindent, minline, doc = cm.doc
  var lim = precise ? -1 : n - (cm.doc.mode.innerMode ? 1000 : 100)
  for (var search = n; search > lim; --search) {
    if (search <= doc.first) { return doc.first }
    var line = getLine(doc, search - 1)
    if (line.stateAfter && (!precise || search <= doc.frontier)) { return search }
    var indented = countColumn(line.text, null, cm.options.tabSize)
    if (minline == null || minindent > indented) {
      minline = search - 1
      minindent = indented
    }
  }
  return minline
}

// LINE DATA STRUCTURE

// Line objects. These hold state related to a line, including
// highlighting info (the styles array).
var Line = function(text, markedSpans, estimateHeight) {
  this.text = text
  attachMarkedSpans(this, markedSpans)
  this.height = estimateHeight ? estimateHeight(this) : 1
};

Line.prototype.lineNo = function () { return lineNo(this) };
eventMixin(Line)

// Change the content (text, markers) of a line. Automatically
// invalidates cached information and tries to re-estimate the
// line's height.
function updateLine(line, text, markedSpans, estimateHeight) {
  line.text = text
  if (line.stateAfter) { line.stateAfter = null }
  if (line.styles) { line.styles = null }
  if (line.order != null) { line.order = null }
  detachMarkedSpans(line)
  attachMarkedSpans(line, markedSpans)
  var estHeight = estimateHeight ? estimateHeight(line) : 1
  if (estHeight != line.height) { updateLineHeight(line, estHeight) }
}

// Detach a line from the document tree and its markers.
function cleanUpLine(line) {
  line.parent = null
  detachMarkedSpans(line)
}

// Convert a style as returned by a mode (either null, or a string
// containing one or more styles) to a CSS style. This is cached,
// and also looks for line-wide styles.
var styleToClassCache = {};
var styleToClassCacheWithMode = {};
function interpretTokenStyle(style, options) {
  if (!style || /^\s*$/.test(style)) { return null }
  var cache = options.addModeClass ? styleToClassCacheWithMode : styleToClassCache
  return cache[style] ||
    (cache[style] = style.replace(/\S+/g, "cm-$&"))
}

// Render the DOM representation of the text of a line. Also builds
// up a 'line map', which points at the DOM nodes that represent
// specific stretches of text, and is used by the measuring code.
// The returned object contains the DOM node, this map, and
// information about line-wide styles that were set by the mode.
function buildLineContent(cm, lineView) {
  // The padding-right forces the element to have a 'border', which
  // is needed on Webkit to be able to get line-level bounding
  // rectangles for it (in measureChar).
  var content = elt("span", null, null, webkit ? "padding-right: .1px" : null)
  var builder = {pre: elt("pre", [content], "CodeMirror-line"), content: content,
                 col: 0, pos: 0, cm: cm,
                 trailingSpace: false,
                 splitSpaces: (ie || webkit) && cm.getOption("lineWrapping")}
  // hide from accessibility tree
  content.setAttribute("role", "presentation")
  builder.pre.setAttribute("role", "presentation")
  lineView.measure = {}

  // Iterate over the logical lines that make up this visual line.
  for (var i = 0; i <= (lineView.rest ? lineView.rest.length : 0); i++) {
    var line = i ? lineView.rest[i - 1] : lineView.line, order = (void 0)
    builder.pos = 0
    builder.addToken = buildToken
    // Optionally wire in some hacks into the token-rendering
    // algorithm, to deal with browser quirks.
    if (hasBadBidiRects(cm.display.measure) && (order = getOrder(line)))
      { builder.addToken = buildTokenBadBidi(builder.addToken, order) }
    builder.map = []
    var allowFrontierUpdate = lineView != cm.display.externalMeasured && lineNo(line)
    insertLineContent(line, builder, getLineStyles(cm, line, allowFrontierUpdate))
    if (line.styleClasses) {
      if (line.styleClasses.bgClass)
        { builder.bgClass = joinClasses(line.styleClasses.bgClass, builder.bgClass || "") }
      if (line.styleClasses.textClass)
        { builder.textClass = joinClasses(line.styleClasses.textClass, builder.textClass || "") }
    }

    // Ensure at least a single node is present, for measuring.
    if (builder.map.length == 0)
      { builder.map.push(0, 0, builder.content.appendChild(zeroWidthElement(cm.display.measure))) }

    // Store the map and a cache object for the current logical line
    if (i == 0) {
      lineView.measure.map = builder.map
      lineView.measure.cache = {}
    } else {
      ;(lineView.measure.maps || (lineView.measure.maps = [])).push(builder.map)
      ;(lineView.measure.caches || (lineView.measure.caches = [])).push({})
    }
  }

  // See issue #2901
  if (webkit) {
    var last = builder.content.lastChild
    if (/\bcm-tab\b/.test(last.className) || (last.querySelector && last.querySelector(".cm-tab")))
      { builder.content.className = "cm-tab-wrap-hack" }
  }

  signal(cm, "renderLine", cm, lineView.line, builder.pre)
  if (builder.pre.className)
    { builder.textClass = joinClasses(builder.pre.className, builder.textClass || "") }

  return builder
}

function defaultSpecialCharPlaceholder(ch) {
  var token = elt("span", "\u2022", "cm-invalidchar")
  token.title = "\\u" + ch.charCodeAt(0).toString(16)
  token.setAttribute("aria-label", token.title)
  return token
}

// Build up the DOM representation for a single token, and add it to
// the line map. Takes care to render special characters separately.
function buildToken(builder, text, style, startStyle, endStyle, title, css) {
  if (!text) { return }
  var displayText = builder.splitSpaces ? splitSpaces(text, builder.trailingSpace) : text
  var special = builder.cm.state.specialChars, mustWrap = false
  var content
  if (!special.test(text)) {
    builder.col += text.length
    content = document.createTextNode(displayText)
    builder.map.push(builder.pos, builder.pos + text.length, content)
    if (ie && ie_version < 9) { mustWrap = true }
    builder.pos += text.length
  } else {
    content = document.createDocumentFragment()
    var pos = 0
    while (true) {
      special.lastIndex = pos
      var m = special.exec(text)
      var skipped = m ? m.index - pos : text.length - pos
      if (skipped) {
        var txt = document.createTextNode(displayText.slice(pos, pos + skipped))
        if (ie && ie_version < 9) { content.appendChild(elt("span", [txt])) }
        else { content.appendChild(txt) }
        builder.map.push(builder.pos, builder.pos + skipped, txt)
        builder.col += skipped
        builder.pos += skipped
      }
      if (!m) { break }
      pos += skipped + 1
      var txt$1 = (void 0)
      if (m[0] == "\t") {
        var tabSize = builder.cm.options.tabSize, tabWidth = tabSize - builder.col % tabSize
        txt$1 = content.appendChild(elt("span", spaceStr(tabWidth), "cm-tab"))
        txt$1.setAttribute("role", "presentation")
        txt$1.setAttribute("cm-text", "\t")
        builder.col += tabWidth
      } else if (m[0] == "\r" || m[0] == "\n") {
        txt$1 = content.appendChild(elt("span", m[0] == "\r" ? "\u240d" : "\u2424", "cm-invalidchar"))
        txt$1.setAttribute("cm-text", m[0])
        builder.col += 1
      } else {
        txt$1 = builder.cm.options.specialCharPlaceholder(m[0])
        txt$1.setAttribute("cm-text", m[0])
        if (ie && ie_version < 9) { content.appendChild(elt("span", [txt$1])) }
        else { content.appendChild(txt$1) }
        builder.col += 1
      }
      builder.map.push(builder.pos, builder.pos + 1, txt$1)
      builder.pos++
    }
  }
  builder.trailingSpace = displayText.charCodeAt(text.length - 1) == 32
  if (style || startStyle || endStyle || mustWrap || css) {
    var fullStyle = style || ""
    if (startStyle) { fullStyle += startStyle }
    if (endStyle) { fullStyle += endStyle }
    var token = elt("span", [content], fullStyle, css)
    if (title) { token.title = title }
    return builder.content.appendChild(token)
  }
  builder.content.appendChild(content)
}

function splitSpaces(text, trailingBefore) {
  if (text.length > 1 && !/  /.test(text)) { return text }
  var spaceBefore = trailingBefore, result = ""
  for (var i = 0; i < text.length; i++) {
    var ch = text.charAt(i)
    if (ch == " " && spaceBefore && (i == text.length - 1 || text.charCodeAt(i + 1) == 32))
      { ch = "\u00a0" }
    result += ch
    spaceBefore = ch == " "
  }
  return result
}

// Work around nonsense dimensions being reported for stretches of
// right-to-left text.
function buildTokenBadBidi(inner, order) {
  return function (builder, text, style, startStyle, endStyle, title, css) {
    style = style ? style + " cm-force-border" : "cm-force-border"
    var start = builder.pos, end = start + text.length
    for (;;) {
      // Find the part that overlaps with the start of this text
      var part = (void 0)
      for (var i = 0; i < order.length; i++) {
        part = order[i]
        if (part.to > start && part.from <= start) { break }
      }
      if (part.to >= end) { return inner(builder, text, style, startStyle, endStyle, title, css) }
      inner(builder, text.slice(0, part.to - start), style, startStyle, null, title, css)
      startStyle = null
      text = text.slice(part.to - start)
      start = part.to
    }
  }
}

function buildCollapsedSpan(builder, size, marker, ignoreWidget) {
  var widget = !ignoreWidget && marker.widgetNode
  if (widget) { builder.map.push(builder.pos, builder.pos + size, widget) }
  if (!ignoreWidget && builder.cm.display.input.needsContentAttribute) {
    if (!widget)
      { widget = builder.content.appendChild(document.createElement("span")) }
    widget.setAttribute("cm-marker", marker.id)
  }
  if (widget) {
    builder.cm.display.input.setUneditable(widget)
    builder.content.appendChild(widget)
  }
  builder.pos += size
  builder.trailingSpace = false
}

// Outputs a number of spans to make up a line, taking highlighting
// and marked text into account.
function insertLineContent(line, builder, styles) {
  var spans = line.markedSpans, allText = line.text, at = 0
  if (!spans) {
    for (var i$1 = 1; i$1 < styles.length; i$1+=2)
      { builder.addToken(builder, allText.slice(at, at = styles[i$1]), interpretTokenStyle(styles[i$1+1], builder.cm.options)) }
    return
  }

  var len = allText.length, pos = 0, i = 1, text = "", style, css
  var nextChange = 0, spanStyle, spanEndStyle, spanStartStyle, title, collapsed
  for (;;) {
    if (nextChange == pos) { // Update current marker set
      spanStyle = spanEndStyle = spanStartStyle = title = css = ""
      collapsed = null; nextChange = Infinity
      var foundBookmarks = [], endStyles = (void 0)
      for (var j = 0; j < spans.length; ++j) {
        var sp = spans[j], m = sp.marker
        if (m.type == "bookmark" && sp.from == pos && m.widgetNode) {
          foundBookmarks.push(m)
        } else if (sp.from <= pos && (sp.to == null || sp.to > pos || m.collapsed && sp.to == pos && sp.from == pos)) {
          if (sp.to != null && sp.to != pos && nextChange > sp.to) {
            nextChange = sp.to
            spanEndStyle = ""
          }
          if (m.className) { spanStyle += " " + m.className }
          if (m.css) { css = (css ? css + ";" : "") + m.css }
          if (m.startStyle && sp.from == pos) { spanStartStyle += " " + m.startStyle }
          if (m.endStyle && sp.to == nextChange) { (endStyles || (endStyles = [])).push(m.endStyle, sp.to) }
          if (m.title && !title) { title = m.title }
          if (m.collapsed && (!collapsed || compareCollapsedMarkers(collapsed.marker, m) < 0))
            { collapsed = sp }
        } else if (sp.from > pos && nextChange > sp.from) {
          nextChange = sp.from
        }
      }
      if (endStyles) { for (var j$1 = 0; j$1 < endStyles.length; j$1 += 2)
        { if (endStyles[j$1 + 1] == nextChange) { spanEndStyle += " " + endStyles[j$1] } } }

      if (!collapsed || collapsed.from == pos) { for (var j$2 = 0; j$2 < foundBookmarks.length; ++j$2)
        { buildCollapsedSpan(builder, 0, foundBookmarks[j$2]) } }
      if (collapsed && (collapsed.from || 0) == pos) {
        buildCollapsedSpan(builder, (collapsed.to == null ? len + 1 : collapsed.to) - pos,
                           collapsed.marker, collapsed.from == null)
        if (collapsed.to == null) { return }
        if (collapsed.to == pos) { collapsed = false }
      }
    }
    if (pos >= len) { break }

    var upto = Math.min(len, nextChange)
    while (true) {
      if (text) {
        var end = pos + text.length
        if (!collapsed) {
          var tokenText = end > upto ? text.slice(0, upto - pos) : text
          builder.addToken(builder, tokenText, style ? style + spanStyle : spanStyle,
                           spanStartStyle, pos + tokenText.length == nextChange ? spanEndStyle : "", title, css)
        }
        if (end >= upto) {text = text.slice(upto - pos); pos = upto; break}
        pos = end
        spanStartStyle = ""
      }
      text = allText.slice(at, at = styles[i++])
      style = interpretTokenStyle(styles[i++], builder.cm.options)
    }
  }
}


// These objects are used to represent the visible (currently drawn)
// part of the document. A LineView may correspond to multiple
// logical lines, if those are connected by collapsed ranges.
function LineView(doc, line, lineN) {
  // The starting line
  this.line = line
  // Continuing lines, if any
  this.rest = visualLineContinued(line)
  // Number of logical lines in this visual line
  this.size = this.rest ? lineNo(lst(this.rest)) - lineN + 1 : 1
  this.node = this.text = null
  this.hidden = lineIsHidden(doc, line)
}

// Create a range of LineView objects for the given lines.
function buildViewArray(cm, from, to) {
  var array = [], nextPos
  for (var pos = from; pos < to; pos = nextPos) {
    var view = new LineView(cm.doc, getLine(cm.doc, pos), pos)
    nextPos = pos + view.size
    array.push(view)
  }
  return array
}

var operationGroup = null

function pushOperation(op) {
  if (operationGroup) {
    operationGroup.ops.push(op)
  } else {
    op.ownsGroup = operationGroup = {
      ops: [op],
      delayedCallbacks: []
    }
  }
}

function fireCallbacksForOps(group) {
  // Calls delayed callbacks and cursorActivity handlers until no
  // new ones appear
  var callbacks = group.delayedCallbacks, i = 0
  do {
    for (; i < callbacks.length; i++)
      { callbacks[i].call(null) }
    for (var j = 0; j < group.ops.length; j++) {
      var op = group.ops[j]
      if (op.cursorActivityHandlers)
        { while (op.cursorActivityCalled < op.cursorActivityHandlers.length)
          { op.cursorActivityHandlers[op.cursorActivityCalled++].call(null, op.cm) } }
    }
  } while (i < callbacks.length)
}

function finishOperation(op, endCb) {
  var group = op.ownsGroup
  if (!group) { return }

  try { fireCallbacksForOps(group) }
  finally {
    operationGroup = null
    endCb(group)
  }
}

var orphanDelayedCallbacks = null

// Often, we want to signal events at a point where we are in the
// middle of some work, but don't want the handler to start calling
// other methods on the editor, which might be in an inconsistent
// state or simply not expect any other events to happen.
// signalLater looks whether there are any handlers, and schedules
// them to be executed when the last operation ends, or, if no
// operation is active, when a timeout fires.
function signalLater(emitter, type /*, values...*/) {
  var arr = getHandlers(emitter, type)
  if (!arr.length) { return }
  var args = Array.prototype.slice.call(arguments, 2), list
  if (operationGroup) {
    list = operationGroup.delayedCallbacks
  } else if (orphanDelayedCallbacks) {
    list = orphanDelayedCallbacks
  } else {
    list = orphanDelayedCallbacks = []
    setTimeout(fireOrphanDelayed, 0)
  }
  var loop = function ( i ) {
    list.push(function () { return arr[i].apply(null, args); })
  };

  for (var i = 0; i < arr.length; ++i)
    loop( i );
}

function fireOrphanDelayed() {
  var delayed = orphanDelayedCallbacks
  orphanDelayedCallbacks = null
  for (var i = 0; i < delayed.length; ++i) { delayed[i]() }
}

// When an aspect of a line changes, a string is added to
// lineView.changes. This updates the relevant part of the line's
// DOM structure.
function updateLineForChanges(cm, lineView, lineN, dims) {
  for (var j = 0; j < lineView.changes.length; j++) {
    var type = lineView.changes[j]
    if (type == "text") { updateLineText(cm, lineView) }
    else if (type == "gutter") { updateLineGutter(cm, lineView, lineN, dims) }
    else if (type == "class") { updateLineClasses(lineView) }
    else if (type == "widget") { updateLineWidgets(cm, lineView, dims) }
  }
  lineView.changes = null
}

// Lines with gutter elements, widgets or a background class need to
// be wrapped, and have the extra elements added to the wrapper div
function ensureLineWrapped(lineView) {
  if (lineView.node == lineView.text) {
    lineView.node = elt("div", null, null, "position: relative")
    if (lineView.text.parentNode)
      { lineView.text.parentNode.replaceChild(lineView.node, lineView.text) }
    lineView.node.appendChild(lineView.text)
    if (ie && ie_version < 8) { lineView.node.style.zIndex = 2 }
  }
  return lineView.node
}

function updateLineBackground(lineView) {
  var cls = lineView.bgClass ? lineView.bgClass + " " + (lineView.line.bgClass || "") : lineView.line.bgClass
  if (cls) { cls += " CodeMirror-linebackground" }
  if (lineView.background) {
    if (cls) { lineView.background.className = cls }
    else { lineView.background.parentNode.removeChild(lineView.background); lineView.background = null }
  } else if (cls) {
    var wrap = ensureLineWrapped(lineView)
    lineView.background = wrap.insertBefore(elt("div", null, cls), wrap.firstChild)
  }
}

// Wrapper around buildLineContent which will reuse the structure
// in display.externalMeasured when possible.
function getLineContent(cm, lineView) {
  var ext = cm.display.externalMeasured
  if (ext && ext.line == lineView.line) {
    cm.display.externalMeasured = null
    lineView.measure = ext.measure
    return ext.built
  }
  return buildLineContent(cm, lineView)
}

// Redraw the line's text. Interacts with the background and text
// classes because the mode may output tokens that influence these
// classes.
function updateLineText(cm, lineView) {
  var cls = lineView.text.className
  var built = getLineContent(cm, lineView)
  if (lineView.text == lineView.node) { lineView.node = built.pre }
  lineView.text.parentNode.replaceChild(built.pre, lineView.text)
  lineView.text = built.pre
  if (built.bgClass != lineView.bgClass || built.textClass != lineView.textClass) {
    lineView.bgClass = built.bgClass
    lineView.textClass = built.textClass
    updateLineClasses(lineView)
  } else if (cls) {
    lineView.text.className = cls
  }
}

function updateLineClasses(lineView) {
  updateLineBackground(lineView)
  if (lineView.line.wrapClass)
    { ensureLineWrapped(lineView).className = lineView.line.wrapClass }
  else if (lineView.node != lineView.text)
    { lineView.node.className = "" }
  var textClass = lineView.textClass ? lineView.textClass + " " + (lineView.line.textClass || "") : lineView.line.textClass
  lineView.text.className = textClass || ""
}

function updateLineGutter(cm, lineView, lineN, dims) {
  if (lineView.gutter) {
    lineView.node.removeChild(lineView.gutter)
    lineView.gutter = null
  }
  if (lineView.gutterBackground) {
    lineView.node.removeChild(lineView.gutterBackground)
    lineView.gutterBackground = null
  }
  if (lineView.line.gutterClass) {
    var wrap = ensureLineWrapped(lineView)
    lineView.gutterBackground = elt("div", null, "CodeMirror-gutter-background " + lineView.line.gutterClass,
                                    ("left: " + (cm.options.fixedGutter ? dims.fixedPos : -dims.gutterTotalWidth) + "px; width: " + (dims.gutterTotalWidth) + "px"))
    wrap.insertBefore(lineView.gutterBackground, lineView.text)
  }
  var markers = lineView.line.gutterMarkers
  if (cm.options.lineNumbers || markers) {
    var wrap$1 = ensureLineWrapped(lineView)
    var gutterWrap = lineView.gutter = elt("div", null, "CodeMirror-gutter-wrapper", ("left: " + (cm.options.fixedGutter ? dims.fixedPos : -dims.gutterTotalWidth) + "px"))
    cm.display.input.setUneditable(gutterWrap)
    wrap$1.insertBefore(gutterWrap, lineView.text)
    if (lineView.line.gutterClass)
      { gutterWrap.className += " " + lineView.line.gutterClass }
    if (cm.options.lineNumbers && (!markers || !markers["CodeMirror-linenumbers"]))
      { lineView.lineNumber = gutterWrap.appendChild(
        elt("div", lineNumberFor(cm.options, lineN),
            "CodeMirror-linenumber CodeMirror-gutter-elt",
            ("left: " + (dims.gutterLeft["CodeMirror-linenumbers"]) + "px; width: " + (cm.display.lineNumInnerWidth) + "px"))) }
    if (markers) { for (var k = 0; k < cm.options.gutters.length; ++k) {
      var id = cm.options.gutters[k], found = markers.hasOwnProperty(id) && markers[id]
      if (found)
        { gutterWrap.appendChild(elt("div", [found], "CodeMirror-gutter-elt",
                                   ("left: " + (dims.gutterLeft[id]) + "px; width: " + (dims.gutterWidth[id]) + "px"))) }
    } }
  }
}

function updateLineWidgets(cm, lineView, dims) {
  if (lineView.alignable) { lineView.alignable = null }
  for (var node = lineView.node.firstChild, next = (void 0); node; node = next) {
    next = node.nextSibling
    if (node.className == "CodeMirror-linewidget")
      { lineView.node.removeChild(node) }
  }
  insertLineWidgets(cm, lineView, dims)
}

// Build a line's DOM representation from scratch
function buildLineElement(cm, lineView, lineN, dims) {
  var built = getLineContent(cm, lineView)
  lineView.text = lineView.node = built.pre
  if (built.bgClass) { lineView.bgClass = built.bgClass }
  if (built.textClass) { lineView.textClass = built.textClass }

  updateLineClasses(lineView)
  updateLineGutter(cm, lineView, lineN, dims)
  insertLineWidgets(cm, lineView, dims)
  return lineView.node
}

// A lineView may contain multiple logical lines (when merged by
// collapsed spans). The widgets for all of them need to be drawn.
function insertLineWidgets(cm, lineView, dims) {
  insertLineWidgetsFor(cm, lineView.line, lineView, dims, true)
  if (lineView.rest) { for (var i = 0; i < lineView.rest.length; i++)
    { insertLineWidgetsFor(cm, lineView.rest[i], lineView, dims, false) } }
}

function insertLineWidgetsFor(cm, line, lineView, dims, allowAbove) {
  if (!line.widgets) { return }
  var wrap = ensureLineWrapped(lineView)
  for (var i = 0, ws = line.widgets; i < ws.length; ++i) {
    var widget = ws[i], node = elt("div", [widget.node], "CodeMirror-linewidget")
    if (!widget.handleMouseEvents) { node.setAttribute("cm-ignore-events", "true") }
    positionLineWidget(widget, node, lineView, dims)
    cm.display.input.setUneditable(node)
    if (allowAbove && widget.above)
      { wrap.insertBefore(node, lineView.gutter || lineView.text) }
    else
      { wrap.appendChild(node) }
    signalLater(widget, "redraw")
  }
}

function positionLineWidget(widget, node, lineView, dims) {
  if (widget.noHScroll) {
    ;(lineView.alignable || (lineView.alignable = [])).push(node)
    var width = dims.wrapperWidth
    node.style.left = dims.fixedPos + "px"
    if (!widget.coverGutter) {
      width -= dims.gutterTotalWidth
      node.style.paddingLeft = dims.gutterTotalWidth + "px"
    }
    node.style.width = width + "px"
  }
  if (widget.coverGutter) {
    node.style.zIndex = 5
    node.style.position = "relative"
    if (!widget.noHScroll) { node.style.marginLeft = -dims.gutterTotalWidth + "px" }
  }
}

function widgetHeight(widget) {
  if (widget.height != null) { return widget.height }
  var cm = widget.doc.cm
  if (!cm) { return 0 }
  if (!contains(document.body, widget.node)) {
    var parentStyle = "position: relative;"
    if (widget.coverGutter)
      { parentStyle += "margin-left: -" + cm.display.gutters.offsetWidth + "px;" }
    if (widget.noHScroll)
      { parentStyle += "width: " + cm.display.wrapper.clientWidth + "px;" }
    removeChildrenAndAdd(cm.display.measure, elt("div", [widget.node], null, parentStyle))
  }
  return widget.height = widget.node.parentNode.offsetHeight
}

// Return true when the given mouse event happened in a widget
function eventInWidget(display, e) {
  for (var n = e_target(e); n != display.wrapper; n = n.parentNode) {
    if (!n || (n.nodeType == 1 && n.getAttribute("cm-ignore-events") == "true") ||
        (n.parentNode == display.sizer && n != display.mover))
      { return true }
  }
}

// POSITION MEASUREMENT

function paddingTop(display) {return display.lineSpace.offsetTop}
function paddingVert(display) {return display.mover.offsetHeight - display.lineSpace.offsetHeight}
function paddingH(display) {
  if (display.cachedPaddingH) { return display.cachedPaddingH }
  var e = removeChildrenAndAdd(display.measure, elt("pre", "x"))
  var style = window.getComputedStyle ? window.getComputedStyle(e) : e.currentStyle
  var data = {left: parseInt(style.paddingLeft), right: parseInt(style.paddingRight)}
  if (!isNaN(data.left) && !isNaN(data.right)) { display.cachedPaddingH = data }
  return data
}

function scrollGap(cm) { return scrollerGap - cm.display.nativeBarWidth }
function displayWidth(cm) {
  return cm.display.scroller.clientWidth - scrollGap(cm) - cm.display.barWidth
}
function displayHeight(cm) {
  return cm.display.scroller.clientHeight - scrollGap(cm) - cm.display.barHeight
}

// Ensure the lineView.wrapping.heights array is populated. This is
// an array of bottom offsets for the lines that make up a drawn
// line. When lineWrapping is on, there might be more than one
// height.
function ensureLineHeights(cm, lineView, rect) {
  var wrapping = cm.options.lineWrapping
  var curWidth = wrapping && displayWidth(cm)
  if (!lineView.measure.heights || wrapping && lineView.measure.width != curWidth) {
    var heights = lineView.measure.heights = []
    if (wrapping) {
      lineView.measure.width = curWidth
      var rects = lineView.text.firstChild.getClientRects()
      for (var i = 0; i < rects.length - 1; i++) {
        var cur = rects[i], next = rects[i + 1]
        if (Math.abs(cur.bottom - next.bottom) > 2)
          { heights.push((cur.bottom + next.top) / 2 - rect.top) }
      }
    }
    heights.push(rect.bottom - rect.top)
  }
}

// Find a line map (mapping character offsets to text nodes) and a
// measurement cache for the given line number. (A line view might
// contain multiple lines when collapsed ranges are present.)
function mapFromLineView(lineView, line, lineN) {
  if (lineView.line == line)
    { return {map: lineView.measure.map, cache: lineView.measure.cache} }
  for (var i = 0; i < lineView.rest.length; i++)
    { if (lineView.rest[i] == line)
      { return {map: lineView.measure.maps[i], cache: lineView.measure.caches[i]} } }
  for (var i$1 = 0; i$1 < lineView.rest.length; i$1++)
    { if (lineNo(lineView.rest[i$1]) > lineN)
      { return {map: lineView.measure.maps[i$1], cache: lineView.measure.caches[i$1], before: true} } }
}

// Render a line into the hidden node display.externalMeasured. Used
// when measurement is needed for a line that's not in the viewport.
function updateExternalMeasurement(cm, line) {
  line = visualLine(line)
  var lineN = lineNo(line)
  var view = cm.display.externalMeasured = new LineView(cm.doc, line, lineN)
  view.lineN = lineN
  var built = view.built = buildLineContent(cm, view)
  view.text = built.pre
  removeChildrenAndAdd(cm.display.lineMeasure, built.pre)
  return view
}

// Get a {top, bottom, left, right} box (in line-local coordinates)
// for a given character.
function measureChar(cm, line, ch, bias) {
  return measureCharPrepared(cm, prepareMeasureForLine(cm, line), ch, bias)
}

// Find a line view that corresponds to the given line number.
function findViewForLine(cm, lineN) {
  if (lineN >= cm.display.viewFrom && lineN < cm.display.viewTo)
    { return cm.display.view[findViewIndex(cm, lineN)] }
  var ext = cm.display.externalMeasured
  if (ext && lineN >= ext.lineN && lineN < ext.lineN + ext.size)
    { return ext }
}

// Measurement can be split in two steps, the set-up work that
// applies to the whole line, and the measurement of the actual
// character. Functions like coordsChar, that need to do a lot of
// measurements in a row, can thus ensure that the set-up work is
// only done once.
function prepareMeasureForLine(cm, line) {
  var lineN = lineNo(line)
  var view = findViewForLine(cm, lineN)
  if (view && !view.text) {
    view = null
  } else if (view && view.changes) {
    updateLineForChanges(cm, view, lineN, getDimensions(cm))
    cm.curOp.forceUpdate = true
  }
  if (!view)
    { view = updateExternalMeasurement(cm, line) }

  var info = mapFromLineView(view, line, lineN)
  return {
    line: line, view: view, rect: null,
    map: info.map, cache: info.cache, before: info.before,
    hasHeights: false
  }
}

// Given a prepared measurement object, measures the position of an
// actual character (or fetches it from the cache).
function measureCharPrepared(cm, prepared, ch, bias, varHeight) {
  if (prepared.before) { ch = -1 }
  var key = ch + (bias || ""), found
  if (prepared.cache.hasOwnProperty(key)) {
    found = prepared.cache[key]
  } else {
    if (!prepared.rect)
      { prepared.rect = prepared.view.text.getBoundingClientRect() }
    if (!prepared.hasHeights) {
      ensureLineHeights(cm, prepared.view, prepared.rect)
      prepared.hasHeights = true
    }
    found = measureCharInner(cm, prepared, ch, bias)
    if (!found.bogus) { prepared.cache[key] = found }
  }
  return {left: found.left, right: found.right,
          top: varHeight ? found.rtop : found.top,
          bottom: varHeight ? found.rbottom : found.bottom}
}

var nullRect = {left: 0, right: 0, top: 0, bottom: 0}

function nodeAndOffsetInLineMap(map, ch, bias) {
  var node, start, end, collapse, mStart, mEnd
  // First, search the line map for the text node corresponding to,
  // or closest to, the target character.
  for (var i = 0; i < map.length; i += 3) {
    mStart = map[i]
    mEnd = map[i + 1]
    if (ch < mStart) {
      start = 0; end = 1
      collapse = "left"
    } else if (ch < mEnd) {
      start = ch - mStart
      end = start + 1
    } else if (i == map.length - 3 || ch == mEnd && map[i + 3] > ch) {
      end = mEnd - mStart
      start = end - 1
      if (ch >= mEnd) { collapse = "right" }
    }
    if (start != null) {
      node = map[i + 2]
      if (mStart == mEnd && bias == (node.insertLeft ? "left" : "right"))
        { collapse = bias }
      if (bias == "left" && start == 0)
        { while (i && map[i - 2] == map[i - 3] && map[i - 1].insertLeft) {
          node = map[(i -= 3) + 2]
          collapse = "left"
        } }
      if (bias == "right" && start == mEnd - mStart)
        { while (i < map.length - 3 && map[i + 3] == map[i + 4] && !map[i + 5].insertLeft) {
          node = map[(i += 3) + 2]
          collapse = "right"
        } }
      break
    }
  }
  return {node: node, start: start, end: end, collapse: collapse, coverStart: mStart, coverEnd: mEnd}
}

function getUsefulRect(rects, bias) {
  var rect = nullRect
  if (bias == "left") { for (var i = 0; i < rects.length; i++) {
    if ((rect = rects[i]).left != rect.right) { break }
  } } else { for (var i$1 = rects.length - 1; i$1 >= 0; i$1--) {
    if ((rect = rects[i$1]).left != rect.right) { break }
  } }
  return rect
}

function measureCharInner(cm, prepared, ch, bias) {
  var place = nodeAndOffsetInLineMap(prepared.map, ch, bias)
  var node = place.node, start = place.start, end = place.end, collapse = place.collapse

  var rect
  if (node.nodeType == 3) { // If it is a text node, use a range to retrieve the coordinates.
    for (var i$1 = 0; i$1 < 4; i$1++) { // Retry a maximum of 4 times when nonsense rectangles are returned
      while (start && isExtendingChar(prepared.line.text.charAt(place.coverStart + start))) { --start }
      while (place.coverStart + end < place.coverEnd && isExtendingChar(prepared.line.text.charAt(place.coverStart + end))) { ++end }
      if (ie && ie_version < 9 && start == 0 && end == place.coverEnd - place.coverStart)
        { rect = node.parentNode.getBoundingClientRect() }
      else
        { rect = getUsefulRect(range(node, start, end).getClientRects(), bias) }
      if (rect.left || rect.right || start == 0) { break }
      end = start
      start = start - 1
      collapse = "right"
    }
    if (ie && ie_version < 11) { rect = maybeUpdateRectForZooming(cm.display.measure, rect) }
  } else { // If it is a widget, simply get the box for the whole widget.
    if (start > 0) { collapse = bias = "right" }
    var rects
    if (cm.options.lineWrapping && (rects = node.getClientRects()).length > 1)
      { rect = rects[bias == "right" ? rects.length - 1 : 0] }
    else
      { rect = node.getBoundingClientRect() }
  }
  if (ie && ie_version < 9 && !start && (!rect || !rect.left && !rect.right)) {
    var rSpan = node.parentNode.getClientRects()[0]
    if (rSpan)
      { rect = {left: rSpan.left, right: rSpan.left + charWidth(cm.display), top: rSpan.top, bottom: rSpan.bottom} }
    else
      { rect = nullRect }
  }

  var rtop = rect.top - prepared.rect.top, rbot = rect.bottom - prepared.rect.top
  var mid = (rtop + rbot) / 2
  var heights = prepared.view.measure.heights
  var i = 0
  for (; i < heights.length - 1; i++)
    { if (mid < heights[i]) { break } }
  var top = i ? heights[i - 1] : 0, bot = heights[i]
  var result = {left: (collapse == "right" ? rect.right : rect.left) - prepared.rect.left,
                right: (collapse == "left" ? rect.left : rect.right) - prepared.rect.left,
                top: top, bottom: bot}
  if (!rect.left && !rect.right) { result.bogus = true }
  if (!cm.options.singleCursorHeightPerLine) { result.rtop = rtop; result.rbottom = rbot }

  return result
}

// Work around problem with bounding client rects on ranges being
// returned incorrectly when zoomed on IE10 and below.
function maybeUpdateRectForZooming(measure, rect) {
  if (!window.screen || screen.logicalXDPI == null ||
      screen.logicalXDPI == screen.deviceXDPI || !hasBadZoomedRects(measure))
    { return rect }
  var scaleX = screen.logicalXDPI / screen.deviceXDPI
  var scaleY = screen.logicalYDPI / screen.deviceYDPI
  return {left: rect.left * scaleX, right: rect.right * scaleX,
          top: rect.top * scaleY, bottom: rect.bottom * scaleY}
}

function clearLineMeasurementCacheFor(lineView) {
  if (lineView.measure) {
    lineView.measure.cache = {}
    lineView.measure.heights = null
    if (lineView.rest) { for (var i = 0; i < lineView.rest.length; i++)
      { lineView.measure.caches[i] = {} } }
  }
}

function clearLineMeasurementCache(cm) {
  cm.display.externalMeasure = null
  removeChildren(cm.display.lineMeasure)
  for (var i = 0; i < cm.display.view.length; i++)
    { clearLineMeasurementCacheFor(cm.display.view[i]) }
}

function clearCaches(cm) {
  clearLineMeasurementCache(cm)
  cm.display.cachedCharWidth = cm.display.cachedTextHeight = cm.display.cachedPaddingH = null
  if (!cm.options.lineWrapping) { cm.display.maxLineChanged = true }
  cm.display.lineNumChars = null
}

function pageScrollX() { return window.pageXOffset || (document.documentElement || document.body).scrollLeft }
function pageScrollY() { return window.pageYOffset || (document.documentElement || document.body).scrollTop }

// Converts a {top, bottom, left, right} box from line-local
// coordinates into another coordinate system. Context may be one of
// "line", "div" (display.lineDiv), "local"./null (editor), "window",
// or "page".
function intoCoordSystem(cm, lineObj, rect, context, includeWidgets) {
  if (!includeWidgets && lineObj.widgets) { for (var i = 0; i < lineObj.widgets.length; ++i) { if (lineObj.widgets[i].above) {
    var size = widgetHeight(lineObj.widgets[i])
    rect.top += size; rect.bottom += size
  } } }
  if (context == "line") { return rect }
  if (!context) { context = "local" }
  var yOff = heightAtLine(lineObj)
  if (context == "local") { yOff += paddingTop(cm.display) }
  else { yOff -= cm.display.viewOffset }
  if (context == "page" || context == "window") {
    var lOff = cm.display.lineSpace.getBoundingClientRect()
    yOff += lOff.top + (context == "window" ? 0 : pageScrollY())
    var xOff = lOff.left + (context == "window" ? 0 : pageScrollX())
    rect.left += xOff; rect.right += xOff
  }
  rect.top += yOff; rect.bottom += yOff
  return rect
}

// Coverts a box from "div" coords to another coordinate system.
// Context may be "window", "page", "div", or "local"./null.
function fromCoordSystem(cm, coords, context) {
  if (context == "div") { return coords }
  var left = coords.left, top = coords.top
  // First move into "page" coordinate system
  if (context == "page") {
    left -= pageScrollX()
    top -= pageScrollY()
  } else if (context == "local" || !context) {
    var localBox = cm.display.sizer.getBoundingClientRect()
    left += localBox.left
    top += localBox.top
  }

  var lineSpaceBox = cm.display.lineSpace.getBoundingClientRect()
  return {left: left - lineSpaceBox.left, top: top - lineSpaceBox.top}
}

function charCoords(cm, pos, context, lineObj, bias) {
  if (!lineObj) { lineObj = getLine(cm.doc, pos.line) }
  return intoCoordSystem(cm, lineObj, measureChar(cm, lineObj, pos.ch, bias), context)
}

// Returns a box for a given cursor position, which may have an
// 'other' property containing the position of the secondary cursor
// on a bidi boundary.
// A cursor Pos(line, char, "before") is on the same visual line as `char - 1`
// and after `char - 1` in writing order of `char - 1`
// A cursor Pos(line, char, "after") is on the same visual line as `char`
// and before `char` in writing order of `char`
// Examples (upper-case letters are RTL, lower-case are LTR):
//     Pos(0, 1, ...)
//     before   after
// ab     a|b     a|b
// aB     a|B     aB|
// Ab     |Ab     A|b
// AB     B|A     B|A
// Every position after the last character on a line is considered to stick
// to the last character on the line.
function cursorCoords(cm, pos, context, lineObj, preparedMeasure, varHeight) {
  lineObj = lineObj || getLine(cm.doc, pos.line)
  if (!preparedMeasure) { preparedMeasure = prepareMeasureForLine(cm, lineObj) }
  function get(ch, right) {
    var m = measureCharPrepared(cm, preparedMeasure, ch, right ? "right" : "left", varHeight)
    if (right) { m.left = m.right; } else { m.right = m.left }
    return intoCoordSystem(cm, lineObj, m, context)
  }
  var order = getOrder(lineObj), ch = pos.ch, sticky = pos.sticky
  if (ch >= lineObj.text.length) {
    ch = lineObj.text.length
    sticky = "before"
  } else if (ch <= 0) {
    ch = 0
    sticky = "after"
  }
  if (!order) { return get(sticky == "before" ? ch - 1 : ch, sticky == "before") }

  function getBidi(ch, partPos, invert) {
    var part = order[partPos], right = (part.level % 2) != 0
    return get(invert ? ch - 1 : ch, right != invert)
  }
  var partPos = getBidiPartAt(order, ch, sticky)
  var other = bidiOther
  var val = getBidi(ch, partPos, sticky == "before")
  if (other != null) { val.other = getBidi(ch, other, sticky != "before") }
  return val
}

// Used to cheaply estimate the coordinates for a position. Used for
// intermediate scroll updates.
function estimateCoords(cm, pos) {
  var left = 0
  pos = clipPos(cm.doc, pos)
  if (!cm.options.lineWrapping) { left = charWidth(cm.display) * pos.ch }
  var lineObj = getLine(cm.doc, pos.line)
  var top = heightAtLine(lineObj) + paddingTop(cm.display)
  return {left: left, right: left, top: top, bottom: top + lineObj.height}
}

// Positions returned by coordsChar contain some extra information.
// xRel is the relative x position of the input coordinates compared
// to the found position (so xRel > 0 means the coordinates are to
// the right of the character position, for example). When outside
// is true, that means the coordinates lie outside the line's
// vertical range.
function PosWithInfo(line, ch, sticky, outside, xRel) {
  var pos = Pos(line, ch, sticky)
  pos.xRel = xRel
  if (outside) { pos.outside = true }
  return pos
}

// Compute the character position closest to the given coordinates.
// Input must be lineSpace-local ("div" coordinate system).
function coordsChar(cm, x, y) {
  var doc = cm.doc
  y += cm.display.viewOffset
  if (y < 0) { return PosWithInfo(doc.first, 0, null, true, -1) }
  var lineN = lineAtHeight(doc, y), last = doc.first + doc.size - 1
  if (lineN > last)
    { return PosWithInfo(doc.first + doc.size - 1, getLine(doc, last).text.length, null, true, 1) }
  if (x < 0) { x = 0 }

  var lineObj = getLine(doc, lineN)
  for (;;) {
    var found = coordsCharInner(cm, lineObj, lineN, x, y)
    var merged = collapsedSpanAtEnd(lineObj)
    var mergedPos = merged && merged.find(0, true)
    if (merged && (found.ch > mergedPos.from.ch || found.ch == mergedPos.from.ch && found.xRel > 0))
      { lineN = lineNo(lineObj = mergedPos.to.line) }
    else
      { return found }
  }
}

function wrappedLineExtent(cm, lineObj, preparedMeasure, y) {
  var measure = function (ch) { return intoCoordSystem(cm, lineObj, measureCharPrepared(cm, preparedMeasure, ch), "line"); }
  var end = lineObj.text.length
  var begin = findFirst(function (ch) { return measure(ch - 1).bottom <= y; }, end, 0)
  end = findFirst(function (ch) { return measure(ch).top > y; }, begin, end)
  return {begin: begin, end: end}
}

function wrappedLineExtentChar(cm, lineObj, preparedMeasure, target) {
  var targetTop = intoCoordSystem(cm, lineObj, measureCharPrepared(cm, preparedMeasure, target), "line").top
  return wrappedLineExtent(cm, lineObj, preparedMeasure, targetTop)
}

function coordsCharInner(cm, lineObj, lineNo, x, y) {
  y -= heightAtLine(lineObj)
  var begin = 0, end = lineObj.text.length
  var preparedMeasure = prepareMeasureForLine(cm, lineObj)
  var pos
  var order = getOrder(lineObj)
  if (order) {
    if (cm.options.lineWrapping) {
      ;var assign;
      ((assign = wrappedLineExtent(cm, lineObj, preparedMeasure, y), begin = assign.begin, end = assign.end, assign))
    }
    pos = new Pos(lineNo, begin)
    var beginLeft = cursorCoords(cm, pos, "line", lineObj, preparedMeasure).left
    var dir = beginLeft < x ? 1 : -1
    var prevDiff, diff = beginLeft - x, prevPos
    do {
      prevDiff = diff
      prevPos = pos
      pos = moveVisually(cm, lineObj, pos, dir)
      if (pos == null || pos.ch < begin || end <= (pos.sticky == "before" ? pos.ch - 1 : pos.ch)) {
        pos = prevPos
        break
      }
      diff = cursorCoords(cm, pos, "line", lineObj, preparedMeasure).left - x
    } while ((dir < 0) != (diff < 0) && (Math.abs(diff) <= Math.abs(prevDiff)))
    if (Math.abs(diff) > Math.abs(prevDiff)) {
      if ((diff < 0) == (prevDiff < 0)) { throw new Error("Broke out of infinite loop in coordsCharInner") }
      pos = prevPos
    }
  } else {
    var ch = findFirst(function (ch) {
      var box = intoCoordSystem(cm, lineObj, measureCharPrepared(cm, preparedMeasure, ch), "line")
      if (box.top > y) {
        // For the cursor stickiness
        end = Math.min(ch, end)
        return true
      }
      else if (box.bottom <= y) { return false }
      else if (box.left > x) { return true }
      else if (box.right < x) { return false }
      else { return (x - box.left < box.right - x) }
    }, begin, end)
    ch = skipExtendingChars(lineObj.text, ch, 1)
    pos = new Pos(lineNo, ch, ch == end ? "before" : "after")
  }
  var coords = cursorCoords(cm, pos, "line", lineObj, preparedMeasure)
  if (y < coords.top || coords.bottom < y) { pos.outside = true }
  pos.xRel = x < coords.left ? -1 : (x > coords.right ? 1 : 0)
  return pos
}

var measureText
// Compute the default text height.
function textHeight(display) {
  if (display.cachedTextHeight != null) { return display.cachedTextHeight }
  if (measureText == null) {
    measureText = elt("pre")
    // Measure a bunch of lines, for browsers that compute
    // fractional heights.
    for (var i = 0; i < 49; ++i) {
      measureText.appendChild(document.createTextNode("x"))
      measureText.appendChild(elt("br"))
    }
    measureText.appendChild(document.createTextNode("x"))
  }
  removeChildrenAndAdd(display.measure, measureText)
  var height = measureText.offsetHeight / 50
  if (height > 3) { display.cachedTextHeight = height }
  removeChildren(display.measure)
  return height || 1
}

// Compute the default character width.
function charWidth(display) {
  if (display.cachedCharWidth != null) { return display.cachedCharWidth }
  var anchor = elt("span", "xxxxxxxxxx")
  var pre = elt("pre", [anchor])
  removeChildrenAndAdd(display.measure, pre)
  var rect = anchor.getBoundingClientRect(), width = (rect.right - rect.left) / 10
  if (width > 2) { display.cachedCharWidth = width }
  return width || 10
}

// Do a bulk-read of the DOM positions and sizes needed to draw the
// view, so that we don't interleave reading and writing to the DOM.
function getDimensions(cm) {
  var d = cm.display, left = {}, width = {}
  var gutterLeft = d.gutters.clientLeft
  for (var n = d.gutters.firstChild, i = 0; n; n = n.nextSibling, ++i) {
    left[cm.options.gutters[i]] = n.offsetLeft + n.clientLeft + gutterLeft
    width[cm.options.gutters[i]] = n.clientWidth
  }
  return {fixedPos: compensateForHScroll(d),
          gutterTotalWidth: d.gutters.offsetWidth,
          gutterLeft: left,
          gutterWidth: width,
          wrapperWidth: d.wrapper.clientWidth}
}

// Computes display.scroller.scrollLeft + display.gutters.offsetWidth,
// but using getBoundingClientRect to get a sub-pixel-accurate
// result.
function compensateForHScroll(display) {
  return display.scroller.getBoundingClientRect().left - display.sizer.getBoundingClientRect().left
}

// Returns a function that estimates the height of a line, to use as
// first approximation until the line becomes visible (and is thus
// properly measurable).
function estimateHeight(cm) {
  var th = textHeight(cm.display), wrapping = cm.options.lineWrapping
  var perLine = wrapping && Math.max(5, cm.display.scroller.clientWidth / charWidth(cm.display) - 3)
  return function (line) {
    if (lineIsHidden(cm.doc, line)) { return 0 }

    var widgetsHeight = 0
    if (line.widgets) { for (var i = 0; i < line.widgets.length; i++) {
      if (line.widgets[i].height) { widgetsHeight += line.widgets[i].height }
    } }

    if (wrapping)
      { return widgetsHeight + (Math.ceil(line.text.length / perLine) || 1) * th }
    else
      { return widgetsHeight + th }
  }
}

function estimateLineHeights(cm) {
  var doc = cm.doc, est = estimateHeight(cm)
  doc.iter(function (line) {
    var estHeight = est(line)
    if (estHeight != line.height) { updateLineHeight(line, estHeight) }
  })
}

// Given a mouse event, find the corresponding position. If liberal
// is false, it checks whether a gutter or scrollbar was clicked,
// and returns null if it was. forRect is used by rectangular
// selections, and tries to estimate a character position even for
// coordinates beyond the right of the text.
function posFromMouse(cm, e, liberal, forRect) {
  var display = cm.display
  if (!liberal && e_target(e).getAttribute("cm-not-content") == "true") { return null }

  var x, y, space = display.lineSpace.getBoundingClientRect()
  // Fails unpredictably on IE[67] when mouse is dragged around quickly.
  try { x = e.clientX - space.left; y = e.clientY - space.top }
  catch (e) { return null }
  var coords = coordsChar(cm, x, y), line
  if (forRect && coords.xRel == 1 && (line = getLine(cm.doc, coords.line).text).length == coords.ch) {
    var colDiff = countColumn(line, line.length, cm.options.tabSize) - line.length
    coords = Pos(coords.line, Math.max(0, Math.round((x - paddingH(cm.display).left) / charWidth(cm.display)) - colDiff))
  }
  return coords
}

// Find the view element corresponding to a given line. Return null
// when the line isn't visible.
function findViewIndex(cm, n) {
  if (n >= cm.display.viewTo) { return null }
  n -= cm.display.viewFrom
  if (n < 0) { return null }
  var view = cm.display.view
  for (var i = 0; i < view.length; i++) {
    n -= view[i].size
    if (n < 0) { return i }
  }
}

function updateSelection(cm) {
  cm.display.input.showSelection(cm.display.input.prepareSelection())
}

function prepareSelection(cm, primary) {
  var doc = cm.doc, result = {}
  var curFragment = result.cursors = document.createDocumentFragment()
  var selFragment = result.selection = document.createDocumentFragment()

  for (var i = 0; i < doc.sel.ranges.length; i++) {
    if (primary === false && i == doc.sel.primIndex) { continue }
    var range = doc.sel.ranges[i]
    if (range.from().line >= cm.display.viewTo || range.to().line < cm.display.viewFrom) { continue }
    var collapsed = range.empty()
    if (collapsed || cm.options.showCursorWhenSelecting)
      { drawSelectionCursor(cm, range.head, curFragment) }
    if (!collapsed)
      { drawSelectionRange(cm, range, selFragment) }
  }
  return result
}

// Draws a cursor for the given range
function drawSelectionCursor(cm, head, output) {
  var pos = cursorCoords(cm, head, "div", null, null, !cm.options.singleCursorHeightPerLine)

  var cursor = output.appendChild(elt("div", "\u00a0", "CodeMirror-cursor"))
  cursor.style.left = pos.left + "px"
  cursor.style.top = pos.top + "px"
  cursor.style.height = Math.max(0, pos.bottom - pos.top) * cm.options.cursorHeight + "px"

  if (pos.other) {
    // Secondary cursor, shown when on a 'jump' in bi-directional text
    var otherCursor = output.appendChild(elt("div", "\u00a0", "CodeMirror-cursor CodeMirror-secondarycursor"))
    otherCursor.style.display = ""
    otherCursor.style.left = pos.other.left + "px"
    otherCursor.style.top = pos.other.top + "px"
    otherCursor.style.height = (pos.other.bottom - pos.other.top) * .85 + "px"
  }
}

// Draws the given range as a highlighted selection
function drawSelectionRange(cm, range, output) {
  var display = cm.display, doc = cm.doc
  var fragment = document.createDocumentFragment()
  var padding = paddingH(cm.display), leftSide = padding.left
  var rightSide = Math.max(display.sizerWidth, displayWidth(cm) - display.sizer.offsetLeft) - padding.right

  function add(left, top, width, bottom) {
    if (top < 0) { top = 0 }
    top = Math.round(top)
    bottom = Math.round(bottom)
    fragment.appendChild(elt("div", null, "CodeMirror-selected", ("position: absolute; left: " + left + "px;\n                             top: " + top + "px; width: " + (width == null ? rightSide - left : width) + "px;\n                             height: " + (bottom - top) + "px")))
  }

  function drawForLine(line, fromArg, toArg) {
    var lineObj = getLine(doc, line)
    var lineLen = lineObj.text.length
    var start, end
    function coords(ch, bias) {
      return charCoords(cm, Pos(line, ch), "div", lineObj, bias)
    }

    iterateBidiSections(getOrder(lineObj), fromArg || 0, toArg == null ? lineLen : toArg, function (from, to, dir) {
      var leftPos = coords(from, "left"), rightPos, left, right
      if (from == to) {
        rightPos = leftPos
        left = right = leftPos.left
      } else {
        rightPos = coords(to - 1, "right")
        if (dir == "rtl") { var tmp = leftPos; leftPos = rightPos; rightPos = tmp }
        left = leftPos.left
        right = rightPos.right
      }
      if (fromArg == null && from == 0) { left = leftSide }
      if (rightPos.top - leftPos.top > 3) { // Different lines, draw top part
        add(left, leftPos.top, null, leftPos.bottom)
        left = leftSide
        if (leftPos.bottom < rightPos.top) { add(left, leftPos.bottom, null, rightPos.top) }
      }
      if (toArg == null && to == lineLen) { right = rightSide }
      if (!start || leftPos.top < start.top || leftPos.top == start.top && leftPos.left < start.left)
        { start = leftPos }
      if (!end || rightPos.bottom > end.bottom || rightPos.bottom == end.bottom && rightPos.right > end.right)
        { end = rightPos }
      if (left < leftSide + 1) { left = leftSide }
      add(left, rightPos.top, right - left, rightPos.bottom)
    })
    return {start: start, end: end}
  }

  var sFrom = range.from(), sTo = range.to()
  if (sFrom.line == sTo.line) {
    drawForLine(sFrom.line, sFrom.ch, sTo.ch)
  } else {
    var fromLine = getLine(doc, sFrom.line), toLine = getLine(doc, sTo.line)
    var singleVLine = visualLine(fromLine) == visualLine(toLine)
    var leftEnd = drawForLine(sFrom.line, sFrom.ch, singleVLine ? fromLine.text.length + 1 : null).end
    var rightStart = drawForLine(sTo.line, singleVLine ? 0 : null, sTo.ch).start
    if (singleVLine) {
      if (leftEnd.top < rightStart.top - 2) {
        add(leftEnd.right, leftEnd.top, null, leftEnd.bottom)
        add(leftSide, rightStart.top, rightStart.left, rightStart.bottom)
      } else {
        add(leftEnd.right, leftEnd.top, rightStart.left - leftEnd.right, leftEnd.bottom)
      }
    }
    if (leftEnd.bottom < rightStart.top)
      { add(leftSide, leftEnd.bottom, null, rightStart.top) }
  }

  output.appendChild(fragment)
}

// Cursor-blinking
function restartBlink(cm) {
  if (!cm.state.focused) { return }
  var display = cm.display
  clearInterval(display.blinker)
  var on = true
  display.cursorDiv.style.visibility = ""
  if (cm.options.cursorBlinkRate > 0)
    { display.blinker = setInterval(function () { return display.cursorDiv.style.visibility = (on = !on) ? "" : "hidden"; },
      cm.options.cursorBlinkRate) }
  else if (cm.options.cursorBlinkRate < 0)
    { display.cursorDiv.style.visibility = "hidden" }
}

function ensureFocus(cm) {
  if (!cm.state.focused) { cm.display.input.focus(); onFocus(cm) }
}

function delayBlurEvent(cm) {
  cm.state.delayingBlurEvent = true
  setTimeout(function () { if (cm.state.delayingBlurEvent) {
    cm.state.delayingBlurEvent = false
    onBlur(cm)
  } }, 100)
}

function onFocus(cm, e) {
  if (cm.state.delayingBlurEvent) { cm.state.delayingBlurEvent = false }

  if (cm.options.readOnly == "nocursor") { return }
  if (!cm.state.focused) {
    signal(cm, "focus", cm, e)
    cm.state.focused = true
    addClass(cm.display.wrapper, "CodeMirror-focused")
    // This test prevents this from firing when a context
    // menu is closed (since the input reset would kill the
    // select-all detection hack)
    if (!cm.curOp && cm.display.selForContextMenu != cm.doc.sel) {
      cm.display.input.reset()
      if (webkit) { setTimeout(function () { return cm.display.input.reset(true); }, 20) } // Issue #1730
    }
    cm.display.input.receivedFocus()
  }
  restartBlink(cm)
}
function onBlur(cm, e) {
  if (cm.state.delayingBlurEvent) { return }

  if (cm.state.focused) {
    signal(cm, "blur", cm, e)
    cm.state.focused = false
    rmClass(cm.display.wrapper, "CodeMirror-focused")
  }
  clearInterval(cm.display.blinker)
  setTimeout(function () { if (!cm.state.focused) { cm.display.shift = false } }, 150)
}

// Re-align line numbers and gutter marks to compensate for
// horizontal scrolling.
function alignHorizontally(cm) {
  var display = cm.display, view = display.view
  if (!display.alignWidgets && (!display.gutters.firstChild || !cm.options.fixedGutter)) { return }
  var comp = compensateForHScroll(display) - display.scroller.scrollLeft + cm.doc.scrollLeft
  var gutterW = display.gutters.offsetWidth, left = comp + "px"
  for (var i = 0; i < view.length; i++) { if (!view[i].hidden) {
    if (cm.options.fixedGutter) {
      if (view[i].gutter)
        { view[i].gutter.style.left = left }
      if (view[i].gutterBackground)
        { view[i].gutterBackground.style.left = left }
    }
    var align = view[i].alignable
    if (align) { for (var j = 0; j < align.length; j++)
      { align[j].style.left = left } }
  } }
  if (cm.options.fixedGutter)
    { display.gutters.style.left = (comp + gutterW) + "px" }
}

// Used to ensure that the line number gutter is still the right
// size for the current document size. Returns true when an update
// is needed.
function maybeUpdateLineNumberWidth(cm) {
  if (!cm.options.lineNumbers) { return false }
  var doc = cm.doc, last = lineNumberFor(cm.options, doc.first + doc.size - 1), display = cm.display
  if (last.length != display.lineNumChars) {
    var test = display.measure.appendChild(elt("div", [elt("div", last)],
                                               "CodeMirror-linenumber CodeMirror-gutter-elt"))
    var innerW = test.firstChild.offsetWidth, padding = test.offsetWidth - innerW
    display.lineGutter.style.width = ""
    display.lineNumInnerWidth = Math.max(innerW, display.lineGutter.offsetWidth - padding) + 1
    display.lineNumWidth = display.lineNumInnerWidth + padding
    display.lineNumChars = display.lineNumInnerWidth ? last.length : -1
    display.lineGutter.style.width = display.lineNumWidth + "px"
    updateGutterSpace(cm)
    return true
  }
  return false
}

// Read the actual heights of the rendered lines, and update their
// stored heights to match.
function updateHeightsInViewport(cm) {
  var display = cm.display
  var prevBottom = display.lineDiv.offsetTop
  for (var i = 0; i < display.view.length; i++) {
    var cur = display.view[i], height = (void 0)
    if (cur.hidden) { continue }
    if (ie && ie_version < 8) {
      var bot = cur.node.offsetTop + cur.node.offsetHeight
      height = bot - prevBottom
      prevBottom = bot
    } else {
      var box = cur.node.getBoundingClientRect()
      height = box.bottom - box.top
    }
    var diff = cur.line.height - height
    if (height < 2) { height = textHeight(display) }
    if (diff > .001 || diff < -.001) {
      updateLineHeight(cur.line, height)
      updateWidgetHeight(cur.line)
      if (cur.rest) { for (var j = 0; j < cur.rest.length; j++)
        { updateWidgetHeight(cur.rest[j]) } }
    }
  }
}

// Read and store the height of line widgets associated with the
// given line.
function updateWidgetHeight(line) {
  if (line.widgets) { for (var i = 0; i < line.widgets.length; ++i)
    { line.widgets[i].height = line.widgets[i].node.parentNode.offsetHeight } }
}

// Compute the lines that are visible in a given viewport (defaults
// the the current scroll position). viewport may contain top,
// height, and ensure (see op.scrollToPos) properties.
function visibleLines(display, doc, viewport) {
  var top = viewport && viewport.top != null ? Math.max(0, viewport.top) : display.scroller.scrollTop
  top = Math.floor(top - paddingTop(display))
  var bottom = viewport && viewport.bottom != null ? viewport.bottom : top + display.wrapper.clientHeight

  var from = lineAtHeight(doc, top), to = lineAtHeight(doc, bottom)
  // Ensure is a {from: {line, ch}, to: {line, ch}} object, and
  // forces those lines into the viewport (if possible).
  if (viewport && viewport.ensure) {
    var ensureFrom = viewport.ensure.from.line, ensureTo = viewport.ensure.to.line
    if (ensureFrom < from) {
      from = ensureFrom
      to = lineAtHeight(doc, heightAtLine(getLine(doc, ensureFrom)) + display.wrapper.clientHeight)
    } else if (Math.min(ensureTo, doc.lastLine()) >= to) {
      from = lineAtHeight(doc, heightAtLine(getLine(doc, ensureTo)) - display.wrapper.clientHeight)
      to = ensureTo
    }
  }
  return {from: from, to: Math.max(to, from + 1)}
}

// Sync the scrollable area and scrollbars, ensure the viewport
// covers the visible area.
function setScrollTop(cm, val) {
  if (Math.abs(cm.doc.scrollTop - val) < 2) { return }
  cm.doc.scrollTop = val
  if (!gecko) { updateDisplaySimple(cm, {top: val}) }
  if (cm.display.scroller.scrollTop != val) { cm.display.scroller.scrollTop = val }
  cm.display.scrollbars.setScrollTop(val)
  if (gecko) { updateDisplaySimple(cm) }
  startWorker(cm, 100)
}
// Sync scroller and scrollbar, ensure the gutter elements are
// aligned.
function setScrollLeft(cm, val, isScroller) {
  if (isScroller ? val == cm.doc.scrollLeft : Math.abs(cm.doc.scrollLeft - val) < 2) { return }
  val = Math.min(val, cm.display.scroller.scrollWidth - cm.display.scroller.clientWidth)
  cm.doc.scrollLeft = val
  alignHorizontally(cm)
  if (cm.display.scroller.scrollLeft != val) { cm.display.scroller.scrollLeft = val }
  cm.display.scrollbars.setScrollLeft(val)
}

// Since the delta values reported on mouse wheel events are
// unstandardized between browsers and even browser versions, and
// generally horribly unpredictable, this code starts by measuring
// the scroll effect that the first few mouse wheel events have,
// and, from that, detects the way it can convert deltas to pixel
// offsets afterwards.
//
// The reason we want to know the amount a wheel event will scroll
// is that it gives us a chance to update the display before the
// actual scrolling happens, reducing flickering.

var wheelSamples = 0;
var wheelPixelsPerUnit = null;
// Fill in a browser-detected starting value on browsers where we
// know one. These don't have to be accurate -- the result of them
// being wrong would just be a slight flicker on the first wheel
// scroll (if it is large enough).
if (ie) { wheelPixelsPerUnit = -.53 }
else if (gecko) { wheelPixelsPerUnit = 15 }
else if (chrome) { wheelPixelsPerUnit = -.7 }
else if (safari) { wheelPixelsPerUnit = -1/3 }

function wheelEventDelta(e) {
  var dx = e.wheelDeltaX, dy = e.wheelDeltaY
  if (dx == null && e.detail && e.axis == e.HORIZONTAL_AXIS) { dx = e.detail }
  if (dy == null && e.detail && e.axis == e.VERTICAL_AXIS) { dy = e.detail }
  else if (dy == null) { dy = e.wheelDelta }
  return {x: dx, y: dy}
}
function wheelEventPixels(e) {
  var delta = wheelEventDelta(e)
  delta.x *= wheelPixelsPerUnit
  delta.y *= wheelPixelsPerUnit
  return delta
}

function onScrollWheel(cm, e) {
  var delta = wheelEventDelta(e), dx = delta.x, dy = delta.y

  var display = cm.display, scroll = display.scroller
  // Quit if there's nothing to scroll here
  var canScrollX = scroll.scrollWidth > scroll.clientWidth
  var canScrollY = scroll.scrollHeight > scroll.clientHeight
  if (!(dx && canScrollX || dy && canScrollY)) { return }

  // Webkit browsers on OS X abort momentum scrolls when the target
  // of the scroll event is removed from the scrollable element.
  // This hack (see related code in patchDisplay) makes sure the
  // element is kept around.
  if (dy && mac && webkit) {
    outer: for (var cur = e.target, view = display.view; cur != scroll; cur = cur.parentNode) {
      for (var i = 0; i < view.length; i++) {
        if (view[i].node == cur) {
          cm.display.currentWheelTarget = cur
          break outer
        }
      }
    }
  }

  // On some browsers, horizontal scrolling will cause redraws to
  // happen before the gutter has been realigned, causing it to
  // wriggle around in a most unseemly way. When we have an
  // estimated pixels/delta value, we just handle horizontal
  // scrolling entirely here. It'll be slightly off from native, but
  // better than glitching out.
  if (dx && !gecko && !presto && wheelPixelsPerUnit != null) {
    if (dy && canScrollY)
      { setScrollTop(cm, Math.max(0, Math.min(scroll.scrollTop + dy * wheelPixelsPerUnit, scroll.scrollHeight - scroll.clientHeight))) }
    setScrollLeft(cm, Math.max(0, Math.min(scroll.scrollLeft + dx * wheelPixelsPerUnit, scroll.scrollWidth - scroll.clientWidth)))
    // Only prevent default scrolling if vertical scrolling is
    // actually possible. Otherwise, it causes vertical scroll
    // jitter on OSX trackpads when deltaX is small and deltaY
    // is large (issue #3579)
    if (!dy || (dy && canScrollY))
      { e_preventDefault(e) }
    display.wheelStartX = null // Abort measurement, if in progress
    return
  }

  // 'Project' the visible viewport to cover the area that is being
  // scrolled into view (if we know enough to estimate it).
  if (dy && wheelPixelsPerUnit != null) {
    var pixels = dy * wheelPixelsPerUnit
    var top = cm.doc.scrollTop, bot = top + display.wrapper.clientHeight
    if (pixels < 0) { top = Math.max(0, top + pixels - 50) }
    else { bot = Math.min(cm.doc.height, bot + pixels + 50) }
    updateDisplaySimple(cm, {top: top, bottom: bot})
  }

  if (wheelSamples < 20) {
    if (display.wheelStartX == null) {
      display.wheelStartX = scroll.scrollLeft; display.wheelStartY = scroll.scrollTop
      display.wheelDX = dx; display.wheelDY = dy
      setTimeout(function () {
        if (display.wheelStartX == null) { return }
        var movedX = scroll.scrollLeft - display.wheelStartX
        var movedY = scroll.scrollTop - display.wheelStartY
        var sample = (movedY && display.wheelDY && movedY / display.wheelDY) ||
          (movedX && display.wheelDX && movedX / display.wheelDX)
        display.wheelStartX = display.wheelStartY = null
        if (!sample) { return }
        wheelPixelsPerUnit = (wheelPixelsPerUnit * wheelSamples + sample) / (wheelSamples + 1)
        ++wheelSamples
      }, 200)
    } else {
      display.wheelDX += dx; display.wheelDY += dy
    }
  }
}

// SCROLLBARS

// Prepare DOM reads needed to update the scrollbars. Done in one
// shot to minimize update/measure roundtrips.
function measureForScrollbars(cm) {
  var d = cm.display, gutterW = d.gutters.offsetWidth
  var docH = Math.round(cm.doc.height + paddingVert(cm.display))
  return {
    clientHeight: d.scroller.clientHeight,
    viewHeight: d.wrapper.clientHeight,
    scrollWidth: d.scroller.scrollWidth, clientWidth: d.scroller.clientWidth,
    viewWidth: d.wrapper.clientWidth,
    barLeft: cm.options.fixedGutter ? gutterW : 0,
    docHeight: docH,
    scrollHeight: docH + scrollGap(cm) + d.barHeight,
    nativeBarWidth: d.nativeBarWidth,
    gutterWidth: gutterW
  }
}

var NativeScrollbars = function(place, scroll, cm) {
  this.cm = cm
  var vert = this.vert = elt("div", [elt("div", null, null, "min-width: 1px")], "CodeMirror-vscrollbar")
  var horiz = this.horiz = elt("div", [elt("div", null, null, "height: 100%; min-height: 1px")], "CodeMirror-hscrollbar")
  place(vert); place(horiz)

  on(vert, "scroll", function () {
    if (vert.clientHeight) { scroll(vert.scrollTop, "vertical") }
  })
  on(horiz, "scroll", function () {
    if (horiz.clientWidth) { scroll(horiz.scrollLeft, "horizontal") }
  })

  this.checkedZeroWidth = false
  // Need to set a minimum width to see the scrollbar on IE7 (but must not set it on IE8).
  if (ie && ie_version < 8) { this.horiz.style.minHeight = this.vert.style.minWidth = "18px" }
};

NativeScrollbars.prototype.update = function (measure) {
  var needsH = measure.scrollWidth > measure.clientWidth + 1
  var needsV = measure.scrollHeight > measure.clientHeight + 1
  var sWidth = measure.nativeBarWidth

  if (needsV) {
    this.vert.style.display = "block"
    this.vert.style.bottom = needsH ? sWidth + "px" : "0"
    var totalHeight = measure.viewHeight - (needsH ? sWidth : 0)
    // A bug in IE8 can cause this value to be negative, so guard it.
    this.vert.firstChild.style.height =
      Math.max(0, measure.scrollHeight - measure.clientHeight + totalHeight) + "px"
  } else {
    this.vert.style.display = ""
    this.vert.firstChild.style.height = "0"
  }

  if (needsH) {
    this.horiz.style.display = "block"
    this.horiz.style.right = needsV ? sWidth + "px" : "0"
    this.horiz.style.left = measure.barLeft + "px"
    var totalWidth = measure.viewWidth - measure.barLeft - (needsV ? sWidth : 0)
    this.horiz.firstChild.style.width =
      Math.max(0, measure.scrollWidth - measure.clientWidth + totalWidth) + "px"
  } else {
    this.horiz.style.display = ""
    this.horiz.firstChild.style.width = "0"
  }

  if (!this.checkedZeroWidth && measure.clientHeight > 0) {
    if (sWidth == 0) { this.zeroWidthHack() }
    this.checkedZeroWidth = true
  }

  return {right: needsV ? sWidth : 0, bottom: needsH ? sWidth : 0}
};

NativeScrollbars.prototype.setScrollLeft = function (pos) {
  if (this.horiz.scrollLeft != pos) { this.horiz.scrollLeft = pos }
  if (this.disableHoriz) { this.enableZeroWidthBar(this.horiz, this.disableHoriz) }
};

NativeScrollbars.prototype.setScrollTop = function (pos) {
  if (this.vert.scrollTop != pos) { this.vert.scrollTop = pos }
  if (this.disableVert) { this.enableZeroWidthBar(this.vert, this.disableVert) }
};

NativeScrollbars.prototype.zeroWidthHack = function () {
  var w = mac && !mac_geMountainLion ? "12px" : "18px"
  this.horiz.style.height = this.vert.style.width = w
  this.horiz.style.pointerEvents = this.vert.style.pointerEvents = "none"
  this.disableHoriz = new Delayed
  this.disableVert = new Delayed
};

NativeScrollbars.prototype.enableZeroWidthBar = function (bar, delay) {
  bar.style.pointerEvents = "auto"
  function maybeDisable() {
    // To find out whether the scrollbar is still visible, we
    // check whether the element under the pixel in the bottom
    // left corner of the scrollbar box is the scrollbar box
    // itself (when the bar is still visible) or its filler child
    // (when the bar is hidden). If it is still visible, we keep
    // it enabled, if it's hidden, we disable pointer events.
    var box = bar.getBoundingClientRect()
    var elt = document.elementFromPoint(box.left + 1, box.bottom - 1)
    if (elt != bar) { bar.style.pointerEvents = "none" }
    else { delay.set(1000, maybeDisable) }
  }
  delay.set(1000, maybeDisable)
};

NativeScrollbars.prototype.clear = function () {
  var parent = this.horiz.parentNode
  parent.removeChild(this.horiz)
  parent.removeChild(this.vert)
};

var NullScrollbars = function () {};

NullScrollbars.prototype.update = function () { return {bottom: 0, right: 0} };
NullScrollbars.prototype.setScrollLeft = function () {};
NullScrollbars.prototype.setScrollTop = function () {};
NullScrollbars.prototype.clear = function () {};

function updateScrollbars(cm, measure) {
  if (!measure) { measure = measureForScrollbars(cm) }
  var startWidth = cm.display.barWidth, startHeight = cm.display.barHeight
  updateScrollbarsInner(cm, measure)
  for (var i = 0; i < 4 && startWidth != cm.display.barWidth || startHeight != cm.display.barHeight; i++) {
    if (startWidth != cm.display.barWidth && cm.options.lineWrapping)
      { updateHeightsInViewport(cm) }
    updateScrollbarsInner(cm, measureForScrollbars(cm))
    startWidth = cm.display.barWidth; startHeight = cm.display.barHeight
  }
}

// Re-synchronize the fake scrollbars with the actual size of the
// content.
function updateScrollbarsInner(cm, measure) {
  var d = cm.display
  var sizes = d.scrollbars.update(measure)

  d.sizer.style.paddingRight = (d.barWidth = sizes.right) + "px"
  d.sizer.style.paddingBottom = (d.barHeight = sizes.bottom) + "px"
  d.heightForcer.style.borderBottom = sizes.bottom + "px solid transparent"

  if (sizes.right && sizes.bottom) {
    d.scrollbarFiller.style.display = "block"
    d.scrollbarFiller.style.height = sizes.bottom + "px"
    d.scrollbarFiller.style.width = sizes.right + "px"
  } else { d.scrollbarFiller.style.display = "" }
  if (sizes.bottom && cm.options.coverGutterNextToScrollbar && cm.options.fixedGutter) {
    d.gutterFiller.style.display = "block"
    d.gutterFiller.style.height = sizes.bottom + "px"
    d.gutterFiller.style.width = measure.gutterWidth + "px"
  } else { d.gutterFiller.style.display = "" }
}

var scrollbarModel = {"native": NativeScrollbars, "null": NullScrollbars}

function initScrollbars(cm) {
  if (cm.display.scrollbars) {
    cm.display.scrollbars.clear()
    if (cm.display.scrollbars.addClass)
      { rmClass(cm.display.wrapper, cm.display.scrollbars.addClass) }
  }

  cm.display.scrollbars = new scrollbarModel[cm.options.scrollbarStyle](function (node) {
    cm.display.wrapper.insertBefore(node, cm.display.scrollbarFiller)
    // Prevent clicks in the scrollbars from killing focus
    on(node, "mousedown", function () {
      if (cm.state.focused) { setTimeout(function () { return cm.display.input.focus(); }, 0) }
    })
    node.setAttribute("cm-not-content", "true")
  }, function (pos, axis) {
    if (axis == "horizontal") { setScrollLeft(cm, pos) }
    else { setScrollTop(cm, pos) }
  }, cm)
  if (cm.display.scrollbars.addClass)
    { addClass(cm.display.wrapper, cm.display.scrollbars.addClass) }
}

// SCROLLING THINGS INTO VIEW

// If an editor sits on the top or bottom of the window, partially
// scrolled out of view, this ensures that the cursor is visible.
function maybeScrollWindow(cm, coords) {
  if (signalDOMEvent(cm, "scrollCursorIntoView")) { return }

  var display = cm.display, box = display.sizer.getBoundingClientRect(), doScroll = null
  if (coords.top + box.top < 0) { doScroll = true }
  else if (coords.bottom + box.top > (window.innerHeight || document.documentElement.clientHeight)) { doScroll = false }
  if (doScroll != null && !phantom) {
    var scrollNode = elt("div", "\u200b", null, ("position: absolute;\n                         top: " + (coords.top - display.viewOffset - paddingTop(cm.display)) + "px;\n                         height: " + (coords.bottom - coords.top + scrollGap(cm) + display.barHeight) + "px;\n                         left: " + (coords.left) + "px; width: 2px;"))
    cm.display.lineSpace.appendChild(scrollNode)
    scrollNode.scrollIntoView(doScroll)
    cm.display.lineSpace.removeChild(scrollNode)
  }
}

// Scroll a given position into view (immediately), verifying that
// it actually became visible (as line heights are accurately
// measured, the position of something may 'drift' during drawing).
function scrollPosIntoView(cm, pos, end, margin) {
  if (margin == null) { margin = 0 }
  var coords
  for (var limit = 0; limit < 5; limit++) {
    var changed = false
    coords = cursorCoords(cm, pos)
    var endCoords = !end || end == pos ? coords : cursorCoords(cm, end)
    var scrollPos = calculateScrollPos(cm, Math.min(coords.left, endCoords.left),
                                       Math.min(coords.top, endCoords.top) - margin,
                                       Math.max(coords.left, endCoords.left),
                                       Math.max(coords.bottom, endCoords.bottom) + margin)
    var startTop = cm.doc.scrollTop, startLeft = cm.doc.scrollLeft
    if (scrollPos.scrollTop != null) {
      setScrollTop(cm, scrollPos.scrollTop)
      if (Math.abs(cm.doc.scrollTop - startTop) > 1) { changed = true }
    }
    if (scrollPos.scrollLeft != null) {
      setScrollLeft(cm, scrollPos.scrollLeft)
      if (Math.abs(cm.doc.scrollLeft - startLeft) > 1) { changed = true }
    }
    if (!changed) { break }
  }
  return coords
}

// Scroll a given set of coordinates into view (immediately).
function scrollIntoView(cm, x1, y1, x2, y2) {
  var scrollPos = calculateScrollPos(cm, x1, y1, x2, y2)
  if (scrollPos.scrollTop != null) { setScrollTop(cm, scrollPos.scrollTop) }
  if (scrollPos.scrollLeft != null) { setScrollLeft(cm, scrollPos.scrollLeft) }
}

// Calculate a new scroll position needed to scroll the given
// rectangle into view. Returns an object with scrollTop and
// scrollLeft properties. When these are undefined, the
// vertical/horizontal position does not need to be adjusted.
function calculateScrollPos(cm, x1, y1, x2, y2) {
  var display = cm.display, snapMargin = textHeight(cm.display)
  if (y1 < 0) { y1 = 0 }
  var screentop = cm.curOp && cm.curOp.scrollTop != null ? cm.curOp.scrollTop : display.scroller.scrollTop
  var screen = displayHeight(cm), result = {}
  if (y2 - y1 > screen) { y2 = y1 + screen }
  var docBottom = cm.doc.height + paddingVert(display)
  var atTop = y1 < snapMargin, atBottom = y2 > docBottom - snapMargin
  if (y1 < screentop) {
    result.scrollTop = atTop ? 0 : y1
  } else if (y2 > screentop + screen) {
    var newTop = Math.min(y1, (atBottom ? docBottom : y2) - screen)
    if (newTop != screentop) { result.scrollTop = newTop }
  }

  var screenleft = cm.curOp && cm.curOp.scrollLeft != null ? cm.curOp.scrollLeft : display.scroller.scrollLeft
  var screenw = displayWidth(cm) - (cm.options.fixedGutter ? display.gutters.offsetWidth : 0)
  var tooWide = x2 - x1 > screenw
  if (tooWide) { x2 = x1 + screenw }
  if (x1 < 10)
    { result.scrollLeft = 0 }
  else if (x1 < screenleft)
    { result.scrollLeft = Math.max(0, x1 - (tooWide ? 0 : 10)) }
  else if (x2 > screenw + screenleft - 3)
    { result.scrollLeft = x2 + (tooWide ? 0 : 10) - screenw }
  return result
}

// Store a relative adjustment to the scroll position in the current
// operation (to be applied when the operation finishes).
function addToScrollPos(cm, left, top) {
  if (left != null || top != null) { resolveScrollToPos(cm) }
  if (left != null)
    { cm.curOp.scrollLeft = (cm.curOp.scrollLeft == null ? cm.doc.scrollLeft : cm.curOp.scrollLeft) + left }
  if (top != null)
    { cm.curOp.scrollTop = (cm.curOp.scrollTop == null ? cm.doc.scrollTop : cm.curOp.scrollTop) + top }
}

// Make sure that at the end of the operation the current cursor is
// shown.
function ensureCursorVisible(cm) {
  resolveScrollToPos(cm)
  var cur = cm.getCursor(), from = cur, to = cur
  if (!cm.options.lineWrapping) {
    from = cur.ch ? Pos(cur.line, cur.ch - 1) : cur
    to = Pos(cur.line, cur.ch + 1)
  }
  cm.curOp.scrollToPos = {from: from, to: to, margin: cm.options.cursorScrollMargin, isCursor: true}
}

// When an operation has its scrollToPos property set, and another
// scroll action is applied before the end of the operation, this
// 'simulates' scrolling that position into view in a cheap way, so
// that the effect of intermediate scroll commands is not ignored.
function resolveScrollToPos(cm) {
  var range = cm.curOp.scrollToPos
  if (range) {
    cm.curOp.scrollToPos = null
    var from = estimateCoords(cm, range.from), to = estimateCoords(cm, range.to)
    var sPos = calculateScrollPos(cm, Math.min(from.left, to.left),
                                  Math.min(from.top, to.top) - range.margin,
                                  Math.max(from.right, to.right),
                                  Math.max(from.bottom, to.bottom) + range.margin)
    cm.scrollTo(sPos.scrollLeft, sPos.scrollTop)
  }
}

// Operations are used to wrap a series of changes to the editor
// state in such a way that each change won't have to update the
// cursor and display (which would be awkward, slow, and
// error-prone). Instead, display updates are batched and then all
// combined and executed at once.

var nextOpId = 0
// Start a new operation.
function startOperation(cm) {
  cm.curOp = {
    cm: cm,
    viewChanged: false,      // Flag that indicates that lines might need to be redrawn
    startHeight: cm.doc.height, // Used to detect need to update scrollbar
    forceUpdate: false,      // Used to force a redraw
    updateInput: null,       // Whether to reset the input textarea
    typing: false,           // Whether this reset should be careful to leave existing text (for compositing)
    changeObjs: null,        // Accumulated changes, for firing change events
    cursorActivityHandlers: null, // Set of handlers to fire cursorActivity on
    cursorActivityCalled: 0, // Tracks which cursorActivity handlers have been called already
    selectionChanged: false, // Whether the selection needs to be redrawn
    updateMaxLine: false,    // Set when the widest line needs to be determined anew
    scrollLeft: null, scrollTop: null, // Intermediate scroll position, not pushed to DOM yet
    scrollToPos: null,       // Used to scroll to a specific position
    focus: false,
    id: ++nextOpId           // Unique ID
  }
  pushOperation(cm.curOp)
}

// Finish an operation, updating the display and signalling delayed events
function endOperation(cm) {
  var op = cm.curOp
  finishOperation(op, function (group) {
    for (var i = 0; i < group.ops.length; i++)
      { group.ops[i].cm.curOp = null }
    endOperations(group)
  })
}

// The DOM updates done when an operation finishes are batched so
// that the minimum number of relayouts are required.
function endOperations(group) {
  var ops = group.ops
  for (var i = 0; i < ops.length; i++) // Read DOM
    { endOperation_R1(ops[i]) }
  for (var i$1 = 0; i$1 < ops.length; i$1++) // Write DOM (maybe)
    { endOperation_W1(ops[i$1]) }
  for (var i$2 = 0; i$2 < ops.length; i$2++) // Read DOM
    { endOperation_R2(ops[i$2]) }
  for (var i$3 = 0; i$3 < ops.length; i$3++) // Write DOM (maybe)
    { endOperation_W2(ops[i$3]) }
  for (var i$4 = 0; i$4 < ops.length; i$4++) // Read DOM
    { endOperation_finish(ops[i$4]) }
}

function endOperation_R1(op) {
  var cm = op.cm, display = cm.display
  maybeClipScrollbars(cm)
  if (op.updateMaxLine) { findMaxLine(cm) }

  op.mustUpdate = op.viewChanged || op.forceUpdate || op.scrollTop != null ||
    op.scrollToPos && (op.scrollToPos.from.line < display.viewFrom ||
                       op.scrollToPos.to.line >= display.viewTo) ||
    display.maxLineChanged && cm.options.lineWrapping
  op.update = op.mustUpdate &&
    new DisplayUpdate(cm, op.mustUpdate && {top: op.scrollTop, ensure: op.scrollToPos}, op.forceUpdate)
}

function endOperation_W1(op) {
  op.updatedDisplay = op.mustUpdate && updateDisplayIfNeeded(op.cm, op.update)
}

function endOperation_R2(op) {
  var cm = op.cm, display = cm.display
  if (op.updatedDisplay) { updateHeightsInViewport(cm) }

  op.barMeasure = measureForScrollbars(cm)

  // If the max line changed since it was last measured, measure it,
  // and ensure the document's width matches it.
  // updateDisplay_W2 will use these properties to do the actual resizing
  if (display.maxLineChanged && !cm.options.lineWrapping) {
    op.adjustWidthTo = measureChar(cm, display.maxLine, display.maxLine.text.length).left + 3
    cm.display.sizerWidth = op.adjustWidthTo
    op.barMeasure.scrollWidth =
      Math.max(display.scroller.clientWidth, display.sizer.offsetLeft + op.adjustWidthTo + scrollGap(cm) + cm.display.barWidth)
    op.maxScrollLeft = Math.max(0, display.sizer.offsetLeft + op.adjustWidthTo - displayWidth(cm))
  }

  if (op.updatedDisplay || op.selectionChanged)
    { op.preparedSelection = display.input.prepareSelection(op.focus) }
}

function endOperation_W2(op) {
  var cm = op.cm

  if (op.adjustWidthTo != null) {
    cm.display.sizer.style.minWidth = op.adjustWidthTo + "px"
    if (op.maxScrollLeft < cm.doc.scrollLeft)
      { setScrollLeft(cm, Math.min(cm.display.scroller.scrollLeft, op.maxScrollLeft), true) }
    cm.display.maxLineChanged = false
  }

  var takeFocus = op.focus && op.focus == activeElt() && (!document.hasFocus || document.hasFocus())
  if (op.preparedSelection)
    { cm.display.input.showSelection(op.preparedSelection, takeFocus) }
  if (op.updatedDisplay || op.startHeight != cm.doc.height)
    { updateScrollbars(cm, op.barMeasure) }
  if (op.updatedDisplay)
    { setDocumentHeight(cm, op.barMeasure) }

  if (op.selectionChanged) { restartBlink(cm) }

  if (cm.state.focused && op.updateInput)
    { cm.display.input.reset(op.typing) }
  if (takeFocus) { ensureFocus(op.cm) }
}

function endOperation_finish(op) {
  var cm = op.cm, display = cm.display, doc = cm.doc

  if (op.updatedDisplay) { postUpdateDisplay(cm, op.update) }

  // Abort mouse wheel delta measurement, when scrolling explicitly
  if (display.wheelStartX != null && (op.scrollTop != null || op.scrollLeft != null || op.scrollToPos))
    { display.wheelStartX = display.wheelStartY = null }

  // Propagate the scroll position to the actual DOM scroller
  if (op.scrollTop != null && (display.scroller.scrollTop != op.scrollTop || op.forceScroll)) {
    doc.scrollTop = Math.max(0, Math.min(display.scroller.scrollHeight - display.scroller.clientHeight, op.scrollTop))
    display.scrollbars.setScrollTop(doc.scrollTop)
    display.scroller.scrollTop = doc.scrollTop
  }
  if (op.scrollLeft != null && (display.scroller.scrollLeft != op.scrollLeft || op.forceScroll)) {
    doc.scrollLeft = Math.max(0, Math.min(display.scroller.scrollWidth - display.scroller.clientWidth, op.scrollLeft))
    display.scrollbars.setScrollLeft(doc.scrollLeft)
    display.scroller.scrollLeft = doc.scrollLeft
    alignHorizontally(cm)
  }
  // If we need to scroll a specific position into view, do so.
  if (op.scrollToPos) {
    var coords = scrollPosIntoView(cm, clipPos(doc, op.scrollToPos.from),
                                   clipPos(doc, op.scrollToPos.to), op.scrollToPos.margin)
    if (op.scrollToPos.isCursor && cm.state.focused) { maybeScrollWindow(cm, coords) }
  }

  // Fire events for markers that are hidden/unidden by editing or
  // undoing
  var hidden = op.maybeHiddenMarkers, unhidden = op.maybeUnhiddenMarkers
  if (hidden) { for (var i = 0; i < hidden.length; ++i)
    { if (!hidden[i].lines.length) { signal(hidden[i], "hide") } } }
  if (unhidden) { for (var i$1 = 0; i$1 < unhidden.length; ++i$1)
    { if (unhidden[i$1].lines.length) { signal(unhidden[i$1], "unhide") } } }

  if (display.wrapper.offsetHeight)
    { doc.scrollTop = cm.display.scroller.scrollTop }

  // Fire change events, and delayed event handlers
  if (op.changeObjs)
    { signal(cm, "changes", cm, op.changeObjs) }
  if (op.update)
    { op.update.finish() }
}

// Run the given function in an operation
function runInOp(cm, f) {
  if (cm.curOp) { return f() }
  startOperation(cm)
  try { return f() }
  finally { endOperation(cm) }
}
// Wraps a function in an operation. Returns the wrapped function.
function operation(cm, f) {
  return function() {
    if (cm.curOp) { return f.apply(cm, arguments) }
    startOperation(cm)
    try { return f.apply(cm, arguments) }
    finally { endOperation(cm) }
  }
}
// Used to add methods to editor and doc instances, wrapping them in
// operations.
function methodOp(f) {
  return function() {
    if (this.curOp) { return f.apply(this, arguments) }
    startOperation(this)
    try { return f.apply(this, arguments) }
    finally { endOperation(this) }
  }
}
function docMethodOp(f) {
  return function() {
    var cm = this.cm
    if (!cm || cm.curOp) { return f.apply(this, arguments) }
    startOperation(cm)
    try { return f.apply(this, arguments) }
    finally { endOperation(cm) }
  }
}

// Updates the display.view data structure for a given change to the
// document. From and to are in pre-change coordinates. Lendiff is
// the amount of lines added or subtracted by the change. This is
// used for changes that span multiple lines, or change the way
// lines are divided into visual lines. regLineChange (below)
// registers single-line changes.
function regChange(cm, from, to, lendiff) {
  if (from == null) { from = cm.doc.first }
  if (to == null) { to = cm.doc.first + cm.doc.size }
  if (!lendiff) { lendiff = 0 }

  var display = cm.display
  if (lendiff && to < display.viewTo &&
      (display.updateLineNumbers == null || display.updateLineNumbers > from))
    { display.updateLineNumbers = from }

  cm.curOp.viewChanged = true

  if (from >= display.viewTo) { // Change after
    if (sawCollapsedSpans && visualLineNo(cm.doc, from) < display.viewTo)
      { resetView(cm) }
  } else if (to <= display.viewFrom) { // Change before
    if (sawCollapsedSpans && visualLineEndNo(cm.doc, to + lendiff) > display.viewFrom) {
      resetView(cm)
    } else {
      display.viewFrom += lendiff
      display.viewTo += lendiff
    }
  } else if (from <= display.viewFrom && to >= display.viewTo) { // Full overlap
    resetView(cm)
  } else if (from <= display.viewFrom) { // Top overlap
    var cut = viewCuttingPoint(cm, to, to + lendiff, 1)
    if (cut) {
      display.view = display.view.slice(cut.index)
      display.viewFrom = cut.lineN
      display.viewTo += lendiff
    } else {
      resetView(cm)
    }
  } else if (to >= display.viewTo) { // Bottom overlap
    var cut$1 = viewCuttingPoint(cm, from, from, -1)
    if (cut$1) {
      display.view = display.view.slice(0, cut$1.index)
      display.viewTo = cut$1.lineN
    } else {
      resetView(cm)
    }
  } else { // Gap in the middle
    var cutTop = viewCuttingPoint(cm, from, from, -1)
    var cutBot = viewCuttingPoint(cm, to, to + lendiff, 1)
    if (cutTop && cutBot) {
      display.view = display.view.slice(0, cutTop.index)
        .concat(buildViewArray(cm, cutTop.lineN, cutBot.lineN))
        .concat(display.view.slice(cutBot.index))
      display.viewTo += lendiff
    } else {
      resetView(cm)
    }
  }

  var ext = display.externalMeasured
  if (ext) {
    if (to < ext.lineN)
      { ext.lineN += lendiff }
    else if (from < ext.lineN + ext.size)
      { display.externalMeasured = null }
  }
}

// Register a change to a single line. Type must be one of "text",
// "gutter", "class", "widget"
function regLineChange(cm, line, type) {
  cm.curOp.viewChanged = true
  var display = cm.display, ext = cm.display.externalMeasured
  if (ext && line >= ext.lineN && line < ext.lineN + ext.size)
    { display.externalMeasured = null }

  if (line < display.viewFrom || line >= display.viewTo) { return }
  var lineView = display.view[findViewIndex(cm, line)]
  if (lineView.node == null) { return }
  var arr = lineView.changes || (lineView.changes = [])
  if (indexOf(arr, type) == -1) { arr.push(type) }
}

// Clear the view.
function resetView(cm) {
  cm.display.viewFrom = cm.display.viewTo = cm.doc.first
  cm.display.view = []
  cm.display.viewOffset = 0
}

function viewCuttingPoint(cm, oldN, newN, dir) {
  var index = findViewIndex(cm, oldN), diff, view = cm.display.view
  if (!sawCollapsedSpans || newN == cm.doc.first + cm.doc.size)
    { return {index: index, lineN: newN} }
  var n = cm.display.viewFrom
  for (var i = 0; i < index; i++)
    { n += view[i].size }
  if (n != oldN) {
    if (dir > 0) {
      if (index == view.length - 1) { return null }
      diff = (n + view[index].size) - oldN
      index++
    } else {
      diff = n - oldN
    }
    oldN += diff; newN += diff
  }
  while (visualLineNo(cm.doc, newN) != newN) {
    if (index == (dir < 0 ? 0 : view.length - 1)) { return null }
    newN += dir * view[index - (dir < 0 ? 1 : 0)].size
    index += dir
  }
  return {index: index, lineN: newN}
}

// Force the view to cover a given range, adding empty view element
// or clipping off existing ones as needed.
function adjustView(cm, from, to) {
  var display = cm.display, view = display.view
  if (view.length == 0 || from >= display.viewTo || to <= display.viewFrom) {
    display.view = buildViewArray(cm, from, to)
    display.viewFrom = from
  } else {
    if (display.viewFrom > from)
      { display.view = buildViewArray(cm, from, display.viewFrom).concat(display.view) }
    else if (display.viewFrom < from)
      { display.view = display.view.slice(findViewIndex(cm, from)) }
    display.viewFrom = from
    if (display.viewTo < to)
      { display.view = display.view.concat(buildViewArray(cm, display.viewTo, to)) }
    else if (display.viewTo > to)
      { display.view = display.view.slice(0, findViewIndex(cm, to)) }
  }
  display.viewTo = to
}

// Count the number of lines in the view whose DOM representation is
// out of date (or nonexistent).
function countDirtyView(cm) {
  var view = cm.display.view, dirty = 0
  for (var i = 0; i < view.length; i++) {
    var lineView = view[i]
    if (!lineView.hidden && (!lineView.node || lineView.changes)) { ++dirty }
  }
  return dirty
}

// HIGHLIGHT WORKER

function startWorker(cm, time) {
  if (cm.doc.mode.startState && cm.doc.frontier < cm.display.viewTo)
    { cm.state.highlight.set(time, bind(highlightWorker, cm)) }
}

function highlightWorker(cm) {
  var doc = cm.doc
  if (doc.frontier < doc.first) { doc.frontier = doc.first }
  if (doc.frontier >= cm.display.viewTo) { return }
  var end = +new Date + cm.options.workTime
  var state = copyState(doc.mode, getStateBefore(cm, doc.frontier))
  var changedLines = []

  doc.iter(doc.frontier, Math.min(doc.first + doc.size, cm.display.viewTo + 500), function (line) {
    if (doc.frontier >= cm.display.viewFrom) { // Visible
      var oldStyles = line.styles, tooLong = line.text.length > cm.options.maxHighlightLength
      var highlighted = highlightLine(cm, line, tooLong ? copyState(doc.mode, state) : state, true)
      line.styles = highlighted.styles
      var oldCls = line.styleClasses, newCls = highlighted.classes
      if (newCls) { line.styleClasses = newCls }
      else if (oldCls) { line.styleClasses = null }
      var ischange = !oldStyles || oldStyles.length != line.styles.length ||
        oldCls != newCls && (!oldCls || !newCls || oldCls.bgClass != newCls.bgClass || oldCls.textClass != newCls.textClass)
      for (var i = 0; !ischange && i < oldStyles.length; ++i) { ischange = oldStyles[i] != line.styles[i] }
      if (ischange) { changedLines.push(doc.frontier) }
      line.stateAfter = tooLong ? state : copyState(doc.mode, state)
    } else {
      if (line.text.length <= cm.options.maxHighlightLength)
        { processLine(cm, line.text, state) }
      line.stateAfter = doc.frontier % 5 == 0 ? copyState(doc.mode, state) : null
    }
    ++doc.frontier
    if (+new Date > end) {
      startWorker(cm, cm.options.workDelay)
      return true
    }
  })
  if (changedLines.length) { runInOp(cm, function () {
    for (var i = 0; i < changedLines.length; i++)
      { regLineChange(cm, changedLines[i], "text") }
  }) }
}

// DISPLAY DRAWING

var DisplayUpdate = function(cm, viewport, force) {
  var display = cm.display

  this.viewport = viewport
  // Store some values that we'll need later (but don't want to force a relayout for)
  this.visible = visibleLines(display, cm.doc, viewport)
  this.editorIsHidden = !display.wrapper.offsetWidth
  this.wrapperHeight = display.wrapper.clientHeight
  this.wrapperWidth = display.wrapper.clientWidth
  this.oldDisplayWidth = displayWidth(cm)
  this.force = force
  this.dims = getDimensions(cm)
  this.events = []
};

DisplayUpdate.prototype.signal = function (emitter, type) {
  if (hasHandler(emitter, type))
    { this.events.push(arguments) }
};
DisplayUpdate.prototype.finish = function () {
    var this$1 = this;

  for (var i = 0; i < this.events.length; i++)
    { signal.apply(null, this$1.events[i]) }
};

function maybeClipScrollbars(cm) {
  var display = cm.display
  if (!display.scrollbarsClipped && display.scroller.offsetWidth) {
    display.nativeBarWidth = display.scroller.offsetWidth - display.scroller.clientWidth
    display.heightForcer.style.height = scrollGap(cm) + "px"
    display.sizer.style.marginBottom = -display.nativeBarWidth + "px"
    display.sizer.style.borderRightWidth = scrollGap(cm) + "px"
    display.scrollbarsClipped = true
  }
}

// Does the actual updating of the line display. Bails out
// (returning false) when there is nothing to be done and forced is
// false.
function updateDisplayIfNeeded(cm, update) {
  var display = cm.display, doc = cm.doc

  if (update.editorIsHidden) {
    resetView(cm)
    return false
  }

  // Bail out if the visible area is already rendered and nothing changed.
  if (!update.force &&
      update.visible.from >= display.viewFrom && update.visible.to <= display.viewTo &&
      (display.updateLineNumbers == null || display.updateLineNumbers >= display.viewTo) &&
      display.renderedView == display.view && countDirtyView(cm) == 0)
    { return false }

  if (maybeUpdateLineNumberWidth(cm)) {
    resetView(cm)
    update.dims = getDimensions(cm)
  }

  // Compute a suitable new viewport (from & to)
  var end = doc.first + doc.size
  var from = Math.max(update.visible.from - cm.options.viewportMargin, doc.first)
  var to = Math.min(end, update.visible.to + cm.options.viewportMargin)
  if (display.viewFrom < from && from - display.viewFrom < 20) { from = Math.max(doc.first, display.viewFrom) }
  if (display.viewTo > to && display.viewTo - to < 20) { to = Math.min(end, display.viewTo) }
  if (sawCollapsedSpans) {
    from = visualLineNo(cm.doc, from)
    to = visualLineEndNo(cm.doc, to)
  }

  var different = from != display.viewFrom || to != display.viewTo ||
    display.lastWrapHeight != update.wrapperHeight || display.lastWrapWidth != update.wrapperWidth
  adjustView(cm, from, to)

  display.viewOffset = heightAtLine(getLine(cm.doc, display.viewFrom))
  // Position the mover div to align with the current scroll position
  cm.display.mover.style.top = display.viewOffset + "px"

  var toUpdate = countDirtyView(cm)
  if (!different && toUpdate == 0 && !update.force && display.renderedView == display.view &&
      (display.updateLineNumbers == null || display.updateLineNumbers >= display.viewTo))
    { return false }

  // For big changes, we hide the enclosing element during the
  // update, since that speeds up the operations on most browsers.
  var focused = activeElt()
  if (toUpdate > 4) { display.lineDiv.style.display = "none" }
  patchDisplay(cm, display.updateLineNumbers, update.dims)
  if (toUpdate > 4) { display.lineDiv.style.display = "" }
  display.renderedView = display.view
  // There might have been a widget with a focused element that got
  // hidden or updated, if so re-focus it.
  if (focused && activeElt() != focused && focused.offsetHeight) { focused.focus() }

  // Prevent selection and cursors from interfering with the scroll
  // width and height.
  removeChildren(display.cursorDiv)
  removeChildren(display.selectionDiv)
  display.gutters.style.height = display.sizer.style.minHeight = 0

  if (different) {
    display.lastWrapHeight = update.wrapperHeight
    display.lastWrapWidth = update.wrapperWidth
    startWorker(cm, 400)
  }

  display.updateLineNumbers = null

  return true
}

function postUpdateDisplay(cm, update) {
  var viewport = update.viewport

  for (var first = true;; first = false) {
    if (!first || !cm.options.lineWrapping || update.oldDisplayWidth == displayWidth(cm)) {
      // Clip forced viewport to actual scrollable area.
      if (viewport && viewport.top != null)
        { viewport = {top: Math.min(cm.doc.height + paddingVert(cm.display) - displayHeight(cm), viewport.top)} }
      // Updated line heights might result in the drawn area not
      // actually covering the viewport. Keep looping until it does.
      update.visible = visibleLines(cm.display, cm.doc, viewport)
      if (update.visible.from >= cm.display.viewFrom && update.visible.to <= cm.display.viewTo)
        { break }
    }
    if (!updateDisplayIfNeeded(cm, update)) { break }
    updateHeightsInViewport(cm)
    var barMeasure = measureForScrollbars(cm)
    updateSelection(cm)
    updateScrollbars(cm, barMeasure)
    setDocumentHeight(cm, barMeasure)
  }

  update.signal(cm, "update", cm)
  if (cm.display.viewFrom != cm.display.reportedViewFrom || cm.display.viewTo != cm.display.reportedViewTo) {
    update.signal(cm, "viewportChange", cm, cm.display.viewFrom, cm.display.viewTo)
    cm.display.reportedViewFrom = cm.display.viewFrom; cm.display.reportedViewTo = cm.display.viewTo
  }
}

function updateDisplaySimple(cm, viewport) {
  var update = new DisplayUpdate(cm, viewport)
  if (updateDisplayIfNeeded(cm, update)) {
    updateHeightsInViewport(cm)
    postUpdateDisplay(cm, update)
    var barMeasure = measureForScrollbars(cm)
    updateSelection(cm)
    updateScrollbars(cm, barMeasure)
    setDocumentHeight(cm, barMeasure)
    update.finish()
  }
}

// Sync the actual display DOM structure with display.view, removing
// nodes for lines that are no longer in view, and creating the ones
// that are not there yet, and updating the ones that are out of
// date.
function patchDisplay(cm, updateNumbersFrom, dims) {
  var display = cm.display, lineNumbers = cm.options.lineNumbers
  var container = display.lineDiv, cur = container.firstChild

  function rm(node) {
    var next = node.nextSibling
    // Works around a throw-scroll bug in OS X Webkit
    if (webkit && mac && cm.display.currentWheelTarget == node)
      { node.style.display = "none" }
    else
      { node.parentNode.removeChild(node) }
    return next
  }

  var view = display.view, lineN = display.viewFrom
  // Loop over the elements in the view, syncing cur (the DOM nodes
  // in display.lineDiv) with the view as we go.
  for (var i = 0; i < view.length; i++) {
    var lineView = view[i]
    if (lineView.hidden) {
    } else if (!lineView.node || lineView.node.parentNode != container) { // Not drawn yet
      var node = buildLineElement(cm, lineView, lineN, dims)
      container.insertBefore(node, cur)
    } else { // Already drawn
      while (cur != lineView.node) { cur = rm(cur) }
      var updateNumber = lineNumbers && updateNumbersFrom != null &&
        updateNumbersFrom <= lineN && lineView.lineNumber
      if (lineView.changes) {
        if (indexOf(lineView.changes, "gutter") > -1) { updateNumber = false }
        updateLineForChanges(cm, lineView, lineN, dims)
      }
      if (updateNumber) {
        removeChildren(lineView.lineNumber)
        lineView.lineNumber.appendChild(document.createTextNode(lineNumberFor(cm.options, lineN)))
      }
      cur = lineView.node.nextSibling
    }
    lineN += lineView.size
  }
  while (cur) { cur = rm(cur) }
}

function updateGutterSpace(cm) {
  var width = cm.display.gutters.offsetWidth
  cm.display.sizer.style.marginLeft = width + "px"
}

function setDocumentHeight(cm, measure) {
  cm.display.sizer.style.minHeight = measure.docHeight + "px"
  cm.display.heightForcer.style.top = measure.docHeight + "px"
  cm.display.gutters.style.height = (measure.docHeight + cm.display.barHeight + scrollGap(cm)) + "px"
}

// Rebuild the gutter elements, ensure the margin to the left of the
// code matches their width.
function updateGutters(cm) {
  var gutters = cm.display.gutters, specs = cm.options.gutters
  removeChildren(gutters)
  var i = 0
  for (; i < specs.length; ++i) {
    var gutterClass = specs[i]
    var gElt = gutters.appendChild(elt("div", null, "CodeMirror-gutter " + gutterClass))
    if (gutterClass == "CodeMirror-linenumbers") {
      cm.display.lineGutter = gElt
      gElt.style.width = (cm.display.lineNumWidth || 1) + "px"
    }
  }
  gutters.style.display = i ? "" : "none"
  updateGutterSpace(cm)
}

// Make sure the gutters options contains the element
// "CodeMirror-linenumbers" when the lineNumbers option is true.
function setGuttersForLineNumbers(options) {
  var found = indexOf(options.gutters, "CodeMirror-linenumbers")
  if (found == -1 && options.lineNumbers) {
    options.gutters = options.gutters.concat(["CodeMirror-linenumbers"])
  } else if (found > -1 && !options.lineNumbers) {
    options.gutters = options.gutters.slice(0)
    options.gutters.splice(found, 1)
  }
}

// Selection objects are immutable. A new one is created every time
// the selection changes. A selection is one or more non-overlapping
// (and non-touching) ranges, sorted, and an integer that indicates
// which one is the primary selection (the one that's scrolled into
// view, that getCursor returns, etc).
var Selection = function(ranges, primIndex) {
  this.ranges = ranges
  this.primIndex = primIndex
};

Selection.prototype.primary = function () { return this.ranges[this.primIndex] };

Selection.prototype.equals = function (other) {
    var this$1 = this;

  if (other == this) { return true }
  if (other.primIndex != this.primIndex || other.ranges.length != this.ranges.length) { return false }
  for (var i = 0; i < this.ranges.length; i++) {
    var here = this$1.ranges[i], there = other.ranges[i]
    if (!equalCursorPos(here.anchor, there.anchor) || !equalCursorPos(here.head, there.head)) { return false }
  }
  return true
};

Selection.prototype.deepCopy = function () {
    var this$1 = this;

  var out = []
  for (var i = 0; i < this.ranges.length; i++)
    { out[i] = new Range(copyPos(this$1.ranges[i].anchor), copyPos(this$1.ranges[i].head)) }
  return new Selection(out, this.primIndex)
};

Selection.prototype.somethingSelected = function () {
    var this$1 = this;

  for (var i = 0; i < this.ranges.length; i++)
    { if (!this$1.ranges[i].empty()) { return true } }
  return false
};

Selection.prototype.contains = function (pos, end) {
    var this$1 = this;

  if (!end) { end = pos }
  for (var i = 0; i < this.ranges.length; i++) {
    var range = this$1.ranges[i]
    if (cmp(end, range.from()) >= 0 && cmp(pos, range.to()) <= 0)
      { return i }
  }
  return -1
};

var Range = function(anchor, head) {
  this.anchor = anchor; this.head = head
};

Range.prototype.from = function () { return minPos(this.anchor, this.head) };
Range.prototype.to = function () { return maxPos(this.anchor, this.head) };
Range.prototype.empty = function () { return this.head.line == this.anchor.line && this.head.ch == this.anchor.ch };

// Take an unsorted, potentially overlapping set of ranges, and
// build a selection out of it. 'Consumes' ranges array (modifying
// it).
function normalizeSelection(ranges, primIndex) {
  var prim = ranges[primIndex]
  ranges.sort(function (a, b) { return cmp(a.from(), b.from()); })
  primIndex = indexOf(ranges, prim)
  for (var i = 1; i < ranges.length; i++) {
    var cur = ranges[i], prev = ranges[i - 1]
    if (cmp(prev.to(), cur.from()) >= 0) {
      var from = minPos(prev.from(), cur.from()), to = maxPos(prev.to(), cur.to())
      var inv = prev.empty() ? cur.from() == cur.head : prev.from() == prev.head
      if (i <= primIndex) { --primIndex }
      ranges.splice(--i, 2, new Range(inv ? to : from, inv ? from : to))
    }
  }
  return new Selection(ranges, primIndex)
}

function simpleSelection(anchor, head) {
  return new Selection([new Range(anchor, head || anchor)], 0)
}

// Compute the position of the end of a change (its 'to' property
// refers to the pre-change end).
function changeEnd(change) {
  if (!change.text) { return change.to }
  return Pos(change.from.line + change.text.length - 1,
             lst(change.text).length + (change.text.length == 1 ? change.from.ch : 0))
}

// Adjust a position to refer to the post-change position of the
// same text, or the end of the change if the change covers it.
function adjustForChange(pos, change) {
  if (cmp(pos, change.from) < 0) { return pos }
  if (cmp(pos, change.to) <= 0) { return changeEnd(change) }

  var line = pos.line + change.text.length - (change.to.line - change.from.line) - 1, ch = pos.ch
  if (pos.line == change.to.line) { ch += changeEnd(change).ch - change.to.ch }
  return Pos(line, ch)
}

function computeSelAfterChange(doc, change) {
  var out = []
  for (var i = 0; i < doc.sel.ranges.length; i++) {
    var range = doc.sel.ranges[i]
    out.push(new Range(adjustForChange(range.anchor, change),
                       adjustForChange(range.head, change)))
  }
  return normalizeSelection(out, doc.sel.primIndex)
}

function offsetPos(pos, old, nw) {
  if (pos.line == old.line)
    { return Pos(nw.line, pos.ch - old.ch + nw.ch) }
  else
    { return Pos(nw.line + (pos.line - old.line), pos.ch) }
}

// Used by replaceSelections to allow moving the selection to the
// start or around the replaced test. Hint may be "start" or "around".
function computeReplacedSel(doc, changes, hint) {
  var out = []
  var oldPrev = Pos(doc.first, 0), newPrev = oldPrev
  for (var i = 0; i < changes.length; i++) {
    var change = changes[i]
    var from = offsetPos(change.from, oldPrev, newPrev)
    var to = offsetPos(changeEnd(change), oldPrev, newPrev)
    oldPrev = change.to
    newPrev = to
    if (hint == "around") {
      var range = doc.sel.ranges[i], inv = cmp(range.head, range.anchor) < 0
      out[i] = new Range(inv ? to : from, inv ? from : to)
    } else {
      out[i] = new Range(from, from)
    }
  }
  return new Selection(out, doc.sel.primIndex)
}

// Used to get the editor into a consistent state again when options change.

function loadMode(cm) {
  cm.doc.mode = getMode(cm.options, cm.doc.modeOption)
  resetModeState(cm)
}

function resetModeState(cm) {
  cm.doc.iter(function (line) {
    if (line.stateAfter) { line.stateAfter = null }
    if (line.styles) { line.styles = null }
  })
  cm.doc.frontier = cm.doc.first
  startWorker(cm, 100)
  cm.state.modeGen++
  if (cm.curOp) { regChange(cm) }
}

// DOCUMENT DATA STRUCTURE

// By default, updates that start and end at the beginning of a line
// are treated specially, in order to make the association of line
// widgets and marker elements with the text behave more intuitive.
function isWholeLineUpdate(doc, change) {
  return change.from.ch == 0 && change.to.ch == 0 && lst(change.text) == "" &&
    (!doc.cm || doc.cm.options.wholeLineUpdateBefore)
}

// Perform a change on the document data structure.
function updateDoc(doc, change, markedSpans, estimateHeight) {
  function spansFor(n) {return markedSpans ? markedSpans[n] : null}
  function update(line, text, spans) {
    updateLine(line, text, spans, estimateHeight)
    signalLater(line, "change", line, change)
  }
  function linesFor(start, end) {
    var result = []
    for (var i = start; i < end; ++i)
      { result.push(new Line(text[i], spansFor(i), estimateHeight)) }
    return result
  }

  var from = change.from, to = change.to, text = change.text
  var firstLine = getLine(doc, from.line), lastLine = getLine(doc, to.line)
  var lastText = lst(text), lastSpans = spansFor(text.length - 1), nlines = to.line - from.line

  // Adjust the line structure
  if (change.full) {
    doc.insert(0, linesFor(0, text.length))
    doc.remove(text.length, doc.size - text.length)
  } else if (isWholeLineUpdate(doc, change)) {
    // This is a whole-line replace. Treated specially to make
    // sure line objects move the way they are supposed to.
    var added = linesFor(0, text.length - 1)
    update(lastLine, lastLine.text, lastSpans)
    if (nlines) { doc.remove(from.line, nlines) }
    if (added.length) { doc.insert(from.line, added) }
  } else if (firstLine == lastLine) {
    if (text.length == 1) {
      update(firstLine, firstLine.text.slice(0, from.ch) + lastText + firstLine.text.slice(to.ch), lastSpans)
    } else {
      var added$1 = linesFor(1, text.length - 1)
      added$1.push(new Line(lastText + firstLine.text.slice(to.ch), lastSpans, estimateHeight))
      update(firstLine, firstLine.text.slice(0, from.ch) + text[0], spansFor(0))
      doc.insert(from.line + 1, added$1)
    }
  } else if (text.length == 1) {
    update(firstLine, firstLine.text.slice(0, from.ch) + text[0] + lastLine.text.slice(to.ch), spansFor(0))
    doc.remove(from.line + 1, nlines)
  } else {
    update(firstLine, firstLine.text.slice(0, from.ch) + text[0], spansFor(0))
    update(lastLine, lastText + lastLine.text.slice(to.ch), lastSpans)
    var added$2 = linesFor(1, text.length - 1)
    if (nlines > 1) { doc.remove(from.line + 1, nlines - 1) }
    doc.insert(from.line + 1, added$2)
  }

  signalLater(doc, "change", doc, change)
}

// Call f for all linked documents.
function linkedDocs(doc, f, sharedHistOnly) {
  function propagate(doc, skip, sharedHist) {
    if (doc.linked) { for (var i = 0; i < doc.linked.length; ++i) {
      var rel = doc.linked[i]
      if (rel.doc == skip) { continue }
      var shared = sharedHist && rel.sharedHist
      if (sharedHistOnly && !shared) { continue }
      f(rel.doc, shared)
      propagate(rel.doc, doc, shared)
    } }
  }
  propagate(doc, null, true)
}

// Attach a document to an editor.
function attachDoc(cm, doc) {
  if (doc.cm) { throw new Error("This document is already in use.") }
  cm.doc = doc
  doc.cm = cm
  estimateLineHeights(cm)
  loadMode(cm)
  if (!cm.options.lineWrapping) { findMaxLine(cm) }
  cm.options.mode = doc.modeOption
  regChange(cm)
}

function History(startGen) {
  // Arrays of change events and selections. Doing something adds an
  // event to done and clears undo. Undoing moves events from done
  // to undone, redoing moves them in the other direction.
  this.done = []; this.undone = []
  this.undoDepth = Infinity
  // Used to track when changes can be merged into a single undo
  // event
  this.lastModTime = this.lastSelTime = 0
  this.lastOp = this.lastSelOp = null
  this.lastOrigin = this.lastSelOrigin = null
  // Used by the isClean() method
  this.generation = this.maxGeneration = startGen || 1
}

// Create a history change event from an updateDoc-style change
// object.
function historyChangeFromChange(doc, change) {
  var histChange = {from: copyPos(change.from), to: changeEnd(change), text: getBetween(doc, change.from, change.to)}
  attachLocalSpans(doc, histChange, change.from.line, change.to.line + 1)
  linkedDocs(doc, function (doc) { return attachLocalSpans(doc, histChange, change.from.line, change.to.line + 1); }, true)
  return histChange
}

// Pop all selection events off the end of a history array. Stop at
// a change event.
function clearSelectionEvents(array) {
  while (array.length) {
    var last = lst(array)
    if (last.ranges) { array.pop() }
    else { break }
  }
}

// Find the top change event in the history. Pop off selection
// events that are in the way.
function lastChangeEvent(hist, force) {
  if (force) {
    clearSelectionEvents(hist.done)
    return lst(hist.done)
  } else if (hist.done.length && !lst(hist.done).ranges) {
    return lst(hist.done)
  } else if (hist.done.length > 1 && !hist.done[hist.done.length - 2].ranges) {
    hist.done.pop()
    return lst(hist.done)
  }
}

// Register a change in the history. Merges changes that are within
// a single operation, or are close together with an origin that
// allows merging (starting with "+") into a single event.
function addChangeToHistory(doc, change, selAfter, opId) {
  var hist = doc.history
  hist.undone.length = 0
  var time = +new Date, cur
  var last

  if ((hist.lastOp == opId ||
       hist.lastOrigin == change.origin && change.origin &&
       ((change.origin.charAt(0) == "+" && doc.cm && hist.lastModTime > time - doc.cm.options.historyEventDelay) ||
        change.origin.charAt(0) == "*")) &&
      (cur = lastChangeEvent(hist, hist.lastOp == opId))) {
    // Merge this change into the last event
    last = lst(cur.changes)
    if (cmp(change.from, change.to) == 0 && cmp(change.from, last.to) == 0) {
      // Optimized case for simple insertion -- don't want to add
      // new changesets for every character typed
      last.to = changeEnd(change)
    } else {
      // Add new sub-event
      cur.changes.push(historyChangeFromChange(doc, change))
    }
  } else {
    // Can not be merged, start a new event.
    var before = lst(hist.done)
    if (!before || !before.ranges)
      { pushSelectionToHistory(doc.sel, hist.done) }
    cur = {changes: [historyChangeFromChange(doc, change)],
           generation: hist.generation}
    hist.done.push(cur)
    while (hist.done.length > hist.undoDepth) {
      hist.done.shift()
      if (!hist.done[0].ranges) { hist.done.shift() }
    }
  }
  hist.done.push(selAfter)
  hist.generation = ++hist.maxGeneration
  hist.lastModTime = hist.lastSelTime = time
  hist.lastOp = hist.lastSelOp = opId
  hist.lastOrigin = hist.lastSelOrigin = change.origin

  if (!last) { signal(doc, "historyAdded") }
}

function selectionEventCanBeMerged(doc, origin, prev, sel) {
  var ch = origin.charAt(0)
  return ch == "*" ||
    ch == "+" &&
    prev.ranges.length == sel.ranges.length &&
    prev.somethingSelected() == sel.somethingSelected() &&
    new Date - doc.history.lastSelTime <= (doc.cm ? doc.cm.options.historyEventDelay : 500)
}

// Called whenever the selection changes, sets the new selection as
// the pending selection in the history, and pushes the old pending
// selection into the 'done' array when it was significantly
// different (in number of selected ranges, emptiness, or time).
function addSelectionToHistory(doc, sel, opId, options) {
  var hist = doc.history, origin = options && options.origin

  // A new event is started when the previous origin does not match
  // the current, or the origins don't allow matching. Origins
  // starting with * are always merged, those starting with + are
  // merged when similar and close together in time.
  if (opId == hist.lastSelOp ||
      (origin && hist.lastSelOrigin == origin &&
       (hist.lastModTime == hist.lastSelTime && hist.lastOrigin == origin ||
        selectionEventCanBeMerged(doc, origin, lst(hist.done), sel))))
    { hist.done[hist.done.length - 1] = sel }
  else
    { pushSelectionToHistory(sel, hist.done) }

  hist.lastSelTime = +new Date
  hist.lastSelOrigin = origin
  hist.lastSelOp = opId
  if (options && options.clearRedo !== false)
    { clearSelectionEvents(hist.undone) }
}

function pushSelectionToHistory(sel, dest) {
  var top = lst(dest)
  if (!(top && top.ranges && top.equals(sel)))
    { dest.push(sel) }
}

// Used to store marked span information in the history.
function attachLocalSpans(doc, change, from, to) {
  var existing = change["spans_" + doc.id], n = 0
  doc.iter(Math.max(doc.first, from), Math.min(doc.first + doc.size, to), function (line) {
    if (line.markedSpans)
      { (existing || (existing = change["spans_" + doc.id] = {}))[n] = line.markedSpans }
    ++n
  })
}

// When un/re-doing restores text containing marked spans, those
// that have been explicitly cleared should not be restored.
function removeClearedSpans(spans) {
  if (!spans) { return null }
  var out
  for (var i = 0; i < spans.length; ++i) {
    if (spans[i].marker.explicitlyCleared) { if (!out) { out = spans.slice(0, i) } }
    else if (out) { out.push(spans[i]) }
  }
  return !out ? spans : out.length ? out : null
}

// Retrieve and filter the old marked spans stored in a change event.
function getOldSpans(doc, change) {
  var found = change["spans_" + doc.id]
  if (!found) { return null }
  var nw = []
  for (var i = 0; i < change.text.length; ++i)
    { nw.push(removeClearedSpans(found[i])) }
  return nw
}

// Used for un/re-doing changes from the history. Combines the
// result of computing the existing spans with the set of spans that
// existed in the history (so that deleting around a span and then
// undoing brings back the span).
function mergeOldSpans(doc, change) {
  var old = getOldSpans(doc, change)
  var stretched = stretchSpansOverChange(doc, change)
  if (!old) { return stretched }
  if (!stretched) { return old }

  for (var i = 0; i < old.length; ++i) {
    var oldCur = old[i], stretchCur = stretched[i]
    if (oldCur && stretchCur) {
      spans: for (var j = 0; j < stretchCur.length; ++j) {
        var span = stretchCur[j]
        for (var k = 0; k < oldCur.length; ++k)
          { if (oldCur[k].marker == span.marker) { continue spans } }
        oldCur.push(span)
      }
    } else if (stretchCur) {
      old[i] = stretchCur
    }
  }
  return old
}

// Used both to provide a JSON-safe object in .getHistory, and, when
// detaching a document, to split the history in two
function copyHistoryArray(events, newGroup, instantiateSel) {
  var copy = []
  for (var i = 0; i < events.length; ++i) {
    var event = events[i]
    if (event.ranges) {
      copy.push(instantiateSel ? Selection.prototype.deepCopy.call(event) : event)
      continue
    }
    var changes = event.changes, newChanges = []
    copy.push({changes: newChanges})
    for (var j = 0; j < changes.length; ++j) {
      var change = changes[j], m = (void 0)
      newChanges.push({from: change.from, to: change.to, text: change.text})
      if (newGroup) { for (var prop in change) { if (m = prop.match(/^spans_(\d+)$/)) {
        if (indexOf(newGroup, Number(m[1])) > -1) {
          lst(newChanges)[prop] = change[prop]
          delete change[prop]
        }
      } } }
    }
  }
  return copy
}

// The 'scroll' parameter given to many of these indicated whether
// the new cursor position should be scrolled into view after
// modifying the selection.

// If shift is held or the extend flag is set, extends a range to
// include a given position (and optionally a second position).
// Otherwise, simply returns the range between the given positions.
// Used for cursor motion and such.
function extendRange(doc, range, head, other) {
  if (doc.cm && doc.cm.display.shift || doc.extend) {
    var anchor = range.anchor
    if (other) {
      var posBefore = cmp(head, anchor) < 0
      if (posBefore != (cmp(other, anchor) < 0)) {
        anchor = head
        head = other
      } else if (posBefore != (cmp(head, other) < 0)) {
        head = other
      }
    }
    return new Range(anchor, head)
  } else {
    return new Range(other || head, head)
  }
}

// Extend the primary selection range, discard the rest.
function extendSelection(doc, head, other, options) {
  setSelection(doc, new Selection([extendRange(doc, doc.sel.primary(), head, other)], 0), options)
}

// Extend all selections (pos is an array of selections with length
// equal the number of selections)
function extendSelections(doc, heads, options) {
  var out = []
  for (var i = 0; i < doc.sel.ranges.length; i++)
    { out[i] = extendRange(doc, doc.sel.ranges[i], heads[i], null) }
  var newSel = normalizeSelection(out, doc.sel.primIndex)
  setSelection(doc, newSel, options)
}

// Updates a single range in the selection.
function replaceOneSelection(doc, i, range, options) {
  var ranges = doc.sel.ranges.slice(0)
  ranges[i] = range
  setSelection(doc, normalizeSelection(ranges, doc.sel.primIndex), options)
}

// Reset the selection to a single range.
function setSimpleSelection(doc, anchor, head, options) {
  setSelection(doc, simpleSelection(anchor, head), options)
}

// Give beforeSelectionChange handlers a change to influence a
// selection update.
function filterSelectionChange(doc, sel, options) {
  var obj = {
    ranges: sel.ranges,
    update: function(ranges) {
      var this$1 = this;

      this.ranges = []
      for (var i = 0; i < ranges.length; i++)
        { this$1.ranges[i] = new Range(clipPos(doc, ranges[i].anchor),
                                   clipPos(doc, ranges[i].head)) }
    },
    origin: options && options.origin
  }
  signal(doc, "beforeSelectionChange", doc, obj)
  if (doc.cm) { signal(doc.cm, "beforeSelectionChange", doc.cm, obj) }
  if (obj.ranges != sel.ranges) { return normalizeSelection(obj.ranges, obj.ranges.length - 1) }
  else { return sel }
}

function setSelectionReplaceHistory(doc, sel, options) {
  var done = doc.history.done, last = lst(done)
  if (last && last.ranges) {
    done[done.length - 1] = sel
    setSelectionNoUndo(doc, sel, options)
  } else {
    setSelection(doc, sel, options)
  }
}

// Set a new selection.
function setSelection(doc, sel, options) {
  setSelectionNoUndo(doc, sel, options)
  addSelectionToHistory(doc, doc.sel, doc.cm ? doc.cm.curOp.id : NaN, options)
}

function setSelectionNoUndo(doc, sel, options) {
  if (hasHandler(doc, "beforeSelectionChange") || doc.cm && hasHandler(doc.cm, "beforeSelectionChange"))
    { sel = filterSelectionChange(doc, sel, options) }

  var bias = options && options.bias ||
    (cmp(sel.primary().head, doc.sel.primary().head) < 0 ? -1 : 1)
  setSelectionInner(doc, skipAtomicInSelection(doc, sel, bias, true))

  if (!(options && options.scroll === false) && doc.cm)
    { ensureCursorVisible(doc.cm) }
}

function setSelectionInner(doc, sel) {
  if (sel.equals(doc.sel)) { return }

  doc.sel = sel

  if (doc.cm) {
    doc.cm.curOp.updateInput = doc.cm.curOp.selectionChanged = true
    signalCursorActivity(doc.cm)
  }
  signalLater(doc, "cursorActivity", doc)
}

// Verify that the selection does not partially select any atomic
// marked ranges.
function reCheckSelection(doc) {
  setSelectionInner(doc, skipAtomicInSelection(doc, doc.sel, null, false), sel_dontScroll)
}

// Return a selection that does not partially select any atomic
// ranges.
function skipAtomicInSelection(doc, sel, bias, mayClear) {
  var out
  for (var i = 0; i < sel.ranges.length; i++) {
    var range = sel.ranges[i]
    var old = sel.ranges.length == doc.sel.ranges.length && doc.sel.ranges[i]
    var newAnchor = skipAtomic(doc, range.anchor, old && old.anchor, bias, mayClear)
    var newHead = skipAtomic(doc, range.head, old && old.head, bias, mayClear)
    if (out || newAnchor != range.anchor || newHead != range.head) {
      if (!out) { out = sel.ranges.slice(0, i) }
      out[i] = new Range(newAnchor, newHead)
    }
  }
  return out ? normalizeSelection(out, sel.primIndex) : sel
}

function skipAtomicInner(doc, pos, oldPos, dir, mayClear) {
  var line = getLine(doc, pos.line)
  if (line.markedSpans) { for (var i = 0; i < line.markedSpans.length; ++i) {
    var sp = line.markedSpans[i], m = sp.marker
    if ((sp.from == null || (m.inclusiveLeft ? sp.from <= pos.ch : sp.from < pos.ch)) &&
        (sp.to == null || (m.inclusiveRight ? sp.to >= pos.ch : sp.to > pos.ch))) {
      if (mayClear) {
        signal(m, "beforeCursorEnter")
        if (m.explicitlyCleared) {
          if (!line.markedSpans) { break }
          else {--i; continue}
        }
      }
      if (!m.atomic) { continue }

      if (oldPos) {
        var near = m.find(dir < 0 ? 1 : -1), diff = (void 0)
        if (dir < 0 ? m.inclusiveRight : m.inclusiveLeft)
          { near = movePos(doc, near, -dir, near && near.line == pos.line ? line : null) }
        if (near && near.line == pos.line && (diff = cmp(near, oldPos)) && (dir < 0 ? diff < 0 : diff > 0))
          { return skipAtomicInner(doc, near, pos, dir, mayClear) }
      }

      var far = m.find(dir < 0 ? -1 : 1)
      if (dir < 0 ? m.inclusiveLeft : m.inclusiveRight)
        { far = movePos(doc, far, dir, far.line == pos.line ? line : null) }
      return far ? skipAtomicInner(doc, far, pos, dir, mayClear) : null
    }
  } }
  return pos
}

// Ensure a given position is not inside an atomic range.
function skipAtomic(doc, pos, oldPos, bias, mayClear) {
  var dir = bias || 1
  var found = skipAtomicInner(doc, pos, oldPos, dir, mayClear) ||
      (!mayClear && skipAtomicInner(doc, pos, oldPos, dir, true)) ||
      skipAtomicInner(doc, pos, oldPos, -dir, mayClear) ||
      (!mayClear && skipAtomicInner(doc, pos, oldPos, -dir, true))
  if (!found) {
    doc.cantEdit = true
    return Pos(doc.first, 0)
  }
  return found
}

function movePos(doc, pos, dir, line) {
  if (dir < 0 && pos.ch == 0) {
    if (pos.line > doc.first) { return clipPos(doc, Pos(pos.line - 1)) }
    else { return null }
  } else if (dir > 0 && pos.ch == (line || getLine(doc, pos.line)).text.length) {
    if (pos.line < doc.first + doc.size - 1) { return Pos(pos.line + 1, 0) }
    else { return null }
  } else {
    return new Pos(pos.line, pos.ch + dir)
  }
}

function selectAll(cm) {
  cm.setSelection(Pos(cm.firstLine(), 0), Pos(cm.lastLine()), sel_dontScroll)
}

// UPDATING

// Allow "beforeChange" event handlers to influence a change
function filterChange(doc, change, update) {
  var obj = {
    canceled: false,
    from: change.from,
    to: change.to,
    text: change.text,
    origin: change.origin,
    cancel: function () { return obj.canceled = true; }
  }
  if (update) { obj.update = function (from, to, text, origin) {
    if (from) { obj.from = clipPos(doc, from) }
    if (to) { obj.to = clipPos(doc, to) }
    if (text) { obj.text = text }
    if (origin !== undefined) { obj.origin = origin }
  } }
  signal(doc, "beforeChange", doc, obj)
  if (doc.cm) { signal(doc.cm, "beforeChange", doc.cm, obj) }

  if (obj.canceled) { return null }
  return {from: obj.from, to: obj.to, text: obj.text, origin: obj.origin}
}

// Apply a change to a document, and add it to the document's
// history, and propagating it to all linked documents.
function makeChange(doc, change, ignoreReadOnly) {
  if (doc.cm) {
    if (!doc.cm.curOp) { return operation(doc.cm, makeChange)(doc, change, ignoreReadOnly) }
    if (doc.cm.state.suppressEdits) { return }
  }

  if (hasHandler(doc, "beforeChange") || doc.cm && hasHandler(doc.cm, "beforeChange")) {
    change = filterChange(doc, change, true)
    if (!change) { return }
  }

  // Possibly split or suppress the update based on the presence
  // of read-only spans in its range.
  var split = sawReadOnlySpans && !ignoreReadOnly && removeReadOnlyRanges(doc, change.from, change.to)
  if (split) {
    for (var i = split.length - 1; i >= 0; --i)
      { makeChangeInner(doc, {from: split[i].from, to: split[i].to, text: i ? [""] : change.text}) }
  } else {
    makeChangeInner(doc, change)
  }
}

function makeChangeInner(doc, change) {
  if (change.text.length == 1 && change.text[0] == "" && cmp(change.from, change.to) == 0) { return }
  var selAfter = computeSelAfterChange(doc, change)
  addChangeToHistory(doc, change, selAfter, doc.cm ? doc.cm.curOp.id : NaN)

  makeChangeSingleDoc(doc, change, selAfter, stretchSpansOverChange(doc, change))
  var rebased = []

  linkedDocs(doc, function (doc, sharedHist) {
    if (!sharedHist && indexOf(rebased, doc.history) == -1) {
      rebaseHist(doc.history, change)
      rebased.push(doc.history)
    }
    makeChangeSingleDoc(doc, change, null, stretchSpansOverChange(doc, change))
  })
}

// Revert a change stored in a document's history.
function makeChangeFromHistory(doc, type, allowSelectionOnly) {
  if (doc.cm && doc.cm.state.suppressEdits && !allowSelectionOnly) { return }

  var hist = doc.history, event, selAfter = doc.sel
  var source = type == "undo" ? hist.done : hist.undone, dest = type == "undo" ? hist.undone : hist.done

  // Verify that there is a useable event (so that ctrl-z won't
  // needlessly clear selection events)
  var i = 0
  for (; i < source.length; i++) {
    event = source[i]
    if (allowSelectionOnly ? event.ranges && !event.equals(doc.sel) : !event.ranges)
      { break }
  }
  if (i == source.length) { return }
  hist.lastOrigin = hist.lastSelOrigin = null

  for (;;) {
    event = source.pop()
    if (event.ranges) {
      pushSelectionToHistory(event, dest)
      if (allowSelectionOnly && !event.equals(doc.sel)) {
        setSelection(doc, event, {clearRedo: false})
        return
      }
      selAfter = event
    }
    else { break }
  }

  // Build up a reverse change object to add to the opposite history
  // stack (redo when undoing, and vice versa).
  var antiChanges = []
  pushSelectionToHistory(selAfter, dest)
  dest.push({changes: antiChanges, generation: hist.generation})
  hist.generation = event.generation || ++hist.maxGeneration

  var filter = hasHandler(doc, "beforeChange") || doc.cm && hasHandler(doc.cm, "beforeChange")

  var loop = function ( i ) {
    var change = event.changes[i]
    change.origin = type
    if (filter && !filterChange(doc, change, false)) {
      source.length = 0
      return {}
    }

    antiChanges.push(historyChangeFromChange(doc, change))

    var after = i ? computeSelAfterChange(doc, change) : lst(source)
    makeChangeSingleDoc(doc, change, after, mergeOldSpans(doc, change))
    if (!i && doc.cm) { doc.cm.scrollIntoView({from: change.from, to: changeEnd(change)}) }
    var rebased = []

    // Propagate to the linked documents
    linkedDocs(doc, function (doc, sharedHist) {
      if (!sharedHist && indexOf(rebased, doc.history) == -1) {
        rebaseHist(doc.history, change)
        rebased.push(doc.history)
      }
      makeChangeSingleDoc(doc, change, null, mergeOldSpans(doc, change))
    })
  };

  for (var i$1 = event.changes.length - 1; i$1 >= 0; --i$1) {
    var returned = loop( i$1 );

    if ( returned ) return returned.v;
  }
}

// Sub-views need their line numbers shifted when text is added
// above or below them in the parent document.
function shiftDoc(doc, distance) {
  if (distance == 0) { return }
  doc.first += distance
  doc.sel = new Selection(map(doc.sel.ranges, function (range) { return new Range(
    Pos(range.anchor.line + distance, range.anchor.ch),
    Pos(range.head.line + distance, range.head.ch)
  ); }), doc.sel.primIndex)
  if (doc.cm) {
    regChange(doc.cm, doc.first, doc.first - distance, distance)
    for (var d = doc.cm.display, l = d.viewFrom; l < d.viewTo; l++)
      { regLineChange(doc.cm, l, "gutter") }
  }
}

// More lower-level change function, handling only a single document
// (not linked ones).
function makeChangeSingleDoc(doc, change, selAfter, spans) {
  if (doc.cm && !doc.cm.curOp)
    { return operation(doc.cm, makeChangeSingleDoc)(doc, change, selAfter, spans) }

  if (change.to.line < doc.first) {
    shiftDoc(doc, change.text.length - 1 - (change.to.line - change.from.line))
    return
  }
  if (change.from.line > doc.lastLine()) { return }

  // Clip the change to the size of this doc
  if (change.from.line < doc.first) {
    var shift = change.text.length - 1 - (doc.first - change.from.line)
    shiftDoc(doc, shift)
    change = {from: Pos(doc.first, 0), to: Pos(change.to.line + shift, change.to.ch),
              text: [lst(change.text)], origin: change.origin}
  }
  var last = doc.lastLine()
  if (change.to.line > last) {
    change = {from: change.from, to: Pos(last, getLine(doc, last).text.length),
              text: [change.text[0]], origin: change.origin}
  }

  change.removed = getBetween(doc, change.from, change.to)

  if (!selAfter) { selAfter = computeSelAfterChange(doc, change) }
  if (doc.cm) { makeChangeSingleDocInEditor(doc.cm, change, spans) }
  else { updateDoc(doc, change, spans) }
  setSelectionNoUndo(doc, selAfter, sel_dontScroll)
}

// Handle the interaction of a change to a document with the editor
// that this document is part of.
function makeChangeSingleDocInEditor(cm, change, spans) {
  var doc = cm.doc, display = cm.display, from = change.from, to = change.to

  var recomputeMaxLength = false, checkWidthStart = from.line
  if (!cm.options.lineWrapping) {
    checkWidthStart = lineNo(visualLine(getLine(doc, from.line)))
    doc.iter(checkWidthStart, to.line + 1, function (line) {
      if (line == display.maxLine) {
        recomputeMaxLength = true
        return true
      }
    })
  }

  if (doc.sel.contains(change.from, change.to) > -1)
    { signalCursorActivity(cm) }

  updateDoc(doc, change, spans, estimateHeight(cm))

  if (!cm.options.lineWrapping) {
    doc.iter(checkWidthStart, from.line + change.text.length, function (line) {
      var len = lineLength(line)
      if (len > display.maxLineLength) {
        display.maxLine = line
        display.maxLineLength = len
        display.maxLineChanged = true
        recomputeMaxLength = false
      }
    })
    if (recomputeMaxLength) { cm.curOp.updateMaxLine = true }
  }

  // Adjust frontier, schedule worker
  doc.frontier = Math.min(doc.frontier, from.line)
  startWorker(cm, 400)

  var lendiff = change.text.length - (to.line - from.line) - 1
  // Remember that these lines changed, for updating the display
  if (change.full)
    { regChange(cm) }
  else if (from.line == to.line && change.text.length == 1 && !isWholeLineUpdate(cm.doc, change))
    { regLineChange(cm, from.line, "text") }
  else
    { regChange(cm, from.line, to.line + 1, lendiff) }

  var changesHandler = hasHandler(cm, "changes"), changeHandler = hasHandler(cm, "change")
  if (changeHandler || changesHandler) {
    var obj = {
      from: from, to: to,
      text: change.text,
      removed: change.removed,
      origin: change.origin
    }
    if (changeHandler) { signalLater(cm, "change", cm, obj) }
    if (changesHandler) { (cm.curOp.changeObjs || (cm.curOp.changeObjs = [])).push(obj) }
  }
  cm.display.selForContextMenu = null
}

function replaceRange(doc, code, from, to, origin) {
  if (!to) { to = from }
  if (cmp(to, from) < 0) { var tmp = to; to = from; from = tmp }
  if (typeof code == "string") { code = doc.splitLines(code) }
  makeChange(doc, {from: from, to: to, text: code, origin: origin})
}

// Rebasing/resetting history to deal with externally-sourced changes

function rebaseHistSelSingle(pos, from, to, diff) {
  if (to < pos.line) {
    pos.line += diff
  } else if (from < pos.line) {
    pos.line = from
    pos.ch = 0
  }
}

// Tries to rebase an array of history events given a change in the
// document. If the change touches the same lines as the event, the
// event, and everything 'behind' it, is discarded. If the change is
// before the event, the event's positions are updated. Uses a
// copy-on-write scheme for the positions, to avoid having to
// reallocate them all on every rebase, but also avoid problems with
// shared position objects being unsafely updated.
function rebaseHistArray(array, from, to, diff) {
  for (var i = 0; i < array.length; ++i) {
    var sub = array[i], ok = true
    if (sub.ranges) {
      if (!sub.copied) { sub = array[i] = sub.deepCopy(); sub.copied = true }
      for (var j = 0; j < sub.ranges.length; j++) {
        rebaseHistSelSingle(sub.ranges[j].anchor, from, to, diff)
        rebaseHistSelSingle(sub.ranges[j].head, from, to, diff)
      }
      continue
    }
    for (var j$1 = 0; j$1 < sub.changes.length; ++j$1) {
      var cur = sub.changes[j$1]
      if (to < cur.from.line) {
        cur.from = Pos(cur.from.line + diff, cur.from.ch)
        cur.to = Pos(cur.to.line + diff, cur.to.ch)
      } else if (from <= cur.to.line) {
        ok = false
        break
      }
    }
    if (!ok) {
      array.splice(0, i + 1)
      i = 0
    }
  }
}

function rebaseHist(hist, change) {
  var from = change.from.line, to = change.to.line, diff = change.text.length - (to - from) - 1
  rebaseHistArray(hist.done, from, to, diff)
  rebaseHistArray(hist.undone, from, to, diff)
}

// Utility for applying a change to a line by handle or number,
// returning the number and optionally registering the line as
// changed.
function changeLine(doc, handle, changeType, op) {
  var no = handle, line = handle
  if (typeof handle == "number") { line = getLine(doc, clipLine(doc, handle)) }
  else { no = lineNo(handle) }
  if (no == null) { return null }
  if (op(line, no) && doc.cm) { regLineChange(doc.cm, no, changeType) }
  return line
}

// The document is represented as a BTree consisting of leaves, with
// chunk of lines in them, and branches, with up to ten leaves or
// other branch nodes below them. The top node is always a branch
// node, and is the document object itself (meaning it has
// additional methods and properties).
//
// All nodes have parent links. The tree is used both to go from
// line numbers to line objects, and to go from objects to numbers.
// It also indexes by height, and is used to convert between height
// and line object, and to find the total height of the document.
//
// See also http://marijnhaverbeke.nl/blog/codemirror-line-tree.html

var LeafChunk = function(lines) {
  var this$1 = this;

  this.lines = lines
  this.parent = null
  var height = 0
  for (var i = 0; i < lines.length; ++i) {
    lines[i].parent = this$1
    height += lines[i].height
  }
  this.height = height
};

LeafChunk.prototype.chunkSize = function () { return this.lines.length };

// Remove the n lines at offset 'at'.
LeafChunk.prototype.removeInner = function (at, n) {
    var this$1 = this;

  for (var i = at, e = at + n; i < e; ++i) {
    var line = this$1.lines[i]
    this$1.height -= line.height
    cleanUpLine(line)
    signalLater(line, "delete")
  }
  this.lines.splice(at, n)
};

// Helper used to collapse a small branch into a single leaf.
LeafChunk.prototype.collapse = function (lines) {
  lines.push.apply(lines, this.lines)
};

// Insert the given array of lines at offset 'at', count them as
// having the given height.
LeafChunk.prototype.insertInner = function (at, lines, height) {
    var this$1 = this;

  this.height += height
  this.lines = this.lines.slice(0, at).concat(lines).concat(this.lines.slice(at))
  for (var i = 0; i < lines.length; ++i) { lines[i].parent = this$1 }
};

// Used to iterate over a part of the tree.
LeafChunk.prototype.iterN = function (at, n, op) {
    var this$1 = this;

  for (var e = at + n; at < e; ++at)
    { if (op(this$1.lines[at])) { return true } }
};

var BranchChunk = function(children) {
  var this$1 = this;

  this.children = children
  var size = 0, height = 0
  for (var i = 0; i < children.length; ++i) {
    var ch = children[i]
    size += ch.chunkSize(); height += ch.height
    ch.parent = this$1
  }
  this.size = size
  this.height = height
  this.parent = null
};

BranchChunk.prototype.chunkSize = function () { return this.size };

BranchChunk.prototype.removeInner = function (at, n) {
    var this$1 = this;

  this.size -= n
  for (var i = 0; i < this.children.length; ++i) {
    var child = this$1.children[i], sz = child.chunkSize()
    if (at < sz) {
      var rm = Math.min(n, sz - at), oldHeight = child.height
      child.removeInner(at, rm)
      this$1.height -= oldHeight - child.height
      if (sz == rm) { this$1.children.splice(i--, 1); child.parent = null }
      if ((n -= rm) == 0) { break }
      at = 0
    } else { at -= sz }
  }
  // If the result is smaller than 25 lines, ensure that it is a
  // single leaf node.
  if (this.size - n < 25 &&
      (this.children.length > 1 || !(this.children[0] instanceof LeafChunk))) {
    var lines = []
    this.collapse(lines)
    this.children = [new LeafChunk(lines)]
    this.children[0].parent = this
  }
};

BranchChunk.prototype.collapse = function (lines) {
    var this$1 = this;

  for (var i = 0; i < this.children.length; ++i) { this$1.children[i].collapse(lines) }
};

BranchChunk.prototype.insertInner = function (at, lines, height) {
    var this$1 = this;

  this.size += lines.length
  this.height += height
  for (var i = 0; i < this.children.length; ++i) {
    var child = this$1.children[i], sz = child.chunkSize()
    if (at <= sz) {
      child.insertInner(at, lines, height)
      if (child.lines && child.lines.length > 50) {
        // To avoid memory thrashing when child.lines is huge (e.g. first view of a large file), it's never spliced.
        // Instead, small slices are taken. They're taken in order because sequential memory accesses are fastest.
        var remaining = child.lines.length % 25 + 25
        for (var pos = remaining; pos < child.lines.length;) {
          var leaf = new LeafChunk(child.lines.slice(pos, pos += 25))
          child.height -= leaf.height
          this$1.children.splice(++i, 0, leaf)
          leaf.parent = this$1
        }
        child.lines = child.lines.slice(0, remaining)
        this$1.maybeSpill()
      }
      break
    }
    at -= sz
  }
};

// When a node has grown, check whether it should be split.
BranchChunk.prototype.maybeSpill = function () {
  if (this.children.length <= 10) { return }
  var me = this
  do {
    var spilled = me.children.splice(me.children.length - 5, 5)
    var sibling = new BranchChunk(spilled)
    if (!me.parent) { // Become the parent node
      var copy = new BranchChunk(me.children)
      copy.parent = me
      me.children = [copy, sibling]
      me = copy
   } else {
      me.size -= sibling.size
      me.height -= sibling.height
      var myIndex = indexOf(me.parent.children, me)
      me.parent.children.splice(myIndex + 1, 0, sibling)
    }
    sibling.parent = me.parent
  } while (me.children.length > 10)
  me.parent.maybeSpill()
};

BranchChunk.prototype.iterN = function (at, n, op) {
    var this$1 = this;

  for (var i = 0; i < this.children.length; ++i) {
    var child = this$1.children[i], sz = child.chunkSize()
    if (at < sz) {
      var used = Math.min(n, sz - at)
      if (child.iterN(at, used, op)) { return true }
      if ((n -= used) == 0) { break }
      at = 0
    } else { at -= sz }
  }
};

// Line widgets are block elements displayed above or below a line.

var LineWidget = function(doc, node, options) {
  var this$1 = this;

  if (options) { for (var opt in options) { if (options.hasOwnProperty(opt))
    { this$1[opt] = options[opt] } } }
  this.doc = doc
  this.node = node
};

LineWidget.prototype.clear = function () {
    var this$1 = this;

  var cm = this.doc.cm, ws = this.line.widgets, line = this.line, no = lineNo(line)
  if (no == null || !ws) { return }
  for (var i = 0; i < ws.length; ++i) { if (ws[i] == this$1) { ws.splice(i--, 1) } }
  if (!ws.length) { line.widgets = null }
  var height = widgetHeight(this)
  updateLineHeight(line, Math.max(0, line.height - height))
  if (cm) {
    runInOp(cm, function () {
      adjustScrollWhenAboveVisible(cm, line, -height)
      regLineChange(cm, no, "widget")
    })
    signalLater(cm, "lineWidgetCleared", cm, this)
  }
};

LineWidget.prototype.changed = function () {
  var oldH = this.height, cm = this.doc.cm, line = this.line
  this.height = null
  var diff = widgetHeight(this) - oldH
  if (!diff) { return }
  updateLineHeight(line, line.height + diff)
  if (cm) { runInOp(cm, function () {
    cm.curOp.forceUpdate = true
    adjustScrollWhenAboveVisible(cm, line, diff)
  }) }
};
eventMixin(LineWidget)

function adjustScrollWhenAboveVisible(cm, line, diff) {
  if (heightAtLine(line) < ((cm.curOp && cm.curOp.scrollTop) || cm.doc.scrollTop))
    { addToScrollPos(cm, null, diff) }
}

function addLineWidget(doc, handle, node, options) {
  var widget = new LineWidget(doc, node, options)
  var cm = doc.cm
  if (cm && widget.noHScroll) { cm.display.alignWidgets = true }
  changeLine(doc, handle, "widget", function (line) {
    var widgets = line.widgets || (line.widgets = [])
    if (widget.insertAt == null) { widgets.push(widget) }
    else { widgets.splice(Math.min(widgets.length - 1, Math.max(0, widget.insertAt)), 0, widget) }
    widget.line = line
    if (cm && !lineIsHidden(doc, line)) {
      var aboveVisible = heightAtLine(line) < doc.scrollTop
      updateLineHeight(line, line.height + widgetHeight(widget))
      if (aboveVisible) { addToScrollPos(cm, null, widget.height) }
      cm.curOp.forceUpdate = true
    }
    return true
  })
  signalLater(cm, "lineWidgetAdded", cm, widget)
  return widget
}

// TEXTMARKERS

// Created with markText and setBookmark methods. A TextMarker is a
// handle that can be used to clear or find a marked position in the
// document. Line objects hold arrays (markedSpans) containing
// {from, to, marker} object pointing to such marker objects, and
// indicating that such a marker is present on that line. Multiple
// lines may point to the same marker when it spans across lines.
// The spans will have null for their from/to properties when the
// marker continues beyond the start/end of the line. Markers have
// links back to the lines they currently touch.

// Collapsed markers have unique ids, in order to be able to order
// them, which is needed for uniquely determining an outer marker
// when they overlap (they may nest, but not partially overlap).
var nextMarkerId = 0

var TextMarker = function(doc, type) {
  this.lines = []
  this.type = type
  this.doc = doc
  this.id = ++nextMarkerId
};

// Clear the marker.
TextMarker.prototype.clear = function () {
    var this$1 = this;

  if (this.explicitlyCleared) { return }
  var cm = this.doc.cm, withOp = cm && !cm.curOp
  if (withOp) { startOperation(cm) }
  if (hasHandler(this, "clear")) {
    var found = this.find()
    if (found) { signalLater(this, "clear", found.from, found.to) }
  }
  var min = null, max = null
  for (var i = 0; i < this.lines.length; ++i) {
    var line = this$1.lines[i]
    var span = getMarkedSpanFor(line.markedSpans, this$1)
    if (cm && !this$1.collapsed) { regLineChange(cm, lineNo(line), "text") }
    else if (cm) {
      if (span.to != null) { max = lineNo(line) }
      if (span.from != null) { min = lineNo(line) }
    }
    line.markedSpans = removeMarkedSpan(line.markedSpans, span)
    if (span.from == null && this$1.collapsed && !lineIsHidden(this$1.doc, line) && cm)
      { updateLineHeight(line, textHeight(cm.display)) }
  }
  if (cm && this.collapsed && !cm.options.lineWrapping) { for (var i$1 = 0; i$1 < this.lines.length; ++i$1) {
    var visual = visualLine(this$1.lines[i$1]), len = lineLength(visual)
    if (len > cm.display.maxLineLength) {
      cm.display.maxLine = visual
      cm.display.maxLineLength = len
      cm.display.maxLineChanged = true
    }
  } }

  if (min != null && cm && this.collapsed) { regChange(cm, min, max + 1) }
  this.lines.length = 0
  this.explicitlyCleared = true
  if (this.atomic && this.doc.cantEdit) {
    this.doc.cantEdit = false
    if (cm) { reCheckSelection(cm.doc) }
  }
  if (cm) { signalLater(cm, "markerCleared", cm, this) }
  if (withOp) { endOperation(cm) }
  if (this.parent) { this.parent.clear() }
};

// Find the position of the marker in the document. Returns a {from,
// to} object by default. Side can be passed to get a specific side
// -- 0 (both), -1 (left), or 1 (right). When lineObj is true, the
// Pos objects returned contain a line object, rather than a line
// number (used to prevent looking up the same line twice).
TextMarker.prototype.find = function (side, lineObj) {
    var this$1 = this;

  if (side == null && this.type == "bookmark") { side = 1 }
  var from, to
  for (var i = 0; i < this.lines.length; ++i) {
    var line = this$1.lines[i]
    var span = getMarkedSpanFor(line.markedSpans, this$1)
    if (span.from != null) {
      from = Pos(lineObj ? line : lineNo(line), span.from)
      if (side == -1) { return from }
    }
    if (span.to != null) {
      to = Pos(lineObj ? line : lineNo(line), span.to)
      if (side == 1) { return to }
    }
  }
  return from && {from: from, to: to}
};

// Signals that the marker's widget changed, and surrounding layout
// should be recomputed.
TextMarker.prototype.changed = function () {
  var pos = this.find(-1, true), widget = this, cm = this.doc.cm
  if (!pos || !cm) { return }
  runInOp(cm, function () {
    var line = pos.line, lineN = lineNo(pos.line)
    var view = findViewForLine(cm, lineN)
    if (view) {
      clearLineMeasurementCacheFor(view)
      cm.curOp.selectionChanged = cm.curOp.forceUpdate = true
    }
    cm.curOp.updateMaxLine = true
    if (!lineIsHidden(widget.doc, line) && widget.height != null) {
      var oldHeight = widget.height
      widget.height = null
      var dHeight = widgetHeight(widget) - oldHeight
      if (dHeight)
        { updateLineHeight(line, line.height + dHeight) }
    }
  })
};

TextMarker.prototype.attachLine = function (line) {
  if (!this.lines.length && this.doc.cm) {
    var op = this.doc.cm.curOp
    if (!op.maybeHiddenMarkers || indexOf(op.maybeHiddenMarkers, this) == -1)
      { (op.maybeUnhiddenMarkers || (op.maybeUnhiddenMarkers = [])).push(this) }
  }
  this.lines.push(line)
};

TextMarker.prototype.detachLine = function (line) {
  this.lines.splice(indexOf(this.lines, line), 1)
  if (!this.lines.length && this.doc.cm) {
    var op = this.doc.cm.curOp
    ;(op.maybeHiddenMarkers || (op.maybeHiddenMarkers = [])).push(this)
  }
};
eventMixin(TextMarker)

// Create a marker, wire it up to the right lines, and
function markText(doc, from, to, options, type) {
  // Shared markers (across linked documents) are handled separately
  // (markTextShared will call out to this again, once per
  // document).
  if (options && options.shared) { return markTextShared(doc, from, to, options, type) }
  // Ensure we are in an operation.
  if (doc.cm && !doc.cm.curOp) { return operation(doc.cm, markText)(doc, from, to, options, type) }

  var marker = new TextMarker(doc, type), diff = cmp(from, to)
  if (options) { copyObj(options, marker, false) }
  // Don't connect empty markers unless clearWhenEmpty is false
  if (diff > 0 || diff == 0 && marker.clearWhenEmpty !== false)
    { return marker }
  if (marker.replacedWith) {
    // Showing up as a widget implies collapsed (widget replaces text)
    marker.collapsed = true
    marker.widgetNode = elt("span", [marker.replacedWith], "CodeMirror-widget")
    marker.widgetNode.setAttribute("role", "presentation") // hide from accessibility tree
    if (!options.handleMouseEvents) { marker.widgetNode.setAttribute("cm-ignore-events", "true") }
    if (options.insertLeft) { marker.widgetNode.insertLeft = true }
  }
  if (marker.collapsed) {
    if (conflictingCollapsedRange(doc, from.line, from, to, marker) ||
        from.line != to.line && conflictingCollapsedRange(doc, to.line, from, to, marker))
      { throw new Error("Inserting collapsed marker partially overlapping an existing one") }
    seeCollapsedSpans()
  }

  if (marker.addToHistory)
    { addChangeToHistory(doc, {from: from, to: to, origin: "markText"}, doc.sel, NaN) }

  var curLine = from.line, cm = doc.cm, updateMaxLine
  doc.iter(curLine, to.line + 1, function (line) {
    if (cm && marker.collapsed && !cm.options.lineWrapping && visualLine(line) == cm.display.maxLine)
      { updateMaxLine = true }
    if (marker.collapsed && curLine != from.line) { updateLineHeight(line, 0) }
    addMarkedSpan(line, new MarkedSpan(marker,
                                       curLine == from.line ? from.ch : null,
                                       curLine == to.line ? to.ch : null))
    ++curLine
  })
  // lineIsHidden depends on the presence of the spans, so needs a second pass
  if (marker.collapsed) { doc.iter(from.line, to.line + 1, function (line) {
    if (lineIsHidden(doc, line)) { updateLineHeight(line, 0) }
  }) }

  if (marker.clearOnEnter) { on(marker, "beforeCursorEnter", function () { return marker.clear(); }) }

  if (marker.readOnly) {
    seeReadOnlySpans()
    if (doc.history.done.length || doc.history.undone.length)
      { doc.clearHistory() }
  }
  if (marker.collapsed) {
    marker.id = ++nextMarkerId
    marker.atomic = true
  }
  if (cm) {
    // Sync editor state
    if (updateMaxLine) { cm.curOp.updateMaxLine = true }
    if (marker.collapsed)
      { regChange(cm, from.line, to.line + 1) }
    else if (marker.className || marker.title || marker.startStyle || marker.endStyle || marker.css)
      { for (var i = from.line; i <= to.line; i++) { regLineChange(cm, i, "text") } }
    if (marker.atomic) { reCheckSelection(cm.doc) }
    signalLater(cm, "markerAdded", cm, marker)
  }
  return marker
}

// SHARED TEXTMARKERS

// A shared marker spans multiple linked documents. It is
// implemented as a meta-marker-object controlling multiple normal
// markers.
var SharedTextMarker = function(markers, primary) {
  var this$1 = this;

  this.markers = markers
  this.primary = primary
  for (var i = 0; i < markers.length; ++i)
    { markers[i].parent = this$1 }
};

SharedTextMarker.prototype.clear = function () {
    var this$1 = this;

  if (this.explicitlyCleared) { return }
  this.explicitlyCleared = true
  for (var i = 0; i < this.markers.length; ++i)
    { this$1.markers[i].clear() }
  signalLater(this, "clear")
};

SharedTextMarker.prototype.find = function (side, lineObj) {
  return this.primary.find(side, lineObj)
};
eventMixin(SharedTextMarker)

function markTextShared(doc, from, to, options, type) {
  options = copyObj(options)
  options.shared = false
  var markers = [markText(doc, from, to, options, type)], primary = markers[0]
  var widget = options.widgetNode
  linkedDocs(doc, function (doc) {
    if (widget) { options.widgetNode = widget.cloneNode(true) }
    markers.push(markText(doc, clipPos(doc, from), clipPos(doc, to), options, type))
    for (var i = 0; i < doc.linked.length; ++i)
      { if (doc.linked[i].isParent) { return } }
    primary = lst(markers)
  })
  return new SharedTextMarker(markers, primary)
}

function findSharedMarkers(doc) {
  return doc.findMarks(Pos(doc.first, 0), doc.clipPos(Pos(doc.lastLine())), function (m) { return m.parent; })
}

function copySharedMarkers(doc, markers) {
  for (var i = 0; i < markers.length; i++) {
    var marker = markers[i], pos = marker.find()
    var mFrom = doc.clipPos(pos.from), mTo = doc.clipPos(pos.to)
    if (cmp(mFrom, mTo)) {
      var subMark = markText(doc, mFrom, mTo, marker.primary, marker.primary.type)
      marker.markers.push(subMark)
      subMark.parent = marker
    }
  }
}

function detachSharedMarkers(markers) {
  var loop = function ( i ) {
    var marker = markers[i], linked = [marker.primary.doc]
    linkedDocs(marker.primary.doc, function (d) { return linked.push(d); })
    for (var j = 0; j < marker.markers.length; j++) {
      var subMarker = marker.markers[j]
      if (indexOf(linked, subMarker.doc) == -1) {
        subMarker.parent = null
        marker.markers.splice(j--, 1)
      }
    }
  };

  for (var i = 0; i < markers.length; i++) loop( i );
}

var nextDocId = 0
var Doc = function(text, mode, firstLine, lineSep) {
  if (!(this instanceof Doc)) { return new Doc(text, mode, firstLine, lineSep) }
  if (firstLine == null) { firstLine = 0 }

  BranchChunk.call(this, [new LeafChunk([new Line("", null)])])
  this.first = firstLine
  this.scrollTop = this.scrollLeft = 0
  this.cantEdit = false
  this.cleanGeneration = 1
  this.frontier = firstLine
  var start = Pos(firstLine, 0)
  this.sel = simpleSelection(start)
  this.history = new History(null)
  this.id = ++nextDocId
  this.modeOption = mode
  this.lineSep = lineSep
  this.extend = false

  if (typeof text == "string") { text = this.splitLines(text) }
  updateDoc(this, {from: start, to: start, text: text})
  setSelection(this, simpleSelection(start), sel_dontScroll)
}

Doc.prototype = createObj(BranchChunk.prototype, {
  constructor: Doc,
  // Iterate over the document. Supports two forms -- with only one
  // argument, it calls that for each line in the document. With
  // three, it iterates over the range given by the first two (with
  // the second being non-inclusive).
  iter: function(from, to, op) {
    if (op) { this.iterN(from - this.first, to - from, op) }
    else { this.iterN(this.first, this.first + this.size, from) }
  },

  // Non-public interface for adding and removing lines.
  insert: function(at, lines) {
    var height = 0
    for (var i = 0; i < lines.length; ++i) { height += lines[i].height }
    this.insertInner(at - this.first, lines, height)
  },
  remove: function(at, n) { this.removeInner(at - this.first, n) },

  // From here, the methods are part of the public interface. Most
  // are also available from CodeMirror (editor) instances.

  getValue: function(lineSep) {
    var lines = getLines(this, this.first, this.first + this.size)
    if (lineSep === false) { return lines }
    return lines.join(lineSep || this.lineSeparator())
  },
  setValue: docMethodOp(function(code) {
    var top = Pos(this.first, 0), last = this.first + this.size - 1
    makeChange(this, {from: top, to: Pos(last, getLine(this, last).text.length),
                      text: this.splitLines(code), origin: "setValue", full: true}, true)
    setSelection(this, simpleSelection(top))
  }),
  replaceRange: function(code, from, to, origin) {
    from = clipPos(this, from)
    to = to ? clipPos(this, to) : from
    replaceRange(this, code, from, to, origin)
  },
  getRange: function(from, to, lineSep) {
    var lines = getBetween(this, clipPos(this, from), clipPos(this, to))
    if (lineSep === false) { return lines }
    return lines.join(lineSep || this.lineSeparator())
  },

  getLine: function(line) {var l = this.getLineHandle(line); return l && l.text},

  getLineHandle: function(line) {if (isLine(this, line)) { return getLine(this, line) }},
  getLineNumber: function(line) {return lineNo(line)},

  getLineHandleVisualStart: function(line) {
    if (typeof line == "number") { line = getLine(this, line) }
    return visualLine(line)
  },

  lineCount: function() {return this.size},
  firstLine: function() {return this.first},
  lastLine: function() {return this.first + this.size - 1},

  clipPos: function(pos) {return clipPos(this, pos)},

  getCursor: function(start) {
    var range = this.sel.primary(), pos
    if (start == null || start == "head") { pos = range.head }
    else if (start == "anchor") { pos = range.anchor }
    else if (start == "end" || start == "to" || start === false) { pos = range.to() }
    else { pos = range.from() }
    return pos
  },
  listSelections: function() { return this.sel.ranges },
  somethingSelected: function() {return this.sel.somethingSelected()},

  setCursor: docMethodOp(function(line, ch, options) {
    setSimpleSelection(this, clipPos(this, typeof line == "number" ? Pos(line, ch || 0) : line), null, options)
  }),
  setSelection: docMethodOp(function(anchor, head, options) {
    setSimpleSelection(this, clipPos(this, anchor), clipPos(this, head || anchor), options)
  }),
  extendSelection: docMethodOp(function(head, other, options) {
    extendSelection(this, clipPos(this, head), other && clipPos(this, other), options)
  }),
  extendSelections: docMethodOp(function(heads, options) {
    extendSelections(this, clipPosArray(this, heads), options)
  }),
  extendSelectionsBy: docMethodOp(function(f, options) {
    var heads = map(this.sel.ranges, f)
    extendSelections(this, clipPosArray(this, heads), options)
  }),
  setSelections: docMethodOp(function(ranges, primary, options) {
    var this$1 = this;

    if (!ranges.length) { return }
    var out = []
    for (var i = 0; i < ranges.length; i++)
      { out[i] = new Range(clipPos(this$1, ranges[i].anchor),
                         clipPos(this$1, ranges[i].head)) }
    if (primary == null) { primary = Math.min(ranges.length - 1, this.sel.primIndex) }
    setSelection(this, normalizeSelection(out, primary), options)
  }),
  addSelection: docMethodOp(function(anchor, head, options) {
    var ranges = this.sel.ranges.slice(0)
    ranges.push(new Range(clipPos(this, anchor), clipPos(this, head || anchor)))
    setSelection(this, normalizeSelection(ranges, ranges.length - 1), options)
  }),

  getSelection: function(lineSep) {
    var this$1 = this;

    var ranges = this.sel.ranges, lines
    for (var i = 0; i < ranges.length; i++) {
      var sel = getBetween(this$1, ranges[i].from(), ranges[i].to())
      lines = lines ? lines.concat(sel) : sel
    }
    if (lineSep === false) { return lines }
    else { return lines.join(lineSep || this.lineSeparator()) }
  },
  getSelections: function(lineSep) {
    var this$1 = this;

    var parts = [], ranges = this.sel.ranges
    for (var i = 0; i < ranges.length; i++) {
      var sel = getBetween(this$1, ranges[i].from(), ranges[i].to())
      if (lineSep !== false) { sel = sel.join(lineSep || this$1.lineSeparator()) }
      parts[i] = sel
    }
    return parts
  },
  replaceSelection: function(code, collapse, origin) {
    var dup = []
    for (var i = 0; i < this.sel.ranges.length; i++)
      { dup[i] = code }
    this.replaceSelections(dup, collapse, origin || "+input")
  },
  replaceSelections: docMethodOp(function(code, collapse, origin) {
    var this$1 = this;

    var changes = [], sel = this.sel
    for (var i = 0; i < sel.ranges.length; i++) {
      var range = sel.ranges[i]
      changes[i] = {from: range.from(), to: range.to(), text: this$1.splitLines(code[i]), origin: origin}
    }
    var newSel = collapse && collapse != "end" && computeReplacedSel(this, changes, collapse)
    for (var i$1 = changes.length - 1; i$1 >= 0; i$1--)
      { makeChange(this$1, changes[i$1]) }
    if (newSel) { setSelectionReplaceHistory(this, newSel) }
    else if (this.cm) { ensureCursorVisible(this.cm) }
  }),
  undo: docMethodOp(function() {makeChangeFromHistory(this, "undo")}),
  redo: docMethodOp(function() {makeChangeFromHistory(this, "redo")}),
  undoSelection: docMethodOp(function() {makeChangeFromHistory(this, "undo", true)}),
  redoSelection: docMethodOp(function() {makeChangeFromHistory(this, "redo", true)}),

  setExtending: function(val) {this.extend = val},
  getExtending: function() {return this.extend},

  historySize: function() {
    var hist = this.history, done = 0, undone = 0
    for (var i = 0; i < hist.done.length; i++) { if (!hist.done[i].ranges) { ++done } }
    for (var i$1 = 0; i$1 < hist.undone.length; i$1++) { if (!hist.undone[i$1].ranges) { ++undone } }
    return {undo: done, redo: undone}
  },
  clearHistory: function() {this.history = new History(this.history.maxGeneration)},

  markClean: function() {
    this.cleanGeneration = this.changeGeneration(true)
  },
  changeGeneration: function(forceSplit) {
    if (forceSplit)
      { this.history.lastOp = this.history.lastSelOp = this.history.lastOrigin = null }
    return this.history.generation
  },
  isClean: function (gen) {
    return this.history.generation == (gen || this.cleanGeneration)
  },

  getHistory: function() {
    return {done: copyHistoryArray(this.history.done),
            undone: copyHistoryArray(this.history.undone)}
  },
  setHistory: function(histData) {
    var hist = this.history = new History(this.history.maxGeneration)
    hist.done = copyHistoryArray(histData.done.slice(0), null, true)
    hist.undone = copyHistoryArray(histData.undone.slice(0), null, true)
  },

  setGutterMarker: docMethodOp(function(line, gutterID, value) {
    return changeLine(this, line, "gutter", function (line) {
      var markers = line.gutterMarkers || (line.gutterMarkers = {})
      markers[gutterID] = value
      if (!value && isEmpty(markers)) { line.gutterMarkers = null }
      return true
    })
  }),

  clearGutter: docMethodOp(function(gutterID) {
    var this$1 = this;

    this.iter(function (line) {
      if (line.gutterMarkers && line.gutterMarkers[gutterID]) {
        changeLine(this$1, line, "gutter", function () {
          line.gutterMarkers[gutterID] = null
          if (isEmpty(line.gutterMarkers)) { line.gutterMarkers = null }
          return true
        })
      }
    })
  }),

  lineInfo: function(line) {
    var n
    if (typeof line == "number") {
      if (!isLine(this, line)) { return null }
      n = line
      line = getLine(this, line)
      if (!line) { return null }
    } else {
      n = lineNo(line)
      if (n == null) { return null }
    }
    return {line: n, handle: line, text: line.text, gutterMarkers: line.gutterMarkers,
            textClass: line.textClass, bgClass: line.bgClass, wrapClass: line.wrapClass,
            widgets: line.widgets}
  },

  addLineClass: docMethodOp(function(handle, where, cls) {
    return changeLine(this, handle, where == "gutter" ? "gutter" : "class", function (line) {
      var prop = where == "text" ? "textClass"
               : where == "background" ? "bgClass"
               : where == "gutter" ? "gutterClass" : "wrapClass"
      if (!line[prop]) { line[prop] = cls }
      else if (classTest(cls).test(line[prop])) { return false }
      else { line[prop] += " " + cls }
      return true
    })
  }),
  removeLineClass: docMethodOp(function(handle, where, cls) {
    return changeLine(this, handle, where == "gutter" ? "gutter" : "class", function (line) {
      var prop = where == "text" ? "textClass"
               : where == "background" ? "bgClass"
               : where == "gutter" ? "gutterClass" : "wrapClass"
      var cur = line[prop]
      if (!cur) { return false }
      else if (cls == null) { line[prop] = null }
      else {
        var found = cur.match(classTest(cls))
        if (!found) { return false }
        var end = found.index + found[0].length
        line[prop] = cur.slice(0, found.index) + (!found.index || end == cur.length ? "" : " ") + cur.slice(end) || null
      }
      return true
    })
  }),

  addLineWidget: docMethodOp(function(handle, node, options) {
    return addLineWidget(this, handle, node, options)
  }),
  removeLineWidget: function(widget) { widget.clear() },

  markText: function(from, to, options) {
    return markText(this, clipPos(this, from), clipPos(this, to), options, options && options.type || "range")
  },
  setBookmark: function(pos, options) {
    var realOpts = {replacedWith: options && (options.nodeType == null ? options.widget : options),
                    insertLeft: options && options.insertLeft,
                    clearWhenEmpty: false, shared: options && options.shared,
                    handleMouseEvents: options && options.handleMouseEvents}
    pos = clipPos(this, pos)
    return markText(this, pos, pos, realOpts, "bookmark")
  },
  findMarksAt: function(pos) {
    pos = clipPos(this, pos)
    var markers = [], spans = getLine(this, pos.line).markedSpans
    if (spans) { for (var i = 0; i < spans.length; ++i) {
      var span = spans[i]
      if ((span.from == null || span.from <= pos.ch) &&
          (span.to == null || span.to >= pos.ch))
        { markers.push(span.marker.parent || span.marker) }
    } }
    return markers
  },
  findMarks: function(from, to, filter) {
    from = clipPos(this, from); to = clipPos(this, to)
    var found = [], lineNo = from.line
    this.iter(from.line, to.line + 1, function (line) {
      var spans = line.markedSpans
      if (spans) { for (var i = 0; i < spans.length; i++) {
        var span = spans[i]
        if (!(span.to != null && lineNo == from.line && from.ch >= span.to ||
              span.from == null && lineNo != from.line ||
              span.from != null && lineNo == to.line && span.from >= to.ch) &&
            (!filter || filter(span.marker)))
          { found.push(span.marker.parent || span.marker) }
      } }
      ++lineNo
    })
    return found
  },
  getAllMarks: function() {
    var markers = []
    this.iter(function (line) {
      var sps = line.markedSpans
      if (sps) { for (var i = 0; i < sps.length; ++i)
        { if (sps[i].from != null) { markers.push(sps[i].marker) } } }
    })
    return markers
  },

  posFromIndex: function(off) {
    var ch, lineNo = this.first, sepSize = this.lineSeparator().length
    this.iter(function (line) {
      var sz = line.text.length + sepSize
      if (sz > off) { ch = off; return true }
      off -= sz
      ++lineNo
    })
    return clipPos(this, Pos(lineNo, ch))
  },
  indexFromPos: function (coords) {
    coords = clipPos(this, coords)
    var index = coords.ch
    if (coords.line < this.first || coords.ch < 0) { return 0 }
    var sepSize = this.lineSeparator().length
    this.iter(this.first, coords.line, function (line) { // iter aborts when callback returns a truthy value
      index += line.text.length + sepSize
    })
    return index
  },

  copy: function(copyHistory) {
    var doc = new Doc(getLines(this, this.first, this.first + this.size),
                      this.modeOption, this.first, this.lineSep)
    doc.scrollTop = this.scrollTop; doc.scrollLeft = this.scrollLeft
    doc.sel = this.sel
    doc.extend = false
    if (copyHistory) {
      doc.history.undoDepth = this.history.undoDepth
      doc.setHistory(this.getHistory())
    }
    return doc
  },

  linkedDoc: function(options) {
    if (!options) { options = {} }
    var from = this.first, to = this.first + this.size
    if (options.from != null && options.from > from) { from = options.from }
    if (options.to != null && options.to < to) { to = options.to }
    var copy = new Doc(getLines(this, from, to), options.mode || this.modeOption, from, this.lineSep)
    if (options.sharedHist) { copy.history = this.history
    ; }(this.linked || (this.linked = [])).push({doc: copy, sharedHist: options.sharedHist})
    copy.linked = [{doc: this, isParent: true, sharedHist: options.sharedHist}]
    copySharedMarkers(copy, findSharedMarkers(this))
    return copy
  },
  unlinkDoc: function(other) {
    var this$1 = this;

    if (other instanceof CodeMirror) { other = other.doc }
    if (this.linked) { for (var i = 0; i < this.linked.length; ++i) {
      var link = this$1.linked[i]
      if (link.doc != other) { continue }
      this$1.linked.splice(i, 1)
      other.unlinkDoc(this$1)
      detachSharedMarkers(findSharedMarkers(this$1))
      break
    } }
    // If the histories were shared, split them again
    if (other.history == this.history) {
      var splitIds = [other.id]
      linkedDocs(other, function (doc) { return splitIds.push(doc.id); }, true)
      other.history = new History(null)
      other.history.done = copyHistoryArray(this.history.done, splitIds)
      other.history.undone = copyHistoryArray(this.history.undone, splitIds)
    }
  },
  iterLinkedDocs: function(f) {linkedDocs(this, f)},

  getMode: function() {return this.mode},
  getEditor: function() {return this.cm},

  splitLines: function(str) {
    if (this.lineSep) { return str.split(this.lineSep) }
    return splitLinesAuto(str)
  },
  lineSeparator: function() { return this.lineSep || "\n" }
})

// Public alias.
Doc.prototype.eachLine = Doc.prototype.iter

// Kludge to work around strange IE behavior where it'll sometimes
// re-fire a series of drag-related events right after the drop (#1551)
var lastDrop = 0

function onDrop(e) {
  var cm = this
  clearDragCursor(cm)
  if (signalDOMEvent(cm, e) || eventInWidget(cm.display, e))
    { return }
  e_preventDefault(e)
  if (ie) { lastDrop = +new Date }
  var pos = posFromMouse(cm, e, true), files = e.dataTransfer.files
  if (!pos || cm.isReadOnly()) { return }
  // Might be a file drop, in which case we simply extract the text
  // and insert it.
  if (files && files.length && window.FileReader && window.File) {
    var n = files.length, text = Array(n), read = 0
    var loadFile = function (file, i) {
      if (cm.options.allowDropFileTypes &&
          indexOf(cm.options.allowDropFileTypes, file.type) == -1)
        { return }

      var reader = new FileReader
      reader.onload = operation(cm, function () {
        var content = reader.result
        if (/[\x00-\x08\x0e-\x1f]{2}/.test(content)) { content = "" }
        text[i] = content
        if (++read == n) {
          pos = clipPos(cm.doc, pos)
          var change = {from: pos, to: pos,
                        text: cm.doc.splitLines(text.join(cm.doc.lineSeparator())),
                        origin: "paste"}
          makeChange(cm.doc, change)
          setSelectionReplaceHistory(cm.doc, simpleSelection(pos, changeEnd(change)))
        }
      })
      reader.readAsText(file)
    }
    for (var i = 0; i < n; ++i) { loadFile(files[i], i) }
  } else { // Normal drop
    // Don't do a replace if the drop happened inside of the selected text.
    if (cm.state.draggingText && cm.doc.sel.contains(pos) > -1) {
      cm.state.draggingText(e)
      // Ensure the editor is re-focused
      setTimeout(function () { return cm.display.input.focus(); }, 20)
      return
    }
    try {
      var text$1 = e.dataTransfer.getData("Text")
      if (text$1) {
        var selected
        if (cm.state.draggingText && !cm.state.draggingText.copy)
          { selected = cm.listSelections() }
        setSelectionNoUndo(cm.doc, simpleSelection(pos, pos))
        if (selected) { for (var i$1 = 0; i$1 < selected.length; ++i$1)
          { replaceRange(cm.doc, "", selected[i$1].anchor, selected[i$1].head, "drag") } }
        cm.replaceSelection(text$1, "around", "paste")
        cm.display.input.focus()
      }
    }
    catch(e){}
  }
}

function onDragStart(cm, e) {
  if (ie && (!cm.state.draggingText || +new Date - lastDrop < 100)) { e_stop(e); return }
  if (signalDOMEvent(cm, e) || eventInWidget(cm.display, e)) { return }

  e.dataTransfer.setData("Text", cm.getSelection())
  e.dataTransfer.effectAllowed = "copyMove"

  // Use dummy image instead of default browsers image.
  // Recent Safari (~6.0.2) have a tendency to segfault when this happens, so we don't do it there.
  if (e.dataTransfer.setDragImage && !safari) {
    var img = elt("img", null, null, "position: fixed; left: 0; top: 0;")
    img.src = "data:image/gif;base64,R0lGODlhAQABAAAAACH5BAEKAAEALAAAAAABAAEAAAICTAEAOw=="
    if (presto) {
      img.width = img.height = 1
      cm.display.wrapper.appendChild(img)
      // Force a relayout, or Opera won't use our image for some obscure reason
      img._top = img.offsetTop
    }
    e.dataTransfer.setDragImage(img, 0, 0)
    if (presto) { img.parentNode.removeChild(img) }
  }
}

function onDragOver(cm, e) {
  var pos = posFromMouse(cm, e)
  if (!pos) { return }
  var frag = document.createDocumentFragment()
  drawSelectionCursor(cm, pos, frag)
  if (!cm.display.dragCursor) {
    cm.display.dragCursor = elt("div", null, "CodeMirror-cursors CodeMirror-dragcursors")
    cm.display.lineSpace.insertBefore(cm.display.dragCursor, cm.display.cursorDiv)
  }
  removeChildrenAndAdd(cm.display.dragCursor, frag)
}

function clearDragCursor(cm) {
  if (cm.display.dragCursor) {
    cm.display.lineSpace.removeChild(cm.display.dragCursor)
    cm.display.dragCursor = null
  }
}

// These must be handled carefully, because naively registering a
// handler for each editor will cause the editors to never be
// garbage collected.

function forEachCodeMirror(f) {
  if (!document.body.getElementsByClassName) { return }
  var byClass = document.body.getElementsByClassName("CodeMirror")
  for (var i = 0; i < byClass.length; i++) {
    var cm = byClass[i].CodeMirror
    if (cm) { f(cm) }
  }
}

var globalsRegistered = false
function ensureGlobalHandlers() {
  if (globalsRegistered) { return }
  registerGlobalHandlers()
  globalsRegistered = true
}
function registerGlobalHandlers() {
  // When the window resizes, we need to refresh active editors.
  var resizeTimer
  on(window, "resize", function () {
    if (resizeTimer == null) { resizeTimer = setTimeout(function () {
      resizeTimer = null
      forEachCodeMirror(onResize)
    }, 100) }
  })
  // When the window loses focus, we want to show the editor as blurred
  on(window, "blur", function () { return forEachCodeMirror(onBlur); })
}
// Called when the window resizes
function onResize(cm) {
  var d = cm.display
  if (d.lastWrapHeight == d.wrapper.clientHeight && d.lastWrapWidth == d.wrapper.clientWidth)
    { return }
  // Might be a text scaling operation, clear size caches.
  d.cachedCharWidth = d.cachedTextHeight = d.cachedPaddingH = null
  d.scrollbarsClipped = false
  cm.setSize()
}

var keyNames = {
  3: "Enter", 8: "Backspace", 9: "Tab", 13: "Enter", 16: "Shift", 17: "Ctrl", 18: "Alt",
  19: "Pause", 20: "CapsLock", 27: "Esc", 32: "Space", 33: "PageUp", 34: "PageDown", 35: "End",
  36: "Home", 37: "Left", 38: "Up", 39: "Right", 40: "Down", 44: "PrintScrn", 45: "Insert",
  46: "Delete", 59: ";", 61: "=", 91: "Mod", 92: "Mod", 93: "Mod",
  106: "*", 107: "=", 109: "-", 110: ".", 111: "/", 127: "Delete",
  173: "-", 186: ";", 187: "=", 188: ",", 189: "-", 190: ".", 191: "/", 192: "`", 219: "[", 220: "\\",
  221: "]", 222: "'", 63232: "Up", 63233: "Down", 63234: "Left", 63235: "Right", 63272: "Delete",
  63273: "Home", 63275: "End", 63276: "PageUp", 63277: "PageDown", 63302: "Insert"
}

// Number keys
for (var i = 0; i < 10; i++) { keyNames[i + 48] = keyNames[i + 96] = String(i) }
// Alphabetic keys
for (var i$1 = 65; i$1 <= 90; i$1++) { keyNames[i$1] = String.fromCharCode(i$1) }
// Function keys
for (var i$2 = 1; i$2 <= 12; i$2++) { keyNames[i$2 + 111] = keyNames[i$2 + 63235] = "F" + i$2 }

var keyMap = {}

keyMap.basic = {
  "Left": "goCharLeft", "Right": "goCharRight", "Up": "goLineUp", "Down": "goLineDown",
  "End": "goLineEnd", "Home": "goLineStartSmart", "PageUp": "goPageUp", "PageDown": "goPageDown",
  "Delete": "delCharAfter", "Backspace": "delCharBefore", "Shift-Backspace": "delCharBefore",
  "Tab": "defaultTab", "Shift-Tab": "indentAuto",
  "Enter": "newlineAndIndent", "Insert": "toggleOverwrite",
  "Esc": "singleSelection"
}
// Note that the save and find-related commands aren't defined by
// default. User code or addons can define them. Unknown commands
// are simply ignored.
keyMap.pcDefault = {
  "Ctrl-A": "selectAll", "Ctrl-D": "deleteLine", "Ctrl-Z": "undo", "Shift-Ctrl-Z": "redo", "Ctrl-Y": "redo",
  "Ctrl-Home": "goDocStart", "Ctrl-End": "goDocEnd", "Ctrl-Up": "goLineUp", "Ctrl-Down": "goLineDown",
  "Ctrl-Left": "goGroupLeft", "Ctrl-Right": "goGroupRight", "Alt-Left": "goLineStart", "Alt-Right": "goLineEnd",
  "Ctrl-Backspace": "delGroupBefore", "Ctrl-Delete": "delGroupAfter", "Ctrl-S": "save", "Ctrl-F": "find",
  "Ctrl-G": "findNext", "Shift-Ctrl-G": "findPrev", "Shift-Ctrl-F": "replace", "Shift-Ctrl-R": "replaceAll",
  "Ctrl-[": "indentLess", "Ctrl-]": "indentMore",
  "Ctrl-U": "undoSelection", "Shift-Ctrl-U": "redoSelection", "Alt-U": "redoSelection",
  fallthrough: "basic"
}
// Very basic readline/emacs-style bindings, which are standard on Mac.
keyMap.emacsy = {
  "Ctrl-F": "goCharRight", "Ctrl-B": "goCharLeft", "Ctrl-P": "goLineUp", "Ctrl-N": "goLineDown",
  "Alt-F": "goWordRight", "Alt-B": "goWordLeft", "Ctrl-A": "goLineStart", "Ctrl-E": "goLineEnd",
  "Ctrl-V": "goPageDown", "Shift-Ctrl-V": "goPageUp", "Ctrl-D": "delCharAfter", "Ctrl-H": "delCharBefore",
  "Alt-D": "delWordAfter", "Alt-Backspace": "delWordBefore", "Ctrl-K": "killLine", "Ctrl-T": "transposeChars",
  "Ctrl-O": "openLine"
}
keyMap.macDefault = {
  "Cmd-A": "selectAll", "Cmd-D": "deleteLine", "Cmd-Z": "undo", "Shift-Cmd-Z": "redo", "Cmd-Y": "redo",
  "Cmd-Home": "goDocStart", "Cmd-Up": "goDocStart", "Cmd-End": "goDocEnd", "Cmd-Down": "goDocEnd", "Alt-Left": "goGroupLeft",
  "Alt-Right": "goGroupRight", "Cmd-Left": "goLineLeft", "Cmd-Right": "goLineRight", "Alt-Backspace": "delGroupBefore",
  "Ctrl-Alt-Backspace": "delGroupAfter", "Alt-Delete": "delGroupAfter", "Cmd-S": "save", "Cmd-F": "find",
  "Cmd-G": "findNext", "Shift-Cmd-G": "findPrev", "Cmd-Alt-F": "replace", "Shift-Cmd-Alt-F": "replaceAll",
  "Cmd-[": "indentLess", "Cmd-]": "indentMore", "Cmd-Backspace": "delWrappedLineLeft", "Cmd-Delete": "delWrappedLineRight",
  "Cmd-U": "undoSelection", "Shift-Cmd-U": "redoSelection", "Ctrl-Up": "goDocStart", "Ctrl-Down": "goDocEnd",
  fallthrough: ["basic", "emacsy"]
}
keyMap["default"] = mac ? keyMap.macDefault : keyMap.pcDefault

// KEYMAP DISPATCH

function normalizeKeyName(name) {
  var parts = name.split(/-(?!$)/)
  name = parts[parts.length - 1]
  var alt, ctrl, shift, cmd
  for (var i = 0; i < parts.length - 1; i++) {
    var mod = parts[i]
    if (/^(cmd|meta|m)$/i.test(mod)) { cmd = true }
    else if (/^a(lt)?$/i.test(mod)) { alt = true }
    else if (/^(c|ctrl|control)$/i.test(mod)) { ctrl = true }
    else if (/^s(hift)?$/i.test(mod)) { shift = true }
    else { throw new Error("Unrecognized modifier name: " + mod) }
  }
  if (alt) { name = "Alt-" + name }
  if (ctrl) { name = "Ctrl-" + name }
  if (cmd) { name = "Cmd-" + name }
  if (shift) { name = "Shift-" + name }
  return name
}

// This is a kludge to keep keymaps mostly working as raw objects
// (backwards compatibility) while at the same time support features
// like normalization and multi-stroke key bindings. It compiles a
// new normalized keymap, and then updates the old object to reflect
// this.
function normalizeKeyMap(keymap) {
  var copy = {}
  for (var keyname in keymap) { if (keymap.hasOwnProperty(keyname)) {
    var value = keymap[keyname]
    if (/^(name|fallthrough|(de|at)tach)$/.test(keyname)) { continue }
    if (value == "...") { delete keymap[keyname]; continue }

    var keys = map(keyname.split(" "), normalizeKeyName)
    for (var i = 0; i < keys.length; i++) {
      var val = (void 0), name = (void 0)
      if (i == keys.length - 1) {
        name = keys.join(" ")
        val = value
      } else {
        name = keys.slice(0, i + 1).join(" ")
        val = "..."
      }
      var prev = copy[name]
      if (!prev) { copy[name] = val }
      else if (prev != val) { throw new Error("Inconsistent bindings for " + name) }
    }
    delete keymap[keyname]
  } }
  for (var prop in copy) { keymap[prop] = copy[prop] }
  return keymap
}

function lookupKey(key, map, handle, context) {
  map = getKeyMap(map)
  var found = map.call ? map.call(key, context) : map[key]
  if (found === false) { return "nothing" }
  if (found === "...") { return "multi" }
  if (found != null && handle(found)) { return "handled" }

  if (map.fallthrough) {
    if (Object.prototype.toString.call(map.fallthrough) != "[object Array]")
      { return lookupKey(key, map.fallthrough, handle, context) }
    for (var i = 0; i < map.fallthrough.length; i++) {
      var result = lookupKey(key, map.fallthrough[i], handle, context)
      if (result) { return result }
    }
  }
}

// Modifier key presses don't count as 'real' key presses for the
// purpose of keymap fallthrough.
function isModifierKey(value) {
  var name = typeof value == "string" ? value : keyNames[value.keyCode]
  return name == "Ctrl" || name == "Alt" || name == "Shift" || name == "Mod"
}

// Look up the name of a key as indicated by an event object.
function keyName(event, noShift) {
  if (presto && event.keyCode == 34 && event["char"]) { return false }
  var base = keyNames[event.keyCode], name = base
  if (name == null || event.altGraphKey) { return false }
  if (event.altKey && base != "Alt") { name = "Alt-" + name }
  if ((flipCtrlCmd ? event.metaKey : event.ctrlKey) && base != "Ctrl") { name = "Ctrl-" + name }
  if ((flipCtrlCmd ? event.ctrlKey : event.metaKey) && base != "Cmd") { name = "Cmd-" + name }
  if (!noShift && event.shiftKey && base != "Shift") { name = "Shift-" + name }
  return name
}

function getKeyMap(val) {
  return typeof val == "string" ? keyMap[val] : val
}

// Helper for deleting text near the selection(s), used to implement
// backspace, delete, and similar functionality.
function deleteNearSelection(cm, compute) {
  var ranges = cm.doc.sel.ranges, kill = []
  // Build up a set of ranges to kill first, merging overlapping
  // ranges.
  for (var i = 0; i < ranges.length; i++) {
    var toKill = compute(ranges[i])
    while (kill.length && cmp(toKill.from, lst(kill).to) <= 0) {
      var replaced = kill.pop()
      if (cmp(replaced.from, toKill.from) < 0) {
        toKill.from = replaced.from
        break
      }
    }
    kill.push(toKill)
  }
  // Next, remove those actual ranges.
  runInOp(cm, function () {
    for (var i = kill.length - 1; i >= 0; i--)
      { replaceRange(cm.doc, "", kill[i].from, kill[i].to, "+delete") }
    ensureCursorVisible(cm)
  })
}

// Commands are parameter-less actions that can be performed on an
// editor, mostly used for keybindings.
var commands = {
  selectAll: selectAll,
  singleSelection: function (cm) { return cm.setSelection(cm.getCursor("anchor"), cm.getCursor("head"), sel_dontScroll); },
  killLine: function (cm) { return deleteNearSelection(cm, function (range) {
    if (range.empty()) {
      var len = getLine(cm.doc, range.head.line).text.length
      if (range.head.ch == len && range.head.line < cm.lastLine())
        { return {from: range.head, to: Pos(range.head.line + 1, 0)} }
      else
        { return {from: range.head, to: Pos(range.head.line, len)} }
    } else {
      return {from: range.from(), to: range.to()}
    }
  }); },
  deleteLine: function (cm) { return deleteNearSelection(cm, function (range) { return ({
    from: Pos(range.from().line, 0),
    to: clipPos(cm.doc, Pos(range.to().line + 1, 0))
  }); }); },
  delLineLeft: function (cm) { return deleteNearSelection(cm, function (range) { return ({
    from: Pos(range.from().line, 0), to: range.from()
  }); }); },
  delWrappedLineLeft: function (cm) { return deleteNearSelection(cm, function (range) {
    var top = cm.charCoords(range.head, "div").top + 5
    var leftPos = cm.coordsChar({left: 0, top: top}, "div")
    return {from: leftPos, to: range.from()}
  }); },
  delWrappedLineRight: function (cm) { return deleteNearSelection(cm, function (range) {
    var top = cm.charCoords(range.head, "div").top + 5
    var rightPos = cm.coordsChar({left: cm.display.lineDiv.offsetWidth + 100, top: top}, "div")
    return {from: range.from(), to: rightPos }
  }); },
  undo: function (cm) { return cm.undo(); },
  redo: function (cm) { return cm.redo(); },
  undoSelection: function (cm) { return cm.undoSelection(); },
  redoSelection: function (cm) { return cm.redoSelection(); },
  goDocStart: function (cm) { return cm.extendSelection(Pos(cm.firstLine(), 0)); },
  goDocEnd: function (cm) { return cm.extendSelection(Pos(cm.lastLine())); },
  goLineStart: function (cm) { return cm.extendSelectionsBy(function (range) { return lineStart(cm, range.head.line); },
    {origin: "+move", bias: 1}
  ); },
  goLineStartSmart: function (cm) { return cm.extendSelectionsBy(function (range) { return lineStartSmart(cm, range.head); },
    {origin: "+move", bias: 1}
  ); },
  goLineEnd: function (cm) { return cm.extendSelectionsBy(function (range) { return lineEnd(cm, range.head.line); },
    {origin: "+move", bias: -1}
  ); },
  goLineRight: function (cm) { return cm.extendSelectionsBy(function (range) {
    var top = cm.charCoords(range.head, "div").top + 5
    return cm.coordsChar({left: cm.display.lineDiv.offsetWidth + 100, top: top}, "div")
  }, sel_move); },
  goLineLeft: function (cm) { return cm.extendSelectionsBy(function (range) {
    var top = cm.charCoords(range.head, "div").top + 5
    return cm.coordsChar({left: 0, top: top}, "div")
  }, sel_move); },
  goLineLeftSmart: function (cm) { return cm.extendSelectionsBy(function (range) {
    var top = cm.charCoords(range.head, "div").top + 5
    var pos = cm.coordsChar({left: 0, top: top}, "div")
    if (pos.ch < cm.getLine(pos.line).search(/\S/)) { return lineStartSmart(cm, range.head) }
    return pos
  }, sel_move); },
  goLineUp: function (cm) { return cm.moveV(-1, "line"); },
  goLineDown: function (cm) { return cm.moveV(1, "line"); },
  goPageUp: function (cm) { return cm.moveV(-1, "page"); },
  goPageDown: function (cm) { return cm.moveV(1, "page"); },
  goCharLeft: function (cm) { return cm.moveH(-1, "char"); },
  goCharRight: function (cm) { return cm.moveH(1, "char"); },
  goColumnLeft: function (cm) { return cm.moveH(-1, "column"); },
  goColumnRight: function (cm) { return cm.moveH(1, "column"); },
  goWordLeft: function (cm) { return cm.moveH(-1, "word"); },
  goGroupRight: function (cm) { return cm.moveH(1, "group"); },
  goGroupLeft: function (cm) { return cm.moveH(-1, "group"); },
  goWordRight: function (cm) { return cm.moveH(1, "word"); },
  delCharBefore: function (cm) { return cm.deleteH(-1, "char"); },
  delCharAfter: function (cm) { return cm.deleteH(1, "char"); },
  delWordBefore: function (cm) { return cm.deleteH(-1, "word"); },
  delWordAfter: function (cm) { return cm.deleteH(1, "word"); },
  delGroupBefore: function (cm) { return cm.deleteH(-1, "group"); },
  delGroupAfter: function (cm) { return cm.deleteH(1, "group"); },
  indentAuto: function (cm) { return cm.indentSelection("smart"); },
  indentMore: function (cm) { return cm.indentSelection("add"); },
  indentLess: function (cm) { return cm.indentSelection("subtract"); },
  insertTab: function (cm) { return cm.replaceSelection("\t"); },
  insertSoftTab: function (cm) {
    var spaces = [], ranges = cm.listSelections(), tabSize = cm.options.tabSize
    for (var i = 0; i < ranges.length; i++) {
      var pos = ranges[i].from()
      var col = countColumn(cm.getLine(pos.line), pos.ch, tabSize)
      spaces.push(spaceStr(tabSize - col % tabSize))
    }
    cm.replaceSelections(spaces)
  },
  defaultTab: function (cm) {
    if (cm.somethingSelected()) { cm.indentSelection("add") }
    else { cm.execCommand("insertTab") }
  },
  // Swap the two chars left and right of each selection's head.
  // Move cursor behind the two swapped characters afterwards.
  //
  // Doesn't consider line feeds a character.
  // Doesn't scan more than one line above to find a character.
  // Doesn't do anything on an empty line.
  // Doesn't do anything with non-empty selections.
  transposeChars: function (cm) { return runInOp(cm, function () {
    var ranges = cm.listSelections(), newSel = []
    for (var i = 0; i < ranges.length; i++) {
      if (!ranges[i].empty()) { continue }
      var cur = ranges[i].head, line = getLine(cm.doc, cur.line).text
      if (line) {
        if (cur.ch == line.length) { cur = new Pos(cur.line, cur.ch - 1) }
        if (cur.ch > 0) {
          cur = new Pos(cur.line, cur.ch + 1)
          cm.replaceRange(line.charAt(cur.ch - 1) + line.charAt(cur.ch - 2),
                          Pos(cur.line, cur.ch - 2), cur, "+transpose")
        } else if (cur.line > cm.doc.first) {
          var prev = getLine(cm.doc, cur.line - 1).text
          if (prev) {
            cur = new Pos(cur.line, 1)
            cm.replaceRange(line.charAt(0) + cm.doc.lineSeparator() +
                            prev.charAt(prev.length - 1),
                            Pos(cur.line - 1, prev.length - 1), cur, "+transpose")
          }
        }
      }
      newSel.push(new Range(cur, cur))
    }
    cm.setSelections(newSel)
  }); },
  newlineAndIndent: function (cm) { return runInOp(cm, function () {
    var sels = cm.listSelections()
    for (var i = sels.length - 1; i >= 0; i--)
      { cm.replaceRange(cm.doc.lineSeparator(), sels[i].anchor, sels[i].head, "+input") }
    sels = cm.listSelections()
    for (var i$1 = 0; i$1 < sels.length; i$1++)
      { cm.indentLine(sels[i$1].from().line, null, true) }
    ensureCursorVisible(cm)
  }); },
  openLine: function (cm) { return cm.replaceSelection("\n", "start"); },
  toggleOverwrite: function (cm) { return cm.toggleOverwrite(); }
}


function lineStart(cm, lineN) {
  var line = getLine(cm.doc, lineN)
  var visual = visualLine(line)
  if (visual != line) { lineN = lineNo(visual) }
  return endOfLine(true, cm, visual, lineN, 1)
}
function lineEnd(cm, lineN) {
  var line = getLine(cm.doc, lineN)
  var visual = visualLineEnd(line)
  if (visual != line) { lineN = lineNo(visual) }
  return endOfLine(true, cm, line, lineN, -1)
}
function lineStartSmart(cm, pos) {
  var start = lineStart(cm, pos.line)
  var line = getLine(cm.doc, start.line)
  var order = getOrder(line)
  if (!order || order[0].level == 0) {
    var firstNonWS = Math.max(0, line.text.search(/\S/))
    var inWS = pos.line == start.line && pos.ch <= firstNonWS && pos.ch
    return Pos(start.line, inWS ? 0 : firstNonWS, start.sticky)
  }
  return start
}

// Run a handler that was bound to a key.
function doHandleBinding(cm, bound, dropShift) {
  if (typeof bound == "string") {
    bound = commands[bound]
    if (!bound) { return false }
  }
  // Ensure previous input has been read, so that the handler sees a
  // consistent view of the document
  cm.display.input.ensurePolled()
  var prevShift = cm.display.shift, done = false
  try {
    if (cm.isReadOnly()) { cm.state.suppressEdits = true }
    if (dropShift) { cm.display.shift = false }
    done = bound(cm) != Pass
  } finally {
    cm.display.shift = prevShift
    cm.state.suppressEdits = false
  }
  return done
}

function lookupKeyForEditor(cm, name, handle) {
  for (var i = 0; i < cm.state.keyMaps.length; i++) {
    var result = lookupKey(name, cm.state.keyMaps[i], handle, cm)
    if (result) { return result }
  }
  return (cm.options.extraKeys && lookupKey(name, cm.options.extraKeys, handle, cm))
    || lookupKey(name, cm.options.keyMap, handle, cm)
}

var stopSeq = new Delayed
function dispatchKey(cm, name, e, handle) {
  var seq = cm.state.keySeq
  if (seq) {
    if (isModifierKey(name)) { return "handled" }
    stopSeq.set(50, function () {
      if (cm.state.keySeq == seq) {
        cm.state.keySeq = null
        cm.display.input.reset()
      }
    })
    name = seq + " " + name
  }
  var result = lookupKeyForEditor(cm, name, handle)

  if (result == "multi")
    { cm.state.keySeq = name }
  if (result == "handled")
    { signalLater(cm, "keyHandled", cm, name, e) }

  if (result == "handled" || result == "multi") {
    e_preventDefault(e)
    restartBlink(cm)
  }

  if (seq && !result && /\'$/.test(name)) {
    e_preventDefault(e)
    return true
  }
  return !!result
}

// Handle a key from the keydown event.
function handleKeyBinding(cm, e) {
  var name = keyName(e, true)
  if (!name) { return false }

  if (e.shiftKey && !cm.state.keySeq) {
    // First try to resolve full name (including 'Shift-'). Failing
    // that, see if there is a cursor-motion command (starting with
    // 'go') bound to the keyname without 'Shift-'.
    return dispatchKey(cm, "Shift-" + name, e, function (b) { return doHandleBinding(cm, b, true); })
        || dispatchKey(cm, name, e, function (b) {
             if (typeof b == "string" ? /^go[A-Z]/.test(b) : b.motion)
               { return doHandleBinding(cm, b) }
           })
  } else {
    return dispatchKey(cm, name, e, function (b) { return doHandleBinding(cm, b); })
  }
}

// Handle a key from the keypress event
function handleCharBinding(cm, e, ch) {
  return dispatchKey(cm, "'" + ch + "'", e, function (b) { return doHandleBinding(cm, b, true); })
}

var lastStoppedKey = null
function onKeyDown(e) {
  var cm = this
  cm.curOp.focus = activeElt()
  if (signalDOMEvent(cm, e)) { return }
  // IE does strange things with escape.
  if (ie && ie_version < 11 && e.keyCode == 27) { e.returnValue = false }
  var code = e.keyCode
  cm.display.shift = code == 16 || e.shiftKey
  var handled = handleKeyBinding(cm, e)
  if (presto) {
    lastStoppedKey = handled ? code : null
    // Opera has no cut event... we try to at least catch the key combo
    if (!handled && code == 88 && !hasCopyEvent && (mac ? e.metaKey : e.ctrlKey))
      { cm.replaceSelection("", null, "cut") }
  }

  // Turn mouse into crosshair when Alt is held on Mac.
  if (code == 18 && !/\bCodeMirror-crosshair\b/.test(cm.display.lineDiv.className))
    { showCrossHair(cm) }
}

function showCrossHair(cm) {
  var lineDiv = cm.display.lineDiv
  addClass(lineDiv, "CodeMirror-crosshair")

  function up(e) {
    if (e.keyCode == 18 || !e.altKey) {
      rmClass(lineDiv, "CodeMirror-crosshair")
      off(document, "keyup", up)
      off(document, "mouseover", up)
    }
  }
  on(document, "keyup", up)
  on(document, "mouseover", up)
}

function onKeyUp(e) {
  if (e.keyCode == 16) { this.doc.sel.shift = false }
  signalDOMEvent(this, e)
}

function onKeyPress(e) {
  var cm = this
  if (eventInWidget(cm.display, e) || signalDOMEvent(cm, e) || e.ctrlKey && !e.altKey || mac && e.metaKey) { return }
  var keyCode = e.keyCode, charCode = e.charCode
  if (presto && keyCode == lastStoppedKey) {lastStoppedKey = null; e_preventDefault(e); return}
  if ((presto && (!e.which || e.which < 10)) && handleKeyBinding(cm, e)) { return }
  var ch = String.fromCharCode(charCode == null ? keyCode : charCode)
  // Some browsers fire keypress events for backspace
  if (ch == "\x08") { return }
  if (handleCharBinding(cm, e, ch)) { return }
  cm.display.input.onKeyPress(e)
}

// A mouse down can be a single click, double click, triple click,
// start of selection drag, start of text drag, new cursor
// (ctrl-click), rectangle drag (alt-drag), or xwin
// middle-click-paste. Or it might be a click on something we should
// not interfere with, such as a scrollbar or widget.
function onMouseDown(e) {
  var cm = this, display = cm.display
  if (signalDOMEvent(cm, e) || display.activeTouch && display.input.supportsTouch()) { return }
  display.input.ensurePolled()
  display.shift = e.shiftKey

  if (eventInWidget(display, e)) {
    if (!webkit) {
      // Briefly turn off draggability, to allow widgets to do
      // normal dragging things.
      display.scroller.draggable = false
      setTimeout(function () { return display.scroller.draggable = true; }, 100)
    }
    return
  }
  if (clickInGutter(cm, e)) { return }
  var start = posFromMouse(cm, e)
  window.focus()

  switch (e_button(e)) {
  case 1:
    // #3261: make sure, that we're not starting a second selection
    if (cm.state.selectingText)
      { cm.state.selectingText(e) }
    else if (start)
      { leftButtonDown(cm, e, start) }
    else if (e_target(e) == display.scroller)
      { e_preventDefault(e) }
    break
  case 2:
    if (webkit) { cm.state.lastMiddleDown = +new Date }
    if (start) { extendSelection(cm.doc, start) }
    setTimeout(function () { return display.input.focus(); }, 20)
    e_preventDefault(e)
    break
  case 3:
    if (captureRightClick) { onContextMenu(cm, e) }
    else { delayBlurEvent(cm) }
    break
  }
}

var lastClick;
var lastDoubleClick;
function leftButtonDown(cm, e, start) {
  if (ie) { setTimeout(bind(ensureFocus, cm), 0) }
  else { cm.curOp.focus = activeElt() }

  var now = +new Date, type
  if (lastDoubleClick && lastDoubleClick.time > now - 400 && cmp(lastDoubleClick.pos, start) == 0) {
    type = "triple"
  } else if (lastClick && lastClick.time > now - 400 && cmp(lastClick.pos, start) == 0) {
    type = "double"
    lastDoubleClick = {time: now, pos: start}
  } else {
    type = "single"
    lastClick = {time: now, pos: start}
  }

  var sel = cm.doc.sel, modifier = mac ? e.metaKey : e.ctrlKey, contained
  if (cm.options.dragDrop && dragAndDrop && !cm.isReadOnly() &&
      type == "single" && (contained = sel.contains(start)) > -1 &&
      (cmp((contained = sel.ranges[contained]).from(), start) < 0 || start.xRel > 0) &&
      (cmp(contained.to(), start) > 0 || start.xRel < 0))
    { leftButtonStartDrag(cm, e, start, modifier) }
  else
    { leftButtonSelect(cm, e, start, type, modifier) }
}

// Start a text drag. When it ends, see if any dragging actually
// happen, and treat as a click if it didn't.
function leftButtonStartDrag(cm, e, start, modifier) {
  var display = cm.display, startTime = +new Date
  var dragEnd = operation(cm, function (e2) {
    if (webkit) { display.scroller.draggable = false }
    cm.state.draggingText = false
    off(document, "mouseup", dragEnd)
    off(display.scroller, "drop", dragEnd)
    if (Math.abs(e.clientX - e2.clientX) + Math.abs(e.clientY - e2.clientY) < 10) {
      e_preventDefault(e2)
      if (!modifier && +new Date - 200 < startTime)
        { extendSelection(cm.doc, start) }
      // Work around unexplainable focus problem in IE9 (#2127) and Chrome (#3081)
      if (webkit || ie && ie_version == 9)
        { setTimeout(function () {document.body.focus(); display.input.focus()}, 20) }
      else
        { display.input.focus() }
    }
  })
  // Let the drag handler handle this.
  if (webkit) { display.scroller.draggable = true }
  cm.state.draggingText = dragEnd
  dragEnd.copy = mac ? e.altKey : e.ctrlKey
  // IE's approach to draggable
  if (display.scroller.dragDrop) { display.scroller.dragDrop() }
  on(document, "mouseup", dragEnd)
  on(display.scroller, "drop", dragEnd)
}

// Normal selection, as opposed to text dragging.
function leftButtonSelect(cm, e, start, type, addNew) {
  var display = cm.display, doc = cm.doc
  e_preventDefault(e)

  var ourRange, ourIndex, startSel = doc.sel, ranges = startSel.ranges
  if (addNew && !e.shiftKey) {
    ourIndex = doc.sel.contains(start)
    if (ourIndex > -1)
      { ourRange = ranges[ourIndex] }
    else
      { ourRange = new Range(start, start) }
  } else {
    ourRange = doc.sel.primary()
    ourIndex = doc.sel.primIndex
  }

  if (chromeOS ? e.shiftKey && e.metaKey : e.altKey) {
    type = "rect"
    if (!addNew) { ourRange = new Range(start, start) }
    start = posFromMouse(cm, e, true, true)
    ourIndex = -1
  } else if (type == "double") {
    var word = cm.findWordAt(start)
    if (cm.display.shift || doc.extend)
      { ourRange = extendRange(doc, ourRange, word.anchor, word.head) }
    else
      { ourRange = word }
  } else if (type == "triple") {
    var line = new Range(Pos(start.line, 0), clipPos(doc, Pos(start.line + 1, 0)))
    if (cm.display.shift || doc.extend)
      { ourRange = extendRange(doc, ourRange, line.anchor, line.head) }
    else
      { ourRange = line }
  } else {
    ourRange = extendRange(doc, ourRange, start)
  }

  if (!addNew) {
    ourIndex = 0
    setSelection(doc, new Selection([ourRange], 0), sel_mouse)
    startSel = doc.sel
  } else if (ourIndex == -1) {
    ourIndex = ranges.length
    setSelection(doc, normalizeSelection(ranges.concat([ourRange]), ourIndex),
                 {scroll: false, origin: "*mouse"})
  } else if (ranges.length > 1 && ranges[ourIndex].empty() && type == "single" && !e.shiftKey) {
    setSelection(doc, normalizeSelection(ranges.slice(0, ourIndex).concat(ranges.slice(ourIndex + 1)), 0),
                 {scroll: false, origin: "*mouse"})
    startSel = doc.sel
  } else {
    replaceOneSelection(doc, ourIndex, ourRange, sel_mouse)
  }

  var lastPos = start
  function extendTo(pos) {
    if (cmp(lastPos, pos) == 0) { return }
    lastPos = pos

    if (type == "rect") {
      var ranges = [], tabSize = cm.options.tabSize
      var startCol = countColumn(getLine(doc, start.line).text, start.ch, tabSize)
      var posCol = countColumn(getLine(doc, pos.line).text, pos.ch, tabSize)
      var left = Math.min(startCol, posCol), right = Math.max(startCol, posCol)
      for (var line = Math.min(start.line, pos.line), end = Math.min(cm.lastLine(), Math.max(start.line, pos.line));
           line <= end; line++) {
        var text = getLine(doc, line).text, leftPos = findColumn(text, left, tabSize)
        if (left == right)
          { ranges.push(new Range(Pos(line, leftPos), Pos(line, leftPos))) }
        else if (text.length > leftPos)
          { ranges.push(new Range(Pos(line, leftPos), Pos(line, findColumn(text, right, tabSize)))) }
      }
      if (!ranges.length) { ranges.push(new Range(start, start)) }
      setSelection(doc, normalizeSelection(startSel.ranges.slice(0, ourIndex).concat(ranges), ourIndex),
                   {origin: "*mouse", scroll: false})
      cm.scrollIntoView(pos)
    } else {
      var oldRange = ourRange
      var anchor = oldRange.anchor, head = pos
      if (type != "single") {
        var range
        if (type == "double")
          { range = cm.findWordAt(pos) }
        else
          { range = new Range(Pos(pos.line, 0), clipPos(doc, Pos(pos.line + 1, 0))) }
        if (cmp(range.anchor, anchor) > 0) {
          head = range.head
          anchor = minPos(oldRange.from(), range.anchor)
        } else {
          head = range.anchor
          anchor = maxPos(oldRange.to(), range.head)
        }
      }
      var ranges$1 = startSel.ranges.slice(0)
      ranges$1[ourIndex] = new Range(clipPos(doc, anchor), head)
      setSelection(doc, normalizeSelection(ranges$1, ourIndex), sel_mouse)
    }
  }

  var editorSize = display.wrapper.getBoundingClientRect()
  // Used to ensure timeout re-tries don't fire when another extend
  // happened in the meantime (clearTimeout isn't reliable -- at
  // least on Chrome, the timeouts still happen even when cleared,
  // if the clear happens after their scheduled firing time).
  var counter = 0

  function extend(e) {
    var curCount = ++counter
    var cur = posFromMouse(cm, e, true, type == "rect")
    if (!cur) { return }
    if (cmp(cur, lastPos) != 0) {
      cm.curOp.focus = activeElt()
      extendTo(cur)
      var visible = visibleLines(display, doc)
      if (cur.line >= visible.to || cur.line < visible.from)
        { setTimeout(operation(cm, function () {if (counter == curCount) { extend(e) }}), 150) }
    } else {
      var outside = e.clientY < editorSize.top ? -20 : e.clientY > editorSize.bottom ? 20 : 0
      if (outside) { setTimeout(operation(cm, function () {
        if (counter != curCount) { return }
        display.scroller.scrollTop += outside
        extend(e)
      }), 50) }
    }
  }

  function done(e) {
    cm.state.selectingText = false
    counter = Infinity
    e_preventDefault(e)
    display.input.focus()
    off(document, "mousemove", move)
    off(document, "mouseup", up)
    doc.history.lastSelOrigin = null
  }

  var move = operation(cm, function (e) {
    if (!e_button(e)) { done(e) }
    else { extend(e) }
  })
  var up = operation(cm, done)
  cm.state.selectingText = up
  on(document, "mousemove", move)
  on(document, "mouseup", up)
}


// Determines whether an event happened in the gutter, and fires the
// handlers for the corresponding event.
function gutterEvent(cm, e, type, prevent) {
  var mX, mY
  try { mX = e.clientX; mY = e.clientY }
  catch(e) { return false }
  if (mX >= Math.floor(cm.display.gutters.getBoundingClientRect().right)) { return false }
  if (prevent) { e_preventDefault(e) }

  var display = cm.display
  var lineBox = display.lineDiv.getBoundingClientRect()

  if (mY > lineBox.bottom || !hasHandler(cm, type)) { return e_defaultPrevented(e) }
  mY -= lineBox.top - display.viewOffset

  for (var i = 0; i < cm.options.gutters.length; ++i) {
    var g = display.gutters.childNodes[i]
    if (g && g.getBoundingClientRect().right >= mX) {
      var line = lineAtHeight(cm.doc, mY)
      var gutter = cm.options.gutters[i]
      signal(cm, type, cm, line, gutter, e)
      return e_defaultPrevented(e)
    }
  }
}

function clickInGutter(cm, e) {
  return gutterEvent(cm, e, "gutterClick", true)
}

// CONTEXT MENU HANDLING

// To make the context menu work, we need to briefly unhide the
// textarea (making it as unobtrusive as possible) to let the
// right-click take effect on it.
function onContextMenu(cm, e) {
  if (eventInWidget(cm.display, e) || contextMenuInGutter(cm, e)) { return }
  if (signalDOMEvent(cm, e, "contextmenu")) { return }
  cm.display.input.onContextMenu(e)
}

function contextMenuInGutter(cm, e) {
  if (!hasHandler(cm, "gutterContextMenu")) { return false }
  return gutterEvent(cm, e, "gutterContextMenu", false)
}

function themeChanged(cm) {
  cm.display.wrapper.className = cm.display.wrapper.className.replace(/\s*cm-s-\S+/g, "") +
    cm.options.theme.replace(/(^|\s)\s*/g, " cm-s-")
  clearCaches(cm)
}

var Init = {toString: function(){return "CodeMirror.Init"}}

var defaults = {}
var optionHandlers = {}

function defineOptions(CodeMirror) {
  var optionHandlers = CodeMirror.optionHandlers

  function option(name, deflt, handle, notOnInit) {
    CodeMirror.defaults[name] = deflt
    if (handle) { optionHandlers[name] =
      notOnInit ? function (cm, val, old) {if (old != Init) { handle(cm, val, old) }} : handle }
  }

  CodeMirror.defineOption = option

  // Passed to option handlers when there is no old value.
  CodeMirror.Init = Init

  // These two are, on init, called from the constructor because they
  // have to be initialized before the editor can start at all.
  option("value", "", function (cm, val) { return cm.setValue(val); }, true)
  option("mode", null, function (cm, val) {
    cm.doc.modeOption = val
    loadMode(cm)
  }, true)

  option("indentUnit", 2, loadMode, true)
  option("indentWithTabs", false)
  option("smartIndent", true)
  option("tabSize", 4, function (cm) {
    resetModeState(cm)
    clearCaches(cm)
    regChange(cm)
  }, true)
  option("lineSeparator", null, function (cm, val) {
    cm.doc.lineSep = val
    if (!val) { return }
    var newBreaks = [], lineNo = cm.doc.first
    cm.doc.iter(function (line) {
      for (var pos = 0;;) {
        var found = line.text.indexOf(val, pos)
        if (found == -1) { break }
        pos = found + val.length
        newBreaks.push(Pos(lineNo, found))
      }
      lineNo++
    })
    for (var i = newBreaks.length - 1; i >= 0; i--)
      { replaceRange(cm.doc, val, newBreaks[i], Pos(newBreaks[i].line, newBreaks[i].ch + val.length)) }
  })
  option("specialChars", /[\u0000-\u001f\u007f\u00ad\u061c\u200b-\u200f\u2028\u2029\ufeff]/g, function (cm, val, old) {
    cm.state.specialChars = new RegExp(val.source + (val.test("\t") ? "" : "|\t"), "g")
    if (old != Init) { cm.refresh() }
  })
  option("specialCharPlaceholder", defaultSpecialCharPlaceholder, function (cm) { return cm.refresh(); }, true)
  option("electricChars", true)
  option("inputStyle", mobile ? "contenteditable" : "textarea", function () {
    throw new Error("inputStyle can not (yet) be changed in a running editor") // FIXME
  }, true)
  option("spellcheck", false, function (cm, val) { return cm.getInputField().spellcheck = val; }, true)
  option("rtlMoveVisually", !windows)
  option("wholeLineUpdateBefore", true)

  option("theme", "default", function (cm) {
    themeChanged(cm)
    guttersChanged(cm)
  }, true)
  option("keyMap", "default", function (cm, val, old) {
    var next = getKeyMap(val)
    var prev = old != Init && getKeyMap(old)
    if (prev && prev.detach) { prev.detach(cm, next) }
    if (next.attach) { next.attach(cm, prev || null) }
  })
  option("extraKeys", null)

  option("lineWrapping", false, wrappingChanged, true)
  option("gutters", [], function (cm) {
    setGuttersForLineNumbers(cm.options)
    guttersChanged(cm)
  }, true)
  option("fixedGutter", true, function (cm, val) {
    cm.display.gutters.style.left = val ? compensateForHScroll(cm.display) + "px" : "0"
    cm.refresh()
  }, true)
  option("coverGutterNextToScrollbar", false, function (cm) { return updateScrollbars(cm); }, true)
  option("scrollbarStyle", "native", function (cm) {
    initScrollbars(cm)
    updateScrollbars(cm)
    cm.display.scrollbars.setScrollTop(cm.doc.scrollTop)
    cm.display.scrollbars.setScrollLeft(cm.doc.scrollLeft)
  }, true)
  option("lineNumbers", false, function (cm) {
    setGuttersForLineNumbers(cm.options)
    guttersChanged(cm)
  }, true)
  option("firstLineNumber", 1, guttersChanged, true)
  option("lineNumberFormatter", function (integer) { return integer; }, guttersChanged, true)
  option("showCursorWhenSelecting", false, updateSelection, true)

  option("resetSelectionOnContextMenu", true)
  option("lineWiseCopyCut", true)

  option("readOnly", false, function (cm, val) {
    if (val == "nocursor") {
      onBlur(cm)
      cm.display.input.blur()
      cm.display.disabled = true
    } else {
      cm.display.disabled = false
    }
    cm.display.input.readOnlyChanged(val)
  })
  option("disableInput", false, function (cm, val) {if (!val) { cm.display.input.reset() }}, true)
  option("dragDrop", true, dragDropChanged)
  option("allowDropFileTypes", null)

  option("cursorBlinkRate", 530)
  option("cursorScrollMargin", 0)
  option("cursorHeight", 1, updateSelection, true)
  option("singleCursorHeightPerLine", true, updateSelection, true)
  option("workTime", 100)
  option("workDelay", 100)
  option("flattenSpans", true, resetModeState, true)
  option("addModeClass", false, resetModeState, true)
  option("pollInterval", 100)
  option("undoDepth", 200, function (cm, val) { return cm.doc.history.undoDepth = val; })
  option("historyEventDelay", 1250)
  option("viewportMargin", 10, function (cm) { return cm.refresh(); }, true)
  option("maxHighlightLength", 10000, resetModeState, true)
  option("moveInputWithCursor", true, function (cm, val) {
    if (!val) { cm.display.input.resetPosition() }
  })

  option("tabindex", null, function (cm, val) { return cm.display.input.getField().tabIndex = val || ""; })
  option("autofocus", null)
}

function guttersChanged(cm) {
  updateGutters(cm)
  regChange(cm)
  alignHorizontally(cm)
}

function dragDropChanged(cm, value, old) {
  var wasOn = old && old != Init
  if (!value != !wasOn) {
    var funcs = cm.display.dragFunctions
    var toggle = value ? on : off
    toggle(cm.display.scroller, "dragstart", funcs.start)
    toggle(cm.display.scroller, "dragenter", funcs.enter)
    toggle(cm.display.scroller, "dragover", funcs.over)
    toggle(cm.display.scroller, "dragleave", funcs.leave)
    toggle(cm.display.scroller, "drop", funcs.drop)
  }
}

function wrappingChanged(cm) {
  if (cm.options.lineWrapping) {
    addClass(cm.display.wrapper, "CodeMirror-wrap")
    cm.display.sizer.style.minWidth = ""
    cm.display.sizerWidth = null
  } else {
    rmClass(cm.display.wrapper, "CodeMirror-wrap")
    findMaxLine(cm)
  }
  estimateLineHeights(cm)
  regChange(cm)
  clearCaches(cm)
  setTimeout(function () { return updateScrollbars(cm); }, 100)
}

// A CodeMirror instance represents an editor. This is the object
// that user code is usually dealing with.

function CodeMirror(place, options) {
  var this$1 = this;

  if (!(this instanceof CodeMirror)) { return new CodeMirror(place, options) }

  this.options = options = options ? copyObj(options) : {}
  // Determine effective options based on given values and defaults.
  copyObj(defaults, options, false)
  setGuttersForLineNumbers(options)

  var doc = options.value
  if (typeof doc == "string") { doc = new Doc(doc, options.mode, null, options.lineSeparator) }
  this.doc = doc

  var input = new CodeMirror.inputStyles[options.inputStyle](this)
  var display = this.display = new Display(place, doc, input)
  display.wrapper.CodeMirror = this
  updateGutters(this)
  themeChanged(this)
  if (options.lineWrapping)
    { this.display.wrapper.className += " CodeMirror-wrap" }
  initScrollbars(this)

  this.state = {
    keyMaps: [],  // stores maps added by addKeyMap
    overlays: [], // highlighting overlays, as added by addOverlay
    modeGen: 0,   // bumped when mode/overlay changes, used to invalidate highlighting info
    overwrite: false,
    delayingBlurEvent: false,
    focused: false,
    suppressEdits: false, // used to disable editing during key handlers when in readOnly mode
    pasteIncoming: false, cutIncoming: false, // help recognize paste/cut edits in input.poll
    selectingText: false,
    draggingText: false,
    highlight: new Delayed(), // stores highlight worker timeout
    keySeq: null,  // Unfinished key sequence
    specialChars: null
  }

  if (options.autofocus && !mobile) { display.input.focus() }

  // Override magic textarea content restore that IE sometimes does
  // on our hidden textarea on reload
  if (ie && ie_version < 11) { setTimeout(function () { return this$1.display.input.reset(true); }, 20) }

  registerEventHandlers(this)
  ensureGlobalHandlers()

  startOperation(this)
  this.curOp.forceUpdate = true
  attachDoc(this, doc)

  if ((options.autofocus && !mobile) || this.hasFocus())
    { setTimeout(bind(onFocus, this), 20) }
  else
    { onBlur(this) }

  for (var opt in optionHandlers) { if (optionHandlers.hasOwnProperty(opt))
    { optionHandlers[opt](this$1, options[opt], Init) } }
  maybeUpdateLineNumberWidth(this)
  if (options.finishInit) { options.finishInit(this) }
  for (var i = 0; i < initHooks.length; ++i) { initHooks[i](this$1) }
  endOperation(this)
  // Suppress optimizelegibility in Webkit, since it breaks text
  // measuring on line wrapping boundaries.
  if (webkit && options.lineWrapping &&
      getComputedStyle(display.lineDiv).textRendering == "optimizelegibility")
    { display.lineDiv.style.textRendering = "auto" }
}

// The default configuration options.
CodeMirror.defaults = defaults
// Functions to run when options are changed.
CodeMirror.optionHandlers = optionHandlers

// Attach the necessary event handlers when initializing the editor
function registerEventHandlers(cm) {
  var d = cm.display
  on(d.scroller, "mousedown", operation(cm, onMouseDown))
  // Older IE's will not fire a second mousedown for a double click
  if (ie && ie_version < 11)
    { on(d.scroller, "dblclick", operation(cm, function (e) {
      if (signalDOMEvent(cm, e)) { return }
      var pos = posFromMouse(cm, e)
      if (!pos || clickInGutter(cm, e) || eventInWidget(cm.display, e)) { return }
      e_preventDefault(e)
      var word = cm.findWordAt(pos)
      extendSelection(cm.doc, word.anchor, word.head)
    })) }
  else
    { on(d.scroller, "dblclick", function (e) { return signalDOMEvent(cm, e) || e_preventDefault(e); }) }
  // Some browsers fire contextmenu *after* opening the menu, at
  // which point we can't mess with it anymore. Context menu is
  // handled in onMouseDown for these browsers.
  if (!captureRightClick) { on(d.scroller, "contextmenu", function (e) { return onContextMenu(cm, e); }) }

  // Used to suppress mouse event handling when a touch happens
  var touchFinished, prevTouch = {end: 0}
  function finishTouch() {
    if (d.activeTouch) {
      touchFinished = setTimeout(function () { return d.activeTouch = null; }, 1000)
      prevTouch = d.activeTouch
      prevTouch.end = +new Date
    }
  }
  function isMouseLikeTouchEvent(e) {
    if (e.touches.length != 1) { return false }
    var touch = e.touches[0]
    return touch.radiusX <= 1 && touch.radiusY <= 1
  }
  function farAway(touch, other) {
    if (other.left == null) { return true }
    var dx = other.left - touch.left, dy = other.top - touch.top
    return dx * dx + dy * dy > 20 * 20
  }
  on(d.scroller, "touchstart", function (e) {
    if (!signalDOMEvent(cm, e) && !isMouseLikeTouchEvent(e)) {
      d.input.ensurePolled()
      clearTimeout(touchFinished)
      var now = +new Date
      d.activeTouch = {start: now, moved: false,
                       prev: now - prevTouch.end <= 300 ? prevTouch : null}
      if (e.touches.length == 1) {
        d.activeTouch.left = e.touches[0].pageX
        d.activeTouch.top = e.touches[0].pageY
      }
    }
  })
  on(d.scroller, "touchmove", function () {
    if (d.activeTouch) { d.activeTouch.moved = true }
  })
  on(d.scroller, "touchend", function (e) {
    var touch = d.activeTouch
    if (touch && !eventInWidget(d, e) && touch.left != null &&
        !touch.moved && new Date - touch.start < 300) {
      var pos = cm.coordsChar(d.activeTouch, "page"), range
      if (!touch.prev || farAway(touch, touch.prev)) // Single tap
        { range = new Range(pos, pos) }
      else if (!touch.prev.prev || farAway(touch, touch.prev.prev)) // Double tap
        { range = cm.findWordAt(pos) }
      else // Triple tap
        { range = new Range(Pos(pos.line, 0), clipPos(cm.doc, Pos(pos.line + 1, 0))) }
      cm.setSelection(range.anchor, range.head)
      cm.focus()
      e_preventDefault(e)
    }
    finishTouch()
  })
  on(d.scroller, "touchcancel", finishTouch)

  // Sync scrolling between fake scrollbars and real scrollable
  // area, ensure viewport is updated when scrolling.
  on(d.scroller, "scroll", function () {
    if (d.scroller.clientHeight) {
      setScrollTop(cm, d.scroller.scrollTop)
      setScrollLeft(cm, d.scroller.scrollLeft, true)
      signal(cm, "scroll", cm)
    }
  })

  // Listen to wheel events in order to try and update the viewport on time.
  on(d.scroller, "mousewheel", function (e) { return onScrollWheel(cm, e); })
  on(d.scroller, "DOMMouseScroll", function (e) { return onScrollWheel(cm, e); })

  // Prevent wrapper from ever scrolling
  on(d.wrapper, "scroll", function () { return d.wrapper.scrollTop = d.wrapper.scrollLeft = 0; })

  d.dragFunctions = {
    enter: function (e) {if (!signalDOMEvent(cm, e)) { e_stop(e) }},
    over: function (e) {if (!signalDOMEvent(cm, e)) { onDragOver(cm, e); e_stop(e) }},
    start: function (e) { return onDragStart(cm, e); },
    drop: operation(cm, onDrop),
    leave: function (e) {if (!signalDOMEvent(cm, e)) { clearDragCursor(cm) }}
  }

  var inp = d.input.getField()
  on(inp, "keyup", function (e) { return onKeyUp.call(cm, e); })
  on(inp, "keydown", operation(cm, onKeyDown))
  on(inp, "keypress", operation(cm, onKeyPress))
  on(inp, "focus", function (e) { return onFocus(cm, e); })
  on(inp, "blur", function (e) { return onBlur(cm, e); })
}

var initHooks = []
CodeMirror.defineInitHook = function (f) { return initHooks.push(f); }

// Indent the given line. The how parameter can be "smart",
// "add"/null, "subtract", or "prev". When aggressive is false
// (typically set to true for forced single-line indents), empty
// lines are not indented, and places where the mode returns Pass
// are left alone.
function indentLine(cm, n, how, aggressive) {
  var doc = cm.doc, state
  if (how == null) { how = "add" }
  if (how == "smart") {
    // Fall back to "prev" when the mode doesn't have an indentation
    // method.
    if (!doc.mode.indent) { how = "prev" }
    else { state = getStateBefore(cm, n) }
  }

  var tabSize = cm.options.tabSize
  var line = getLine(doc, n), curSpace = countColumn(line.text, null, tabSize)
  if (line.stateAfter) { line.stateAfter = null }
  var curSpaceString = line.text.match(/^\s*/)[0], indentation
  if (!aggressive && !/\S/.test(line.text)) {
    indentation = 0
    how = "not"
  } else if (how == "smart") {
    indentation = doc.mode.indent(state, line.text.slice(curSpaceString.length), line.text)
    if (indentation == Pass || indentation > 150) {
      if (!aggressive) { return }
      how = "prev"
    }
  }
  if (how == "prev") {
    if (n > doc.first) { indentation = countColumn(getLine(doc, n-1).text, null, tabSize) }
    else { indentation = 0 }
  } else if (how == "add") {
    indentation = curSpace + cm.options.indentUnit
  } else if (how == "subtract") {
    indentation = curSpace - cm.options.indentUnit
  } else if (typeof how == "number") {
    indentation = curSpace + how
  }
  indentation = Math.max(0, indentation)

  var indentString = "", pos = 0
  if (cm.options.indentWithTabs)
    { for (var i = Math.floor(indentation / tabSize); i; --i) {pos += tabSize; indentString += "\t"} }
  if (pos < indentation) { indentString += spaceStr(indentation - pos) }

  if (indentString != curSpaceString) {
    replaceRange(doc, indentString, Pos(n, 0), Pos(n, curSpaceString.length), "+input")
    line.stateAfter = null
    return true
  } else {
    // Ensure that, if the cursor was in the whitespace at the start
    // of the line, it is moved to the end of that space.
    for (var i$1 = 0; i$1 < doc.sel.ranges.length; i$1++) {
      var range = doc.sel.ranges[i$1]
      if (range.head.line == n && range.head.ch < curSpaceString.length) {
        var pos$1 = Pos(n, curSpaceString.length)
        replaceOneSelection(doc, i$1, new Range(pos$1, pos$1))
        break
      }
    }
  }
}

// This will be set to a {lineWise: bool, text: [string]} object, so
// that, when pasting, we know what kind of selections the copied
// text was made out of.
var lastCopied = null

function setLastCopied(newLastCopied) {
  lastCopied = newLastCopied
}

function applyTextInput(cm, inserted, deleted, sel, origin) {
  var doc = cm.doc
  cm.display.shift = false
  if (!sel) { sel = doc.sel }

  var paste = cm.state.pasteIncoming || origin == "paste"
  var textLines = splitLinesAuto(inserted), multiPaste = null
  // When pasing N lines into N selections, insert one line per selection
  if (paste && sel.ranges.length > 1) {
    if (lastCopied && lastCopied.text.join("\n") == inserted) {
      if (sel.ranges.length % lastCopied.text.length == 0) {
        multiPaste = []
        for (var i = 0; i < lastCopied.text.length; i++)
          { multiPaste.push(doc.splitLines(lastCopied.text[i])) }
      }
    } else if (textLines.length == sel.ranges.length) {
      multiPaste = map(textLines, function (l) { return [l]; })
    }
  }

  var updateInput
  // Normal behavior is to insert the new text into every selection
  for (var i$1 = sel.ranges.length - 1; i$1 >= 0; i$1--) {
    var range = sel.ranges[i$1]
    var from = range.from(), to = range.to()
    if (range.empty()) {
      if (deleted && deleted > 0) // Handle deletion
        { from = Pos(from.line, from.ch - deleted) }
      else if (cm.state.overwrite && !paste) // Handle overwrite
        { to = Pos(to.line, Math.min(getLine(doc, to.line).text.length, to.ch + lst(textLines).length)) }
      else if (lastCopied && lastCopied.lineWise && lastCopied.text.join("\n") == inserted)
        { from = to = Pos(from.line, 0) }
    }
    updateInput = cm.curOp.updateInput
    var changeEvent = {from: from, to: to, text: multiPaste ? multiPaste[i$1 % multiPaste.length] : textLines,
                       origin: origin || (paste ? "paste" : cm.state.cutIncoming ? "cut" : "+input")}
    makeChange(cm.doc, changeEvent)
    signalLater(cm, "inputRead", cm, changeEvent)
  }
  if (inserted && !paste)
    { triggerElectric(cm, inserted) }

  ensureCursorVisible(cm)
  cm.curOp.updateInput = updateInput
  cm.curOp.typing = true
  cm.state.pasteIncoming = cm.state.cutIncoming = false
}

function handlePaste(e, cm) {
  var pasted = e.clipboardData && e.clipboardData.getData("Text")
  if (pasted) {
    e.preventDefault()
    if (!cm.isReadOnly() && !cm.options.disableInput)
      { runInOp(cm, function () { return applyTextInput(cm, pasted, 0, null, "paste"); }) }
    return true
  }
}

function triggerElectric(cm, inserted) {
  // When an 'electric' character is inserted, immediately trigger a reindent
  if (!cm.options.electricChars || !cm.options.smartIndent) { return }
  var sel = cm.doc.sel

  for (var i = sel.ranges.length - 1; i >= 0; i--) {
    var range = sel.ranges[i]
    if (range.head.ch > 100 || (i && sel.ranges[i - 1].head.line == range.head.line)) { continue }
    var mode = cm.getModeAt(range.head)
    var indented = false
    if (mode.electricChars) {
      for (var j = 0; j < mode.electricChars.length; j++)
        { if (inserted.indexOf(mode.electricChars.charAt(j)) > -1) {
          indented = indentLine(cm, range.head.line, "smart")
          break
        } }
    } else if (mode.electricInput) {
      if (mode.electricInput.test(getLine(cm.doc, range.head.line).text.slice(0, range.head.ch)))
        { indented = indentLine(cm, range.head.line, "smart") }
    }
    if (indented) { signalLater(cm, "electricInput", cm, range.head.line) }
  }
}

function copyableRanges(cm) {
  var text = [], ranges = []
  for (var i = 0; i < cm.doc.sel.ranges.length; i++) {
    var line = cm.doc.sel.ranges[i].head.line
    var lineRange = {anchor: Pos(line, 0), head: Pos(line + 1, 0)}
    ranges.push(lineRange)
    text.push(cm.getRange(lineRange.anchor, lineRange.head))
  }
  return {text: text, ranges: ranges}
}

function disableBrowserMagic(field, spellcheck) {
  field.setAttribute("autocorrect", "off")
  field.setAttribute("autocapitalize", "off")
  field.setAttribute("spellcheck", !!spellcheck)
}

function hiddenTextarea() {
  var te = elt("textarea", null, null, "position: absolute; bottom: -1em; padding: 0; width: 1px; height: 1em; outline: none")
  var div = elt("div", [te], null, "overflow: hidden; position: relative; width: 3px; height: 0px;")
  // The textarea is kept positioned near the cursor to prevent the
  // fact that it'll be scrolled into view on input from scrolling
  // our fake cursor out of view. On webkit, when wrap=off, paste is
  // very slow. So make the area wide instead.
  if (webkit) { te.style.width = "1000px" }
  else { te.setAttribute("wrap", "off") }
  // If border: 0; -- iOS fails to open keyboard (issue #1287)
  if (ios) { te.style.border = "1px solid black" }
  disableBrowserMagic(te)
  return div
}

// The publicly visible API. Note that methodOp(f) means
// 'wrap f in an operation, performed on its `this` parameter'.

// This is not the complete set of editor methods. Most of the
// methods defined on the Doc type are also injected into
// CodeMirror.prototype, for backwards compatibility and
// convenience.

function addEditorMethods(CodeMirror) {
  var optionHandlers = CodeMirror.optionHandlers

  var helpers = CodeMirror.helpers = {}

  CodeMirror.prototype = {
    constructor: CodeMirror,
    focus: function(){window.focus(); this.display.input.focus()},

    setOption: function(option, value) {
      var options = this.options, old = options[option]
      if (options[option] == value && option != "mode") { return }
      options[option] = value
      if (optionHandlers.hasOwnProperty(option))
        { operation(this, optionHandlers[option])(this, value, old) }
      signal(this, "optionChange", this, option)
    },

    getOption: function(option) {return this.options[option]},
    getDoc: function() {return this.doc},

    addKeyMap: function(map, bottom) {
      this.state.keyMaps[bottom ? "push" : "unshift"](getKeyMap(map))
    },
    removeKeyMap: function(map) {
      var maps = this.state.keyMaps
      for (var i = 0; i < maps.length; ++i)
        { if (maps[i] == map || maps[i].name == map) {
          maps.splice(i, 1)
          return true
        } }
    },

    addOverlay: methodOp(function(spec, options) {
      var mode = spec.token ? spec : CodeMirror.getMode(this.options, spec)
      if (mode.startState) { throw new Error("Overlays may not be stateful.") }
      insertSorted(this.state.overlays,
                   {mode: mode, modeSpec: spec, opaque: options && options.opaque,
                    priority: (options && options.priority) || 0},
                   function (overlay) { return overlay.priority; })
      this.state.modeGen++
      regChange(this)
    }),
    removeOverlay: methodOp(function(spec) {
      var this$1 = this;

      var overlays = this.state.overlays
      for (var i = 0; i < overlays.length; ++i) {
        var cur = overlays[i].modeSpec
        if (cur == spec || typeof spec == "string" && cur.name == spec) {
          overlays.splice(i, 1)
          this$1.state.modeGen++
          regChange(this$1)
          return
        }
      }
    }),

    indentLine: methodOp(function(n, dir, aggressive) {
      if (typeof dir != "string" && typeof dir != "number") {
        if (dir == null) { dir = this.options.smartIndent ? "smart" : "prev" }
        else { dir = dir ? "add" : "subtract" }
      }
      if (isLine(this.doc, n)) { indentLine(this, n, dir, aggressive) }
    }),
    indentSelection: methodOp(function(how) {
      var this$1 = this;

      var ranges = this.doc.sel.ranges, end = -1
      for (var i = 0; i < ranges.length; i++) {
        var range = ranges[i]
        if (!range.empty()) {
          var from = range.from(), to = range.to()
          var start = Math.max(end, from.line)
          end = Math.min(this$1.lastLine(), to.line - (to.ch ? 0 : 1)) + 1
          for (var j = start; j < end; ++j)
            { indentLine(this$1, j, how) }
          var newRanges = this$1.doc.sel.ranges
          if (from.ch == 0 && ranges.length == newRanges.length && newRanges[i].from().ch > 0)
            { replaceOneSelection(this$1.doc, i, new Range(from, newRanges[i].to()), sel_dontScroll) }
        } else if (range.head.line > end) {
          indentLine(this$1, range.head.line, how, true)
          end = range.head.line
          if (i == this$1.doc.sel.primIndex) { ensureCursorVisible(this$1) }
        }
      }
    }),

    // Fetch the parser token for a given character. Useful for hacks
    // that want to inspect the mode state (say, for completion).
    getTokenAt: function(pos, precise) {
      return takeToken(this, pos, precise)
    },

    getLineTokens: function(line, precise) {
      return takeToken(this, Pos(line), precise, true)
    },

    getTokenTypeAt: function(pos) {
      pos = clipPos(this.doc, pos)
      var styles = getLineStyles(this, getLine(this.doc, pos.line))
      var before = 0, after = (styles.length - 1) / 2, ch = pos.ch
      var type
      if (ch == 0) { type = styles[2] }
      else { for (;;) {
        var mid = (before + after) >> 1
        if ((mid ? styles[mid * 2 - 1] : 0) >= ch) { after = mid }
        else if (styles[mid * 2 + 1] < ch) { before = mid + 1 }
        else { type = styles[mid * 2 + 2]; break }
      } }
      var cut = type ? type.indexOf("overlay ") : -1
      return cut < 0 ? type : cut == 0 ? null : type.slice(0, cut - 1)
    },

    getModeAt: function(pos) {
      var mode = this.doc.mode
      if (!mode.innerMode) { return mode }
      return CodeMirror.innerMode(mode, this.getTokenAt(pos).state).mode
    },

    getHelper: function(pos, type) {
      return this.getHelpers(pos, type)[0]
    },

    getHelpers: function(pos, type) {
      var this$1 = this;

      var found = []
      if (!helpers.hasOwnProperty(type)) { return found }
      var help = helpers[type], mode = this.getModeAt(pos)
      if (typeof mode[type] == "string") {
        if (help[mode[type]]) { found.push(help[mode[type]]) }
      } else if (mode[type]) {
        for (var i = 0; i < mode[type].length; i++) {
          var val = help[mode[type][i]]
          if (val) { found.push(val) }
        }
      } else if (mode.helperType && help[mode.helperType]) {
        found.push(help[mode.helperType])
      } else if (help[mode.name]) {
        found.push(help[mode.name])
      }
      for (var i$1 = 0; i$1 < help._global.length; i$1++) {
        var cur = help._global[i$1]
        if (cur.pred(mode, this$1) && indexOf(found, cur.val) == -1)
          { found.push(cur.val) }
      }
      return found
    },

    getStateAfter: function(line, precise) {
      var doc = this.doc
      line = clipLine(doc, line == null ? doc.first + doc.size - 1: line)
      return getStateBefore(this, line + 1, precise)
    },

    cursorCoords: function(start, mode) {
      var pos, range = this.doc.sel.primary()
      if (start == null) { pos = range.head }
      else if (typeof start == "object") { pos = clipPos(this.doc, start) }
      else { pos = start ? range.from() : range.to() }
      return cursorCoords(this, pos, mode || "page")
    },

    charCoords: function(pos, mode) {
      return charCoords(this, clipPos(this.doc, pos), mode || "page")
    },

    coordsChar: function(coords, mode) {
      coords = fromCoordSystem(this, coords, mode || "page")
      return coordsChar(this, coords.left, coords.top)
    },

    lineAtHeight: function(height, mode) {
      height = fromCoordSystem(this, {top: height, left: 0}, mode || "page").top
      return lineAtHeight(this.doc, height + this.display.viewOffset)
    },
    heightAtLine: function(line, mode, includeWidgets) {
      var end = false, lineObj
      if (typeof line == "number") {
        var last = this.doc.first + this.doc.size - 1
        if (line < this.doc.first) { line = this.doc.first }
        else if (line > last) { line = last; end = true }
        lineObj = getLine(this.doc, line)
      } else {
        lineObj = line
      }
      return intoCoordSystem(this, lineObj, {top: 0, left: 0}, mode || "page", includeWidgets || end).top +
        (end ? this.doc.height - heightAtLine(lineObj) : 0)
    },

    defaultTextHeight: function() { return textHeight(this.display) },
    defaultCharWidth: function() { return charWidth(this.display) },

    getViewport: function() { return {from: this.display.viewFrom, to: this.display.viewTo}},

    addWidget: function(pos, node, scroll, vert, horiz) {
      var display = this.display
      pos = cursorCoords(this, clipPos(this.doc, pos))
      var top = pos.bottom, left = pos.left
      node.style.position = "absolute"
      node.setAttribute("cm-ignore-events", "true")
      this.display.input.setUneditable(node)
      display.sizer.appendChild(node)
      if (vert == "over") {
        top = pos.top
      } else if (vert == "above" || vert == "near") {
        var vspace = Math.max(display.wrapper.clientHeight, this.doc.height),
        hspace = Math.max(display.sizer.clientWidth, display.lineSpace.clientWidth)
        // Default to positioning above (if specified and possible); otherwise default to positioning below
        if ((vert == 'above' || pos.bottom + node.offsetHeight > vspace) && pos.top > node.offsetHeight)
          { top = pos.top - node.offsetHeight }
        else if (pos.bottom + node.offsetHeight <= vspace)
          { top = pos.bottom }
        if (left + node.offsetWidth > hspace)
          { left = hspace - node.offsetWidth }
      }
      node.style.top = top + "px"
      node.style.left = node.style.right = ""
      if (horiz == "right") {
        left = display.sizer.clientWidth - node.offsetWidth
        node.style.right = "0px"
      } else {
        if (horiz == "left") { left = 0 }
        else if (horiz == "middle") { left = (display.sizer.clientWidth - node.offsetWidth) / 2 }
        node.style.left = left + "px"
      }
      if (scroll)
        { scrollIntoView(this, left, top, left + node.offsetWidth, top + node.offsetHeight) }
    },

    triggerOnKeyDown: methodOp(onKeyDown),
    triggerOnKeyPress: methodOp(onKeyPress),
    triggerOnKeyUp: onKeyUp,

    execCommand: function(cmd) {
      if (commands.hasOwnProperty(cmd))
        { return commands[cmd].call(null, this) }
    },

    triggerElectric: methodOp(function(text) { triggerElectric(this, text) }),

    findPosH: function(from, amount, unit, visually) {
      var this$1 = this;

      var dir = 1
      if (amount < 0) { dir = -1; amount = -amount }
      var cur = clipPos(this.doc, from)
      for (var i = 0; i < amount; ++i) {
        cur = findPosH(this$1.doc, cur, dir, unit, visually)
        if (cur.hitSide) { break }
      }
      return cur
    },

    moveH: methodOp(function(dir, unit) {
      var this$1 = this;

      this.extendSelectionsBy(function (range) {
        if (this$1.display.shift || this$1.doc.extend || range.empty())
          { return findPosH(this$1.doc, range.head, dir, unit, this$1.options.rtlMoveVisually) }
        else
          { return dir < 0 ? range.from() : range.to() }
      }, sel_move)
    }),

    deleteH: methodOp(function(dir, unit) {
      var sel = this.doc.sel, doc = this.doc
      if (sel.somethingSelected())
        { doc.replaceSelection("", null, "+delete") }
      else
        { deleteNearSelection(this, function (range) {
          var other = findPosH(doc, range.head, dir, unit, false)
          return dir < 0 ? {from: other, to: range.head} : {from: range.head, to: other}
        }) }
    }),

    findPosV: function(from, amount, unit, goalColumn) {
      var this$1 = this;

      var dir = 1, x = goalColumn
      if (amount < 0) { dir = -1; amount = -amount }
      var cur = clipPos(this.doc, from)
      for (var i = 0; i < amount; ++i) {
        var coords = cursorCoords(this$1, cur, "div")
        if (x == null) { x = coords.left }
        else { coords.left = x }
        cur = findPosV(this$1, coords, dir, unit)
        if (cur.hitSide) { break }
      }
      return cur
    },

    moveV: methodOp(function(dir, unit) {
      var this$1 = this;

      var doc = this.doc, goals = []
      var collapse = !this.display.shift && !doc.extend && doc.sel.somethingSelected()
      doc.extendSelectionsBy(function (range) {
        if (collapse)
          { return dir < 0 ? range.from() : range.to() }
        var headPos = cursorCoords(this$1, range.head, "div")
        if (range.goalColumn != null) { headPos.left = range.goalColumn }
        goals.push(headPos.left)
        var pos = findPosV(this$1, headPos, dir, unit)
        if (unit == "page" && range == doc.sel.primary())
          { addToScrollPos(this$1, null, charCoords(this$1, pos, "div").top - headPos.top) }
        return pos
      }, sel_move)
      if (goals.length) { for (var i = 0; i < doc.sel.ranges.length; i++)
        { doc.sel.ranges[i].goalColumn = goals[i] } }
    }),

    // Find the word at the given position (as returned by coordsChar).
    findWordAt: function(pos) {
      var doc = this.doc, line = getLine(doc, pos.line).text
      var start = pos.ch, end = pos.ch
      if (line) {
        var helper = this.getHelper(pos, "wordChars")
        if ((pos.sticky == "before" || end == line.length) && start) { --start; } else { ++end }
        var startChar = line.charAt(start)
        var check = isWordChar(startChar, helper)
          ? function (ch) { return isWordChar(ch, helper); }
          : /\s/.test(startChar) ? function (ch) { return /\s/.test(ch); }
          : function (ch) { return (!/\s/.test(ch) && !isWordChar(ch)); }
        while (start > 0 && check(line.charAt(start - 1))) { --start }
        while (end < line.length && check(line.charAt(end))) { ++end }
      }
      return new Range(Pos(pos.line, start), Pos(pos.line, end))
    },

    toggleOverwrite: function(value) {
      if (value != null && value == this.state.overwrite) { return }
      if (this.state.overwrite = !this.state.overwrite)
        { addClass(this.display.cursorDiv, "CodeMirror-overwrite") }
      else
        { rmClass(this.display.cursorDiv, "CodeMirror-overwrite") }

      signal(this, "overwriteToggle", this, this.state.overwrite)
    },
    hasFocus: function() { return this.display.input.getField() == activeElt() },
    isReadOnly: function() { return !!(this.options.readOnly || this.doc.cantEdit) },

    scrollTo: methodOp(function(x, y) {
      if (x != null || y != null) { resolveScrollToPos(this) }
      if (x != null) { this.curOp.scrollLeft = x }
      if (y != null) { this.curOp.scrollTop = y }
    }),
    getScrollInfo: function() {
      var scroller = this.display.scroller
      return {left: scroller.scrollLeft, top: scroller.scrollTop,
              height: scroller.scrollHeight - scrollGap(this) - this.display.barHeight,
              width: scroller.scrollWidth - scrollGap(this) - this.display.barWidth,
              clientHeight: displayHeight(this), clientWidth: displayWidth(this)}
    },

    scrollIntoView: methodOp(function(range, margin) {
      if (range == null) {
        range = {from: this.doc.sel.primary().head, to: null}
        if (margin == null) { margin = this.options.cursorScrollMargin }
      } else if (typeof range == "number") {
        range = {from: Pos(range, 0), to: null}
      } else if (range.from == null) {
        range = {from: range, to: null}
      }
      if (!range.to) { range.to = range.from }
      range.margin = margin || 0

      if (range.from.line != null) {
        resolveScrollToPos(this)
        this.curOp.scrollToPos = range
      } else {
        var sPos = calculateScrollPos(this, Math.min(range.from.left, range.to.left),
                                      Math.min(range.from.top, range.to.top) - range.margin,
                                      Math.max(range.from.right, range.to.right),
                                      Math.max(range.from.bottom, range.to.bottom) + range.margin)
        this.scrollTo(sPos.scrollLeft, sPos.scrollTop)
      }
    }),

    setSize: methodOp(function(width, height) {
      var this$1 = this;

      var interpret = function (val) { return typeof val == "number" || /^\d+$/.test(String(val)) ? val + "px" : val; }
      if (width != null) { this.display.wrapper.style.width = interpret(width) }
      if (height != null) { this.display.wrapper.style.height = interpret(height) }
      if (this.options.lineWrapping) { clearLineMeasurementCache(this) }
      var lineNo = this.display.viewFrom
      this.doc.iter(lineNo, this.display.viewTo, function (line) {
        if (line.widgets) { for (var i = 0; i < line.widgets.length; i++)
          { if (line.widgets[i].noHScroll) { regLineChange(this$1, lineNo, "widget"); break } } }
        ++lineNo
      })
      this.curOp.forceUpdate = true
      signal(this, "refresh", this)
    }),

    operation: function(f){return runInOp(this, f)},

    refresh: methodOp(function() {
      var oldHeight = this.display.cachedTextHeight
      regChange(this)
      this.curOp.forceUpdate = true
      clearCaches(this)
      this.scrollTo(this.doc.scrollLeft, this.doc.scrollTop)
      updateGutterSpace(this)
      if (oldHeight == null || Math.abs(oldHeight - textHeight(this.display)) > .5)
        { estimateLineHeights(this) }
      signal(this, "refresh", this)
    }),

    swapDoc: methodOp(function(doc) {
      var old = this.doc
      old.cm = null
      attachDoc(this, doc)
      clearCaches(this)
      this.display.input.reset()
      this.scrollTo(doc.scrollLeft, doc.scrollTop)
      this.curOp.forceScroll = true
      signalLater(this, "swapDoc", this, old)
      return old
    }),

    getInputField: function(){return this.display.input.getField()},
    getWrapperElement: function(){return this.display.wrapper},
    getScrollerElement: function(){return this.display.scroller},
    getGutterElement: function(){return this.display.gutters}
  }
  eventMixin(CodeMirror)

  CodeMirror.registerHelper = function(type, name, value) {
    if (!helpers.hasOwnProperty(type)) { helpers[type] = CodeMirror[type] = {_global: []} }
    helpers[type][name] = value
  }
  CodeMirror.registerGlobalHelper = function(type, name, predicate, value) {
    CodeMirror.registerHelper(type, name, value)
    helpers[type]._global.push({pred: predicate, val: value})
  }
}

// Used for horizontal relative motion. Dir is -1 or 1 (left or
// right), unit can be "char", "column" (like char, but doesn't
// cross line boundaries), "word" (across next word), or "group" (to
// the start of next group of word or non-word-non-whitespace
// chars). The visually param controls whether, in right-to-left
// text, direction 1 means to move towards the next index in the
// string, or towards the character to the right of the current
// position. The resulting position will have a hitSide=true
// property if it reached the end of the document.
function findPosH(doc, pos, dir, unit, visually) {
  var oldPos = pos
  var origDir = dir
  var lineObj = getLine(doc, pos.line)
  function findNextLine() {
    var l = pos.line + dir
    if (l < doc.first || l >= doc.first + doc.size) { return false }
    pos = new Pos(l, pos.ch, pos.sticky)
    return lineObj = getLine(doc, l)
  }
  function moveOnce(boundToLine) {
    var next
    if (visually) {
      next = moveVisually(doc.cm, lineObj, pos, dir)
    } else {
      next = moveLogically(lineObj, pos, dir)
    }
    if (next == null) {
      if (!boundToLine && findNextLine())
        { pos = endOfLine(visually, doc.cm, lineObj, pos.line, dir) }
      else
        { return false }
    } else {
      pos = next
    }
    return true
  }

  if (unit == "char") {
    moveOnce()
  } else if (unit == "column") {
    moveOnce(true)
  } else if (unit == "word" || unit == "group") {
    var sawType = null, group = unit == "group"
    var helper = doc.cm && doc.cm.getHelper(pos, "wordChars")
    for (var first = true;; first = false) {
      if (dir < 0 && !moveOnce(!first)) { break }
      var cur = lineObj.text.charAt(pos.ch) || "\n"
      var type = isWordChar(cur, helper) ? "w"
        : group && cur == "\n" ? "n"
        : !group || /\s/.test(cur) ? null
        : "p"
      if (group && !first && !type) { type = "s" }
      if (sawType && sawType != type) {
        if (dir < 0) {dir = 1; moveOnce(); pos.sticky = "after"}
        break
      }

      if (type) { sawType = type }
      if (dir > 0 && !moveOnce(!first)) { break }
    }
  }
  var result = skipAtomic(doc, pos, oldPos, origDir, true)
  if (equalCursorPos(oldPos, result)) { result.hitSide = true }
  return result
}

// For relative vertical movement. Dir may be -1 or 1. Unit can be
// "page" or "line". The resulting position will have a hitSide=true
// property if it reached the end of the document.
function findPosV(cm, pos, dir, unit) {
  var doc = cm.doc, x = pos.left, y
  if (unit == "page") {
    var pageSize = Math.min(cm.display.wrapper.clientHeight, window.innerHeight || document.documentElement.clientHeight)
    var moveAmount = Math.max(pageSize - .5 * textHeight(cm.display), 3)
    y = (dir > 0 ? pos.bottom : pos.top) + dir * moveAmount

  } else if (unit == "line") {
    y = dir > 0 ? pos.bottom + 3 : pos.top - 3
  }
  var target
  for (;;) {
    target = coordsChar(cm, x, y)
    if (!target.outside) { break }
    if (dir < 0 ? y <= 0 : y >= doc.height) { target.hitSide = true; break }
    y += dir * 5
  }
  return target
}

// CONTENTEDITABLE INPUT STYLE

var ContentEditableInput = function(cm) {
  this.cm = cm
  this.lastAnchorNode = this.lastAnchorOffset = this.lastFocusNode = this.lastFocusOffset = null
  this.polling = new Delayed()
  this.composing = null
  this.gracePeriod = false
  this.readDOMTimeout = null
};

ContentEditableInput.prototype.init = function (display) {
    var this$1 = this;

  var input = this, cm = input.cm
  var div = input.div = display.lineDiv
  disableBrowserMagic(div, cm.options.spellcheck)

  on(div, "paste", function (e) {
    if (signalDOMEvent(cm, e) || handlePaste(e, cm)) { return }
    // IE doesn't fire input events, so we schedule a read for the pasted content in this way
    if (ie_version <= 11) { setTimeout(operation(cm, function () {
      if (!input.pollContent()) { regChange(cm) }
    }), 20) }
  })

  on(div, "compositionstart", function (e) {
    this$1.composing = {data: e.data, done: false}
  })
  on(div, "compositionupdate", function (e) {
    if (!this$1.composing) { this$1.composing = {data: e.data, done: false} }
  })
  on(div, "compositionend", function (e) {
    if (this$1.composing) {
      if (e.data != this$1.composing.data) { this$1.readFromDOMSoon() }
      this$1.composing.done = true
    }
  })

  on(div, "touchstart", function () { return input.forceCompositionEnd(); })

  on(div, "input", function () {
    if (!this$1.composing) { this$1.readFromDOMSoon() }
  })

  function onCopyCut(e) {
    if (signalDOMEvent(cm, e)) { return }
    if (cm.somethingSelected()) {
      setLastCopied({lineWise: false, text: cm.getSelections()})
      if (e.type == "cut") { cm.replaceSelection("", null, "cut") }
    } else if (!cm.options.lineWiseCopyCut) {
      return
    } else {
      var ranges = copyableRanges(cm)
      setLastCopied({lineWise: true, text: ranges.text})
      if (e.type == "cut") {
        cm.operation(function () {
          cm.setSelections(ranges.ranges, 0, sel_dontScroll)
          cm.replaceSelection("", null, "cut")
        })
      }
    }
    if (e.clipboardData) {
      e.clipboardData.clearData()
      var content = lastCopied.text.join("\n")
      // iOS exposes the clipboard API, but seems to discard content inserted into it
      e.clipboardData.setData("Text", content)
      if (e.clipboardData.getData("Text") == content) {
        e.preventDefault()
        return
      }
    }
    // Old-fashioned briefly-focus-a-textarea hack
    var kludge = hiddenTextarea(), te = kludge.firstChild
    cm.display.lineSpace.insertBefore(kludge, cm.display.lineSpace.firstChild)
    te.value = lastCopied.text.join("\n")
    var hadFocus = document.activeElement
    selectInput(te)
    setTimeout(function () {
      cm.display.lineSpace.removeChild(kludge)
      hadFocus.focus()
      if (hadFocus == div) { input.showPrimarySelection() }
    }, 50)
  }
  on(div, "copy", onCopyCut)
  on(div, "cut", onCopyCut)
};

ContentEditableInput.prototype.prepareSelection = function () {
  var result = prepareSelection(this.cm, false)
  result.focus = this.cm.state.focused
  return result
};

ContentEditableInput.prototype.showSelection = function (info, takeFocus) {
  if (!info || !this.cm.display.view.length) { return }
  if (info.focus || takeFocus) { this.showPrimarySelection() }
  this.showMultipleSelections(info)
};

ContentEditableInput.prototype.showPrimarySelection = function () {
  var sel = window.getSelection(), prim = this.cm.doc.sel.primary()
  var curAnchor = domToPos(this.cm, sel.anchorNode, sel.anchorOffset)
  var curFocus = domToPos(this.cm, sel.focusNode, sel.focusOffset)
  if (curAnchor && !curAnchor.bad && curFocus && !curFocus.bad &&
      cmp(minPos(curAnchor, curFocus), prim.from()) == 0 &&
      cmp(maxPos(curAnchor, curFocus), prim.to()) == 0)
    { return }

  var start = posToDOM(this.cm, prim.from())
  var end = posToDOM(this.cm, prim.to())
  if (!start && !end) { return }

  var view = this.cm.display.view
  var old = sel.rangeCount && sel.getRangeAt(0)
  if (!start) {
    start = {node: view[0].measure.map[2], offset: 0}
  } else if (!end) { // FIXME dangerously hacky
    var measure = view[view.length - 1].measure
    var map = measure.maps ? measure.maps[measure.maps.length - 1] : measure.map
    end = {node: map[map.length - 1], offset: map[map.length - 2] - map[map.length - 3]}
  }

  var rng
  try { rng = range(start.node, start.offset, end.offset, end.node) }
  catch(e) {} // Our model of the DOM might be outdated, in which case the range we try to set can be impossible
  if (rng) {
    if (!gecko && this.cm.state.focused) {
      sel.collapse(start.node, start.offset)
      if (!rng.collapsed) {
        sel.removeAllRanges()
        sel.addRange(rng)
      }
    } else {
      sel.removeAllRanges()
      sel.addRange(rng)
    }
    if (old && sel.anchorNode == null) { sel.addRange(old) }
    else if (gecko) { this.startGracePeriod() }
  }
  this.rememberSelection()
};

ContentEditableInput.prototype.startGracePeriod = function () {
    var this$1 = this;

  clearTimeout(this.gracePeriod)
  this.gracePeriod = setTimeout(function () {
    this$1.gracePeriod = false
    if (this$1.selectionChanged())
      { this$1.cm.operation(function () { return this$1.cm.curOp.selectionChanged = true; }) }
  }, 20)
};

ContentEditableInput.prototype.showMultipleSelections = function (info) {
  removeChildrenAndAdd(this.cm.display.cursorDiv, info.cursors)
  removeChildrenAndAdd(this.cm.display.selectionDiv, info.selection)
};

ContentEditableInput.prototype.rememberSelection = function () {
  var sel = window.getSelection()
  this.lastAnchorNode = sel.anchorNode; this.lastAnchorOffset = sel.anchorOffset
  this.lastFocusNode = sel.focusNode; this.lastFocusOffset = sel.focusOffset
};

ContentEditableInput.prototype.selectionInEditor = function () {
  var sel = window.getSelection()
  if (!sel.rangeCount) { return false }
  var node = sel.getRangeAt(0).commonAncestorContainer
  return contains(this.div, node)
};

ContentEditableInput.prototype.focus = function () {
  if (this.cm.options.readOnly != "nocursor") {
    if (!this.selectionInEditor())
      { this.showSelection(this.prepareSelection(), true) }
    this.div.focus()
  }
};
ContentEditableInput.prototype.blur = function () { this.div.blur() };
ContentEditableInput.prototype.getField = function () { return this.div };

ContentEditableInput.prototype.supportsTouch = function () { return true };

ContentEditableInput.prototype.receivedFocus = function () {
  var input = this
  if (this.selectionInEditor())
    { this.pollSelection() }
  else
    { runInOp(this.cm, function () { return input.cm.curOp.selectionChanged = true; }) }

  function poll() {
    if (input.cm.state.focused) {
      input.pollSelection()
      input.polling.set(input.cm.options.pollInterval, poll)
    }
  }
  this.polling.set(this.cm.options.pollInterval, poll)
};

ContentEditableInput.prototype.selectionChanged = function () {
  var sel = window.getSelection()
  return sel.anchorNode != this.lastAnchorNode || sel.anchorOffset != this.lastAnchorOffset ||
    sel.focusNode != this.lastFocusNode || sel.focusOffset != this.lastFocusOffset
};

ContentEditableInput.prototype.pollSelection = function () {
  if (!this.composing && this.readDOMTimeout == null && !this.gracePeriod && this.selectionChanged()) {
    var sel = window.getSelection(), cm = this.cm
    this.rememberSelection()
    var anchor = domToPos(cm, sel.anchorNode, sel.anchorOffset)
    var head = domToPos(cm, sel.focusNode, sel.focusOffset)
    if (anchor && head) { runInOp(cm, function () {
      setSelection(cm.doc, simpleSelection(anchor, head), sel_dontScroll)
      if (anchor.bad || head.bad) { cm.curOp.selectionChanged = true }
    }) }
  }
};

ContentEditableInput.prototype.pollContent = function () {
  if (this.readDOMTimeout != null) {
    clearTimeout(this.readDOMTimeout)
    this.readDOMTimeout = null
  }

  var cm = this.cm, display = cm.display, sel = cm.doc.sel.primary()
  var from = sel.from(), to = sel.to()
  if (from.ch == 0 && from.line > cm.firstLine())
    { from = Pos(from.line - 1, getLine(cm.doc, from.line - 1).length) }
  if (to.ch == getLine(cm.doc, to.line).text.length && to.line < cm.lastLine())
    { to = Pos(to.line + 1, 0) }
  if (from.line < display.viewFrom || to.line > display.viewTo - 1) { return false }

  var fromIndex, fromLine, fromNode
  if (from.line == display.viewFrom || (fromIndex = findViewIndex(cm, from.line)) == 0) {
    fromLine = lineNo(display.view[0].line)
    fromNode = display.view[0].node
  } else {
    fromLine = lineNo(display.view[fromIndex].line)
    fromNode = display.view[fromIndex - 1].node.nextSibling
  }
  var toIndex = findViewIndex(cm, to.line)
  var toLine, toNode
  if (toIndex == display.view.length - 1) {
    toLine = display.viewTo - 1
    toNode = display.lineDiv.lastChild
  } else {
    toLine = lineNo(display.view[toIndex + 1].line) - 1
    toNode = display.view[toIndex + 1].node.previousSibling
  }

  if (!fromNode) { return false }
  var newText = cm.doc.splitLines(domTextBetween(cm, fromNode, toNode, fromLine, toLine))
  var oldText = getBetween(cm.doc, Pos(fromLine, 0), Pos(toLine, getLine(cm.doc, toLine).text.length))
  while (newText.length > 1 && oldText.length > 1) {
    if (lst(newText) == lst(oldText)) { newText.pop(); oldText.pop(); toLine-- }
    else if (newText[0] == oldText[0]) { newText.shift(); oldText.shift(); fromLine++ }
    else { break }
  }

  var cutFront = 0, cutEnd = 0
  var newTop = newText[0], oldTop = oldText[0], maxCutFront = Math.min(newTop.length, oldTop.length)
  while (cutFront < maxCutFront && newTop.charCodeAt(cutFront) == oldTop.charCodeAt(cutFront))
    { ++cutFront }
  var newBot = lst(newText), oldBot = lst(oldText)
  var maxCutEnd = Math.min(newBot.length - (newText.length == 1 ? cutFront : 0),
                           oldBot.length - (oldText.length == 1 ? cutFront : 0))
  while (cutEnd < maxCutEnd &&
         newBot.charCodeAt(newBot.length - cutEnd - 1) == oldBot.charCodeAt(oldBot.length - cutEnd - 1))
    { ++cutEnd }

  newText[newText.length - 1] = newBot.slice(0, newBot.length - cutEnd).replace(/^\u200b+/, "")
  newText[0] = newText[0].slice(cutFront).replace(/\u200b+$/, "")

  var chFrom = Pos(fromLine, cutFront)
  var chTo = Pos(toLine, oldText.length ? lst(oldText).length - cutEnd : 0)
  if (newText.length > 1 || newText[0] || cmp(chFrom, chTo)) {
    replaceRange(cm.doc, newText, chFrom, chTo, "+input")
    return true
  }
};

ContentEditableInput.prototype.ensurePolled = function () {
  this.forceCompositionEnd()
};
ContentEditableInput.prototype.reset = function () {
  this.forceCompositionEnd()
};
ContentEditableInput.prototype.forceCompositionEnd = function () {
  if (!this.composing) { return }
  clearTimeout(this.readDOMTimeout)
  this.composing = null
  if (!this.pollContent()) { regChange(this.cm) }
  this.div.blur()
  this.div.focus()
};
ContentEditableInput.prototype.readFromDOMSoon = function () {
    var this$1 = this;

  if (this.readDOMTimeout != null) { return }
  this.readDOMTimeout = setTimeout(function () {
    this$1.readDOMTimeout = null
    if (this$1.composing) {
      if (this$1.composing.done) { this$1.composing = null }
      else { return }
    }
    if (this$1.cm.isReadOnly() || !this$1.pollContent())
      { runInOp(this$1.cm, function () { return regChange(this$1.cm); }) }
  }, 80)
};

ContentEditableInput.prototype.setUneditable = function (node) {
  node.contentEditable = "false"
};

ContentEditableInput.prototype.onKeyPress = function (e) {
  if (e.charCode == 0) { return }
  e.preventDefault()
  if (!this.cm.isReadOnly())
    { operation(this.cm, applyTextInput)(this.cm, String.fromCharCode(e.charCode == null ? e.keyCode : e.charCode), 0) }
};

ContentEditableInput.prototype.readOnlyChanged = function (val) {
  this.div.contentEditable = String(val != "nocursor")
};

ContentEditableInput.prototype.onContextMenu = function () {};
ContentEditableInput.prototype.resetPosition = function () {};

ContentEditableInput.prototype.needsContentAttribute = true

function posToDOM(cm, pos) {
  var view = findViewForLine(cm, pos.line)
  if (!view || view.hidden) { return null }
  var line = getLine(cm.doc, pos.line)
  var info = mapFromLineView(view, line, pos.line)

  var order = getOrder(line), side = "left"
  if (order) {
    var partPos = getBidiPartAt(order, pos.ch)
    side = partPos % 2 ? "right" : "left"
  }
  var result = nodeAndOffsetInLineMap(info.map, pos.ch, side)
  result.offset = result.collapse == "right" ? result.end : result.start
  return result
}

function badPos(pos, bad) { if (bad) { pos.bad = true; } return pos }

function domTextBetween(cm, from, to, fromLine, toLine) {
  var text = "", closing = false, lineSep = cm.doc.lineSeparator()
  function recognizeMarker(id) { return function (marker) { return marker.id == id; } }
  function walk(node) {
    if (node.nodeType == 1) {
      var cmText = node.getAttribute("cm-text")
      if (cmText != null) {
        if (cmText == "") { text += node.textContent.replace(/\u200b/g, "") }
        else { text += cmText }
        return
      }
      var markerID = node.getAttribute("cm-marker"), range
      if (markerID) {
        var found = cm.findMarks(Pos(fromLine, 0), Pos(toLine + 1, 0), recognizeMarker(+markerID))
        if (found.length && (range = found[0].find()))
          { text += getBetween(cm.doc, range.from, range.to).join(lineSep) }
        return
      }
      if (node.getAttribute("contenteditable") == "false") { return }
      for (var i = 0; i < node.childNodes.length; i++)
        { walk(node.childNodes[i]) }
      if (/^(pre|div|p)$/i.test(node.nodeName))
        { closing = true }
    } else if (node.nodeType == 3) {
      var val = node.nodeValue
      if (!val) { return }
      if (closing) {
        text += lineSep
        closing = false
      }
      text += val
    }
  }
  for (;;) {
    walk(from)
    if (from == to) { break }
    from = from.nextSibling
  }
  return text
}

function domToPos(cm, node, offset) {
  var lineNode
  if (node == cm.display.lineDiv) {
    lineNode = cm.display.lineDiv.childNodes[offset]
    if (!lineNode) { return badPos(cm.clipPos(Pos(cm.display.viewTo - 1)), true) }
    node = null; offset = 0
  } else {
    for (lineNode = node;; lineNode = lineNode.parentNode) {
      if (!lineNode || lineNode == cm.display.lineDiv) { return null }
      if (lineNode.parentNode && lineNode.parentNode == cm.display.lineDiv) { break }
    }
  }
  for (var i = 0; i < cm.display.view.length; i++) {
    var lineView = cm.display.view[i]
    if (lineView.node == lineNode)
      { return locateNodeInLineView(lineView, node, offset) }
  }
}

function locateNodeInLineView(lineView, node, offset) {
  var wrapper = lineView.text.firstChild, bad = false
  if (!node || !contains(wrapper, node)) { return badPos(Pos(lineNo(lineView.line), 0), true) }
  if (node == wrapper) {
    bad = true
    node = wrapper.childNodes[offset]
    offset = 0
    if (!node) {
      var line = lineView.rest ? lst(lineView.rest) : lineView.line
      return badPos(Pos(lineNo(line), line.text.length), bad)
    }
  }

  var textNode = node.nodeType == 3 ? node : null, topNode = node
  if (!textNode && node.childNodes.length == 1 && node.firstChild.nodeType == 3) {
    textNode = node.firstChild
    if (offset) { offset = textNode.nodeValue.length }
  }
  while (topNode.parentNode != wrapper) { topNode = topNode.parentNode }
  var measure = lineView.measure, maps = measure.maps

  function find(textNode, topNode, offset) {
    for (var i = -1; i < (maps ? maps.length : 0); i++) {
      var map = i < 0 ? measure.map : maps[i]
      for (var j = 0; j < map.length; j += 3) {
        var curNode = map[j + 2]
        if (curNode == textNode || curNode == topNode) {
          var line = lineNo(i < 0 ? lineView.line : lineView.rest[i])
          var ch = map[j] + offset
          if (offset < 0 || curNode != textNode) { ch = map[j + (offset ? 1 : 0)] }
          return Pos(line, ch)
        }
      }
    }
  }
  var found = find(textNode, topNode, offset)
  if (found) { return badPos(found, bad) }

  // FIXME this is all really shaky. might handle the few cases it needs to handle, but likely to cause problems
  for (var after = topNode.nextSibling, dist = textNode ? textNode.nodeValue.length - offset : 0; after; after = after.nextSibling) {
    found = find(after, after.firstChild, 0)
    if (found)
      { return badPos(Pos(found.line, found.ch - dist), bad) }
    else
      { dist += after.textContent.length }
  }
  for (var before = topNode.previousSibling, dist$1 = offset; before; before = before.previousSibling) {
    found = find(before, before.firstChild, -1)
    if (found)
      { return badPos(Pos(found.line, found.ch + dist$1), bad) }
    else
      { dist$1 += before.textContent.length }
  }
}

// TEXTAREA INPUT STYLE

var TextareaInput = function(cm) {
  this.cm = cm
  // See input.poll and input.reset
  this.prevInput = ""

  // Flag that indicates whether we expect input to appear real soon
  // now (after some event like 'keypress' or 'input') and are
  // polling intensively.
  this.pollingFast = false
  // Self-resetting timeout for the poller
  this.polling = new Delayed()
  // Tracks when input.reset has punted to just putting a short
  // string into the textarea instead of the full selection.
  this.inaccurateSelection = false
  // Used to work around IE issue with selection being forgotten when focus moves away from textarea
  this.hasSelection = false
  this.composing = null
};

TextareaInput.prototype.init = function (display) {
    var this$1 = this;

  var input = this, cm = this.cm

  // Wraps and hides input textarea
  var div = this.wrapper = hiddenTextarea()
  // The semihidden textarea that is focused when the editor is
  // focused, and receives input.
  var te = this.textarea = div.firstChild
  display.wrapper.insertBefore(div, display.wrapper.firstChild)

  // Needed to hide big blue blinking cursor on Mobile Safari (doesn't seem to work in iOS 8 anymore)
  if (ios) { te.style.width = "0px" }

  on(te, "input", function () {
    if (ie && ie_version >= 9 && this$1.hasSelection) { this$1.hasSelection = null }
    input.poll()
  })

  on(te, "paste", function (e) {
    if (signalDOMEvent(cm, e) || handlePaste(e, cm)) { return }

    cm.state.pasteIncoming = true
    input.fastPoll()
  })

  function prepareCopyCut(e) {
    if (signalDOMEvent(cm, e)) { return }
    if (cm.somethingSelected()) {
      setLastCopied({lineWise: false, text: cm.getSelections()})
      if (input.inaccurateSelection) {
        input.prevInput = ""
        input.inaccurateSelection = false
        te.value = lastCopied.text.join("\n")
        selectInput(te)
      }
    } else if (!cm.options.lineWiseCopyCut) {
      return
    } else {
      var ranges = copyableRanges(cm)
      setLastCopied({lineWise: true, text: ranges.text})
      if (e.type == "cut") {
        cm.setSelections(ranges.ranges, null, sel_dontScroll)
      } else {
        input.prevInput = ""
        te.value = ranges.text.join("\n")
        selectInput(te)
      }
    }
    if (e.type == "cut") { cm.state.cutIncoming = true }
  }
  on(te, "cut", prepareCopyCut)
  on(te, "copy", prepareCopyCut)

  on(display.scroller, "paste", function (e) {
    if (eventInWidget(display, e) || signalDOMEvent(cm, e)) { return }
    cm.state.pasteIncoming = true
    input.focus()
  })

  // Prevent normal selection in the editor (we handle our own)
  on(display.lineSpace, "selectstart", function (e) {
    if (!eventInWidget(display, e)) { e_preventDefault(e) }
  })

  on(te, "compositionstart", function () {
    var start = cm.getCursor("from")
    if (input.composing) { input.composing.range.clear() }
    input.composing = {
      start: start,
      range: cm.markText(start, cm.getCursor("to"), {className: "CodeMirror-composing"})
    }
  })
  on(te, "compositionend", function () {
    if (input.composing) {
      input.poll()
      input.composing.range.clear()
      input.composing = null
    }
  })
};

TextareaInput.prototype.prepareSelection = function () {
  // Redraw the selection and/or cursor
  var cm = this.cm, display = cm.display, doc = cm.doc
  var result = prepareSelection(cm)

  // Move the hidden textarea near the cursor to prevent scrolling artifacts
  if (cm.options.moveInputWithCursor) {
    var headPos = cursorCoords(cm, doc.sel.primary().head, "div")
    var wrapOff = display.wrapper.getBoundingClientRect(), lineOff = display.lineDiv.getBoundingClientRect()
    result.teTop = Math.max(0, Math.min(display.wrapper.clientHeight - 10,
                                        headPos.top + lineOff.top - wrapOff.top))
    result.teLeft = Math.max(0, Math.min(display.wrapper.clientWidth - 10,
                                         headPos.left + lineOff.left - wrapOff.left))
  }

  return result
};

TextareaInput.prototype.showSelection = function (drawn) {
  var cm = this.cm, display = cm.display
  removeChildrenAndAdd(display.cursorDiv, drawn.cursors)
  removeChildrenAndAdd(display.selectionDiv, drawn.selection)
  if (drawn.teTop != null) {
    this.wrapper.style.top = drawn.teTop + "px"
    this.wrapper.style.left = drawn.teLeft + "px"
  }
};

// Reset the input to correspond to the selection (or to be empty,
// when not typing and nothing is selected)
TextareaInput.prototype.reset = function (typing) {
  if (this.contextMenuPending) { return }
  var minimal, selected, cm = this.cm, doc = cm.doc
  if (cm.somethingSelected()) {
    this.prevInput = ""
    var range = doc.sel.primary()
    minimal = hasCopyEvent &&
      (range.to().line - range.from().line > 100 || (selected = cm.getSelection()).length > 1000)
    var content = minimal ? "-" : selected || cm.getSelection()
    this.textarea.value = content
    if (cm.state.focused) { selectInput(this.textarea) }
    if (ie && ie_version >= 9) { this.hasSelection = content }
  } else if (!typing) {
    this.prevInput = this.textarea.value = ""
    if (ie && ie_version >= 9) { this.hasSelection = null }
  }
  this.inaccurateSelection = minimal
};

TextareaInput.prototype.getField = function () { return this.textarea };

TextareaInput.prototype.supportsTouch = function () { return false };

TextareaInput.prototype.focus = function () {
  if (this.cm.options.readOnly != "nocursor" && (!mobile || activeElt() != this.textarea)) {
    try { this.textarea.focus() }
    catch (e) {} // IE8 will throw if the textarea is display: none or not in DOM
  }
};

TextareaInput.prototype.blur = function () { this.textarea.blur() };

TextareaInput.prototype.resetPosition = function () {
  this.wrapper.style.top = this.wrapper.style.left = 0
};

TextareaInput.prototype.receivedFocus = function () { this.slowPoll() };

// Poll for input changes, using the normal rate of polling. This
// runs as long as the editor is focused.
TextareaInput.prototype.slowPoll = function () {
    var this$1 = this;

  if (this.pollingFast) { return }
  this.polling.set(this.cm.options.pollInterval, function () {
    this$1.poll()
    if (this$1.cm.state.focused) { this$1.slowPoll() }
  })
};

// When an event has just come in that is likely to add or change
// something in the input textarea, we poll faster, to ensure that
// the change appears on the screen quickly.
TextareaInput.prototype.fastPoll = function () {
  var missed = false, input = this
  input.pollingFast = true
  function p() {
    var changed = input.poll()
    if (!changed && !missed) {missed = true; input.polling.set(60, p)}
    else {input.pollingFast = false; input.slowPoll()}
  }
  input.polling.set(20, p)
};

// Read input from the textarea, and update the document to match.
// When something is selected, it is present in the textarea, and
// selected (unless it is huge, in which case a placeholder is
// used). When nothing is selected, the cursor sits after previously
// seen text (can be empty), which is stored in prevInput (we must
// not reset the textarea when typing, because that breaks IME).
TextareaInput.prototype.poll = function () {
    var this$1 = this;

  var cm = this.cm, input = this.textarea, prevInput = this.prevInput
  // Since this is called a *lot*, try to bail out as cheaply as
  // possible when it is clear that nothing happened. hasSelection
  // will be the case when there is a lot of text in the textarea,
  // in which case reading its value would be expensive.
  if (this.contextMenuPending || !cm.state.focused ||
      (hasSelection(input) && !prevInput && !this.composing) ||
      cm.isReadOnly() || cm.options.disableInput || cm.state.keySeq)
    { return false }

  var text = input.value
  // If nothing changed, bail.
  if (text == prevInput && !cm.somethingSelected()) { return false }
  // Work around nonsensical selection resetting in IE9/10, and
  // inexplicable appearance of private area unicode characters on
  // some key combos in Mac (#2689).
  if (ie && ie_version >= 9 && this.hasSelection === text ||
      mac && /[\uf700-\uf7ff]/.test(text)) {
    cm.display.input.reset()
    return false
  }

  if (cm.doc.sel == cm.display.selForContextMenu) {
    var first = text.charCodeAt(0)
    if (first == 0x200b && !prevInput) { prevInput = "\u200b" }
    if (first == 0x21da) { this.reset(); return this.cm.execCommand("undo") }
  }
  // Find the part of the input that is actually new
  var same = 0, l = Math.min(prevInput.length, text.length)
  while (same < l && prevInput.charCodeAt(same) == text.charCodeAt(same)) { ++same }

  runInOp(cm, function () {
    applyTextInput(cm, text.slice(same), prevInput.length - same,
                   null, this$1.composing ? "*compose" : null)

    // Don't leave long text in the textarea, since it makes further polling slow
    if (text.length > 1000 || text.indexOf("\n") > -1) { input.value = this$1.prevInput = "" }
    else { this$1.prevInput = text }

    if (this$1.composing) {
      this$1.composing.range.clear()
      this$1.composing.range = cm.markText(this$1.composing.start, cm.getCursor("to"),
                                         {className: "CodeMirror-composing"})
    }
  })
  return true
};

TextareaInput.prototype.ensurePolled = function () {
  if (this.pollingFast && this.poll()) { this.pollingFast = false }
};

TextareaInput.prototype.onKeyPress = function () {
  if (ie && ie_version >= 9) { this.hasSelection = null }
  this.fastPoll()
};

TextareaInput.prototype.onContextMenu = function (e) {
  var input = this, cm = input.cm, display = cm.display, te = input.textarea
  var pos = posFromMouse(cm, e), scrollPos = display.scroller.scrollTop
  if (!pos || presto) { return } // Opera is difficult.

  // Reset the current text selection only if the click is done outside of the selection
  // and 'resetSelectionOnContextMenu' option is true.
  var reset = cm.options.resetSelectionOnContextMenu
  if (reset && cm.doc.sel.contains(pos) == -1)
    { operation(cm, setSelection)(cm.doc, simpleSelection(pos), sel_dontScroll) }

  var oldCSS = te.style.cssText, oldWrapperCSS = input.wrapper.style.cssText
  input.wrapper.style.cssText = "position: absolute"
  var wrapperBox = input.wrapper.getBoundingClientRect()
  te.style.cssText = "position: absolute; width: 30px; height: 30px;\n      top: " + (e.clientY - wrapperBox.top - 5) + "px; left: " + (e.clientX - wrapperBox.left - 5) + "px;\n      z-index: 1000; background: " + (ie ? "rgba(255, 255, 255, .05)" : "transparent") + ";\n      outline: none; border-width: 0; outline: none; overflow: hidden; opacity: .05; filter: alpha(opacity=5);"
  var oldScrollY
  if (webkit) { oldScrollY = window.scrollY } // Work around Chrome issue (#2712)
  display.input.focus()
  if (webkit) { window.scrollTo(null, oldScrollY) }
  display.input.reset()
  // Adds "Select all" to context menu in FF
  if (!cm.somethingSelected()) { te.value = input.prevInput = " " }
  input.contextMenuPending = true
  display.selForContextMenu = cm.doc.sel
  clearTimeout(display.detectingSelectAll)

  // Select-all will be greyed out if there's nothing to select, so
  // this adds a zero-width space so that we can later check whether
  // it got selected.
  function prepareSelectAllHack() {
    if (te.selectionStart != null) {
      var selected = cm.somethingSelected()
      var extval = "\u200b" + (selected ? te.value : "")
      te.value = "\u21da" // Used to catch context-menu undo
      te.value = extval
      input.prevInput = selected ? "" : "\u200b"
      te.selectionStart = 1; te.selectionEnd = extval.length
      // Re-set this, in case some other handler touched the
      // selection in the meantime.
      display.selForContextMenu = cm.doc.sel
    }
  }
  function rehide() {
    input.contextMenuPending = false
    input.wrapper.style.cssText = oldWrapperCSS
    te.style.cssText = oldCSS
    if (ie && ie_version < 9) { display.scrollbars.setScrollTop(display.scroller.scrollTop = scrollPos) }

    // Try to detect the user choosing select-all
    if (te.selectionStart != null) {
      if (!ie || (ie && ie_version < 9)) { prepareSelectAllHack() }
      var i = 0, poll = function () {
        if (display.selForContextMenu == cm.doc.sel && te.selectionStart == 0 &&
            te.selectionEnd > 0 && input.prevInput == "\u200b") {
          operation(cm, selectAll)(cm)
        } else if (i++ < 10) {
          display.detectingSelectAll = setTimeout(poll, 500)
        } else {
          display.selForContextMenu = null
          display.input.reset()
        }
      }
      display.detectingSelectAll = setTimeout(poll, 200)
    }
  }

  if (ie && ie_version >= 9) { prepareSelectAllHack() }
  if (captureRightClick) {
    e_stop(e)
    var mouseup = function () {
      off(window, "mouseup", mouseup)
      setTimeout(rehide, 20)
    }
    on(window, "mouseup", mouseup)
  } else {
    setTimeout(rehide, 50)
  }
};

TextareaInput.prototype.readOnlyChanged = function (val) {
  if (!val) { this.reset() }
};

TextareaInput.prototype.setUneditable = function () {};

TextareaInput.prototype.needsContentAttribute = false

function fromTextArea(textarea, options) {
  options = options ? copyObj(options) : {}
  options.value = textarea.value
  if (!options.tabindex && textarea.tabIndex)
    { options.tabindex = textarea.tabIndex }
  if (!options.placeholder && textarea.placeholder)
    { options.placeholder = textarea.placeholder }
  // Set autofocus to true if this textarea is focused, or if it has
  // autofocus and no other element is focused.
  if (options.autofocus == null) {
    var hasFocus = activeElt()
    options.autofocus = hasFocus == textarea ||
      textarea.getAttribute("autofocus") != null && hasFocus == document.body
  }

  function save() {textarea.value = cm.getValue()}

  var realSubmit
  if (textarea.form) {
    on(textarea.form, "submit", save)
    // Deplorable hack to make the submit method do the right thing.
    if (!options.leaveSubmitMethodAlone) {
      var form = textarea.form
      realSubmit = form.submit
      try {
        var wrappedSubmit = form.submit = function () {
          save()
          form.submit = realSubmit
          form.submit()
          form.submit = wrappedSubmit
        }
      } catch(e) {}
    }
  }

  options.finishInit = function (cm) {
    cm.save = save
    cm.getTextArea = function () { return textarea; }
    cm.toTextArea = function () {
      cm.toTextArea = isNaN // Prevent this from being ran twice
      save()
      textarea.parentNode.removeChild(cm.getWrapperElement())
      textarea.style.display = ""
      if (textarea.form) {
        off(textarea.form, "submit", save)
        if (typeof textarea.form.submit == "function")
          { textarea.form.submit = realSubmit }
      }
    }
  }

  textarea.style.display = "none"
  var cm = CodeMirror(function (node) { return textarea.parentNode.insertBefore(node, textarea.nextSibling); },
    options)
  return cm
}

function addLegacyProps(CodeMirror) {
  CodeMirror.off = off
  CodeMirror.on = on
  CodeMirror.wheelEventPixels = wheelEventPixels
  CodeMirror.Doc = Doc
  CodeMirror.splitLines = splitLinesAuto
  CodeMirror.countColumn = countColumn
  CodeMirror.findColumn = findColumn
  CodeMirror.isWordChar = isWordCharBasic
  CodeMirror.Pass = Pass
  CodeMirror.signal = signal
  CodeMirror.Line = Line
  CodeMirror.changeEnd = changeEnd
  CodeMirror.scrollbarModel = scrollbarModel
  CodeMirror.Pos = Pos
  CodeMirror.cmpPos = cmp
  CodeMirror.modes = modes
  CodeMirror.mimeModes = mimeModes
  CodeMirror.resolveMode = resolveMode
  CodeMirror.getMode = getMode
  CodeMirror.modeExtensions = modeExtensions
  CodeMirror.extendMode = extendMode
  CodeMirror.copyState = copyState
  CodeMirror.startState = startState
  CodeMirror.innerMode = innerMode
  CodeMirror.commands = commands
  CodeMirror.keyMap = keyMap
  CodeMirror.keyName = keyName
  CodeMirror.isModifierKey = isModifierKey
  CodeMirror.lookupKey = lookupKey
  CodeMirror.normalizeKeyMap = normalizeKeyMap
  CodeMirror.StringStream = StringStream
  CodeMirror.SharedTextMarker = SharedTextMarker
  CodeMirror.TextMarker = TextMarker
  CodeMirror.LineWidget = LineWidget
  CodeMirror.e_preventDefault = e_preventDefault
  CodeMirror.e_stopPropagation = e_stopPropagation
  CodeMirror.e_stop = e_stop
  CodeMirror.addClass = addClass
  CodeMirror.contains = contains
  CodeMirror.rmClass = rmClass
  CodeMirror.keyNames = keyNames
}

// EDITOR CONSTRUCTOR

defineOptions(CodeMirror)

addEditorMethods(CodeMirror)

// Set up methods on CodeMirror's prototype to redirect to the editor's document.
var dontDelegate = "iter insert remove copy getEditor constructor".split(" ")
for (var prop in Doc.prototype) { if (Doc.prototype.hasOwnProperty(prop) && indexOf(dontDelegate, prop) < 0)
  { CodeMirror.prototype[prop] = (function(method) {
    return function() {return method.apply(this.doc, arguments)}
  })(Doc.prototype[prop]) } }

eventMixin(Doc)

// INPUT HANDLING

CodeMirror.inputStyles = {"textarea": TextareaInput, "contenteditable": ContentEditableInput}

// MODE DEFINITION AND QUERYING

// Extra arguments are stored as the mode's dependencies, which is
// used by (legacy) mechanisms like loadmode.js to automatically
// load a mode. (Preferred mechanism is the require/define calls.)
CodeMirror.defineMode = function(name/*, mode, …*/) {
  if (!CodeMirror.defaults.mode && name != "null") { CodeMirror.defaults.mode = name }
  defineMode.apply(this, arguments)
}

CodeMirror.defineMIME = defineMIME

// Minimal default mode.
CodeMirror.defineMode("null", function () { return ({token: function (stream) { return stream.skipToEnd(); }}); })
CodeMirror.defineMIME("text/plain", "null")

// EXTENSIONS

CodeMirror.defineExtension = function (name, func) {
  CodeMirror.prototype[name] = func
}
CodeMirror.defineDocExtension = function (name, func) {
  Doc.prototype[name] = func
}

CodeMirror.fromTextArea = fromTextArea

addLegacyProps(CodeMirror)

CodeMirror.version = "5.24.0"

return CodeMirror;

})));
var CLOSURE_NO_DEPS = true;
var COMPILED = false;
var goog = goog || {};
goog.global = this;
goog.global.CLOSURE_UNCOMPILED_DEFINES;
goog.global.CLOSURE_DEFINES;
goog.isDef = function(val) {
  return val !== void 0;
};
goog.exportPath_ = function(name, opt_object, opt_objectToExportTo) {
  var parts = name.split(".");
  var cur = opt_objectToExportTo || goog.global;
  if (!(parts[0] in cur) && cur.execScript) {
    cur.execScript("var " + parts[0]);
  }
  for (var part;parts.length && (part = parts.shift());) {
    if (!parts.length && goog.isDef(opt_object)) {
      cur[part] = opt_object;
    } else {
      if (cur[part]) {
        cur = cur[part];
      } else {
        cur = cur[part] = {};
      }
    }
  }
};
goog.define = function(name, defaultValue) {
  var value = defaultValue;
  if (!COMPILED) {
    if (goog.global.CLOSURE_UNCOMPILED_DEFINES && Object.prototype.hasOwnProperty.call(goog.global.CLOSURE_UNCOMPILED_DEFINES, name)) {
      value = goog.global.CLOSURE_UNCOMPILED_DEFINES[name];
    } else {
      if (goog.global.CLOSURE_DEFINES && Object.prototype.hasOwnProperty.call(goog.global.CLOSURE_DEFINES, name)) {
        value = goog.global.CLOSURE_DEFINES[name];
      }
    }
  }
  goog.exportPath_(name, value);
};
goog.define("goog.DEBUG", true);
goog.define("goog.LOCALE", "en");
goog.define("goog.TRUSTED_SITE", true);
goog.define("goog.STRICT_MODE_COMPATIBLE", false);
goog.define("goog.DISALLOW_TEST_ONLY_CODE", COMPILED && !goog.DEBUG);
goog.define("goog.ENABLE_CHROME_APP_SAFE_SCRIPT_LOADING", false);
goog.provide = function(name) {
  if (goog.isInModuleLoader_()) {
    throw Error("goog.provide can not be used within a goog.module.");
  }
  if (!COMPILED) {
    if (goog.isProvided_(name)) {
      throw Error('Namespace "' + name + '" already declared.');
    }
  }
  goog.constructNamespace_(name);
};
goog.constructNamespace_ = function(name, opt_obj) {
  if (!COMPILED) {
    delete goog.implicitNamespaces_[name];
    var namespace = name;
    while (namespace = namespace.substring(0, namespace.lastIndexOf("."))) {
      if (goog.getObjectByName(namespace)) {
        break;
      }
      goog.implicitNamespaces_[namespace] = true;
    }
  }
  goog.exportPath_(name, opt_obj);
};
goog.VALID_MODULE_RE_ = /^[a-zA-Z_$][a-zA-Z0-9._$]*$/;
goog.module = function(name) {
  if (!goog.isString(name) || !name || name.search(goog.VALID_MODULE_RE_) == -1) {
    throw Error("Invalid module identifier");
  }
  if (!goog.isInModuleLoader_()) {
    throw Error("Module " + name + " has been loaded incorrectly.");
  }
  if (goog.moduleLoaderState_.moduleName) {
    throw Error("goog.module may only be called once per module.");
  }
  goog.moduleLoaderState_.moduleName = name;
  if (!COMPILED) {
    if (goog.isProvided_(name)) {
      throw Error('Namespace "' + name + '" already declared.');
    }
    delete goog.implicitNamespaces_[name];
  }
};
goog.module.get = function(name) {
  return goog.module.getInternal_(name);
};
goog.module.getInternal_ = function(name) {
  if (!COMPILED) {
    if (goog.isProvided_(name)) {
      return name in goog.loadedModules_ ? goog.loadedModules_[name] : goog.getObjectByName(name);
    } else {
      return null;
    }
  }
};
goog.moduleLoaderState_ = null;
goog.isInModuleLoader_ = function() {
  return goog.moduleLoaderState_ != null;
};
goog.module.declareLegacyNamespace = function() {
  if (!COMPILED && !goog.isInModuleLoader_()) {
    throw new Error("goog.module.declareLegacyNamespace must be called from " + "within a goog.module");
  }
  if (!COMPILED && !goog.moduleLoaderState_.moduleName) {
    throw Error("goog.module must be called prior to " + "goog.module.declareLegacyNamespace.");
  }
  goog.moduleLoaderState_.declareLegacyNamespace = true;
};
goog.setTestOnly = function(opt_message) {
  if (goog.DISALLOW_TEST_ONLY_CODE) {
    opt_message = opt_message || "";
    throw Error("Importing test-only code into non-debug environment" + (opt_message ? ": " + opt_message : "."));
  }
};
goog.forwardDeclare = function(name) {
};
goog.forwardDeclare("Document");
goog.forwardDeclare("HTMLScriptElement");
goog.forwardDeclare("XMLHttpRequest");
if (!COMPILED) {
  goog.isProvided_ = function(name) {
    return name in goog.loadedModules_ || !goog.implicitNamespaces_[name] && goog.isDefAndNotNull(goog.getObjectByName(name));
  };
  goog.implicitNamespaces_ = {"goog.module":true};
}
goog.getObjectByName = function(name, opt_obj) {
  var parts = name.split(".");
  var cur = opt_obj || goog.global;
  for (var part;part = parts.shift();) {
    if (goog.isDefAndNotNull(cur[part])) {
      cur = cur[part];
    } else {
      return null;
    }
  }
  return cur;
};
goog.globalize = function(obj, opt_global) {
  var global = opt_global || goog.global;
  for (var x in obj) {
    global[x] = obj[x];
  }
};
goog.addDependency = function(relPath, provides, requires, opt_loadFlags) {
  if (goog.DEPENDENCIES_ENABLED) {
    var provide, require;
    var path = relPath.replace(/\\/g, "/");
    var deps = goog.dependencies_;
    if (!opt_loadFlags || typeof opt_loadFlags === "boolean") {
      opt_loadFlags = opt_loadFlags ? {"module":"goog"} : {};
    }
    for (var i = 0;provide = provides[i];i++) {
      deps.nameToPath[provide] = path;
      deps.loadFlags[path] = opt_loadFlags;
    }
    for (var j = 0;require = requires[j];j++) {
      if (!(path in deps.requires)) {
        deps.requires[path] = {};
      }
      deps.requires[path][require] = true;
    }
  }
};
goog.define("goog.ENABLE_DEBUG_LOADER", true);
goog.logToConsole_ = function(msg) {
  if (goog.global.console) {
    goog.global.console["error"](msg);
  }
};
goog.require = function(name) {
  if (!COMPILED) {
    if (goog.ENABLE_DEBUG_LOADER && goog.IS_OLD_IE_) {
      goog.maybeProcessDeferredDep_(name);
    }
    if (goog.isProvided_(name)) {
      if (goog.isInModuleLoader_()) {
        return goog.module.getInternal_(name);
      } else {
        return null;
      }
    }
    if (goog.ENABLE_DEBUG_LOADER) {
      var path = goog.getPathFromDeps_(name);
      if (path) {
        goog.writeScripts_(path);
        return null;
      }
    }
    var errorMessage = "goog.require could not find: " + name;
    goog.logToConsole_(errorMessage);
    throw Error(errorMessage);
  }
};
goog.basePath = "";
goog.global.CLOSURE_BASE_PATH;
goog.global.CLOSURE_NO_DEPS;
goog.global.CLOSURE_IMPORT_SCRIPT;
goog.nullFunction = function() {
};
goog.abstractMethod = function() {
  throw Error("unimplemented abstract method");
};
goog.addSingletonGetter = function(ctor) {
  ctor.getInstance = function() {
    if (ctor.instance_) {
      return ctor.instance_;
    }
    if (goog.DEBUG) {
      goog.instantiatedSingletons_[goog.instantiatedSingletons_.length] = ctor;
    }
    return ctor.instance_ = new ctor;
  };
};
goog.instantiatedSingletons_ = [];
goog.define("goog.LOAD_MODULE_USING_EVAL", true);
goog.define("goog.SEAL_MODULE_EXPORTS", goog.DEBUG);
goog.loadedModules_ = {};
goog.DEPENDENCIES_ENABLED = !COMPILED && goog.ENABLE_DEBUG_LOADER;
goog.define("goog.TRANSPILE", "detect");
goog.define("goog.TRANSPILER", "transpile.js");
if (goog.DEPENDENCIES_ENABLED) {
  goog.dependencies_ = {loadFlags:{}, nameToPath:{}, requires:{}, visited:{}, written:{}, deferred:{}};
  goog.inHtmlDocument_ = function() {
    var doc = goog.global.document;
    return doc != null && "write" in doc;
  };
  goog.findBasePath_ = function() {
    if (goog.isDef(goog.global.CLOSURE_BASE_PATH)) {
      goog.basePath = goog.global.CLOSURE_BASE_PATH;
      return;
    } else {
      if (!goog.inHtmlDocument_()) {
        return;
      }
    }
    var doc = goog.global.document;
    var scripts = doc.getElementsByTagName("SCRIPT");
    for (var i = scripts.length - 1;i >= 0;--i) {
      var script = (scripts[i]);
      var src = script.src;
      var qmark = src.lastIndexOf("?");
      var l = qmark == -1 ? src.length : qmark;
      if (src.substr(l - 7, 7) == "base.js") {
        goog.basePath = src.substr(0, l - 7);
        return;
      }
    }
  };
  goog.importScript_ = function(src, opt_sourceText) {
    var importScript = goog.global.CLOSURE_IMPORT_SCRIPT || goog.writeScriptTag_;
    if (importScript(src, opt_sourceText)) {
      goog.dependencies_.written[src] = true;
    }
  };
  goog.IS_OLD_IE_ = !!(!goog.global.atob && goog.global.document && goog.global.document.all);
  goog.importProcessedScript_ = function(src, isModule, needsTranspile) {
    var bootstrap = 'goog.retrieveAndExec_("' + src + '", ' + isModule + ", " + needsTranspile + ");";
    goog.importScript_("", bootstrap);
  };
  goog.queuedModules_ = [];
  goog.wrapModule_ = function(srcUrl, scriptText) {
    if (!goog.LOAD_MODULE_USING_EVAL || !goog.isDef(goog.global.JSON)) {
      return "" + "goog.loadModule(function(exports) {" + '"use strict";' + scriptText + "\n" + ";return exports" + "});" + "\n//# sourceURL\x3d" + srcUrl + "\n";
    } else {
      return "" + "goog.loadModule(" + goog.global.JSON.stringify(scriptText + "\n//# sourceURL\x3d" + srcUrl + "\n") + ");";
    }
  };
  goog.loadQueuedModules_ = function() {
    var count = goog.queuedModules_.length;
    if (count > 0) {
      var queue = goog.queuedModules_;
      goog.queuedModules_ = [];
      for (var i = 0;i < count;i++) {
        var path = queue[i];
        goog.maybeProcessDeferredPath_(path);
      }
    }
  };
  goog.maybeProcessDeferredDep_ = function(name) {
    if (goog.isDeferredModule_(name) && goog.allDepsAreAvailable_(name)) {
      var path = goog.getPathFromDeps_(name);
      goog.maybeProcessDeferredPath_(goog.basePath + path);
    }
  };
  goog.isDeferredModule_ = function(name) {
    var path = goog.getPathFromDeps_(name);
    var loadFlags = path && goog.dependencies_.loadFlags[path] || {};
    if (path && (loadFlags["module"] == "goog" || goog.needsTranspile_(loadFlags["lang"]))) {
      var abspath = goog.basePath + path;
      return abspath in goog.dependencies_.deferred;
    }
    return false;
  };
  goog.allDepsAreAvailable_ = function(name) {
    var path = goog.getPathFromDeps_(name);
    if (path && path in goog.dependencies_.requires) {
      for (var requireName in goog.dependencies_.requires[path]) {
        if (!goog.isProvided_(requireName) && !goog.isDeferredModule_(requireName)) {
          return false;
        }
      }
    }
    return true;
  };
  goog.maybeProcessDeferredPath_ = function(abspath) {
    if (abspath in goog.dependencies_.deferred) {
      var src = goog.dependencies_.deferred[abspath];
      delete goog.dependencies_.deferred[abspath];
      goog.globalEval(src);
    }
  };
  goog.loadModuleFromUrl = function(url) {
    goog.retrieveAndExec_(url, true, false);
  };
  goog.loadModule = function(moduleDef) {
    var previousState = goog.moduleLoaderState_;
    try {
      goog.moduleLoaderState_ = {moduleName:undefined, declareLegacyNamespace:false};
      var exports;
      if (goog.isFunction(moduleDef)) {
        exports = moduleDef.call(goog.global, {});
      } else {
        if (goog.isString(moduleDef)) {
          exports = goog.loadModuleFromSource_.call(goog.global, moduleDef);
        } else {
          throw Error("Invalid module definition");
        }
      }
      var moduleName = goog.moduleLoaderState_.moduleName;
      if (!goog.isString(moduleName) || !moduleName) {
        throw Error('Invalid module name "' + moduleName + '"');
      }
      if (goog.moduleLoaderState_.declareLegacyNamespace) {
        goog.constructNamespace_(moduleName, exports);
      } else {
        if (goog.SEAL_MODULE_EXPORTS && Object.seal) {
          Object.seal(exports);
        }
      }
      goog.loadedModules_[moduleName] = exports;
    } finally {
      goog.moduleLoaderState_ = previousState;
    }
  };
  goog.loadModuleFromSource_ = function() {
    var exports = {};
    eval(arguments[0]);
    return exports;
  };
  goog.writeScriptSrcNode_ = function(src) {
    goog.global.document.write('\x3cscript type\x3d"text/javascript" src\x3d"' + src + '"\x3e\x3c/' + "script\x3e");
  };
  goog.appendScriptSrcNode_ = function(src) {
    var doc = goog.global.document;
    var scriptEl = (doc.createElement("script"));
    scriptEl.type = "text/javascript";
    scriptEl.src = src;
    scriptEl.defer = false;
    scriptEl.async = false;
    doc.head.appendChild(scriptEl);
  };
  goog.writeScriptTag_ = function(src, opt_sourceText) {
    if (goog.inHtmlDocument_()) {
      var doc = goog.global.document;
      if (!goog.ENABLE_CHROME_APP_SAFE_SCRIPT_LOADING && doc.readyState == "complete") {
        var isDeps = /\bdeps.js$/.test(src);
        if (isDeps) {
          return false;
        } else {
          throw Error('Cannot write "' + src + '" after document load');
        }
      }
      if (opt_sourceText === undefined) {
        if (!goog.IS_OLD_IE_) {
          if (goog.ENABLE_CHROME_APP_SAFE_SCRIPT_LOADING) {
            goog.appendScriptSrcNode_(src);
          } else {
            goog.writeScriptSrcNode_(src);
          }
        } else {
          var state = " onreadystatechange\x3d'goog.onScriptLoad_(this, " + ++goog.lastNonModuleScriptIndex_ + ")' ";
          doc.write('\x3cscript type\x3d"text/javascript" src\x3d"' + src + '"' + state + "\x3e\x3c/" + "script\x3e");
        }
      } else {
        doc.write('\x3cscript type\x3d"text/javascript"\x3e' + opt_sourceText + "\x3c/" + "script\x3e");
      }
      return true;
    } else {
      return false;
    }
  };
  goog.needsTranspile_ = function(lang) {
    if (goog.TRANSPILE == "always") {
      return true;
    } else {
      if (goog.TRANSPILE == "never") {
        return false;
      } else {
        if (!goog.transpiledLanguages_) {
          goog.transpiledLanguages_ = {"es5":true, "es6":true, "es6-impl":true};
          try {
            goog.transpiledLanguages_["es5"] = eval("[1,].length!\x3d1");
            var es6implTest = "let a\x3d{};const X\x3dclass{constructor(){}x(z){return new Map([" + "...arguments]).get(z[0])\x3d\x3d3}};return new X().x([a,3])";
            var es6fullTest = "class X{constructor(){if(new.target!\x3dString)throw 1;this.x\x3d42}}" + "let q\x3dReflect.construct(X,[],String);if(q.x!\x3d42||!(q instanceof " + "String))throw 1;for(const a of[2,3]){if(a\x3d\x3d2)continue;function " + "f(z\x3d{a}){let a\x3d0;return z.a}{function f(){return 0;}}return f()" + "\x3d\x3d3}";
            if (eval('(()\x3d\x3e{"use strict";' + es6implTest + "})()")) {
              goog.transpiledLanguages_["es6-impl"] = false;
            }
            if (eval('(()\x3d\x3e{"use strict";' + es6fullTest + "})()")) {
              goog.transpiledLanguages_["es6"] = false;
            }
          } catch (err) {
          }
        }
      }
    }
    return !!goog.transpiledLanguages_[lang];
  };
  goog.transpiledLanguages_ = null;
  goog.lastNonModuleScriptIndex_ = 0;
  goog.onScriptLoad_ = function(script, scriptIndex) {
    if (script.readyState == "complete" && goog.lastNonModuleScriptIndex_ == scriptIndex) {
      goog.loadQueuedModules_();
    }
    return true;
  };
  goog.writeScripts_ = function(pathToLoad) {
    var scripts = [];
    var seenScript = {};
    var deps = goog.dependencies_;
    function visitNode(path) {
      if (path in deps.written) {
        return;
      }
      if (path in deps.visited) {
        return;
      }
      deps.visited[path] = true;
      if (path in deps.requires) {
        for (var requireName in deps.requires[path]) {
          if (!goog.isProvided_(requireName)) {
            if (requireName in deps.nameToPath) {
              visitNode(deps.nameToPath[requireName]);
            } else {
              throw Error("Undefined nameToPath for " + requireName);
            }
          }
        }
      }
      if (!(path in seenScript)) {
        seenScript[path] = true;
        scripts.push(path);
      }
    }
    visitNode(pathToLoad);
    for (var i = 0;i < scripts.length;i++) {
      var path = scripts[i];
      goog.dependencies_.written[path] = true;
    }
    var moduleState = goog.moduleLoaderState_;
    goog.moduleLoaderState_ = null;
    for (var i = 0;i < scripts.length;i++) {
      var path = scripts[i];
      if (path) {
        var loadFlags = deps.loadFlags[path] || {};
        var needsTranspile = goog.needsTranspile_(loadFlags["lang"]);
        if (loadFlags["module"] == "goog" || needsTranspile) {
          goog.importProcessedScript_(goog.basePath + path, loadFlags["module"] == "goog", needsTranspile);
        } else {
          goog.importScript_(goog.basePath + path);
        }
      } else {
        goog.moduleLoaderState_ = moduleState;
        throw Error("Undefined script input");
      }
    }
    goog.moduleLoaderState_ = moduleState;
  };
  goog.getPathFromDeps_ = function(rule) {
    if (rule in goog.dependencies_.nameToPath) {
      return goog.dependencies_.nameToPath[rule];
    } else {
      return null;
    }
  };
  goog.findBasePath_();
  if (!goog.global.CLOSURE_NO_DEPS) {
    goog.importScript_(goog.basePath + "deps.js");
  }
}
goog.normalizePath_ = function(path) {
  var components = path.split("/");
  var i = 0;
  while (i < components.length) {
    if (components[i] == ".") {
      components.splice(i, 1);
    } else {
      if (i && components[i] == ".." && components[i - 1] && components[i - 1] != "..") {
        components.splice(--i, 2);
      } else {
        i++;
      }
    }
  }
  return components.join("/");
};
goog.loadFileSync_ = function(src) {
  if (goog.global.CLOSURE_LOAD_FILE_SYNC) {
    return goog.global.CLOSURE_LOAD_FILE_SYNC(src);
  } else {
    try {
      var xhr = new goog.global["XMLHttpRequest"];
      xhr.open("get", src, false);
      xhr.send();
      return xhr.status == 0 || xhr.status == 200 ? xhr.responseText : null;
    } catch (err) {
      return null;
    }
  }
};
goog.retrieveAndExec_ = function(src, isModule, needsTranspile) {
  if (!COMPILED) {
    var originalPath = src;
    src = goog.normalizePath_(src);
    var importScript = goog.global.CLOSURE_IMPORT_SCRIPT || goog.writeScriptTag_;
    var scriptText = goog.loadFileSync_(src);
    if (scriptText == null) {
      throw new Error('Load of "' + src + '" failed');
    }
    if (needsTranspile) {
      scriptText = goog.transpile_.call(goog.global, scriptText, src);
    }
    if (isModule) {
      scriptText = goog.wrapModule_(src, scriptText);
    } else {
      scriptText += "\n//# sourceURL\x3d" + src;
    }
    var isOldIE = goog.IS_OLD_IE_;
    if (isOldIE) {
      goog.dependencies_.deferred[originalPath] = scriptText;
      goog.queuedModules_.push(originalPath);
    } else {
      importScript(src, scriptText);
    }
  }
};
goog.transpile_ = function(code, path) {
  var jscomp = goog.global["$jscomp"];
  if (!jscomp) {
    goog.global["$jscomp"] = jscomp = {};
  }
  var transpile = jscomp.transpile;
  if (!transpile) {
    var transpilerPath = goog.basePath + goog.TRANSPILER;
    var transpilerCode = goog.loadFileSync_(transpilerPath);
    if (transpilerCode) {
      eval(transpilerCode + "\n//# sourceURL\x3d" + transpilerPath);
      jscomp = goog.global["$jscomp"];
      transpile = jscomp.transpile;
    }
  }
  if (!transpile) {
    var suffix = " requires transpilation but no transpiler was found.";
    transpile = jscomp.transpile = function(code, path) {
      goog.logToConsole_(path + suffix);
      return code;
    };
  }
  return transpile(code, path);
};
goog.typeOf = function(value) {
  var s = typeof value;
  if (s == "object") {
    if (value) {
      if (value instanceof Array) {
        return "array";
      } else {
        if (value instanceof Object) {
          return s;
        }
      }
      var className = Object.prototype.toString.call((value));
      if (className == "[object Window]") {
        return "object";
      }
      if (className == "[object Array]" || typeof value.length == "number" && typeof value.splice != "undefined" && typeof value.propertyIsEnumerable != "undefined" && !value.propertyIsEnumerable("splice")) {
        return "array";
      }
      if (className == "[object Function]" || typeof value.call != "undefined" && typeof value.propertyIsEnumerable != "undefined" && !value.propertyIsEnumerable("call")) {
        return "function";
      }
    } else {
      return "null";
    }
  } else {
    if (s == "function" && typeof value.call == "undefined") {
      return "object";
    }
  }
  return s;
};
goog.isNull = function(val) {
  return val === null;
};
goog.isDefAndNotNull = function(val) {
  return val != null;
};
goog.isArray = function(val) {
  return goog.typeOf(val) == "array";
};
goog.isArrayLike = function(val) {
  var type = goog.typeOf(val);
  return type == "array" || type == "object" && typeof val.length == "number";
};
goog.isDateLike = function(val) {
  return goog.isObject(val) && typeof val.getFullYear == "function";
};
goog.isString = function(val) {
  return typeof val == "string";
};
goog.isBoolean = function(val) {
  return typeof val == "boolean";
};
goog.isNumber = function(val) {
  return typeof val == "number";
};
goog.isFunction = function(val) {
  return goog.typeOf(val) == "function";
};
goog.isObject = function(val) {
  var type = typeof val;
  return type == "object" && val != null || type == "function";
};
goog.getUid = function(obj) {
  return obj[goog.UID_PROPERTY_] || (obj[goog.UID_PROPERTY_] = ++goog.uidCounter_);
};
goog.hasUid = function(obj) {
  return !!obj[goog.UID_PROPERTY_];
};
goog.removeUid = function(obj) {
  if (obj !== null && "removeAttribute" in obj) {
    obj.removeAttribute(goog.UID_PROPERTY_);
  }
  try {
    delete obj[goog.UID_PROPERTY_];
  } catch (ex) {
  }
};
goog.UID_PROPERTY_ = "closure_uid_" + (Math.random() * 1e9 >>> 0);
goog.uidCounter_ = 0;
goog.getHashCode = goog.getUid;
goog.removeHashCode = goog.removeUid;
goog.cloneObject = function(obj) {
  var type = goog.typeOf(obj);
  if (type == "object" || type == "array") {
    if (obj.clone) {
      return obj.clone();
    }
    var clone = type == "array" ? [] : {};
    for (var key in obj) {
      clone[key] = goog.cloneObject(obj[key]);
    }
    return clone;
  }
  return obj;
};
goog.bindNative_ = function(fn, selfObj, var_args) {
  return (fn.call.apply(fn.bind, arguments));
};
goog.bindJs_ = function(fn, selfObj, var_args) {
  if (!fn) {
    throw new Error;
  }
  if (arguments.length > 2) {
    var boundArgs = Array.prototype.slice.call(arguments, 2);
    return function() {
      var newArgs = Array.prototype.slice.call(arguments);
      Array.prototype.unshift.apply(newArgs, boundArgs);
      return fn.apply(selfObj, newArgs);
    };
  } else {
    return function() {
      return fn.apply(selfObj, arguments);
    };
  }
};
goog.bind = function(fn, selfObj, var_args) {
  if (Function.prototype.bind && Function.prototype.bind.toString().indexOf("native code") != -1) {
    goog.bind = goog.bindNative_;
  } else {
    goog.bind = goog.bindJs_;
  }
  return goog.bind.apply(null, arguments);
};
goog.partial = function(fn, var_args) {
  var args = Array.prototype.slice.call(arguments, 1);
  return function() {
    var newArgs = args.slice();
    newArgs.push.apply(newArgs, arguments);
    return fn.apply(this, newArgs);
  };
};
goog.mixin = function(target, source) {
  for (var x in source) {
    target[x] = source[x];
  }
};
goog.now = goog.TRUSTED_SITE && Date.now || function() {
  return +new Date;
};
goog.globalEval = function(script) {
  if (goog.global.execScript) {
    goog.global.execScript(script, "JavaScript");
  } else {
    if (goog.global.eval) {
      if (goog.evalWorksForGlobals_ == null) {
        goog.global.eval("var _evalTest_ \x3d 1;");
        if (typeof goog.global["_evalTest_"] != "undefined") {
          try {
            delete goog.global["_evalTest_"];
          } catch (ignore) {
          }
          goog.evalWorksForGlobals_ = true;
        } else {
          goog.evalWorksForGlobals_ = false;
        }
      }
      if (goog.evalWorksForGlobals_) {
        goog.global.eval(script);
      } else {
        var doc = goog.global.document;
        var scriptElt = (doc.createElement("SCRIPT"));
        scriptElt.type = "text/javascript";
        scriptElt.defer = false;
        scriptElt.appendChild(doc.createTextNode(script));
        doc.body.appendChild(scriptElt);
        doc.body.removeChild(scriptElt);
      }
    } else {
      throw Error("goog.globalEval not available");
    }
  }
};
goog.evalWorksForGlobals_ = null;
goog.cssNameMapping_;
goog.cssNameMappingStyle_;
goog.getCssName = function(className, opt_modifier) {
  var getMapping = function(cssName) {
    return goog.cssNameMapping_[cssName] || cssName;
  };
  var renameByParts = function(cssName) {
    var parts = cssName.split("-");
    var mapped = [];
    for (var i = 0;i < parts.length;i++) {
      mapped.push(getMapping(parts[i]));
    }
    return mapped.join("-");
  };
  var rename;
  if (goog.cssNameMapping_) {
    rename = goog.cssNameMappingStyle_ == "BY_WHOLE" ? getMapping : renameByParts;
  } else {
    rename = function(a) {
      return a;
    };
  }
  if (opt_modifier) {
    return className + "-" + rename(opt_modifier);
  } else {
    return rename(className);
  }
};
goog.setCssNameMapping = function(mapping, opt_style) {
  goog.cssNameMapping_ = mapping;
  goog.cssNameMappingStyle_ = opt_style;
};
goog.global.CLOSURE_CSS_NAME_MAPPING;
if (!COMPILED && goog.global.CLOSURE_CSS_NAME_MAPPING) {
  goog.cssNameMapping_ = goog.global.CLOSURE_CSS_NAME_MAPPING;
}
goog.getMsg = function(str, opt_values) {
  if (opt_values) {
    str = str.replace(/\{\$([^}]+)}/g, function(match, key) {
      return opt_values != null && key in opt_values ? opt_values[key] : match;
    });
  }
  return str;
};
goog.getMsgWithFallback = function(a, b) {
  return a;
};
goog.exportSymbol = function(publicPath, object, opt_objectToExportTo) {
  goog.exportPath_(publicPath, object, opt_objectToExportTo);
};
goog.exportProperty = function(object, publicName, symbol) {
  object[publicName] = symbol;
};
goog.inherits = function(childCtor, parentCtor) {
  function tempCtor() {
  }
  tempCtor.prototype = parentCtor.prototype;
  childCtor.superClass_ = parentCtor.prototype;
  childCtor.prototype = new tempCtor;
  childCtor.prototype.constructor = childCtor;
  childCtor.base = function(me, methodName, var_args) {
    var args = new Array(arguments.length - 2);
    for (var i = 2;i < arguments.length;i++) {
      args[i - 2] = arguments[i];
    }
    return parentCtor.prototype[methodName].apply(me, args);
  };
};
goog.base = function(me, opt_methodName, var_args) {
  var caller = arguments.callee.caller;
  if (goog.STRICT_MODE_COMPATIBLE || goog.DEBUG && !caller) {
    throw Error("arguments.caller not defined.  goog.base() cannot be used " + "with strict mode code. See " + "http://www.ecma-international.org/ecma-262/5.1/#sec-C");
  }
  if (caller.superClass_) {
    var ctorArgs = new Array(arguments.length - 1);
    for (var i = 1;i < arguments.length;i++) {
      ctorArgs[i - 1] = arguments[i];
    }
    return caller.superClass_.constructor.apply(me, ctorArgs);
  }
  var args = new Array(arguments.length - 2);
  for (var i = 2;i < arguments.length;i++) {
    args[i - 2] = arguments[i];
  }
  var foundCaller = false;
  for (var ctor = me.constructor;ctor;ctor = ctor.superClass_ && ctor.superClass_.constructor) {
    if (ctor.prototype[opt_methodName] === caller) {
      foundCaller = true;
    } else {
      if (foundCaller) {
        return ctor.prototype[opt_methodName].apply(me, args);
      }
    }
  }
  if (me[opt_methodName] === caller) {
    return me.constructor.prototype[opt_methodName].apply(me, args);
  } else {
    throw Error("goog.base called from a method of one name " + "to a method of a different name");
  }
};
goog.scope = function(fn) {
  if (goog.isInModuleLoader_()) {
    throw Error("goog.scope is not supported within a goog.module.");
  }
  fn.call(goog.global);
};
if (!COMPILED) {
  goog.global["COMPILED"] = COMPILED;
}
goog.defineClass = function(superClass, def) {
  var constructor = def.constructor;
  var statics = def.statics;
  if (!constructor || constructor == Object.prototype.constructor) {
    constructor = function() {
      throw Error("cannot instantiate an interface (no constructor defined).");
    };
  }
  var cls = goog.defineClass.createSealingConstructor_(constructor, superClass);
  if (superClass) {
    goog.inherits(cls, superClass);
  }
  delete def.constructor;
  delete def.statics;
  goog.defineClass.applyProperties_(cls.prototype, def);
  if (statics != null) {
    if (statics instanceof Function) {
      statics(cls);
    } else {
      goog.defineClass.applyProperties_(cls, statics);
    }
  }
  return cls;
};
goog.defineClass.ClassDescriptor;
goog.define("goog.defineClass.SEAL_CLASS_INSTANCES", goog.DEBUG);
goog.defineClass.createSealingConstructor_ = function(ctr, superClass) {
  if (!goog.defineClass.SEAL_CLASS_INSTANCES) {
    return ctr;
  }
  var superclassSealable = !goog.defineClass.isUnsealable_(superClass);
  var wrappedCtr = function() {
    var instance = ctr.apply(this, arguments) || this;
    instance[goog.UID_PROPERTY_] = instance[goog.UID_PROPERTY_];
    if (this.constructor === wrappedCtr && superclassSealable && Object.seal instanceof Function) {
      Object.seal(instance);
    }
    return instance;
  };
  return wrappedCtr;
};
goog.defineClass.isUnsealable_ = function(ctr) {
  return ctr && ctr.prototype && ctr.prototype[goog.UNSEALABLE_CONSTRUCTOR_PROPERTY_];
};
goog.defineClass.OBJECT_PROTOTYPE_FIELDS_ = ["constructor", "hasOwnProperty", "isPrototypeOf", "propertyIsEnumerable", "toLocaleString", "toString", "valueOf"];
goog.defineClass.applyProperties_ = function(target, source) {
  var key;
  for (key in source) {
    if (Object.prototype.hasOwnProperty.call(source, key)) {
      target[key] = source[key];
    }
  }
  for (var i = 0;i < goog.defineClass.OBJECT_PROTOTYPE_FIELDS_.length;i++) {
    key = goog.defineClass.OBJECT_PROTOTYPE_FIELDS_[i];
    if (Object.prototype.hasOwnProperty.call(source, key)) {
      target[key] = source[key];
    }
  }
};
goog.tagUnsealableClass = function(ctr) {
  if (!COMPILED && goog.defineClass.SEAL_CLASS_INSTANCES) {
    ctr.prototype[goog.UNSEALABLE_CONSTRUCTOR_PROPERTY_] = true;
  }
};
goog.UNSEALABLE_CONSTRUCTOR_PROPERTY_ = "goog_defineClass_legacy_unsealable";
goog.provide("goog.string");
goog.provide("goog.string.Unicode");
goog.define("goog.string.DETECT_DOUBLE_ESCAPING", false);
goog.define("goog.string.FORCE_NON_DOM_HTML_UNESCAPING", false);
goog.string.Unicode = {NBSP:" "};
goog.string.startsWith = function(str, prefix) {
  return str.lastIndexOf(prefix, 0) == 0;
};
goog.string.endsWith = function(str, suffix) {
  var l = str.length - suffix.length;
  return l >= 0 && str.indexOf(suffix, l) == l;
};
goog.string.caseInsensitiveStartsWith = function(str, prefix) {
  return goog.string.caseInsensitiveCompare(prefix, str.substr(0, prefix.length)) == 0;
};
goog.string.caseInsensitiveEndsWith = function(str, suffix) {
  return goog.string.caseInsensitiveCompare(suffix, str.substr(str.length - suffix.length, suffix.length)) == 0;
};
goog.string.caseInsensitiveEquals = function(str1, str2) {
  return str1.toLowerCase() == str2.toLowerCase();
};
goog.string.subs = function(str, var_args) {
  var splitParts = str.split("%s");
  var returnString = "";
  var subsArguments = Array.prototype.slice.call(arguments, 1);
  while (subsArguments.length && splitParts.length > 1) {
    returnString += splitParts.shift() + subsArguments.shift();
  }
  return returnString + splitParts.join("%s");
};
goog.string.collapseWhitespace = function(str) {
  return str.replace(/[\s\xa0]+/g, " ").replace(/^\s+|\s+$/g, "");
};
goog.string.isEmptyOrWhitespace = function(str) {
  return /^[\s\xa0]*$/.test(str);
};
goog.string.isEmptyString = function(str) {
  return str.length == 0;
};
goog.string.isEmpty = goog.string.isEmptyOrWhitespace;
goog.string.isEmptyOrWhitespaceSafe = function(str) {
  return goog.string.isEmptyOrWhitespace(goog.string.makeSafe(str));
};
goog.string.isEmptySafe = goog.string.isEmptyOrWhitespaceSafe;
goog.string.isBreakingWhitespace = function(str) {
  return !/[^\t\n\r ]/.test(str);
};
goog.string.isAlpha = function(str) {
  return !/[^a-zA-Z]/.test(str);
};
goog.string.isNumeric = function(str) {
  return !/[^0-9]/.test(str);
};
goog.string.isAlphaNumeric = function(str) {
  return !/[^a-zA-Z0-9]/.test(str);
};
goog.string.isSpace = function(ch) {
  return ch == " ";
};
goog.string.isUnicodeChar = function(ch) {
  return ch.length == 1 && ch >= " " && ch <= "~" || ch >= "" && ch <= "�";
};
goog.string.stripNewlines = function(str) {
  return str.replace(/(\r\n|\r|\n)+/g, " ");
};
goog.string.canonicalizeNewlines = function(str) {
  return str.replace(/(\r\n|\r|\n)/g, "\n");
};
goog.string.normalizeWhitespace = function(str) {
  return str.replace(/\xa0|\s/g, " ");
};
goog.string.normalizeSpaces = function(str) {
  return str.replace(/\xa0|[ \t]+/g, " ");
};
goog.string.collapseBreakingSpaces = function(str) {
  return str.replace(/[\t\r\n ]+/g, " ").replace(/^[\t\r\n ]+|[\t\r\n ]+$/g, "");
};
goog.string.trim = goog.TRUSTED_SITE && String.prototype.trim ? function(str) {
  return str.trim();
} : function(str) {
  return str.replace(/^[\s\xa0]+|[\s\xa0]+$/g, "");
};
goog.string.trimLeft = function(str) {
  return str.replace(/^[\s\xa0]+/, "");
};
goog.string.trimRight = function(str) {
  return str.replace(/[\s\xa0]+$/, "");
};
goog.string.caseInsensitiveCompare = function(str1, str2) {
  var test1 = String(str1).toLowerCase();
  var test2 = String(str2).toLowerCase();
  if (test1 < test2) {
    return -1;
  } else {
    if (test1 == test2) {
      return 0;
    } else {
      return 1;
    }
  }
};
goog.string.numberAwareCompare_ = function(str1, str2, tokenizerRegExp) {
  if (str1 == str2) {
    return 0;
  }
  if (!str1) {
    return -1;
  }
  if (!str2) {
    return 1;
  }
  var tokens1 = str1.toLowerCase().match(tokenizerRegExp);
  var tokens2 = str2.toLowerCase().match(tokenizerRegExp);
  var count = Math.min(tokens1.length, tokens2.length);
  for (var i = 0;i < count;i++) {
    var a = tokens1[i];
    var b = tokens2[i];
    if (a != b) {
      var num1 = parseInt(a, 10);
      if (!isNaN(num1)) {
        var num2 = parseInt(b, 10);
        if (!isNaN(num2) && num1 - num2) {
          return num1 - num2;
        }
      }
      return a < b ? -1 : 1;
    }
  }
  if (tokens1.length != tokens2.length) {
    return tokens1.length - tokens2.length;
  }
  return str1 < str2 ? -1 : 1;
};
goog.string.intAwareCompare = function(str1, str2) {
  return goog.string.numberAwareCompare_(str1, str2, /\d+|\D+/g);
};
goog.string.floatAwareCompare = function(str1, str2) {
  return goog.string.numberAwareCompare_(str1, str2, /\d+|\.\d+|\D+/g);
};
goog.string.numerateCompare = goog.string.floatAwareCompare;
goog.string.urlEncode = function(str) {
  return encodeURIComponent(String(str));
};
goog.string.urlDecode = function(str) {
  return decodeURIComponent(str.replace(/\+/g, " "));
};
goog.string.newLineToBr = function(str, opt_xml) {
  return str.replace(/(\r\n|\r|\n)/g, opt_xml ? "\x3cbr /\x3e" : "\x3cbr\x3e");
};
goog.string.htmlEscape = function(str, opt_isLikelyToContainHtmlChars) {
  if (opt_isLikelyToContainHtmlChars) {
    str = str.replace(goog.string.AMP_RE_, "\x26amp;").replace(goog.string.LT_RE_, "\x26lt;").replace(goog.string.GT_RE_, "\x26gt;").replace(goog.string.QUOT_RE_, "\x26quot;").replace(goog.string.SINGLE_QUOTE_RE_, "\x26#39;").replace(goog.string.NULL_RE_, "\x26#0;");
    if (goog.string.DETECT_DOUBLE_ESCAPING) {
      str = str.replace(goog.string.E_RE_, "\x26#101;");
    }
    return str;
  } else {
    if (!goog.string.ALL_RE_.test(str)) {
      return str;
    }
    if (str.indexOf("\x26") != -1) {
      str = str.replace(goog.string.AMP_RE_, "\x26amp;");
    }
    if (str.indexOf("\x3c") != -1) {
      str = str.replace(goog.string.LT_RE_, "\x26lt;");
    }
    if (str.indexOf("\x3e") != -1) {
      str = str.replace(goog.string.GT_RE_, "\x26gt;");
    }
    if (str.indexOf('"') != -1) {
      str = str.replace(goog.string.QUOT_RE_, "\x26quot;");
    }
    if (str.indexOf("'") != -1) {
      str = str.replace(goog.string.SINGLE_QUOTE_RE_, "\x26#39;");
    }
    if (str.indexOf("\x00") != -1) {
      str = str.replace(goog.string.NULL_RE_, "\x26#0;");
    }
    if (goog.string.DETECT_DOUBLE_ESCAPING && str.indexOf("e") != -1) {
      str = str.replace(goog.string.E_RE_, "\x26#101;");
    }
    return str;
  }
};
goog.string.AMP_RE_ = /&/g;
goog.string.LT_RE_ = /</g;
goog.string.GT_RE_ = />/g;
goog.string.QUOT_RE_ = /"/g;
goog.string.SINGLE_QUOTE_RE_ = /'/g;
goog.string.NULL_RE_ = /\x00/g;
goog.string.E_RE_ = /e/g;
goog.string.ALL_RE_ = goog.string.DETECT_DOUBLE_ESCAPING ? /[\x00&<>"'e]/ : /[\x00&<>"']/;
goog.string.unescapeEntities = function(str) {
  if (goog.string.contains(str, "\x26")) {
    if (!goog.string.FORCE_NON_DOM_HTML_UNESCAPING && "document" in goog.global) {
      return goog.string.unescapeEntitiesUsingDom_(str);
    } else {
      return goog.string.unescapePureXmlEntities_(str);
    }
  }
  return str;
};
goog.string.unescapeEntitiesWithDocument = function(str, document) {
  if (goog.string.contains(str, "\x26")) {
    return goog.string.unescapeEntitiesUsingDom_(str, document);
  }
  return str;
};
goog.string.unescapeEntitiesUsingDom_ = function(str, opt_document) {
  var seen = {"\x26amp;":"\x26", "\x26lt;":"\x3c", "\x26gt;":"\x3e", "\x26quot;":'"'};
  var div;
  if (opt_document) {
    div = opt_document.createElement("div");
  } else {
    div = goog.global.document.createElement("div");
  }
  return str.replace(goog.string.HTML_ENTITY_PATTERN_, function(s, entity) {
    var value = seen[s];
    if (value) {
      return value;
    }
    if (entity.charAt(0) == "#") {
      var n = Number("0" + entity.substr(1));
      if (!isNaN(n)) {
        value = String.fromCharCode(n);
      }
    }
    if (!value) {
      div.innerHTML = s + " ";
      value = div.firstChild.nodeValue.slice(0, -1);
    }
    return seen[s] = value;
  });
};
goog.string.unescapePureXmlEntities_ = function(str) {
  return str.replace(/&([^;]+);/g, function(s, entity) {
    switch(entity) {
      case "amp":
        return "\x26";
      case "lt":
        return "\x3c";
      case "gt":
        return "\x3e";
      case "quot":
        return '"';
      default:
        if (entity.charAt(0) == "#") {
          var n = Number("0" + entity.substr(1));
          if (!isNaN(n)) {
            return String.fromCharCode(n);
          }
        }
        return s;
    }
  });
};
goog.string.HTML_ENTITY_PATTERN_ = /&([^;\s<&]+);?/g;
goog.string.whitespaceEscape = function(str, opt_xml) {
  return goog.string.newLineToBr(str.replace(/  /g, " \x26#160;"), opt_xml);
};
goog.string.preserveSpaces = function(str) {
  return str.replace(/(^|[\n ]) /g, "$1" + goog.string.Unicode.NBSP);
};
goog.string.stripQuotes = function(str, quoteChars) {
  var length = quoteChars.length;
  for (var i = 0;i < length;i++) {
    var quoteChar = length == 1 ? quoteChars : quoteChars.charAt(i);
    if (str.charAt(0) == quoteChar && str.charAt(str.length - 1) == quoteChar) {
      return str.substring(1, str.length - 1);
    }
  }
  return str;
};
goog.string.truncate = function(str, chars, opt_protectEscapedCharacters) {
  if (opt_protectEscapedCharacters) {
    str = goog.string.unescapeEntities(str);
  }
  if (str.length > chars) {
    str = str.substring(0, chars - 3) + "...";
  }
  if (opt_protectEscapedCharacters) {
    str = goog.string.htmlEscape(str);
  }
  return str;
};
goog.string.truncateMiddle = function(str, chars, opt_protectEscapedCharacters, opt_trailingChars) {
  if (opt_protectEscapedCharacters) {
    str = goog.string.unescapeEntities(str);
  }
  if (opt_trailingChars && str.length > chars) {
    if (opt_trailingChars > chars) {
      opt_trailingChars = chars;
    }
    var endPoint = str.length - opt_trailingChars;
    var startPoint = chars - opt_trailingChars;
    str = str.substring(0, startPoint) + "..." + str.substring(endPoint);
  } else {
    if (str.length > chars) {
      var half = Math.floor(chars / 2);
      var endPos = str.length - half;
      half += chars % 2;
      str = str.substring(0, half) + "..." + str.substring(endPos);
    }
  }
  if (opt_protectEscapedCharacters) {
    str = goog.string.htmlEscape(str);
  }
  return str;
};
goog.string.specialEscapeChars_ = {"\x00":"\\0", "\b":"\\b", "\f":"\\f", "\n":"\\n", "\r":"\\r", "\t":"\\t", "\x0B":"\\x0B", '"':'\\"', "\\":"\\\\", "\x3c":"\x3c"};
goog.string.jsEscapeCache_ = {"'":"\\'"};
goog.string.quote = function(s) {
  s = String(s);
  var sb = ['"'];
  for (var i = 0;i < s.length;i++) {
    var ch = s.charAt(i);
    var cc = ch.charCodeAt(0);
    sb[i + 1] = goog.string.specialEscapeChars_[ch] || (cc > 31 && cc < 127 ? ch : goog.string.escapeChar(ch));
  }
  sb.push('"');
  return sb.join("");
};
goog.string.escapeString = function(str) {
  var sb = [];
  for (var i = 0;i < str.length;i++) {
    sb[i] = goog.string.escapeChar(str.charAt(i));
  }
  return sb.join("");
};
goog.string.escapeChar = function(c) {
  if (c in goog.string.jsEscapeCache_) {
    return goog.string.jsEscapeCache_[c];
  }
  if (c in goog.string.specialEscapeChars_) {
    return goog.string.jsEscapeCache_[c] = goog.string.specialEscapeChars_[c];
  }
  var rv = c;
  var cc = c.charCodeAt(0);
  if (cc > 31 && cc < 127) {
    rv = c;
  } else {
    if (cc < 256) {
      rv = "\\x";
      if (cc < 16 || cc > 256) {
        rv += "0";
      }
    } else {
      rv = "\\u";
      if (cc < 4096) {
        rv += "0";
      }
    }
    rv += cc.toString(16).toUpperCase();
  }
  return goog.string.jsEscapeCache_[c] = rv;
};
goog.string.contains = function(str, subString) {
  return str.indexOf(subString) != -1;
};
goog.string.caseInsensitiveContains = function(str, subString) {
  return goog.string.contains(str.toLowerCase(), subString.toLowerCase());
};
goog.string.countOf = function(s, ss) {
  return s && ss ? s.split(ss).length - 1 : 0;
};
goog.string.removeAt = function(s, index, stringLength) {
  var resultStr = s;
  if (index >= 0 && index < s.length && stringLength > 0) {
    resultStr = s.substr(0, index) + s.substr(index + stringLength, s.length - index - stringLength);
  }
  return resultStr;
};
goog.string.remove = function(s, ss) {
  var re = new RegExp(goog.string.regExpEscape(ss), "");
  return s.replace(re, "");
};
goog.string.removeAll = function(s, ss) {
  var re = new RegExp(goog.string.regExpEscape(ss), "g");
  return s.replace(re, "");
};
goog.string.regExpEscape = function(s) {
  return String(s).replace(/([-()\[\]{}+?*.$\^|,:#<!\\])/g, "\\$1").replace(/\x08/g, "\\x08");
};
goog.string.repeat = String.prototype.repeat ? function(string, length) {
  return string.repeat(length);
} : function(string, length) {
  return (new Array(length + 1)).join(string);
};
goog.string.padNumber = function(num, length, opt_precision) {
  var s = goog.isDef(opt_precision) ? num.toFixed(opt_precision) : String(num);
  var index = s.indexOf(".");
  if (index == -1) {
    index = s.length;
  }
  return goog.string.repeat("0", Math.max(0, length - index)) + s;
};
goog.string.makeSafe = function(obj) {
  return obj == null ? "" : String(obj);
};
goog.string.buildString = function(var_args) {
  return Array.prototype.join.call(arguments, "");
};
goog.string.getRandomString = function() {
  var x = 2147483648;
  return Math.floor(Math.random() * x).toString(36) + Math.abs(Math.floor(Math.random() * x) ^ goog.now()).toString(36);
};
goog.string.compareVersions = function(version1, version2) {
  var order = 0;
  var v1Subs = goog.string.trim(String(version1)).split(".");
  var v2Subs = goog.string.trim(String(version2)).split(".");
  var subCount = Math.max(v1Subs.length, v2Subs.length);
  for (var subIdx = 0;order == 0 && subIdx < subCount;subIdx++) {
    var v1Sub = v1Subs[subIdx] || "";
    var v2Sub = v2Subs[subIdx] || "";
    var v1CompParser = new RegExp("(\\d*)(\\D*)", "g");
    var v2CompParser = new RegExp("(\\d*)(\\D*)", "g");
    do {
      var v1Comp = v1CompParser.exec(v1Sub) || ["", "", ""];
      var v2Comp = v2CompParser.exec(v2Sub) || ["", "", ""];
      if (v1Comp[0].length == 0 && v2Comp[0].length == 0) {
        break;
      }
      var v1CompNum = v1Comp[1].length == 0 ? 0 : parseInt(v1Comp[1], 10);
      var v2CompNum = v2Comp[1].length == 0 ? 0 : parseInt(v2Comp[1], 10);
      order = goog.string.compareElements_(v1CompNum, v2CompNum) || goog.string.compareElements_(v1Comp[2].length == 0, v2Comp[2].length == 0) || goog.string.compareElements_(v1Comp[2], v2Comp[2]);
    } while (order == 0);
  }
  return order;
};
goog.string.compareElements_ = function(left, right) {
  if (left < right) {
    return -1;
  } else {
    if (left > right) {
      return 1;
    }
  }
  return 0;
};
goog.string.hashCode = function(str) {
  var result = 0;
  for (var i = 0;i < str.length;++i) {
    result = 31 * result + str.charCodeAt(i) >>> 0;
  }
  return result;
};
goog.string.uniqueStringCounter_ = Math.random() * 2147483648 | 0;
goog.string.createUniqueString = function() {
  return "goog_" + goog.string.uniqueStringCounter_++;
};
goog.string.toNumber = function(str) {
  var num = Number(str);
  if (num == 0 && goog.string.isEmptyOrWhitespace(str)) {
    return NaN;
  }
  return num;
};
goog.string.isLowerCamelCase = function(str) {
  return /^[a-z]+([A-Z][a-z]*)*$/.test(str);
};
goog.string.isUpperCamelCase = function(str) {
  return /^([A-Z][a-z]*)+$/.test(str);
};
goog.string.toCamelCase = function(str) {
  return String(str).replace(/\-([a-z])/g, function(all, match) {
    return match.toUpperCase();
  });
};
goog.string.toSelectorCase = function(str) {
  return String(str).replace(/([A-Z])/g, "-$1").toLowerCase();
};
goog.string.toTitleCase = function(str, opt_delimiters) {
  var delimiters = goog.isString(opt_delimiters) ? goog.string.regExpEscape(opt_delimiters) : "\\s";
  delimiters = delimiters ? "|[" + delimiters + "]+" : "";
  var regexp = new RegExp("(^" + delimiters + ")([a-z])", "g");
  return str.replace(regexp, function(all, p1, p2) {
    return p1 + p2.toUpperCase();
  });
};
goog.string.capitalize = function(str) {
  return String(str.charAt(0)).toUpperCase() + String(str.substr(1)).toLowerCase();
};
goog.string.parseInt = function(value) {
  if (isFinite(value)) {
    value = String(value);
  }
  if (goog.isString(value)) {
    return /^\s*-?0x/i.test(value) ? parseInt(value, 16) : parseInt(value, 10);
  }
  return NaN;
};
goog.string.splitLimit = function(str, separator, limit) {
  var parts = str.split(separator);
  var returnVal = [];
  while (limit > 0 && parts.length) {
    returnVal.push(parts.shift());
    limit--;
  }
  if (parts.length) {
    returnVal.push(parts.join(separator));
  }
  return returnVal;
};
goog.string.lastComponent = function(str, separators) {
  if (!separators) {
    return str;
  } else {
    if (typeof separators == "string") {
      separators = [separators];
    }
  }
  var lastSeparatorIndex = -1;
  for (var i = 0;i < separators.length;i++) {
    if (separators[i] == "") {
      continue;
    }
    var currentSeparatorIndex = str.lastIndexOf(separators[i]);
    if (currentSeparatorIndex > lastSeparatorIndex) {
      lastSeparatorIndex = currentSeparatorIndex;
    }
  }
  if (lastSeparatorIndex == -1) {
    return str;
  }
  return str.slice(lastSeparatorIndex + 1);
};
goog.string.editDistance = function(a, b) {
  var v0 = [];
  var v1 = [];
  if (a == b) {
    return 0;
  }
  if (!a.length || !b.length) {
    return Math.max(a.length, b.length);
  }
  for (var i = 0;i < b.length + 1;i++) {
    v0[i] = i;
  }
  for (var i = 0;i < a.length;i++) {
    v1[0] = i + 1;
    for (var j = 0;j < b.length;j++) {
      var cost = Number(a[i] != b[j]);
      v1[j + 1] = Math.min(v1[j] + 1, v0[j + 1] + 1, v0[j] + cost);
    }
    for (var j = 0;j < v0.length;j++) {
      v0[j] = v1[j];
    }
  }
  return v1[b.length];
};
goog.provide("goog.object");
goog.object.is = function(v, v2) {
  if (v === v2) {
    return v !== 0 || 1 / v === 1 / (v2);
  }
  return v !== v && v2 !== v2;
};
goog.object.forEach = function(obj, f, opt_obj) {
  for (var key in obj) {
    f.call((opt_obj), obj[key], key, obj);
  }
};
goog.object.filter = function(obj, f, opt_obj) {
  var res = {};
  for (var key in obj) {
    if (f.call((opt_obj), obj[key], key, obj)) {
      res[key] = obj[key];
    }
  }
  return res;
};
goog.object.map = function(obj, f, opt_obj) {
  var res = {};
  for (var key in obj) {
    res[key] = f.call((opt_obj), obj[key], key, obj);
  }
  return res;
};
goog.object.some = function(obj, f, opt_obj) {
  for (var key in obj) {
    if (f.call((opt_obj), obj[key], key, obj)) {
      return true;
    }
  }
  return false;
};
goog.object.every = function(obj, f, opt_obj) {
  for (var key in obj) {
    if (!f.call((opt_obj), obj[key], key, obj)) {
      return false;
    }
  }
  return true;
};
goog.object.getCount = function(obj) {
  var rv = 0;
  for (var key in obj) {
    rv++;
  }
  return rv;
};
goog.object.getAnyKey = function(obj) {
  for (var key in obj) {
    return key;
  }
};
goog.object.getAnyValue = function(obj) {
  for (var key in obj) {
    return obj[key];
  }
};
goog.object.contains = function(obj, val) {
  return goog.object.containsValue(obj, val);
};
goog.object.getValues = function(obj) {
  var res = [];
  var i = 0;
  for (var key in obj) {
    res[i++] = obj[key];
  }
  return res;
};
goog.object.getKeys = function(obj) {
  var res = [];
  var i = 0;
  for (var key in obj) {
    res[i++] = key;
  }
  return res;
};
goog.object.getValueByKeys = function(obj, var_args) {
  var isArrayLike = goog.isArrayLike(var_args);
  var keys = isArrayLike ? var_args : arguments;
  for (var i = isArrayLike ? 0 : 1;i < keys.length;i++) {
    obj = obj[keys[i]];
    if (!goog.isDef(obj)) {
      break;
    }
  }
  return obj;
};
goog.object.containsKey = function(obj, key) {
  return obj !== null && key in obj;
};
goog.object.containsValue = function(obj, val) {
  for (var key in obj) {
    if (obj[key] == val) {
      return true;
    }
  }
  return false;
};
goog.object.findKey = function(obj, f, opt_this) {
  for (var key in obj) {
    if (f.call((opt_this), obj[key], key, obj)) {
      return key;
    }
  }
  return undefined;
};
goog.object.findValue = function(obj, f, opt_this) {
  var key = goog.object.findKey(obj, f, opt_this);
  return key && obj[key];
};
goog.object.isEmpty = function(obj) {
  for (var key in obj) {
    return false;
  }
  return true;
};
goog.object.clear = function(obj) {
  for (var i in obj) {
    delete obj[i];
  }
};
goog.object.remove = function(obj, key) {
  var rv;
  if (rv = key in (obj)) {
    delete obj[key];
  }
  return rv;
};
goog.object.add = function(obj, key, val) {
  if (obj !== null && key in obj) {
    throw Error('The object already contains the key "' + key + '"');
  }
  goog.object.set(obj, key, val);
};
goog.object.get = function(obj, key, opt_val) {
  if (obj !== null && key in obj) {
    return obj[key];
  }
  return opt_val;
};
goog.object.set = function(obj, key, value) {
  obj[key] = value;
};
goog.object.setIfUndefined = function(obj, key, value) {
  return key in (obj) ? obj[key] : obj[key] = value;
};
goog.object.setWithReturnValueIfNotSet = function(obj, key, f) {
  if (key in obj) {
    return obj[key];
  }
  var val = f();
  obj[key] = val;
  return val;
};
goog.object.equals = function(a, b) {
  for (var k in a) {
    if (!(k in b) || a[k] !== b[k]) {
      return false;
    }
  }
  for (var k in b) {
    if (!(k in a)) {
      return false;
    }
  }
  return true;
};
goog.object.clone = function(obj) {
  var res = {};
  for (var key in obj) {
    res[key] = obj[key];
  }
  return res;
};
goog.object.unsafeClone = function(obj) {
  var type = goog.typeOf(obj);
  if (type == "object" || type == "array") {
    if (goog.isFunction(obj.clone)) {
      return obj.clone();
    }
    var clone = type == "array" ? [] : {};
    for (var key in obj) {
      clone[key] = goog.object.unsafeClone(obj[key]);
    }
    return clone;
  }
  return obj;
};
goog.object.transpose = function(obj) {
  var transposed = {};
  for (var key in obj) {
    transposed[obj[key]] = key;
  }
  return transposed;
};
goog.object.PROTOTYPE_FIELDS_ = ["constructor", "hasOwnProperty", "isPrototypeOf", "propertyIsEnumerable", "toLocaleString", "toString", "valueOf"];
goog.object.extend = function(target, var_args) {
  var key, source;
  for (var i = 1;i < arguments.length;i++) {
    source = arguments[i];
    for (key in source) {
      target[key] = source[key];
    }
    for (var j = 0;j < goog.object.PROTOTYPE_FIELDS_.length;j++) {
      key = goog.object.PROTOTYPE_FIELDS_[j];
      if (Object.prototype.hasOwnProperty.call(source, key)) {
        target[key] = source[key];
      }
    }
  }
};
goog.object.create = function(var_args) {
  var argLength = arguments.length;
  if (argLength == 1 && goog.isArray(arguments[0])) {
    return goog.object.create.apply(null, arguments[0]);
  }
  if (argLength % 2) {
    throw Error("Uneven number of arguments");
  }
  var rv = {};
  for (var i = 0;i < argLength;i += 2) {
    rv[arguments[i]] = arguments[i + 1];
  }
  return rv;
};
goog.object.createSet = function(var_args) {
  var argLength = arguments.length;
  if (argLength == 1 && goog.isArray(arguments[0])) {
    return goog.object.createSet.apply(null, arguments[0]);
  }
  var rv = {};
  for (var i = 0;i < argLength;i++) {
    rv[arguments[i]] = true;
  }
  return rv;
};
goog.object.createImmutableView = function(obj) {
  var result = obj;
  if (Object.isFrozen && !Object.isFrozen(obj)) {
    result = Object.create(obj);
    Object.freeze(result);
  }
  return result;
};
goog.object.isImmutableView = function(obj) {
  return !!Object.isFrozen && Object.isFrozen(obj);
};
goog.provide("goog.math.Integer");
goog.math.Integer = function(bits, sign) {
  this.bits_ = [];
  this.sign_ = sign;
  var top = true;
  for (var i = bits.length - 1;i >= 0;i--) {
    var val = bits[i] | 0;
    if (!top || val != sign) {
      this.bits_[i] = val;
      top = false;
    }
  }
};
goog.math.Integer.IntCache_ = {};
goog.math.Integer.fromInt = function(value) {
  if (-128 <= value && value < 128) {
    var cachedObj = goog.math.Integer.IntCache_[value];
    if (cachedObj) {
      return cachedObj;
    }
  }
  var obj = new goog.math.Integer([value | 0], value < 0 ? -1 : 0);
  if (-128 <= value && value < 128) {
    goog.math.Integer.IntCache_[value] = obj;
  }
  return obj;
};
goog.math.Integer.fromNumber = function(value) {
  if (isNaN(value) || !isFinite(value)) {
    return goog.math.Integer.ZERO;
  } else {
    if (value < 0) {
      return goog.math.Integer.fromNumber(-value).negate();
    } else {
      var bits = [];
      var pow = 1;
      for (var i = 0;value >= pow;i++) {
        bits[i] = value / pow | 0;
        pow *= goog.math.Integer.TWO_PWR_32_DBL_;
      }
      return new goog.math.Integer(bits, 0);
    }
  }
};
goog.math.Integer.fromBits = function(bits) {
  var high = bits[bits.length - 1];
  return new goog.math.Integer(bits, high & 1 << 31 ? -1 : 0);
};
goog.math.Integer.fromString = function(str, opt_radix) {
  if (str.length == 0) {
    throw Error("number format error: empty string");
  }
  var radix = opt_radix || 10;
  if (radix < 2 || 36 < radix) {
    throw Error("radix out of range: " + radix);
  }
  if (str.charAt(0) == "-") {
    return goog.math.Integer.fromString(str.substring(1), radix).negate();
  } else {
    if (str.indexOf("-") >= 0) {
      throw Error('number format error: interior "-" character');
    }
  }
  var radixToPower = goog.math.Integer.fromNumber(Math.pow(radix, 8));
  var result = goog.math.Integer.ZERO;
  for (var i = 0;i < str.length;i += 8) {
    var size = Math.min(8, str.length - i);
    var value = parseInt(str.substring(i, i + size), radix);
    if (size < 8) {
      var power = goog.math.Integer.fromNumber(Math.pow(radix, size));
      result = result.multiply(power).add(goog.math.Integer.fromNumber(value));
    } else {
      result = result.multiply(radixToPower);
      result = result.add(goog.math.Integer.fromNumber(value));
    }
  }
  return result;
};
goog.math.Integer.TWO_PWR_32_DBL_ = (1 << 16) * (1 << 16);
goog.math.Integer.ZERO = goog.math.Integer.fromInt(0);
goog.math.Integer.ONE = goog.math.Integer.fromInt(1);
goog.math.Integer.TWO_PWR_24_ = goog.math.Integer.fromInt(1 << 24);
goog.math.Integer.prototype.toInt = function() {
  return this.bits_.length > 0 ? this.bits_[0] : this.sign_;
};
goog.math.Integer.prototype.toNumber = function() {
  if (this.isNegative()) {
    return -this.negate().toNumber();
  } else {
    var val = 0;
    var pow = 1;
    for (var i = 0;i < this.bits_.length;i++) {
      val += this.getBitsUnsigned(i) * pow;
      pow *= goog.math.Integer.TWO_PWR_32_DBL_;
    }
    return val;
  }
};
goog.math.Integer.prototype.toString = function(opt_radix) {
  var radix = opt_radix || 10;
  if (radix < 2 || 36 < radix) {
    throw Error("radix out of range: " + radix);
  }
  if (this.isZero()) {
    return "0";
  } else {
    if (this.isNegative()) {
      return "-" + this.negate().toString(radix);
    }
  }
  var radixToPower = goog.math.Integer.fromNumber(Math.pow(radix, 6));
  var rem = this;
  var result = "";
  while (true) {
    var remDiv = rem.divide(radixToPower);
    var intval = rem.subtract(remDiv.multiply(radixToPower)).toInt() >>> 0;
    var digits = intval.toString(radix);
    rem = remDiv;
    if (rem.isZero()) {
      return digits + result;
    } else {
      while (digits.length < 6) {
        digits = "0" + digits;
      }
      result = "" + digits + result;
    }
  }
};
goog.math.Integer.prototype.getBits = function(index) {
  if (index < 0) {
    return 0;
  } else {
    if (index < this.bits_.length) {
      return this.bits_[index];
    } else {
      return this.sign_;
    }
  }
};
goog.math.Integer.prototype.getBitsUnsigned = function(index) {
  var val = this.getBits(index);
  return val >= 0 ? val : goog.math.Integer.TWO_PWR_32_DBL_ + val;
};
goog.math.Integer.prototype.getSign = function() {
  return this.sign_;
};
goog.math.Integer.prototype.isZero = function() {
  if (this.sign_ != 0) {
    return false;
  }
  for (var i = 0;i < this.bits_.length;i++) {
    if (this.bits_[i] != 0) {
      return false;
    }
  }
  return true;
};
goog.math.Integer.prototype.isNegative = function() {
  return this.sign_ == -1;
};
goog.math.Integer.prototype.isOdd = function() {
  return this.bits_.length == 0 && this.sign_ == -1 || this.bits_.length > 0 && (this.bits_[0] & 1) != 0;
};
goog.math.Integer.prototype.equals = function(other) {
  if (this.sign_ != other.sign_) {
    return false;
  }
  var len = Math.max(this.bits_.length, other.bits_.length);
  for (var i = 0;i < len;i++) {
    if (this.getBits(i) != other.getBits(i)) {
      return false;
    }
  }
  return true;
};
goog.math.Integer.prototype.notEquals = function(other) {
  return !this.equals(other);
};
goog.math.Integer.prototype.greaterThan = function(other) {
  return this.compare(other) > 0;
};
goog.math.Integer.prototype.greaterThanOrEqual = function(other) {
  return this.compare(other) >= 0;
};
goog.math.Integer.prototype.lessThan = function(other) {
  return this.compare(other) < 0;
};
goog.math.Integer.prototype.lessThanOrEqual = function(other) {
  return this.compare(other) <= 0;
};
goog.math.Integer.prototype.compare = function(other) {
  var diff = this.subtract(other);
  if (diff.isNegative()) {
    return -1;
  } else {
    if (diff.isZero()) {
      return 0;
    } else {
      return +1;
    }
  }
};
goog.math.Integer.prototype.shorten = function(numBits) {
  var arr_index = numBits - 1 >> 5;
  var bit_index = (numBits - 1) % 32;
  var bits = [];
  for (var i = 0;i < arr_index;i++) {
    bits[i] = this.getBits(i);
  }
  var sigBits = bit_index == 31 ? 4294967295 : (1 << bit_index + 1) - 1;
  var val = this.getBits(arr_index) & sigBits;
  if (val & 1 << bit_index) {
    val |= 4294967295 - sigBits;
    bits[arr_index] = val;
    return new goog.math.Integer(bits, -1);
  } else {
    bits[arr_index] = val;
    return new goog.math.Integer(bits, 0);
  }
};
goog.math.Integer.prototype.negate = function() {
  return this.not().add(goog.math.Integer.ONE);
};
goog.math.Integer.prototype.add = function(other) {
  var len = Math.max(this.bits_.length, other.bits_.length);
  var arr = [];
  var carry = 0;
  for (var i = 0;i <= len;i++) {
    var a1 = this.getBits(i) >>> 16;
    var a0 = this.getBits(i) & 65535;
    var b1 = other.getBits(i) >>> 16;
    var b0 = other.getBits(i) & 65535;
    var c0 = carry + a0 + b0;
    var c1 = (c0 >>> 16) + a1 + b1;
    carry = c1 >>> 16;
    c0 &= 65535;
    c1 &= 65535;
    arr[i] = c1 << 16 | c0;
  }
  return goog.math.Integer.fromBits(arr);
};
goog.math.Integer.prototype.subtract = function(other) {
  return this.add(other.negate());
};
goog.math.Integer.prototype.multiply = function(other) {
  if (this.isZero()) {
    return goog.math.Integer.ZERO;
  } else {
    if (other.isZero()) {
      return goog.math.Integer.ZERO;
    }
  }
  if (this.isNegative()) {
    if (other.isNegative()) {
      return this.negate().multiply(other.negate());
    } else {
      return this.negate().multiply(other).negate();
    }
  } else {
    if (other.isNegative()) {
      return this.multiply(other.negate()).negate();
    }
  }
  if (this.lessThan(goog.math.Integer.TWO_PWR_24_) && other.lessThan(goog.math.Integer.TWO_PWR_24_)) {
    return goog.math.Integer.fromNumber(this.toNumber() * other.toNumber());
  }
  var len = this.bits_.length + other.bits_.length;
  var arr = [];
  for (var i = 0;i < 2 * len;i++) {
    arr[i] = 0;
  }
  for (var i = 0;i < this.bits_.length;i++) {
    for (var j = 0;j < other.bits_.length;j++) {
      var a1 = this.getBits(i) >>> 16;
      var a0 = this.getBits(i) & 65535;
      var b1 = other.getBits(j) >>> 16;
      var b0 = other.getBits(j) & 65535;
      arr[2 * i + 2 * j] += a0 * b0;
      goog.math.Integer.carry16_(arr, 2 * i + 2 * j);
      arr[2 * i + 2 * j + 1] += a1 * b0;
      goog.math.Integer.carry16_(arr, 2 * i + 2 * j + 1);
      arr[2 * i + 2 * j + 1] += a0 * b1;
      goog.math.Integer.carry16_(arr, 2 * i + 2 * j + 1);
      arr[2 * i + 2 * j + 2] += a1 * b1;
      goog.math.Integer.carry16_(arr, 2 * i + 2 * j + 2);
    }
  }
  for (var i = 0;i < len;i++) {
    arr[i] = arr[2 * i + 1] << 16 | arr[2 * i];
  }
  for (var i = len;i < 2 * len;i++) {
    arr[i] = 0;
  }
  return new goog.math.Integer(arr, 0);
};
goog.math.Integer.carry16_ = function(bits, index) {
  while ((bits[index] & 65535) != bits[index]) {
    bits[index + 1] += bits[index] >>> 16;
    bits[index] &= 65535;
  }
};
goog.math.Integer.prototype.slowDivide_ = function(other) {
  if (this.isNegative() || other.isNegative()) {
    throw Error("slowDivide_ only works with positive integers.");
  }
  var twoPower = goog.math.Integer.ONE;
  var multiple = other;
  while (multiple.lessThanOrEqual(this)) {
    twoPower = twoPower.shiftLeft(1);
    multiple = multiple.shiftLeft(1);
  }
  var res = twoPower.shiftRight(1);
  var total = multiple.shiftRight(1);
  var total2;
  multiple = multiple.shiftRight(2);
  twoPower = twoPower.shiftRight(2);
  while (!multiple.isZero()) {
    total2 = total.add(multiple);
    if (total2.lessThanOrEqual(this)) {
      res = res.add(twoPower);
      total = total2;
    }
    multiple = multiple.shiftRight(1);
    twoPower = twoPower.shiftRight(1);
  }
  return res;
};
goog.math.Integer.prototype.divide = function(other) {
  if (other.isZero()) {
    throw Error("division by zero");
  } else {
    if (this.isZero()) {
      return goog.math.Integer.ZERO;
    }
  }
  if (this.isNegative()) {
    if (other.isNegative()) {
      return this.negate().divide(other.negate());
    } else {
      return this.negate().divide(other).negate();
    }
  } else {
    if (other.isNegative()) {
      return this.divide(other.negate()).negate();
    }
  }
  if (this.bits_.length > 30) {
    return this.slowDivide_(other);
  }
  var res = goog.math.Integer.ZERO;
  var rem = this;
  while (rem.greaterThanOrEqual(other)) {
    var approx = Math.max(1, Math.floor(rem.toNumber() / other.toNumber()));
    var log2 = Math.ceil(Math.log(approx) / Math.LN2);
    var delta = log2 <= 48 ? 1 : Math.pow(2, log2 - 48);
    var approxRes = goog.math.Integer.fromNumber(approx);
    var approxRem = approxRes.multiply(other);
    while (approxRem.isNegative() || approxRem.greaterThan(rem)) {
      approx -= delta;
      approxRes = goog.math.Integer.fromNumber(approx);
      approxRem = approxRes.multiply(other);
    }
    if (approxRes.isZero()) {
      approxRes = goog.math.Integer.ONE;
    }
    res = res.add(approxRes);
    rem = rem.subtract(approxRem);
  }
  return res;
};
goog.math.Integer.prototype.modulo = function(other) {
  return this.subtract(this.divide(other).multiply(other));
};
goog.math.Integer.prototype.not = function() {
  var len = this.bits_.length;
  var arr = [];
  for (var i = 0;i < len;i++) {
    arr[i] = ~this.bits_[i];
  }
  return new goog.math.Integer(arr, ~this.sign_);
};
goog.math.Integer.prototype.and = function(other) {
  var len = Math.max(this.bits_.length, other.bits_.length);
  var arr = [];
  for (var i = 0;i < len;i++) {
    arr[i] = this.getBits(i) & other.getBits(i);
  }
  return new goog.math.Integer(arr, this.sign_ & other.sign_);
};
goog.math.Integer.prototype.or = function(other) {
  var len = Math.max(this.bits_.length, other.bits_.length);
  var arr = [];
  for (var i = 0;i < len;i++) {
    arr[i] = this.getBits(i) | other.getBits(i);
  }
  return new goog.math.Integer(arr, this.sign_ | other.sign_);
};
goog.math.Integer.prototype.xor = function(other) {
  var len = Math.max(this.bits_.length, other.bits_.length);
  var arr = [];
  for (var i = 0;i < len;i++) {
    arr[i] = this.getBits(i) ^ other.getBits(i);
  }
  return new goog.math.Integer(arr, this.sign_ ^ other.sign_);
};
goog.math.Integer.prototype.shiftLeft = function(numBits) {
  var arr_delta = numBits >> 5;
  var bit_delta = numBits % 32;
  var len = this.bits_.length + arr_delta + (bit_delta > 0 ? 1 : 0);
  var arr = [];
  for (var i = 0;i < len;i++) {
    if (bit_delta > 0) {
      arr[i] = this.getBits(i - arr_delta) << bit_delta | this.getBits(i - arr_delta - 1) >>> 32 - bit_delta;
    } else {
      arr[i] = this.getBits(i - arr_delta);
    }
  }
  return new goog.math.Integer(arr, this.sign_);
};
goog.math.Integer.prototype.shiftRight = function(numBits) {
  var arr_delta = numBits >> 5;
  var bit_delta = numBits % 32;
  var len = this.bits_.length - arr_delta;
  var arr = [];
  for (var i = 0;i < len;i++) {
    if (bit_delta > 0) {
      arr[i] = this.getBits(i + arr_delta) >>> bit_delta | this.getBits(i + arr_delta + 1) << 32 - bit_delta;
    } else {
      arr[i] = this.getBits(i + arr_delta);
    }
  }
  return new goog.math.Integer(arr, this.sign_);
};
goog.provide("goog.string.StringBuffer");
goog.string.StringBuffer = function(opt_a1, var_args) {
  if (opt_a1 != null) {
    this.append.apply(this, arguments);
  }
};
goog.string.StringBuffer.prototype.buffer_ = "";
goog.string.StringBuffer.prototype.set = function(s) {
  this.buffer_ = "" + s;
};
goog.string.StringBuffer.prototype.append = function(a1, opt_a2, var_args) {
  this.buffer_ += String(a1);
  if (opt_a2 != null) {
    for (var i = 1;i < arguments.length;i++) {
      this.buffer_ += arguments[i];
    }
  }
  return this;
};
goog.string.StringBuffer.prototype.clear = function() {
  this.buffer_ = "";
};
goog.string.StringBuffer.prototype.getLength = function() {
  return this.buffer_.length;
};
goog.string.StringBuffer.prototype.toString = function() {
  return this.buffer_;
};
goog.provide("goog.debug.Error");
goog.debug.Error = function(opt_msg) {
  if (Error.captureStackTrace) {
    Error.captureStackTrace(this, goog.debug.Error);
  } else {
    var stack = (new Error).stack;
    if (stack) {
      this.stack = stack;
    }
  }
  if (opt_msg) {
    this.message = String(opt_msg);
  }
  this.reportErrorToServer = true;
};
goog.inherits(goog.debug.Error, Error);
goog.debug.Error.prototype.name = "CustomError";
goog.provide("goog.dom.NodeType");
goog.dom.NodeType = {ELEMENT:1, ATTRIBUTE:2, TEXT:3, CDATA_SECTION:4, ENTITY_REFERENCE:5, ENTITY:6, PROCESSING_INSTRUCTION:7, COMMENT:8, DOCUMENT:9, DOCUMENT_TYPE:10, DOCUMENT_FRAGMENT:11, NOTATION:12};
goog.provide("goog.asserts");
goog.provide("goog.asserts.AssertionError");
goog.require("goog.debug.Error");
goog.require("goog.dom.NodeType");
goog.require("goog.string");
goog.define("goog.asserts.ENABLE_ASSERTS", goog.DEBUG);
goog.asserts.AssertionError = function(messagePattern, messageArgs) {
  messageArgs.unshift(messagePattern);
  goog.debug.Error.call(this, goog.string.subs.apply(null, messageArgs));
  messageArgs.shift();
  this.messagePattern = messagePattern;
};
goog.inherits(goog.asserts.AssertionError, goog.debug.Error);
goog.asserts.AssertionError.prototype.name = "AssertionError";
goog.asserts.DEFAULT_ERROR_HANDLER = function(e) {
  throw e;
};
goog.asserts.errorHandler_ = goog.asserts.DEFAULT_ERROR_HANDLER;
goog.asserts.doAssertFailure_ = function(defaultMessage, defaultArgs, givenMessage, givenArgs) {
  var message = "Assertion failed";
  if (givenMessage) {
    message += ": " + givenMessage;
    var args = givenArgs;
  } else {
    if (defaultMessage) {
      message += ": " + defaultMessage;
      args = defaultArgs;
    }
  }
  var e = new goog.asserts.AssertionError("" + message, args || []);
  goog.asserts.errorHandler_(e);
};
goog.asserts.setErrorHandler = function(errorHandler) {
  if (goog.asserts.ENABLE_ASSERTS) {
    goog.asserts.errorHandler_ = errorHandler;
  }
};
goog.asserts.assert = function(condition, opt_message, var_args) {
  if (goog.asserts.ENABLE_ASSERTS && !condition) {
    goog.asserts.doAssertFailure_("", null, opt_message, Array.prototype.slice.call(arguments, 2));
  }
  return condition;
};
goog.asserts.fail = function(opt_message, var_args) {
  if (goog.asserts.ENABLE_ASSERTS) {
    goog.asserts.errorHandler_(new goog.asserts.AssertionError("Failure" + (opt_message ? ": " + opt_message : ""), Array.prototype.slice.call(arguments, 1)));
  }
};
goog.asserts.assertNumber = function(value, opt_message, var_args) {
  if (goog.asserts.ENABLE_ASSERTS && !goog.isNumber(value)) {
    goog.asserts.doAssertFailure_("Expected number but got %s: %s.", [goog.typeOf(value), value], opt_message, Array.prototype.slice.call(arguments, 2));
  }
  return (value);
};
goog.asserts.assertString = function(value, opt_message, var_args) {
  if (goog.asserts.ENABLE_ASSERTS && !goog.isString(value)) {
    goog.asserts.doAssertFailure_("Expected string but got %s: %s.", [goog.typeOf(value), value], opt_message, Array.prototype.slice.call(arguments, 2));
  }
  return (value);
};
goog.asserts.assertFunction = function(value, opt_message, var_args) {
  if (goog.asserts.ENABLE_ASSERTS && !goog.isFunction(value)) {
    goog.asserts.doAssertFailure_("Expected function but got %s: %s.", [goog.typeOf(value), value], opt_message, Array.prototype.slice.call(arguments, 2));
  }
  return (value);
};
goog.asserts.assertObject = function(value, opt_message, var_args) {
  if (goog.asserts.ENABLE_ASSERTS && !goog.isObject(value)) {
    goog.asserts.doAssertFailure_("Expected object but got %s: %s.", [goog.typeOf(value), value], opt_message, Array.prototype.slice.call(arguments, 2));
  }
  return (value);
};
goog.asserts.assertArray = function(value, opt_message, var_args) {
  if (goog.asserts.ENABLE_ASSERTS && !goog.isArray(value)) {
    goog.asserts.doAssertFailure_("Expected array but got %s: %s.", [goog.typeOf(value), value], opt_message, Array.prototype.slice.call(arguments, 2));
  }
  return (value);
};
goog.asserts.assertBoolean = function(value, opt_message, var_args) {
  if (goog.asserts.ENABLE_ASSERTS && !goog.isBoolean(value)) {
    goog.asserts.doAssertFailure_("Expected boolean but got %s: %s.", [goog.typeOf(value), value], opt_message, Array.prototype.slice.call(arguments, 2));
  }
  return (value);
};
goog.asserts.assertElement = function(value, opt_message, var_args) {
  if (goog.asserts.ENABLE_ASSERTS && (!goog.isObject(value) || value.nodeType != goog.dom.NodeType.ELEMENT)) {
    goog.asserts.doAssertFailure_("Expected Element but got %s: %s.", [goog.typeOf(value), value], opt_message, Array.prototype.slice.call(arguments, 2));
  }
  return (value);
};
goog.asserts.assertInstanceof = function(value, type, opt_message, var_args) {
  if (goog.asserts.ENABLE_ASSERTS && !(value instanceof type)) {
    goog.asserts.doAssertFailure_("Expected instanceof %s but got %s.", [goog.asserts.getType_(type), goog.asserts.getType_(value)], opt_message, Array.prototype.slice.call(arguments, 3));
  }
  return value;
};
goog.asserts.assertObjectPrototypeIsIntact = function() {
  for (var key in Object.prototype) {
    goog.asserts.fail(key + " should not be enumerable in Object.prototype.");
  }
};
goog.asserts.getType_ = function(value) {
  if (value instanceof Function) {
    return value.displayName || value.name || "unknown type name";
  } else {
    if (value instanceof Object) {
      return value.constructor.displayName || value.constructor.name || Object.prototype.toString.call(value);
    } else {
      return value === null ? "null" : typeof value;
    }
  }
};
goog.provide("goog.array");
goog.require("goog.asserts");
goog.define("goog.NATIVE_ARRAY_PROTOTYPES", goog.TRUSTED_SITE);
goog.define("goog.array.ASSUME_NATIVE_FUNCTIONS", false);
goog.array.peek = function(array) {
  return array[array.length - 1];
};
goog.array.last = goog.array.peek;
goog.array.indexOf = goog.NATIVE_ARRAY_PROTOTYPES && (goog.array.ASSUME_NATIVE_FUNCTIONS || Array.prototype.indexOf) ? function(arr, obj, opt_fromIndex) {
  goog.asserts.assert(arr.length != null);
  return Array.prototype.indexOf.call(arr, obj, opt_fromIndex);
} : function(arr, obj, opt_fromIndex) {
  var fromIndex = opt_fromIndex == null ? 0 : opt_fromIndex < 0 ? Math.max(0, arr.length + opt_fromIndex) : opt_fromIndex;
  if (goog.isString(arr)) {
    if (!goog.isString(obj) || obj.length != 1) {
      return -1;
    }
    return arr.indexOf(obj, fromIndex);
  }
  for (var i = fromIndex;i < arr.length;i++) {
    if (i in arr && arr[i] === obj) {
      return i;
    }
  }
  return -1;
};
goog.array.lastIndexOf = goog.NATIVE_ARRAY_PROTOTYPES && (goog.array.ASSUME_NATIVE_FUNCTIONS || Array.prototype.lastIndexOf) ? function(arr, obj, opt_fromIndex) {
  goog.asserts.assert(arr.length != null);
  var fromIndex = opt_fromIndex == null ? arr.length - 1 : opt_fromIndex;
  return Array.prototype.lastIndexOf.call(arr, obj, fromIndex);
} : function(arr, obj, opt_fromIndex) {
  var fromIndex = opt_fromIndex == null ? arr.length - 1 : opt_fromIndex;
  if (fromIndex < 0) {
    fromIndex = Math.max(0, arr.length + fromIndex);
  }
  if (goog.isString(arr)) {
    if (!goog.isString(obj) || obj.length != 1) {
      return -1;
    }
    return arr.lastIndexOf(obj, fromIndex);
  }
  for (var i = fromIndex;i >= 0;i--) {
    if (i in arr && arr[i] === obj) {
      return i;
    }
  }
  return -1;
};
goog.array.forEach = goog.NATIVE_ARRAY_PROTOTYPES && (goog.array.ASSUME_NATIVE_FUNCTIONS || Array.prototype.forEach) ? function(arr, f, opt_obj) {
  goog.asserts.assert(arr.length != null);
  Array.prototype.forEach.call(arr, f, opt_obj);
} : function(arr, f, opt_obj) {
  var l = arr.length;
  var arr2 = goog.isString(arr) ? arr.split("") : arr;
  for (var i = 0;i < l;i++) {
    if (i in arr2) {
      f.call((opt_obj), arr2[i], i, arr);
    }
  }
};
goog.array.forEachRight = function(arr, f, opt_obj) {
  var l = arr.length;
  var arr2 = goog.isString(arr) ? arr.split("") : arr;
  for (var i = l - 1;i >= 0;--i) {
    if (i in arr2) {
      f.call((opt_obj), arr2[i], i, arr);
    }
  }
};
goog.array.filter = goog.NATIVE_ARRAY_PROTOTYPES && (goog.array.ASSUME_NATIVE_FUNCTIONS || Array.prototype.filter) ? function(arr, f, opt_obj) {
  goog.asserts.assert(arr.length != null);
  return Array.prototype.filter.call(arr, f, opt_obj);
} : function(arr, f, opt_obj) {
  var l = arr.length;
  var res = [];
  var resLength = 0;
  var arr2 = goog.isString(arr) ? arr.split("") : arr;
  for (var i = 0;i < l;i++) {
    if (i in arr2) {
      var val = arr2[i];
      if (f.call((opt_obj), val, i, arr)) {
        res[resLength++] = val;
      }
    }
  }
  return res;
};
goog.array.map = goog.NATIVE_ARRAY_PROTOTYPES && (goog.array.ASSUME_NATIVE_FUNCTIONS || Array.prototype.map) ? function(arr, f, opt_obj) {
  goog.asserts.assert(arr.length != null);
  return Array.prototype.map.call(arr, f, opt_obj);
} : function(arr, f, opt_obj) {
  var l = arr.length;
  var res = new Array(l);
  var arr2 = goog.isString(arr) ? arr.split("") : arr;
  for (var i = 0;i < l;i++) {
    if (i in arr2) {
      res[i] = f.call((opt_obj), arr2[i], i, arr);
    }
  }
  return res;
};
goog.array.reduce = goog.NATIVE_ARRAY_PROTOTYPES && (goog.array.ASSUME_NATIVE_FUNCTIONS || Array.prototype.reduce) ? function(arr, f, val, opt_obj) {
  goog.asserts.assert(arr.length != null);
  if (opt_obj) {
    f = goog.bind(f, opt_obj);
  }
  return Array.prototype.reduce.call(arr, f, val);
} : function(arr, f, val, opt_obj) {
  var rval = val;
  goog.array.forEach(arr, function(val, index) {
    rval = f.call((opt_obj), rval, val, index, arr);
  });
  return rval;
};
goog.array.reduceRight = goog.NATIVE_ARRAY_PROTOTYPES && (goog.array.ASSUME_NATIVE_FUNCTIONS || Array.prototype.reduceRight) ? function(arr, f, val, opt_obj) {
  goog.asserts.assert(arr.length != null);
  goog.asserts.assert(f != null);
  if (opt_obj) {
    f = goog.bind(f, opt_obj);
  }
  return Array.prototype.reduceRight.call(arr, f, val);
} : function(arr, f, val, opt_obj) {
  var rval = val;
  goog.array.forEachRight(arr, function(val, index) {
    rval = f.call((opt_obj), rval, val, index, arr);
  });
  return rval;
};
goog.array.some = goog.NATIVE_ARRAY_PROTOTYPES && (goog.array.ASSUME_NATIVE_FUNCTIONS || Array.prototype.some) ? function(arr, f, opt_obj) {
  goog.asserts.assert(arr.length != null);
  return Array.prototype.some.call(arr, f, opt_obj);
} : function(arr, f, opt_obj) {
  var l = arr.length;
  var arr2 = goog.isString(arr) ? arr.split("") : arr;
  for (var i = 0;i < l;i++) {
    if (i in arr2 && f.call((opt_obj), arr2[i], i, arr)) {
      return true;
    }
  }
  return false;
};
goog.array.every = goog.NATIVE_ARRAY_PROTOTYPES && (goog.array.ASSUME_NATIVE_FUNCTIONS || Array.prototype.every) ? function(arr, f, opt_obj) {
  goog.asserts.assert(arr.length != null);
  return Array.prototype.every.call(arr, f, opt_obj);
} : function(arr, f, opt_obj) {
  var l = arr.length;
  var arr2 = goog.isString(arr) ? arr.split("") : arr;
  for (var i = 0;i < l;i++) {
    if (i in arr2 && !f.call((opt_obj), arr2[i], i, arr)) {
      return false;
    }
  }
  return true;
};
goog.array.count = function(arr, f, opt_obj) {
  var count = 0;
  goog.array.forEach(arr, function(element, index, arr) {
    if (f.call((opt_obj), element, index, arr)) {
      ++count;
    }
  }, opt_obj);
  return count;
};
goog.array.find = function(arr, f, opt_obj) {
  var i = goog.array.findIndex(arr, f, opt_obj);
  return i < 0 ? null : goog.isString(arr) ? arr.charAt(i) : arr[i];
};
goog.array.findIndex = function(arr, f, opt_obj) {
  var l = arr.length;
  var arr2 = goog.isString(arr) ? arr.split("") : arr;
  for (var i = 0;i < l;i++) {
    if (i in arr2 && f.call((opt_obj), arr2[i], i, arr)) {
      return i;
    }
  }
  return -1;
};
goog.array.findRight = function(arr, f, opt_obj) {
  var i = goog.array.findIndexRight(arr, f, opt_obj);
  return i < 0 ? null : goog.isString(arr) ? arr.charAt(i) : arr[i];
};
goog.array.findIndexRight = function(arr, f, opt_obj) {
  var l = arr.length;
  var arr2 = goog.isString(arr) ? arr.split("") : arr;
  for (var i = l - 1;i >= 0;i--) {
    if (i in arr2 && f.call((opt_obj), arr2[i], i, arr)) {
      return i;
    }
  }
  return -1;
};
goog.array.contains = function(arr, obj) {
  return goog.array.indexOf(arr, obj) >= 0;
};
goog.array.isEmpty = function(arr) {
  return arr.length == 0;
};
goog.array.clear = function(arr) {
  if (!goog.isArray(arr)) {
    for (var i = arr.length - 1;i >= 0;i--) {
      delete arr[i];
    }
  }
  arr.length = 0;
};
goog.array.insert = function(arr, obj) {
  if (!goog.array.contains(arr, obj)) {
    arr.push(obj);
  }
};
goog.array.insertAt = function(arr, obj, opt_i) {
  goog.array.splice(arr, opt_i, 0, obj);
};
goog.array.insertArrayAt = function(arr, elementsToAdd, opt_i) {
  goog.partial(goog.array.splice, arr, opt_i, 0).apply(null, elementsToAdd);
};
goog.array.insertBefore = function(arr, obj, opt_obj2) {
  var i;
  if (arguments.length == 2 || (i = goog.array.indexOf(arr, opt_obj2)) < 0) {
    arr.push(obj);
  } else {
    goog.array.insertAt(arr, obj, i);
  }
};
goog.array.remove = function(arr, obj) {
  var i = goog.array.indexOf(arr, obj);
  var rv;
  if (rv = i >= 0) {
    goog.array.removeAt(arr, i);
  }
  return rv;
};
goog.array.removeLast = function(arr, obj) {
  var i = goog.array.lastIndexOf(arr, obj);
  if (i >= 0) {
    goog.array.removeAt(arr, i);
    return true;
  }
  return false;
};
goog.array.removeAt = function(arr, i) {
  goog.asserts.assert(arr.length != null);
  return Array.prototype.splice.call(arr, i, 1).length == 1;
};
goog.array.removeIf = function(arr, f, opt_obj) {
  var i = goog.array.findIndex(arr, f, opt_obj);
  if (i >= 0) {
    goog.array.removeAt(arr, i);
    return true;
  }
  return false;
};
goog.array.removeAllIf = function(arr, f, opt_obj) {
  var removedCount = 0;
  goog.array.forEachRight(arr, function(val, index) {
    if (f.call((opt_obj), val, index, arr)) {
      if (goog.array.removeAt(arr, index)) {
        removedCount++;
      }
    }
  });
  return removedCount;
};
goog.array.concat = function(var_args) {
  return Array.prototype.concat.apply(Array.prototype, arguments);
};
goog.array.join = function(var_args) {
  return Array.prototype.concat.apply(Array.prototype, arguments);
};
goog.array.toArray = function(object) {
  var length = object.length;
  if (length > 0) {
    var rv = new Array(length);
    for (var i = 0;i < length;i++) {
      rv[i] = object[i];
    }
    return rv;
  }
  return [];
};
goog.array.clone = goog.array.toArray;
goog.array.extend = function(arr1, var_args) {
  for (var i = 1;i < arguments.length;i++) {
    var arr2 = arguments[i];
    if (goog.isArrayLike(arr2)) {
      var len1 = arr1.length || 0;
      var len2 = arr2.length || 0;
      arr1.length = len1 + len2;
      for (var j = 0;j < len2;j++) {
        arr1[len1 + j] = arr2[j];
      }
    } else {
      arr1.push(arr2);
    }
  }
};
goog.array.splice = function(arr, index, howMany, var_args) {
  goog.asserts.assert(arr.length != null);
  return Array.prototype.splice.apply(arr, goog.array.slice(arguments, 1));
};
goog.array.slice = function(arr, start, opt_end) {
  goog.asserts.assert(arr.length != null);
  if (arguments.length <= 2) {
    return Array.prototype.slice.call(arr, start);
  } else {
    return Array.prototype.slice.call(arr, start, opt_end);
  }
};
goog.array.removeDuplicates = function(arr, opt_rv, opt_hashFn) {
  var returnArray = opt_rv || arr;
  var defaultHashFn = function(item) {
    return goog.isObject(item) ? "o" + goog.getUid(item) : (typeof item).charAt(0) + item;
  };
  var hashFn = opt_hashFn || defaultHashFn;
  var seen = {}, cursorInsert = 0, cursorRead = 0;
  while (cursorRead < arr.length) {
    var current = arr[cursorRead++];
    var key = hashFn(current);
    if (!Object.prototype.hasOwnProperty.call(seen, key)) {
      seen[key] = true;
      returnArray[cursorInsert++] = current;
    }
  }
  returnArray.length = cursorInsert;
};
goog.array.binarySearch = function(arr, target, opt_compareFn) {
  return goog.array.binarySearch_(arr, opt_compareFn || goog.array.defaultCompare, false, target);
};
goog.array.binarySelect = function(arr, evaluator, opt_obj) {
  return goog.array.binarySearch_(arr, evaluator, true, undefined, opt_obj);
};
goog.array.binarySearch_ = function(arr, compareFn, isEvaluator, opt_target, opt_selfObj) {
  var left = 0;
  var right = arr.length;
  var found;
  while (left < right) {
    var middle = left + right >> 1;
    var compareResult;
    if (isEvaluator) {
      compareResult = compareFn.call(opt_selfObj, arr[middle], middle, arr);
    } else {
      compareResult = (compareFn)(opt_target, arr[middle]);
    }
    if (compareResult > 0) {
      left = middle + 1;
    } else {
      right = middle;
      found = !compareResult;
    }
  }
  return found ? left : ~left;
};
goog.array.sort = function(arr, opt_compareFn) {
  arr.sort(opt_compareFn || goog.array.defaultCompare);
};
goog.array.stableSort = function(arr, opt_compareFn) {
  var compArr = new Array(arr.length);
  for (var i = 0;i < arr.length;i++) {
    compArr[i] = {index:i, value:arr[i]};
  }
  var valueCompareFn = opt_compareFn || goog.array.defaultCompare;
  function stableCompareFn(obj1, obj2) {
    return valueCompareFn(obj1.value, obj2.value) || obj1.index - obj2.index;
  }
  goog.array.sort(compArr, stableCompareFn);
  for (var i = 0;i < arr.length;i++) {
    arr[i] = compArr[i].value;
  }
};
goog.array.sortByKey = function(arr, keyFn, opt_compareFn) {
  var keyCompareFn = opt_compareFn || goog.array.defaultCompare;
  goog.array.sort(arr, function(a, b) {
    return keyCompareFn(keyFn(a), keyFn(b));
  });
};
goog.array.sortObjectsByKey = function(arr, key, opt_compareFn) {
  goog.array.sortByKey(arr, function(obj) {
    return obj[key];
  }, opt_compareFn);
};
goog.array.isSorted = function(arr, opt_compareFn, opt_strict) {
  var compare = opt_compareFn || goog.array.defaultCompare;
  for (var i = 1;i < arr.length;i++) {
    var compareResult = compare(arr[i - 1], arr[i]);
    if (compareResult > 0 || compareResult == 0 && opt_strict) {
      return false;
    }
  }
  return true;
};
goog.array.equals = function(arr1, arr2, opt_equalsFn) {
  if (!goog.isArrayLike(arr1) || !goog.isArrayLike(arr2) || arr1.length != arr2.length) {
    return false;
  }
  var l = arr1.length;
  var equalsFn = opt_equalsFn || goog.array.defaultCompareEquality;
  for (var i = 0;i < l;i++) {
    if (!equalsFn(arr1[i], arr2[i])) {
      return false;
    }
  }
  return true;
};
goog.array.compare3 = function(arr1, arr2, opt_compareFn) {
  var compare = opt_compareFn || goog.array.defaultCompare;
  var l = Math.min(arr1.length, arr2.length);
  for (var i = 0;i < l;i++) {
    var result = compare(arr1[i], arr2[i]);
    if (result != 0) {
      return result;
    }
  }
  return goog.array.defaultCompare(arr1.length, arr2.length);
};
goog.array.defaultCompare = function(a, b) {
  return a > b ? 1 : a < b ? -1 : 0;
};
goog.array.inverseDefaultCompare = function(a, b) {
  return -goog.array.defaultCompare(a, b);
};
goog.array.defaultCompareEquality = function(a, b) {
  return a === b;
};
goog.array.binaryInsert = function(array, value, opt_compareFn) {
  var index = goog.array.binarySearch(array, value, opt_compareFn);
  if (index < 0) {
    goog.array.insertAt(array, value, -(index + 1));
    return true;
  }
  return false;
};
goog.array.binaryRemove = function(array, value, opt_compareFn) {
  var index = goog.array.binarySearch(array, value, opt_compareFn);
  return index >= 0 ? goog.array.removeAt(array, index) : false;
};
goog.array.bucket = function(array, sorter, opt_obj) {
  var buckets = {};
  for (var i = 0;i < array.length;i++) {
    var value = array[i];
    var key = sorter.call((opt_obj), value, i, array);
    if (goog.isDef(key)) {
      var bucket = buckets[key] || (buckets[key] = []);
      bucket.push(value);
    }
  }
  return buckets;
};
goog.array.toObject = function(arr, keyFunc, opt_obj) {
  var ret = {};
  goog.array.forEach(arr, function(element, index) {
    ret[keyFunc.call((opt_obj), element, index, arr)] = element;
  });
  return ret;
};
goog.array.range = function(startOrEnd, opt_end, opt_step) {
  var array = [];
  var start = 0;
  var end = startOrEnd;
  var step = opt_step || 1;
  if (opt_end !== undefined) {
    start = startOrEnd;
    end = opt_end;
  }
  if (step * (end - start) < 0) {
    return [];
  }
  if (step > 0) {
    for (var i = start;i < end;i += step) {
      array.push(i);
    }
  } else {
    for (var i = start;i > end;i += step) {
      array.push(i);
    }
  }
  return array;
};
goog.array.repeat = function(value, n) {
  var array = [];
  for (var i = 0;i < n;i++) {
    array[i] = value;
  }
  return array;
};
goog.array.flatten = function(var_args) {
  var CHUNK_SIZE = 8192;
  var result = [];
  for (var i = 0;i < arguments.length;i++) {
    var element = arguments[i];
    if (goog.isArray(element)) {
      for (var c = 0;c < element.length;c += CHUNK_SIZE) {
        var chunk = goog.array.slice(element, c, c + CHUNK_SIZE);
        var recurseResult = goog.array.flatten.apply(null, chunk);
        for (var r = 0;r < recurseResult.length;r++) {
          result.push(recurseResult[r]);
        }
      }
    } else {
      result.push(element);
    }
  }
  return result;
};
goog.array.rotate = function(array, n) {
  goog.asserts.assert(array.length != null);
  if (array.length) {
    n %= array.length;
    if (n > 0) {
      Array.prototype.unshift.apply(array, array.splice(-n, n));
    } else {
      if (n < 0) {
        Array.prototype.push.apply(array, array.splice(0, -n));
      }
    }
  }
  return array;
};
goog.array.moveItem = function(arr, fromIndex, toIndex) {
  goog.asserts.assert(fromIndex >= 0 && fromIndex < arr.length);
  goog.asserts.assert(toIndex >= 0 && toIndex < arr.length);
  var removedItems = Array.prototype.splice.call(arr, fromIndex, 1);
  Array.prototype.splice.call(arr, toIndex, 0, removedItems[0]);
};
goog.array.zip = function(var_args) {
  if (!arguments.length) {
    return [];
  }
  var result = [];
  var minLen = arguments[0].length;
  for (var i = 1;i < arguments.length;i++) {
    if (arguments[i].length < minLen) {
      minLen = arguments[i].length;
    }
  }
  for (var i = 0;i < minLen;i++) {
    var value = [];
    for (var j = 0;j < arguments.length;j++) {
      value.push(arguments[j][i]);
    }
    result.push(value);
  }
  return result;
};
goog.array.shuffle = function(arr, opt_randFn) {
  var randFn = opt_randFn || Math.random;
  for (var i = arr.length - 1;i > 0;i--) {
    var j = Math.floor(randFn() * (i + 1));
    var tmp = arr[i];
    arr[i] = arr[j];
    arr[j] = tmp;
  }
};
goog.array.copyByIndex = function(arr, index_arr) {
  var result = [];
  goog.array.forEach(index_arr, function(index) {
    result.push(arr[index]);
  });
  return result;
};
goog.array.concatMap = function(arr, f, opt_obj) {
  return goog.array.concat.apply([], goog.array.map(arr, f, opt_obj));
};
goog.provide("goog.reflect");
goog.reflect.object = function(type, object) {
  return object;
};
goog.reflect.objectProperty = function(prop, object) {
  return prop;
};
goog.reflect.sinkValue = function(x) {
  goog.reflect.sinkValue[" "](x);
  return x;
};
goog.reflect.sinkValue[" "] = goog.nullFunction;
goog.reflect.canAccessProperty = function(obj, prop) {
  try {
    goog.reflect.sinkValue(obj[prop]);
    return true;
  } catch (e) {
  }
  return false;
};
goog.reflect.cache = function(cacheObj, key, valueFn, opt_keyFn) {
  var storedKey = opt_keyFn ? opt_keyFn(key) : key;
  if (Object.prototype.hasOwnProperty.call(cacheObj, storedKey)) {
    return cacheObj[storedKey];
  }
  return cacheObj[storedKey] = valueFn(key);
};
goog.provide("goog.math.Long");
goog.require("goog.reflect");
goog.math.Long = function(low, high) {
  this.low_ = low | 0;
  this.high_ = high | 0;
};
goog.math.Long.IntCache_ = {};
goog.math.Long.valueCache_ = {};
goog.math.Long.fromInt = function(value) {
  if (-128 <= value && value < 128) {
    return goog.reflect.cache(goog.math.Long.IntCache_, value, function(val) {
      return new goog.math.Long(val | 0, val < 0 ? -1 : 0);
    });
  } else {
    return new goog.math.Long(value | 0, value < 0 ? -1 : 0);
  }
};
goog.math.Long.fromNumber = function(value) {
  if (isNaN(value)) {
    return goog.math.Long.getZero();
  } else {
    if (value <= -goog.math.Long.TWO_PWR_63_DBL_) {
      return goog.math.Long.getMinValue();
    } else {
      if (value + 1 >= goog.math.Long.TWO_PWR_63_DBL_) {
        return goog.math.Long.getMaxValue();
      } else {
        if (value < 0) {
          return goog.math.Long.fromNumber(-value).negate();
        } else {
          return new goog.math.Long(value % goog.math.Long.TWO_PWR_32_DBL_ | 0, value / goog.math.Long.TWO_PWR_32_DBL_ | 0);
        }
      }
    }
  }
};
goog.math.Long.fromBits = function(lowBits, highBits) {
  return new goog.math.Long(lowBits, highBits);
};
goog.math.Long.fromString = function(str, opt_radix) {
  if (str.length == 0) {
    throw Error("number format error: empty string");
  }
  var radix = opt_radix || 10;
  if (radix < 2 || 36 < radix) {
    throw Error("radix out of range: " + radix);
  }
  if (str.charAt(0) == "-") {
    return goog.math.Long.fromString(str.substring(1), radix).negate();
  } else {
    if (str.indexOf("-") >= 0) {
      throw Error('number format error: interior "-" character: ' + str);
    }
  }
  var radixToPower = goog.math.Long.fromNumber(Math.pow(radix, 8));
  var result = goog.math.Long.getZero();
  for (var i = 0;i < str.length;i += 8) {
    var size = Math.min(8, str.length - i);
    var value = parseInt(str.substring(i, i + size), radix);
    if (size < 8) {
      var power = goog.math.Long.fromNumber(Math.pow(radix, size));
      result = result.multiply(power).add(goog.math.Long.fromNumber(value));
    } else {
      result = result.multiply(radixToPower);
      result = result.add(goog.math.Long.fromNumber(value));
    }
  }
  return result;
};
goog.math.Long.TWO_PWR_16_DBL_ = 1 << 16;
goog.math.Long.TWO_PWR_32_DBL_ = goog.math.Long.TWO_PWR_16_DBL_ * goog.math.Long.TWO_PWR_16_DBL_;
goog.math.Long.TWO_PWR_64_DBL_ = goog.math.Long.TWO_PWR_32_DBL_ * goog.math.Long.TWO_PWR_32_DBL_;
goog.math.Long.TWO_PWR_63_DBL_ = goog.math.Long.TWO_PWR_64_DBL_ / 2;
goog.math.Long.getZero = function() {
  return goog.reflect.cache(goog.math.Long.valueCache_, goog.math.Long.ValueCacheId_.ZERO, function() {
    return goog.math.Long.fromInt(0);
  });
};
goog.math.Long.getOne = function() {
  return goog.reflect.cache(goog.math.Long.valueCache_, goog.math.Long.ValueCacheId_.ONE, function() {
    return goog.math.Long.fromInt(1);
  });
};
goog.math.Long.getNegOne = function() {
  return goog.reflect.cache(goog.math.Long.valueCache_, goog.math.Long.ValueCacheId_.NEG_ONE, function() {
    return goog.math.Long.fromInt(-1);
  });
};
goog.math.Long.getMaxValue = function() {
  return goog.reflect.cache(goog.math.Long.valueCache_, goog.math.Long.ValueCacheId_.MAX_VALUE, function() {
    return goog.math.Long.fromBits(4294967295 | 0, 2147483647 | 0);
  });
};
goog.math.Long.getMinValue = function() {
  return goog.reflect.cache(goog.math.Long.valueCache_, goog.math.Long.ValueCacheId_.MIN_VALUE, function() {
    return goog.math.Long.fromBits(0, 2147483648 | 0);
  });
};
goog.math.Long.getTwoPwr24 = function() {
  return goog.reflect.cache(goog.math.Long.valueCache_, goog.math.Long.ValueCacheId_.TWO_PWR_24, function() {
    return goog.math.Long.fromInt(1 << 24);
  });
};
goog.math.Long.prototype.toInt = function() {
  return this.low_;
};
goog.math.Long.prototype.toNumber = function() {
  return this.high_ * goog.math.Long.TWO_PWR_32_DBL_ + this.getLowBitsUnsigned();
};
goog.math.Long.prototype.toString = function(opt_radix) {
  var radix = opt_radix || 10;
  if (radix < 2 || 36 < radix) {
    throw Error("radix out of range: " + radix);
  }
  if (this.isZero()) {
    return "0";
  }
  if (this.isNegative()) {
    if (this.equals(goog.math.Long.getMinValue())) {
      var radixLong = goog.math.Long.fromNumber(radix);
      var div = this.div(radixLong);
      var rem = div.multiply(radixLong).subtract(this);
      return div.toString(radix) + rem.toInt().toString(radix);
    } else {
      return "-" + this.negate().toString(radix);
    }
  }
  var radixToPower = goog.math.Long.fromNumber(Math.pow(radix, 6));
  var rem = this;
  var result = "";
  while (true) {
    var remDiv = rem.div(radixToPower);
    var intval = rem.subtract(remDiv.multiply(radixToPower)).toInt() >>> 0;
    var digits = intval.toString(radix);
    rem = remDiv;
    if (rem.isZero()) {
      return digits + result;
    } else {
      while (digits.length < 6) {
        digits = "0" + digits;
      }
      result = "" + digits + result;
    }
  }
};
goog.math.Long.prototype.getHighBits = function() {
  return this.high_;
};
goog.math.Long.prototype.getLowBits = function() {
  return this.low_;
};
goog.math.Long.prototype.getLowBitsUnsigned = function() {
  return this.low_ >= 0 ? this.low_ : goog.math.Long.TWO_PWR_32_DBL_ + this.low_;
};
goog.math.Long.prototype.getNumBitsAbs = function() {
  if (this.isNegative()) {
    if (this.equals(goog.math.Long.getMinValue())) {
      return 64;
    } else {
      return this.negate().getNumBitsAbs();
    }
  } else {
    var val = this.high_ != 0 ? this.high_ : this.low_;
    for (var bit = 31;bit > 0;bit--) {
      if ((val & 1 << bit) != 0) {
        break;
      }
    }
    return this.high_ != 0 ? bit + 33 : bit + 1;
  }
};
goog.math.Long.prototype.isZero = function() {
  return this.high_ == 0 && this.low_ == 0;
};
goog.math.Long.prototype.isNegative = function() {
  return this.high_ < 0;
};
goog.math.Long.prototype.isOdd = function() {
  return (this.low_ & 1) == 1;
};
goog.math.Long.prototype.equals = function(other) {
  return this.high_ == other.high_ && this.low_ == other.low_;
};
goog.math.Long.prototype.notEquals = function(other) {
  return this.high_ != other.high_ || this.low_ != other.low_;
};
goog.math.Long.prototype.lessThan = function(other) {
  return this.compare(other) < 0;
};
goog.math.Long.prototype.lessThanOrEqual = function(other) {
  return this.compare(other) <= 0;
};
goog.math.Long.prototype.greaterThan = function(other) {
  return this.compare(other) > 0;
};
goog.math.Long.prototype.greaterThanOrEqual = function(other) {
  return this.compare(other) >= 0;
};
goog.math.Long.prototype.compare = function(other) {
  if (this.equals(other)) {
    return 0;
  }
  var thisNeg = this.isNegative();
  var otherNeg = other.isNegative();
  if (thisNeg && !otherNeg) {
    return -1;
  }
  if (!thisNeg && otherNeg) {
    return 1;
  }
  if (this.subtract(other).isNegative()) {
    return -1;
  } else {
    return 1;
  }
};
goog.math.Long.prototype.negate = function() {
  if (this.equals(goog.math.Long.getMinValue())) {
    return goog.math.Long.getMinValue();
  } else {
    return this.not().add(goog.math.Long.getOne());
  }
};
goog.math.Long.prototype.add = function(other) {
  var a48 = this.high_ >>> 16;
  var a32 = this.high_ & 65535;
  var a16 = this.low_ >>> 16;
  var a00 = this.low_ & 65535;
  var b48 = other.high_ >>> 16;
  var b32 = other.high_ & 65535;
  var b16 = other.low_ >>> 16;
  var b00 = other.low_ & 65535;
  var c48 = 0, c32 = 0, c16 = 0, c00 = 0;
  c00 += a00 + b00;
  c16 += c00 >>> 16;
  c00 &= 65535;
  c16 += a16 + b16;
  c32 += c16 >>> 16;
  c16 &= 65535;
  c32 += a32 + b32;
  c48 += c32 >>> 16;
  c32 &= 65535;
  c48 += a48 + b48;
  c48 &= 65535;
  return goog.math.Long.fromBits(c16 << 16 | c00, c48 << 16 | c32);
};
goog.math.Long.prototype.subtract = function(other) {
  return this.add(other.negate());
};
goog.math.Long.prototype.multiply = function(other) {
  if (this.isZero()) {
    return goog.math.Long.getZero();
  } else {
    if (other.isZero()) {
      return goog.math.Long.getZero();
    }
  }
  if (this.equals(goog.math.Long.getMinValue())) {
    return other.isOdd() ? goog.math.Long.getMinValue() : goog.math.Long.getZero();
  } else {
    if (other.equals(goog.math.Long.getMinValue())) {
      return this.isOdd() ? goog.math.Long.getMinValue() : goog.math.Long.getZero();
    }
  }
  if (this.isNegative()) {
    if (other.isNegative()) {
      return this.negate().multiply(other.negate());
    } else {
      return this.negate().multiply(other).negate();
    }
  } else {
    if (other.isNegative()) {
      return this.multiply(other.negate()).negate();
    }
  }
  if (this.lessThan(goog.math.Long.getTwoPwr24()) && other.lessThan(goog.math.Long.getTwoPwr24())) {
    return goog.math.Long.fromNumber(this.toNumber() * other.toNumber());
  }
  var a48 = this.high_ >>> 16;
  var a32 = this.high_ & 65535;
  var a16 = this.low_ >>> 16;
  var a00 = this.low_ & 65535;
  var b48 = other.high_ >>> 16;
  var b32 = other.high_ & 65535;
  var b16 = other.low_ >>> 16;
  var b00 = other.low_ & 65535;
  var c48 = 0, c32 = 0, c16 = 0, c00 = 0;
  c00 += a00 * b00;
  c16 += c00 >>> 16;
  c00 &= 65535;
  c16 += a16 * b00;
  c32 += c16 >>> 16;
  c16 &= 65535;
  c16 += a00 * b16;
  c32 += c16 >>> 16;
  c16 &= 65535;
  c32 += a32 * b00;
  c48 += c32 >>> 16;
  c32 &= 65535;
  c32 += a16 * b16;
  c48 += c32 >>> 16;
  c32 &= 65535;
  c32 += a00 * b32;
  c48 += c32 >>> 16;
  c32 &= 65535;
  c48 += a48 * b00 + a32 * b16 + a16 * b32 + a00 * b48;
  c48 &= 65535;
  return goog.math.Long.fromBits(c16 << 16 | c00, c48 << 16 | c32);
};
goog.math.Long.prototype.div = function(other) {
  if (other.isZero()) {
    throw Error("division by zero");
  } else {
    if (this.isZero()) {
      return goog.math.Long.getZero();
    }
  }
  if (this.equals(goog.math.Long.getMinValue())) {
    if (other.equals(goog.math.Long.getOne()) || other.equals(goog.math.Long.getNegOne())) {
      return goog.math.Long.getMinValue();
    } else {
      if (other.equals(goog.math.Long.getMinValue())) {
        return goog.math.Long.getOne();
      } else {
        var halfThis = this.shiftRight(1);
        var approx = halfThis.div(other).shiftLeft(1);
        if (approx.equals(goog.math.Long.getZero())) {
          return other.isNegative() ? goog.math.Long.getOne() : goog.math.Long.getNegOne();
        } else {
          var rem = this.subtract(other.multiply(approx));
          var result = approx.add(rem.div(other));
          return result;
        }
      }
    }
  } else {
    if (other.equals(goog.math.Long.getMinValue())) {
      return goog.math.Long.getZero();
    }
  }
  if (this.isNegative()) {
    if (other.isNegative()) {
      return this.negate().div(other.negate());
    } else {
      return this.negate().div(other).negate();
    }
  } else {
    if (other.isNegative()) {
      return this.div(other.negate()).negate();
    }
  }
  var res = goog.math.Long.getZero();
  var rem = this;
  while (rem.greaterThanOrEqual(other)) {
    var approx = Math.max(1, Math.floor(rem.toNumber() / other.toNumber()));
    var log2 = Math.ceil(Math.log(approx) / Math.LN2);
    var delta = log2 <= 48 ? 1 : Math.pow(2, log2 - 48);
    var approxRes = goog.math.Long.fromNumber(approx);
    var approxRem = approxRes.multiply(other);
    while (approxRem.isNegative() || approxRem.greaterThan(rem)) {
      approx -= delta;
      approxRes = goog.math.Long.fromNumber(approx);
      approxRem = approxRes.multiply(other);
    }
    if (approxRes.isZero()) {
      approxRes = goog.math.Long.getOne();
    }
    res = res.add(approxRes);
    rem = rem.subtract(approxRem);
  }
  return res;
};
goog.math.Long.prototype.modulo = function(other) {
  return this.subtract(this.div(other).multiply(other));
};
goog.math.Long.prototype.not = function() {
  return goog.math.Long.fromBits(~this.low_, ~this.high_);
};
goog.math.Long.prototype.and = function(other) {
  return goog.math.Long.fromBits(this.low_ & other.low_, this.high_ & other.high_);
};
goog.math.Long.prototype.or = function(other) {
  return goog.math.Long.fromBits(this.low_ | other.low_, this.high_ | other.high_);
};
goog.math.Long.prototype.xor = function(other) {
  return goog.math.Long.fromBits(this.low_ ^ other.low_, this.high_ ^ other.high_);
};
goog.math.Long.prototype.shiftLeft = function(numBits) {
  numBits &= 63;
  if (numBits == 0) {
    return this;
  } else {
    var low = this.low_;
    if (numBits < 32) {
      var high = this.high_;
      return goog.math.Long.fromBits(low << numBits, high << numBits | low >>> 32 - numBits);
    } else {
      return goog.math.Long.fromBits(0, low << numBits - 32);
    }
  }
};
goog.math.Long.prototype.shiftRight = function(numBits) {
  numBits &= 63;
  if (numBits == 0) {
    return this;
  } else {
    var high = this.high_;
    if (numBits < 32) {
      var low = this.low_;
      return goog.math.Long.fromBits(low >>> numBits | high << 32 - numBits, high >> numBits);
    } else {
      return goog.math.Long.fromBits(high >> numBits - 32, high >= 0 ? 0 : -1);
    }
  }
};
goog.math.Long.prototype.shiftRightUnsigned = function(numBits) {
  numBits &= 63;
  if (numBits == 0) {
    return this;
  } else {
    var high = this.high_;
    if (numBits < 32) {
      var low = this.low_;
      return goog.math.Long.fromBits(low >>> numBits | high << 32 - numBits, high >>> numBits);
    } else {
      if (numBits == 32) {
        return goog.math.Long.fromBits(high, 0);
      } else {
        return goog.math.Long.fromBits(high >>> numBits - 32, 0);
      }
    }
  }
};
goog.math.Long.ValueCacheId_ = {MAX_VALUE:1, MIN_VALUE:2, ZERO:3, ONE:4, NEG_ONE:5, TWO_PWR_24:6};
goog.provide("cljs.core");
goog.require("goog.math.Long");
goog.require("goog.math.Integer");
goog.require("goog.string");
goog.require("goog.object");
goog.require("goog.array");
goog.require("goog.string.StringBuffer");
cljs.core._STAR_clojurescript_version_STAR_ = "1.9.521";
cljs.core._STAR_unchecked_if_STAR_ = false;
cljs.core._STAR_warn_on_infer_STAR_ = false;
if (typeof cljs.core.PROTOCOL_SENTINEL !== "undefined") {
} else {
  cljs.core.PROTOCOL_SENTINEL = {};
}
goog.define("cljs.core._STAR_target_STAR_", "default");
cljs.core._STAR_ns_STAR_ = null;
cljs.core._STAR_out_STAR_ = null;
cljs.core._STAR_assert_STAR_ = true;
if (typeof cljs.core._STAR_print_fn_STAR_ !== "undefined") {
} else {
  cljs.core._STAR_print_fn_STAR_ = function cljs$core$_STAR_print_fn_STAR_(_) {
    throw new Error("No *print-fn* fn set for evaluation environment");
  };
}
if (typeof cljs.core._STAR_print_err_fn_STAR_ !== "undefined") {
} else {
  cljs.core._STAR_print_err_fn_STAR_ = function cljs$core$_STAR_print_err_fn_STAR_(_) {
    throw new Error("No *print-err-fn* fn set for evaluation environment");
  };
}
cljs.core.set_print_fn_BANG_ = function cljs$core$set_print_fn_BANG_(f) {
  return cljs.core._STAR_print_fn_STAR_ = f;
};
cljs.core.set_print_err_fn_BANG_ = function cljs$core$set_print_err_fn_BANG_(f) {
  return cljs.core._STAR_print_err_fn_STAR_ = f;
};
cljs.core._STAR_flush_on_newline_STAR_ = true;
cljs.core._STAR_print_newline_STAR_ = true;
cljs.core._STAR_print_readably_STAR_ = true;
cljs.core._STAR_print_meta_STAR_ = false;
cljs.core._STAR_print_dup_STAR_ = false;
cljs.core._STAR_print_namespace_maps_STAR_ = false;
cljs.core._STAR_print_length_STAR_ = null;
cljs.core._STAR_print_level_STAR_ = null;
if (typeof cljs.core._STAR_loaded_libs_STAR_ !== "undefined") {
} else {
  cljs.core._STAR_loaded_libs_STAR_ = null;
}
cljs.core.pr_opts = function cljs$core$pr_opts() {
  return new cljs.core.PersistentArrayMap(null, 5, [new cljs.core.Keyword(null, "flush-on-newline", "flush-on-newline", -151457939), cljs.core._STAR_flush_on_newline_STAR_, new cljs.core.Keyword(null, "readably", "readably", 1129599760), cljs.core._STAR_print_readably_STAR_, new cljs.core.Keyword(null, "meta", "meta", 1499536964), cljs.core._STAR_print_meta_STAR_, new cljs.core.Keyword(null, "dup", "dup", 556298533), cljs.core._STAR_print_dup_STAR_, new cljs.core.Keyword(null, "print-length", "print-length", 
  1931866356), cljs.core._STAR_print_length_STAR_], null);
};
cljs.core.enable_console_print_BANG_ = function cljs$core$enable_console_print_BANG_() {
  cljs.core._STAR_print_newline_STAR_ = false;
  cljs.core._STAR_print_fn_STAR_ = function() {
    var G__9571__delegate = function(args) {
      return console.log.apply(console, cljs.core.into_array.cljs$core$IFn$_invoke$arity$1 ? cljs.core.into_array.cljs$core$IFn$_invoke$arity$1(args) : cljs.core.into_array.call(null, args));
    };
    var G__9571 = function(var_args) {
      var args = null;
      if (arguments.length > 0) {
        var G__9572__i = 0, G__9572__a = new Array(arguments.length - 0);
        while (G__9572__i < G__9572__a.length) {
          G__9572__a[G__9572__i] = arguments[G__9572__i + 0];
          ++G__9572__i;
        }
        args = new cljs.core.IndexedSeq(G__9572__a, 0);
      }
      return G__9571__delegate.call(this, args);
    };
    G__9571.cljs$lang$maxFixedArity = 0;
    G__9571.cljs$lang$applyTo = function(arglist__9573) {
      var args = cljs.core.seq(arglist__9573);
      return G__9571__delegate(args);
    };
    G__9571.cljs$core$IFn$_invoke$arity$variadic = G__9571__delegate;
    return G__9571;
  }();
  cljs.core._STAR_print_err_fn_STAR_ = function() {
    var G__9574__delegate = function(args) {
      return console.error.apply(console, cljs.core.into_array.cljs$core$IFn$_invoke$arity$1 ? cljs.core.into_array.cljs$core$IFn$_invoke$arity$1(args) : cljs.core.into_array.call(null, args));
    };
    var G__9574 = function(var_args) {
      var args = null;
      if (arguments.length > 0) {
        var G__9575__i = 0, G__9575__a = new Array(arguments.length - 0);
        while (G__9575__i < G__9575__a.length) {
          G__9575__a[G__9575__i] = arguments[G__9575__i + 0];
          ++G__9575__i;
        }
        args = new cljs.core.IndexedSeq(G__9575__a, 0);
      }
      return G__9574__delegate.call(this, args);
    };
    G__9574.cljs$lang$maxFixedArity = 0;
    G__9574.cljs$lang$applyTo = function(arglist__9576) {
      var args = cljs.core.seq(arglist__9576);
      return G__9574__delegate(args);
    };
    G__9574.cljs$core$IFn$_invoke$arity$variadic = G__9574__delegate;
    return G__9574;
  }();
  return null;
};
cljs.core.truth_ = function cljs$core$truth_(x) {
  return x != null && x !== false;
};
cljs.core.not_native = null;
cljs.core.identical_QMARK_ = function cljs$core$identical_QMARK_(x, y) {
  return x === y;
};
cljs.core.nil_QMARK_ = function cljs$core$nil_QMARK_(x) {
  return x == null;
};
cljs.core.array_QMARK_ = function cljs$core$array_QMARK_(x) {
  if (cljs.core._STAR_target_STAR_ === "nodejs") {
    return Array.isArray(x);
  } else {
    return x instanceof Array;
  }
};
cljs.core.number_QMARK_ = function cljs$core$number_QMARK_(x) {
  return typeof x === "number";
};
cljs.core.not = function cljs$core$not(x) {
  if (x == null) {
    return true;
  } else {
    if (x === false) {
      return true;
    } else {
      return false;
    }
  }
};
cljs.core.some_QMARK_ = function cljs$core$some_QMARK_(x) {
  return !(x == null);
};
cljs.core.object_QMARK_ = function cljs$core$object_QMARK_(x) {
  if (!(x == null)) {
    return x.constructor === Object;
  } else {
    return false;
  }
};
cljs.core.string_QMARK_ = function cljs$core$string_QMARK_(x) {
  return goog.isString(x);
};
cljs.core.char_QMARK_ = function cljs$core$char_QMARK_(x) {
  return typeof x === "string" && 1 === x.length;
};
cljs.core.any_QMARK_ = function cljs$core$any_QMARK_(x) {
  return true;
};
cljs.core.native_satisfies_QMARK_ = function cljs$core$native_satisfies_QMARK_(p, x) {
  var x__$1 = x == null ? null : x;
  if (p[goog.typeOf(x__$1)]) {
    return true;
  } else {
    if (p["_"]) {
      return true;
    } else {
      return false;
    }
  }
};
cljs.core.is_proto_ = function cljs$core$is_proto_(x) {
  return x.constructor.prototype === x;
};
cljs.core._STAR_main_cli_fn_STAR_ = null;
cljs.core.type = function cljs$core$type(x) {
  if (x == null) {
    return null;
  } else {
    return x.constructor;
  }
};
cljs.core.missing_protocol = function cljs$core$missing_protocol(proto, obj) {
  var ty = cljs.core.type(obj);
  var ty__$1 = cljs.core.truth_(function() {
    var and__7248__auto__ = ty;
    if (cljs.core.truth_(and__7248__auto__)) {
      return ty.cljs$lang$type;
    } else {
      return and__7248__auto__;
    }
  }()) ? ty.cljs$lang$ctorStr : goog.typeOf(obj);
  return new Error(["No protocol method ", proto, " defined for type ", ty__$1, ": ", obj].join(""));
};
cljs.core.type__GT_str = function cljs$core$type__GT_str(ty) {
  var temp__4655__auto__ = ty.cljs$lang$ctorStr;
  if (cljs.core.truth_(temp__4655__auto__)) {
    var s = temp__4655__auto__;
    return s;
  } else {
    return [cljs.core.str.cljs$core$IFn$_invoke$arity$1(ty)].join("");
  }
};
cljs.core.load_file = function cljs$core$load_file(file) {
  if (cljs.core.truth_(COMPILED)) {
    return null;
  } else {
    return goog.nodeGlobalRequire(file);
  }
};
if (typeof Symbol !== "undefined" && goog.typeOf(Symbol) === "function") {
  cljs.core.ITER_SYMBOL = Symbol.iterator;
} else {
  cljs.core.ITER_SYMBOL = "@@iterator";
}
cljs.core.CHAR_MAP = {"]":"_RBRACK_", "'":"_SINGLEQUOTE_", "\x3d":"_EQ_", '"':"_DOUBLEQUOTE_", "!":"_BANG_", "*":"_STAR_", "%":"_PERCENT_", "|":"_BAR_", "~":"_TILDE_", "/":"_SLASH_", "\\":"_BSLASH_", "-":"_", "?":"_QMARK_", "\x26":"_AMPERSAND_", ":":"_COLON_", "\x3c":"_LT_", "{":"_LBRACE_", "}":"_RBRACE_", "[":"_LBRACK_", "#":"_SHARP_", "^":"_CARET_", "+":"_PLUS_", "@":"_CIRCA_", "\x3e":"_GT_"};
cljs.core.DEMUNGE_MAP = {"_RBRACE_":"}", "_COLON_":":", "_BANG_":"!", "_QMARK_":"?", "_BSLASH_":"\\\\", "_SLASH_":"/", "_PERCENT_":"%", "_PLUS_":"+", "_SHARP_":"#", "_LBRACE_":"{", "_BAR_":"|", "_LBRACK_":"[", "_EQ_":"\x3d", "_":"-", "_TILDE_":"~", "_RBRACK_":"]", "_GT_":"\x3e", "_SINGLEQUOTE_":"'", "_CIRCA_":"@", "_AMPERSAND_":"\x26", "_DOUBLEQUOTE_":'\\"', "_CARET_":"^", "_LT_":"\x3c", "_STAR_":"*"};
cljs.core.DEMUNGE_PATTERN = null;
cljs.core.system_time = function cljs$core$system_time() {
  if (typeof performance !== "undefined" && !(performance.now == null)) {
    return performance.now();
  } else {
    if (typeof process !== "undefined" && !(process.hrtime == null)) {
      var t = process.hrtime();
      return (t[0] * 1.0E9 + t[1]) / 1000000.0;
    } else {
      return (new Date).getTime();
    }
  }
};
cljs.core.make_array = function cljs$core$make_array(var_args) {
  var args9577 = [];
  var len__8373__auto___9583 = arguments.length;
  var i__8374__auto___9584 = 0;
  while (true) {
    if (i__8374__auto___9584 < len__8373__auto___9583) {
      args9577.push(arguments[i__8374__auto___9584]);
      var G__9585 = i__8374__auto___9584 + 1;
      i__8374__auto___9584 = G__9585;
      continue;
    } else {
    }
    break;
  }
  var G__9582 = args9577.length;
  switch(G__9582) {
    case 1:
      return cljs.core.make_array.cljs$core$IFn$_invoke$arity$1(arguments[0]);
      break;
    case 2:
      return cljs.core.make_array.cljs$core$IFn$_invoke$arity$2(arguments[0], arguments[1]);
      break;
    default:
      var argseq__8392__auto__ = new cljs.core.IndexedSeq(args9577.slice(2), 0, null);
      return cljs.core.make_array.cljs$core$IFn$_invoke$arity$variadic(arguments[0], arguments[1], argseq__8392__auto__);
  }
};
cljs.core.make_array.cljs$core$IFn$_invoke$arity$1 = function(size) {
  return new Array(size);
};
cljs.core.make_array.cljs$core$IFn$_invoke$arity$2 = function(type, size) {
  return new Array(size);
};
cljs.core.make_array.cljs$core$IFn$_invoke$arity$variadic = function(type, size, more_sizes) {
  var dims = more_sizes;
  var dimarray = new Array(size);
  var n__8183__auto___9587 = dimarray.length;
  var i_9588 = 0;
  while (true) {
    if (i_9588 < n__8183__auto___9587) {
      dimarray[i_9588] = cljs.core.apply.cljs$core$IFn$_invoke$arity$3 ? cljs.core.apply.cljs$core$IFn$_invoke$arity$3(cljs.core.make_array, null, dims) : cljs.core.apply.call(null, cljs.core.make_array, null, dims);
      var G__9589 = i_9588 + 1;
      i_9588 = G__9589;
      continue;
    } else {
    }
    break;
  }
  return dimarray;
};
cljs.core.make_array.cljs$lang$applyTo = function(seq9578) {
  var G__9579 = cljs.core.first(seq9578);
  var seq9578__$1 = cljs.core.next(seq9578);
  var G__9580 = cljs.core.first(seq9578__$1);
  var seq9578__$2 = cljs.core.next(seq9578__$1);
  return cljs.core.make_array.cljs$core$IFn$_invoke$arity$variadic(G__9579, G__9580, seq9578__$2);
};
cljs.core.make_array.cljs$lang$maxFixedArity = 2;
cljs.core.aclone = function cljs$core$aclone(arr) {
  var len = arr.length;
  var new_arr = new Array(len);
  var n__8183__auto___9590 = len;
  var i_9591 = 0;
  while (true) {
    if (i_9591 < n__8183__auto___9590) {
      new_arr[i_9591] = arr[i_9591];
      var G__9592 = i_9591 + 1;
      i_9591 = G__9592;
      continue;
    } else {
    }
    break;
  }
  return new_arr;
};
cljs.core.array = function cljs$core$array(var_args) {
  var a = new Array(arguments.length);
  var i = 0;
  while (true) {
    if (i < a.length) {
      a[i] = arguments[i];
      var G__9593 = i + 1;
      i = G__9593;
      continue;
    } else {
      return a;
    }
    break;
  }
};
cljs.core.aget = function cljs$core$aget(var_args) {
  var args9594 = [];
  var len__8373__auto___9603 = arguments.length;
  var i__8374__auto___9604 = 0;
  while (true) {
    if (i__8374__auto___9604 < len__8373__auto___9603) {
      args9594.push(arguments[i__8374__auto___9604]);
      var G__9605 = i__8374__auto___9604 + 1;
      i__8374__auto___9604 = G__9605;
      continue;
    } else {
    }
    break;
  }
  var G__9599 = args9594.length;
  switch(G__9599) {
    case 2:
      return cljs.core.aget.cljs$core$IFn$_invoke$arity$2(arguments[0], arguments[1]);
      break;
    default:
      var argseq__8392__auto__ = new cljs.core.IndexedSeq(args9594.slice(2), 0, null);
      return cljs.core.aget.cljs$core$IFn$_invoke$arity$variadic(arguments[0], arguments[1], argseq__8392__auto__);
  }
};
cljs.core.aget.cljs$core$IFn$_invoke$arity$2 = function(array, i) {
  return array[i];
};
cljs.core.aget.cljs$core$IFn$_invoke$arity$variadic = function(array, i, idxs) {
  var G__9600 = cljs.core.aget;
  var G__9601 = array[i];
  var G__9602 = idxs;
  return cljs.core.apply.cljs$core$IFn$_invoke$arity$3 ? cljs.core.apply.cljs$core$IFn$_invoke$arity$3(G__9600, G__9601, G__9602) : cljs.core.apply.call(null, G__9600, G__9601, G__9602);
};
cljs.core.aget.cljs$lang$applyTo = function(seq9595) {
  var G__9596 = cljs.core.first(seq9595);
  var seq9595__$1 = cljs.core.next(seq9595);
  var G__9597 = cljs.core.first(seq9595__$1);
  var seq9595__$2 = cljs.core.next(seq9595__$1);
  return cljs.core.aget.cljs$core$IFn$_invoke$arity$variadic(G__9596, G__9597, seq9595__$2);
};
cljs.core.aget.cljs$lang$maxFixedArity = 2;
cljs.core.aset = function cljs$core$aset(var_args) {
  var args9607 = [];
  var len__8373__auto___9618 = arguments.length;
  var i__8374__auto___9619 = 0;
  while (true) {
    if (i__8374__auto___9619 < len__8373__auto___9618) {
      args9607.push(arguments[i__8374__auto___9619]);
      var G__9620 = i__8374__auto___9619 + 1;
      i__8374__auto___9619 = G__9620;
      continue;
    } else {
    }
    break;
  }
  var G__9613 = args9607.length;
  switch(G__9613) {
    case 3:
      return cljs.core.aset.cljs$core$IFn$_invoke$arity$3(arguments[0], arguments[1], arguments[2]);
      break;
    default:
      var argseq__8392__auto__ = new cljs.core.IndexedSeq(args9607.slice(3), 0, null);
      return cljs.core.aset.cljs$core$IFn$_invoke$arity$variadic(arguments[0], arguments[1], arguments[2], argseq__8392__auto__);
  }
};
cljs.core.aset.cljs$core$IFn$_invoke$arity$3 = function(array, i, val) {
  return array[i] = val;
};
cljs.core.aset.cljs$core$IFn$_invoke$arity$variadic = function(array, idx, idx2, idxv) {
  var G__9614 = cljs.core.aset;
  var G__9615 = array[idx];
  var G__9616 = idx2;
  var G__9617 = idxv;
  return cljs.core.apply.cljs$core$IFn$_invoke$arity$4 ? cljs.core.apply.cljs$core$IFn$_invoke$arity$4(G__9614, G__9615, G__9616, G__9617) : cljs.core.apply.call(null, G__9614, G__9615, G__9616, G__9617);
};
cljs.core.aset.cljs$lang$applyTo = function(seq9608) {
  var G__9609 = cljs.core.first(seq9608);
  var seq9608__$1 = cljs.core.next(seq9608);
  var G__9610 = cljs.core.first(seq9608__$1);
  var seq9608__$2 = cljs.core.next(seq9608__$1);
  var G__9611 = cljs.core.first(seq9608__$2);
  var seq9608__$3 = cljs.core.next(seq9608__$2);
  return cljs.core.aset.cljs$core$IFn$_invoke$arity$variadic(G__9609, G__9610, G__9611, seq9608__$3);
};
cljs.core.aset.cljs$lang$maxFixedArity = 3;
cljs.core.alength = function cljs$core$alength(array) {
  return array.length;
};
cljs.core.into_array = function cljs$core$into_array(var_args) {
  var args9622 = [];
  var len__8373__auto___9628 = arguments.length;
  var i__8374__auto___9629 = 0;
  while (true) {
    if (i__8374__auto___9629 < len__8373__auto___9628) {
      args9622.push(arguments[i__8374__auto___9629]);
      var G__9630 = i__8374__auto___9629 + 1;
      i__8374__auto___9629 = G__9630;
      continue;
    } else {
    }
    break;
  }
  var G__9624 = args9622.length;
  switch(G__9624) {
    case 1:
      return cljs.core.into_array.cljs$core$IFn$_invoke$arity$1(arguments[0]);
      break;
    case 2:
      return cljs.core.into_array.cljs$core$IFn$_invoke$arity$2(arguments[0], arguments[1]);
      break;
    default:
      throw new Error([cljs.core.str.cljs$core$IFn$_invoke$arity$1("Invalid arity: "), cljs.core.str.cljs$core$IFn$_invoke$arity$1(args9622.length)].join(""));
  }
};
cljs.core.into_array.cljs$core$IFn$_invoke$arity$1 = function(aseq) {
  return cljs.core.into_array.cljs$core$IFn$_invoke$arity$2(null, aseq);
};
cljs.core.into_array.cljs$core$IFn$_invoke$arity$2 = function(type, aseq) {
  var G__9625 = function(a, x) {
    a.push(x);
    return a;
  };
  var G__9626 = [];
  var G__9627 = aseq;
  return cljs.core.reduce.cljs$core$IFn$_invoke$arity$3 ? cljs.core.reduce.cljs$core$IFn$_invoke$arity$3(G__9625, G__9626, G__9627) : cljs.core.reduce.call(null, G__9625, G__9626, G__9627);
};
cljs.core.into_array.cljs$lang$maxFixedArity = 2;
cljs.core.js_invoke = function cljs$core$js_invoke(var_args) {
  var args__8380__auto__ = [];
  var len__8373__auto___9635 = arguments.length;
  var i__8374__auto___9636 = 0;
  while (true) {
    if (i__8374__auto___9636 < len__8373__auto___9635) {
      args__8380__auto__.push(arguments[i__8374__auto___9636]);
      var G__9637 = i__8374__auto___9636 + 1;
      i__8374__auto___9636 = G__9637;
      continue;
    } else {
    }
    break;
  }
  var argseq__8381__auto__ = 2 < args__8380__auto__.length ? new cljs.core.IndexedSeq(args__8380__auto__.slice(2), 0, null) : null;
  return cljs.core.js_invoke.cljs$core$IFn$_invoke$arity$variadic(arguments[0], arguments[1], argseq__8381__auto__);
};
cljs.core.js_invoke.cljs$core$IFn$_invoke$arity$variadic = function(obj, s, args) {
  return obj[s].apply(obj, cljs.core.into_array.cljs$core$IFn$_invoke$arity$1(args));
};
cljs.core.js_invoke.cljs$lang$maxFixedArity = 2;
cljs.core.js_invoke.cljs$lang$applyTo = function(seq9632) {
  var G__9633 = cljs.core.first(seq9632);
  var seq9632__$1 = cljs.core.next(seq9632);
  var G__9634 = cljs.core.first(seq9632__$1);
  var seq9632__$2 = cljs.core.next(seq9632__$1);
  return cljs.core.js_invoke.cljs$core$IFn$_invoke$arity$variadic(G__9633, G__9634, seq9632__$2);
};
cljs.core.Fn = function() {
};
cljs.core.IFn = function() {
};
cljs.core._invoke = function cljs$core$_invoke(var_args) {
  var args9638 = [];
  var len__8373__auto___9641 = arguments.length;
  var i__8374__auto___9642 = 0;
  while (true) {
    if (i__8374__auto___9642 < len__8373__auto___9641) {
      args9638.push(arguments[i__8374__auto___9642]);
      var G__9643 = i__8374__auto___9642 + 1;
      i__8374__auto___9642 = G__9643;
      continue;
    } else {
    }
    break;
  }
  var G__9640 = args9638.length;
  switch(G__9640) {
    case 1:
      return cljs.core._invoke.cljs$core$IFn$_invoke$arity$1(arguments[0]);
      break;
    case 2:
      return cljs.core._invoke.cljs$core$IFn$_invoke$arity$2(arguments[0], arguments[1]);
      break;
    case 3:
      return cljs.core._invoke.cljs$core$IFn$_invoke$arity$3(arguments[0], arguments[1], arguments[2]);
      break;
    case 4:
      return cljs.core._invoke.cljs$core$IFn$_invoke$arity$4(arguments[0], arguments[1], arguments[2], arguments[3]);
      break;
    case 5:
      return cljs.core._invoke.cljs$core$IFn$_invoke$arity$5(arguments[0], arguments[1], arguments[2], arguments[3], arguments[4]);
      break;
    case 6:
      return cljs.core._invoke.cljs$core$IFn$_invoke$arity$6(arguments[0], arguments[1], arguments[2], arguments[3], arguments[4], arguments[5]);
      break;
    case 7:
      return cljs.core._invoke.cljs$core$IFn$_invoke$arity$7(arguments[0], arguments[1], arguments[2], arguments[3], arguments[4], arguments[5], arguments[6]);
      break;
    case 8:
      return cljs.core._invoke.cljs$core$IFn$_invoke$arity$8(arguments[0], arguments[1], arguments[2], arguments[3], arguments[4], arguments[5], arguments[6], arguments[7]);
      break;
    case 9:
      return cljs.core._invoke.cljs$core$IFn$_invoke$arity$9(arguments[0], arguments[1], arguments[2], arguments[3], arguments[4], arguments[5], arguments[6], arguments[7], arguments[8]);
      break;
    case 10:
      return cljs.core._invoke.cljs$core$IFn$_invoke$arity$10(arguments[0], arguments[1], arguments[2], arguments[3], arguments[4], arguments[5], arguments[6], arguments[7], arguments[8], arguments[9]);
      break;
    case 11:
      return cljs.core._invoke.cljs$core$IFn$_invoke$arity$11(arguments[0], arguments[1], arguments[2], arguments[3], arguments[4], arguments[5], arguments[6], arguments[7], arguments[8], arguments[9], arguments[10]);
      break;
    case 12:
      return cljs.core._invoke.cljs$core$IFn$_invoke$arity$12(arguments[0], arguments[1], arguments[2], arguments[3], arguments[4], arguments[5], arguments[6], arguments[7], arguments[8], arguments[9], arguments[10], arguments[11]);
      break;
    case 13:
      return cljs.core._invoke.cljs$core$IFn$_invoke$arity$13(arguments[0], arguments[1], arguments[2], arguments[3], arguments[4], arguments[5], arguments[6], arguments[7], arguments[8], arguments[9], arguments[10], arguments[11], arguments[12]);
      break;
    case 14:
      return cljs.core._invoke.cljs$core$IFn$_invoke$arity$14(arguments[0], arguments[1], arguments[2], arguments[3], arguments[4], arguments[5], arguments[6], arguments[7], arguments[8], arguments[9], arguments[10], arguments[11], arguments[12], arguments[13]);
      break;
    case 15:
      return cljs.core._invoke.cljs$core$IFn$_invoke$arity$15(arguments[0], arguments[1], arguments[2], arguments[3], arguments[4], arguments[5], arguments[6], arguments[7], arguments[8], arguments[9], arguments[10], arguments[11], arguments[12], arguments[13], arguments[14]);
      break;
    case 16:
      return cljs.core._invoke.cljs$core$IFn$_invoke$arity$16(arguments[0], arguments[1], arguments[2], arguments[3], arguments[4], arguments[5], arguments[6], arguments[7], arguments[8], arguments[9], arguments[10], arguments[11], arguments[12], arguments[13], arguments[14], arguments[15]);
      break;
    case 17:
      return cljs.core._invoke.cljs$core$IFn$_invoke$arity$17(arguments[0], arguments[1], arguments[2], arguments[3], arguments[4], arguments[5], arguments[6], arguments[7], arguments[8], arguments[9], arguments[10], arguments[11], arguments[12], arguments[13], arguments[14], arguments[15], arguments[16]);
      break;
    case 18:
      return cljs.core._invoke.cljs$core$IFn$_invoke$arity$18(arguments[0], arguments[1], arguments[2], arguments[3], arguments[4], arguments[5], arguments[6], arguments[7], arguments[8], arguments[9], arguments[10], arguments[11], arguments[12], arguments[13], arguments[14], arguments[15], arguments[16], arguments[17]);
      break;
    case 19:
      return cljs.core._invoke.cljs$core$IFn$_invoke$arity$19(arguments[0], arguments[1], arguments[2], arguments[3], arguments[4], arguments[5], arguments[6], arguments[7], arguments[8], arguments[9], arguments[10], arguments[11], arguments[12], arguments[13], arguments[14], arguments[15], arguments[16], arguments[17], arguments[18]);
      break;
    case 20:
      return cljs.core._invoke.cljs$core$IFn$_invoke$arity$20(arguments[0], arguments[1], arguments[2], arguments[3], arguments[4], arguments[5], arguments[6], arguments[7], arguments[8], arguments[9], arguments[10], arguments[11], arguments[12], arguments[13], arguments[14], arguments[15], arguments[16], arguments[17], arguments[18], arguments[19]);
      break;
    case 21:
      return cljs.core._invoke.cljs$core$IFn$_invoke$arity$21(arguments[0], arguments[1], arguments[2], arguments[3], arguments[4], arguments[5], arguments[6], arguments[7], arguments[8], arguments[9], arguments[10], arguments[11], arguments[12], arguments[13], arguments[14], arguments[15], arguments[16], arguments[17], arguments[18], arguments[19], arguments[20]);
      break;
    case 22:
      return cljs.core._invoke.cljs$core$IFn$_invoke$arity$22(arguments[0], arguments[1], arguments[2], arguments[3], arguments[4], arguments[5], arguments[6], arguments[7], arguments[8], arguments[9], arguments[10], arguments[11], arguments[12], arguments[13], arguments[14], arguments[15], arguments[16], arguments[17], arguments[18], arguments[19], arguments[20], arguments[21]);
      break;
    default:
      throw new Error([cljs.core.str.cljs$core$IFn$_invoke$arity$1("Invalid arity: "), cljs.core.str.cljs$core$IFn$_invoke$arity$1(args9638.length)].join(""));
  }
};
cljs.core._invoke.cljs$core$IFn$_invoke$arity$1 = function(this$) {
  if (!(this$ == null) && !(this$.cljs$core$IFn$_invoke$arity$1 == null)) {
    return this$.cljs$core$IFn$_invoke$arity$1(this$);
  } else {
    var x__7928__auto__ = this$ == null ? null : this$;
    var m__7929__auto__ = cljs.core._invoke[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$1 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$1(this$) : m__7929__auto__.call(null, this$);
    } else {
      var m__7929__auto____$1 = cljs.core._invoke["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$1 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$1(this$) : m__7929__auto____$1.call(null, this$);
      } else {
        throw cljs.core.missing_protocol("IFn.-invoke", this$);
      }
    }
  }
};
cljs.core._invoke.cljs$core$IFn$_invoke$arity$2 = function(this$, a) {
  if (!(this$ == null) && !(this$.cljs$core$IFn$_invoke$arity$2 == null)) {
    return this$.cljs$core$IFn$_invoke$arity$2(this$, a);
  } else {
    var x__7928__auto__ = this$ == null ? null : this$;
    var m__7929__auto__ = cljs.core._invoke[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$2 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$2(this$, a) : m__7929__auto__.call(null, this$, a);
    } else {
      var m__7929__auto____$1 = cljs.core._invoke["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$2 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$2(this$, a) : m__7929__auto____$1.call(null, this$, a);
      } else {
        throw cljs.core.missing_protocol("IFn.-invoke", this$);
      }
    }
  }
};
cljs.core._invoke.cljs$core$IFn$_invoke$arity$3 = function(this$, a, b) {
  if (!(this$ == null) && !(this$.cljs$core$IFn$_invoke$arity$3 == null)) {
    return this$.cljs$core$IFn$_invoke$arity$3(this$, a, b);
  } else {
    var x__7928__auto__ = this$ == null ? null : this$;
    var m__7929__auto__ = cljs.core._invoke[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$3 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$3(this$, a, b) : m__7929__auto__.call(null, this$, a, b);
    } else {
      var m__7929__auto____$1 = cljs.core._invoke["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$3 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$3(this$, a, b) : m__7929__auto____$1.call(null, this$, a, b);
      } else {
        throw cljs.core.missing_protocol("IFn.-invoke", this$);
      }
    }
  }
};
cljs.core._invoke.cljs$core$IFn$_invoke$arity$4 = function(this$, a, b, c) {
  if (!(this$ == null) && !(this$.cljs$core$IFn$_invoke$arity$4 == null)) {
    return this$.cljs$core$IFn$_invoke$arity$4(this$, a, b, c);
  } else {
    var x__7928__auto__ = this$ == null ? null : this$;
    var m__7929__auto__ = cljs.core._invoke[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$4 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$4(this$, a, b, c) : m__7929__auto__.call(null, this$, a, b, c);
    } else {
      var m__7929__auto____$1 = cljs.core._invoke["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$4 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$4(this$, a, b, c) : m__7929__auto____$1.call(null, this$, a, b, c);
      } else {
        throw cljs.core.missing_protocol("IFn.-invoke", this$);
      }
    }
  }
};
cljs.core._invoke.cljs$core$IFn$_invoke$arity$5 = function(this$, a, b, c, d) {
  if (!(this$ == null) && !(this$.cljs$core$IFn$_invoke$arity$5 == null)) {
    return this$.cljs$core$IFn$_invoke$arity$5(this$, a, b, c, d);
  } else {
    var x__7928__auto__ = this$ == null ? null : this$;
    var m__7929__auto__ = cljs.core._invoke[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$5 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$5(this$, a, b, c, d) : m__7929__auto__.call(null, this$, a, b, c, d);
    } else {
      var m__7929__auto____$1 = cljs.core._invoke["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$5 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$5(this$, a, b, c, d) : m__7929__auto____$1.call(null, this$, a, b, c, d);
      } else {
        throw cljs.core.missing_protocol("IFn.-invoke", this$);
      }
    }
  }
};
cljs.core._invoke.cljs$core$IFn$_invoke$arity$6 = function(this$, a, b, c, d, e) {
  if (!(this$ == null) && !(this$.cljs$core$IFn$_invoke$arity$6 == null)) {
    return this$.cljs$core$IFn$_invoke$arity$6(this$, a, b, c, d, e);
  } else {
    var x__7928__auto__ = this$ == null ? null : this$;
    var m__7929__auto__ = cljs.core._invoke[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$6 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$6(this$, a, b, c, d, e) : m__7929__auto__.call(null, this$, a, b, c, d, e);
    } else {
      var m__7929__auto____$1 = cljs.core._invoke["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$6 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$6(this$, a, b, c, d, e) : m__7929__auto____$1.call(null, this$, a, b, c, d, e);
      } else {
        throw cljs.core.missing_protocol("IFn.-invoke", this$);
      }
    }
  }
};
cljs.core._invoke.cljs$core$IFn$_invoke$arity$7 = function(this$, a, b, c, d, e, f) {
  if (!(this$ == null) && !(this$.cljs$core$IFn$_invoke$arity$7 == null)) {
    return this$.cljs$core$IFn$_invoke$arity$7(this$, a, b, c, d, e, f);
  } else {
    var x__7928__auto__ = this$ == null ? null : this$;
    var m__7929__auto__ = cljs.core._invoke[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$7 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$7(this$, a, b, c, d, e, f) : m__7929__auto__.call(null, this$, a, b, c, d, e, f);
    } else {
      var m__7929__auto____$1 = cljs.core._invoke["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$7 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$7(this$, a, b, c, d, e, f) : m__7929__auto____$1.call(null, this$, a, b, c, d, e, f);
      } else {
        throw cljs.core.missing_protocol("IFn.-invoke", this$);
      }
    }
  }
};
cljs.core._invoke.cljs$core$IFn$_invoke$arity$8 = function(this$, a, b, c, d, e, f, g) {
  if (!(this$ == null) && !(this$.cljs$core$IFn$_invoke$arity$8 == null)) {
    return this$.cljs$core$IFn$_invoke$arity$8(this$, a, b, c, d, e, f, g);
  } else {
    var x__7928__auto__ = this$ == null ? null : this$;
    var m__7929__auto__ = cljs.core._invoke[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$8 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$8(this$, a, b, c, d, e, f, g) : m__7929__auto__.call(null, this$, a, b, c, d, e, f, g);
    } else {
      var m__7929__auto____$1 = cljs.core._invoke["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$8 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$8(this$, a, b, c, d, e, f, g) : m__7929__auto____$1.call(null, this$, a, b, c, d, e, f, g);
      } else {
        throw cljs.core.missing_protocol("IFn.-invoke", this$);
      }
    }
  }
};
cljs.core._invoke.cljs$core$IFn$_invoke$arity$9 = function(this$, a, b, c, d, e, f, g, h) {
  if (!(this$ == null) && !(this$.cljs$core$IFn$_invoke$arity$9 == null)) {
    return this$.cljs$core$IFn$_invoke$arity$9(this$, a, b, c, d, e, f, g, h);
  } else {
    var x__7928__auto__ = this$ == null ? null : this$;
    var m__7929__auto__ = cljs.core._invoke[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$9 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$9(this$, a, b, c, d, e, f, g, h) : m__7929__auto__.call(null, this$, a, b, c, d, e, f, g, h);
    } else {
      var m__7929__auto____$1 = cljs.core._invoke["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$9 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$9(this$, a, b, c, d, e, f, g, h) : m__7929__auto____$1.call(null, this$, a, b, c, d, e, f, g, h);
      } else {
        throw cljs.core.missing_protocol("IFn.-invoke", this$);
      }
    }
  }
};
cljs.core._invoke.cljs$core$IFn$_invoke$arity$10 = function(this$, a, b, c, d, e, f, g, h, i) {
  if (!(this$ == null) && !(this$.cljs$core$IFn$_invoke$arity$10 == null)) {
    return this$.cljs$core$IFn$_invoke$arity$10(this$, a, b, c, d, e, f, g, h, i);
  } else {
    var x__7928__auto__ = this$ == null ? null : this$;
    var m__7929__auto__ = cljs.core._invoke[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$10 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$10(this$, a, b, c, d, e, f, g, h, i) : m__7929__auto__.call(null, this$, a, b, c, d, e, f, g, h, i);
    } else {
      var m__7929__auto____$1 = cljs.core._invoke["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$10 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$10(this$, a, b, c, d, e, f, g, h, i) : m__7929__auto____$1.call(null, this$, a, b, c, d, e, f, g, h, i);
      } else {
        throw cljs.core.missing_protocol("IFn.-invoke", this$);
      }
    }
  }
};
cljs.core._invoke.cljs$core$IFn$_invoke$arity$11 = function(this$, a, b, c, d, e, f, g, h, i, j) {
  if (!(this$ == null) && !(this$.cljs$core$IFn$_invoke$arity$11 == null)) {
    return this$.cljs$core$IFn$_invoke$arity$11(this$, a, b, c, d, e, f, g, h, i, j);
  } else {
    var x__7928__auto__ = this$ == null ? null : this$;
    var m__7929__auto__ = cljs.core._invoke[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$11 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$11(this$, a, b, c, d, e, f, g, h, i, j) : m__7929__auto__.call(null, this$, a, b, c, d, e, f, g, h, i, j);
    } else {
      var m__7929__auto____$1 = cljs.core._invoke["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$11 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$11(this$, a, b, c, d, e, f, g, h, i, j) : m__7929__auto____$1.call(null, this$, a, b, c, d, e, f, g, h, i, j);
      } else {
        throw cljs.core.missing_protocol("IFn.-invoke", this$);
      }
    }
  }
};
cljs.core._invoke.cljs$core$IFn$_invoke$arity$12 = function(this$, a, b, c, d, e, f, g, h, i, j, k) {
  if (!(this$ == null) && !(this$.cljs$core$IFn$_invoke$arity$12 == null)) {
    return this$.cljs$core$IFn$_invoke$arity$12(this$, a, b, c, d, e, f, g, h, i, j, k);
  } else {
    var x__7928__auto__ = this$ == null ? null : this$;
    var m__7929__auto__ = cljs.core._invoke[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$12 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$12(this$, a, b, c, d, e, f, g, h, i, j, k) : m__7929__auto__.call(null, this$, a, b, c, d, e, f, g, h, i, j, k);
    } else {
      var m__7929__auto____$1 = cljs.core._invoke["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$12 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$12(this$, a, b, c, d, e, f, g, h, i, j, k) : m__7929__auto____$1.call(null, this$, a, b, c, d, e, f, g, h, i, j, k);
      } else {
        throw cljs.core.missing_protocol("IFn.-invoke", this$);
      }
    }
  }
};
cljs.core._invoke.cljs$core$IFn$_invoke$arity$13 = function(this$, a, b, c, d, e, f, g, h, i, j, k, l) {
  if (!(this$ == null) && !(this$.cljs$core$IFn$_invoke$arity$13 == null)) {
    return this$.cljs$core$IFn$_invoke$arity$13(this$, a, b, c, d, e, f, g, h, i, j, k, l);
  } else {
    var x__7928__auto__ = this$ == null ? null : this$;
    var m__7929__auto__ = cljs.core._invoke[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$13 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$13(this$, a, b, c, d, e, f, g, h, i, j, k, l) : m__7929__auto__.call(null, this$, a, b, c, d, e, f, g, h, i, j, k, l);
    } else {
      var m__7929__auto____$1 = cljs.core._invoke["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$13 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$13(this$, a, b, c, d, e, f, g, h, i, j, k, l) : m__7929__auto____$1.call(null, this$, a, b, c, d, e, f, g, h, i, j, k, l);
      } else {
        throw cljs.core.missing_protocol("IFn.-invoke", this$);
      }
    }
  }
};
cljs.core._invoke.cljs$core$IFn$_invoke$arity$14 = function(this$, a, b, c, d, e, f, g, h, i, j, k, l, m) {
  if (!(this$ == null) && !(this$.cljs$core$IFn$_invoke$arity$14 == null)) {
    return this$.cljs$core$IFn$_invoke$arity$14(this$, a, b, c, d, e, f, g, h, i, j, k, l, m);
  } else {
    var x__7928__auto__ = this$ == null ? null : this$;
    var m__7929__auto__ = cljs.core._invoke[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$14 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$14(this$, a, b, c, d, e, f, g, h, i, j, k, l, m) : m__7929__auto__.call(null, this$, a, b, c, d, e, f, g, h, i, j, k, l, m);
    } else {
      var m__7929__auto____$1 = cljs.core._invoke["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$14 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$14(this$, a, b, c, d, e, f, g, h, i, j, k, l, m) : m__7929__auto____$1.call(null, this$, a, b, c, d, e, f, g, h, i, j, k, l, m);
      } else {
        throw cljs.core.missing_protocol("IFn.-invoke", this$);
      }
    }
  }
};
cljs.core._invoke.cljs$core$IFn$_invoke$arity$15 = function(this$, a, b, c, d, e, f, g, h, i, j, k, l, m, n) {
  if (!(this$ == null) && !(this$.cljs$core$IFn$_invoke$arity$15 == null)) {
    return this$.cljs$core$IFn$_invoke$arity$15(this$, a, b, c, d, e, f, g, h, i, j, k, l, m, n);
  } else {
    var x__7928__auto__ = this$ == null ? null : this$;
    var m__7929__auto__ = cljs.core._invoke[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$15 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$15(this$, a, b, c, d, e, f, g, h, i, j, k, l, m, n) : m__7929__auto__.call(null, this$, a, b, c, d, e, f, g, h, i, j, k, l, m, n);
    } else {
      var m__7929__auto____$1 = cljs.core._invoke["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$15 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$15(this$, a, b, c, d, e, f, g, h, i, j, k, l, m, n) : m__7929__auto____$1.call(null, this$, a, b, c, d, e, f, g, h, i, j, k, l, m, n);
      } else {
        throw cljs.core.missing_protocol("IFn.-invoke", this$);
      }
    }
  }
};
cljs.core._invoke.cljs$core$IFn$_invoke$arity$16 = function(this$, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o) {
  if (!(this$ == null) && !(this$.cljs$core$IFn$_invoke$arity$16 == null)) {
    return this$.cljs$core$IFn$_invoke$arity$16(this$, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o);
  } else {
    var x__7928__auto__ = this$ == null ? null : this$;
    var m__7929__auto__ = cljs.core._invoke[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$16 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$16(this$, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o) : m__7929__auto__.call(null, this$, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o);
    } else {
      var m__7929__auto____$1 = cljs.core._invoke["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$16 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$16(this$, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o) : m__7929__auto____$1.call(null, this$, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o);
      } else {
        throw cljs.core.missing_protocol("IFn.-invoke", this$);
      }
    }
  }
};
cljs.core._invoke.cljs$core$IFn$_invoke$arity$17 = function(this$, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p) {
  if (!(this$ == null) && !(this$.cljs$core$IFn$_invoke$arity$17 == null)) {
    return this$.cljs$core$IFn$_invoke$arity$17(this$, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p);
  } else {
    var x__7928__auto__ = this$ == null ? null : this$;
    var m__7929__auto__ = cljs.core._invoke[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$17 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$17(this$, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p) : m__7929__auto__.call(null, this$, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p);
    } else {
      var m__7929__auto____$1 = cljs.core._invoke["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$17 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$17(this$, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p) : m__7929__auto____$1.call(null, this$, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p);
      } else {
        throw cljs.core.missing_protocol("IFn.-invoke", this$);
      }
    }
  }
};
cljs.core._invoke.cljs$core$IFn$_invoke$arity$18 = function(this$, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q) {
  if (!(this$ == null) && !(this$.cljs$core$IFn$_invoke$arity$18 == null)) {
    return this$.cljs$core$IFn$_invoke$arity$18(this$, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q);
  } else {
    var x__7928__auto__ = this$ == null ? null : this$;
    var m__7929__auto__ = cljs.core._invoke[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$18 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$18(this$, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q) : m__7929__auto__.call(null, this$, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q);
    } else {
      var m__7929__auto____$1 = cljs.core._invoke["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$18 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$18(this$, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q) : m__7929__auto____$1.call(null, this$, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q);
      } else {
        throw cljs.core.missing_protocol("IFn.-invoke", this$);
      }
    }
  }
};
cljs.core._invoke.cljs$core$IFn$_invoke$arity$19 = function(this$, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r) {
  if (!(this$ == null) && !(this$.cljs$core$IFn$_invoke$arity$19 == null)) {
    return this$.cljs$core$IFn$_invoke$arity$19(this$, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r);
  } else {
    var x__7928__auto__ = this$ == null ? null : this$;
    var m__7929__auto__ = cljs.core._invoke[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$19 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$19(this$, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r) : m__7929__auto__.call(null, this$, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r);
    } else {
      var m__7929__auto____$1 = cljs.core._invoke["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$19 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$19(this$, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r) : m__7929__auto____$1.call(null, this$, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r);
      } else {
        throw cljs.core.missing_protocol("IFn.-invoke", this$);
      }
    }
  }
};
cljs.core._invoke.cljs$core$IFn$_invoke$arity$20 = function(this$, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s) {
  if (!(this$ == null) && !(this$.cljs$core$IFn$_invoke$arity$20 == null)) {
    return this$.cljs$core$IFn$_invoke$arity$20(this$, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s);
  } else {
    var x__7928__auto__ = this$ == null ? null : this$;
    var m__7929__auto__ = cljs.core._invoke[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$20 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$20(this$, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s) : m__7929__auto__.call(null, this$, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s);
    } else {
      var m__7929__auto____$1 = cljs.core._invoke["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$20 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$20(this$, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s) : m__7929__auto____$1.call(null, this$, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s);
      } else {
        throw cljs.core.missing_protocol("IFn.-invoke", this$);
      }
    }
  }
};
cljs.core._invoke.cljs$core$IFn$_invoke$arity$21 = function(this$, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s, t) {
  if (!(this$ == null) && !(this$.cljs$core$IFn$_invoke$arity$21 == null)) {
    return this$.cljs$core$IFn$_invoke$arity$21(this$, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s, t);
  } else {
    var x__7928__auto__ = this$ == null ? null : this$;
    var m__7929__auto__ = cljs.core._invoke[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$21 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$21(this$, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s, t) : m__7929__auto__.call(null, this$, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s, t);
    } else {
      var m__7929__auto____$1 = cljs.core._invoke["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$21 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$21(this$, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s, t) : m__7929__auto____$1.call(null, this$, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s, t);
      } else {
        throw cljs.core.missing_protocol("IFn.-invoke", this$);
      }
    }
  }
};
cljs.core._invoke.cljs$core$IFn$_invoke$arity$22 = function(this$, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s, t, rest) {
  if (!(this$ == null) && !(this$.cljs$core$IFn$_invoke$arity$22 == null)) {
    return this$.cljs$core$IFn$_invoke$arity$22(this$, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s, t, rest);
  } else {
    var x__7928__auto__ = this$ == null ? null : this$;
    var m__7929__auto__ = cljs.core._invoke[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$22 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$22(this$, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s, t, rest) : m__7929__auto__.call(null, this$, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s, t, rest);
    } else {
      var m__7929__auto____$1 = cljs.core._invoke["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$22 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$22(this$, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s, t, rest) : m__7929__auto____$1.call(null, this$, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s, t, rest);
      } else {
        throw cljs.core.missing_protocol("IFn.-invoke", this$);
      }
    }
  }
};
cljs.core._invoke.cljs$lang$maxFixedArity = 22;
cljs.core.ICloneable = function() {
};
cljs.core._clone = function cljs$core$_clone(value) {
  if (!(value == null) && !(value.cljs$core$ICloneable$_clone$arity$1 == null)) {
    return value.cljs$core$ICloneable$_clone$arity$1(value);
  } else {
    var x__7928__auto__ = value == null ? null : value;
    var m__7929__auto__ = cljs.core._clone[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$1 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$1(value) : m__7929__auto__.call(null, value);
    } else {
      var m__7929__auto____$1 = cljs.core._clone["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$1 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$1(value) : m__7929__auto____$1.call(null, value);
      } else {
        throw cljs.core.missing_protocol("ICloneable.-clone", value);
      }
    }
  }
};
cljs.core.ICounted = function() {
};
cljs.core._count = function cljs$core$_count(coll) {
  if (!(coll == null) && !(coll.cljs$core$ICounted$_count$arity$1 == null)) {
    return coll.cljs$core$ICounted$_count$arity$1(coll);
  } else {
    var x__7928__auto__ = coll == null ? null : coll;
    var m__7929__auto__ = cljs.core._count[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$1 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$1(coll) : m__7929__auto__.call(null, coll);
    } else {
      var m__7929__auto____$1 = cljs.core._count["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$1 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$1(coll) : m__7929__auto____$1.call(null, coll);
      } else {
        throw cljs.core.missing_protocol("ICounted.-count", coll);
      }
    }
  }
};
cljs.core.IEmptyableCollection = function() {
};
cljs.core._empty = function cljs$core$_empty(coll) {
  if (!(coll == null) && !(coll.cljs$core$IEmptyableCollection$_empty$arity$1 == null)) {
    return coll.cljs$core$IEmptyableCollection$_empty$arity$1(coll);
  } else {
    var x__7928__auto__ = coll == null ? null : coll;
    var m__7929__auto__ = cljs.core._empty[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$1 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$1(coll) : m__7929__auto__.call(null, coll);
    } else {
      var m__7929__auto____$1 = cljs.core._empty["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$1 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$1(coll) : m__7929__auto____$1.call(null, coll);
      } else {
        throw cljs.core.missing_protocol("IEmptyableCollection.-empty", coll);
      }
    }
  }
};
cljs.core.ICollection = function() {
};
cljs.core._conj = function cljs$core$_conj(coll, o) {
  if (!(coll == null) && !(coll.cljs$core$ICollection$_conj$arity$2 == null)) {
    return coll.cljs$core$ICollection$_conj$arity$2(coll, o);
  } else {
    var x__7928__auto__ = coll == null ? null : coll;
    var m__7929__auto__ = cljs.core._conj[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$2 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$2(coll, o) : m__7929__auto__.call(null, coll, o);
    } else {
      var m__7929__auto____$1 = cljs.core._conj["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$2 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$2(coll, o) : m__7929__auto____$1.call(null, coll, o);
      } else {
        throw cljs.core.missing_protocol("ICollection.-conj", coll);
      }
    }
  }
};
cljs.core.IIndexed = function() {
};
cljs.core._nth = function cljs$core$_nth(var_args) {
  var args9645 = [];
  var len__8373__auto___9648 = arguments.length;
  var i__8374__auto___9649 = 0;
  while (true) {
    if (i__8374__auto___9649 < len__8373__auto___9648) {
      args9645.push(arguments[i__8374__auto___9649]);
      var G__9650 = i__8374__auto___9649 + 1;
      i__8374__auto___9649 = G__9650;
      continue;
    } else {
    }
    break;
  }
  var G__9647 = args9645.length;
  switch(G__9647) {
    case 2:
      return cljs.core._nth.cljs$core$IFn$_invoke$arity$2(arguments[0], arguments[1]);
      break;
    case 3:
      return cljs.core._nth.cljs$core$IFn$_invoke$arity$3(arguments[0], arguments[1], arguments[2]);
      break;
    default:
      throw new Error([cljs.core.str.cljs$core$IFn$_invoke$arity$1("Invalid arity: "), cljs.core.str.cljs$core$IFn$_invoke$arity$1(args9645.length)].join(""));
  }
};
cljs.core._nth.cljs$core$IFn$_invoke$arity$2 = function(coll, n) {
  if (!(coll == null) && !(coll.cljs$core$IIndexed$_nth$arity$2 == null)) {
    return coll.cljs$core$IIndexed$_nth$arity$2(coll, n);
  } else {
    var x__7928__auto__ = coll == null ? null : coll;
    var m__7929__auto__ = cljs.core._nth[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$2 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$2(coll, n) : m__7929__auto__.call(null, coll, n);
    } else {
      var m__7929__auto____$1 = cljs.core._nth["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$2 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$2(coll, n) : m__7929__auto____$1.call(null, coll, n);
      } else {
        throw cljs.core.missing_protocol("IIndexed.-nth", coll);
      }
    }
  }
};
cljs.core._nth.cljs$core$IFn$_invoke$arity$3 = function(coll, n, not_found) {
  if (!(coll == null) && !(coll.cljs$core$IIndexed$_nth$arity$3 == null)) {
    return coll.cljs$core$IIndexed$_nth$arity$3(coll, n, not_found);
  } else {
    var x__7928__auto__ = coll == null ? null : coll;
    var m__7929__auto__ = cljs.core._nth[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$3 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$3(coll, n, not_found) : m__7929__auto__.call(null, coll, n, not_found);
    } else {
      var m__7929__auto____$1 = cljs.core._nth["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$3 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$3(coll, n, not_found) : m__7929__auto____$1.call(null, coll, n, not_found);
      } else {
        throw cljs.core.missing_protocol("IIndexed.-nth", coll);
      }
    }
  }
};
cljs.core._nth.cljs$lang$maxFixedArity = 3;
cljs.core.ASeq = function() {
};
cljs.core.ISeq = function() {
};
cljs.core._first = function cljs$core$_first(coll) {
  if (!(coll == null) && !(coll.cljs$core$ISeq$_first$arity$1 == null)) {
    return coll.cljs$core$ISeq$_first$arity$1(coll);
  } else {
    var x__7928__auto__ = coll == null ? null : coll;
    var m__7929__auto__ = cljs.core._first[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$1 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$1(coll) : m__7929__auto__.call(null, coll);
    } else {
      var m__7929__auto____$1 = cljs.core._first["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$1 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$1(coll) : m__7929__auto____$1.call(null, coll);
      } else {
        throw cljs.core.missing_protocol("ISeq.-first", coll);
      }
    }
  }
};
cljs.core._rest = function cljs$core$_rest(coll) {
  if (!(coll == null) && !(coll.cljs$core$ISeq$_rest$arity$1 == null)) {
    return coll.cljs$core$ISeq$_rest$arity$1(coll);
  } else {
    var x__7928__auto__ = coll == null ? null : coll;
    var m__7929__auto__ = cljs.core._rest[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$1 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$1(coll) : m__7929__auto__.call(null, coll);
    } else {
      var m__7929__auto____$1 = cljs.core._rest["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$1 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$1(coll) : m__7929__auto____$1.call(null, coll);
      } else {
        throw cljs.core.missing_protocol("ISeq.-rest", coll);
      }
    }
  }
};
cljs.core.INext = function() {
};
cljs.core._next = function cljs$core$_next(coll) {
  if (!(coll == null) && !(coll.cljs$core$INext$_next$arity$1 == null)) {
    return coll.cljs$core$INext$_next$arity$1(coll);
  } else {
    var x__7928__auto__ = coll == null ? null : coll;
    var m__7929__auto__ = cljs.core._next[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$1 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$1(coll) : m__7929__auto__.call(null, coll);
    } else {
      var m__7929__auto____$1 = cljs.core._next["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$1 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$1(coll) : m__7929__auto____$1.call(null, coll);
      } else {
        throw cljs.core.missing_protocol("INext.-next", coll);
      }
    }
  }
};
cljs.core.ILookup = function() {
};
cljs.core._lookup = function cljs$core$_lookup(var_args) {
  var args9652 = [];
  var len__8373__auto___9655 = arguments.length;
  var i__8374__auto___9656 = 0;
  while (true) {
    if (i__8374__auto___9656 < len__8373__auto___9655) {
      args9652.push(arguments[i__8374__auto___9656]);
      var G__9657 = i__8374__auto___9656 + 1;
      i__8374__auto___9656 = G__9657;
      continue;
    } else {
    }
    break;
  }
  var G__9654 = args9652.length;
  switch(G__9654) {
    case 2:
      return cljs.core._lookup.cljs$core$IFn$_invoke$arity$2(arguments[0], arguments[1]);
      break;
    case 3:
      return cljs.core._lookup.cljs$core$IFn$_invoke$arity$3(arguments[0], arguments[1], arguments[2]);
      break;
    default:
      throw new Error([cljs.core.str.cljs$core$IFn$_invoke$arity$1("Invalid arity: "), cljs.core.str.cljs$core$IFn$_invoke$arity$1(args9652.length)].join(""));
  }
};
cljs.core._lookup.cljs$core$IFn$_invoke$arity$2 = function(o, k) {
  if (!(o == null) && !(o.cljs$core$ILookup$_lookup$arity$2 == null)) {
    return o.cljs$core$ILookup$_lookup$arity$2(o, k);
  } else {
    var x__7928__auto__ = o == null ? null : o;
    var m__7929__auto__ = cljs.core._lookup[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$2 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$2(o, k) : m__7929__auto__.call(null, o, k);
    } else {
      var m__7929__auto____$1 = cljs.core._lookup["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$2 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$2(o, k) : m__7929__auto____$1.call(null, o, k);
      } else {
        throw cljs.core.missing_protocol("ILookup.-lookup", o);
      }
    }
  }
};
cljs.core._lookup.cljs$core$IFn$_invoke$arity$3 = function(o, k, not_found) {
  if (!(o == null) && !(o.cljs$core$ILookup$_lookup$arity$3 == null)) {
    return o.cljs$core$ILookup$_lookup$arity$3(o, k, not_found);
  } else {
    var x__7928__auto__ = o == null ? null : o;
    var m__7929__auto__ = cljs.core._lookup[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$3 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$3(o, k, not_found) : m__7929__auto__.call(null, o, k, not_found);
    } else {
      var m__7929__auto____$1 = cljs.core._lookup["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$3 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$3(o, k, not_found) : m__7929__auto____$1.call(null, o, k, not_found);
      } else {
        throw cljs.core.missing_protocol("ILookup.-lookup", o);
      }
    }
  }
};
cljs.core._lookup.cljs$lang$maxFixedArity = 3;
cljs.core.IAssociative = function() {
};
cljs.core._contains_key_QMARK_ = function cljs$core$_contains_key_QMARK_(coll, k) {
  if (!(coll == null) && !(coll.cljs$core$IAssociative$_contains_key_QMARK_$arity$2 == null)) {
    return coll.cljs$core$IAssociative$_contains_key_QMARK_$arity$2(coll, k);
  } else {
    var x__7928__auto__ = coll == null ? null : coll;
    var m__7929__auto__ = cljs.core._contains_key_QMARK_[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$2 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$2(coll, k) : m__7929__auto__.call(null, coll, k);
    } else {
      var m__7929__auto____$1 = cljs.core._contains_key_QMARK_["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$2 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$2(coll, k) : m__7929__auto____$1.call(null, coll, k);
      } else {
        throw cljs.core.missing_protocol("IAssociative.-contains-key?", coll);
      }
    }
  }
};
cljs.core._assoc = function cljs$core$_assoc(coll, k, v) {
  if (!(coll == null) && !(coll.cljs$core$IAssociative$_assoc$arity$3 == null)) {
    return coll.cljs$core$IAssociative$_assoc$arity$3(coll, k, v);
  } else {
    var x__7928__auto__ = coll == null ? null : coll;
    var m__7929__auto__ = cljs.core._assoc[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$3 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$3(coll, k, v) : m__7929__auto__.call(null, coll, k, v);
    } else {
      var m__7929__auto____$1 = cljs.core._assoc["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$3 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$3(coll, k, v) : m__7929__auto____$1.call(null, coll, k, v);
      } else {
        throw cljs.core.missing_protocol("IAssociative.-assoc", coll);
      }
    }
  }
};
cljs.core.IFind = function() {
};
cljs.core._find = function cljs$core$_find(coll, k) {
  if (!(coll == null) && !(coll.cljs$core$IFind$_find$arity$2 == null)) {
    return coll.cljs$core$IFind$_find$arity$2(coll, k);
  } else {
    var x__7928__auto__ = coll == null ? null : coll;
    var m__7929__auto__ = cljs.core._find[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$2 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$2(coll, k) : m__7929__auto__.call(null, coll, k);
    } else {
      var m__7929__auto____$1 = cljs.core._find["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$2 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$2(coll, k) : m__7929__auto____$1.call(null, coll, k);
      } else {
        throw cljs.core.missing_protocol("IFind.-find", coll);
      }
    }
  }
};
cljs.core.IMap = function() {
};
cljs.core._dissoc = function cljs$core$_dissoc(coll, k) {
  if (!(coll == null) && !(coll.cljs$core$IMap$_dissoc$arity$2 == null)) {
    return coll.cljs$core$IMap$_dissoc$arity$2(coll, k);
  } else {
    var x__7928__auto__ = coll == null ? null : coll;
    var m__7929__auto__ = cljs.core._dissoc[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$2 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$2(coll, k) : m__7929__auto__.call(null, coll, k);
    } else {
      var m__7929__auto____$1 = cljs.core._dissoc["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$2 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$2(coll, k) : m__7929__auto____$1.call(null, coll, k);
      } else {
        throw cljs.core.missing_protocol("IMap.-dissoc", coll);
      }
    }
  }
};
cljs.core.IMapEntry = function() {
};
cljs.core._key = function cljs$core$_key(coll) {
  if (!(coll == null) && !(coll.cljs$core$IMapEntry$_key$arity$1 == null)) {
    return coll.cljs$core$IMapEntry$_key$arity$1(coll);
  } else {
    var x__7928__auto__ = coll == null ? null : coll;
    var m__7929__auto__ = cljs.core._key[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$1 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$1(coll) : m__7929__auto__.call(null, coll);
    } else {
      var m__7929__auto____$1 = cljs.core._key["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$1 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$1(coll) : m__7929__auto____$1.call(null, coll);
      } else {
        throw cljs.core.missing_protocol("IMapEntry.-key", coll);
      }
    }
  }
};
cljs.core._val = function cljs$core$_val(coll) {
  if (!(coll == null) && !(coll.cljs$core$IMapEntry$_val$arity$1 == null)) {
    return coll.cljs$core$IMapEntry$_val$arity$1(coll);
  } else {
    var x__7928__auto__ = coll == null ? null : coll;
    var m__7929__auto__ = cljs.core._val[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$1 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$1(coll) : m__7929__auto__.call(null, coll);
    } else {
      var m__7929__auto____$1 = cljs.core._val["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$1 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$1(coll) : m__7929__auto____$1.call(null, coll);
      } else {
        throw cljs.core.missing_protocol("IMapEntry.-val", coll);
      }
    }
  }
};
cljs.core.ISet = function() {
};
cljs.core._disjoin = function cljs$core$_disjoin(coll, v) {
  if (!(coll == null) && !(coll.cljs$core$ISet$_disjoin$arity$2 == null)) {
    return coll.cljs$core$ISet$_disjoin$arity$2(coll, v);
  } else {
    var x__7928__auto__ = coll == null ? null : coll;
    var m__7929__auto__ = cljs.core._disjoin[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$2 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$2(coll, v) : m__7929__auto__.call(null, coll, v);
    } else {
      var m__7929__auto____$1 = cljs.core._disjoin["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$2 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$2(coll, v) : m__7929__auto____$1.call(null, coll, v);
      } else {
        throw cljs.core.missing_protocol("ISet.-disjoin", coll);
      }
    }
  }
};
cljs.core.IStack = function() {
};
cljs.core._peek = function cljs$core$_peek(coll) {
  if (!(coll == null) && !(coll.cljs$core$IStack$_peek$arity$1 == null)) {
    return coll.cljs$core$IStack$_peek$arity$1(coll);
  } else {
    var x__7928__auto__ = coll == null ? null : coll;
    var m__7929__auto__ = cljs.core._peek[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$1 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$1(coll) : m__7929__auto__.call(null, coll);
    } else {
      var m__7929__auto____$1 = cljs.core._peek["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$1 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$1(coll) : m__7929__auto____$1.call(null, coll);
      } else {
        throw cljs.core.missing_protocol("IStack.-peek", coll);
      }
    }
  }
};
cljs.core._pop = function cljs$core$_pop(coll) {
  if (!(coll == null) && !(coll.cljs$core$IStack$_pop$arity$1 == null)) {
    return coll.cljs$core$IStack$_pop$arity$1(coll);
  } else {
    var x__7928__auto__ = coll == null ? null : coll;
    var m__7929__auto__ = cljs.core._pop[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$1 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$1(coll) : m__7929__auto__.call(null, coll);
    } else {
      var m__7929__auto____$1 = cljs.core._pop["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$1 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$1(coll) : m__7929__auto____$1.call(null, coll);
      } else {
        throw cljs.core.missing_protocol("IStack.-pop", coll);
      }
    }
  }
};
cljs.core.IVector = function() {
};
cljs.core._assoc_n = function cljs$core$_assoc_n(coll, n, val) {
  if (!(coll == null) && !(coll.cljs$core$IVector$_assoc_n$arity$3 == null)) {
    return coll.cljs$core$IVector$_assoc_n$arity$3(coll, n, val);
  } else {
    var x__7928__auto__ = coll == null ? null : coll;
    var m__7929__auto__ = cljs.core._assoc_n[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$3 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$3(coll, n, val) : m__7929__auto__.call(null, coll, n, val);
    } else {
      var m__7929__auto____$1 = cljs.core._assoc_n["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$3 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$3(coll, n, val) : m__7929__auto____$1.call(null, coll, n, val);
      } else {
        throw cljs.core.missing_protocol("IVector.-assoc-n", coll);
      }
    }
  }
};
cljs.core.IDeref = function() {
};
cljs.core._deref = function cljs$core$_deref(o) {
  if (!(o == null) && !(o.cljs$core$IDeref$_deref$arity$1 == null)) {
    return o.cljs$core$IDeref$_deref$arity$1(o);
  } else {
    var x__7928__auto__ = o == null ? null : o;
    var m__7929__auto__ = cljs.core._deref[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$1 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$1(o) : m__7929__auto__.call(null, o);
    } else {
      var m__7929__auto____$1 = cljs.core._deref["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$1 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$1(o) : m__7929__auto____$1.call(null, o);
      } else {
        throw cljs.core.missing_protocol("IDeref.-deref", o);
      }
    }
  }
};
cljs.core.IDerefWithTimeout = function() {
};
cljs.core._deref_with_timeout = function cljs$core$_deref_with_timeout(o, msec, timeout_val) {
  if (!(o == null) && !(o.cljs$core$IDerefWithTimeout$_deref_with_timeout$arity$3 == null)) {
    return o.cljs$core$IDerefWithTimeout$_deref_with_timeout$arity$3(o, msec, timeout_val);
  } else {
    var x__7928__auto__ = o == null ? null : o;
    var m__7929__auto__ = cljs.core._deref_with_timeout[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$3 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$3(o, msec, timeout_val) : m__7929__auto__.call(null, o, msec, timeout_val);
    } else {
      var m__7929__auto____$1 = cljs.core._deref_with_timeout["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$3 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$3(o, msec, timeout_val) : m__7929__auto____$1.call(null, o, msec, timeout_val);
      } else {
        throw cljs.core.missing_protocol("IDerefWithTimeout.-deref-with-timeout", o);
      }
    }
  }
};
cljs.core.IMeta = function() {
};
cljs.core._meta = function cljs$core$_meta(o) {
  if (!(o == null) && !(o.cljs$core$IMeta$_meta$arity$1 == null)) {
    return o.cljs$core$IMeta$_meta$arity$1(o);
  } else {
    var x__7928__auto__ = o == null ? null : o;
    var m__7929__auto__ = cljs.core._meta[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$1 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$1(o) : m__7929__auto__.call(null, o);
    } else {
      var m__7929__auto____$1 = cljs.core._meta["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$1 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$1(o) : m__7929__auto____$1.call(null, o);
      } else {
        throw cljs.core.missing_protocol("IMeta.-meta", o);
      }
    }
  }
};
cljs.core.IWithMeta = function() {
};
cljs.core._with_meta = function cljs$core$_with_meta(o, meta) {
  if (!(o == null) && !(o.cljs$core$IWithMeta$_with_meta$arity$2 == null)) {
    return o.cljs$core$IWithMeta$_with_meta$arity$2(o, meta);
  } else {
    var x__7928__auto__ = o == null ? null : o;
    var m__7929__auto__ = cljs.core._with_meta[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$2 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$2(o, meta) : m__7929__auto__.call(null, o, meta);
    } else {
      var m__7929__auto____$1 = cljs.core._with_meta["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$2 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$2(o, meta) : m__7929__auto____$1.call(null, o, meta);
      } else {
        throw cljs.core.missing_protocol("IWithMeta.-with-meta", o);
      }
    }
  }
};
cljs.core.IReduce = function() {
};
cljs.core._reduce = function cljs$core$_reduce(var_args) {
  var args9659 = [];
  var len__8373__auto___9662 = arguments.length;
  var i__8374__auto___9663 = 0;
  while (true) {
    if (i__8374__auto___9663 < len__8373__auto___9662) {
      args9659.push(arguments[i__8374__auto___9663]);
      var G__9664 = i__8374__auto___9663 + 1;
      i__8374__auto___9663 = G__9664;
      continue;
    } else {
    }
    break;
  }
  var G__9661 = args9659.length;
  switch(G__9661) {
    case 2:
      return cljs.core._reduce.cljs$core$IFn$_invoke$arity$2(arguments[0], arguments[1]);
      break;
    case 3:
      return cljs.core._reduce.cljs$core$IFn$_invoke$arity$3(arguments[0], arguments[1], arguments[2]);
      break;
    default:
      throw new Error([cljs.core.str.cljs$core$IFn$_invoke$arity$1("Invalid arity: "), cljs.core.str.cljs$core$IFn$_invoke$arity$1(args9659.length)].join(""));
  }
};
cljs.core._reduce.cljs$core$IFn$_invoke$arity$2 = function(coll, f) {
  if (!(coll == null) && !(coll.cljs$core$IReduce$_reduce$arity$2 == null)) {
    return coll.cljs$core$IReduce$_reduce$arity$2(coll, f);
  } else {
    var x__7928__auto__ = coll == null ? null : coll;
    var m__7929__auto__ = cljs.core._reduce[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$2 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$2(coll, f) : m__7929__auto__.call(null, coll, f);
    } else {
      var m__7929__auto____$1 = cljs.core._reduce["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$2 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$2(coll, f) : m__7929__auto____$1.call(null, coll, f);
      } else {
        throw cljs.core.missing_protocol("IReduce.-reduce", coll);
      }
    }
  }
};
cljs.core._reduce.cljs$core$IFn$_invoke$arity$3 = function(coll, f, start) {
  if (!(coll == null) && !(coll.cljs$core$IReduce$_reduce$arity$3 == null)) {
    return coll.cljs$core$IReduce$_reduce$arity$3(coll, f, start);
  } else {
    var x__7928__auto__ = coll == null ? null : coll;
    var m__7929__auto__ = cljs.core._reduce[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$3 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$3(coll, f, start) : m__7929__auto__.call(null, coll, f, start);
    } else {
      var m__7929__auto____$1 = cljs.core._reduce["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$3 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$3(coll, f, start) : m__7929__auto____$1.call(null, coll, f, start);
      } else {
        throw cljs.core.missing_protocol("IReduce.-reduce", coll);
      }
    }
  }
};
cljs.core._reduce.cljs$lang$maxFixedArity = 3;
cljs.core.IKVReduce = function() {
};
cljs.core._kv_reduce = function cljs$core$_kv_reduce(coll, f, init) {
  if (!(coll == null) && !(coll.cljs$core$IKVReduce$_kv_reduce$arity$3 == null)) {
    return coll.cljs$core$IKVReduce$_kv_reduce$arity$3(coll, f, init);
  } else {
    var x__7928__auto__ = coll == null ? null : coll;
    var m__7929__auto__ = cljs.core._kv_reduce[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$3 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$3(coll, f, init) : m__7929__auto__.call(null, coll, f, init);
    } else {
      var m__7929__auto____$1 = cljs.core._kv_reduce["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$3 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$3(coll, f, init) : m__7929__auto____$1.call(null, coll, f, init);
      } else {
        throw cljs.core.missing_protocol("IKVReduce.-kv-reduce", coll);
      }
    }
  }
};
cljs.core.IEquiv = function() {
};
cljs.core._equiv = function cljs$core$_equiv(o, other) {
  if (!(o == null) && !(o.cljs$core$IEquiv$_equiv$arity$2 == null)) {
    return o.cljs$core$IEquiv$_equiv$arity$2(o, other);
  } else {
    var x__7928__auto__ = o == null ? null : o;
    var m__7929__auto__ = cljs.core._equiv[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$2 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$2(o, other) : m__7929__auto__.call(null, o, other);
    } else {
      var m__7929__auto____$1 = cljs.core._equiv["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$2 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$2(o, other) : m__7929__auto____$1.call(null, o, other);
      } else {
        throw cljs.core.missing_protocol("IEquiv.-equiv", o);
      }
    }
  }
};
cljs.core.IHash = function() {
};
cljs.core._hash = function cljs$core$_hash(o) {
  if (!(o == null) && !(o.cljs$core$IHash$_hash$arity$1 == null)) {
    return o.cljs$core$IHash$_hash$arity$1(o);
  } else {
    var x__7928__auto__ = o == null ? null : o;
    var m__7929__auto__ = cljs.core._hash[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$1 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$1(o) : m__7929__auto__.call(null, o);
    } else {
      var m__7929__auto____$1 = cljs.core._hash["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$1 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$1(o) : m__7929__auto____$1.call(null, o);
      } else {
        throw cljs.core.missing_protocol("IHash.-hash", o);
      }
    }
  }
};
cljs.core.ISeqable = function() {
};
cljs.core._seq = function cljs$core$_seq(o) {
  if (!(o == null) && !(o.cljs$core$ISeqable$_seq$arity$1 == null)) {
    return o.cljs$core$ISeqable$_seq$arity$1(o);
  } else {
    var x__7928__auto__ = o == null ? null : o;
    var m__7929__auto__ = cljs.core._seq[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$1 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$1(o) : m__7929__auto__.call(null, o);
    } else {
      var m__7929__auto____$1 = cljs.core._seq["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$1 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$1(o) : m__7929__auto____$1.call(null, o);
      } else {
        throw cljs.core.missing_protocol("ISeqable.-seq", o);
      }
    }
  }
};
cljs.core.ISequential = function() {
};
cljs.core.IList = function() {
};
cljs.core.IRecord = function() {
};
cljs.core.IReversible = function() {
};
cljs.core._rseq = function cljs$core$_rseq(coll) {
  if (!(coll == null) && !(coll.cljs$core$IReversible$_rseq$arity$1 == null)) {
    return coll.cljs$core$IReversible$_rseq$arity$1(coll);
  } else {
    var x__7928__auto__ = coll == null ? null : coll;
    var m__7929__auto__ = cljs.core._rseq[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$1 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$1(coll) : m__7929__auto__.call(null, coll);
    } else {
      var m__7929__auto____$1 = cljs.core._rseq["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$1 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$1(coll) : m__7929__auto____$1.call(null, coll);
      } else {
        throw cljs.core.missing_protocol("IReversible.-rseq", coll);
      }
    }
  }
};
cljs.core.ISorted = function() {
};
cljs.core._sorted_seq = function cljs$core$_sorted_seq(coll, ascending_QMARK_) {
  if (!(coll == null) && !(coll.cljs$core$ISorted$_sorted_seq$arity$2 == null)) {
    return coll.cljs$core$ISorted$_sorted_seq$arity$2(coll, ascending_QMARK_);
  } else {
    var x__7928__auto__ = coll == null ? null : coll;
    var m__7929__auto__ = cljs.core._sorted_seq[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$2 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$2(coll, ascending_QMARK_) : m__7929__auto__.call(null, coll, ascending_QMARK_);
    } else {
      var m__7929__auto____$1 = cljs.core._sorted_seq["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$2 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$2(coll, ascending_QMARK_) : m__7929__auto____$1.call(null, coll, ascending_QMARK_);
      } else {
        throw cljs.core.missing_protocol("ISorted.-sorted-seq", coll);
      }
    }
  }
};
cljs.core._sorted_seq_from = function cljs$core$_sorted_seq_from(coll, k, ascending_QMARK_) {
  if (!(coll == null) && !(coll.cljs$core$ISorted$_sorted_seq_from$arity$3 == null)) {
    return coll.cljs$core$ISorted$_sorted_seq_from$arity$3(coll, k, ascending_QMARK_);
  } else {
    var x__7928__auto__ = coll == null ? null : coll;
    var m__7929__auto__ = cljs.core._sorted_seq_from[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$3 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$3(coll, k, ascending_QMARK_) : m__7929__auto__.call(null, coll, k, ascending_QMARK_);
    } else {
      var m__7929__auto____$1 = cljs.core._sorted_seq_from["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$3 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$3(coll, k, ascending_QMARK_) : m__7929__auto____$1.call(null, coll, k, ascending_QMARK_);
      } else {
        throw cljs.core.missing_protocol("ISorted.-sorted-seq-from", coll);
      }
    }
  }
};
cljs.core._entry_key = function cljs$core$_entry_key(coll, entry) {
  if (!(coll == null) && !(coll.cljs$core$ISorted$_entry_key$arity$2 == null)) {
    return coll.cljs$core$ISorted$_entry_key$arity$2(coll, entry);
  } else {
    var x__7928__auto__ = coll == null ? null : coll;
    var m__7929__auto__ = cljs.core._entry_key[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$2 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$2(coll, entry) : m__7929__auto__.call(null, coll, entry);
    } else {
      var m__7929__auto____$1 = cljs.core._entry_key["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$2 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$2(coll, entry) : m__7929__auto____$1.call(null, coll, entry);
      } else {
        throw cljs.core.missing_protocol("ISorted.-entry-key", coll);
      }
    }
  }
};
cljs.core._comparator = function cljs$core$_comparator(coll) {
  if (!(coll == null) && !(coll.cljs$core$ISorted$_comparator$arity$1 == null)) {
    return coll.cljs$core$ISorted$_comparator$arity$1(coll);
  } else {
    var x__7928__auto__ = coll == null ? null : coll;
    var m__7929__auto__ = cljs.core._comparator[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$1 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$1(coll) : m__7929__auto__.call(null, coll);
    } else {
      var m__7929__auto____$1 = cljs.core._comparator["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$1 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$1(coll) : m__7929__auto____$1.call(null, coll);
      } else {
        throw cljs.core.missing_protocol("ISorted.-comparator", coll);
      }
    }
  }
};
cljs.core.IWriter = function() {
};
cljs.core._write = function cljs$core$_write(writer, s) {
  if (!(writer == null) && !(writer.cljs$core$IWriter$_write$arity$2 == null)) {
    return writer.cljs$core$IWriter$_write$arity$2(writer, s);
  } else {
    var x__7928__auto__ = writer == null ? null : writer;
    var m__7929__auto__ = cljs.core._write[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$2 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$2(writer, s) : m__7929__auto__.call(null, writer, s);
    } else {
      var m__7929__auto____$1 = cljs.core._write["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$2 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$2(writer, s) : m__7929__auto____$1.call(null, writer, s);
      } else {
        throw cljs.core.missing_protocol("IWriter.-write", writer);
      }
    }
  }
};
cljs.core._flush = function cljs$core$_flush(writer) {
  if (!(writer == null) && !(writer.cljs$core$IWriter$_flush$arity$1 == null)) {
    return writer.cljs$core$IWriter$_flush$arity$1(writer);
  } else {
    var x__7928__auto__ = writer == null ? null : writer;
    var m__7929__auto__ = cljs.core._flush[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$1 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$1(writer) : m__7929__auto__.call(null, writer);
    } else {
      var m__7929__auto____$1 = cljs.core._flush["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$1 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$1(writer) : m__7929__auto____$1.call(null, writer);
      } else {
        throw cljs.core.missing_protocol("IWriter.-flush", writer);
      }
    }
  }
};
cljs.core.IPrintWithWriter = function() {
};
cljs.core._pr_writer = function cljs$core$_pr_writer(o, writer, opts) {
  if (!(o == null) && !(o.cljs$core$IPrintWithWriter$_pr_writer$arity$3 == null)) {
    return o.cljs$core$IPrintWithWriter$_pr_writer$arity$3(o, writer, opts);
  } else {
    var x__7928__auto__ = o == null ? null : o;
    var m__7929__auto__ = cljs.core._pr_writer[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$3 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$3(o, writer, opts) : m__7929__auto__.call(null, o, writer, opts);
    } else {
      var m__7929__auto____$1 = cljs.core._pr_writer["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$3 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$3(o, writer, opts) : m__7929__auto____$1.call(null, o, writer, opts);
      } else {
        throw cljs.core.missing_protocol("IPrintWithWriter.-pr-writer", o);
      }
    }
  }
};
cljs.core.IPending = function() {
};
cljs.core._realized_QMARK_ = function cljs$core$_realized_QMARK_(x) {
  if (!(x == null) && !(x.cljs$core$IPending$_realized_QMARK_$arity$1 == null)) {
    return x.cljs$core$IPending$_realized_QMARK_$arity$1(x);
  } else {
    var x__7928__auto__ = x == null ? null : x;
    var m__7929__auto__ = cljs.core._realized_QMARK_[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$1 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$1(x) : m__7929__auto__.call(null, x);
    } else {
      var m__7929__auto____$1 = cljs.core._realized_QMARK_["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$1 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$1(x) : m__7929__auto____$1.call(null, x);
      } else {
        throw cljs.core.missing_protocol("IPending.-realized?", x);
      }
    }
  }
};
cljs.core.IWatchable = function() {
};
cljs.core._notify_watches = function cljs$core$_notify_watches(this$, oldval, newval) {
  if (!(this$ == null) && !(this$.cljs$core$IWatchable$_notify_watches$arity$3 == null)) {
    return this$.cljs$core$IWatchable$_notify_watches$arity$3(this$, oldval, newval);
  } else {
    var x__7928__auto__ = this$ == null ? null : this$;
    var m__7929__auto__ = cljs.core._notify_watches[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$3 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$3(this$, oldval, newval) : m__7929__auto__.call(null, this$, oldval, newval);
    } else {
      var m__7929__auto____$1 = cljs.core._notify_watches["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$3 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$3(this$, oldval, newval) : m__7929__auto____$1.call(null, this$, oldval, newval);
      } else {
        throw cljs.core.missing_protocol("IWatchable.-notify-watches", this$);
      }
    }
  }
};
cljs.core._add_watch = function cljs$core$_add_watch(this$, key, f) {
  if (!(this$ == null) && !(this$.cljs$core$IWatchable$_add_watch$arity$3 == null)) {
    return this$.cljs$core$IWatchable$_add_watch$arity$3(this$, key, f);
  } else {
    var x__7928__auto__ = this$ == null ? null : this$;
    var m__7929__auto__ = cljs.core._add_watch[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$3 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$3(this$, key, f) : m__7929__auto__.call(null, this$, key, f);
    } else {
      var m__7929__auto____$1 = cljs.core._add_watch["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$3 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$3(this$, key, f) : m__7929__auto____$1.call(null, this$, key, f);
      } else {
        throw cljs.core.missing_protocol("IWatchable.-add-watch", this$);
      }
    }
  }
};
cljs.core._remove_watch = function cljs$core$_remove_watch(this$, key) {
  if (!(this$ == null) && !(this$.cljs$core$IWatchable$_remove_watch$arity$2 == null)) {
    return this$.cljs$core$IWatchable$_remove_watch$arity$2(this$, key);
  } else {
    var x__7928__auto__ = this$ == null ? null : this$;
    var m__7929__auto__ = cljs.core._remove_watch[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$2 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$2(this$, key) : m__7929__auto__.call(null, this$, key);
    } else {
      var m__7929__auto____$1 = cljs.core._remove_watch["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$2 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$2(this$, key) : m__7929__auto____$1.call(null, this$, key);
      } else {
        throw cljs.core.missing_protocol("IWatchable.-remove-watch", this$);
      }
    }
  }
};
cljs.core.IEditableCollection = function() {
};
cljs.core._as_transient = function cljs$core$_as_transient(coll) {
  if (!(coll == null) && !(coll.cljs$core$IEditableCollection$_as_transient$arity$1 == null)) {
    return coll.cljs$core$IEditableCollection$_as_transient$arity$1(coll);
  } else {
    var x__7928__auto__ = coll == null ? null : coll;
    var m__7929__auto__ = cljs.core._as_transient[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$1 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$1(coll) : m__7929__auto__.call(null, coll);
    } else {
      var m__7929__auto____$1 = cljs.core._as_transient["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$1 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$1(coll) : m__7929__auto____$1.call(null, coll);
      } else {
        throw cljs.core.missing_protocol("IEditableCollection.-as-transient", coll);
      }
    }
  }
};
cljs.core.ITransientCollection = function() {
};
cljs.core._conj_BANG_ = function cljs$core$_conj_BANG_(tcoll, val) {
  if (!(tcoll == null) && !(tcoll.cljs$core$ITransientCollection$_conj_BANG_$arity$2 == null)) {
    return tcoll.cljs$core$ITransientCollection$_conj_BANG_$arity$2(tcoll, val);
  } else {
    var x__7928__auto__ = tcoll == null ? null : tcoll;
    var m__7929__auto__ = cljs.core._conj_BANG_[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$2 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$2(tcoll, val) : m__7929__auto__.call(null, tcoll, val);
    } else {
      var m__7929__auto____$1 = cljs.core._conj_BANG_["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$2 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$2(tcoll, val) : m__7929__auto____$1.call(null, tcoll, val);
      } else {
        throw cljs.core.missing_protocol("ITransientCollection.-conj!", tcoll);
      }
    }
  }
};
cljs.core._persistent_BANG_ = function cljs$core$_persistent_BANG_(tcoll) {
  if (!(tcoll == null) && !(tcoll.cljs$core$ITransientCollection$_persistent_BANG_$arity$1 == null)) {
    return tcoll.cljs$core$ITransientCollection$_persistent_BANG_$arity$1(tcoll);
  } else {
    var x__7928__auto__ = tcoll == null ? null : tcoll;
    var m__7929__auto__ = cljs.core._persistent_BANG_[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$1 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$1(tcoll) : m__7929__auto__.call(null, tcoll);
    } else {
      var m__7929__auto____$1 = cljs.core._persistent_BANG_["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$1 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$1(tcoll) : m__7929__auto____$1.call(null, tcoll);
      } else {
        throw cljs.core.missing_protocol("ITransientCollection.-persistent!", tcoll);
      }
    }
  }
};
cljs.core.ITransientAssociative = function() {
};
cljs.core._assoc_BANG_ = function cljs$core$_assoc_BANG_(tcoll, key, val) {
  if (!(tcoll == null) && !(tcoll.cljs$core$ITransientAssociative$_assoc_BANG_$arity$3 == null)) {
    return tcoll.cljs$core$ITransientAssociative$_assoc_BANG_$arity$3(tcoll, key, val);
  } else {
    var x__7928__auto__ = tcoll == null ? null : tcoll;
    var m__7929__auto__ = cljs.core._assoc_BANG_[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$3 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$3(tcoll, key, val) : m__7929__auto__.call(null, tcoll, key, val);
    } else {
      var m__7929__auto____$1 = cljs.core._assoc_BANG_["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$3 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$3(tcoll, key, val) : m__7929__auto____$1.call(null, tcoll, key, val);
      } else {
        throw cljs.core.missing_protocol("ITransientAssociative.-assoc!", tcoll);
      }
    }
  }
};
cljs.core.ITransientMap = function() {
};
cljs.core._dissoc_BANG_ = function cljs$core$_dissoc_BANG_(tcoll, key) {
  if (!(tcoll == null) && !(tcoll.cljs$core$ITransientMap$_dissoc_BANG_$arity$2 == null)) {
    return tcoll.cljs$core$ITransientMap$_dissoc_BANG_$arity$2(tcoll, key);
  } else {
    var x__7928__auto__ = tcoll == null ? null : tcoll;
    var m__7929__auto__ = cljs.core._dissoc_BANG_[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$2 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$2(tcoll, key) : m__7929__auto__.call(null, tcoll, key);
    } else {
      var m__7929__auto____$1 = cljs.core._dissoc_BANG_["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$2 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$2(tcoll, key) : m__7929__auto____$1.call(null, tcoll, key);
      } else {
        throw cljs.core.missing_protocol("ITransientMap.-dissoc!", tcoll);
      }
    }
  }
};
cljs.core.ITransientVector = function() {
};
cljs.core._assoc_n_BANG_ = function cljs$core$_assoc_n_BANG_(tcoll, n, val) {
  if (!(tcoll == null) && !(tcoll.cljs$core$ITransientVector$_assoc_n_BANG_$arity$3 == null)) {
    return tcoll.cljs$core$ITransientVector$_assoc_n_BANG_$arity$3(tcoll, n, val);
  } else {
    var x__7928__auto__ = tcoll == null ? null : tcoll;
    var m__7929__auto__ = cljs.core._assoc_n_BANG_[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$3 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$3(tcoll, n, val) : m__7929__auto__.call(null, tcoll, n, val);
    } else {
      var m__7929__auto____$1 = cljs.core._assoc_n_BANG_["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$3 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$3(tcoll, n, val) : m__7929__auto____$1.call(null, tcoll, n, val);
      } else {
        throw cljs.core.missing_protocol("ITransientVector.-assoc-n!", tcoll);
      }
    }
  }
};
cljs.core._pop_BANG_ = function cljs$core$_pop_BANG_(tcoll) {
  if (!(tcoll == null) && !(tcoll.cljs$core$ITransientVector$_pop_BANG_$arity$1 == null)) {
    return tcoll.cljs$core$ITransientVector$_pop_BANG_$arity$1(tcoll);
  } else {
    var x__7928__auto__ = tcoll == null ? null : tcoll;
    var m__7929__auto__ = cljs.core._pop_BANG_[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$1 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$1(tcoll) : m__7929__auto__.call(null, tcoll);
    } else {
      var m__7929__auto____$1 = cljs.core._pop_BANG_["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$1 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$1(tcoll) : m__7929__auto____$1.call(null, tcoll);
      } else {
        throw cljs.core.missing_protocol("ITransientVector.-pop!", tcoll);
      }
    }
  }
};
cljs.core.ITransientSet = function() {
};
cljs.core._disjoin_BANG_ = function cljs$core$_disjoin_BANG_(tcoll, v) {
  if (!(tcoll == null) && !(tcoll.cljs$core$ITransientSet$_disjoin_BANG_$arity$2 == null)) {
    return tcoll.cljs$core$ITransientSet$_disjoin_BANG_$arity$2(tcoll, v);
  } else {
    var x__7928__auto__ = tcoll == null ? null : tcoll;
    var m__7929__auto__ = cljs.core._disjoin_BANG_[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$2 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$2(tcoll, v) : m__7929__auto__.call(null, tcoll, v);
    } else {
      var m__7929__auto____$1 = cljs.core._disjoin_BANG_["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$2 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$2(tcoll, v) : m__7929__auto____$1.call(null, tcoll, v);
      } else {
        throw cljs.core.missing_protocol("ITransientSet.-disjoin!", tcoll);
      }
    }
  }
};
cljs.core.IComparable = function() {
};
cljs.core._compare = function cljs$core$_compare(x, y) {
  if (!(x == null) && !(x.cljs$core$IComparable$_compare$arity$2 == null)) {
    return x.cljs$core$IComparable$_compare$arity$2(x, y);
  } else {
    var x__7928__auto__ = x == null ? null : x;
    var m__7929__auto__ = cljs.core._compare[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$2 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$2(x, y) : m__7929__auto__.call(null, x, y);
    } else {
      var m__7929__auto____$1 = cljs.core._compare["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$2 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$2(x, y) : m__7929__auto____$1.call(null, x, y);
      } else {
        throw cljs.core.missing_protocol("IComparable.-compare", x);
      }
    }
  }
};
cljs.core.IChunk = function() {
};
cljs.core._drop_first = function cljs$core$_drop_first(coll) {
  if (!(coll == null) && !(coll.cljs$core$IChunk$_drop_first$arity$1 == null)) {
    return coll.cljs$core$IChunk$_drop_first$arity$1(coll);
  } else {
    var x__7928__auto__ = coll == null ? null : coll;
    var m__7929__auto__ = cljs.core._drop_first[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$1 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$1(coll) : m__7929__auto__.call(null, coll);
    } else {
      var m__7929__auto____$1 = cljs.core._drop_first["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$1 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$1(coll) : m__7929__auto____$1.call(null, coll);
      } else {
        throw cljs.core.missing_protocol("IChunk.-drop-first", coll);
      }
    }
  }
};
cljs.core.IChunkedSeq = function() {
};
cljs.core._chunked_first = function cljs$core$_chunked_first(coll) {
  if (!(coll == null) && !(coll.cljs$core$IChunkedSeq$_chunked_first$arity$1 == null)) {
    return coll.cljs$core$IChunkedSeq$_chunked_first$arity$1(coll);
  } else {
    var x__7928__auto__ = coll == null ? null : coll;
    var m__7929__auto__ = cljs.core._chunked_first[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$1 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$1(coll) : m__7929__auto__.call(null, coll);
    } else {
      var m__7929__auto____$1 = cljs.core._chunked_first["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$1 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$1(coll) : m__7929__auto____$1.call(null, coll);
      } else {
        throw cljs.core.missing_protocol("IChunkedSeq.-chunked-first", coll);
      }
    }
  }
};
cljs.core._chunked_rest = function cljs$core$_chunked_rest(coll) {
  if (!(coll == null) && !(coll.cljs$core$IChunkedSeq$_chunked_rest$arity$1 == null)) {
    return coll.cljs$core$IChunkedSeq$_chunked_rest$arity$1(coll);
  } else {
    var x__7928__auto__ = coll == null ? null : coll;
    var m__7929__auto__ = cljs.core._chunked_rest[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$1 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$1(coll) : m__7929__auto__.call(null, coll);
    } else {
      var m__7929__auto____$1 = cljs.core._chunked_rest["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$1 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$1(coll) : m__7929__auto____$1.call(null, coll);
      } else {
        throw cljs.core.missing_protocol("IChunkedSeq.-chunked-rest", coll);
      }
    }
  }
};
cljs.core.IChunkedNext = function() {
};
cljs.core._chunked_next = function cljs$core$_chunked_next(coll) {
  if (!(coll == null) && !(coll.cljs$core$IChunkedNext$_chunked_next$arity$1 == null)) {
    return coll.cljs$core$IChunkedNext$_chunked_next$arity$1(coll);
  } else {
    var x__7928__auto__ = coll == null ? null : coll;
    var m__7929__auto__ = cljs.core._chunked_next[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$1 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$1(coll) : m__7929__auto__.call(null, coll);
    } else {
      var m__7929__auto____$1 = cljs.core._chunked_next["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$1 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$1(coll) : m__7929__auto____$1.call(null, coll);
      } else {
        throw cljs.core.missing_protocol("IChunkedNext.-chunked-next", coll);
      }
    }
  }
};
cljs.core.INamed = function() {
};
cljs.core._name = function cljs$core$_name(x) {
  if (!(x == null) && !(x.cljs$core$INamed$_name$arity$1 == null)) {
    return x.cljs$core$INamed$_name$arity$1(x);
  } else {
    var x__7928__auto__ = x == null ? null : x;
    var m__7929__auto__ = cljs.core._name[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$1 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$1(x) : m__7929__auto__.call(null, x);
    } else {
      var m__7929__auto____$1 = cljs.core._name["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$1 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$1(x) : m__7929__auto____$1.call(null, x);
      } else {
        throw cljs.core.missing_protocol("INamed.-name", x);
      }
    }
  }
};
cljs.core._namespace = function cljs$core$_namespace(x) {
  if (!(x == null) && !(x.cljs$core$INamed$_namespace$arity$1 == null)) {
    return x.cljs$core$INamed$_namespace$arity$1(x);
  } else {
    var x__7928__auto__ = x == null ? null : x;
    var m__7929__auto__ = cljs.core._namespace[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$1 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$1(x) : m__7929__auto__.call(null, x);
    } else {
      var m__7929__auto____$1 = cljs.core._namespace["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$1 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$1(x) : m__7929__auto____$1.call(null, x);
      } else {
        throw cljs.core.missing_protocol("INamed.-namespace", x);
      }
    }
  }
};
cljs.core.IAtom = function() {
};
cljs.core.IReset = function() {
};
cljs.core._reset_BANG_ = function cljs$core$_reset_BANG_(o, new_value) {
  if (!(o == null) && !(o.cljs$core$IReset$_reset_BANG_$arity$2 == null)) {
    return o.cljs$core$IReset$_reset_BANG_$arity$2(o, new_value);
  } else {
    var x__7928__auto__ = o == null ? null : o;
    var m__7929__auto__ = cljs.core._reset_BANG_[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$2 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$2(o, new_value) : m__7929__auto__.call(null, o, new_value);
    } else {
      var m__7929__auto____$1 = cljs.core._reset_BANG_["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$2 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$2(o, new_value) : m__7929__auto____$1.call(null, o, new_value);
      } else {
        throw cljs.core.missing_protocol("IReset.-reset!", o);
      }
    }
  }
};
cljs.core.ISwap = function() {
};
cljs.core._swap_BANG_ = function cljs$core$_swap_BANG_(var_args) {
  var args9666 = [];
  var len__8373__auto___9669 = arguments.length;
  var i__8374__auto___9670 = 0;
  while (true) {
    if (i__8374__auto___9670 < len__8373__auto___9669) {
      args9666.push(arguments[i__8374__auto___9670]);
      var G__9671 = i__8374__auto___9670 + 1;
      i__8374__auto___9670 = G__9671;
      continue;
    } else {
    }
    break;
  }
  var G__9668 = args9666.length;
  switch(G__9668) {
    case 2:
      return cljs.core._swap_BANG_.cljs$core$IFn$_invoke$arity$2(arguments[0], arguments[1]);
      break;
    case 3:
      return cljs.core._swap_BANG_.cljs$core$IFn$_invoke$arity$3(arguments[0], arguments[1], arguments[2]);
      break;
    case 4:
      return cljs.core._swap_BANG_.cljs$core$IFn$_invoke$arity$4(arguments[0], arguments[1], arguments[2], arguments[3]);
      break;
    case 5:
      return cljs.core._swap_BANG_.cljs$core$IFn$_invoke$arity$5(arguments[0], arguments[1], arguments[2], arguments[3], arguments[4]);
      break;
    default:
      throw new Error([cljs.core.str.cljs$core$IFn$_invoke$arity$1("Invalid arity: "), cljs.core.str.cljs$core$IFn$_invoke$arity$1(args9666.length)].join(""));
  }
};
cljs.core._swap_BANG_.cljs$core$IFn$_invoke$arity$2 = function(o, f) {
  if (!(o == null) && !(o.cljs$core$ISwap$_swap_BANG_$arity$2 == null)) {
    return o.cljs$core$ISwap$_swap_BANG_$arity$2(o, f);
  } else {
    var x__7928__auto__ = o == null ? null : o;
    var m__7929__auto__ = cljs.core._swap_BANG_[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$2 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$2(o, f) : m__7929__auto__.call(null, o, f);
    } else {
      var m__7929__auto____$1 = cljs.core._swap_BANG_["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$2 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$2(o, f) : m__7929__auto____$1.call(null, o, f);
      } else {
        throw cljs.core.missing_protocol("ISwap.-swap!", o);
      }
    }
  }
};
cljs.core._swap_BANG_.cljs$core$IFn$_invoke$arity$3 = function(o, f, a) {
  if (!(o == null) && !(o.cljs$core$ISwap$_swap_BANG_$arity$3 == null)) {
    return o.cljs$core$ISwap$_swap_BANG_$arity$3(o, f, a);
  } else {
    var x__7928__auto__ = o == null ? null : o;
    var m__7929__auto__ = cljs.core._swap_BANG_[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$3 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$3(o, f, a) : m__7929__auto__.call(null, o, f, a);
    } else {
      var m__7929__auto____$1 = cljs.core._swap_BANG_["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$3 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$3(o, f, a) : m__7929__auto____$1.call(null, o, f, a);
      } else {
        throw cljs.core.missing_protocol("ISwap.-swap!", o);
      }
    }
  }
};
cljs.core._swap_BANG_.cljs$core$IFn$_invoke$arity$4 = function(o, f, a, b) {
  if (!(o == null) && !(o.cljs$core$ISwap$_swap_BANG_$arity$4 == null)) {
    return o.cljs$core$ISwap$_swap_BANG_$arity$4(o, f, a, b);
  } else {
    var x__7928__auto__ = o == null ? null : o;
    var m__7929__auto__ = cljs.core._swap_BANG_[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$4 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$4(o, f, a, b) : m__7929__auto__.call(null, o, f, a, b);
    } else {
      var m__7929__auto____$1 = cljs.core._swap_BANG_["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$4 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$4(o, f, a, b) : m__7929__auto____$1.call(null, o, f, a, b);
      } else {
        throw cljs.core.missing_protocol("ISwap.-swap!", o);
      }
    }
  }
};
cljs.core._swap_BANG_.cljs$core$IFn$_invoke$arity$5 = function(o, f, a, b, xs) {
  if (!(o == null) && !(o.cljs$core$ISwap$_swap_BANG_$arity$5 == null)) {
    return o.cljs$core$ISwap$_swap_BANG_$arity$5(o, f, a, b, xs);
  } else {
    var x__7928__auto__ = o == null ? null : o;
    var m__7929__auto__ = cljs.core._swap_BANG_[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$5 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$5(o, f, a, b, xs) : m__7929__auto__.call(null, o, f, a, b, xs);
    } else {
      var m__7929__auto____$1 = cljs.core._swap_BANG_["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$5 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$5(o, f, a, b, xs) : m__7929__auto____$1.call(null, o, f, a, b, xs);
      } else {
        throw cljs.core.missing_protocol("ISwap.-swap!", o);
      }
    }
  }
};
cljs.core._swap_BANG_.cljs$lang$maxFixedArity = 5;
cljs.core.IVolatile = function() {
};
cljs.core._vreset_BANG_ = function cljs$core$_vreset_BANG_(o, new_value) {
  if (!(o == null) && !(o.cljs$core$IVolatile$_vreset_BANG_$arity$2 == null)) {
    return o.cljs$core$IVolatile$_vreset_BANG_$arity$2(o, new_value);
  } else {
    var x__7928__auto__ = o == null ? null : o;
    var m__7929__auto__ = cljs.core._vreset_BANG_[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$2 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$2(o, new_value) : m__7929__auto__.call(null, o, new_value);
    } else {
      var m__7929__auto____$1 = cljs.core._vreset_BANG_["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$2 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$2(o, new_value) : m__7929__auto____$1.call(null, o, new_value);
      } else {
        throw cljs.core.missing_protocol("IVolatile.-vreset!", o);
      }
    }
  }
};
cljs.core.IIterable = function() {
};
cljs.core._iterator = function cljs$core$_iterator(coll) {
  if (!(coll == null) && !(coll.cljs$core$IIterable$_iterator$arity$1 == null)) {
    return coll.cljs$core$IIterable$_iterator$arity$1(coll);
  } else {
    var x__7928__auto__ = coll == null ? null : coll;
    var m__7929__auto__ = cljs.core._iterator[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$1 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$1(coll) : m__7929__auto__.call(null, coll);
    } else {
      var m__7929__auto____$1 = cljs.core._iterator["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$1 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$1(coll) : m__7929__auto____$1.call(null, coll);
      } else {
        throw cljs.core.missing_protocol("IIterable.-iterator", coll);
      }
    }
  }
};
cljs.core.StringBufferWriter = function(sb) {
  this.sb = sb;
  this.cljs$lang$protocol_mask$partition0$ = 1073741824;
  this.cljs$lang$protocol_mask$partition1$ = 0;
};
cljs.core.StringBufferWriter.prototype.cljs$core$IWriter$_write$arity$2 = function(_, s) {
  var self__ = this;
  var ___$1 = this;
  return self__.sb.append(s);
};
cljs.core.StringBufferWriter.prototype.cljs$core$IWriter$_flush$arity$1 = function(_) {
  var self__ = this;
  var ___$1 = this;
  return null;
};
cljs.core.StringBufferWriter.getBasis = function() {
  return new cljs.core.PersistentVector(null, 1, 5, cljs.core.PersistentVector.EMPTY_NODE, [new cljs.core.Symbol(null, "sb", "sb", -1249746442, null)], null);
};
cljs.core.StringBufferWriter.cljs$lang$type = true;
cljs.core.StringBufferWriter.cljs$lang$ctorStr = "cljs.core/StringBufferWriter";
cljs.core.StringBufferWriter.cljs$lang$ctorPrWriter = function(this__7871__auto__, writer__7872__auto__, opt__7873__auto__) {
  return cljs.core._write(writer__7872__auto__, "cljs.core/StringBufferWriter");
};
cljs.core.__GT_StringBufferWriter = function cljs$core$__GT_StringBufferWriter(sb) {
  return new cljs.core.StringBufferWriter(sb);
};
cljs.core.pr_str_STAR_ = function cljs$core$pr_str_STAR_(obj) {
  var sb = new goog.string.StringBuffer;
  var writer = new cljs.core.StringBufferWriter(sb);
  obj.cljs$core$IPrintWithWriter$_pr_writer$arity$3(null, writer, cljs.core.pr_opts());
  writer.cljs$core$IWriter$_flush$arity$1(null);
  return [cljs.core.str.cljs$core$IFn$_invoke$arity$1(sb)].join("");
};
cljs.core.int_rotate_left = function cljs$core$int_rotate_left(x, n) {
  return x << n | x >>> -n;
};
if (typeof Math.imul !== "undefined" && !(Math.imul(4294967295, 5) === 0)) {
  cljs.core.imul = function cljs$core$imul(a, b) {
    return Math.imul(a, b);
  };
} else {
  cljs.core.imul = function cljs$core$imul(a, b) {
    var ah = a >>> 16 & 65535;
    var al = a & 65535;
    var bh = b >>> 16 & 65535;
    var bl = b & 65535;
    return al * bl + (ah * bl + al * bh << 16 >>> 0) | 0;
  };
}
cljs.core.m3_seed = 0;
cljs.core.m3_C1 = 3432918353 | 0;
cljs.core.m3_C2 = 461845907 | 0;
cljs.core.m3_mix_K1 = function cljs$core$m3_mix_K1(k1) {
  return cljs.core.imul(cljs.core.int_rotate_left(cljs.core.imul(k1 | 0, cljs.core.m3_C1), 15), cljs.core.m3_C2);
};
cljs.core.m3_mix_H1 = function cljs$core$m3_mix_H1(h1, k1) {
  return cljs.core.imul(cljs.core.int_rotate_left((h1 | 0) ^ (k1 | 0), 13), 5) + (3864292196 | 0) | 0;
};
cljs.core.m3_fmix = function cljs$core$m3_fmix(h1, len) {
  var h1__$1 = h1 | 0;
  var h1__$2 = h1__$1 ^ len;
  var h1__$3 = h1__$2 ^ h1__$2 >>> 16;
  var h1__$4 = cljs.core.imul(h1__$3, 2246822507 | 0);
  var h1__$5 = h1__$4 ^ h1__$4 >>> 13;
  var h1__$6 = cljs.core.imul(h1__$5, 3266489909 | 0);
  return h1__$6 ^ h1__$6 >>> 16;
};
cljs.core.m3_hash_int = function cljs$core$m3_hash_int(in$) {
  if (in$ === 0) {
    return in$;
  } else {
    var k1 = cljs.core.m3_mix_K1(in$);
    var h1 = cljs.core.m3_mix_H1(cljs.core.m3_seed, k1);
    return cljs.core.m3_fmix(h1, 4);
  }
};
cljs.core.m3_hash_unencoded_chars = function cljs$core$m3_hash_unencoded_chars(in$) {
  var h1 = function() {
    var i = 1;
    var h1 = cljs.core.m3_seed;
    while (true) {
      if (i < in$.length) {
        var G__9673 = i + 2;
        var G__9674 = cljs.core.m3_mix_H1(h1, cljs.core.m3_mix_K1(in$.charCodeAt(i - 1) | in$.charCodeAt(i) << 16));
        i = G__9673;
        h1 = G__9674;
        continue;
      } else {
        return h1;
      }
      break;
    }
  }();
  var h1__$1 = (in$.length & 1) === 1 ? h1 ^ cljs.core.m3_mix_K1(in$.charCodeAt(in$.length - 1)) : h1;
  return cljs.core.m3_fmix(h1__$1, cljs.core.imul(2, in$.length));
};
cljs.core.string_hash_cache = {};
cljs.core.string_hash_cache_count = 0;
cljs.core.hash_string_STAR_ = function cljs$core$hash_string_STAR_(s) {
  if (!(s == null)) {
    var len = s.length;
    if (len > 0) {
      var i = 0;
      var hash = 0;
      while (true) {
        if (i < len) {
          var G__9677 = i + 1;
          var G__9678 = cljs.core.imul(31, hash) + s.charCodeAt(i);
          i = G__9677;
          hash = G__9678;
          continue;
        } else {
          return hash;
        }
        break;
      }
    } else {
      return 0;
    }
  } else {
    return 0;
  }
};
cljs.core.add_to_string_hash_cache = function cljs$core$add_to_string_hash_cache(k) {
  var h = cljs.core.hash_string_STAR_(k);
  cljs.core.string_hash_cache[k] = h;
  cljs.core.string_hash_cache_count = cljs.core.string_hash_cache_count + 1;
  return h;
};
cljs.core.hash_string = function cljs$core$hash_string(k) {
  if (cljs.core.string_hash_cache_count > 255) {
    cljs.core.string_hash_cache = {};
    cljs.core.string_hash_cache_count = 0;
  } else {
  }
  if (k == null) {
    return 0;
  } else {
    var h = cljs.core.string_hash_cache[k];
    if (typeof h === "number") {
      return h;
    } else {
      return cljs.core.add_to_string_hash_cache(k);
    }
  }
};
cljs.core.hash = function cljs$core$hash(o) {
  if (!(o == null) ? o.cljs$lang$protocol_mask$partition0$ & 4194304 || cljs.core.PROTOCOL_SENTINEL === o.cljs$core$IHash$ ? true : false : false) {
    return o.cljs$core$IHash$_hash$arity$1(null) ^ 0;
  } else {
    if (typeof o === "number") {
      if (cljs.core.truth_(isFinite(o))) {
        return Math.floor(o) % 2147483647;
      } else {
        var G__9686 = o;
        switch(G__9686) {
          case Infinity:
            return 2146435072;
            break;
          case -Infinity:
            return -1048576;
            break;
          default:
            return 2146959360;
        }
      }
    } else {
      if (o === true) {
        return 1231;
      } else {
        if (o === false) {
          return 1237;
        } else {
          if (typeof o === "string") {
            return cljs.core.m3_hash_int(cljs.core.hash_string(o));
          } else {
            if (o instanceof Date) {
              return o.valueOf() ^ 0;
            } else {
              if (o == null) {
                return 0;
              } else {
                return cljs.core._hash(o) ^ 0;
              }
            }
          }
        }
      }
    }
  }
};
cljs.core.hash_combine = function cljs$core$hash_combine(seed, hash) {
  return seed ^ hash + 2654435769 + (seed << 6) + (seed >> 2);
};
cljs.core.instance_QMARK_ = function cljs$core$instance_QMARK_(c, x) {
  return x instanceof c;
};
cljs.core.symbol_QMARK_ = function cljs$core$symbol_QMARK_(x) {
  return x instanceof cljs.core.Symbol;
};
cljs.core.hash_symbol = function cljs$core$hash_symbol(sym) {
  return cljs.core.hash_combine(cljs.core.m3_hash_unencoded_chars(sym.name), cljs.core.hash_string(sym.ns));
};
cljs.core.compare_symbols = function cljs$core$compare_symbols(a, b) {
  if (a.str === b.str) {
    return 0;
  } else {
    if (cljs.core.truth_(function() {
      var and__7248__auto__ = cljs.core.not(a.ns);
      if (and__7248__auto__) {
        return b.ns;
      } else {
        return and__7248__auto__;
      }
    }())) {
      return -1;
    } else {
      if (cljs.core.truth_(a.ns)) {
        if (cljs.core.not(b.ns)) {
          return 1;
        } else {
          var nsc = function() {
            var G__9694 = a.ns;
            var G__9695 = b.ns;
            return goog.array.defaultCompare(G__9694, G__9695);
          }();
          if (0 === nsc) {
            var G__9696 = a.name;
            var G__9697 = b.name;
            return goog.array.defaultCompare(G__9696, G__9697);
          } else {
            return nsc;
          }
        }
      } else {
        var G__9698 = a.name;
        var G__9699 = b.name;
        return goog.array.defaultCompare(G__9698, G__9699);
      }
    }
  }
};
cljs.core.Symbol = function(ns, name, str, _hash, _meta) {
  this.ns = ns;
  this.name = name;
  this.str = str;
  this._hash = _hash;
  this._meta = _meta;
  this.cljs$lang$protocol_mask$partition0$ = 2154168321;
  this.cljs$lang$protocol_mask$partition1$ = 4096;
};
cljs.core.Symbol.prototype.toString = function() {
  var self__ = this;
  var _ = this;
  return self__.str;
};
cljs.core.Symbol.prototype.equiv = function(other) {
  var self__ = this;
  var this$ = this;
  return this$.cljs$core$IEquiv$_equiv$arity$2(null, other);
};
cljs.core.Symbol.prototype.cljs$core$IEquiv$_equiv$arity$2 = function(_, other) {
  var self__ = this;
  var ___$1 = this;
  if (other instanceof cljs.core.Symbol) {
    return self__.str === other.str;
  } else {
    return false;
  }
};
cljs.core.Symbol.prototype.call = function() {
  var G__9701 = null;
  var G__9701__2 = function(self__, coll) {
    var self__ = this;
    var self____$1 = this;
    var sym = self____$1;
    return cljs.core.get.cljs$core$IFn$_invoke$arity$2 ? cljs.core.get.cljs$core$IFn$_invoke$arity$2(coll, sym) : cljs.core.get.call(null, coll, sym);
  };
  var G__9701__3 = function(self__, coll, not_found) {
    var self__ = this;
    var self____$1 = this;
    var sym = self____$1;
    return cljs.core.get.cljs$core$IFn$_invoke$arity$3 ? cljs.core.get.cljs$core$IFn$_invoke$arity$3(coll, sym, not_found) : cljs.core.get.call(null, coll, sym, not_found);
  };
  G__9701 = function(self__, coll, not_found) {
    switch(arguments.length) {
      case 2:
        return G__9701__2.call(this, self__, coll);
      case 3:
        return G__9701__3.call(this, self__, coll, not_found);
    }
    throw new Error("Invalid arity: " + (arguments.length - 1));
  };
  G__9701.cljs$core$IFn$_invoke$arity$2 = G__9701__2;
  G__9701.cljs$core$IFn$_invoke$arity$3 = G__9701__3;
  return G__9701;
}();
cljs.core.Symbol.prototype.apply = function(self__, args9700) {
  var self__ = this;
  var self____$1 = this;
  return self____$1.call.apply(self____$1, [self____$1].concat(cljs.core.aclone(args9700)));
};
cljs.core.Symbol.prototype.cljs$core$IFn$_invoke$arity$1 = function(coll) {
  var self__ = this;
  var sym = this;
  return cljs.core.get.cljs$core$IFn$_invoke$arity$2 ? cljs.core.get.cljs$core$IFn$_invoke$arity$2(coll, sym) : cljs.core.get.call(null, coll, sym);
};
cljs.core.Symbol.prototype.cljs$core$IFn$_invoke$arity$2 = function(coll, not_found) {
  var self__ = this;
  var sym = this;
  return cljs.core.get.cljs$core$IFn$_invoke$arity$3 ? cljs.core.get.cljs$core$IFn$_invoke$arity$3(coll, sym, not_found) : cljs.core.get.call(null, coll, sym, not_found);
};
cljs.core.Symbol.prototype.cljs$core$IMeta$_meta$arity$1 = function(_) {
  var self__ = this;
  var ___$1 = this;
  return self__._meta;
};
cljs.core.Symbol.prototype.cljs$core$IWithMeta$_with_meta$arity$2 = function(_, new_meta) {
  var self__ = this;
  var ___$1 = this;
  return new cljs.core.Symbol(self__.ns, self__.name, self__.str, self__._hash, new_meta);
};
cljs.core.Symbol.prototype.cljs$core$IHash$_hash$arity$1 = function(sym) {
  var self__ = this;
  var sym__$1 = this;
  var h__7700__auto__ = self__._hash;
  if (!(h__7700__auto__ == null)) {
    return h__7700__auto__;
  } else {
    var h__7700__auto____$1 = cljs.core.hash_symbol(sym__$1);
    self__._hash = h__7700__auto____$1;
    return h__7700__auto____$1;
  }
};
cljs.core.Symbol.prototype.cljs$core$INamed$_name$arity$1 = function(_) {
  var self__ = this;
  var ___$1 = this;
  return self__.name;
};
cljs.core.Symbol.prototype.cljs$core$INamed$_namespace$arity$1 = function(_) {
  var self__ = this;
  var ___$1 = this;
  return self__.ns;
};
cljs.core.Symbol.prototype.cljs$core$IPrintWithWriter$_pr_writer$arity$3 = function(o, writer, _) {
  var self__ = this;
  var o__$1 = this;
  return cljs.core._write(writer, self__.str);
};
cljs.core.Symbol.getBasis = function() {
  return new cljs.core.PersistentVector(null, 5, 5, cljs.core.PersistentVector.EMPTY_NODE, [new cljs.core.Symbol(null, "ns", "ns", 2082130287, null), new cljs.core.Symbol(null, "name", "name", -810760592, null), new cljs.core.Symbol(null, "str", "str", -1564826950, null), cljs.core.with_meta(new cljs.core.Symbol(null, "_hash", "_hash", -2130838312, null), new cljs.core.PersistentArrayMap(null, 1, [new cljs.core.Keyword(null, "mutable", "mutable", 875778266), true], null)), new cljs.core.Symbol(null, 
  "_meta", "_meta", -1716892533, null)], null);
};
cljs.core.Symbol.cljs$lang$type = true;
cljs.core.Symbol.cljs$lang$ctorStr = "cljs.core/Symbol";
cljs.core.Symbol.cljs$lang$ctorPrWriter = function(this__7871__auto__, writer__7872__auto__, opt__7873__auto__) {
  return cljs.core._write(writer__7872__auto__, "cljs.core/Symbol");
};
cljs.core.__GT_Symbol = function cljs$core$__GT_Symbol(ns, name, str, _hash, _meta) {
  return new cljs.core.Symbol(ns, name, str, _hash, _meta);
};
cljs.core.symbol = function cljs$core$symbol(var_args) {
  var args9702 = [];
  var len__8373__auto___9705 = arguments.length;
  var i__8374__auto___9706 = 0;
  while (true) {
    if (i__8374__auto___9706 < len__8373__auto___9705) {
      args9702.push(arguments[i__8374__auto___9706]);
      var G__9707 = i__8374__auto___9706 + 1;
      i__8374__auto___9706 = G__9707;
      continue;
    } else {
    }
    break;
  }
  var G__9704 = args9702.length;
  switch(G__9704) {
    case 1:
      return cljs.core.symbol.cljs$core$IFn$_invoke$arity$1(arguments[0]);
      break;
    case 2:
      return cljs.core.symbol.cljs$core$IFn$_invoke$arity$2(arguments[0], arguments[1]);
      break;
    default:
      throw new Error([cljs.core.str.cljs$core$IFn$_invoke$arity$1("Invalid arity: "), cljs.core.str.cljs$core$IFn$_invoke$arity$1(args9702.length)].join(""));
  }
};
cljs.core.symbol.cljs$core$IFn$_invoke$arity$1 = function(name) {
  if (name instanceof cljs.core.Symbol) {
    return name;
  } else {
    var idx = name.indexOf("/");
    if (idx < 1) {
      return cljs.core.symbol.cljs$core$IFn$_invoke$arity$2(null, name);
    } else {
      return cljs.core.symbol.cljs$core$IFn$_invoke$arity$2(name.substring(0, idx), name.substring(idx + 1, name.length));
    }
  }
};
cljs.core.symbol.cljs$core$IFn$_invoke$arity$2 = function(ns, name) {
  var sym_str = !(ns == null) ? [cljs.core.str.cljs$core$IFn$_invoke$arity$1(ns), cljs.core.str.cljs$core$IFn$_invoke$arity$1("/"), cljs.core.str.cljs$core$IFn$_invoke$arity$1(name)].join("") : name;
  return new cljs.core.Symbol(ns, name, sym_str, null, null);
};
cljs.core.symbol.cljs$lang$maxFixedArity = 2;
cljs.core.Var = function(val, sym, _meta) {
  this.val = val;
  this.sym = sym;
  this._meta = _meta;
  this.cljs$lang$protocol_mask$partition0$ = 6717441;
  this.cljs$lang$protocol_mask$partition1$ = 0;
};
cljs.core.Var.prototype.isMacro = function() {
  var self__ = this;
  var _ = this;
  return (self__.val.cljs$core$IFn$_invoke$arity$0 ? self__.val.cljs$core$IFn$_invoke$arity$0() : self__.val.call(null)).cljs$lang$macro;
};
cljs.core.Var.prototype.toString = function() {
  var self__ = this;
  var _ = this;
  return [cljs.core.str.cljs$core$IFn$_invoke$arity$1("#'"), cljs.core.str.cljs$core$IFn$_invoke$arity$1(self__.sym)].join("");
};
cljs.core.Var.prototype.cljs$core$IDeref$_deref$arity$1 = function(_) {
  var self__ = this;
  var ___$1 = this;
  return self__.val.cljs$core$IFn$_invoke$arity$0 ? self__.val.cljs$core$IFn$_invoke$arity$0() : self__.val.call(null);
};
cljs.core.Var.prototype.cljs$core$IMeta$_meta$arity$1 = function(_) {
  var self__ = this;
  var ___$1 = this;
  return self__._meta;
};
cljs.core.Var.prototype.cljs$core$IWithMeta$_with_meta$arity$2 = function(_, new_meta) {
  var self__ = this;
  var ___$1 = this;
  return new cljs.core.Var(self__.val, self__.sym, new_meta);
};
cljs.core.Var.prototype.cljs$core$IEquiv$_equiv$arity$2 = function(this$, other) {
  var self__ = this;
  var this$__$1 = this;
  if (other instanceof cljs.core.Var) {
    var G__9710 = this$__$1.sym;
    var G__9711 = other.sym;
    return cljs.core._EQ_.cljs$core$IFn$_invoke$arity$2 ? cljs.core._EQ_.cljs$core$IFn$_invoke$arity$2(G__9710, G__9711) : cljs.core._EQ_.call(null, G__9710, G__9711);
  } else {
    return false;
  }
};
cljs.core.Var.prototype.cljs$core$IHash$_hash$arity$1 = function(_) {
  var self__ = this;
  var ___$1 = this;
  return cljs.core.hash_symbol(self__.sym);
};
cljs.core.Var.prototype.cljs$core$Fn$ = cljs.core.PROTOCOL_SENTINEL;
cljs.core.Var.prototype.call = function() {
  var G__9756 = null;
  var G__9756__1 = function(self__) {
    var self__ = this;
    var self____$1 = this;
    var _ = self____$1;
    return (self__.val.cljs$core$IFn$_invoke$arity$0 ? self__.val.cljs$core$IFn$_invoke$arity$0() : self__.val.call(null)).call(null);
  };
  var G__9756__2 = function(self__, a) {
    var self__ = this;
    var self____$1 = this;
    var _ = self____$1;
    return (self__.val.cljs$core$IFn$_invoke$arity$0 ? self__.val.cljs$core$IFn$_invoke$arity$0() : self__.val.call(null)).call(null, a);
  };
  var G__9756__3 = function(self__, a, b) {
    var self__ = this;
    var self____$1 = this;
    var _ = self____$1;
    return (self__.val.cljs$core$IFn$_invoke$arity$0 ? self__.val.cljs$core$IFn$_invoke$arity$0() : self__.val.call(null)).call(null, a, b);
  };
  var G__9756__4 = function(self__, a, b, c) {
    var self__ = this;
    var self____$1 = this;
    var _ = self____$1;
    return (self__.val.cljs$core$IFn$_invoke$arity$0 ? self__.val.cljs$core$IFn$_invoke$arity$0() : self__.val.call(null)).call(null, a, b, c);
  };
  var G__9756__5 = function(self__, a, b, c, d) {
    var self__ = this;
    var self____$1 = this;
    var _ = self____$1;
    return (self__.val.cljs$core$IFn$_invoke$arity$0 ? self__.val.cljs$core$IFn$_invoke$arity$0() : self__.val.call(null)).call(null, a, b, c, d);
  };
  var G__9756__6 = function(self__, a, b, c, d, e) {
    var self__ = this;
    var self____$1 = this;
    var _ = self____$1;
    return (self__.val.cljs$core$IFn$_invoke$arity$0 ? self__.val.cljs$core$IFn$_invoke$arity$0() : self__.val.call(null)).call(null, a, b, c, d, e);
  };
  var G__9756__7 = function(self__, a, b, c, d, e, f) {
    var self__ = this;
    var self____$1 = this;
    var _ = self____$1;
    return (self__.val.cljs$core$IFn$_invoke$arity$0 ? self__.val.cljs$core$IFn$_invoke$arity$0() : self__.val.call(null)).call(null, a, b, c, d, e, f);
  };
  var G__9756__8 = function(self__, a, b, c, d, e, f, g) {
    var self__ = this;
    var self____$1 = this;
    var _ = self____$1;
    return (self__.val.cljs$core$IFn$_invoke$arity$0 ? self__.val.cljs$core$IFn$_invoke$arity$0() : self__.val.call(null)).call(null, a, b, c, d, e, f, g);
  };
  var G__9756__9 = function(self__, a, b, c, d, e, f, g, h) {
    var self__ = this;
    var self____$1 = this;
    var _ = self____$1;
    return (self__.val.cljs$core$IFn$_invoke$arity$0 ? self__.val.cljs$core$IFn$_invoke$arity$0() : self__.val.call(null)).call(null, a, b, c, d, e, f, g, h);
  };
  var G__9756__10 = function(self__, a, b, c, d, e, f, g, h, i) {
    var self__ = this;
    var self____$1 = this;
    var _ = self____$1;
    return (self__.val.cljs$core$IFn$_invoke$arity$0 ? self__.val.cljs$core$IFn$_invoke$arity$0() : self__.val.call(null)).call(null, a, b, c, d, e, f, g, h, i);
  };
  var G__9756__11 = function(self__, a, b, c, d, e, f, g, h, i, j) {
    var self__ = this;
    var self____$1 = this;
    var _ = self____$1;
    return (self__.val.cljs$core$IFn$_invoke$arity$0 ? self__.val.cljs$core$IFn$_invoke$arity$0() : self__.val.call(null)).call(null, a, b, c, d, e, f, g, h, i, j);
  };
  var G__9756__12 = function(self__, a, b, c, d, e, f, g, h, i, j, k) {
    var self__ = this;
    var self____$1 = this;
    var _ = self____$1;
    return (self__.val.cljs$core$IFn$_invoke$arity$0 ? self__.val.cljs$core$IFn$_invoke$arity$0() : self__.val.call(null)).call(null, a, b, c, d, e, f, g, h, i, j, k);
  };
  var G__9756__13 = function(self__, a, b, c, d, e, f, g, h, i, j, k, l) {
    var self__ = this;
    var self____$1 = this;
    var _ = self____$1;
    return (self__.val.cljs$core$IFn$_invoke$arity$0 ? self__.val.cljs$core$IFn$_invoke$arity$0() : self__.val.call(null)).call(null, a, b, c, d, e, f, g, h, i, j, k, l);
  };
  var G__9756__14 = function(self__, a, b, c, d, e, f, g, h, i, j, k, l, m) {
    var self__ = this;
    var self____$1 = this;
    var _ = self____$1;
    return (self__.val.cljs$core$IFn$_invoke$arity$0 ? self__.val.cljs$core$IFn$_invoke$arity$0() : self__.val.call(null)).call(null, a, b, c, d, e, f, g, h, i, j, k, l, m);
  };
  var G__9756__15 = function(self__, a, b, c, d, e, f, g, h, i, j, k, l, m, n) {
    var self__ = this;
    var self____$1 = this;
    var _ = self____$1;
    return (self__.val.cljs$core$IFn$_invoke$arity$0 ? self__.val.cljs$core$IFn$_invoke$arity$0() : self__.val.call(null)).call(null, a, b, c, d, e, f, g, h, i, j, k, l, m, n);
  };
  var G__9756__16 = function(self__, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o) {
    var self__ = this;
    var self____$1 = this;
    var _ = self____$1;
    return (self__.val.cljs$core$IFn$_invoke$arity$0 ? self__.val.cljs$core$IFn$_invoke$arity$0() : self__.val.call(null)).call(null, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o);
  };
  var G__9756__17 = function(self__, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p) {
    var self__ = this;
    var self____$1 = this;
    var _ = self____$1;
    return (self__.val.cljs$core$IFn$_invoke$arity$0 ? self__.val.cljs$core$IFn$_invoke$arity$0() : self__.val.call(null)).call(null, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p);
  };
  var G__9756__18 = function(self__, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q) {
    var self__ = this;
    var self____$1 = this;
    var _ = self____$1;
    return (self__.val.cljs$core$IFn$_invoke$arity$0 ? self__.val.cljs$core$IFn$_invoke$arity$0() : self__.val.call(null)).call(null, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q);
  };
  var G__9756__19 = function(self__, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r) {
    var self__ = this;
    var self____$1 = this;
    var _ = self____$1;
    return (self__.val.cljs$core$IFn$_invoke$arity$0 ? self__.val.cljs$core$IFn$_invoke$arity$0() : self__.val.call(null)).call(null, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r);
  };
  var G__9756__20 = function(self__, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s) {
    var self__ = this;
    var self____$1 = this;
    var _ = self____$1;
    return (self__.val.cljs$core$IFn$_invoke$arity$0 ? self__.val.cljs$core$IFn$_invoke$arity$0() : self__.val.call(null)).call(null, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s);
  };
  var G__9756__21 = function(self__, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s, t) {
    var self__ = this;
    var self____$1 = this;
    var _ = self____$1;
    return (self__.val.cljs$core$IFn$_invoke$arity$0 ? self__.val.cljs$core$IFn$_invoke$arity$0() : self__.val.call(null)).call(null, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s, t);
  };
  var G__9756__22 = function(self__, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s, t, rest) {
    var self__ = this;
    var self____$1 = this;
    var _ = self____$1;
    var G__9712 = self__.val.cljs$core$IFn$_invoke$arity$0 ? self__.val.cljs$core$IFn$_invoke$arity$0() : self__.val.call(null);
    var G__9713 = a;
    var G__9714 = b;
    var G__9715 = c;
    var G__9716 = d;
    var G__9717 = e;
    var G__9718 = f;
    var G__9719 = g;
    var G__9720 = h;
    var G__9721 = i;
    var G__9722 = j;
    var G__9723 = k;
    var G__9724 = l;
    var G__9725 = m;
    var G__9726 = n;
    var G__9727 = o;
    var G__9728 = p;
    var G__9729 = q;
    var G__9730 = r;
    var G__9731 = s;
    var G__9732 = t;
    var G__9733 = rest;
    return cljs.core.apply.cljs$core$IFn$_invoke$arity$22 ? cljs.core.apply.cljs$core$IFn$_invoke$arity$22(G__9712, G__9713, G__9714, G__9715, G__9716, G__9717, G__9718, G__9719, G__9720, G__9721, G__9722, G__9723, G__9724, G__9725, G__9726, G__9727, G__9728, G__9729, G__9730, G__9731, G__9732, G__9733) : cljs.core.apply.call(null, G__9712, G__9713, G__9714, G__9715, G__9716, G__9717, G__9718, G__9719, G__9720, G__9721, G__9722, G__9723, G__9724, G__9725, G__9726, G__9727, G__9728, G__9729, G__9730, 
    G__9731, G__9732, G__9733);
  };
  G__9756 = function(self__, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s, t, rest) {
    switch(arguments.length) {
      case 1:
        return G__9756__1.call(this, self__);
      case 2:
        return G__9756__2.call(this, self__, a);
      case 3:
        return G__9756__3.call(this, self__, a, b);
      case 4:
        return G__9756__4.call(this, self__, a, b, c);
      case 5:
        return G__9756__5.call(this, self__, a, b, c, d);
      case 6:
        return G__9756__6.call(this, self__, a, b, c, d, e);
      case 7:
        return G__9756__7.call(this, self__, a, b, c, d, e, f);
      case 8:
        return G__9756__8.call(this, self__, a, b, c, d, e, f, g);
      case 9:
        return G__9756__9.call(this, self__, a, b, c, d, e, f, g, h);
      case 10:
        return G__9756__10.call(this, self__, a, b, c, d, e, f, g, h, i);
      case 11:
        return G__9756__11.call(this, self__, a, b, c, d, e, f, g, h, i, j);
      case 12:
        return G__9756__12.call(this, self__, a, b, c, d, e, f, g, h, i, j, k);
      case 13:
        return G__9756__13.call(this, self__, a, b, c, d, e, f, g, h, i, j, k, l);
      case 14:
        return G__9756__14.call(this, self__, a, b, c, d, e, f, g, h, i, j, k, l, m);
      case 15:
        return G__9756__15.call(this, self__, a, b, c, d, e, f, g, h, i, j, k, l, m, n);
      case 16:
        return G__9756__16.call(this, self__, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o);
      case 17:
        return G__9756__17.call(this, self__, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p);
      case 18:
        return G__9756__18.call(this, self__, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q);
      case 19:
        return G__9756__19.call(this, self__, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r);
      case 20:
        return G__9756__20.call(this, self__, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s);
      case 21:
        return G__9756__21.call(this, self__, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s, t);
      case 22:
        return G__9756__22.call(this, self__, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s, t, rest);
    }
    throw new Error("Invalid arity: " + (arguments.length - 1));
  };
  G__9756.cljs$core$IFn$_invoke$arity$1 = G__9756__1;
  G__9756.cljs$core$IFn$_invoke$arity$2 = G__9756__2;
  G__9756.cljs$core$IFn$_invoke$arity$3 = G__9756__3;
  G__9756.cljs$core$IFn$_invoke$arity$4 = G__9756__4;
  G__9756.cljs$core$IFn$_invoke$arity$5 = G__9756__5;
  G__9756.cljs$core$IFn$_invoke$arity$6 = G__9756__6;
  G__9756.cljs$core$IFn$_invoke$arity$7 = G__9756__7;
  G__9756.cljs$core$IFn$_invoke$arity$8 = G__9756__8;
  G__9756.cljs$core$IFn$_invoke$arity$9 = G__9756__9;
  G__9756.cljs$core$IFn$_invoke$arity$10 = G__9756__10;
  G__9756.cljs$core$IFn$_invoke$arity$11 = G__9756__11;
  G__9756.cljs$core$IFn$_invoke$arity$12 = G__9756__12;
  G__9756.cljs$core$IFn$_invoke$arity$13 = G__9756__13;
  G__9756.cljs$core$IFn$_invoke$arity$14 = G__9756__14;
  G__9756.cljs$core$IFn$_invoke$arity$15 = G__9756__15;
  G__9756.cljs$core$IFn$_invoke$arity$16 = G__9756__16;
  G__9756.cljs$core$IFn$_invoke$arity$17 = G__9756__17;
  G__9756.cljs$core$IFn$_invoke$arity$18 = G__9756__18;
  G__9756.cljs$core$IFn$_invoke$arity$19 = G__9756__19;
  G__9756.cljs$core$IFn$_invoke$arity$20 = G__9756__20;
  G__9756.cljs$core$IFn$_invoke$arity$21 = G__9756__21;
  G__9756.cljs$core$IFn$_invoke$arity$22 = G__9756__22;
  return G__9756;
}();
cljs.core.Var.prototype.apply = function(self__, args9709) {
  var self__ = this;
  var self____$1 = this;
  return self____$1.call.apply(self____$1, [self____$1].concat(cljs.core.aclone(args9709)));
};
cljs.core.Var.prototype.cljs$core$IFn$_invoke$arity$0 = function() {
  var self__ = this;
  var _ = this;
  return (self__.val.cljs$core$IFn$_invoke$arity$0 ? self__.val.cljs$core$IFn$_invoke$arity$0() : self__.val.call(null)).call(null);
};
cljs.core.Var.prototype.cljs$core$IFn$_invoke$arity$1 = function(a) {
  var self__ = this;
  var _ = this;
  return (self__.val.cljs$core$IFn$_invoke$arity$0 ? self__.val.cljs$core$IFn$_invoke$arity$0() : self__.val.call(null)).call(null, a);
};
cljs.core.Var.prototype.cljs$core$IFn$_invoke$arity$2 = function(a, b) {
  var self__ = this;
  var _ = this;
  return (self__.val.cljs$core$IFn$_invoke$arity$0 ? self__.val.cljs$core$IFn$_invoke$arity$0() : self__.val.call(null)).call(null, a, b);
};
cljs.core.Var.prototype.cljs$core$IFn$_invoke$arity$3 = function(a, b, c) {
  var self__ = this;
  var _ = this;
  return (self__.val.cljs$core$IFn$_invoke$arity$0 ? self__.val.cljs$core$IFn$_invoke$arity$0() : self__.val.call(null)).call(null, a, b, c);
};
cljs.core.Var.prototype.cljs$core$IFn$_invoke$arity$4 = function(a, b, c, d) {
  var self__ = this;
  var _ = this;
  return (self__.val.cljs$core$IFn$_invoke$arity$0 ? self__.val.cljs$core$IFn$_invoke$arity$0() : self__.val.call(null)).call(null, a, b, c, d);
};
cljs.core.Var.prototype.cljs$core$IFn$_invoke$arity$5 = function(a, b, c, d, e) {
  var self__ = this;
  var _ = this;
  return (self__.val.cljs$core$IFn$_invoke$arity$0 ? self__.val.cljs$core$IFn$_invoke$arity$0() : self__.val.call(null)).call(null, a, b, c, d, e);
};
cljs.core.Var.prototype.cljs$core$IFn$_invoke$arity$6 = function(a, b, c, d, e, f) {
  var self__ = this;
  var _ = this;
  return (self__.val.cljs$core$IFn$_invoke$arity$0 ? self__.val.cljs$core$IFn$_invoke$arity$0() : self__.val.call(null)).call(null, a, b, c, d, e, f);
};
cljs.core.Var.prototype.cljs$core$IFn$_invoke$arity$7 = function(a, b, c, d, e, f, g) {
  var self__ = this;
  var _ = this;
  return (self__.val.cljs$core$IFn$_invoke$arity$0 ? self__.val.cljs$core$IFn$_invoke$arity$0() : self__.val.call(null)).call(null, a, b, c, d, e, f, g);
};
cljs.core.Var.prototype.cljs$core$IFn$_invoke$arity$8 = function(a, b, c, d, e, f, g, h) {
  var self__ = this;
  var _ = this;
  return (self__.val.cljs$core$IFn$_invoke$arity$0 ? self__.val.cljs$core$IFn$_invoke$arity$0() : self__.val.call(null)).call(null, a, b, c, d, e, f, g, h);
};
cljs.core.Var.prototype.cljs$core$IFn$_invoke$arity$9 = function(a, b, c, d, e, f, g, h, i) {
  var self__ = this;
  var _ = this;
  return (self__.val.cljs$core$IFn$_invoke$arity$0 ? self__.val.cljs$core$IFn$_invoke$arity$0() : self__.val.call(null)).call(null, a, b, c, d, e, f, g, h, i);
};
cljs.core.Var.prototype.cljs$core$IFn$_invoke$arity$10 = function(a, b, c, d, e, f, g, h, i, j) {
  var self__ = this;
  var _ = this;
  return (self__.val.cljs$core$IFn$_invoke$arity$0 ? self__.val.cljs$core$IFn$_invoke$arity$0() : self__.val.call(null)).call(null, a, b, c, d, e, f, g, h, i, j);
};
cljs.core.Var.prototype.cljs$core$IFn$_invoke$arity$11 = function(a, b, c, d, e, f, g, h, i, j, k) {
  var self__ = this;
  var _ = this;
  return (self__.val.cljs$core$IFn$_invoke$arity$0 ? self__.val.cljs$core$IFn$_invoke$arity$0() : self__.val.call(null)).call(null, a, b, c, d, e, f, g, h, i, j, k);
};
cljs.core.Var.prototype.cljs$core$IFn$_invoke$arity$12 = function(a, b, c, d, e, f, g, h, i, j, k, l) {
  var self__ = this;
  var _ = this;
  return (self__.val.cljs$core$IFn$_invoke$arity$0 ? self__.val.cljs$core$IFn$_invoke$arity$0() : self__.val.call(null)).call(null, a, b, c, d, e, f, g, h, i, j, k, l);
};
cljs.core.Var.prototype.cljs$core$IFn$_invoke$arity$13 = function(a, b, c, d, e, f, g, h, i, j, k, l, m) {
  var self__ = this;
  var _ = this;
  return (self__.val.cljs$core$IFn$_invoke$arity$0 ? self__.val.cljs$core$IFn$_invoke$arity$0() : self__.val.call(null)).call(null, a, b, c, d, e, f, g, h, i, j, k, l, m);
};
cljs.core.Var.prototype.cljs$core$IFn$_invoke$arity$14 = function(a, b, c, d, e, f, g, h, i, j, k, l, m, n) {
  var self__ = this;
  var _ = this;
  return (self__.val.cljs$core$IFn$_invoke$arity$0 ? self__.val.cljs$core$IFn$_invoke$arity$0() : self__.val.call(null)).call(null, a, b, c, d, e, f, g, h, i, j, k, l, m, n);
};
cljs.core.Var.prototype.cljs$core$IFn$_invoke$arity$15 = function(a, b, c, d, e, f, g, h, i, j, k, l, m, n, o) {
  var self__ = this;
  var _ = this;
  return (self__.val.cljs$core$IFn$_invoke$arity$0 ? self__.val.cljs$core$IFn$_invoke$arity$0() : self__.val.call(null)).call(null, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o);
};
cljs.core.Var.prototype.cljs$core$IFn$_invoke$arity$16 = function(a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p) {
  var self__ = this;
  var _ = this;
  return (self__.val.cljs$core$IFn$_invoke$arity$0 ? self__.val.cljs$core$IFn$_invoke$arity$0() : self__.val.call(null)).call(null, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p);
};
cljs.core.Var.prototype.cljs$core$IFn$_invoke$arity$17 = function(a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q) {
  var self__ = this;
  var _ = this;
  return (self__.val.cljs$core$IFn$_invoke$arity$0 ? self__.val.cljs$core$IFn$_invoke$arity$0() : self__.val.call(null)).call(null, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q);
};
cljs.core.Var.prototype.cljs$core$IFn$_invoke$arity$18 = function(a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r) {
  var self__ = this;
  var _ = this;
  return (self__.val.cljs$core$IFn$_invoke$arity$0 ? self__.val.cljs$core$IFn$_invoke$arity$0() : self__.val.call(null)).call(null, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r);
};
cljs.core.Var.prototype.cljs$core$IFn$_invoke$arity$19 = function(a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s) {
  var self__ = this;
  var _ = this;
  return (self__.val.cljs$core$IFn$_invoke$arity$0 ? self__.val.cljs$core$IFn$_invoke$arity$0() : self__.val.call(null)).call(null, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s);
};
cljs.core.Var.prototype.cljs$core$IFn$_invoke$arity$20 = function(a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s, t) {
  var self__ = this;
  var _ = this;
  return (self__.val.cljs$core$IFn$_invoke$arity$0 ? self__.val.cljs$core$IFn$_invoke$arity$0() : self__.val.call(null)).call(null, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s, t);
};
cljs.core.Var.prototype.cljs$core$IFn$_invoke$arity$21 = function(a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s, t, rest) {
  var self__ = this;
  var _ = this;
  var G__9734 = self__.val.cljs$core$IFn$_invoke$arity$0 ? self__.val.cljs$core$IFn$_invoke$arity$0() : self__.val.call(null);
  var G__9735 = a;
  var G__9736 = b;
  var G__9737 = c;
  var G__9738 = d;
  var G__9739 = e;
  var G__9740 = f;
  var G__9741 = g;
  var G__9742 = h;
  var G__9743 = i;
  var G__9744 = j;
  var G__9745 = k;
  var G__9746 = l;
  var G__9747 = m;
  var G__9748 = n;
  var G__9749 = o;
  var G__9750 = p;
  var G__9751 = q;
  var G__9752 = r;
  var G__9753 = s;
  var G__9754 = t;
  var G__9755 = rest;
  return cljs.core.apply.cljs$core$IFn$_invoke$arity$22 ? cljs.core.apply.cljs$core$IFn$_invoke$arity$22(G__9734, G__9735, G__9736, G__9737, G__9738, G__9739, G__9740, G__9741, G__9742, G__9743, G__9744, G__9745, G__9746, G__9747, G__9748, G__9749, G__9750, G__9751, G__9752, G__9753, G__9754, G__9755) : cljs.core.apply.call(null, G__9734, G__9735, G__9736, G__9737, G__9738, G__9739, G__9740, G__9741, G__9742, G__9743, G__9744, G__9745, G__9746, G__9747, G__9748, G__9749, G__9750, G__9751, G__9752, 
  G__9753, G__9754, G__9755);
};
cljs.core.Var.getBasis = function() {
  return new cljs.core.PersistentVector(null, 3, 5, cljs.core.PersistentVector.EMPTY_NODE, [new cljs.core.Symbol(null, "val", "val", 1769233139, null), new cljs.core.Symbol(null, "sym", "sym", 195671222, null), new cljs.core.Symbol(null, "_meta", "_meta", -1716892533, null)], null);
};
cljs.core.Var.cljs$lang$type = true;
cljs.core.Var.cljs$lang$ctorStr = "cljs.core/Var";
cljs.core.Var.cljs$lang$ctorPrWriter = function(this__7871__auto__, writer__7872__auto__, opt__7873__auto__) {
  return cljs.core._write(writer__7872__auto__, "cljs.core/Var");
};
cljs.core.__GT_Var = function cljs$core$__GT_Var(val, sym, _meta) {
  return new cljs.core.Var(val, sym, _meta);
};
cljs.core.var_QMARK_ = function cljs$core$var_QMARK_(v) {
  return v instanceof cljs.core.Var;
};
cljs.core.iterable_QMARK_ = function cljs$core$iterable_QMARK_(x) {
  if (!(x == null)) {
    if (false || cljs.core.PROTOCOL_SENTINEL === x.cljs$core$IIterable$) {
      return true;
    } else {
      if (!x.cljs$lang$protocol_mask$partition$) {
        return cljs.core.native_satisfies_QMARK_(cljs.core.IIterable, x);
      } else {
        return false;
      }
    }
  } else {
    return cljs.core.native_satisfies_QMARK_(cljs.core.IIterable, x);
  }
};
cljs.core.clone = function cljs$core$clone(value) {
  return cljs.core._clone(value);
};
cljs.core.cloneable_QMARK_ = function cljs$core$cloneable_QMARK_(value) {
  if (!(value == null)) {
    if (value.cljs$lang$protocol_mask$partition1$ & 8192 || cljs.core.PROTOCOL_SENTINEL === value.cljs$core$ICloneable$) {
      return true;
    } else {
      if (!value.cljs$lang$protocol_mask$partition1$) {
        return cljs.core.native_satisfies_QMARK_(cljs.core.ICloneable, value);
      } else {
        return false;
      }
    }
  } else {
    return cljs.core.native_satisfies_QMARK_(cljs.core.ICloneable, value);
  }
};
cljs.core.seq = function cljs$core$seq(coll) {
  if (coll == null) {
    return null;
  } else {
    if (!(coll == null) ? coll.cljs$lang$protocol_mask$partition0$ & 8388608 || cljs.core.PROTOCOL_SENTINEL === coll.cljs$core$ISeqable$ ? true : false : false) {
      return coll.cljs$core$ISeqable$_seq$arity$1(null);
    } else {
      if (cljs.core.array_QMARK_(coll)) {
        if (coll.length === 0) {
          return null;
        } else {
          return new cljs.core.IndexedSeq(coll, 0, null);
        }
      } else {
        if (typeof coll === "string") {
          if (coll.length === 0) {
            return null;
          } else {
            return new cljs.core.IndexedSeq(coll, 0, null);
          }
        } else {
          if (cljs.core.native_satisfies_QMARK_(cljs.core.ISeqable, coll)) {
            return cljs.core._seq(coll);
          } else {
            throw new Error([cljs.core.str.cljs$core$IFn$_invoke$arity$1(coll), cljs.core.str.cljs$core$IFn$_invoke$arity$1(" is not ISeqable")].join(""));
          }
        }
      }
    }
  }
};
cljs.core.first = function cljs$core$first(coll) {
  if (coll == null) {
    return null;
  } else {
    if (!(coll == null) ? coll.cljs$lang$protocol_mask$partition0$ & 64 || cljs.core.PROTOCOL_SENTINEL === coll.cljs$core$ISeq$ ? true : false : false) {
      return coll.cljs$core$ISeq$_first$arity$1(null);
    } else {
      var s = cljs.core.seq(coll);
      if (s == null) {
        return null;
      } else {
        return cljs.core._first(s);
      }
    }
  }
};
cljs.core.rest = function cljs$core$rest(coll) {
  if (!(coll == null)) {
    if (!(coll == null) ? coll.cljs$lang$protocol_mask$partition0$ & 64 || cljs.core.PROTOCOL_SENTINEL === coll.cljs$core$ISeq$ ? true : false : false) {
      return coll.cljs$core$ISeq$_rest$arity$1(null);
    } else {
      var s = cljs.core.seq(coll);
      if (s) {
        return cljs.core._rest(s);
      } else {
        return cljs.core.List.EMPTY;
      }
    }
  } else {
    return cljs.core.List.EMPTY;
  }
};
cljs.core.next = function cljs$core$next(coll) {
  if (coll == null) {
    return null;
  } else {
    if (!(coll == null) ? coll.cljs$lang$protocol_mask$partition0$ & 128 || cljs.core.PROTOCOL_SENTINEL === coll.cljs$core$INext$ ? true : false : false) {
      return coll.cljs$core$INext$_next$arity$1(null);
    } else {
      return cljs.core.seq(cljs.core.rest(coll));
    }
  }
};
cljs.core._EQ_ = function cljs$core$_EQ_(var_args) {
  var args9769 = [];
  var len__8373__auto___9775 = arguments.length;
  var i__8374__auto___9776 = 0;
  while (true) {
    if (i__8374__auto___9776 < len__8373__auto___9775) {
      args9769.push(arguments[i__8374__auto___9776]);
      var G__9777 = i__8374__auto___9776 + 1;
      i__8374__auto___9776 = G__9777;
      continue;
    } else {
    }
    break;
  }
  var G__9774 = args9769.length;
  switch(G__9774) {
    case 1:
      return cljs.core._EQ_.cljs$core$IFn$_invoke$arity$1(arguments[0]);
      break;
    case 2:
      return cljs.core._EQ_.cljs$core$IFn$_invoke$arity$2(arguments[0], arguments[1]);
      break;
    default:
      var argseq__8392__auto__ = new cljs.core.IndexedSeq(args9769.slice(2), 0, null);
      return cljs.core._EQ_.cljs$core$IFn$_invoke$arity$variadic(arguments[0], arguments[1], argseq__8392__auto__);
  }
};
cljs.core._EQ_.cljs$core$IFn$_invoke$arity$1 = function(x) {
  return true;
};
cljs.core._EQ_.cljs$core$IFn$_invoke$arity$2 = function(x, y) {
  if (x == null) {
    return y == null;
  } else {
    return x === y || cljs.core._equiv(x, y);
  }
};
cljs.core._EQ_.cljs$core$IFn$_invoke$arity$variadic = function(x, y, more) {
  while (true) {
    if (cljs.core._EQ_.cljs$core$IFn$_invoke$arity$2(x, y)) {
      if (cljs.core.next(more)) {
        var G__9779 = y;
        var G__9780 = cljs.core.first(more);
        var G__9781 = cljs.core.next(more);
        x = G__9779;
        y = G__9780;
        more = G__9781;
        continue;
      } else {
        return cljs.core._EQ_.cljs$core$IFn$_invoke$arity$2(y, cljs.core.first(more));
      }
    } else {
      return false;
    }
    break;
  }
};
cljs.core._EQ_.cljs$lang$applyTo = function(seq9770) {
  var G__9771 = cljs.core.first(seq9770);
  var seq9770__$1 = cljs.core.next(seq9770);
  var G__9772 = cljs.core.first(seq9770__$1);
  var seq9770__$2 = cljs.core.next(seq9770__$1);
  return cljs.core._EQ_.cljs$core$IFn$_invoke$arity$variadic(G__9771, G__9772, seq9770__$2);
};
cljs.core._EQ_.cljs$lang$maxFixedArity = 2;
cljs.core.ES6Iterator = function(s) {
  this.s = s;
};
cljs.core.ES6Iterator.prototype.next = function() {
  var self__ = this;
  var _ = this;
  if (!(self__.s == null)) {
    var x = cljs.core.first(self__.s);
    self__.s = cljs.core.next(self__.s);
    return {"value":x, "done":false};
  } else {
    return {"value":null, "done":true};
  }
};
cljs.core.ES6Iterator.getBasis = function() {
  return new cljs.core.PersistentVector(null, 1, 5, cljs.core.PersistentVector.EMPTY_NODE, [cljs.core.with_meta(new cljs.core.Symbol(null, "s", "s", -948495851, null), new cljs.core.PersistentArrayMap(null, 1, [new cljs.core.Keyword(null, "mutable", "mutable", 875778266), true], null))], null);
};
cljs.core.ES6Iterator.cljs$lang$type = true;
cljs.core.ES6Iterator.cljs$lang$ctorStr = "cljs.core/ES6Iterator";
cljs.core.ES6Iterator.cljs$lang$ctorPrWriter = function(this__7871__auto__, writer__7872__auto__, opt__7873__auto__) {
  return cljs.core._write(writer__7872__auto__, "cljs.core/ES6Iterator");
};
cljs.core.__GT_ES6Iterator = function cljs$core$__GT_ES6Iterator(s) {
  return new cljs.core.ES6Iterator(s);
};
cljs.core.es6_iterator = function cljs$core$es6_iterator(coll) {
  return new cljs.core.ES6Iterator(cljs.core.seq(coll));
};
cljs.core.ES6IteratorSeq = function(value, iter, _rest) {
  this.value = value;
  this.iter = iter;
  this._rest = _rest;
  this.cljs$lang$protocol_mask$partition0$ = 8388672;
  this.cljs$lang$protocol_mask$partition1$ = 0;
};
cljs.core.ES6IteratorSeq.prototype.cljs$core$ISeqable$_seq$arity$1 = function(this$) {
  var self__ = this;
  var this$__$1 = this;
  return this$__$1;
};
cljs.core.ES6IteratorSeq.prototype.cljs$core$ISeq$_first$arity$1 = function(_) {
  var self__ = this;
  var ___$1 = this;
  return self__.value;
};
cljs.core.ES6IteratorSeq.prototype.cljs$core$ISeq$_rest$arity$1 = function(_) {
  var self__ = this;
  var ___$1 = this;
  if (self__._rest == null) {
    self__._rest = cljs.core.es6_iterator_seq.cljs$core$IFn$_invoke$arity$1 ? cljs.core.es6_iterator_seq.cljs$core$IFn$_invoke$arity$1(self__.iter) : cljs.core.es6_iterator_seq.call(null, self__.iter);
  } else {
  }
  return self__._rest;
};
cljs.core.ES6IteratorSeq.getBasis = function() {
  return new cljs.core.PersistentVector(null, 3, 5, cljs.core.PersistentVector.EMPTY_NODE, [new cljs.core.Symbol(null, "value", "value", 1946509744, null), new cljs.core.Symbol(null, "iter", "iter", -1346195486, null), cljs.core.with_meta(new cljs.core.Symbol(null, "_rest", "_rest", -2100466189, null), new cljs.core.PersistentArrayMap(null, 1, [new cljs.core.Keyword(null, "mutable", "mutable", 875778266), true], null))], null);
};
cljs.core.ES6IteratorSeq.cljs$lang$type = true;
cljs.core.ES6IteratorSeq.cljs$lang$ctorStr = "cljs.core/ES6IteratorSeq";
cljs.core.ES6IteratorSeq.cljs$lang$ctorPrWriter = function(this__7871__auto__, writer__7872__auto__, opt__7873__auto__) {
  return cljs.core._write(writer__7872__auto__, "cljs.core/ES6IteratorSeq");
};
cljs.core.__GT_ES6IteratorSeq = function cljs$core$__GT_ES6IteratorSeq(value, iter, _rest) {
  return new cljs.core.ES6IteratorSeq(value, iter, _rest);
};
cljs.core.es6_iterator_seq = function cljs$core$es6_iterator_seq(iter) {
  var v = iter.next();
  if (cljs.core.truth_(v.done)) {
    return cljs.core.List.EMPTY;
  } else {
    return new cljs.core.ES6IteratorSeq(v.value, iter, null);
  }
};
cljs.core.mix_collection_hash = function cljs$core$mix_collection_hash(hash_basis, count) {
  var h1 = cljs.core.m3_seed;
  var k1 = cljs.core.m3_mix_K1(hash_basis);
  var h1__$1 = cljs.core.m3_mix_H1(h1, k1);
  return cljs.core.m3_fmix(h1__$1, count);
};
cljs.core.hash_ordered_coll = function cljs$core$hash_ordered_coll(coll) {
  var n = 0;
  var hash_code = 1;
  var coll__$1 = cljs.core.seq(coll);
  while (true) {
    if (!(coll__$1 == null)) {
      var G__9782 = n + 1;
      var G__9783 = cljs.core.imul(31, hash_code) + cljs.core.hash(cljs.core.first(coll__$1)) | 0;
      var G__9784 = cljs.core.next(coll__$1);
      n = G__9782;
      hash_code = G__9783;
      coll__$1 = G__9784;
      continue;
    } else {
      return cljs.core.mix_collection_hash(hash_code, n);
    }
    break;
  }
};
cljs.core.empty_ordered_hash = cljs.core.mix_collection_hash(1, 0);
cljs.core.hash_unordered_coll = function cljs$core$hash_unordered_coll(coll) {
  var n = 0;
  var hash_code = 0;
  var coll__$1 = cljs.core.seq(coll);
  while (true) {
    if (!(coll__$1 == null)) {
      var G__9785 = n + 1;
      var G__9786 = hash_code + cljs.core.hash(cljs.core.first(coll__$1)) | 0;
      var G__9787 = cljs.core.next(coll__$1);
      n = G__9785;
      hash_code = G__9786;
      coll__$1 = G__9787;
      continue;
    } else {
      return cljs.core.mix_collection_hash(hash_code, n);
    }
    break;
  }
};
cljs.core.empty_unordered_hash = cljs.core.mix_collection_hash(0, 0);
cljs.core.ICounted["null"] = true;
cljs.core._count["null"] = function(_) {
  return 0;
};
Date.prototype.cljs$core$IEquiv$ = cljs.core.PROTOCOL_SENTINEL;
Date.prototype.cljs$core$IEquiv$_equiv$arity$2 = function(o, other) {
  var o__$1 = this;
  return other instanceof Date && o__$1.valueOf() === other.valueOf();
};
Date.prototype.cljs$core$IComparable$ = cljs.core.PROTOCOL_SENTINEL;
Date.prototype.cljs$core$IComparable$_compare$arity$2 = function(this$, other) {
  var this$__$1 = this;
  if (other instanceof Date) {
    var G__9788 = this$__$1.valueOf();
    var G__9789 = other.valueOf();
    return goog.array.defaultCompare(G__9788, G__9789);
  } else {
    throw new Error([cljs.core.str.cljs$core$IFn$_invoke$arity$1("Cannot compare "), cljs.core.str.cljs$core$IFn$_invoke$arity$1(this$__$1), cljs.core.str.cljs$core$IFn$_invoke$arity$1(" to "), cljs.core.str.cljs$core$IFn$_invoke$arity$1(other)].join(""));
  }
};
cljs.core.Inst = function() {
};
cljs.core.inst_ms_STAR_ = function cljs$core$inst_ms_STAR_(inst) {
  if (!(inst == null) && !(inst.cljs$core$Inst$inst_ms_STAR_$arity$1 == null)) {
    return inst.cljs$core$Inst$inst_ms_STAR_$arity$1(inst);
  } else {
    var x__7928__auto__ = inst == null ? null : inst;
    var m__7929__auto__ = cljs.core.inst_ms_STAR_[goog.typeOf(x__7928__auto__)];
    if (!(m__7929__auto__ == null)) {
      return m__7929__auto__.cljs$core$IFn$_invoke$arity$1 ? m__7929__auto__.cljs$core$IFn$_invoke$arity$1(inst) : m__7929__auto__.call(null, inst);
    } else {
      var m__7929__auto____$1 = cljs.core.inst_ms_STAR_["_"];
      if (!(m__7929__auto____$1 == null)) {
        return m__7929__auto____$1.cljs$core$IFn$_invoke$arity$1 ? m__7929__auto____$1.cljs$core$IFn$_invoke$arity$1(inst) : m__7929__auto____$1.call(null, inst);
      } else {
        throw cljs.core.missing_protocol("Inst.inst-ms*", inst);
      }
    }
  }
};
Date.prototype.cljs$core$Inst$ = cljs.core.PROTOCOL_SENTINEL;
Date.prototype.cljs$core$Inst$inst_ms_STAR_$arity$1 = function(inst) {
  var inst__$1 = this;
  return inst__$1.getTime();
};
cljs.core.inst_ms = function cljs$core$inst_ms(inst) {
  return cljs.core.inst_ms_STAR_(inst);
};
cljs.core.inst_QMARK_ = function cljs$core$inst_QMARK_(x) {
  if (!(x == null)) {
    if (false || cljs.core.PROTOCOL_SENTINEL === x.cljs$core$Inst$) {
      return true;
    } else {
      if (!x.cljs$lang$protocol_mask$partition$) {
        return cljs.core.native_satisfies_QMARK_(cljs.core.Inst, x);
      } else {
        return false;
      }
    }
  } else {
    return cljs.core.native_satisfies_QMARK_(cljs.core.Inst, x);
  }
};
cljs.core.IEquiv["number"] = true;
cljs.core._equiv["number"] = function(x, o) {
  return x === o;
};
cljs.core.Fn["function"] = true;
cljs.core.IMeta["function"] = true;
cljs.core._meta["function"] = function(_) {
  return null;
};
cljs.core.IHash["_"] = true;
cljs.core._hash["_"] = function(o) {
  return goog.getUid(o);
};
cljs.core.inc = function cljs$core$inc(x) {
  return x + 1;
};
cljs.core.Reduced = function(val) {
  this.val = val;
  this.cljs$lang$protocol_mask$partition0$ = 32768;
  this.cljs$lang$protocol_mask$partition1$ = 0;
};
cljs.core.Reduced.prototype.cljs$core$IDeref$_deref$arity$1 = function(o) {
  var self__ = this;
  var o__$1 = this;
  return self__.val;
};
cljs.core.Reduced.getBasis = function() {
  return new cljs.core.PersistentVector(null, 1, 5, cljs.core.PersistentVector.EMPTY_NODE, [new cljs.core.Symbol(null, "val", "val", 1769233139, null)], null);
};
cljs.core.Reduced.cljs$lang$type = true;
cljs.core.Reduced.cljs$lang$ctorStr = "cljs.core/Reduced";
cljs.core.Reduced.cljs$lang$ctorPrWriter = function(this__7871__auto__, writer__7872__auto__, opt__7873__auto__) {
  return cljs.core._write(writer__7872__auto__, "cljs.core/Reduced");
};
cljs.core.__GT_Reduced = function cljs$core$__GT_Reduced(val) {
  return new cljs.core.Reduced(val);
};
cljs.core.reduced = function cljs$core$reduced(x) {
  return new cljs.core.Reduced(x);
};
cljs.core.reduced_QMARK_ = function cljs$core$reduced_QMARK_(r) {
  return r instanceof cljs.core.Reduced;
};
cljs.core.ensure_reduced = function cljs$core$ensure_reduced(x) {
  if (cljs.core.reduced_QMARK_(x)) {
    return x;
  } else {
    return cljs.core.reduced(x);
  }
};
cljs.core.unreduced = function cljs$core$unreduced(x) {
  if (cljs.core.reduced_QMARK_(x)) {
    return cljs.core.deref.cljs$core$IFn$_invoke$arity$1 ? cljs.core.deref.cljs$core$IFn$_invoke$arity$1(x) : cljs.core.deref.call(null, x);
  } else {
    return x;
  }
};
cljs.core.deref = function cljs$core$deref(o) {
  return cljs.core._deref(o);
};
cljs.core.ci_reduce = function cljs$core$ci_reduce(var_args) {
  var args9792 = [];
  var len__8373__auto___9801 = arguments.length;
  var i__8374__auto___9802 = 0;
  while (true) {
    if (i__8374__auto___9802 < len__8373__auto___9801) {
      args9792.push(arguments[i__8374__auto___9802]);
      var G__9803 = i__8374__auto___9802 + 1;
      i__8374__auto___9802 = G__9803;
      continue;
    } else {
    }
    break;
  }
  var G__9794 = args9792.length;
  switch(G__9794) {
    case 2:
      return cljs.core.ci_reduce.cljs$core$IFn$_invoke$arity$2(arguments[0], arguments[1]);
      break;
    case 3:
      return cljs.core.ci_reduce.cljs$core$IFn$_invoke$arity$3(arguments[0], arguments[1], arguments[2]);
      break;
    case 4:
      return cljs.core.ci_reduce.cljs$core$IFn$_invoke$arity$4(arguments[0], arguments[1], arguments[2], arguments[3]);
      break;
    default:
      throw new Error([cljs.core.str.cljs$core$IFn$_invoke$arity$1("Invalid arity: "), cljs.core.str.cljs$core$IFn$_invoke$arity$1(args9792.length)].join(""));
  }
};
cljs.core.ci_reduce.cljs$core$IFn$_invoke$arity$2 = function(cicoll, f) {
  var cnt = cljs.core._count(cicoll);
  if (cnt === 0) {
    return f.cljs$core$IFn$_invoke$arity$0 ? f.cljs$core$IFn$_invoke$arity$0() : f.call(null);
  } else {
    var val = cljs.core._nth.cljs$core$IFn$_invoke$arity$2(cicoll, 0);
    var n = 1;
    while (true) {
      if (n < cnt) {
        var nval = function() {
          var G__9795 = val;
          var G__9796 = cljs.core._nth.cljs$core$IFn$_invoke$arity$2(cicoll, n);
          return f.cljs$core$IFn$_invoke$arity$2 ? f.cljs$core$IFn$_invoke$arity$2(G__9795, G__9796) : f.call(null, G__9795, G__9796);
        }();
        if (cljs.core.reduced_QMARK_(nval)) {
          return cljs.core.deref(nval);
        } else {
          var G__9805 = nval;
          var G__9806 = n + 1;
          val = G__9805;
          n = G__9806;
          continue;
        }
      } else {
        return val;
      }
      break;
    }
  }
};
cljs.core.ci_reduce.cljs$core$IFn$_invoke$arity$3 = function(cicoll, f, val) {
  var cnt = cljs.core._count(cicoll);
  var val__$1 = val;
  var n = 0;
  while (true) {
    if (n < cnt) {
      var nval = function() {
        var G__9797 = val__$1;
        var G__9798 = cljs.core._nth.cljs$core$IFn$_invoke$arity$2(cicoll, n);
        return f.cljs$core$IFn$_invoke$arity$2 ? f.cljs$core$IFn$_invoke$arity$2(G__9797, G__9798) : f.call(null, G__9797, G__9798);
      }();
      if (cljs.core.reduced_QMARK_(nval)) {
        return cljs.core.deref(nval);
      } else {
        var G__9807 = nval;
        var G__9808 = n + 1;
        val__$1 = G__9807;
        n = G__9808;
        continue;
      }
    } else {
      return val__$1;
    }
    break;
  }
};
cljs.core.ci_reduce.cljs$core$IFn$_invoke$arity$4 = function(cicoll, f, val, idx) {
  var cnt = cljs.core._count(cicoll);
  var val__$1 = val;
  var n = idx;
  while (true) {
    if (n < cnt) {
      var nval = function() {
        var G__9799 = val__$1;
        var G__9800 = cljs.core._nth.cljs$core$IFn$_invoke$arity$2(cicoll, n);
        return f.cljs$core$IFn$_invoke$arity$2 ? f.cljs$core$IFn$_invoke$arity$2(G__9799, G__9800) : f.call(null, G__9799, G__9800);
      }();
      if (cljs.core.reduced_QMARK_(nval)) {
        return cljs.core.deref(nval);
      } else {
        var G__9809 = nval;
        var G__9810 = n + 1;
        val__$1 = G__9809;
        n = G__9810;
        continue;
      }
    } else {
      return val__$1;
    }
    break;
  }
};
cljs.core.ci_reduce.cljs$lang$maxFixedArity = 4;
cljs.core.array_reduce = function cljs$core$array_reduce(var_args) {
  var args9811 = [];
  var len__8373__auto___9820 = arguments.length;
  var i__8374__auto___9821 = 0;
  while (true) {
    if (i__8374__auto___9821 < len__8373__auto___9820) {
      args9811.push(arguments[i__8374__auto___9821]);
      var G__9822 = i__8374__auto___9821 + 1;
      i__8374__auto___9821 = G__9822;
      continue;
    } else {
    }
    break;
  }
  var G__9813 = args9811.length;
  switch(G__9813) {
    case 2:
      return cljs.core.array_reduce.cljs$core$IFn$_invoke$arity$2(arguments[0], arguments[1]);
      break;
    case 3:
      return cljs.core.array_reduce.cljs$core$IFn$_invoke$arity$3(arguments[0], arguments[1], arguments[2]);
      break;
    case 4:
      return cljs.core.array_reduce.cljs$core$IFn$_invoke$arity$4(arguments[0], arguments[1], arguments[2], arguments[3]);
      break;
    default:
      throw new Error([cljs.core.str.cljs$core$IFn$_invoke$arity$1("Invalid arity: "), cljs.core.str.cljs$core$IFn$_invoke$arity$1(args9811.length)].join(""));
  }
};
cljs.core.array_reduce.cljs$core$IFn$_invoke$arity$2 = function(arr, f) {
  var cnt = arr.length;
  if (arr.length === 0) {
    return f.cljs$core$IFn$_invoke$arity$0 ? f.cljs$core$IFn$_invoke$arity$0() : f.call(null);
  } else {
    var val = arr[0];
    var n = 1;
    while (true) {
      if (n < cnt) {
        var nval = function() {
          var G__9814 = val;
          var G__9815 = arr[n];
          return f.cljs$core$IFn$_invoke$arity$2 ? f.cljs$core$IFn$_invoke$arity$2(G__9814, G__9815) : f.call(null, G__9814, G__9815);
        }();
        if (cljs.core.reduced_QMARK_(nval)) {
          return cljs.core.deref(nval);
        } else {
          var G__9824 = nval;
          var G__9825 = n + 1;
          val = G__9824;
          n = G__9825;
          continue;
        }
      } else {
        return val;
      }
      break;
    }
  }
};
cljs.core.array_reduce.cljs$core$IFn$_invoke$arity$3 = function(arr, f, val) {
  var cnt = arr.length;
  var val__$1 = val;
  var n = 0;
  while (true) {
    if (n < cnt) {
      var nval = function() {
        var G__9816 = val__$1;
        var G__9817 = arr[n];
        return f.cljs$core$IFn$_invoke$arity$2 ? f.cljs$core$IFn$_invoke$arity$2(G__9816, G__9817) : f.call(null, G__9816, G__9817);
      }();
      if (cljs.core.reduced_QMARK_(nval)) {
        return cljs.core.deref(nval);
      } else {
        var G__9826 = nval;
        var G__9827 = n + 1;
        val__$1 = G__9826;
        n = G__9827;
        continue;
      }
    } else {
      return val__$1;
    }
    break;
  }
};
cljs.core.array_reduce.cljs$core$IFn$_invoke$arity$4 = function(arr, f, val, idx) {
  var cnt = arr.length;
  var val__$1 = val;
  var n = idx;
  while (true) {
    if (n < cnt) {
      var nval = function() {
        var G__9818 = val__$1;
        var G__9819 = arr[n];
        return f.cljs$core$IFn$_invoke$arity$2 ? f.cljs$core$IFn$_invoke$arity$2(G__9818, G__9819) : f.call(null, G__9818, G__9819);
      }();
      if (cljs.core.reduced_QMARK_(nval)) {
        return cljs.core.deref(nval);
      } else {
        var G__9828 = nval;
        var G__9829 = n + 1;
        val__$1 = G__9828;
        n = G__9829;
        continue;
      }
    } else {
      return val__$1;
    }
    break;
  }
};
cljs.core.array_reduce.cljs$lang$maxFixedArity = 4;
cljs.core.counted_QMARK_ = function cljs$core$counted_QMARK_(x) {
  if (!(x == null)) {
    if (x.cljs$lang$protocol_mask$partition0$ & 2 || cljs.core.PROTOCOL_SENTINEL === x.cljs$core$ICounted$) {
      return true;
    } else {
      if (!x.cljs$lang$protocol_mask$partition0$) {
        return cljs.core.native_satisfies_QMARK_(cljs.core.ICounted, x);
      } else {
        return false;
      }
    }
  } else {
    return cljs.core.native_satisfies_QMARK_(cljs.core.ICounted, x);
  }
};
cljs.core.indexed_QMARK_ = function cljs$core$indexed_QMARK_(x) {
  if (!(x == null)) {
    if (x.cljs$lang$protocol_mask$partition0$ & 16 || cljs.core.PROTOCOL_SENTINEL === x.cljs$core$IIndexed$) {
      return true;
    } else {
      if (!x.cljs$lang$protocol_mask$partition0$) {
        return cljs.core.native_satisfies_QMARK_(cljs.core.IIndexed, x);
      } else {
        return false;
      }
    }
  } else {
    return cljs.core.native_satisfies_QMARK_(cljs.core.IIndexed, x);
  }
};
cljs.core._indexOf = function cljs$core$_indexOf(var_args) {
  var args9834 = [];
  var len__8373__auto___9837 = arguments.length;
  var i__8374__auto___9838 = 0;
  while (true) {
    if (i__8374__auto___9838 < len__8373__auto___9837) {
      args9834.push(arguments[i__8374__auto___9838]);
      var G__9839 = i__8374__auto___9838 + 1;
      i__8374__auto___9838 = G__9839;
      continue;
    } else {
    }
    break;
  }
  var G__9836 = args9834.length;
  switch(G__9836) {
    case 2:
      return cljs.core._indexOf.cljs$core$IFn$_invoke$arity$2(arguments[0], arguments[1]);
      break;
    case 3:
      return cljs.core._indexOf.cljs$core$IFn$_invoke$arity$3(arguments[0], arguments[1], arguments[2]);
      break;
    default:
      throw new Error([cljs.core.str.cljs$core$IFn$_invoke$arity$1("Invalid arity: "), cljs.core.str.cljs$core$IFn$_invoke$arity$1(args9834.length)].join(""));
  }
};
cljs.core._indexOf.cljs$core$IFn$_invoke$arity$2 = function(coll, x) {
  return cljs.core._indexOf.cljs$core$IFn$_invoke$arity$3(coll, x, 0);
};
cljs.core._indexOf.cljs$core$IFn$_invoke$arity$3 = function(coll, x, start) {
  var len = cljs.core.count.cljs$core$IFn$_invoke$arity$1 ? cljs.core.count.cljs$core$IFn$_invoke$arity$1(coll) : cljs.core.count.call(null, coll);
  if (start >= len) {
    return -1;
  } else {
    var idx = start > 0 ? start : start < 0 ? function() {
      var x__7596__auto__ = 0;
      var y__7597__auto__ = start + len;
      return x__7596__auto__ > y__7597__auto__ ? x__7596__auto__ : y__7597__auto__;
    }() : start;
    while (true) {
      if (idx < len) {
        if (cljs.core._EQ_.cljs$core$IFn$_invoke$arity$2(cljs.core.nth.cljs$core$IFn$_invoke$arity$2 ? cljs.core.nth.cljs$core$IFn$_invoke$arity$2(coll, idx) : cljs.core.nth.call(null, coll, idx), x)) {
          return idx;
        } else {
          var G__9841 = idx + 1;
          idx = G__9841;
          continue;
        }
      } else {
        return -1;
      }
      break;
    }
  }
};
cljs.core._indexOf.cljs$lang$maxFixedArity = 3;
cljs.core._lastIndexOf = function cljs$core$_lastIndexOf(var_args) {
  var args9842 = [];
  var len__8373__auto___9845 = arguments.length;
  var i__8374__auto___9846 = 0;
  while (true) {
    if (i__8374__auto___9846 < len__8373__auto___9845) {
      args9842.push(arguments[i__8374__auto___9846]);
      var G__9847 = i__8374__auto___9846 + 1;
      i__8374__auto___9846 = G__9847;
      continue;
    } else {
    }
    break;
  }
  var G__9844 = args9842.length;
  switch(G__9844) {
    case 2:
      return cljs.core._lastIndexOf.cljs$core$IFn$_invoke$arity$2(arguments[0], arguments[1]);
      break;
    case 3:
      return cljs.core._lastIndexOf.cljs$core$IFn$_invoke$arity$3(arguments[0], arguments[1], arguments[2]);
      break;
    default:
      throw new Error([cljs.core.str.cljs$core$IFn$_invoke$arity$1("Invalid arity: "), cljs.core.str.cljs$core$IFn$_invoke$arity$1(args9842.length)].join(""));
  }
};
cljs.core._lastIndexOf.cljs$core$IFn$_invoke$arity$2 = function(coll, x) {
  return cljs.core._lastIndexOf.cljs$core$IFn$_invoke$arity$3(coll, x, cljs.core.count.cljs$core$IFn$_invoke$arity$1 ? cljs.core.count.cljs$core$IFn$_invoke$arity$1(coll) : cljs.core.count.call(null, coll));
};
cljs.core._lastIndexOf.cljs$core$IFn$_invoke$arity$3 = function(coll, x, start) {
  var len = cljs.core.count.cljs$core$IFn$_invoke$arity$1 ? cljs.core.count.cljs$core$IFn$_invoke$arity$1(coll) : cljs.core.count.call(null, coll);
  if (len === 0) {
    return -1;
  } else {
    var idx = start > 0 ? function() {
      var x__7603__auto__ = len - 1;
      var y__7604__auto__ = start;
      return x__7603__auto__ < y__7604__auto__ ? x__7603__auto__ : y__7604__auto__;
    }() : start < 0 ? len + start : start;
    while (true) {
      if (idx >= 0) {
        if (cljs.core._EQ_.cljs$core$IFn$_invoke$arity$2(cljs.core.nth.cljs$core$IFn$_invoke$arity$2 ? cljs.core.nth.cljs$core$IFn$_invoke$arity$2(coll, idx) : cljs.core.nth.call(null, coll, idx), x)) {
          return idx;
        } else {
          var G__9849 = idx - 1;
          idx = G__9849;
          continue;
        }
      } else {
        return -1;
      }
      break;
    }
  }
};
cljs.core._lastIndexOf.cljs$lang$maxFixedArity = 3;
cljs.core.IndexedSeqIterator = function(arr, i) {
  this.arr = arr;
  this.i = i;
};
cljs.core.IndexedSeqIterator.prototype.hasNext = function() {
  var self__ = this;
  var _ = this;
  return self__.i < self__.arr.length;
};
cljs.core.IndexedSeqIterator.prototype.next = function() {
  var self__ = this;
  var _ = this;
  var ret = self__.arr[self__.i];
  self__.i = self__.i + 1;
  return ret;
};
cljs.core.IndexedSeqIterator.getBasis = function() {
  return new cljs.core.PersistentVector(null, 2, 5, cljs.core.PersistentVector.EMPTY_NODE, [new cljs.core.Symbol(null, "arr", "arr", 2115492975, null), cljs.core.with_meta(new cljs.core.Symbol(null, "i", "i", 253690212, null), new cljs.core.PersistentArrayMap(null, 1, [new cljs.core.Keyword(null, "mutable", "mutable", 875778266), true], null))], null);
};
cljs.core.IndexedSeqIterator.cljs$lang$type = true;
cljs.core.IndexedSeqIterator.cljs$lang$ctorStr = "cljs.core/IndexedSeqIterator";
cljs.core.IndexedSeqIterator.cljs$lang$ctorPrWriter = function(this__7871__auto__, writer__7872__auto__, opt__7873__auto__) {
  return cljs.core._write(writer__7872__auto__, "cljs.core/IndexedSeqIterator");
};
cljs.core.__GT_IndexedSeqIterator = function cljs$core$__GT_IndexedSeqIterator(arr, i) {
  return new cljs.core.IndexedSeqIterator(arr, i);
};
cljs.core.IndexedSeq = function(arr, i, meta) {
  this.arr = arr;
  this.i = i;
  this.meta = meta;
  this.cljs$lang$protocol_mask$partition0$ = 166592766;
  this.cljs$lang$protocol_mask$partition1$ = 8192;
};
cljs.core.IndexedSeq.prototype.toString = function() {
  var self__ = this;
  var coll = this;
  return cljs.core.pr_str_STAR_(coll);
};
cljs.core.IndexedSeq.prototype.equiv = function(other) {
  var self__ = this;
  var this$ = this;
  return this$.cljs$core$IEquiv$_equiv$arity$2(null, other);
};
cljs.core.IndexedSeq.prototype.indexOf = function() {
  var G__9850 = null;
  var G__9850__1 = function(x) {
    var self__ = this;
    var coll = this;
    return cljs.core._indexOf.cljs$core$IFn$_invoke$arity$3(coll, x, 0);
  };
  var G__9850__2 = function(x, start) {
    var self__ = this;
    var coll = this;
    return cljs.core._indexOf.cljs$core$IFn$_invoke$arity$3(coll, x, start);
  };
  G__9850 = function(x, start) {
    switch(arguments.length) {
      case 1:
        return G__9850__1.call(this, x);
      case 2:
        return G__9850__2.call(this, x, start);
    }
    throw new Error("Invalid arity: " + (arguments.length - 1));
  };
  G__9850.cljs$core$IFn$_invoke$arity$1 = G__9850__1;
  G__9850.cljs$core$IFn$_invoke$arity$2 = G__9850__2;
  return G__9850;
}();
cljs.core.IndexedSeq.prototype.lastIndexOf = function() {
  var G__9851 = null;
  var G__9851__1 = function(x) {
    var self__ = this;
    var coll = this;
    return cljs.core._lastIndexOf.cljs$core$IFn$_invoke$arity$3(coll, x, cljs.core.count.cljs$core$IFn$_invoke$arity$1 ? cljs.core.count.cljs$core$IFn$_invoke$arity$1(coll) : cljs.core.count.call(null, coll));
  };
  var G__9851__2 = function(x, start) {
    var self__ = this;
    var coll = this;
    return cljs.core._lastIndexOf.cljs$core$IFn$_invoke$arity$3(coll, x, start);
  };
  G__9851 = function(x, start) {
    switch(arguments.length) {
      case 1:
        return G__9851__1.call(this, x);
      case 2:
        return G__9851__2.call(this, x, start);
    }
    throw new Error("Invalid arity: " + (arguments.length - 1));
  };
  G__9851.cljs$core$IFn$_invoke$arity$1 = G__9851__1;
  G__9851.cljs$core$IFn$_invoke$arity$2 = G__9851__2;
  return G__9851;
}();
cljs.core.IndexedSeq.prototype.cljs$core$IIndexed$_nth$arity$2 = function(coll, n) {
  var self__ = this;
  var coll__$1 = this;
  var i__$1 = n + self__.i;
  if (0 <= i__$1 && i__$1 < self__.arr.length) {
    return self__.arr[i__$1];
  } else {
    throw new Error("Index out of bounds");
  }
};
cljs.core.IndexedSeq.prototype.cljs$core$IIndexed$_nth$arity$3 = function(coll, n, not_found) {
  var self__ = this;
  var coll__$1 = this;
  var i__$1 = n + self__.i;
  if (0 <= i__$1 && i__$1 < self__.arr.length) {
    return self__.arr[i__$1];
  } else {
    return not_found;
  }
};
cljs.core.IndexedSeq.prototype.cljs$core$IIterable$ = cljs.core.PROTOCOL_SENTINEL;
cljs.core.IndexedSeq.prototype.cljs$core$IIterable$_iterator$arity$1 = function(coll) {
  var self__ = this;
  var coll__$1 = this;
  return new cljs.core.IndexedSeqIterator(self__.arr, self__.i);
};
cljs.core.IndexedSeq.prototype.cljs$core$IMeta$_meta$arity$1 = function(coll) {
  var self__ = this;
  var coll__$1 = this;
  return self__.meta;
};
cljs.core.IndexedSeq.prototype.cljs$core$ICloneable$_clone$arity$1 = function(_) {
  var self__ = this;
  var ___$1 = this;
  return new cljs.core.IndexedSeq(self__.arr, self__.i, self__.meta);
};
cljs.core.IndexedSeq.prototype.cljs$core$INext$_next$arity$1 = function(_) {
  var self__ = this;
  var ___$1 = this;
  if (self__.i + 1 < self__.arr.length) {
    return new cljs.core.IndexedSeq(self__.arr, self__.i + 1, null);
  } else {
    return null;
  }
};
cljs.core.IndexedSeq.prototype.cljs$core$ICounted$_count$arity$1 = function(_) {
  var self__ = this;
  var ___$1 = this;
  var x__7596__auto__ = 0;
  var y__7597__auto__ = self__.arr.length - self__.i;
  return x__7596__auto__ > y__7597__auto__ ? x__7596__auto__ : y__7597__auto__;
};
cljs.core.IndexedSeq.prototype.cljs$core$IReversible$_rseq$arity$1 = function(coll) {
  var self__ = this;
  var coll__$1 = this;
  var c = coll__$1.cljs$core$ICounted$_count$arity$1(null);
  if (c > 0) {
    return new cljs.core.RSeq(coll__$1, c - 1, null);
  } else {
    return null;
  }
};
cljs.core.IndexedSeq.prototype.cljs$core$IHash$_hash$arity$1 = function(coll) {
  var self__ = this;
  var coll__$1 = this;
  return cljs.core.hash_ordered_coll(coll__$1);
};
cljs.core.IndexedSeq.prototype.cljs$core$IEquiv$_equiv$arity$2 = function(coll, other) {
  var self__ = this;
  var coll__$1 = this;
  return cljs.core.equiv_sequential.cljs$core$IFn$_invoke$arity$2 ? cljs.core.equiv_sequential.cljs$core$IFn$_invoke$arity$2(coll__$1, other) : cljs.core.equiv_sequential.call(null, coll__$1, other);
};
cljs.core.IndexedSeq.prototype.cljs$core$IEmptyableCollection$_empty$arity$1 = function(coll) {
  var self__ = this;
  var coll__$1 = this;
  return cljs.core.List.EMPTY;
};
cljs.core.IndexedSeq.prototype.cljs$core$IReduce$_reduce$arity$2 = function(coll, f) {
  var self__ = this;
  var coll__$1 = this;
  return cljs.core.array_reduce.cljs$core$IFn$_invoke$arity$4(self__.arr, f, self__.arr[self__.i], self__.i + 1);
};
cljs.core.IndexedSeq.prototype.cljs$core$IReduce$_reduce$arity$3 = function(coll, f, start) {
  var self__ = this;
  var coll__$1 = this;
  return cljs.core.array_reduce.cljs$core$IFn$_invoke$arity$4(self__.arr, f, start, self__.i);
};
cljs.core.IndexedSeq.prototype.cljs$core$ISeq$_first$arity$1 = function(_) {
  var self__ = this;
  var ___$1 = this;
  return self__.arr[self__.i];
};
cljs.core.IndexedSeq.prototype.cljs$core$ISeq$_rest$arity$1 = function(_) {
  var self__ = this;
  var ___$1 = this;
  if (self__.i + 1 < self__.arr.length) {
    return new cljs.core.IndexedSeq(self__.arr, self__.i + 1, null);
  } else {
    return cljs.core.List.EMPTY;
  }
};
cljs.core.IndexedSeq.prototype.cljs$core$ISeqable$_seq$arity$1 = function(this$) {
  var self__ = this;
  var this$__$1 = this;
  if (self__.i < self__.arr.length) {
    return this$__$1;
  } else {
    return null;
  }
};
cljs.core.IndexedSeq.prototype.cljs$core$IWithMeta$_with_meta$arity$2 = function(coll, new_meta) {
  var self__ = this;
  var coll__$1 = this;
  return new cljs.core.IndexedSeq(self__.arr, self__.i, new_meta);
};
cljs.core.IndexedSeq.prototype.cljs$core$ICollection$_conj$arity$2 = function(coll, o) {
  var self__ = this;
  var coll__$1 = this;
  return cljs.core.cons.cljs$core$IFn$_invoke$arity$2 ? cljs.core.cons.cljs$core$IFn$_invoke$arity$2(o, coll__$1) : cljs.core.cons.call(null, o, coll__$1);
};
cljs.core.IndexedSeq.getBasis = function() {
  return new cljs.core.PersistentVector(null, 3, 5, cljs.core.PersistentVector.EMPTY_NODE, [new cljs.core.Symbol(null, "arr", "arr", 2115492975, null), new cljs.core.Symbol(null, "i", "i", 253690212, null), new cljs.core.Symbol(null, "meta", "meta", -1154898805, null)], null);
};
cljs.core.IndexedSeq.cljs$lang$type = true;
cljs.core.IndexedSeq.cljs$lang$ctorStr = "cljs.core/IndexedSeq";
cljs.core.IndexedSeq.cljs$lang$ctorPrWriter = function(this__7871__auto__, writer__7872__auto__, opt__7873__auto__) {
  return cljs.core._write(writer__7872__auto__, "cljs.core/IndexedSeq");
};
cljs.core.__GT_IndexedSeq = function cljs$core$__GT_IndexedSeq(arr, i, meta) {
  return new cljs.core.IndexedSeq(arr, i, meta);
};
cljs.core.IndexedSeq.prototype[cljs.core.ITER_SYMBOL] = function() {
  var this__8274__auto__ = this;
  return cljs.core.es6_iterator(this__8274__auto__);
};
cljs.core.prim_seq = function cljs$core$prim_seq(var_args) {
  var args9852 = [];
  var len__8373__auto___9855 = arguments.length;
  var i__8374__auto___9856 = 0;
  while (true) {
    if (i__8374__auto___9856 < len__8373__auto___9855) {
      args9852.push(arguments[i__8374__auto___9856]);
      var G__9857 = i__8374__auto___9856 + 1;
      i__8374__auto___9856 = G__9857;
      continue;
    } else {
    }
    break;
  }
  var G__9854 = args9852.length;
  switch(G__9854) {
    case 1:
      return cljs.core.prim_seq.cljs$core$IFn$_invoke$arity$1(arguments[0]);
      break;
    case 2:
      return cljs.core.prim_seq.cljs$core$IFn$_invoke$arity$2(arguments[0], arguments[1]);
      break;
    default:
      throw new Error([cljs.core.str.cljs$core$IFn$_invoke$arity$1("Invalid arity: "), cljs.core.str.cljs$core$IFn$_invoke$arity$1(args9852.length)].join(""));
  }
};
cljs.core.prim_seq.cljs$core$IFn$_invoke$arity$1 = function(prim) {
  return cljs.core.prim_seq.cljs$core$IFn$_invoke$arity$2(prim, 0);
};
cljs.core.prim_seq.cljs$core$IFn$_invoke$arity$2 = function(prim, i) {
  if (i < prim.length) {
    return new cljs.core.IndexedSeq(prim, i, null);
  } else {
    return null;
  }
};
cljs.core.prim_seq.cljs$lang$maxFixedArity = 2;
cljs.core.array_seq = function cljs$core$array_seq(var_args) {
  var args9859 = [];
  var len__8373__auto___9862 = arguments.length;
  var i__8374__auto___9863 = 0;
  while (true) {
    if (i__8374__auto___9863 < len__8373__auto___9862) {
      args9859.push(arguments[i__8374__auto___9863]);
      var G__9864 = i__8374__auto___9863 + 1;
      i__8374__auto___9863 = G__9864;
      continue;
    } else {
    }
    break;
  }
  var G__9861 = args9859.length;
  switch(G__9861) {
    case 1:
      return cljs.core.array_seq.cljs$core$IFn$_invoke$arity$1(arguments[0]);
      break;
    case 2:
      return cljs.core.array_seq.cljs$core$IFn$_invoke$arity$2(arguments[0], arguments[1]);
      break;
    default:
      throw new Error([cljs.core.str.cljs$core$IFn$_invoke$arity$1("Invalid arity: "), cljs.core.str.cljs$core$IFn$_invoke$arity$1(args9859.length)].join(""));
  }
};
cljs.core.array_seq.cljs$core$IFn$_invoke$arity$1 = function(array) {
  return cljs.core.prim_seq.cljs$core$IFn$_invoke$arity$2(array, 0);
};
cljs.core.array_seq.cljs$core$IFn$_invoke$arity$2 = function(array, i) {
  return cljs.core.prim_seq.cljs$core$IFn$_invoke$arity$2(array, i);
};
cljs.core.array_seq.cljs$lang$maxFixedArity = 2;
cljs.core.RSeq = function(ci, i, meta) {
  this.ci = ci;
  this.i = i;
  this.meta = meta;
  this.cljs$lang$protocol_mask$partition0$ = 32374990;
  this.cljs$lang$protocol_mask$partition1$ = 8192;
};
cljs.core.RSeq.prototype.toString = function() {
  var self__ = this;
  var coll = this;
  return cljs.core.pr_str_STAR_(coll);
};
cljs.core.RSeq.prototype.equiv = function(other) {
  var self__ = this;
  var this$ = this;
  return this$.cljs$core$IEquiv$_equiv$arity$2(null, other);
};
cljs.core.RSeq.prototype.indexOf = function() {
  var G__9868 = null;
  var G__9868__1 = function(x) {
    var self__ = this;
    var coll = this;
    return cljs.core._indexOf.cljs$core$IFn$_invoke$arity$3(coll, x, 0);
  };
  var G__9868__2 = function(x, start) {
    var self__ = this;
    var coll = this;
    return cljs.core._indexOf.cljs$core$IFn$_invoke$arity$3(coll, x, start);
  };
  G__9868 = function(x, start) {
    switch(arguments.length) {
      case 1:
        return G__9868__1.call(this, x);
      case 2:
        return G__9868__2.call(this, x, start);
    }
    throw new Error("Invalid arity: " + (arguments.length - 1));
  };
  G__9868.cljs$core$IFn$_invoke$arity$1 = G__9868__1;
  G__9868.cljs$core$IFn$_invoke$arity$2 = G__9868__2;
  return G__9868;
}();
cljs.core.RSeq.prototype.lastIndexOf = function() {
  var G__9869 = null;
  var G__9869__1 = function(x) {
    var self__ = this;
    var coll = this;
    return cljs.core._lastIndexOf.cljs$core$IFn$_invoke$arity$3(coll, x, cljs.core.count.cljs$core$IFn$_invoke$arity$1 ? cljs.core.count.cljs$core$IFn$_invoke$arity$1(coll) : cljs.core.count.call(null, coll));
  };
  var G__9869__2 = function(x, start) {
    var self__ = this;
    var coll = this;
    return cljs.core._lastIndexOf.cljs$core$IFn$_invoke$arity$3(coll, x, start);
  };
  G__9869 = function(x, start) {
    switch(arguments.length) {
      case 1:
        return G__9869__1.call(this, x);
      case 2:
        return G__9869__2.call(this, x, start);
    }
    throw new Error("Invalid arity: " + (arguments.length - 1));
  };
  G__9869.cljs$core$IFn$_invoke$arity$1 = G__9869__1;
  G__9869.cljs$core$IFn$_invoke$arity$2 = G__9869__2;
  return G__9869;
}();
cljs.core.RSeq.prototype.cljs$core$IMeta$_meta$arity$1 = function(coll) {
  var self__ = this;
  var coll__$1 = this;
  return self__.meta;
};
cljs.core.RSeq.prototype.cljs$core$ICloneable$_clone$arity$1 = function(_) {
  var self__ = this;
  var ___$1 = this;
  return new cljs.core.RSeq(self__.ci, self__.i, self__.meta);
};
cljs.core.RSeq.prototype.cljs$core$INext$_next$arity$1 = function(coll) {
  var self__ = this;
  var coll__$1 = this;
  if (self__.i > 0) {
    return new cljs.core.RSeq(self__.ci, self__.i - 1, null);
  } else {
    return null;
  }
};
cljs.core.RSeq.prototype.cljs$core$ICounted$_count$arity$1 = function(coll) {
  var self__ = this;
  var coll__$1 = this;
  return self__.i + 1;
};
cljs.core.RSeq.prototype.cljs$core$IHash$_hash$arity$1 = function(coll) {
  var self__ = this;
  var coll__$1 = this;
  return cljs.core.hash_ordered_coll(coll__$1);
};
cljs.core.RSeq.prototype.cljs$core$IEquiv$_equiv$arity$2 = function(coll, other) {
  var self__ = this;
  var coll__$1 = this;
  return cljs.core.equiv_sequential.cljs$core$IFn$_invoke$arity$2 ? cljs.core.equiv_sequential.cljs$core$IFn$_invoke$arity$2(coll__$1, other) : cljs.core.equiv_sequential.call(null, coll__$1, other);
};
cljs.core.RSeq.prototype.cljs$core$IEmptyableCollection$_empty$arity$1 = function(coll) {
  var self__ = this;
  var coll__$1 = this;
  var G__9866 = cljs.core.List.EMPTY;
  var G__9867 = self__.meta;
  return cljs.core.with_meta.cljs$core$IFn$_invoke$arity$2 ? cljs.core.with_meta.cljs$core$IFn$_invoke$arity$2(G__9866, G__9867) : cljs.core.with_meta.call(null, G__9866, G__9867);
};
cljs.core.RSeq.prototype.cljs$core$IReduce$_reduce$arity$2 = function(col, f) {
  var self__ = this;
  var col__$1 = this;
  return cljs.core.seq_reduce.cljs$core$IFn$_invoke$arity$2 ? cljs.core.seq_reduce.cljs$core$IFn$_invoke$arity$2(f, col__$1) : cljs.core.seq_reduce.call(null, f, col__$1);
};
cljs.core.RSeq.prototype.cljs$core$IReduce$_reduce$arity$3 = function(col, f, start) {
  var self__ = this;
  var col__$1 = this;
  return cljs.core.seq_reduce.cljs$core$IFn$_invoke$arity$3 ? cljs.core.seq_reduce.cljs$core$IFn$_invoke$arity$3(f, start, col__$1) : cljs.core.seq_reduce.call(null, f, start, col__$1);
};
cljs.core.RSeq.prototype.cljs$core$ISeq$_first$arity$1 = function(coll) {
  var self__ = this;
  var coll__$1 = this;
  return cljs.core._nth.cljs$core$IFn$_invoke$arity$2(self__.ci, self__.i);
};
cljs.core.RSeq.prototype.cljs$core$ISeq$_rest$arity$1 = function(coll) {
  var self__ = this;
  var coll__$1 = this;
  if (self__.i > 0) {
    return new cljs.core.RSeq(self__.ci, self__.i - 1, null);
  } else {
    return cljs.core.List.EMPTY;
  }
};
cljs.core.RSeq.prototype.cljs$core$ISeqable$_seq$arity$1 = function(coll) {
  var self__ = this;
  var coll__$1 = this;
  return coll__$1;
};
cljs.core.RSeq.prototype.cljs$core$IWithMeta$_with_meta$arity$2 = function(coll, new_meta) {
  var self__ = this;
  var coll__$1 = this;
  return new cljs.core.RSeq(self__.ci, self__.i, new_meta);
};
cljs.core.RSeq.prototype.cljs$core$ICollection$_conj$arity$2 = function(coll, o) {
  var self__ = this;
  var coll__$1 = this;
  return cljs.core.cons.cljs$core$IFn$_invoke$arity$2 ? cljs.core.cons.cljs$core$IFn$_invoke$arity$2(o, coll__$1) : cljs.core.cons.call(null, o, coll__$1);
};
cljs.core.RSeq.getBasis = function() {
  return new cljs.core.PersistentVector(null, 3, 5, cljs.core.PersistentVector.EMPTY_NODE, [new cljs.core.Symbol(null, "ci", "ci", 2049808339, null), new cljs.core.Symbol(null, "i", "i", 253690212, null), new cljs.core.Symbol(null, "meta", "meta", -1154898805, null)], null);
};
cljs.core.RSeq.cljs$lang$type = true;
cljs.core.RSeq.cljs$lang$ctorStr = "cljs.core/RSeq";
cljs.core.RSeq.cljs$lang$ctorPrWriter = function(this__7871__auto__, writer__7872__auto__, opt__7873__auto__) {
  return cljs.core._write(writer__7872__auto__, "cljs.core/RSeq");
};
cljs.core.__GT_RSeq = function cljs$core$__GT_RSeq(ci, i, meta) {
  return new cljs.core.RSeq(ci, i, meta);
};
cljs.core.RSeq.prototype[cljs.core.ITER_SYMBOL] = function() {
  var this__8274__auto__ = this;
  return cljs.core.es6_iterator(this__8274__auto__);
};
cljs.core.second = function cljs$core$second(coll) {
  return cljs.core.first(cljs.core.next(coll));
};
cljs.core.ffirst = function cljs$core$ffirst(coll) {
  return cljs.core.first(cljs.core.first(coll));
};
cljs.core.nfirst = function cljs$core$nfirst(coll) {
  return cljs.core.next(cljs.core.first(coll));
};
cljs.core.fnext = function cljs$core$fnext(coll) {
  return cljs.core.first(cljs.core.next(coll));
};
cljs.core.nnext = function cljs$core$nnext(coll) {
  return cljs.core.next(cljs.core.next(coll));
};
cljs.core.last = function cljs$core$last(s) {
  while (true) {
    var sn = cljs.core.next(s);
    if (!(sn == null)) {
      var G__9870 = sn;
      s = G__9870;
      continue;
    } else {
      return cljs.core.first(s);
    }
    break;
  }
};
cljs.core.IEquiv["_"] = true;
cljs.core._equiv["_"] = function(x, o) {
  return x === o;
};
cljs.core.conj = function cljs$core$conj(var_args) {
  var args9871 = [];
  var len__8373__auto___9877 = arguments.length;
  var i__8374__auto___9878 = 0;
  while (true) {
    if (i__8374__auto___9878 < len__8373__auto___9877) {
      args9871.push(arguments[i__8374__auto___9878]);
      var G__9879 = i__8374__auto___9878 + 1;
      i__8374__auto___9878 = G__9879;
      continue;
    } else {
    }
    break;
  }
  var G__9876 = args9871.length;
  switch(G__9876) {
    case 0:
      return cljs.core.conj.cljs$core$IFn$_invoke$arity$0();
      break;
    case 1:
      return cljs.core.conj.cljs$core$IFn$_invoke$arity$1(arguments[0]);
      break;
    case 2:
      return cljs.core.conj.cljs$core$IFn$_invoke$arity$2(arguments[0], arguments[1]);
      break;
    default:
      var argseq__8392__auto__ = new cljs.core.IndexedSeq(args9871.slice(2), 0, null);
      return cljs.core.conj.cljs$core$IFn$_invoke$arity$variadic(arguments[0], arguments[1], argseq__8392__auto__);
  }
};
cljs.core.conj.cljs$core$IFn$_invoke$arity$0 = function() {
  return cljs.core.PersistentVector.EMPTY;
};
cljs.core.conj.cljs$core$IFn$_invoke$arity$1 = function(coll) {
  return coll;
};
cljs.core.conj.cljs$core$IFn$_invoke$arity$2 = function(coll, x) {
  if (!(coll == null)) {
    return cljs.core._conj(coll, x);
  } else {
    var x__8102__auto__ = x;
    return cljs.core._conj(cljs.core.List.EMPTY, x__8102__auto__);
  }
};
cljs.core.conj.cljs$core$IFn$_invoke$arity$variadic = function(coll, x, xs) {
  while (true) {
    if (cljs.core.truth_(xs)) {
      var G__9881 = cljs.core.conj.cljs$core$IFn$_invoke$arity$2(coll, x);
      var G__9882 = cljs.core.first(xs);
      var G__9883 = cljs.core.next(xs);
      coll = G__9881;
      x = G__9882;
      xs = G__9883;
      continue;
    } else {
      return cljs.core.conj.cljs$core$IFn$_invoke$arity$2(coll, x);
    }
    break;
  }
};
cljs.core.conj.cljs$lang$applyTo = function(seq9872) {
  var G__9873 = cljs.core.first(seq9872);
  var seq9872__$1 = cljs.core.next(seq9872);
  var G__9874 = cljs.core.first(seq9872__$1);
  var seq9872__$2 = cljs.core.next(seq9872__$1);
  return cljs.core.conj.cljs$core$IFn$_invoke$arity$variadic(G__9873, G__9874, seq9872__$2);
};
cljs.core.conj.cljs$lang$maxFixedArity = 2;
cljs.core.empty = function cljs$core$empty(coll) {
  if (coll == null) {
    return null;
  } else {
    return cljs.core._empty(coll);
  }
};
cljs.core.accumulating_seq_count = function cljs$core$accumulating_seq_count(coll) {
  var s = cljs.core.seq(coll);
  var acc = 0;
  while (true) {
    if (cljs.core.counted_QMARK_(s)) {
      return acc + cljs.core._count(s);
    } else {
      var G__9884 = cljs.core.next(s);
      var G__9885 = acc + 1;
      s = G__9884;
      acc = G__9885;
      continue;
    }
    break;
  }
};
cljs.core.count = function cljs$core$count(coll) {
  if (!(coll == null)) {
    if (!(coll == null) ? coll.cljs$lang$protocol_mask$partition0$ & 2 || cljs.core.PROTOCOL_SENTINEL === coll.cljs$core$ICounted$ ? true : false : false) {
      return coll.cljs$core$ICounted$_count$arity$1(null);
    } else {
      if (cljs.core.array_QMARK_(coll)) {
        return coll.length;
      } else {
        if (typeof coll === "string") {
          return coll.length;
        } else {
          if (!(coll == null) ? coll.cljs$lang$protocol_mask$partition0$ & 8388608 || cljs.core.PROTOCOL_SENTINEL === coll.cljs$core$ISeqable$ ? true : false : false) {
            return cljs.core.accumulating_seq_count(coll);
          } else {
            return cljs.core._count(coll);
          }
        }
      }
    }
  } else {
    return 0;
  }
};
cljs.core.linear_traversal_nth = function cljs$core$linear_traversal_nth(var_args) {
  var args9890 = [];
  var len__8373__auto___9893 = arguments.length;
  var i__8374__auto___9894 = 0;
  while (true) {
    if (i__8374__auto___9894 < len__8373__auto___9893) {
      args9890.push(arguments[i__8374__auto___9894]);
      var G__9895 = i__8374__auto___9894 + 1;
      i__8374__auto___9894 = G__9895;
      continue;
    } else {
    }
    break;
  }
  var G__9892 = args9890.length;
  switch(G__9892) {
    case 2:
      return cljs.core.linear_traversal_nth.cljs$core$IFn$_invoke$arity$2(arguments[0], arguments[1]);
      break;
    case 3:
      return cljs.core.linear_traversal_nth.cljs$core$IFn$_invoke$arity$3(arguments[0], arguments[1], arguments[2]);
      break;
    default:
      throw new Error([cljs.core.str.cljs$core$IFn$_invoke$arity$1("Invalid arity: "), cljs.core.str.cljs$core$IFn$_invoke$arity$1(args9890.length)].join(""));
  }
};
cljs.core.linear_traversal_nth.cljs$core$IFn$_invoke$arity$2 = function(coll, n) {
  while (true) {
    if (coll == null) {
      throw new Error("Index out of bounds");
    } else {
      if (n === 0) {
        if (cljs.core.seq(coll)) {
          return cljs.core.first(coll);
        } else {
          throw new Error("Index out of bounds");
        }
      } else {
        if (cljs.core.indexed_QMARK_(coll)) {
          return cljs.core._nth.cljs$core$IFn$_invoke$arity$2(coll, n);
        } else {
          if (cljs.core.seq(coll)) {
            var G__9897 = cljs.core.next(coll);
            var G__9898 = n - 1;
            coll = G__9897;
            n = G__9898;
            continue;
          } else {
            throw new Error("Index out of bounds");
          }
        }
      }
    }
    break;
  }
};
cljs.core.linear_traversal_nth.cljs$core$IFn$_invoke$arity$3 = function(coll, n, not_found) {
  while (true) {
    if (coll == null) {
      return not_found;
    } else {
      if (n === 0) {
        if (cljs.core.seq(coll)) {
          return cljs.core.first(coll);
        } else {
          return not_found;
        }
      } else {
        if (cljs.core.indexed_QMARK_(coll)) {
          return cljs.core._nth.cljs$core$IFn$_invoke$arity$3(coll, n, not_found);
        } else {
          if (cljs.core.seq(coll)) {
            var G__9899 = cljs.core.next(coll);
            var G__9900 = n - 1;
            var G__9901 = not_found;
            coll = G__9899;
            n = G__9900;
            not_found = G__9901;
            continue;
          } else {
            return not_found;
          }
        }
      }
    }
    break;
  }
};
cljs.core.linear_traversal_nth.cljs$lang$maxFixedArity = 3;
cljs.core.nth = function cljs$core$nth(var_args) {
  var args9902 = [];
  var len__8373__auto___9909 = arguments.length;
  var i__8374__auto___9910 = 0;
  while (true) {
    if (i__8374__auto___9910 < len__8373__auto___9909) {
      args9902.push(arguments[i__8374__auto___9910]);
      var G__9911 = i__8374__auto___9910 + 1;
      i__8374__auto___9910 = G__9911;
      continue;
    } else {
    }
    break;
  }
  var G__9904 = args9902.length;
  switch(G__9904) {
    case 2:
      return cljs.core.nth.cljs$core$IFn$_invoke$arity$2(arguments[0], arguments[1]);
      break;
    case 3:
      return cljs.core.nth.cljs$core$IFn$_invoke$arity$3(arguments[0], arguments[1], arguments[2]);
      break;
    default:
      throw new Error([cljs.core.str.cljs$core$IFn$_invoke$arity$1("Invalid arity: "), cljs.core.str.cljs$core$IFn$_invoke$arity$1(args9902.length)].join(""));
  }
};
cljs.core.nth.cljs$core$IFn$_invoke$arity$2 = function(coll, n) {
  if (!(typeof n === "number")) {
    throw new Error("Index argument to nth must be a number");
  } else {
    if (coll == null) {
      return coll;
    } else {
      if (!(coll == null) ? coll.cljs$lang$protocol_mask$partition0$ & 16 || cljs.core.PROTOCOL_SENTINEL === coll.cljs$core$IIndexed$ ? true : false : false) {
        return coll.cljs$core$IIndexed$_nth$arity$2(null, n);
      } else {
        if (cljs.core.array_QMARK_(coll)) {
          if (n >= 0 && n < coll.length) {
            return coll[n];
          } else {
            throw new Error("Index out of bounds");
          }
        } else {
          if (typeof coll === "string") {
            if (n >= 0 && n < coll.length) {
              return coll.charAt(n);
            } else {
              throw new Error("Index out of bounds");
            }
          } else {
            if (!(coll == null) ? coll.cljs$lang$protocol_mask$partition0$ & 64 || cljs.core.PROTOCOL_SENTINEL === coll.cljs$core$ISeq$ ? true : false : false) {
              return cljs.core.linear_traversal_nth.cljs$core$IFn$_invoke$arity$2(coll, n);
            } else {
              if (cljs.core.native_satisfies_QMARK_(cljs.core.IIndexed, coll)) {
                return cljs.core._nth.cljs$core$IFn$_invoke$arity$2(coll, n);
              } else {
                throw new Error([cljs.core.str.cljs$core$IFn$_invoke$arity$1("nth not supported on this type "), cljs.core.str.cljs$core$IFn$_invoke$arity$1(cljs.core.type__GT_str(cljs.core.type(coll)))].join(""));
              }
            }
          }
        }
      }
    }
  }
};
cljs.core.nth.cljs$core$IFn$_invoke$arity$3 = function(coll, n, not_found) {
  if (!(typeof n === "number")) {
    throw new Error("Index argument to nth must be a number.");
  } else {
    if (coll == null) {
      return not_found;
    } else {
      if (!(coll == null) ? coll.cljs$lang$protocol_mask$partition0$ & 16 || cljs.core.PROTOCOL_SENTINEL === coll.cljs$core$IIndexed$ ? true : false : false) {
        return coll.cljs$core$IIndexed$_nth$arity$3(null, n, not_found);
      } else {
        if (cljs.core.array_QMARK_(coll)) {
          if (n >= 0 && n < coll.length) {
            return coll[n];
          } else {
            return not_found;
          }
        } else {
          if (typeof coll === "string") {
            if (n >= 0 && n < coll.length) {
              return coll.charAt(n);
            } else {
              return not_found;
            }
          } else {
            if (!(coll == null) ? coll.cljs$lang$protocol_mask$partition0$ & 64 || cljs.core.PROTOCOL_SENTINEL === coll.cljs$core$ISeq$ ? true : false : false) {
              return cljs.core.linear_traversal_nth.cljs$core$IFn$_invoke$arity$3(coll, n, not_found);
            } else {
              if (cljs.core.native_satisfies_QMARK_(cljs.core.IIndexed, coll)) {
                return cljs.core._nth.cljs$core$IFn$_invoke$arity$2(coll, n);
              } else {
                throw new Error([cljs.core.str.cljs$core$IFn$_invoke$arity$1("nth not supported on this type "), cljs.core.str.cljs$core$IFn$_invoke$arity$1(cljs.core.type__GT_str(cljs.core.type(coll)))].join(""));
              }
            }
          }
        }
      }
    }
  }
};
cljs.core.nth.cljs$lang$maxFixedArity = 3;
cljs.core.nthrest = function cljs$core$nthrest(coll, n) {
  var n__$1 = n;
  var xs = coll;
  while (true) {
    if (n__$1 > 0 && cljs.core.seq(xs)) {
      var G__9913 = n__$1 - 1;
      var G__9914 = cljs.core.rest(xs);
      n__$1 = G__9913;
      xs = G__9914;
      continue;
    } else {
      return xs;
    }
    break;
  }
};
cljs.core.get = function cljs$core$get(var_args) {
  var args9915 = [];
  var len__8373__auto___9920 = arguments.length;
  var i__8374__auto___9921 = 0;
  while (true) {
    if (i__8374__auto___9921 < len__8373__auto___9920) {
      args9915.push(arguments[i__8374__auto___9921]);
      var G__9922 = i__8374__auto___9921 + 1;
      i__8374__auto___9921 = G__9922;
      continue;
    } else {
    }
    break;
  }
  var G__9917 = args9915.length;
  switch(G__9917) {
    case 2:
      return cljs.core.get.cljs$core$IFn$_invoke$arity$2(arguments[0], arguments[1]);
      break;
    case 3:
      return cljs.core.get.cljs$core$IFn$_invoke$arity$3(arguments[0], arguments[1], arguments[2]);
      break;
    default:
      throw new Error([cljs.core.str.cljs$core$IFn$_invoke$arity$1("Invalid arity: "), cljs.core.str.cljs$core$IFn$_invoke$arity$1(args9915.length)].join(""));
  }
};
cljs.core.get.cljs$core$IFn$_invoke$arity$2 = function(o, k) {
  if (o == null) {
    return null;
  } else {
    if (!(o == null) ? o.cljs$lang$protocol_mask$partition0$ & 256 || cljs.core.PROTOCOL_SENTINEL === o.cljs$core$ILookup$ ? true : false : false) {
      return o.cljs$core$ILookup$_lookup$arity$2(null, k);
    } else {
      if (cljs.core.array_QMARK_(o)) {
        if (!(k == null) && k < o.length) {
          return o[k | 0];
        } else {
          return null;
        }
      } else {
        if (typeof o === "string") {
          if (!(k == null) && k < o.length) {
            return o.charAt(k | 0);
          } else {
            return null;
          }
        } else {
          if (cljs.core.native_satisfies_QMARK_(cljs.core.ILookup, o)) {
            return cljs.core._lookup.cljs$core$IFn$_invoke$arity$2(o, k);
          } else {
            return null;
          }
        }
      }
    }
  }
};
cljs.core.get.cljs$core$IFn$_invoke$arity$3 = function(o, k, not_found) {
  if (!(o == null)) {
    if (!(o == null) ? o.cljs$lang$protocol_mask$partition0$ & 256 || cljs.core.PROTOCOL_SENTINEL === o.cljs$core$ILookup$ ? true : false : false) {
      return o.cljs$core$ILookup$_lookup$arity$3(null, k, not_found);
    } else {
      if (cljs.core.array_QMARK_(o)) {
        if (!(k == null) && k >= 0 && k < o.length) {
          return o[k | 0];
        } else {
          return not_found;
        }
      } else {
        if (typeof o === "string") {
          if (!(k == null) && k >= 0 && k < o.length) {
            return o.charAt(k | 0);
          } else {
            return not_found;
          }
        } else {
          if (cljs.core.native_satisfies_QMARK_(cljs.core.ILookup, o)) {
            return cljs.core._lookup.cljs$core$IFn$_invoke$arity$3(o, k, not_found);
          } else {
            return not_found;
          }
        }
      }
    }
  } else {
    return not_found;
  }
};
cljs.core.get.cljs$lang$maxFixedArity = 3;
cljs.core.assoc = function cljs$core$assoc(var_args) {
  var args9924 = [];
  var len__8373__auto___9931 = arguments.length;
  var i__8374__auto___9932 = 0;
  while (true) {
    if (i__8374__auto___9932 < len__8373__auto___9931) {
      args9924.push(arguments[i__8374__auto___9932]);
      var G__9933 = i__8374__auto___9932 + 1;
      i__8374__auto___9932 = G__9933;
      continue;
    } else {
    }
    break;
  }
  var G__9930 = args9924.length;
  switch(G__9930) {
    case 3:
      return cljs.core.assoc.cljs$core$IFn$_invoke$arity$3(arguments[0], arguments[1], arguments[2]);
      break;
    default:
      var argseq__8392__auto__ = new cljs.core.IndexedSeq(args9924.slice(3), 0, null);
      return cljs.core.assoc.cljs$core$IFn$_invoke$arity$variadic(arguments[0], arguments[1], arguments[2], argseq__8392__auto__);
  }
};
cljs.core.assoc.cljs$core$IFn$_invoke$arity$3 = function(coll, k, v) {
  if (!(coll == null)) {
    return cljs.core._assoc(coll, k, v);
  } else {
    return cljs.core.PersistentArrayMap.createAsIfByAssoc([k, v]);
  }
};
cljs.core.assoc.cljs$core$IFn$_invoke$arity$variadic = function(coll, k, v, kvs) {
  while (true) {
    var ret = cljs.core.assoc.cljs$core$IFn$_invoke$arity$3(coll, k, v);
    if (cljs.core.truth_(kvs)) {
      var G__9935 = ret;
      var G__9936 = cljs.core.first(kvs);
      var G__9937 = cljs.core.second(kvs);
      var G__9938 = cljs.core.nnext(kvs);
      coll = G__9935;
      k = G__9936;
      v = G__9937;
      kvs = G__9938;
      continue;
    } else {
      return ret;
    }
    break;
  }
};
cljs.core.assoc.cljs$lang$applyTo = function(seq9925) {
  var G__9926 = cljs.core.first(seq9925);
  var seq9925__$1 = cljs.core.next(seq9925);
  var G__9927 = cljs.core.first(seq9925__$1);
  var seq9925__$2 = cljs.core.next(seq9925__$1);
  var G__9928 = cljs.core.first(seq9925__$2);
  var seq9925__$3 = cljs.core.next(seq9925__$2);
  return cljs.core.assoc.cljs$core$IFn$_invoke$arity$variadic(G__9926, G__9927, G__9928, seq9925__$3);
};
cljs.core.assoc.cljs$lang$maxFixedArity = 3;
cljs.core.dissoc = function cljs$core$dissoc(var_args) {
  var args9939 = [];
  var len__8373__auto___9945 = arguments.length;
  var i__8374__auto___9946 = 0;
  while (true) {
    if (i__8374__auto___9946 < len__8373__auto___9945) {
      args9939.push(arguments[i__8374__auto___9946]);
      var G__9947 = i__8374__auto___9946 + 1;
      i__8374__auto___9946 = G__9947;
      continue;
    } else {
    }
    break;
  }
  var G__9944 = args9939.length;
  switch(G__9944) {
    case 1:
      return cljs.core.dissoc.cljs$core$IFn$_invoke$arity$1(arguments[0]);
      break;
    case 2:
      return cljs.core.dissoc.cljs$core$IFn$_invoke$arity$2(arguments[0], arguments[1]);
      break;
    default:
      var argseq__8392__auto__ = new cljs.core.IndexedSeq(args9939.slice(2), 0, null);
      return cljs.core.dissoc.cljs$core$IFn$_invoke$arity$variadic(arguments[0], arguments[1], argseq__8392__auto__);
  }
};
cljs.core.dissoc.cljs$core$IFn$_invoke$arity$1 = function(coll) {
  return coll;
};
cljs.core.dissoc.cljs$core$IFn$_invoke$arity$2 = function(coll, k) {
  if (coll == null) {
    return null;
  } else {
    return cljs.core._dissoc(coll, k);
  }
};
cljs.core.dissoc.cljs$core$IFn$_invoke$arity$variadic = function(coll, k, ks) {
  while (true) {
    if (coll == null) {
      return null;
    } else {
      var ret = cljs.core.dissoc.cljs$core$IFn$_invoke$arity$2(coll, k);
      if (cljs.core.truth_(ks)) {
        var G__9949 = ret;
        var G__9950 = cljs.core.first(ks);
        var G__9951 = cljs.core.next(ks);
        coll = G__9949;
        k = G__9950;
        ks = G__9951;
        continue;
      } else {
        return ret;
      }
    }
    break;
  }
};
cljs.core.dissoc.cljs$lang$applyTo = function(seq9940) {
  var G__9941 = cljs.core.first(seq9940);
  var seq9940__$1 = cljs.core.next(seq9940);
  var G__9942 = cljs.core.first(seq9940__$1);
  var seq9940__$2 = cljs.core.next(seq9940__$1);
  return cljs.core.dissoc.cljs$core$IFn$_invoke$arity$variadic(G__9941, G__9942, seq9940__$2);
};
cljs.core.dissoc.cljs$lang$maxFixedArity = 2;
cljs.core.fn_QMARK_ = function cljs$core$fn_QMARK_(f) {
  var or__7260__auto__ = goog.isFunction(f);
  if (or__7260__auto__) {
    return or__7260__auto__;
  } else {
    if (!(f == null)) {
      if (false || cljs.core.PROTOCOL_SENTINEL === f.cljs$core$Fn$) {
        return true;
      } else {
        if (!f.cljs$lang$protocol_mask$partition$) {
          return cljs.core.native_satisfies_QMARK_(cljs.core.Fn, f);
        } else {
          return false;
        }
      }
    } else {
      return cljs.core.native_satisfies_QMARK_(cljs.core.Fn, f);
    }
  }
};
cljs.core.MetaFn = function(afn, meta) {
  this.afn = afn;
  this.meta = meta;
  this.cljs$lang$protocol_mask$partition0$ = 393217;
  this.cljs$lang$protocol_mask$partition1$ = 0;
};
cljs.core.MetaFn.prototype.cljs$core$IMeta$_meta$arity$1 = function(_) {
  var self__ = this;
  var ___$1 = this;
  return self__.meta;
};
cljs.core.MetaFn.prototype.cljs$core$IWithMeta$_with_meta$arity$2 = function(_, new_meta) {
  var self__ = this;
  var ___$1 = this;
  return new cljs.core.MetaFn(self__.afn, new_meta);
};
cljs.core.MetaFn.prototype.cljs$core$Fn$ = cljs.core.PROTOCOL_SENTINEL;
cljs.core.MetaFn.prototype.call = function() {
  var G__9957 = null;
  var G__9957__1 = function(self__) {
    var self__ = this;
    var self____$1 = this;
    var _ = self____$1;
    return self__.afn.cljs$core$IFn$_invoke$arity$0 ? self__.afn.cljs$core$IFn$_invoke$arity$0() : self__.afn.call(null);
  };
  var G__9957__2 = function(self__, a) {
    var self__ = this;
    var self____$1 = this;
    var _ = self____$1;
    return self__.afn.cljs$core$IFn$_invoke$arity$1 ? self__.afn.cljs$core$IFn$_invoke$arity$1(a) : self__.afn.call(null, a);
  };
  var G__9957__3 = function(self__, a, b) {
    var self__ = this;
    var self____$1 = this;
    var _ = self____$1;
    return self__.afn.cljs$core$IFn$_invoke$arity$2 ? self__.afn.cljs$core$IFn$_invoke$arity$2(a, b) : self__.afn.call(null, a, b);
  };
  var G__9957__4 = function(self__, a, b, c) {
    var self__ = this;
    var self____$1 = this;
    var _ = self____$1;
    return self__.afn.cljs$core$IFn$_invoke$arity$3 ? self__.afn.cljs$core$IFn$_invoke$arity$3(a, b, c) : self__.afn.call(null, a, b, c);
  };
  var G__9957__5 = function(self__, a, b, c, d) {
    var self__ = this;
    var self____$1 = this;
    var _ = self____$1;
    return self__.afn.cljs$core$IFn$_invoke$arity$4 ? self__.afn.cljs$core$IFn$_invoke$arity$4(a, b, c, d) : self__.afn.call(null, a, b, c, d);
  };
  var G__9957__6 = function(self__, a, b, c, d, e) {
    var self__ = this;
    var self____$1 = this;
    var _ = self____$1;
    return self__.afn.cljs$core$IFn$_invoke$arity$5 ? self__.afn.cljs$core$IFn$_invoke$arity$5(a, b, c, d, e) : self__.afn.call(null, a, b, c, d, e);
  };
  var G__9957__7 = function(self__, a, b, c, d, e, f) {
    var self__ = this;
    var self____$1 = this;
    var _ = self____$1;
    return self__.afn.cljs$core$IFn$_invoke$arity$6 ? self__.afn.cljs$core$IFn$_invoke$arity$6(a, b, c, d, e, f) : self__.afn.call(null, a, b, c, d, e, f);
  };
  var G__9957__8 = function(self__, a, b, c, d, e, f, g) {
    var self__ = this;
    var self____$1 = this;
    var _ = self____$1;
    return self__.afn.cljs$core$IFn$_invoke$arity$7 ? self__.afn.cljs$core$IFn$_invoke$arity$7(a, b, c, d, e, f, g) : self__.afn.call(null, a, b, c, d, e, f, g);
  };
  var G__9957__9 = function(self__, a, b, c, d, e, f, g, h) {
    var self__ = this;
    var self____$1 = this;
    var _ = self____$1;
    return self__.afn.cljs$core$IFn$_invoke$arity$8 ? self__.afn.cljs$core$IFn$_invoke$arity$8(a, b, c, d, e, f, g, h) : self__.afn.call(null, a, b, c, d, e, f, g, h);
  };
  var G__9957__10 = function(self__, a, b, c, d, e, f, g, h, i) {
    var self__ = this;
    var self____$1 = this;
    var _ = self____$1;
    return self__.afn.cljs$core$IFn$_invoke$arity$9 ? self__.afn.cljs$core$IFn$_invoke$arity$9(a, b, c, d, e, f, g, h, i) : self__.afn.call(null, a, b, c, d, e, f, g, h, i);
  };
  var G__9957__11 = function(self__, a, b, c, d, e, f, g, h, i, j) {
    var self__ = this;
    var self____$1 = this;
    var _ = self____$1;
    return self__.afn.cljs$core$IFn$_invoke$arity$10 ? self__.afn.cljs$core$IFn$_invoke$arity$10(a, b, c, d, e, f, g, h, i, j) : self__.afn.call(null, a, b, c, d, e, f, g, h, i, j);
  };
  var G__9957__12 = function(self__, a, b, c, d, e, f, g, h, i, j, k) {
    var self__ = this;
    var self____$1 = this;
    var _ = self____$1;
    return self__.afn.cljs$core$IFn$_invoke$arity$11 ? self__.afn.cljs$core$IFn$_invoke$arity$11(a, b, c, d, e, f, g, h, i, j, k) : self__.afn.call(null, a, b, c, d, e, f, g, h, i, j, k);
  };
  var G__9957__13 = function(self__, a, b, c, d, e, f, g, h, i, j, k, l) {
    var self__ = this;
    var self____$1 = this;
    var _ = self____$1;
    return self__.afn.cljs$core$IFn$_invoke$arity$12 ? self__.afn.cljs$core$IFn$_invoke$arity$12(a, b, c, d, e, f, g, h, i, j, k, l) : self__.afn.call(null, a, b, c, d, e, f, g, h, i, j, k, l);
  };
  var G__9957__14 = function(self__, a, b, c, d, e, f, g, h, i, j, k, l, m) {
    var self__ = this;
    var self____$1 = this;
    var _ = self____$1;
    return self__.afn.cljs$core$IFn$_invoke$arity$13 ? self__.afn.cljs$core$IFn$_invoke$arity$13(a, b, c, d, e, f, g, h, i, j, k, l, m) : self__.afn.call(null, a, b, c, d, e, f, g, h, i, j, k, l, m);
  };
  var G__9957__15 = function(self__, a, b, c, d, e, f, g, h, i, j, k, l, m, n) {
    var self__ = this;
    var self____$1 = this;
    var _ = self____$1;
    return self__.afn.cljs$core$IFn$_invoke$arity$14 ? self__.afn.cljs$core$IFn$_invoke$arity$14(a, b, c, d, e, f, g, h, i, j, k, l, m, n) : self__.afn.call(null, a, b, c, d, e, f, g, h, i, j, k, l, m, n);
  };
  var G__9957__16 = function(self__, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o) {
    var self__ = this;
    var self____$1 = this;
    var _ = self____$1;
    return self__.afn.cljs$core$IFn$_invoke$arity$15 ? self__.afn.cljs$core$IFn$_invoke$arity$15(a, b, c, d, e, f, g, h, i, j, k, l, m, n, o) : self__.afn.call(null, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o);
  };
  var G__9957__17 = function(self__, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p) {
    var self__ = this;
    var self____$1 = this;
    var _ = self____$1;
    return self__.afn.cljs$core$IFn$_invoke$arity$16 ? self__.afn.cljs$core$IFn$_invoke$arity$16(a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p) : self__.afn.call(null, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p);
  };
  var G__9957__18 = function(self__, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q) {
    var self__ = this;
    var self____$1 = this;
    var _ = self____$1;
    return self__.afn.cljs$core$IFn$_invoke$arity$17 ? self__.afn.cljs$core$IFn$_invoke$arity$17(a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q) : self__.afn.call(null, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q);
  };
  var G__9957__19 = function(self__, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r) {
    var self__ = this;
    var self____$1 = this;
    var _ = self____$1;
    return self__.afn.cljs$core$IFn$_invoke$arity$18 ? self__.afn.cljs$core$IFn$_invoke$arity$18(a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r) : self__.afn.call(null, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r);
  };
  var G__9957__20 = function(self__, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s) {
    var self__ = this;
    var self____$1 = this;
    var _ = self____$1;
    return self__.afn.cljs$core$IFn$_invoke$arity$19 ? self__.afn.cljs$core$IFn$_invoke$arity$19(a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s) : self__.afn.call(null, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s);
  };
  var G__9957__21 = function(self__, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s, t) {
    var self__ = this;
    var self____$1 = this;
    var _ = self____$1;
    return self__.afn.cljs$core$IFn$_invoke$arity$20 ? self__.afn.cljs$core$IFn$_invoke$arity$20(a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s, t) : self__.afn.call(null, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s, t);
  };
  var G__9957__22 = function(self__, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s, t, rest) {
    var self__ = this;
    var self____$1 = this;
    var _ = self____$1;
    return cljs.core.apply.cljs$core$IFn$_invoke$arity$22 ? cljs.core.apply.cljs$core$IFn$_invoke$arity$22(self__.afn, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s, t, rest) : cljs.core.apply.call(null, self__.afn, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s, t, rest);
  };
  G__9957 = function(self__, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s, t, rest) {
    switch(arguments.length) {
      case 1:
        return G__9957__1.call(this, self__);
      case 2:
        return G__9957__2.call(this, self__, a);
      case 3:
        return G__9957__3.call(this, self__, a, b);
      case 4:
        return G__9957__4.call(this, self__, a, b, c);
      case 5:
        return G__9957__5.call(this, self__, a, b, c, d);
      case 6:
        return G__9957__6.call(this, self__, a, b, c, d, e);
      case 7:
        return G__9957__7.call(this, self__, a, b, c, d, e, f);
      case 8:
        return G__9957__8.call(this, self__, a, b, c, d, e, f, g);
      case 9:
        return G__9957__9.call(this, self__, a, b, c, d, e, f, g, h);
      case 10:
        return G__9957__10.call(this, self__, a, b, c, d, e, f, g, h, i);
      case 11:
        return G__9957__11.call(this, self__, a, b, c, d, e, f, g, h, i, j);
      case 12:
        return G__9957__12.call(this, self__, a, b, c, d, e, f, g, h, i, j, k);
      case 13:
        return G__9957__13.call(this, self__, a, b, c, d, e, f, g, h, i, j, k, l);
      case 14:
        return G__9957__14.call(this, self__, a, b, c, d, e, f, g, h, i, j, k, l, m);
      case 15:
        return G__9957__15.call(this, self__, a, b, c, d, e, f, g, h, i, j, k, l, m, n);
      case 16:
        return G__9957__16.call(this, self__, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o);
      case 17:
        return G__9957__17.call(this, self__, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p);
      case 18:
        return G__9957__18.call(this, self__, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q);
      case 19:
        return G__9957__19.call(this, self__, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r);
      case 20:
        return G__9957__20.call(this, self__, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s);
      case 21:
        return G__9957__21.call(this, self__, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s, t);
      case 22:
        return G__9957__22.call(this, self__, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s, t, rest);
    }
    throw new Error("Invalid arity: " + (arguments.length - 1));
  };
  G__9957.cljs$core$IFn$_invoke$arity$1 = G__9957__1;
  G__9957.cljs$core$IFn$_invoke$arity$2 = G__9957__2;
  G__9957.cljs$core$IFn$_invoke$arity$3 = G__9957__3;
  G__9957.cljs$core$IFn$_invoke$arity$4 = G__9957__4;
  G__9957.cljs$core$IFn$_invoke$arity$5 = G__9957__5;
  G__9957.cljs$core$IFn$_invoke$arity$6 = G__9957__6;
  G__9957.cljs$core$IFn$_invoke$arity$7 = G__9957__7;
  G__9957.cljs$core$IFn$_invoke$arity$8 = G__9957__8;
  G__9957.cljs$core$IFn$_invoke$arity$9 = G__9957__9;
  G__9957.cljs$core$IFn$_invoke$arity$10 = G__9957__10;
  G__9957.cljs$core$IFn$_invoke$arity$11 = G__9957__11;
  G__9957.cljs$core$IFn$_invoke$arity$12 = G__9957__12;
  G__9957.cljs$core$IFn$_invoke$arity$13 = G__9957__13;
  G__9957.cljs$core$IFn$_invoke$arity$14 = G__9957__14;
  G__9957.cljs$core$IFn$_invoke$arity$15 = G__9957__15;
  G__9957.cljs$core$IFn$_invoke$arity$16 = G__9957__16;
  G__9957.cljs$core$IFn$_invoke$arity$17 = G__9957__17;
  G__9957.cljs$core$IFn$_invoke$arity$18 = G__9957__18;
  G__9957.cljs$core$IFn$_invoke$arity$19 = G__9957__19;
  G__9957.cljs$core$IFn$_invoke$arity$20 = G__9957__20;
  G__9957.cljs$core$IFn$_invoke$arity$21 = G__9957__21;
  G__9957.cljs$core$IFn$_invoke$arity$22 = G__9957__22;
  return G__9957;
}();
cljs.core.MetaFn.prototype.apply = function(self__, args9956) {
  var self__ = this;
  var self____$1 = this;
  return self____$1.call.apply(self____$1, [self____$1].concat(cljs.core.aclone(args9956)));
};
cljs.core.MetaFn.prototype.cljs$core$IFn$_invoke$arity$0 = function() {
  var self__ = this;
  var _ = this;
  return self__.afn.cljs$core$IFn$_invoke$arity$0 ? self__.afn.cljs$core$IFn$_invoke$arity$0() : self__.afn.call(null);
};
cljs.core.MetaFn.prototype.cljs$core$IFn$_invoke$arity$1 = function(a) {
  var self__ = this;
  var _ = this;
  return self__.afn.cljs$core$IFn$_invoke$arity$1 ? self__.afn.cljs$core$IFn$_invoke$arity$1(a) : self__.afn.call(null, a);
};
cljs.core.MetaFn.prototype.cljs$core$IFn$_invoke$arity$2 = function(a, b) {
  var self__ = this;
  var _ = this;
  return self__.afn.cljs$core$IFn$_invoke$arity$2 ? self__.afn.cljs$core$IFn$_invoke$arity$2(a, b) : self__.afn.call(null, a, b);
};
cljs.core.MetaFn.prototype.cljs$core$IFn$_invoke$arity$3 = function(a, b, c) {
  var self__ = this;
  var _ = this;
  return self__.afn.cljs$core$IFn$_invoke$arity$3 ? self__.afn.cljs$core$IFn$_invoke$arity$3(a, b, c) : self__.afn.call(null, a, b, c);
};
cljs.core.MetaFn.prototype.cljs$core$IFn$_invoke$arity$4 = function(a, b, c, d) {
  var self__ = this;
  var _ = this;
  return self__.afn.cljs$core$IFn$_invoke$arity$4 ? self__.afn.cljs$core$IFn$_invoke$arity$4(a, b, c, d) : self__.afn.call(null, a, b, c, d);
};
cljs.core.MetaFn.prototype.cljs$core$IFn$_invoke$arity$5 = function(a, b, c, d, e) {
  var self__ = this;
  var _ = this;
  return self__.afn.cljs$core$IFn$_invoke$arity$5 ? self__.afn.cljs$core$IFn$_invoke$arity$5(a, b, c, d, e) : self__.afn.call(null, a, b, c, d, e);
};
cljs.core.MetaFn.prototype.cljs$core$IFn$_invoke$arity$6 = function(a, b, c, d, e, f) {
  var self__ = this;
  var _ = this;
  return self__.afn.cljs$core$IFn$_invoke$arity$6 ? self__.afn.cljs$core$IFn$_invoke$arity$6(a, b, c, d, e, f) : self__.afn.call(null, a, b, c, d, e, f);
};
cljs.core.MetaFn.prototype.cljs$core$IFn$_invoke$arity$7 = function(a, b, c, d, e, f, g) {
  var self__ = this;
  var _ = this;
  return self__.afn.cljs$core$IFn$_invoke$arity$7 ? self__.afn.cljs$core$IFn$_invoke$arity$7(a, b, c, d, e, f, g) : self__.afn.call(null, a, b, c, d, e, f, g);
};
cljs.core.MetaFn.prototype.cljs$core$IFn$_invoke$arity$8 = function(a, b, c, d, e, f, g, h) {
  var self__ = this;
  var _ = this;
  return self__.afn.cljs$core$IFn$_invoke$arity$8 ? self__.afn.cljs$core$IFn$_invoke$arity$8(a, b, c, d, e, f, g, h) : self__.afn.call(null, a, b, c, d, e, f, g, h);
};
cljs.core.MetaFn.prototype.cljs$core$IFn$_invoke$arity$9 = function(a, b, c, d, e, f, g, h, i) {
  var self__ = this;
  var _ = this;
  return self__.afn.cljs$core$IFn$_invoke$arity$9 ? self__.afn.cljs$core$IFn$_invoke$arity$9(a, b, c, d, e, f, g, h, i) : self__.afn.call(null, a, b, c, d, e, f, g, h, i);
};
cljs.core.MetaFn.prototype.cljs$core$IFn$_invoke$arity$10 = function(a, b, c, d, e, f, g, h, i, j) {
  var self__ = this;
  var _ = this;
  return self__.afn.cljs$core$IFn$_invoke$arity$10 ? self__.afn.cljs$core$IFn$_invoke$arity$10(a, b, c, d, e, f, g, h, i, j) : self__.afn.call(null, a, b, c, d, e, f, g, h, i, j);
};
cljs.core.MetaFn.prototype.cljs$core$IFn$_invoke$arity$11 = function(a, b, c, d, e, f, g, h, i, j, k) {
  var self__ = this;
  var _ = this;
  return self__.afn.cljs$core$IFn$_invoke$arity$11 ? self__.afn.cljs$core$IFn$_invoke$arity$11(a, b, c, d, e, f, g, h, i, j, k) : self__.afn.call(null, a, b, c, d, e, f, g, h, i, j, k);
};
cljs.core.MetaFn.prototype.cljs$core$IFn$_invoke$arity$12 = function(a, b, c, d, e, f, g, h, i, j, k, l) {
  var self__ = this;
  var _ = this;
  return self__.afn.cljs$core$IFn$_invoke$arity$12 ? self__.afn.cljs$core$IFn$_invoke$arity$12(a, b, c, d, e, f, g, h, i, j, k, l) : self__.afn.call(null, a, b, c, d, e, f, g, h, i, j, k, l);
};
cljs.core.MetaFn.prototype.cljs$core$IFn$_invoke$arity$13 = function(a, b, c, d, e, f, g, h, i, j, k, l, m) {
  var self__ = this;
  var _ = this;
  return self__.afn.cljs$core$IFn$_invoke$arity$13 ? self__.afn.cljs$core$IFn$_invoke$arity$13(a, b, c, d, e, f, g, h, i, j, k, l, m) : self__.afn.call(null, a, b, c, d, e, f, g, h, i, j, k, l, m);
};
cljs.core.MetaFn.prototype.cljs$core$IFn$_invoke$arity$14 = function(a, b, c, d, e, f, g, h, i, j, k, l, m, n) {
  var self__ = this;
  var _ = this;
  return self__.afn.cljs$core$IFn$_invoke$arity$14 ? self__.afn.cljs$core$IFn$_invoke$arity$14(a, b, c, d, e, f, g, h, i, j, k, l, m, n) : self__.afn.call(null, a, b, c, d, e, f, g, h, i, j, k, l, m, n);
};
cljs.core.MetaFn.prototype.cljs$core$IFn$_invoke$arity$15 = function(a, b, c, d, e, f, g, h, i, j, k, l, m, n, o) {
  var self__ = this;
  var _ = this;
  return self__.afn.cljs$core$IFn$_invoke$arity$15 ? self__.afn.cljs$core$IFn$_invoke$arity$15(a, b, c, d, e, f, g, h, i, j, k, l, m, n, o) : self__.afn.call(null, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o);
};
cljs.core.MetaFn.prototype.cljs$core$IFn$_invoke$arity$16 = function(a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p) {
  var self__ = this;
  var _ = this;
  return self__.afn.cljs$core$IFn$_invoke$arity$16 ? self__.afn.cljs$core$IFn$_invoke$arity$16(a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p) : self__.afn.call(null, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p);
};
cljs.core.MetaFn.prototype.cljs$core$IFn$_invoke$arity$17 = function(a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q) {
  var self__ = this;
  var _ = this;
  return self__.afn.cljs$core$IFn$_invoke$arity$17 ? self__.afn.cljs$core$IFn$_invoke$arity$17(a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q) : self__.afn.call(null, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q);
};
cljs.core.MetaFn.prototype.cljs$core$IFn$_invoke$arity$18 = function(a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r) {
  var self__ = this;
  var _ = this;
  return self__.afn.cljs$core$IFn$_invoke$arity$18 ? self__.afn.cljs$core$IFn$_invoke$arity$18(a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r) : self__.afn.call(null, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r);
};
cljs.core.MetaFn.prototype.cljs$core$IFn$_invoke$arity$19 = function(a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s) {
  var self__ = this;
  var _ = this;
  return self__.afn.cljs$core$IFn$_invoke$arity$19 ? self__.afn.cljs$core$IFn$_invoke$arity$19(a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s) : self__.afn.call(null, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s);
};
cljs.core.MetaFn.prototype.cljs$core$IFn$_invoke$arity$20 = function(a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s, t) {
  var self__ = this;
  var _ = this;
  return self__.afn.cljs$core$IFn$_invoke$arity$20 ? self__.afn.cljs$core$IFn$_invoke$arity$20(a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s, t) : self__.afn.call(null, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s, t);
};
cljs.core.MetaFn.prototype.cljs$core$IFn$_invoke$arity$21 = function(a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s, t, rest) {
  var self__ = this;
  var _ = this;
  return cljs.core.apply.cljs$core$IFn$_invoke$arity$22 ? cljs.core.apply.cljs$core$IFn$_invoke$arity$22(self__.afn, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s, t, rest) : cljs.core.apply.call(null, self__.afn, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s, t, rest);
};
cljs.core.MetaFn.getBasis = function() {
  return new cljs.core.PersistentVector(null, 2, 5, cljs.core.PersistentVector.EMPTY_NODE, [new cljs.core.Symbol(null, "afn", "afn", 216963467, null), new cljs.core.Symbol(null, "meta", "meta", -1154898805, null)], null);
};
cljs.core.MetaFn.cljs$lang$type = true;
cljs.core.MetaFn.cljs$lang$ctorStr = "cljs.core/MetaFn";
cljs.core.MetaFn.cljs$lang$ctorPrWriter = function(this__7871__auto__, writer__7872__auto__, opt__7873__auto__) {
  return cljs.core._write(writer__7872__auto__, "cljs.core/MetaFn");
};
cljs.core.__GT_MetaFn = function cljs$core$__GT_MetaFn(afn, meta) {
  return new cljs.core.MetaFn(afn, meta);
};
cljs.core.with_meta = function cljs$core$with_meta(o, meta) {
  if (goog.isFunction(o)) {
    return new cljs.core.MetaFn(o, meta);
  } else {
    if (o == null) {
      return null;
    } else {
      return cljs.core._with_meta(o, meta);
    }
  }
};
cljs.core.meta = function cljs$core$meta(o) {
  if (function() {
    var and__7248__auto__ = !(o == null);
    if (and__7248__auto__) {
      if (!(o == null)) {
        if (o.cljs$lang$protocol_mask$partition0$ & 131072 || cljs.core.PROTOCOL_SENTINEL === o.cljs$core$IMeta$) {
          return true;
        } else {
          if (!o.cljs$lang$protocol_mask$partition0$) {
            return cljs.core.native_satisfies_QMARK_(cljs.core.IMeta, o);
          } else {
            return false;
          }
        }
      } else {
        return cljs.core.native_satisfies_QMARK_(cljs.core.IMeta, o);
      }
    } else {
      return and__7248__auto__;
    }
  }()) {
    return cljs.core._meta(o);
  } else {
    return null;
  }
};
cljs.core.peek = function cljs$core$peek(coll) {
  if (coll == null) {
    return null;
  } else {
    return cljs.core._peek(coll);
  }
};
cljs.core.pop = function cljs$core$pop(coll) {
  if (coll == null) {
    return null;
  } else {
    return cljs.core._pop(coll);
  }
};
cljs.core.disj = function cljs$core$disj(var_args) {
  var args9962 = [];
  var len__8373__auto___9968 = arguments.length;
  var i__8374__auto___9969 = 0;
  while (true) {
    if (i__8374__auto___9969 < len__8373__auto___9968) {
      args9962.push(arguments[i__8374__auto___9969]);
      var G__9970 = i__8374__auto___9969 + 1;
      i__8374__auto___9969 = G__9970;
      continue;
    } else {
    }
    break;
  }
  var G__9967 = args9962.length;
  switch(G__9967) {
    case 1:
      return cljs.core.disj.cljs$core$IFn$_invoke$arity$1(arguments[0]);
      break;
    case 2:
      return cljs.core.disj.cljs$core$IFn$_invoke$arity$2(arguments[0], arguments[1]);
      break;
    default:
      var argseq__8392__auto__ = new cljs.core.IndexedSeq(args9962.slice(2), 0, null);
      return cljs.core.disj.cljs$core$IFn$_invoke$arity$variadic(arguments[0], arguments[1], argseq__8392__auto__);
  }
};
cljs.core.disj.cljs$core$IFn$_invoke$arity$1 = function(coll) {
  return coll;
};
cljs.core.disj.cljs$core$IFn$_invoke$arity$2 = function(coll, k) {
  if (coll == null) {
    return null;
  } else {
    return cljs.core._disjoin(coll, k);
  }
};
cljs.core.disj.cljs$core$IFn$_invoke$arity$variadic = function(coll, k, ks) {
  while (true) {
    if (coll == null) {
      return null;
    } else {
      var ret = cljs.core.disj.cljs$core$IFn$_invoke$arity$2(coll, k);
      if (cljs.core.truth_(ks)) {
        var G__9972 = ret;
        var G__9973 = cljs.core.first(ks);
        var G__9974 = cljs.core.next(ks);
        coll = G__9972;
        k = G__9973;
        ks = G__9974;
        continue;
      } else {
        return ret;
      }
    }
    break;
  }
};
cljs.core.disj.cljs$lang$applyTo = function(seq9963) {
  var G__9964 = cljs.core.first(seq9963);
  var seq9963__$1 = cljs.core.next(seq9963);
  var G__9965 = cljs.core.first(seq9963__$1);
  var seq9963__$2 = cljs.core.next(seq9963__$1);
  return cljs.core.disj.cljs$core$IFn$_invoke$arity$variadic(G__9964, G__9965, seq9963__$2);
};
cljs.core.disj.cljs$lang$maxFixedArity = 2;
cljs.core.empty_QMARK_ = function cljs$core$empty_QMARK_(coll) {
  return coll == null || cljs.core.not(cljs.core.seq(coll));
};
cljs.core.coll_QMARK_ = function cljs$core$coll_QMARK_(x) {
  if (x == null) {
    return false;
  } else {
    if (!(x == null)) {
      if (x.cljs$lang$protocol_mask$partition0$ & 8 || cljs.core.PROTOCOL_SENTINEL === x.cljs$core$ICollection$) {
        return true;
      } else {
        if (!x.cljs$lang$protocol_mask$partition0$) {
          return cljs.core.native_satisfies_QMARK_(cljs.core.ICollection, x);
        } else {
          return false;
        }
      }
    } else {
      return cljs.core.native_satisfies_QMARK_(cljs.core.ICollection, x);
    }
  }
};
cljs.core.set_QMARK_ = function cljs$core$set_QMARK_(x) {
  if (x == null) {
    return false;
  } else {
    if (!(x == null)) {
      if (x.cljs$lang$protocol_mask$partition0$ & 4096 || cljs.core.PROTOCOL_SENTINEL === x.cljs$core$ISet$) {
        return true;
      } else {
        if (!x.cljs$lang$protocol_mask$partition0$) {
          return cljs.core.native_satisfies_QMARK_(cljs.core.ISet, x);
        } else {
          return false;
        }
      }
    } else {
      return cljs.core.native_satisfies_QMARK_(cljs.core.ISet, x);
    }
  }
};
cljs.core.associative_QMARK_ = function cljs$core$associative_QMARK_(x) {
  if (!(x == null)) {
    if (x.cljs$lang$protocol_mask$partition0$ & 512 || cljs.core.PROTOCOL_SENTINEL === x.cljs$core$IAssociative$) {
      return true;
    } else {
      if (!x.cljs$lang$protocol_mask$partition0$) {
        return cljs.core.native_satisfies_QMARK_(cljs.core.IAssociative, x);
      } else {
        return false;
      }
    }
  } else {
    return cljs.core.native_satisfies_QMARK_(cljs.core.IAssociative, x);
  }
};
cljs.core.ifind_QMARK_ = function cljs$core$ifind_QMARK_(x) {
  if (!(x == null)) {
    if (false || cljs.core.PROTOCOL_SENTINEL === x.cljs$core$IFind$) {
      return true;
    } else {
      if (!x.cljs$lang$protocol_mask$partition$) {
        return cljs.core.native_satisfies_QMARK_(cljs.core.IFind, x);
      } else {
        return false;
      }
    }
  } else {
    return cljs.core.native_satisfies_QMARK_(cljs.core.IFind, x);
  }
};
cljs.core.sequential_QMARK_ = function cljs$core$sequential_QMARK_(x) {
  if (!(x == null)) {
    if (x.cljs$lang$protocol_mask$partition0$ & 16777216 || cljs.core.PROTOCOL_SENTINEL === x.cljs$core$ISequential$) {
      return true;
    } else {
      if (!x.cljs$lang$protocol_mask$partition0$) {
        return cljs.core.native_satisfies_QMARK_(cljs.core.ISequential, x);
      } else {
        return false;
      }
    }
  } else {
    return cljs.core.native_satisfies_QMARK_(cljs.core.ISequential, x);
  }
};
cljs.core.sorted_QMARK_ = function cljs$core$sorted_QMARK_(x) {
  if (!(x == null)) {
    if (x.cljs$lang$protocol_mask$partition0$ & 268435456 || cljs.core.PROTOCOL_SENTINEL === x.cljs$core$ISorted$) {
      return true;
    } else {
      if (!x.cljs$lang$protocol_mask$partition0$) {
        return cljs.core.native_satisfies_QMARK_(cljs.core.ISorted, x);
      } else {
        return false;
      }
    }
  } else {
    return cljs.core.native_satisfies_QMARK_(cljs.core.ISorted, x);
  }
};
cljs.core.reduceable_QMARK_ = function cljs$core$reduceable_QMARK_(x) {
  if (!(x == null)) {
    if (x.cljs$lang$protocol_mask$partition0$ & 524288 || cljs.core.PROTOCOL_SENTINEL === x.cljs$core$IReduce$) {
      return true;
    } else {
      if (!x.cljs$lang$protocol_mask$partition0$) {
        return cljs.core.native_satisfies_QMARK_(cljs.core.IReduce, x);
      } else {
        return false;
      }
    }
  } else {
    return cljs.core.native_satisfies_QMARK_(cljs.core.IReduce, x);
  }
};
cljs.core.map_QMARK_ = function cljs$core$map_QMARK_(x) {
  if (x == null) {
    return false;
  } else {
    if (!(x == null)) {
      if (x.cljs$lang$protocol_mask$partition0$ & 1024 || cljs.core.PROTOCOL_SENTINEL === x.cljs$core$IMap$) {
        return true;
      } else {
        if (!x.cljs$lang$protocol_mask$partition0$) {
          return cljs.core.native_satisfies_QMARK_(cljs.core.IMap, x);
        } else {
          return false;
        }
      }
    } else {
      return cljs.core.native_satisfies_QMARK_(cljs.core.IMap, x);
    }
  }
};
cljs.core.record_QMARK_ = function cljs$core$record_QMARK_(x) {
  if (!(x == null)) {
    if (x.cljs$lang$protocol_mask$partition0$ & 67108864 || cljs.core.PROTOCOL_SENTINEL === x.cljs$core$IRecord$) {
      return true;
    } else {
      if (!x.cljs$lang$protocol_mask$partition0$) {
        return cljs.core.native_satisfies_QMARK_(cljs.core.IRecord, x);
      } else {
        return false;
      }
    }
  } else {
    return cljs.core.native_satisfies_QMARK_(cljs.core.IRecord, x);
  }
};
cljs.core.vector_QMARK_ = function cljs$core$vector_QMARK_(x) {
  if (!(x == null)) {
    if (x.cljs$lang$protocol_mask$partition0$ & 16384 || cljs.core.PROTOCOL_SENTINEL === x.cljs$core$IVector$) {
      return true;
    } else {
      if (!x.cljs$lang$protocol_mask$partition0$) {
        return cljs.core.native_satisfies_QMARK_(cljs.core.IVector, x);
      } else {
        return false;
      }
    }
  } else {
    return cljs.core.native_satisfies_QMARK_(cljs.core.IVector, x);
  }
};
cljs.core.chunked_seq_QMARK_ = function cljs$core$chunked_seq_QMARK_(x) {
  if (!(x == null)) {
    if (x.cljs$lang$protocol_mask$partition1$ & 512 || cljs.core.PROTOCOL_SENTINEL === x.cljs$core$IChunkedSeq$) {
      return true;
    } else {
      return false;
    }
  } else {
    return false;
  }
};
cljs.core.js_obj = function cljs$core$js_obj(var_args) {
  var args9997 = [];
  var len__8373__auto___10003 = arguments.length;
  var i__8374__auto___10004 = 0;
  while (true) {
    if (i__8374__auto___10004 < len__8373__auto___10003) {
      args9997.push(arguments[i__8374__auto___10004]);
      var G__10005 = i__8374__auto___10004 + 1;
      i__8374__auto___10004 = G__10005;
      continue;
    } else {
    }
    break;
  }
  var G__10000 = args9997.length;
  switch(G__10000) {
    case 0:
      return cljs.core.js_obj.cljs$core$IFn$_invoke$arity$0();
      break;
    default:
      var argseq__8392__auto__ = new cljs.core.IndexedSeq(args9997.slice(0), 0, null);
      return cljs.core.js_obj.cljs$core$IFn$_invoke$arity$variadic(argseq__8392__auto__);
  }
};
cljs.core.js_obj.cljs$core$IFn$_invoke$arity$0 = function() {
  return {};
};
cljs.core.js_obj.cljs$core$IFn$_invoke$arity$variadic = function(keyvals) {
  return cljs.core.apply.cljs$core$IFn$_invoke$arity$2 ? cljs.core.apply.cljs$core$IFn$_invoke$arity$2(goog.object.create, keyvals) : cljs.core.apply.call(null, goog.object.create, keyvals);
};
cljs.core.js_obj.cljs$lang$applyTo = function(seq9998) {
  return cljs.core.js_obj.cljs$core$IFn$_invoke$arity$variadic(cljs.core.seq(seq9998));
};
cljs.core.js_obj.cljs$lang$maxFixedArity = 0;
cljs.core.js_keys = function cljs$core$js_keys(obj) {
  var keys = [];
  var G__10009_10011 = obj;
  var G__10010_10012 = function(G__10009_10011, keys) {
    return function(val, key, obj__$1) {
      return keys.push(key);
    };
  }(G__10009_10011, keys);
  goog.object.forEach(G__10009_10011, G__10010_10012);
  return keys;
};
cljs.core.js_delete = function cljs$core$js_delete(obj, key) {
  return delete obj[key];
};
cljs.core.array_copy = function cljs$core$array_copy(from, i, to, j, len) {
  var i__$1 = i;
  var j__$1 = j;
  var len__$1 = len;
  while (true) {
    if (len__$1 === 0) {
      return to;
    } else {
      to[j__$1] = from[i__$1];
      var G__10013 = i__$1 + 1;
      var G__10014 = j__$1 + 1;
      var G__10015 = len__$1 - 1;
      i__$1 = G__10013;
      j__$1 = G__10014;
      len__$1 = G__10015;
      continue;
    }
    break;
  }
};
cljs.core.array_copy_downward = function cljs$core$array_copy_downward(from, i, to, j, len) {
  var i__$1 = i + (len - 1);
  var j__$1 = j + (len - 1);
  var len__$1 = len;
  while (true) {
    if (len__$1 === 0) {
      return to;
    } else {
      to[j__$1] = from[i__$1];
      var G__10016 = i__$1 - 1;
      var G__10017 = j__$1 - 1;
      var G__10018 = len__$1 - 1;
      i__$1 = G__10016;
      j__$1 = G__10017;
      len__$1 = G__10018;
      continue;
    }
    break;
  }
};
cljs.core.lookup_sentinel = {};
cljs.core.false_QMARK_ = function cljs$core$false_QMARK_(x) {
  return x === false;
};
cljs.core.true_QMARK_ = function cljs$core$true_QMARK_(x) {
  return x === true;
};
cljs.core.boolean_QMARK_ = function cljs$core$boolean_QMARK_(x) {
  return x === true || x === false;
};
cljs.core.undefined_QMARK_ = function cljs$core$undefined_QMARK_(x) {
  return void 0 === x;
};
cljs.core.seq_QMARK_ = function cljs$core$seq_QMARK_(s) {
  if (s == null) {
    return false;
  } else {
    if (!(s == null)) {
      if (s.cljs$lang$protocol_mask$partition0$ & 64 || cljs.core.PROTOCOL_SENTINEL === s.cljs$core$ISeq$) {
        return true;
      } else {
        if (!s.cljs$lang$protocol_mask$partition0$) {
          return cljs.core.native_satisfies_QMARK_(cljs.core.ISeq, s);
        } else {
          return false;
        }
      }
    } else {
      return cljs.core.native_satisfies_QMARK_(cljs.core.ISeq, s);
    }
  }
};
cljs.core.seqable_QMARK_ = function cljs$core$seqable_QMARK_(s) {
  var or__7260__auto__ = !(s == null) ? s.cljs$lang$protocol_mask$partition0$ & 8388608 || cljs.core.PROTOCOL_SENTINEL === s.cljs$core$ISeqable$ ? true : !s.cljs$lang$protocol_mask$partition0$ ? cljs.core.native_satisfies_QMARK_(cljs.core.ISeqable, s) : false : cljs.core.native_satisfies_QMARK_(cljs.core.ISeqable, s);
  if (or__7260__auto__) {
    return or__7260__auto__;
  } else {
    return cljs.core.array_QMARK_(s) || typeof s === "string";
  }
};
cljs.core.boolean$ = function cljs$core$boolean(x) {
  if (x == null) {
    return false;
  } else {
    if (x === false) {
      return false;
    } else {
      return true;
    }
  }
};
cljs.core.ifn_QMARK_ = function cljs$core$ifn_QMARK_(f) {
  var or__7260__auto__ = cljs.core.fn_QMARK_(f);
  if (or__7260__auto__) {
    return or__7260__auto__;
  } else {
    if (!(f == null)) {
      if (f.cljs$lang$protocol_mask$partition0$ & 1 || cljs.core.PROTOCOL_SENTINEL === f.cljs$core$IFn$) {
        return true;
      } else {
        if (!f.cljs$lang$protocol_mask$partition0$) {
          return cljs.core.native_satisfies_QMARK_(cljs.core.IFn, f);
        } else {
          return false;
        }
      }
    } else {
      return cljs.core.native_satisfies_QMARK_(cljs.core.IFn, f);
    }
  }
};
cljs.core.integer_QMARK_ = function cljs$core$integer_QMARK_(n) {
  return typeof n === "number" && !isNaN(n) && !(n === Infinity) && parseFloat(n) === parseInt(n, 10);
};
cljs.core.int_QMARK_ = function cljs$core$int_QMARK_(x) {
  return cljs.core.integer_QMARK_(x) || x instanceof goog.math.Integer || x instanceof goog.math.Long;
};
cljs.core.pos_int_QMARK_ = function cljs$core$pos_int_QMARK_(x) {
  if (cljs.core.integer_QMARK_(x)) {
    return x > 0;
  } else {
    if (x instanceof goog.math.Integer) {
      return cljs.core.not(x.isNegative()) && cljs.core.not(x.isZero());
    } else {
      if (x instanceof goog.math.Long) {
        return cljs.core.not(x.isNegative()) && cljs.core.not(x.isZero());
      } else {
        return false;
      }
    }
  }
};
cljs.core.neg_int_QMARK_ = function cljs$core$neg_int_QMARK_(x) {
  if (cljs.core.integer_QMARK_(x)) {
    return x < 0;
  } else {
    if (x instanceof goog.math.Integer) {
      return x.isNegative();
    } else {
      if (x instanceof goog.math.Long) {
        return x.isNegative();
      } else {
        return false;
      }
    }
  }
};
cljs.core.nat_int_QMARK_ = function cljs$core$nat_int_QMARK_(x) {
  if (cljs.core.integer_QMARK_(x)) {
    return !(x < 0) || x === 0;
  } else {
    if (x instanceof goog.math.Integer) {
      var or__7260__auto__ = cljs.core.not(x.isNegative());
      if (or__7260__auto__) {
        return or__7260__auto__;
      } else {
        return x.isZero();
      }
    } else {
      if (x instanceof goog.math.Long) {
        var or__7260__auto__ = cljs.core.not(x.isNegative());
        if (or__7260__auto__) {
          return or__7260__auto__;
        } else {
          return x.isZero();
        }
      } else {
        return false;
      }
    }
  }
};
cljs.core.float_QMARK_ = function cljs$core$float_QMARK_(x) {
  return typeof x === "number";
};
cljs.core.double_QMARK_ = function cljs$core$double_QMARK_(x) {
  return typeof x === "number";
};
cljs.core.infinite_QMARK_ = function cljs$core$infinite_QMARK_(x) {
  return x === Number.POSITIVE_INFINITY || x === Number.NEGATIVE_INFINITY;
};
cljs.core.contains_QMARK_ = function cljs$core$contains_QMARK_(coll, v) {
  if (cljs.core.get.cljs$core$IFn$_invoke$arity$3(coll, v, cljs.core.lookup_sentinel) === cljs.core.lookup_sentinel) {
    return false;
  } else {
    return true;
  }
};
cljs.core.find = function cljs$core$find(coll, k) {
  if (!(coll == null) && cljs.core.associative_QMARK_(coll) && cljs.core.contains_QMARK_(coll, k)) {
    if (cljs.core.ifind_QMARK_(coll)) {
      return cljs.core._find(coll, k);
    } else {
      return new cljs.core.PersistentVector(null, 2, 5, cljs.core.PersistentVector.EMPTY_NODE, [k, cljs.core.get.cljs$core$IFn$_invoke$arity$2(coll, k)], null);
    }
  } else {
    return null;
  }
};
cljs.core.distinct_QMARK_ = function cljs$core$distinct_QMARK_(var_args) {
  var args10031 = [];
  var len__8373__auto___10037 = arguments.length;
  var i__8374__auto___10038 = 0;
  while (true) {
    if (i__8374__auto___10038 < len__8373__auto___10037) {
      args10031.push(arguments[i__8374__auto___10038]);
      var G__10039 = i__8374__auto___10038 + 1;
      i__8374__auto___10038 = G__10039;
      continue;
    } else {
    }
    break;
  }
  var G__10036 = args10031.length;
  switch(G__10036) {
    case 1:
      return cljs.core.distinct_QMARK_.cljs$core$IFn$_invoke$arity$1(arguments[0]);
      break;
    case 2:
      return cljs.core.distinct_QMARK_.cljs$core$IFn$_invoke$arity$2(arguments[0], arguments[1]);
      break;
    default:
      var argseq__8392__auto__ = new cljs.core.IndexedSeq(args10031.slice(2), 0, null);
      return cljs.core.distinct_QMARK_.cljs$core$IFn$_invoke$arity$variadic(arguments[0], arguments[1], argseq__8392__auto__);
  }
};
cljs.core.distinct_QMARK_.cljs$core$IFn$_invoke$arity$1 = function(x) {
  return true;
};
cljs.core.distinct_QMARK_.cljs$core$IFn$_invoke$arity$2 = function(x, y) {
  return !cljs.core._EQ_.cljs$core$IFn$_invoke$arity$2(x, y);
};
cljs.core.distinct_QMARK_.cljs$core$IFn$_invoke$arity$variadic = function(x, y, more) {
  if (!cljs.core._EQ_.cljs$core$IFn$_invoke$arity$2(x, y)) {
    var s = cljs.core.PersistentHashSet.createAsIfByAssoc([x, y], true);
    var xs = more;
    while (true) {
      var x__$1 = cljs.core.first(xs);
      var etc = cljs.core.next(xs);
      if (cljs.core.truth_(xs)) {
        if (cljs.core.contains_QMARK_(s, x__$1)) {
          return false;
        } else {
          var G__10041 = cljs.core.conj.cljs$core$IFn$_invoke$arity$2(s, x__$1);
          var G__10042 = etc;
          s = G__10041;
          xs = G__10042;
          continue;
        }
      } else {
        return true;
      }
      break;
    }
  } else {
    return false;
  }
};
cljs.core.distinct_QMARK_.cljs$lang$applyTo = function(seq10032) {
  var G__10033 = cljs.core.first(seq10032);
  var seq10032__$1 = cljs.core.next(seq10032);
  var G__10034 = cljs.core.first(seq10032__$1);
  var seq10032__$2 = cljs.core.next(seq10032__$1);
  return cljs.core.distinct_QMARK_.cljs$core$IFn$_invoke$arity$variadic(G__10033, G__10034, seq10032__$2);
};
cljs.core.distinct_QMARK_.cljs$lang$maxFixedArity = 2;
cljs.core.compare = function cljs$core$compare(x, y) {
  if (x === y) {
    return 0;
  } else {
    if (x == null) {
      return -1;
    } else {
      if (y == null) {
        return 1;
      } else {
        if (typeof x === "number") {
          if (typeof y === "number") {
            return goog.array.defaultCompare(x, y);
          } else {
            throw new Error([cljs.core.str.cljs$core$IFn$_invoke$arity$1("Cannot compare "), cljs.core.str.cljs$core$IFn$_invoke$arity$1(x), cljs.core.str.cljs$core$IFn$_invoke$arity$1(" to "), cljs.core.str.cljs$core$IFn$_invoke$arity$1(y)].join(""));
          }
        } else {
          if (!(x == null) ? x.cljs$lang$protocol_mask$partition1$ & 2048 || cljs.core.PROTOCOL_SENTINEL === x.cljs$core$IComparable$ ? true : !x.cljs$lang$protocol_mask$partition1$ ? cljs.core.native_satisfies_QMARK_(cljs.core.IComparable, x) : false : cljs.core.native_satisfies_QMARK_(cljs.core.IComparable, x)) {
            return cljs.core._compare(x, y);
          } else {
            if ((typeof x === "string" || cljs.core.array_QMARK_(x) || x === true || x === false) && cljs.core.type(x) === cljs.core.type(y)) {
              return goog.array.defaultCompare(x, y);
            } else {
              throw new Error([cljs.core.str.cljs$core$IFn$_invoke$arity$1("Cannot compare "), cljs.core.str.cljs$core$IFn$_invoke$arity$1(x), cljs.core.str.cljs$core$IFn$_invoke$arity$1(" to "), cljs.core.str.cljs$core$IFn$_invoke$arity$1(y)].join(""));
            }
          }
        }
      }
    }
  }
};
cljs.core.compare_indexed = function cljs$core$compare_indexed(var_args) {
  var args10045 = [];
  var len__8373__auto___10048 = arguments.length;
  var i__8374__auto___10049 = 0;
  while (true) {
    if (i__8374__auto___10049 < len__8373__auto___10048) {
      args10045.push(arguments[i__8374__auto___10049]);
      var G__10050 = i__8374__auto___10049 + 1;
      i__8374__auto___10049 = G__10050;
      continue;
    } else {
    }
    break;
  }
  var G__10047 = args10045.length;
  switch(G__10047) {
    case 2:
      return cljs.core.compare_indexed.cljs$core$IFn$_invoke$arity$2(arguments[0], arguments[1]);
      break;
    case 4:
      return cljs.core.compare_indexed.cljs$core$IFn$_invoke$arity$4(arguments[0], arguments[1], arguments[2], arguments[3]);
      break;
    default:
      throw new Error([cljs.core.str.cljs$core$IFn$_invoke$arity$1("Invalid arity: "), cljs.core.str.cljs$core$IFn$_invoke$arity$1(args10045.length)].join(""));
  }
};
cljs.core.compare_indexed.cljs$core$IFn$_invoke$arity$2 = function(xs, ys) {
  var xl = cljs.core.count(xs);
  var yl = cljs.core.count(ys);
  if (xl < yl) {
    return -1;
  } else {
    if (xl > yl) {
      return 1;
    } else {
      if (xl === 0) {
        return 0;
      } else {
        return cljs.core.compare_indexed.cljs$core$IFn$_invoke$arity$4(xs, ys, xl, 0);
      }
    }
  }
};
cljs.core.compare_indexed.cljs$core$IFn$_invoke$arity$4 = function(xs, ys, len, n) {
  while (true) {
    var d = cljs.core.compare(cljs.core.nth.cljs$core$IFn$_invoke$arity$2(xs, n), cljs.core.nth.cljs$core$IFn$_invoke$arity$2(ys, n));
    if (d === 0 && n + 1 < len) {
      var G__10052 = xs;
      var G__10053 = ys;
      var G__10054 = len;
      var G__10055 = n + 1;
      xs = G__10052;
      ys = G__10053;
      len = G__10054;
      n = G__10055;
      continue;
    } else {
      return d;
    }
    break;
  }
};
cljs.core.compare_indexed.cljs$lang$maxFixedArity = 4;
cljs.core.fn__GT_comparator = function cljs$core$fn__GT_comparator(f) {
  if (cljs.core._EQ_.cljs$core$IFn$_invoke$arity$2(f, cljs.core.compare)) {
    return cljs.core.compare;
  } else {
    return function(x, y) {
      var r = f.cljs$core$IFn$_invoke$arity$2 ? f.cljs$core$IFn$_invoke$arity$2(x, y) : f.call(null, x, y);
      if (typeof r === "number") {
        return r;
      } else {
        if (cljs.core.truth_(r)) {
          return -1;
        } else {
          if (cljs.core.truth_(f.cljs$core$IFn$_invoke$arity$2 ? f.cljs$core$IFn$_invoke$arity$2(y, x) : f.call(null, y, x))) {
            return 1;
          } else {
            return 0;
          }
        }
      }
    };
  }
};
cljs.core.sort = function cljs$core$sort(var_args) {
  var args10056 = [];
  var len__8373__auto___10061 = arguments.length;
  var i__8374__auto___10062 = 0;
  while (true) {
    if (i__8374__auto___10062 < len__8373__auto___10061) {
      args10056.push(arguments[i__8374__auto___10062]);
      var G__10063 = i__8374__auto___10062 + 1;
      i__8374__auto___10062 = G__10063;
      continue;
    } else {
    }
    break;
  }
  var G__10058 = args10056.length;
  switch(G__10058) {
    case 1:
      return cljs.core.sort.cljs$core$IFn$_invoke$arity$1(arguments[0]);
      break;
    case 2:
      return cljs.core.sort.cljs$core$IFn$_invoke$arity$2(arguments[0], arguments[1]);
      break;
    default:
      throw new Error([cljs.core.str.cljs$core$IFn$_invoke$arity$1("Invalid arity: "), cljs.core.str.cljs$core$IFn$_invoke$arity$1(args10056.length)].join(""));
  }
};
cljs.core.sort.cljs$core$IFn$_invoke$arity$1 = function(coll) {
  return cljs.core.sort.cljs$core$IFn$_invoke$arity$2(cljs.core.compare, coll);
};
cljs.core.sort.cljs$core$IFn$_invoke$arity$2 = function(comp, coll) {
  if (cljs.core.seq(coll)) {
    var a = cljs.core.to_array.cljs$core$IFn$_invoke$arity$1 ? cljs.core.to_array.cljs$core$IFn$_invoke$arity$1(coll) : cljs.core.to_array.call(null, coll);
    var G__10059_10065 = a;
    var G__10060_10066 = cljs.core.fn__GT_comparator(comp);
    goog.array.stableSort(G__10059_10065, G__10060_10066);
    return cljs.core.seq(a);
  } else {
    return cljs.core.List.EMPTY;
  }
};
cljs.core.sort.cljs$lang$maxFixedArity = 2;
cljs.core.sort_by = function cljs$core$sort_by(var_args) {
  var args10067 = [];
  var len__8373__auto___10070 = arguments.length;
  var i__8374__auto___10071 = 0;
  while (true) {
    if (i__8374__auto___10071 < len__8373__auto___10070) {
      args10067.push(arguments[i__8374__auto___10071]);
      var G__10072 = i__8374__auto___10071 + 1;
      i__8374__auto___10071 = G__10072;
      continue;
    } else {
    }
    break;
  }
  var G__10069 = args10067.length;
  switch(G__10069) {
    case 2:
      return cljs.core.sort_by.cljs$core$IFn$_invoke$arity$2(arguments[0], arguments[1]);
      break;
    case 3:
      return cljs.core.sort_by.cljs$core$IFn$_invoke$arity$3(arguments[0], arguments[1], arguments[2]);
      break;
    default:
      throw new Error([cljs.core.str.cljs$core$IFn$_invoke$arity$1("Invalid arity: "), cljs.core.str.cljs$core$IFn$_invoke$arity$1(args10067.length)].join(""));
  }
};
cljs.core.sort_by.cljs$core$IFn$_invoke$arity$2 = function(keyfn, coll) {
  return cljs.core.sort_by.cljs$core$IFn$_invoke$arity$3(keyfn, cljs.core.compare, coll);
};
cljs.core.sort_by.cljs$core$IFn$_invoke$arity$3 = function(keyfn, comp, coll) {
  return cljs.core.sort.cljs$core$IFn$_invoke$arity$2(function(x, y) {
    return cljs.core.fn__GT_comparator(comp).call(null, keyfn.cljs$core$IFn$_invoke$arity$1 ? keyfn.cljs$core$IFn$_invoke$arity$1(x) : keyfn.call(null, x), keyfn.cljs$core$IFn$_invoke$arity$1 ? keyfn.cljs$core$IFn$_invoke$arity$1(y) : keyfn.call(null, y));
  }, coll);
};
cljs.core.sort_by.cljs$lang$maxFixedArity = 3;
cljs.core.seq_reduce = function cljs$core$seq_reduce(var_args) {
  var args10074 = [];
  var len__8373__auto___10082 = arguments.length;
  var i__8374__auto___10083 = 0;
  while (true) {
    if (i__8374__auto___10083 < len__8373__auto___10082) {
      args10074.push(arguments[i__8374__auto___10083]);
      var G__10084 = i__8374__auto___10083 + 1;
      i__8374__auto___10083 = G__10084;
      continue;
    } else {
    }
    break;
  }
  var G__10076 = args10074.length;
  switch(G__10076) {
    case 2:
      return cljs.core.seq_reduce.cljs$core$IFn$_invoke$arity$2(arguments[0], arguments[1]);
      break;
    case 3:
      return cljs.core.seq_reduce.cljs$core$IFn$_invoke$arity$3(arguments[0], arguments[1], arguments[2]);
      break;
    default:
      throw new Error([cljs.core.str.cljs$core$IFn$_invoke$arity$1("Invalid arity: "), cljs.core.str.cljs$core$IFn$_invoke$arity$1(args10074.length)].join(""));
  }
};
cljs.core.seq_reduce.cljs$core$IFn$_invoke$arity$2 = function(f, coll) {
  var temp__4655__auto__ = cljs.core.seq(coll);
  if (temp__4655__auto__) {
    var s = temp__4655__auto__;
    var G__10077 = f;
    var G__10078 = cljs.core.first(s);
    var G__10079 = cljs.core.next(s);
    return cljs.core.reduce.cljs$core$IFn$_invoke$arity$3 ? cljs.core.reduce.cljs$core$IFn$_invoke$arity$3(G__10077, G__10078, G__10079) : cljs.core.reduce.call(null, G__10077, G__10078, G__10079);
  } else {
    return f.cljs$core$IFn$_invoke$arity$0 ? f.cljs$core$IFn$_invoke$arity$0() : f.call(null);
  }
};
cljs.core.seq_reduce.cljs$core$IFn$_invoke$arity$3 = function(f, val, coll) {
  var val__$1 = val;
  var coll__$1 = cljs.core.seq(coll);
  while (true) {
    if (coll__$1) {
      var nval = function() {
        var G__10080 = val__$1;
        var G__10081 = cljs.core.first(coll__$1);
        return f.cljs$core$IFn$_invoke$arity$2 ? f.cljs$core$IFn$_invoke$arity$2(G__10080, G__10081) : f.call(null, G__10080, G__10081);
      }();
      if (cljs.core.reduced_QMARK_(nval)) {
        return cljs.core.deref(nval);
      } else {
        var G__10086 = nval;
        var G__10087 = cljs.core.next(coll__$1);
        val__$1 = G__10086;
        coll__$1 = G__10087;
        continue;
      }
    } else {
      return val__$1;
    }
    break;
  }
};
cljs.core.seq_reduce.cljs$lang$maxFixedArity = 3;
cljs.core.shuffle = function cljs$core$shuffle(coll) {
  var a = cljs.core.to_array.cljs$core$IFn$_invoke$arity$1 ? cljs.core.to_array.cljs$core$IFn$_invoke$arity$1(coll) : cljs.core.to_array.call(null, coll);
  goog.array.shuffle(a);
  return cljs.core.vec.cljs$core$IFn$_invoke$arity$1 ? cljs.core.vec.cljs$core$IFn$_invoke$arity$1(a) : cljs.core.vec.call(null, a);
};
cljs.core.reduce = function cljs$core$reduce(var_args) {
  var args10088 = [];
  var len__8373__auto___10093 = arguments.length;
  var i__8374__auto___10094 = 0;
  while (true) {
    if (i__8374__auto___10094 < len__8373__auto___10093) {
      args10088.push(arguments[i__8374__auto___10094]);
      var G__10095 = i__8374__auto___10094 + 1;
      i__8374__auto___10094 = G__10095;
      continue;
    } else {
    }
    break;
  }
  var G__10090 = args10088.length;
  switch(G__10090) {
    case 2:
      return cljs.core.reduce.cljs$core$IFn$_invoke$arity$2(arguments[0], arguments[1]);
      break;
    case 3:
      return cljs.core.reduce.cljs$core$IFn$_invoke$arity$3(arguments[0], arguments[1], arguments[2]);
      break;
    default:
      throw new Error([cljs.core.str.cljs$core$IFn$_invoke$arity$1("Invalid arity: "), cljs.core.str.cljs$core$IFn$_invoke$arity$1(args10088.length)].join(""));
  }
};
cljs.core.reduce.cljs$core$IFn$_invoke$arity$2 = function(f, coll) {
  if (!(coll == null) ? coll.cljs$lang$protocol_mask$partition0$ & 524288 || cljs.core.PROTOCOL_SENTINEL === coll.cljs$core$IReduce$ ? true : false : false) {
    return coll.cljs$core$IReduce$_reduce$arity$2(null, f);
  } else {
    if (cljs.core.array_QMARK_(coll)) {
      return cljs.core.array_reduce.cljs$core$IFn$_invoke$arity$2(coll, f);
    } else {
      if (typeof coll === "string") {
        return cljs.core.array_reduce.cljs$core$IFn$_invoke$arity$2(coll, f);
      } else {
        if (cljs.core.native_satisfies_QMARK_(cljs.core.IReduce, coll)) {
          return cljs.core._reduce.cljs$core$IFn$_invoke$arity$2(coll, f);
        } else {
          return cljs.core.seq_reduce.cljs$core$IFn$_invoke$arity$2(f, coll);
        }
      }
    }
  }
};
cljs.core.reduce.cljs$core$IFn$_invoke$arity$3 = function(f, val, coll) {
  if (!(coll == null) ? coll.cljs$lang$protocol_mask$partition0$ & 524288 || cljs.core.PROTOCOL_SENTINEL === coll.cljs$core$IReduce$ ? true : false : false) {
    return coll.cljs$core$IReduce$_reduce$arity$3(null, f, val);
  } else {
    if (cljs.core.array_QMARK_(coll)) {
      return cljs.core.array_reduce.cljs$core$IFn$_invoke$arity$3(coll, f, val);
    } else {
      if (typeof coll === "string") {
        return cljs.core.array_reduce.cljs$core$IFn$_invoke$arity$3(coll, f, val);
      } else {
        if (cljs.core.native_satisfies_QMARK_(cljs.core.IReduce, coll)) {
          return cljs.core._reduce.cljs$core$IFn$_invoke$arity$3(coll, f, val);
        } else {
          return cljs.core.seq_reduce.cljs$core$IFn$_invoke$arity$3(f, val, coll);
        }
      }
    }
  }
};
cljs.core.reduce.cljs$lang$maxFixedArity = 3;
cljs.core.reduce_kv = function cljs$core$reduce_kv(f, init, coll) {
  if (!(coll == null)) {
    return cljs.core._kv_reduce(coll, f, init);
  } else {
    return init;
  }
};
cljs.core.identity = function cljs$core$identity(x) {
  return x;
};
cljs.core.completing = function cljs$core$completing(var_args) {
  var args10097 = [];
  var len__8373__auto___10100 = arguments.length;
  var i__8374__auto___10101 = 0;
  while (true) {
    if (i__8374__auto___10101 < len__8373__auto___10100) {
      args10097.push(arguments[i__8374__auto___10101]);
      var G__10102 = i__8374__auto___10101 + 1;
      i__8374__auto___10101 = G__10102;
      continue;
    } else {
    }
    break;
  }
  var G__10099 = args10097.length;
  switch(G__10099) {
    case 1:
      return cljs.core.completing.cljs$core$IFn$_invoke$arity$1(arguments[0]);
      break;
    case 2:
      return cljs.core.completing.cljs$core$IFn$_invoke$arity$2(arguments[0], arguments[1]);
      break;
    default:
      throw new Error([cljs.core.str.cljs$core$IFn$_invoke$arity$1("Invalid arity: "), cljs.core.str.cljs$core$IFn$_invoke$arity$1(args10097.length)].join(""));
  }
};
cljs.core.completing.cljs$core$IFn$_invoke$arity$1 = function(f) {
  return cljs.core.completing.cljs$core$IFn$_invoke$arity$2(f, cljs.core.identity);
};
cljs.core.completing.cljs$core$IFn$_invoke$arity$2 = function(f, cf) {
  return function() {
    var G__10104 = null;
    var G__10104__0 = function() {
      return f.cljs$core$IFn$_invoke$arity$0 ? f.cljs$core$IFn$_invoke$arity$0() : f.call(null);
    };
    var G__10104__1 = function(x) {
      return cf.cljs$core$IFn$_invoke$arity$1 ? cf.cljs$core$IFn$_invoke$arity$1(x) : cf.call(null, x);
    };
    var G__10104__2 = function(x, y) {
      return f.cljs$core$IFn$_invoke$arity$2 ? f.cljs$core$IFn$_invoke$arity$2(x, y) : f.call(null, x, y);
    };
    G__10104 = function(x, y) {
      switch(arguments.length) {
        case 0:
          return G__10104__0.call(this);
        case 1:
          return G__10104__1.call(this, x);
        case 2:
          return G__10104__2.call(this, x, y);
      }
      throw new Error("Invalid arity: " + (arguments.length - 1));
    };
    G__10104.cljs$core$IFn$_invoke$arity$0 = G__10104__0;
    G__10104.cljs$core$IFn$_invoke$arity$1 = G__10104__1;
    G__10104.cljs$core$IFn$_invoke$arity$2 = G__10104__2;
    return G__10104;
  }();
};
cljs.core.completing.cljs$lang$maxFixedArity = 2;
cljs.core.transduce = function cljs$core$transduce(var_args) {
  var args10105 = [];
  var len__8373__auto___10108 = arguments.length;
  var i__8374__auto___10109 = 0;
  while (true) {
    if (i__8374__auto___10109 < len__8373__auto___10108) {
      args10105.push(arguments[i__8374__auto___10109]);
      var G__10110 = i__8374__auto___10109 + 1;
      i__8374__auto___10109 = G__10110;
      continue;
    } else {
    }
    break;
  }
  var G__10107 = args10105.length;
  switch(G__10107) {
    case 3:
      return cljs.core.transduce.cljs$core$IFn$_invoke$arity$3(arguments[0], arguments[1], arguments[2]);
      break;
    case 4:
      return cljs.core.transduce.cljs$core$IFn$_invoke$arity$4(arguments[0], arguments[1], arguments[2], arguments[3]);
      break;
    default:
      throw new Error([cljs.core.str.cljs$core$IFn$_invoke$arity$1("Invalid arity: "), cljs.core.str.cljs$core$IFn$_invoke$arity$1(args10105.length)].join(""));
  }
};
cljs.core.transduce.cljs$core$IFn$_invoke$arity$3 = function(xform, f, coll) {
  return cljs.core.transduce.cljs$core$IFn$_invoke$arity$4(xform, f, f.cljs$core$IFn$_invoke$arity$0 ? f.cljs$core$IFn$_invoke$arity$0() : f.call(null), coll);
};
cljs.core.transduce.cljs$core$IFn$_invoke$arity$4 = function(xform, f, init, coll) {
  var f__$1 = xform.cljs$core$IFn$_invoke$arity$1 ? xform.cljs$core$IFn$_invoke$arity$1(f) : xform.call(null, f);
  var ret = cljs.core.reduce.cljs$core$IFn$_invoke$arity$3(f__$1, init, coll);
  return f__$1.cljs$core$IFn$_invoke$arity$1 ? f__$1.cljs$core$IFn$_invoke$arity$1(ret) : f__$1.call(null, ret);
};
cljs.core.transduce.cljs$lang$maxFixedArity = 4;
cljs.core._PLUS_ = function cljs$core$_PLUS_(var_args) {
  var args10112 = [];
  var len__8373__auto___10118 = arguments.length;
  var i__8374__auto___10119 = 0;
  while (true) {
    if (i__8374__auto___10119 < len__8373__auto___10118) {
      args10112.push(arguments[i__8374__auto___10119]);
      var G__10120 = i__8374__auto___10119 + 1;
      i__8374__auto___10119 = G__10120;
      continue;
    } else {
    }
    break;
  }
  var G__10117 = args10112.length;
  switch(G__10117) {
    case 0:
      return cljs.core._PLUS_.cljs$core$IFn$_invoke$arity$0();
      break;
    case 1:
      return cljs.core._PLUS_.cljs$core$IFn$_invoke$arity$1(arguments[0]);
      break;
    case 2:
      return cljs.core._PLUS_.cljs$core$IFn$_invoke$arity$2(arguments[0], arguments[1]);
      break;
    default:
      var argseq__8392__auto__ = new cljs.core.IndexedSeq(args10112.slice(2), 0, null);
      return cljs.core._PLUS_.cljs$core$IFn$_invoke$arity$variadic(arguments[0], arguments[1], argseq__8392__auto__);
  }
};
cljs.core._PLUS_.cljs$core$IFn$_invoke$arity$0 = function() {
  return 0;
};
cljs.core._PLUS_.cljs$core$IFn$_invoke$arity$1 = function(x) {
  return x;
};
cljs.core._PLUS_.cljs$core$IFn$_invoke$arity$2 = function(x, y) {
  return x + y;
};
cljs.core._PLUS_.cljs$core$IFn$_invoke$arity$variadic = function(x, y, more) {
  return cljs.core.reduce.cljs$core$IFn$_invoke$arity$3(cljs.core._PLUS_, x + y, more);
};
cljs.core._PLUS_.cljs$lang$applyTo = function(seq10113) {
  var G__10114 = cljs.core.first(seq10113);
  var seq10113__$1 = cljs.core.next(seq10113);
  var G__10115 = cljs.core.first(seq10113__$1);
  var seq10113__$2 = cljs.core.next(seq10113__$1);
  return cljs.core._PLUS_.cljs$core$IFn$_invoke$arity$variadic(G__10114, G__10115, seq10113__$2);
};
cljs.core._PLUS_.cljs$lang$maxFixedArity = 2;
cljs.core._ = function cljs$core$_(var_args) {
  var args10122 = [];
  var len__8373__auto___10128 = arguments.length;
  var i__8374__auto___10129 = 0;
  while (true) {
    if (i__8374__auto___10129 < len__8373__auto___10128) {
      args10122.push(arguments[i__8374__auto___10129]);
      var G__10130 = i__8374__auto___10129 + 1;
      i__8374__auto___10129 = G__10130;
      continue;
    } else {
    }
    break;
  }
  var G__10127 = args10122.length;
  switch(G__10127) {
    case 1:
      return cljs.core._.cljs$core$IFn$_invoke$arity$1(arguments[0]);
      break;
    case 2:
      return cljs.core._.cljs$core$IFn$_invoke$arity$2(arguments[0], arguments[1]);
      break;
    default:
      var argseq__8392__auto__ = new cljs.core.IndexedSeq(args10122.slice(2), 0, null);
      return cljs.core._.cljs$core$IFn$_invoke$arity$variadic(arguments[0], arguments[1], argseq__8392__auto__);
  }
};
cljs.core._.cljs$core$IFn$_invoke$arity$1 = function(x) {
  return -x;
};
cljs.core._.cljs$core$IFn$_invoke$arity$2 = function(x, y) {
  return x - y;
};
cljs.core._.cljs$core$IFn$_invoke$arity$variadic = function(x, y, more) {
  return cljs.core.reduce.cljs$core$IFn$_invoke$arity$3(cljs.core._, x - y, more);
};
cljs.core._.cljs$lang$applyTo = function(seq10123) {
  var G__10124 = cljs.core.first(seq10123);
  var seq10123__$1 = cljs.core.next(seq10123);
  var G__10125 = cljs.core.first(seq10123__$1);
  var seq10123__$2 = cljs.core.next(seq10123__$1);
  return cljs.core._.cljs$core$IFn$_invoke$arity$variadic(G__10124, G__10125, seq10123__$2);
};
cljs.core._.cljs$lang$maxFixedArity = 2;
cljs.core._STAR_ = function cljs$core$_STAR_(var_args) {
  var args10132 = [];
  var len__8373__auto___10138 = arguments.length;
  var i__8374__auto___10139 = 0;
  while (true) {
    if (i__8374__auto___10139 < len__8373__auto___10138) {
      args10132.push(arguments[i__8374__auto___10139]);
      var G__10140 = i__8374__auto___10139 + 1;
      i__8374__auto___10139 = G__10140;
      continue;
    } else {
    }
    break;
  }
  var G__10137 = args10132.length;
  switch(G__10137) {
    case 0:
      return cljs.core._STAR_.cljs$core$IFn$_invoke$arity$0();
      break;
    case 1:
      return cljs.core._STAR_.cljs$core$IFn$_invoke$arity$1(arguments[0]);
      break;
    case 2:
      return cljs.core._STAR_.cljs$core$IFn$_invoke$arity$2(arguments[0], arguments[1]);
      break;
    default:
      var argseq__8392__auto__ = new cljs.core.IndexedSeq(args10132.slice(2), 0, null);
      return cljs.core._STAR_.cljs$core$IFn$_invoke$arity$variadic(arguments[0], arguments[1], argseq__8392__auto__);
  }
};
cljs.core._STAR_.cljs$core$IFn$_invoke$arity$0 = function() {
  return 1;
};
cljs.core._STAR_.cljs$core$IFn$_invoke$arity$1 = function(x) {
  return x;
};
cljs.core._STAR_.cljs$core$IFn$_invoke$arity$2 = function(x, y) {
  return x * y;
};
cljs.core._STAR_.cljs$core$IFn$_invoke$arity$variadic = function(x, y, more) {
  return cljs.core.reduce.cljs$core$IFn$_invoke$arity$3(cljs.core._STAR_, x * y, more);
};
cljs.core._STAR_.cljs$lang$applyTo = function(seq10133) {
  var G__10134 = cljs.core.first(seq10133);
  var seq10133__$1 = cljs.core.next(seq10133);
  var G__10135 = cljs.core.first(seq10133__$1);
  var seq10133__$2 = cljs.core.next(seq10133__$1);
  return cljs.core._STAR_.cljs$core$IFn$_invoke$arity$variadic(G__10134, G__10135, seq10133__$2);
};
cljs.core._STAR_.cljs$lang$maxFixedArity = 2;
cljs.core._SLASH_ = function cljs$core$_SLASH_(var_args) {
  var args10142 = [];
  var len__8373__auto___10148 = arguments.length;
  var i__8374__auto___10149 = 0;
  while (true) {
    if (i__8374__auto___10149 < len__8373__auto___10148) {
      args10142.push(arguments[i__8374__auto___10149]);
      var G__10150 = i__8374__auto___10149 + 1;
      i__8374__auto___10149 = G__10150;
      continue;
    } else {
    }
    break;
  }
  var G__10147 = args10142.length;
  switch(G__10147) {
    case 1:
      return cljs.core._SLASH_.cljs$core$IFn$_invoke$arity$1(arguments[0]);
      break;
    case 2:
      return cljs.core._SLASH_.cljs$core$IFn$_invoke$arity$2(arguments[0], arguments[1]);
      break;
    default:
      var argseq__8392__auto__ = new cljs.core.IndexedSeq(args10142.slice(2), 0, null);
      return cljs.core._SLASH_.cljs$core$IFn$_invoke$arity$variadic(arguments[0], arguments[1], argseq__8392__auto__);
  }
};
cljs.core._SLASH_.cljs$core$IFn$_invoke$arity$1 = function(x) {
  return 1 / x;
};
cljs.core._SLASH_.cljs$core$IFn$_invoke$arity$2 = function(x, y) {
  return x / y;
};
cljs.core._SLASH_.cljs$core$IFn$_invoke$arity$variadic = function(x, y, more) {
  return cljs.core.reduce.cljs$core$IFn$_invoke$arity$3(cljs.core._SLASH_, x / y, more);
};
cljs.core._SLASH_.cljs$lang$applyTo = function(seq10143) {
  var G__10144 = cljs.core.first(seq10143);
  var seq10143__$1 = cljs.core.next(seq10143);
  var G__10145 = cljs.core.first(seq10143__$1);
  var seq10143__$2 = cljs.core.next(seq10143__$1);
  return cljs.core._SLASH_.cljs$core$IFn$_invoke$arity$variadic(G__10144, G__10145, seq10143__$2);
};
cljs.core._SLASH_.cljs$lang$maxFixedArity = 2;
cljs.core._LT_ = function cljs$core$_LT_(var_args) {
  var args10152 = [];
  var len__8373__auto___10158 = arguments.length;
  var i__8374__auto___10159 = 0;
  while (true) {
    if (i__8374__auto___10159 < len__8373__auto___10158) {
      args10152.push(arguments[i__8374__auto___10159]);
      var G__10160 = i__8374__auto___10159 + 1;
      i__8374__auto___10159 = G__10160;
      continue;
    } else {
    }
    break;
  }
  var G__10157 = args10152.length;
  switch(G__10157) {
    case 1:
      return cljs.core._LT_.cljs$core$IFn$_invoke$arity$1(arguments[0]);
      break;
    case 2:
      return cljs.core._LT_.cljs$core$IFn$_invoke$arity$2(arguments[0], arguments[1]);
      break;
    default:
      var argseq__8392__auto__ = new cljs.core.IndexedSeq(args10152.slice(2), 0, null);
      return cljs.core._LT_.cljs$core$IFn$_invoke$arity$variadic(arguments[0], arguments[1], argseq__8392__auto__);
  }
};
cljs.core._LT_.cljs$core$IFn$_invoke$arity$1 = function(x) {
  return true;
};
cljs.core._LT_.cljs$core$IFn$_invoke$arity$2 = function(x, y) {
  return x < y;
};
cljs.core._LT_.cljs$core$IFn$_invoke$arity$variadic = function(x, y, more) {
  while (true) {
    if (x < y) {
      if (cljs.core.next(more)) {
        var G__10162 = y;
        var G__10163 = cljs.core.first(more);
        var G__10164 = cljs.core.next(more);
        x = G__10162;
        y = G__10163;
        more = G__10164;
        continue;
      } else {
        return y < cljs.core.first(more);
      }
    } else {
      return false;
    }
    break;
  }
};
cljs.core._LT_.cljs$lang$applyTo = function(seq10153) {
  var G__10154 = cljs.core.first(seq10153);
  var seq10153__$1 = cljs.core.next(seq10153);
  var G__10155 = cljs.core.first(seq10153__$1);
  var seq10153__$2 = cljs.core.next(seq10153__$1);
  return cljs.core._LT_.cljs$core$IFn$_invoke$arity$variadic(G__10154, G__10155, seq10153__$2);
};
cljs.core._LT_.cljs$lang$maxFixedArity = 2;
cljs.core._LT__EQ_ = function cljs$core$_LT__EQ_(var_args) {
  var args10165 = [];
  var len__8373__auto___10171 = arguments.length;
  var i__8374__auto___10172 = 0;
  while (true) {
    if (i__8374__auto___10172 < len__8373__auto___10171) {
      args10165.push(arguments[i__8374__auto___10172]);
      var G__10173 = i__8374__auto___10172 + 1;
      i__8374__auto___10172 = G__10173;
      continue;
    } else {
    }
    break;
  }
  var G__10170 = args10165.length;
  switch(G__10170) {
    case 1:
      return cljs.core._LT__EQ_.cljs$core$IFn$_invoke$arity$1(arguments[0]);
      break;
    case 2:
      return cljs.core._LT__EQ_.cljs$core$IFn$_invoke$arity$2(arguments[0], arguments[1]);
      break;
    default:
      var argseq__8392__auto__ = new cljs.core.IndexedSeq(args10165.slice(2), 0, null);
      return cljs.core._LT__EQ_.cljs$core$IFn$_invoke$arity$variadic(arguments[0], arguments[1], argseq__8392__auto__);
  }
};
cljs.core._LT__EQ_.cljs$core$IFn$_invoke$arity$1 = function(x) {
  return true;
};
cljs.core._LT__EQ_.cljs$core$IFn$_invoke$arity$2 = function(x, y) {
  return x <= y;
};
cljs.core._LT__EQ_.cljs$core$IFn$_invoke$arity$variadic = function(x, y, more) {
  while (true) {
    if (x <= y) {
      if (cljs.core.next(more)) {
        var G__10175 = y;
        var G__10176 = cljs.core.first(more);
        var G__10177 = cljs.core.next(more);
        x = G__10175;
        y = G__10176;
        more = G__10177;
        continue;
      } else {
        return y <= cljs.core.first(more);
      }
    } else {
      return false;
    }
    break;
  }
};
cljs.core._LT__EQ_.cljs$lang$applyTo = function(seq10166) {
  var G__10167 = cljs.core.first(seq10166);
  var seq10166__$1 = cljs.core.next(seq10166);
  var G__10168 = cljs.core.first(seq10166__$1);
  var seq10166__$2 = cljs.core.next(seq10166__$1);
  return cljs.core._LT__EQ_.cljs$core$IFn$_invoke$arity$variadic(G__10167, G__10168, seq10166__$2);
};
cljs.core._LT__EQ_.cljs$lang$maxFixedArity = 2;
cljs.core._GT_ = function cljs$core$_GT_(var_args) {
  var args10178 = [];
  var len__8373__auto___10184 = arguments.length;
  var i__8374__auto___10185 = 0;
  while (true) {
    if (i__8374__auto___10185 < len__8373__auto___10184) {
      args10178.push(arguments[i__8374__auto___10185]);
      var G__10186 = i__8374__auto___10185 + 1;
      i__8374__auto___10185 = G__10186;
      continue;
    } else {
    }
    break;
  }
  var G__10183 = args10178.length;
  switch(G__10183) {
    case 1:
      return cljs.core._GT_.cljs$core$IFn$_invoke$arity$1(arguments[0]);
      break;
    case 2:
      return cljs.core._GT_.cljs$core$IFn$_invoke$arity$2(arguments[0], arguments[1]);
      break;
    default:
      var argseq__8392__auto__ = new cljs.core.IndexedSeq(args10178.slice(2), 0, null);
      return cljs.core._GT_.cljs$core$IFn$_invoke$arity$variadic(arguments[0], arguments[1], argseq__8392__auto__);
  }
};
cljs.core._GT_.cljs$core$IFn$_invoke$arity$1 = function(x) {
  return true;
};
cljs.core._GT_.cljs$core$IFn$_invoke$arity$2 = function(x, y) {
  return x > y;
};
cljs.core._GT_.cljs$core$IFn$_invoke$arity$variadic = function(x, y, more) {
  while (true) {
    if (x > y) {
      if (cljs.core.next(more)) {
        var G__10188 = y;
        var G__10189 = cljs.core.first(more);
        var G__10190 = cljs.core.next(more);
        x = G__10188;
        y = G__10189;
        more = G__10190;
        continue;
      } else {
        return y > cljs.core.first(more);
      }
    } else {
      return false;
    }
    break;
  }
};
cljs.core._GT_.cljs$lang$applyTo = function(seq10179) {
  var G__10180 = cljs.core.first(seq10179);
  var seq10179__$1 = cljs.core.next(seq10179);
  var G__10181 = cljs.core.first(seq10179__$1);
  var seq10179__$2 = cljs.core.next(seq10179__$1);
  return cljs.core._GT_.cljs$core$IFn$_invoke$arity$variadic(G__10180, G__10181, seq10179__$2);
};
cljs.core._GT_.cljs$lang$maxFixedArity = 2;
cljs.core._GT__EQ_ = function cljs$core$_GT__EQ_(var_args) {
  var args10191 = [];
  var len__8373__auto___10197 = arguments.length;
  var i__8374__auto___10198 = 0;
  while (true) {
    if (i__8374__auto___10198 < len__8373__auto___10197) {
      args10191.push(arguments[i__8374__auto___10198]);
      var G__10199 = i__8374__auto___10198 + 1;
      i__8374__auto___10198 = G__10199;
      continue;
    } else {
    }
    break;
  }
  var G__10196 = args10191.length;
  switch(G__10196) {
    case 1:
      return cljs.core._GT__EQ_.cljs$core$IFn$_invoke$arity$1(arguments[0]);
      break;
    case 2:
      return cljs.core._GT__EQ_.cljs$core$IFn$_invoke$arity$2(arguments[0], arguments[1]);
      break;
    default:
      var argseq__8392__auto__ = new cljs.core.IndexedSeq(args10191.slice(2), 0, null);
      return cljs.core._GT__EQ_.cljs$core$IFn$_invoke$arity$variadic(arguments[0], arguments[1], argseq__8392__auto__);
  }
};
cljs.core._GT__EQ_.cljs$core$IFn$_invoke$arity$1 = function(x) {
  return true;
};
cljs.core._GT__EQ_.cljs$core$IFn$_invoke$arity$2 = function(x, y) {
  return x >= y;
};
cljs.core._GT__EQ_.cljs$core$IFn$_invoke$arity$variadic = function(x, y, more) {
  while (true) {
    if (x >= y) {
      if (cljs.core.next(more)) {
        var G__10201 = y;
        var G__10202 = cljs.core.first(more);
        var G__10203 = cljs.core.next(more);
        x = G__10201;
        y = G__10202;
        more = G__10203;
        continue;
      } else {
        return y >= cljs.core.first(more);
      }
    } else {
      return false;
    }
    break;
  }
};
cljs.core._GT__EQ_.cljs$lang$applyTo = function(seq10192) {
  var G__10193 = cljs.core.first(seq10192);
  var seq10192__$1 = cljs.core.next(seq10192);
  var G__10194 = cljs.core.first(seq10192__$1);
  var seq10192__$2 = cljs.core.next(seq10192__$1);
  return cljs.core._GT__EQ_.cljs$core$IFn$_invoke$arity$variadic(G__10193, G__10194, seq10192__$2);
};
cljs.core._GT__EQ_.cljs$lang$maxFixedArity = 2;
cljs.core.dec = function cljs$core$dec(x) {
  return x - 1;
};
cljs.core.max = function cljs$core$max(var_args) {
  var args10204 = [];
  var len__8373__auto___10210 = arguments.length;
  var i__8374__auto___10211 = 0;
  while (true) {
    if (i__8374__auto___10211 < len__8373__auto___10210) {
      args10204.push(arguments[i__8374__auto___10211]);
      var G__10212 = i__8374__auto___10211 + 1;
      i__8374__auto___10211 = G__10212;
      continue;
    } else {
    }
    break;
  }
  var G__10209 = args10204.length;
  switch(G__10209) {
    case 1:
      return cljs.core.max.cljs$core$IFn$_invoke$arity$1(arguments[0]);
      break;
    case 2:
      return cljs.core.max.cljs$core$IFn$_invoke$arity$2(arguments[0], arguments[1]);
      break;
    default:
      var argseq__8392__auto__ = new cljs.core.IndexedSeq(args10204.slice(2), 0, null);
      return cljs.core.max.cljs$core$IFn$_invoke$arity$variadic(arguments[0], arguments[1], argseq__8392__auto__);
  }
};
cljs.core.max.cljs$core$IFn$_invoke$arity$1 = function(x) {
  return x;
};
cljs.core.max.cljs$core$IFn$_invoke$arity$2 = function(x, y) {
  var x__7596__auto__ = x;
  var y__7597__auto__ = y;
  return x__7596__auto__ > y__7597__auto__ ? x__7596__auto__ : y__7597__auto__;
};
cljs.core.max.cljs$core$IFn$_invoke$arity$variadic = function(x, y, more) {
  return cljs.core.reduce.cljs$core$IFn$_invoke$arity$3(cljs.core.max, function() {
    var x__7596__auto__ = x;
    var y__7597__auto__ = y;
    return x__7596__auto__ > y__7597__auto__ ? x__7596__auto__ : y__7597__auto__;
  }(), more);
};
cljs.core.max.cljs$lang$applyTo = function(seq10205) {
  var G__10206 = cljs.core.first(seq10205);
  var seq10205__$1 = cljs.core.next(seq10205);
  var G__10207 = cljs.core.first(seq10205__$1);
  var seq10205__$2 = cljs.core.next(seq10205__$1);
  return cljs.core.max.cljs$core$IFn$_invoke$arity$variadic(G__10206, G__10207, seq10205__$2);
};
cljs.core.max.cljs$lang$maxFixedArity = 2;
cljs.core.min = function cljs$core$min(var_args) {
  var args10214 = [];
  var len__8373__auto___10220 = arguments.length;
  var i__8374__auto___10221 = 0;
  while (true) {
    if (i__8374__auto___10221 < len__8373__auto___10220) {
      args10214.push(arguments[i__8374__auto___10221]);
      var G__10222 = i__8374__auto___10221 + 1;
      i__8374__auto___10221 = G__10222;
      continue;
    } else {
    }
    break;
  }
  var G__10219 = args10214.length;
  switch(G__10219) {
    case 1:
      return cljs.core.min.cljs$core$IFn$_invoke$arity$1(arguments[0]);
      break;
    case 2:
      return cljs.core.min.cljs$core$IFn$_invoke$arity$2(arguments[0], arguments[1]);
      break;
    default:
      var argseq__8392__auto__ = new cljs.core.IndexedSeq(args10214.slice(2), 0, null);
      return cljs.core.min.cljs$core$IFn$_invoke$arity$variadic(arguments[0], arguments[1], argseq__8392__auto__);
  }
};
cljs.core.min.cljs$core$IFn$_invoke$arity$1 = function(x) {
  return x;
};
cljs.core.min.cljs$core$IFn$_invoke$arity$2 = function(x, y) {
  var x__7603__auto__ = x;
  var y__7604__auto__ = y;
  return x__7603__auto__ < y__7604__auto__ ? x__7603__auto__ : y__7604__auto__;
};
cljs.core.min.cljs$core$IFn$_invoke$arity$variadic = function(x, y, more) {
  return cljs.core.reduce.cljs$core$IFn$_invoke$arity$3(cljs.core.min, function() {
    var x__7603__auto__ = x;
    var y__7604__auto__ = y;
    return x__7603__auto__ < y__7604__auto__ ? x__7603__auto__ : y__7604__auto__;
  }(), more);
};
cljs.core.min.cljs$lang$applyTo = function(seq10215) {
  var G__10216 = cljs.core.first(seq10215);
  var seq10215__$1 = cljs.core.next(seq10215);
  var G__10217 = cljs.core.first(seq10215__$1);
  var seq10215__$2 = cljs.core.next(seq10215__$1);
  return cljs.core.min.cljs$core$IFn$_invoke$arity$variadic(G__10216, G__10217, seq10215__$2);
};
cljs.core.min.cljs$lang$maxFixedArity = 2;
cljs.core.byte$ = function cljs$core$byte(x) {
  return x;
};
cljs.core.char$ = function cljs$core$char(x) {
  if (typeof x === "number") {
    return String.fromCharCode(x);
  } else {
    if (typeof x === "string" && x.length === 1) {
      return x;
    } else {
      throw new Error("Argument to char must be a character or number");
    }
  }
};
cljs.core.short$ = function cljs$core$short(x) {
  return x;
};
cljs.core.float$ = function cljs$core$float(x) {
  return x;
};
cljs.core.double$ = function cljs$core$double(x) {
  return x;
};
cljs.core.unchecked_byte = function cljs$core$unchecked_byte(x) {
  return x;
};
cljs.core.unchecked_char = function cljs$core$unchecked_char(x) {
  return x;
};
cljs.core.unchecked_short = function cljs$core$unchecked_short(x) {
  return x;
};
cljs.core.unchecked_float = function cljs$core$unchecked_float(x) {
  return x;
};
cljs.core.unchecked_double = function cljs$core$unchecked_double(x) {
  return x;
};
cljs.core.unchecked_add = function cljs$core$unchecked_add(var_args) {
  var args10224 = [];
  var len__8373__auto___10230 = arguments.length;
  var i__8374__auto___10231 = 0;
  while (true) {
    if (i__8374__auto___10231 < len__8373__auto___10230) {
      args10224.push(arguments[i__8374__auto___10231]);
      var G__10232 = i__8374__auto___10231 + 1;
      i__8374__auto___10231 = G__10232;
      continue;
    } else {
    }
    break;
  }
  var G__10229 = args10224.length;
  switch(G__10229) {
    case 0:
      return cljs.core.unchecked_add.cljs$core$IFn$_invoke$arity$0();
      break;
    case 1:
      return cljs.core.unchecked_add.cljs$core$IFn$_invoke$arity$1(arguments[0]);
      break;
    case 2:
      return cljs.core.unchecked_add.cljs$core$IFn$_invoke$arity$2(arguments[0], arguments[1]);
      break;
    default:
      var argseq__8392__auto__ = new cljs.core.IndexedSeq(args10224.slice(2), 0, null);
      return cljs.core.unchecked_add.cljs$core$IFn$_invoke$arity$variadic(arguments[0], arguments[1], argseq__8392__auto__);
  }
};
cljs.core.unchecked_add.cljs$core$IFn$_invoke$arity$0 = function() {
  return 0;
};
cljs.core.unchecked_add.cljs$core$IFn$_invoke$arity$1 = function(x) {
  return x;
};
cljs.core.unchecked_add.cljs$core$IFn$_invoke$arity$2 = function(x, y) {
  return x + y;
};
cljs.core.unchecked_add.cljs$core$IFn$_invoke$arity$variadic = function(x, y, more) {
  return cljs.core.reduce.cljs$core$IFn$_invoke$arity$3(cljs.core.unchecked_add, x + y, more);
};
cljs.core.unchecked_add.cljs$lang$applyTo = function(seq10225) {
  var G__10226 = cljs.core.first(seq10225);
  var seq10225__$1 = cljs.core.next(seq10225);
  var G__10227 = cljs.core.first(seq10225__$1);
  var seq10225__$2 = cljs.core.next(seq10225__$1);
  return cljs.core.unchecked_add.cljs$core$IFn$_invoke$arity$variadic(G__10226, G__10227, seq10225__$2);
};
cljs.core.unchecked_add.cljs$lang$maxFixedArity = 2;
cljs.core.unchecked_add_int = function cljs$core$unchecked_add_int(var_args) {
  var args10234 = [];
  var len__8373__auto___10240 = arguments.length;
  var i__8374__auto___10241 = 0;
  while (true) {
    if (i__8374__auto___10241 < len__8373__auto___10240) {
      args10234.push(arguments[i__8374__auto___10241]);
      var G__10242 = i__8374__auto___10241 + 1;
      i__8374__auto___10241 = G__10242;
      continue;
    } else {
    }
    break;
  }
  var G__10239 = args10234.length;
  switch(G__10239) {
    case 0:
      return cljs.core.unchecked_add_int.cljs$core$IFn$_invoke$arity$0();
      break;
    case 1:
      return cljs.core.unchecked_add_int.cljs$core$IFn$_invoke$arity$1(arguments[0]);
      break;
    case 2:
      return cljs.core.unchecked_add_int.cljs$core$IFn$_invoke$arity$2(arguments[0], arguments[1]);
      break;
    default:
      var argseq__8392__auto__ = new cljs.core.IndexedSeq(args10234.slice(2), 0, null);
      return cljs.core.unchecked_add_int.cljs$core$IFn$_invoke$arity$variadic(arguments[0], arguments[1], argseq__8392__auto__);
  }
};
cljs.core.unchecked_add_int.cljs$core$IFn$_invoke$arity$0 = function() {
  return 0;
};
cljs.core.unchecked_add_int.cljs$core$IFn$_invoke$arity$1 = function(x) {
  return x;
};
cljs.core.unchecked_add_int.cljs$core$IFn$_invoke$arity$2 = function(x, y) {
  return x + y;
};
cljs.core.unchecked_add_int.cljs$core$IFn$_invoke$arity$variadic = function(x, y, more) {
  return cljs.core.reduce.cljs$core$IFn$_invoke$arity$3(cljs.core.unchecked_add_int, x + y, more);
};
cljs.core.unchecked_add_int.cljs$lang$applyTo = function(seq10235) {
  var G__10236 = cljs.core.first(seq10235);
  var seq10235__$1 = cljs.core.next(seq10235);
  var G__10237 = cljs.core.first(seq10235__$1);
  var seq10235__$2 = cljs.core.next(seq10235__$1);
  return cljs.core.unchecked_add_int.cljs$core$IFn$_invoke$arity$variadic(G__10236, G__10237, seq10235__$2);
};
cljs.core.unchecked_add_int.cljs$lang$maxFixedArity = 2;
cljs.core.unchecked_dec = function cljs$core$unchecked_dec(x) {
  return x - 1;
};
cljs.core.unchecked_dec_int = function cljs$core$unchecked_dec_int(x) {
  return x - 1;
};
cljs.core.unchecked_divide_int = function cljs$core$unchecked_divide_int(var_args) {
  var args10244 = [];
  var len__8373__auto___10250 = arguments.length;
  var i__8374__auto___10251 = 0;
  while (true) {
    if (i__8374__auto___10251 < len__8373__auto___10250) {
      args10244.push(arguments[i__8374__auto___10251]);
      var G__10252 = i__8374__auto___10251 + 1;
      i__8374__auto___10251 = G__10252;
      continue;
    } else {
    }
    break;
  }
  var G__10249 = args10244.length;
  switch(G__10249) {
    case 1:
      return cljs.core.unchecked_divide_int.cljs$core$IFn$_invoke$arity$1(arguments[0]);
      break;
    case 2:
      return cljs.core.unchecked_divide_int.cljs$core$IFn$_invoke$arity$2(arguments[0], arguments[1]);
      break;
    default:
      var argseq__8392__auto__ = new cljs.core.IndexedSeq(args10244.slice(2), 0, null);
      return cljs.core.unchecked_divide_int.cljs$core$IFn$_invoke$arity$variadic(arguments[0], arguments[1], argseq__8392__auto__);
  }
};
cljs.core.unchecked_divide_int.cljs$core$IFn$_invoke$arity$1 = function(x) {
  return 1 / x;
};
cljs.core.unchecked_divide_int.cljs$core$IFn$_invoke$arity$2 = function(x, y) {
  return x / y;
};
cljs.core.unchecked_divide_int.cljs$core$IFn$_invoke$arity$variadic = function(x, y, more) {
  return cljs.core.reduce.cljs$core$IFn$_invoke$arity$3(cljs.core.unchecked_divide_int, x / y, more);
};
cljs.core.unchecked_divide_int.cljs$lang$applyTo = function(seq10245) {
  var G__10246 = cljs.core.first(seq10245);
  var seq10245__$1 = cljs.core.next(seq10245);
  var G__10247 = cljs.core.first(seq10245__$1);
  var seq10245__$2 = cljs.core.next(seq10245__$1);
  return cljs.core.unchecked_divide_int.cljs$core$IFn$_invoke$arity$variadic(G__10246, G__10247, seq10245__$2);
};
cljs.core.unchecked_divide_int.cljs$lang$maxFixedArity = 2;
cljs.core.unchecked_inc = function cljs$core$unchecked_inc(x) {
  return x + 1;
};
cljs.core.unchecked_inc_int = function cljs$core$unchecked_inc_int(x) {
  return x + 1;
};
cljs.core.unchecked_multiply = function cljs$core$unchecked_multiply(var_args) {
  var args10254 = [];
  var len__8373__auto___10260 = arguments.length;
  var i__8374__auto___10261 = 0;
  while (true) {
    if (i__8374__auto___10261 < len__8373__auto___10260) {
      args10254.push(arguments[i__8374__auto___10261]);
      var G__10262 = i__8374__auto___10261 + 1;
      i__8374__auto___10261 = G__10262;
      continue;
    } else {
    }
    break;
  }
  var G__10259 = args10254.length;
  switch(G__10259) {
    case 0:
      return cljs.core.unchecked_multiply.cljs$core$IFn$_invoke$arity$0();
      break;
    case 1:
      return cljs.core.unchecked_multiply.cljs$core$IFn$_invoke$arity$1(arguments[0]);
      break;
    case 2:
      return cljs.core.unchecked_multiply.cljs$core$IFn$_invoke$arity$2(arguments[0], arguments[1]);
      break;
    default:
      var argseq__8392__auto__ = new cljs.core.IndexedSeq(args10254.slice(2), 0, null);
      return cljs.core.unchecked_multiply.cljs$core$IFn$_invoke$arity$variadic(arguments[0], arguments[1], argseq__8392__auto__);
  }
};
cljs.core.unchecked_multiply.cljs$core$IFn$_invoke$arity$0 = function() {
  return 1;
};
cljs.core.unchecked_multiply.cljs$core$IFn$_invoke$arity$1 = function(x) {
  return x;
};
cljs.core.unchecked_multiply.cljs$core$IFn$_invoke$arity$2 = function(x, y) {
  return x * y;
};
cljs.core.unchecked_multiply.cljs$core$IFn$_invoke$arity$variadic = function(x, y, more) {
  return cljs.core.reduce.cljs$core$IFn$_invoke$arity$3(cljs.core.unchecked_multiply, x * y, more);
};
cljs.core.unchecked_multiply.cljs$lang$applyTo = function(seq10255) {
  var G__10256 = cljs.core.first(seq10255);
  var seq10255__$1 = cljs.core.next(seq10255);
  var G__10257 = cljs.core.first(seq10255__$1);
  var seq10255__$2 = cljs.core.next(seq10255__$1);
  return cljs.core.unchecked_multiply.cljs$core$IFn$_invoke$arity$variadic(G__10256, G__10257, seq10255__$2);
};
cljs.core.unchecked_multiply.cljs$lang$maxFixedArity = 2;
cljs.core.unchecked_multiply_int = function cljs$core$unchecked_multiply_int(var_args) {
  var args10264 = [];
  var len__8373__auto___10270 = arguments.length;
  var i__8374__auto___10271 = 0;
  while (true) {
    if (i__8374__auto___10271 < len__8373__auto___10270) {
      args10264.push(arguments[i__8374__auto___10271]);
      var G__10272 = i__8374__auto___10271 + 1;
      i__8374__auto___10271 = G__10272;
      continue;
    } else {
    }
    break;
  }
  var G__10269 = args10264.length;
  switch(G__10269) {
    case 0:
      return cljs.core.unchecked_multiply_int.cljs$core$IFn$_invoke$arity$0();
      break;
    case 1:
      return cljs.core.unchecked_multiply_int.cljs$core$IFn$_invoke$arity$1(arguments[0]);
      break;
    case 2:
      return cljs.core.unchecked_multiply_int.cljs$core$IFn$_invoke$arity$2(arguments[0], arguments[1]);
      break;
    default:
      var argseq__8392__auto__ = new cljs.core.IndexedSeq(args10264.slice(2), 0, null);
      return cljs.core.unchecked_multiply_int.cljs$core$IFn$_invoke$arity$variadic(arguments[0], arguments[1], argseq__8392__auto__);
  }
};
cljs.core.unchecked_multiply_int.cljs$core$IFn$_invoke$arity$0 = function() {
  return 1;
};
cljs.core.unchecked_multiply_int.cljs$core$IFn$_invoke$arity$1 = function(x) {
  return x;
};
cljs.core.unchecked_multiply_int.cljs$core$IFn$_invoke$arity$2 = function(x, y) {
  return x * y;
};
cljs.core.unchecked_multiply_int.cljs$core$IFn$_invoke$arity$variadic = function(x, y, more) {
  return cljs.core.reduce.cljs$core$IFn$_invoke$arity$3(cljs.core.unchecked_multiply_int, x * y, more);
};
cljs.core.unchecked_multiply_int.cljs$lang$applyTo = function(seq10265) {
  var G__10266 = cljs.core.first(seq10265);
  var seq10265__$1 = cljs.core.next(seq10265);
  var G__10267 = cljs.core.first(seq10265__$1);
  var seq10265__$2 = cljs.core.next(seq10265__$1);
  return cljs.core.unchecked_multiply_int.cljs$core$IFn$_invoke$arity$variadic(G__10266, G__10267, seq10265__$2);
};
cljs.core.unchecked_multiply_int.cljs$lang$maxFixedArity = 2;
cljs.core.unchecked_negate = function cljs$core$unchecked_negate(x) {
  return -x;
};
cljs.core.unchecked_negate_int = function cljs$core$unchecked_negate_int(x) {
  return -x;
};
cljs.core.unchecked_remainder_int = function cljs$core$unchecked_remainder_int(x, n) {
  return cljs.core.mod.cljs$core$IFn$_invoke$arity$2 ? cljs.core.mod.cljs$core$IFn$_invoke$arity$2(x, n) : cljs.core.mod.call(null, x, n);
};
cljs.core.unchecked_subtract = function cljs$core$unchecked_subtract(var_args) {
  var args10274 = [];
  var len__8373__auto___10280 = arguments.length;
  var i__8374__auto___10281 = 0;
  while (true) {
    if (i__8374__auto___10281 < len__8373__auto___10280) {
      args10274.push(arguments[i__8374__auto___10281]);
      var G__10282 = i__8374__auto___10281 + 1;
      i__8374__auto___10281 = G__10282;
      continue;
    } else {
    }
    break;
  }
  var G__10279 = args10274.length;
  switch(G__10279) {
    case 1:
      return cljs.core.unchecked_subtract.cljs$core$IFn$_invoke$arity$1(arguments[0]);
      break;
    case 2:
      return cljs.core.unchecked_subtract.cljs$core$IFn$_invoke$arity$2(arguments[0], arguments[1]);
      break;
    default:
      var argseq__8392__auto__ = new cljs.core.IndexedSeq(args10274.slice(2), 0, null);
      return cljs.core.unchecked_subtract.cljs$core$IFn$_invoke$arity$variadic(arguments[0], arguments[1], argseq__8392__auto__);
  }
};
cljs.core.unchecked_subtract.cljs$core$IFn$_invoke$arity$1 = function(x) {
  return -x;
};
cljs.core.unchecked_subtract.cljs$core$IFn$_invoke$arity$2 = function(x, y) {
  return x - y;
};
cljs.core.unchecked_subtract.cljs$core$IFn$_invoke$arity$variadic = function(x, y, more) {
  return cljs.core.reduce.cljs$core$IFn$_invoke$arity$3(cljs.core.unchecked_subtract, x - y, more);
};
cljs.core.unchecked_subtract.cljs$lang$applyTo = function(seq10275) {
  var G__10276 = cljs.core.first(seq10275);
  var seq10275__$1 = cljs.core.next(seq10275);
  var G__10277 = cljs.core.first(seq10275__$1);
  var seq10275__$2 = cljs.core.next(seq10275__$1);
  return cljs.core.unchecked_subtract.cljs$core$IFn$_invoke$arity$variadic(G__10276, G__10277, seq10275__$2);
};
cljs.core.unchecked_subtract.cljs$lang$maxFixedArity = 2;
cljs.core.unchecked_subtract_int = function cljs$core$unchecked_subtract_int(var_args) {
  var args10284 = [];
  var len__8373__auto___10290 = arguments.length;
  var i__8374__auto___10291 = 0;
  while (true) {
    if (i__8374__auto___10291 < len__8373__auto___10290) {
      args10284.push(arguments[i__8374__auto___10291]);
      var G__10292 = i__8374__auto___10291 + 1;
      i__8374__auto___10291 = G__10292;
      continue;
    } else {
    }
    break;
  }
  var G__10289 = args10284.length;
  switch(G__10289) {
    case 1:
      return cljs.core.unchecked_subtract_int.cljs$core$IFn$_invoke$arity$1(arguments[0]);
      break;
    case 2:
      return cljs.core.unchecked_subtract_int.cljs$core$IFn$_invoke$arity$2(arguments[0], arguments[1]);
      break;
    default:
      var argseq__8392__auto__ = new cljs.core.IndexedSeq(args10284.slice(2), 0, null);
      return cljs.core.unchecked_subtract_int.cljs$core$IFn$_invoke$arity$variadic(arguments[0], arguments[1], argseq__8392__auto__);
  }
};
cljs.core.unchecked_subtract_int.cljs$core$IFn$_invoke$arity$1 = function(x) {
  return -x;
};
cljs.core.unchecked_subtract_int.cljs$core$IFn$_invoke$arity$2 = function(x, y) {
  return x - y;
};
cljs.core.unchecked_subtract_int.cljs$core$IFn$_invoke$arity$variadic = function(x, y, more) {
  return cljs.core.reduce.cljs$core$IFn$_invoke$arity$3(cljs.core.unchecked_subtract_int, x - y, more);
};
cljs.core.unchecked_subtract_int.cljs$lang$applyTo = function(seq10285) {
  var G__10286 = cljs.core.first(seq10285);
  var seq10285__$1 = cljs.core.next(seq10285);
  var G__10287 = cljs.core.first(seq10285__$1);
  var seq10285__$2 = cljs.core.next(seq10285__$1);
  return cljs.core.unchecked_subtract_int.cljs$core$IFn$_invoke$arity$variadic(G__10286, G__10287, seq10285__$2);
};
cljs.core.unchecked_subtract_int.cljs$lang$maxFixedArity = 2;
cljs.core.fix = function cljs$core$fix(q) {
  if (q >= 0) {
    return Math.floor(q);
  } else {
    return Math.ceil(q);
  }
};
cljs.core.int$ = function cljs$core$int(x) {
  return x | 0;
};
cljs.core.unchecked_int = function cljs$core$unchecked_int(x) {
  return cljs.core.fix(x);
};
cljs.core.long$ = function cljs$core$long(x) {
  return cljs.core.fix(x);
};
cljs.core.unchecked_long = function cljs$core$unchecked_long(x) {
  return cljs.core.fix(x);
};
cljs.core.booleans = function cljs$core$booleans(x) {
  return x;
};
cljs.core.bytes = function cljs$core$bytes(x) {
  return x;
};
cljs.core.chars = function cljs$core$chars(x) {
  return x;
};
cljs.core.shorts = function cljs$core$shorts(x) {
  return x;
};
cljs.core.ints = function cljs$core$ints(x) {
  return x;
};
cljs.core.floats = function cljs$core$floats(x) {
  return x;
};
cljs.core.doubles = function cljs$core$doubles(x) {
  return x;
};
cljs.core.longs = function cljs$core$longs(x) {
  return x;
};
cljs.core.js_mod = function cljs$core$js_mod(n, d) {
  return n % d;
};
cljs.core.mod = function cljs$core$mod(n, d) {
  return (n % d + d) % d;
};
cljs.core.quot = function cljs$core$quot(n, d) {
  var rem = n % d;
  return cljs.core.fix((n - rem) / d);
};
cljs.core.rem = function cljs$core$rem(n, d) {
  var q = cljs.core.quot(n, d);
  return n - d * q;
};
cljs.core.bit_xor = function cljs$core$bit_xor(var_args) {
  var args10294 = [];
  var len__8373__auto___10300 = arguments.length;
  var i__8374__auto___10301 = 0;
  while (true) {
    if (i__8374__auto___10301 < len__8373__auto___10300) {
      args10294.push(arguments[i__8374__auto___10301]);
      var G__10302 = i__8374__auto___10301 + 1;
      i__8374__auto___10301 = G__10302;
      continue;
    } else {
    }
    break;
  }
  var G__10299 = args10294.length;
  switch(G__10299) {
    case 2:
      return cljs.core.bit_xor.cljs$core$IFn$_invoke$arity$2(arguments[0], arguments[1]);
      break;
    default:
      var argseq__8392__auto__ = new cljs.core.IndexedSeq(args10294.slice(2), 0, null);
      return cljs.core.bit_xor.cljs$core$IFn$_invoke$arity$variadic(arguments[0], arguments[1], argseq__8392__auto__);
  }
};
cljs.core.bit_xor.cljs$core$IFn$_invoke$arity$2 = function(x, y) {
  return x ^ y;
};
cljs.core.bit_xor.cljs$core$IFn$_invoke$arity$variadic = function(x, y, more) {
  return cljs.core.reduce.cljs$core$IFn$_invoke$arity$3(cljs.core.bit_xor, x ^ y, more);
};
cljs.core.bit_xor.cljs$lang$applyTo = function(seq10295) {
  var G__10296 = cljs.core.first(seq10295);
  var seq10295__$1 = cljs.core.next(seq10295);
  var G__10297 = cljs.core.first(seq10295__$1);
  var seq10295__$2 = cljs.core.next(seq10295__$1);
  return cljs.core.bit_xor.cljs$core$IFn$_invoke$arity$variadic(G__10296, G__10297, seq10295__$2);
};
cljs.core.bit_xor.cljs$lang$maxFixedArity = 2;
cljs.core.bit_and = function cljs$core$bit_and(var_args) {
  var args10304 = [];
  var len__8373__auto___10310 = arguments.length;
  var i__8374__auto___10311 = 0;
  while (true) {
    if (i__8374__auto___10311 < len__8373__auto___10310) {
      args10304.push(arguments[i__8374__auto___10311]);
      var G__10312 = i__8374__auto___10311 + 1;
      i__8374__auto___10311 = G__10312;
      continue;
    } else {
    }
    break;
  }
  var G__10309 = args10304.length;
  switch(G__10309) {
    case 2:
      return cljs.core.bit_and.cljs$core$IFn$_invoke$arity$2(arguments[0], arguments[1]);
      break;
    default:
      var argseq__8392__auto__ = new cljs.core.IndexedSeq(args10304.slice(2), 0, null);
      return cljs.core.bit_and.cljs$core$IFn$_invoke$arity$variadic(arguments[0], arguments[1], argseq__8392__auto__);
  }
};
cljs.core.bit_and.cljs$core$IFn$_invoke$arity$2 = function(x, y) {
  return x & y;
};
cljs.core.bit_and.cljs$core$IFn$_invoke$arity$variadic = function(x, y, more) {
  return cljs.core.reduce.cljs$core$IFn$_invoke$arity$3(cljs.core.bit_and, x & y, more);
};
cljs.core.bit_and.cljs$lang$applyTo = function(seq10305) {
  var G__10306 = cljs.core.first(seq10305);
  var seq10305__$1 = cljs.core.next(seq10305);
  var G__10307 = cljs.core.first(seq10305__$1);
  var seq10305__$2 = cljs.core.next(seq10305__$1);
  return cljs.core.bit_and.cljs$core$IFn$_invoke$arity$variadic(G__10306, G__10307, seq10305__$2);
};
cljs.core.bit_and.cljs$lang$maxFixedArity = 2;
cljs.core.bit_or = function cljs$core$bit_or(var_args) {
  var args10314 = [];
  var len__8373__auto___10320 = arguments.length;
  var i__8374__auto___10321 = 0;
  while (true) {
    if (i__8374__auto___10321 < len__8373__auto___10320) {
      args10314.push(arguments[i__8374__auto___10321]);
      var G__10322 = i__8374__auto___10321 + 1;
      i__8374__auto___10321 = G__10322;
      continue;
    } else {
    }
    break;
  }
  var G__10319 = args10314.length;
  switch(G__10319) {
    case 2:
      return cljs.core.bit_or.cljs$core$IFn$_invoke$arity$2(arguments[0], arguments[1]);
      break;
    default:
      var argseq__8392__auto__ = new cljs.core.IndexedSeq(args10314.slice(2), 0, null);
      return cljs.core.bit_or.cljs$core$IFn$_invoke$arity$variadic(arguments[0], arguments[1], argseq__8392__auto__);
  }
};
cljs.core.bit_or.cljs$core$IFn$_invoke$arity$2 = function(x, y) {
  return x | y;
};
cljs.core.bit_or.cljs$core$IFn$_invoke$arity$variadic = function(x, y, more) {
  return cljs.core.reduce.cljs$core$IFn$_invoke$arity$3(cljs.core.bit_or, x | y, more);
};
cljs.core.bit_or.cljs$lang$applyTo = function(seq10315) {
  var G__10316 = cljs.core.first(seq10315);
  var seq10315__$1 = cljs.core.next(seq10315);
  var G__10317 = cljs.core.first(seq10315__$1);
  var seq10315__$2 = cljs.core.next(seq10315__$1);
  return cljs.core.bit_or.cljs$core$IFn$_invoke$arity$variadic(G__10316, G__10317, seq10315__$2);
};
cljs.core.bit_or.cljs$lang$maxFixedArity = 2;
cljs.core.bit_and_not = function cljs$core$bit_and_not(var_args) {
  var args10324 = [];
  var len__8373__auto___10330 = arguments.length;
  var i__8374__auto___10331 = 0;
  while (true) {
    if (i__8374__auto___10331 < len__8373__auto___10330) {
      args10324.push(arguments[i__8374__auto___10331]);
      var G__10332 = i__8374__auto___10331 + 1;
      i__8374__auto___10331 = G__10332;
      continue;
    } else {
    }
    break;
  }
  var G__10329 = args10324.length;
  switch(G__10329) {
    case 2:
      return cljs.core.bit_and_not.cljs$core$IFn$_invoke$arity$2(arguments[0], arguments[1]);
      break;
    default:
      var argseq__8392__auto__ = new cljs.core.IndexedSeq(args10324.slice(2), 0, null);
      return cljs.core.bit_and_not.cljs$core$IFn$_invoke$arity$variadic(arguments[0], arguments[1], argseq__8392__auto__);
  }
};
cljs.core.bit_and_not.cljs$core$IFn$_invoke$arity$2 = function(x, y) {
  return x & ~y;
};
cljs.core.bit_and_not.cljs$core$IFn$_invoke$arity$variadic = function(x, y, more) {
  return cljs.core.reduce.cljs$core$IFn$_invoke$arity$3(cljs.core.bit_and_not, x & ~y, more);
};
cljs.core.bit_and_not.cljs$lang$applyTo = function(seq10325) {
  var G__10326 = cljs.core.first(seq10325);
  var seq10325__$1 = cljs.core.next(seq10325);
  var G__10327 = cljs.core.first(seq10325__$1);
  var seq10325__$2 = cljs.core.next(seq10325__$1);
  return cljs.core.bit_and_not.cljs$core$IFn$_invoke$arity$variadic(G__10326, G__10327, seq10325__$2);
};
cljs.core.bit_and_not.cljs$lang$maxFixedArity = 2;
cljs.core.bit_clear = function cljs$core$bit_clear(x, n) {
  return x & ~(1 << n);
};
cljs.core.bit_flip = function cljs$core$bit_flip(x, n) {
  return x ^ 1 << n;
};
cljs.core.bit_not = function cljs$core$bit_not(x) {
  return ~x;
};
cljs.core.bit_set = function cljs$core$bit_set(x, n) {
  return x | 1 << n;
};
cljs.core.bit_test = function cljs$core$bit_test(x, n) {
  return (x & 1 << n) != 0;
};
cljs.core.bit_shift_left = function cljs$core$bit_shift_left(x, n) {
  return x << n;
};
cljs.core.bit_shift_right = function cljs$core$bit_shift_right(x, n) {
  return x >> n;
};
cljs.core.bit_shift_right_zero_fill = function cljs$core$bit_shift_right_zero_fill(x, n) {
  return x >>> n;
};
cljs.core.unsigned_bit_shift_right = function cljs$core$unsigned_bit_shift_right(x, n) {
  return x >>> n;
};
cljs.core.bit_count = function cljs$core$bit_count(v) {
  var v__$1 = v - (v >> 1 & 1431655765);
  var v__$2 = (v__$1 & 858993459) + (v__$1 >> 2 & 858993459);
  return (v__$2 + (v__$2 >> 4) & 252645135) * 16843009 >> 24;
};
cljs.core._EQ__EQ_ = function cljs$core$_EQ__EQ_(var_args) {
  var args10334 = [];
  var len__8373__auto___10340 = arguments.length;
  var i__8374__auto___10341 = 0;
  while (true) {
    if (i__8374__auto___10341 < len__8373__auto___10340) {
      args10334.push(arguments[i__8374__auto___10341]);
      var G__10342 = i__8374__auto___10341 + 1;
      i__8374__auto___10341 = G__10342;
      continue;
    } else {
    }
    break;
  }
  var G__10339 = args10334.length;
  switch(G__10339) {
    case 1:
      return cljs.core._EQ__EQ_.cljs$core$IFn$_invoke$arity$1(arguments[0]);
      break;
    case 2:
      return cljs.core._EQ__EQ_.cljs$core$IFn$_invoke$arity$2(arguments[0], arguments[1]);
      break;
    default:
      var argseq__8392__auto__ = new cljs.core.IndexedSeq(args10334.slice(2), 0, null);
      return cljs.core._EQ__EQ_.cljs$core$IFn$_invoke$arity$variadic(arguments[0], arguments[1], argseq__8392__auto__);
  }
};
cljs.core._EQ__EQ_.cljs$core$IFn$_invoke$arity$1 = function(x) {
  return true;
};
cljs.core._EQ__EQ_.cljs$core$IFn$_invoke$arity$2 = function(x, y) {
  return cljs.core._equiv(x, y);
};
cljs.core._EQ__EQ_.cljs$core$IFn$_invoke$arity$variadic = function(x, y, more) {
  while (true) {
    if (x === y) {
      if (cljs.core.next(more)) {
        var G__10344 = y;
        var G__10345 = cljs.core.first(more);
        var G__10346 = cljs.core.next(more);
        x = G__10344;
        y = G__10345;
        more = G__10346;
        continue;
      } else {
        return y === cljs.core.first(more);
      }
    } else {
      return false;
    }
    break;
  }
};
cljs.core._EQ__EQ_.cljs$lang$applyTo = function(seq10335) {
  var G__10336 = cljs.core.first(seq10335);
  var seq10335__$1 = cljs.core.next(seq10335);
  var G__10337 = cljs.core.first(seq10335__$1);
  var seq10335__$2 = cljs.core.next(seq10335__$1);
  return cljs.core._EQ__EQ_.cljs$core$IFn$_invoke$arity$variadic(G__10336, G__10337, seq10335__$2);
};
cljs.core._EQ__EQ_.cljs$lang$maxFixedArity = 2;
cljs.core.pos_QMARK_ = function cljs$core$pos_QMARK_(x) {
  return x > 0;
};
cljs.core.zero_QMARK_ = function cljs$core$zero_QMARK_(x) {
  return x === 0;
};
cljs.core.neg_QMARK_ = function cljs$core$neg_QMARK_(x) {
  return x < 0;
};
cljs.core.nthnext = function cljs$core$nthnext(coll, n) {
  var n__$1 = n;
  var xs = cljs.core.seq(coll);
  while (true) {
    if (xs && n__$1 > 0) {
      var G__10347 = n__$1 - 1;
      var G__10348 = cljs.core.next(xs);
      n__$1 = G__10347;
      xs = G__10348;
      continue;
    } else {
      return xs;
    }
    break;
  }
};
cljs.core.str = function cljs$core$str(var_args) {
  var args10349 = [];
  var len__8373__auto___10354 = arguments.length;
  var i__8374__auto___10355 = 0;
  while (true) {
    if (i__8374__auto___10355 < len__8373__auto___10354) {
      args10349.push(arguments[i__8374__auto___10355]);
      var G__10356 = i__8374__auto___10355 + 1;
      i__8374__auto___10355 = G__10356;
      continue;
    } else {
    }
    break;
  }
  var G__10353 = args10349.length;
  switch(G__10353) {
    case 0:
      return cljs.core.str.cljs$core$IFn$_invoke$arity$0();
      break;
    case 1:
      return cljs.core.str.cljs$core$IFn$_invoke$arity$1(arguments[0]);
      break;
    default:
      var argseq__8392__auto__ = new cljs.core.IndexedSeq(args10349.slice(1), 0, null);
      return cljs.core.str.cljs$core$IFn$_invoke$arity$variadic(arguments[0], argseq__8392__auto__);
  }
};
cljs.core.str.cljs$core$IFn$_invoke$arity$0 = function() {
  return "";
};
cljs.core.str.cljs$core$IFn$_invoke$arity$1 = function(x) {
  if (x == null) {
    return "";
  } else {
    return [x].join("");
  }
};
cljs.core.str.cljs$core$IFn$_invoke$arity$variadic = function(x, ys) {
  var sb = new goog.string.StringBuffer([cljs.core.str.cljs$core$IFn$_invoke$arity$1(x)].join(""));
  var more = ys;
  while (true) {
    if (cljs.core.truth_(more)) {
      var G__10358 = sb.append([cljs.core.str.cljs$core$IFn$_invoke$arity$1(cljs.core.first(more))].join(""));
      var G__10359 = cljs.core.next(more);
      sb = G__10358;
      more = G__10359;
      continue;
    } else {
      return sb.toString();
    }
    break;
  }
};
cljs.core.str.cljs$lang$applyTo = function(seq10350) {
  var G__10351 = cljs.core.first(seq10350);
  var seq10350__$1 = cljs.core.next(seq10350);
  return cljs.core.str.cljs$core$IFn$_invoke$arity$variadic(G__10351, seq10350__$1);
};
cljs.core.str.cljs$lang$maxFixedArity = 1;
cljs.core.subs = function cljs$core$subs(var_args) {
  var args10360 = [];
  var len__8373__auto___10363 = arguments.length;
  var i__8374__auto___10364 = 0;
  while (true) {
    if (i__8374__auto___10364 < len__8373__auto___10363) {
      args10360.push(arguments[i__8374__auto___10364]);
      var G__10365 = i__8374__auto___10364 + 1;
      i__8374__auto___10364 = G__10365;
      continue;
    } else {
    }
    break;
  }
  var G__10362 = args10360.length;
  switch(G__10362) {
    case 2:
      return cljs.core.subs.cljs$core$IFn$_invoke$arity$2(arguments[0], arguments[1]);
      break;
    case 3:
      return cljs.core.subs.cljs$core$IFn$_invoke$arity$3(arguments[0], arguments[1], arguments[2]);
      break;
    default:
      throw new Error([cljs.core.str.cljs$core$IFn$_invoke$arity$1("Invalid arity: "), cljs.core.str.cljs$core$IFn$_invoke$arity$1(args10360.length)].join(""));
  }
};
cljs.core.subs.cljs$core$IFn$_invoke$arity$2 = function(s, start) {
  return s.substring(start);
};
cljs.core.subs.cljs$core$IFn$_invoke$arity$3 = function(s, start, end) {
  return s.substring(start, end);
};
cljs.core.subs.cljs$lang$maxFixedArity = 3;
cljs.core.equiv_sequential = function cljs$core$equiv_sequential(x, y) {
  return cljs.core.boolean$(cljs.core.sequential_QMARK_(y) ? cljs.core.counted_QMARK_(x) && cljs.core.counted_QMARK_(y) && !(cljs.core.count(x) === cljs.core.count(y)) ? false : function() {
    var xs = cljs.core.seq(x);
    var ys = cljs.core.seq(y);
    while (true) {
      if (xs == null) {
        return ys == null;
      } else {
        if (ys == null) {
          return false;
        } else {
          if (cljs.core._EQ_.cljs$core$IFn$_invoke$arity$2(cljs.core.first(xs), cljs.core.first(ys))) {
            var G__10367 = cljs.core.next(xs);
            var G__10368 = cljs.core.next(ys);
            xs = G__10367;
            ys = G__10368;
            continue;
          } else {
            return false;
          }
        }
      }
      break;
    }
  }() : null);
};
cljs.core.hash_coll = function cljs$core$hash_coll(coll) {
  if (cljs.core.seq(coll)) {
    var res = cljs.core.hash(cljs.core.first(coll));
    var s = cljs.core.next(coll);
    while (true) {
      if (s == null) {
        return res;
      } else {
        var G__10369 = cljs.core.hash_combine(res, cljs.core.hash(cljs.core.first(s)));
        var G__10370 = cljs.core.next(s);
        res = G__10369;
        s = G__10370;
        continue;
      }
      break;
    }
  } else {
    return 0;
  }
};
cljs.core.hash_imap = function cljs$core$hash_imap(m) {
  var h = 0;
  var s = cljs.core.seq(m);
  while (true) {
    if (s) {
      var e = cljs.core.first(s);
      var G__10371 = (h + (cljs.core.hash(cljs.core.key.cljs$core$IFn$_invoke$arity$1 ? cljs.core.key.cljs$core$IFn$_invoke$arity$1(e) : cljs.core.key.call(null, e)) ^ cljs.core.hash(cljs.core.val.cljs$core$IFn$_invoke$arity$1 ? cljs.core.val.cljs$core$IFn$_invoke$arity$1(e) : cljs.core.val.call(null, e)))) % 4503599627370496;
      var G__10372 = cljs.core.next(s);
      h = G__10371;
      s = G__10372;
      continue;
    } else {
      return h;
    }
    break;
  }
};
cljs.core.hash_iset = function cljs$core$hash_iset(s) {
  var h = 0;
  var s__$1 = cljs.core.seq(s);
  while (true) {
    if (s__$1) {
      var e = cljs.core.first(s__$1);
      var G__10373 = (h + cljs.core.hash(e)) % 4503599627370496;
      var G__10374 = cljs.core.next(s__$1);
      h = G__10373;
      s__$1 = G__10374;
      continue;
    } else {
      return h;
    }
    break;
  }
};
cljs.core.extend_object_BANG_ = function cljs$core$extend_object_BANG_(obj, fn_map) {
  var seq__10385_10395 = cljs.core.seq(fn_map);
  var chunk__10386_10396 = null;
  var count__10387_10397 = 0;
  var i__10388_10398 = 0;
  while (true) {
    if (i__10388_10398 < count__10387_10397) {
      var vec__10389_10399 = chunk__10386_10396.cljs$core$IIndexed$_nth$arity$2(null, i__10388_10398);
      var key_name_10400 = cljs.core.nth.cljs$core$IFn$_invoke$arity$3(vec__10389_10399, 0, null);
      var f_10401 = cljs.core.nth.cljs$core$IFn$_invoke$arity$3(vec__10389_10399, 1, null);
      var str_name_10402 = cljs.core.name.cljs$core$IFn$_invoke$arity$1 ? cljs.core.name.cljs$core$IFn$_invoke$arity$1(key_name_10400) : cljs.core.name.call(null, key_name_10400);
      obj[str_name_10402] = f_10401;
      var G__10403 = seq__10385_10395;
      var G__10404 = chunk__10386_10396;
      var G__10405 = count__10387_10397;
      var G__10406 = i__10388_10398 + 1;
      seq__10385_10395 = G__10403;
      chunk__10386_10396 = G__10404;
      count__10387_10397 = G__10405;
      i__10388_10398 = G__10406;
      continue;
    } else {
      var temp__4657__auto___10407 = cljs.core.seq(seq__10385_10395);
      if (temp__4657__auto___10407) {
        var seq__10385_10408__$1 = temp__4657__auto___10407;
        if (cljs.core.chunked_seq_QMARK_(seq__10385_10408__$1)) {
          var c__8079__auto___10409 = cljs.core.chunk_first.cljs$core$IFn$_invoke$arity$1 ? cljs.core.chunk_first.cljs$core$IFn$_invoke$arity$1(seq__10385_10408__$1) : cljs.core.chunk_first.call(null, seq__10385_10408__$1);
          var G__10410 = cljs.core.chunk_rest.cljs$core$IFn$_invoke$arity$1 ? cljs.core.chunk_rest.cljs$core$IFn$_invoke$arity$1(seq__10385_10408__$1) : cljs.core.chunk_rest.call(null, seq__10385_10408__$1);
          var G__10411 = c__8079__auto___10409;
          var G__10412 = cljs.core.count(c__8079__auto___10409);
          var G__10413 = 0;
          seq__10385_10395 = G__10410;
          chunk__10386_10396 = G__10411;
          count__10387_10397 = G__10412;
          i__10388_10398 = G__10413;
          continue;
        } else {
          var vec__10392_10414 = cljs.core.first(seq__10385_10408__$1);
          var key_name_10415 = cljs.core.nth.cljs$core$IFn$_invoke$arity$3(vec__10392_10414, 0, null);
          var f_10416 = cljs.core.nth.cljs$core$IFn$_invoke$arity$3(vec__10392_10414, 1, null);
          var str_name_10417 = cljs.core.name.cljs$core$IFn$_invoke$arity$1 ? cljs.core.name.cljs$core$IFn$_invoke$arity$1(key_name_10415) : cljs.core.name.call(null, key_name_10415);
          obj[str_name_10417] = f_10416;
          var G__10418 = cljs.core.next(seq__10385_10408__$1);
          var G__10419 = null;
          var G__10420 = 0;
          var G__10421 = 0;
          seq__10385_10395 = G__10418;
          chunk__10386_10396 = G__10419;
          count__10387_10397 = G__10420;
          i__10388_10398 = G__10421;
          continue;
        }
      } else {
      }
    }
    break;
  }
  return obj;
};
cljs.core.List = function(meta, first, rest, count, __hash) {
  this.meta = meta;
  this.first = first;
  this.rest = rest;
  this.count = count;
  this.__hash = __hash;
  this.cljs$lang$protocol_mask$partition0$ = 65937646;
  this.cljs$lang$protocol_mask$partition1$ = 8192;
};
cljs.core.List.prototype.toString = function() {
  var self__ = this;
  var coll = this;
  return cljs.core.pr_str_STAR_(coll);
};
cljs.core.List.prototype.equiv = function(other) {
  var self__ = this;
  var this$ = this;
  return this$.cljs$core$IEquiv$_equiv$arity$2(null, other);
};
cljs.core.List.prototype.indexOf = function() {
  var G__10422 = null;
  var G__10422__1 = function(x) {
    var self__ = this;
    var coll = this;
    return cljs.core._indexOf.cljs$core$IFn$_invoke$arity$3(coll, x, 0);
  };
  var G__10422__2 = function(x, start) {
    var self__ = this;
    var coll = this;
    return cljs.core._indexOf.cljs$core$IFn$_invoke$arity$3(coll, x, start);
  };
  G__10422 = function(x, start) {
    switch(arguments.length) {
      case 1:
        return G__10422__1.call(this, x);
      case 2:
        return G__10422__2.call(this, x, start);
    }
    throw new Error("Invalid arity: " + (arguments.length - 1));
  };
  G__10422.cljs$core$IFn$_invoke$arity$1 = G__10422__1;
  G__10422.cljs$core$IFn$_invoke$arity$2 = G__10422__2;
  return G__10422;
}();
cljs.core.List.prototype.lastIndexOf = function() {
  var G__10423 = null;
  var G__10423__1 = function(x) {
    var self__ = this;
    var coll = this;
    return cljs.core._lastIndexOf.cljs$core$IFn$_invoke$arity$3(coll, x, self__.count);
  };
  var G__10423__2 = function(x, start) {
    var self__ = this;
    var coll = this;
    return cljs.core._lastIndexOf.cljs$core$IFn$_invoke$arity$3(coll, x, start);
  };
  G__10423 = function(x, start) {
    switch(arguments.length) {
      case 1:
        return G__10423__1.call(this, x);
      case 2:
        return G__10423__2.call(this, x, start);
    }
    throw new Error("Invalid arity: " + (arguments.length - 1));
  };
  G__10423.cljs$core$IFn$_invoke$arity$1 = G__10423__1;
  G__10423.cljs$core$IFn$_invoke$arity$2 = G__10423__2;
  return G__10423;
}();
cljs.core.List.prototype.cljs$core$IMeta$_meta$arity$1 = function(coll) {
  var self__ = this;
  var coll__$1 = this;
  return self__.meta;
};
cljs.core.List.prototype.cljs$core$ICloneable$_clone$arity$1 = function(_) {
  var self__ = this;
  var ___$1 = this;
  return new cljs.core.List(self__.meta, self__.first, self__.rest, self__.count, self__.__hash);
};
cljs.core.List.prototype.cljs$core$INext$_next$arity$1 = function(coll) {
  var self__ = this;
  var coll__$1 = this;
  if (self__.count === 1) {
    return null;
  } else {
    return self__.rest;
  }
};
cljs.core.List.prototype.cljs$core$ICounted$_count$arity$1 = function(coll) {
  var self__ = this;
  var coll__$1 = this;
  return self__.count;
};
cljs.core.List.prototype.cljs$core$IStack$_peek$arity$1 = function(coll) {
  var self__ = this;
  var coll__$1 = this;
  return self__.first;
};
cljs.core.List.prototype.cljs$core$IStack$_pop$arity$1 = function(coll) {
  var self__ = this;
  var coll__$1 = this;
  return coll__$1.cljs$core$ISeq$_rest$arity$1(null);
};
cljs.core.List.prototype.cljs$core$IHash$_hash$arity$1 = function(coll) {
  var self__ = this;
  var coll__$1 = this;
  var h__7700__auto__ = self__.__hash;
  if (!(h__7700__auto__ == null)) {
    return h__7700__auto__;
  } else {
    var h__7700__auto____$1 = cljs.core.hash_ordered_coll(coll__$1);
    self__.__hash = h__7700__auto____$1;
    return h__7700__auto____$1;
  }
};
cljs.core.List.prototype.cljs$core$IEquiv$_equiv$arity$2 = function(coll, other) {
  var self__ = this;
  var coll__$1 = this;
  return cljs.core.equiv_sequential(coll__$1, other);
};
cljs.core.List.prototype.cljs$core$IEmptyableCollection$_empty$arity$1 = function(coll) {
  var self__ = this;
  var coll__$1 = this;
  return cljs.core._with_meta(cljs.core.List.EMPTY, self__.meta);
};
cljs.core.List.prototype.cljs$core$IReduce$_reduce$arity$2 = function(coll, f) {
  var self__ = this;
  var coll__$1 = this;
  return cljs.core.seq_reduce.cljs$core$IFn$_invoke$arity$2(f, coll__$1);
};
cljs.core.List.prototype.cljs$core$IReduce$_reduce$arity$3 = function(coll, f, start) {
  var self__ = this;
  var coll__$1 = this;
  return cljs.core.seq_reduce.cljs$core$IFn$_invoke$arity$3(f, start, coll__$1);
};
cljs.core.List.prototype.cljs$core$ISeq$_first$arity$1 = function(coll) {
  var self__ = this;
  var coll__$1 = this;
  return self__.first;
};
cljs.core.List.prototype.cljs$core$ISeq$_rest$arity$1 = function(coll) {
  var self__ = this;
  var coll__$1 = this;
  if (self__.count === 1) {
    return cljs.core.List.EMPTY;
  } else {
    return self__.rest;
  }
};
cljs.core.List.prototype.cljs$core$ISeqable$_seq$arity$1 = function(coll) {
  var self__ = this;
  var coll__$1 = this;
  return coll__$1;
};
cljs.core.List.prototype.cljs$core$IWithMeta$_with_meta$arity$2 = function(coll, meta__$1) {
  var self__ = this;
  var coll__$1 = this;
  return new cljs.core.List(meta__$1, self__.first, self__.rest, self__.count, self__.__hash);
};
cljs.core.List.prototype.cljs$core$ICollection$_conj$arity$2 = function(coll, o) {
  var self__ = this;
  var coll__$1 = this;
  return new cljs.core.List(self__.meta, o, coll__$1, self__.count + 1, null);
};
cljs.core.List.getBasis = function() {
  return new cljs.core.PersistentVector(null, 5, 5, cljs.core.PersistentVector.EMPTY_NODE, [new cljs.core.Symbol(null, "meta", "meta", -1154898805, null), new cljs.core.Symbol(null, "first", "first", 996428481, null), new cljs.core.Symbol(null, "rest", "rest", 398835108, null), new cljs.core.Symbol(null, "count", "count", -514511684, null), cljs.core.with_meta(new cljs.core.Symbol(null, "__hash", "__hash", -1328796629, null), new cljs.core.PersistentArrayMap(null, 1, [new cljs.core.Keyword(null, 
  "mutable", "mutable", 875778266), true], null))], null);
};
cljs.core.List.cljs$lang$type = true;
cljs.core.List.cljs$lang$ctorStr = "cljs.core/List";
cljs.core.List.cljs$lang$ctorPrWriter = function(this__7871__auto__, writer__7872__auto__, opt__7873__auto__) {
  return cljs.core._write(writer__7872__auto__, "cljs.core/List");
};
cljs.core.__GT_List = function cljs$core$__GT_List(meta, first, rest, count, __hash) {
  return new cljs.core.List(meta, first, rest, count, __hash);
};
cljs.core.list_QMARK_ = function cljs$core$list_QMARK_(x) {
  if (!(x == null)) {
    if (x.cljs$lang$protocol_mask$partition0$ & 33554432 || cljs.core.PROTOCOL_SENTINEL === x.cljs$core$IList$) {
      return true;
    } else {
      if (!x.cljs$lang$protocol_mask$partition0$) {
        return cljs.core.native_satisfies_QMARK_(cljs.core.IList, x);
      } else {
        return false;
      }
    }
  } else {
    return cljs.core.native_satisfies_QMARK_(cljs.core.IList, x);
  }
};
cljs.core.List.prototype[cljs.core.ITER_SYMBOL] = function() {
  var this__8274__auto__ = this;
  return cljs.core.es6_iterator(this__8274__auto__);
};
cljs.core.EmptyList = function(meta) {
  this.meta = meta;
  this.cljs$lang$protocol_mask$partition0$ = 65937614;
  this.cljs$lang$protocol_mask$partition1$ = 8192;
};
cljs.core.EmptyList.prototype.toString = function() {
  var self__ = this;
  var coll = this;
  return cljs.core.pr_str_STAR_(coll);
};
cljs.core.EmptyList.prototype.equiv = function(other) {
  var self__ = this;
  var this$ = this;
  return this$.cljs$core$IEquiv$_equiv$arity$2(null, other);
};
cljs.core.EmptyList.prototype.indexOf = function() {
  var G__10426 = null;
  var G__10426__1 = function(x) {
    var self__ = this;
    var coll = this;
    return cljs.core._indexOf.cljs$core$IFn$_invoke$arity$3(coll, x, 0);
  };
  var G__10426__2 = function(x, start) {
    var self__ = this;
    var coll = this;
    return cljs.core._indexOf.cljs$core$IFn$_invoke$arity$3(coll, x, start);
  };
  G__10426 = function(x, start) {
    switch(arguments.length) {
      case 1:
        return G__10426__1.call(this, x);
      case 2:
        return G__10426__2.call(this, x, start);
    }
    throw new Error("Invalid arity: " + (arguments.length - 1));
  };
  G__10426.cljs$core$IFn$_invoke$arity$1 = G__10426__1;
  G__10426.cljs$core$IFn$_invoke$arity$2 = G__10426__2;
  return G__10426;
}();
cljs.core.EmptyList.prototype.lastIndexOf = function() {
  var G__10427 = null;
  var G__10427__1 = function(x) {
    var self__ = this;
    var coll = this;
    return cljs.core._lastIndexOf.cljs$core$IFn$_invoke$arity$3(coll, x, cljs.core.count(coll));
  };
  var G__10427__2 = function(x, start) {
    var self__ = this;
    var coll = this;
    return cljs.core._lastIndexOf.cljs$core$IFn$_invoke$arity$3(coll, x, start);
  };
  G__10427 = function(x, start) {
    switch(arguments.length) {
      case 1:
        return G__10427__1.call(this, x);
      case 2:
        return G__10427__2.call(this, x, start);
    }
    throw new Error("Invalid arity: " + (arguments.length - 1));
  };
  G__10427.cljs$core$IFn$_invoke$arity$1 = G__10427__1;
  G__10427.cljs$core$IFn$_invoke$arity$2 = G__10427__2;
  return G__10427;
}();
cljs.core.EmptyList.prototype.cljs$core$IMeta$_meta$arity$1 = function(coll) {
  var self__ = this;
  var coll__$1 = this;
  return self__.meta;
};
cljs.core.EmptyList.prototype.cljs$core$ICloneable$_clone$arity$1 = function(_) {
  var self__ = this;
  var ___$1 = this;
  return new cljs.core.EmptyList(self__.meta);
};
cljs.core.EmptyList.prototype.cljs$core$INext$_next$arity$1 = function(coll) {
  var self__ = this;
  var coll__$1 = this;
  return null;
};
cljs.core.EmptyList.prototype.cljs$core$ICounted$_count$arity$1 = function(coll) {
  var self__ = this;
  var coll__$1 = this;
  return 0;
};
cljs.core.EmptyList.prototype.cljs$core$IStack$_peek$arity$1 = function(coll) {
  var self__ = this;
  var coll__$1 = this;
  return null;
};
cljs.core.EmptyList.prototype.cljs$core$IStack$_pop$arity$1 = function(coll) {
  var self__ = this;
  var coll__$1 = this;
  throw new Error("Can't pop empty list");
};
cljs.core.EmptyList.prototype.cljs$core$IHash$_hash$arity$1 = function(coll) {
  var self__ = this;
  var coll__$1 = this;
  return cljs.core.empty_ordered_hash;
};
cljs.core.EmptyList.prototype.cljs$core$IEquiv$_equiv$arity$2 = function(coll, other) {
  var self__ = this;
  var coll__$1 = this;
  if (cljs.core.list_QMARK_(other) || cljs.core.sequential_QMARK_(other)) {
    return cljs.core.seq(other) == null;
  } else {
    return false;
  }
};
cljs.core.EmptyList.prototype.cljs$core$IEmptyableCollection$_empty$arity$1 = function(coll) {
  var self__ = this;
  var coll__$1 = this;
  return coll__$1;
};
cljs.core.EmptyList.prototype.cljs$core$IReduce$_reduce$arity$2 = function(coll, f) {
  var self__ = this;
  var coll__$1 = this;
  return cljs.core.seq_reduce.cljs$core$IFn$_invoke$arity$2(f, coll__$1);
};
cljs.core.EmptyList.prototype.cljs$core$IReduce$_reduce$arity$3 = function(coll, f, start) {
  var self__ = this;
  var coll__$1 = this;
  return cljs.core.seq_reduce.cljs$core$IFn$_invoke$arity$3(f, start, coll__$1);
};
cljs.core.EmptyList.prototype.cljs$core$ISeq$_first$arity$1 = function(coll) {
  var self__ = this;
  var coll__$1 = this;
  return null;
};
cljs.core.EmptyList.prototype.cljs$core$ISeq$_rest$arity$1 = function(coll) {
  var self__ = this;
  var coll__$1 = this;
  return cljs.core.List.EMPTY;
};
cljs.core.EmptyList.prototype.cljs$core$ISeqable$_seq$arity$1 = function(coll) {
  var self__ = this;
  var coll__$1 = this;
  return null;
};
cljs.core.EmptyList.prototype.cljs$core$IWithMeta$_with_meta$arity$2 = function(coll, meta__$1) {
  var self__ = this;
  var coll__$1 = this;
  return new cljs.core.EmptyList(meta__$1);
};
cljs.core.EmptyList.prototype.cljs$core$ICollection$_conj$arity$2 = function(coll, o) {
  var self__ = this;
  var coll__$1 = this;
  return new cljs.core.List(self__.meta, o, null, 1, null);
};
cljs.core.EmptyList.getBasis = function() {
  return new cljs.core.PersistentVector(null, 1, 5, cljs.core.PersistentVector.EMPTY_NODE, [new cljs.core.Symbol(null, "meta", "meta", -1154898805, null)], null);
};
cljs.core.EmptyList.cljs$lang$type = true;
cljs.core.EmptyList.cljs$lang$ctorStr = "cljs.core/EmptyList";
cljs.core.EmptyList.cljs$lang$ctorPrWriter = function(this__7871__auto__, writer__7872__auto__, opt__7873__auto__) {
  return cljs.core._write(writer__7872__auto__, "cljs.core/EmptyList");
};
cljs.core.__GT_EmptyList = function cljs$core$__GT_EmptyList(meta) {
  return new cljs.core.EmptyList(meta);
};
cljs.core.List.EMPTY = new cljs.core.EmptyList(null);
cljs.core.EmptyList.prototype[cljs.core.ITER_SYMBOL] = function() {
  var this__8274__auto__ = this;
  return cljs.core.es6_iterator(this__8274__auto__);
};
cljs.core.reversible_QMARK_ = function cljs$core$reversible_QMARK_(coll) {
  if (!(coll == null)) {
    if (coll.cljs$lang$protocol_mask$partition0$ & 134217728 || cljs.core.PROTOCOL_SENTINEL === coll.cljs$core$IReversible$) {
      return true;
    } else {
      if (!coll.cljs$lang$protocol_mask$partition0$) {
        return cljs.core.native_satisfies_QMARK_(cljs.core.IReversible, coll);
      } else {
        return false;
      }
    }
  } else {
    return cljs.core.native_satisfies_QMARK_(cljs.core.IReversible, coll);
  }
};
cljs.core.rseq = function cljs$core$rseq(rev) {
  return cljs.core._rseq(rev);
};
cljs.core.reverse = function cljs$core$reverse(coll) {
  if (cljs.core.reversible_QMARK_(coll)) {
    return cljs.core.rseq(coll);
  } else {
    return cljs.core.reduce.cljs$core$IFn$_invoke$arity$3(cljs.core.conj, cljs.core.List.EMPTY, coll);
  }
};
cljs.core.list = function cljs$core$list(var_args) {
  var args__8380__auto__ = [];
  var len__8373__auto___10431 = arguments.length;
  var i__8374__auto___10432 = 0;
  while (true) {
    if (i__8374__auto___10432 < len__8373__auto___10431) {
      args__8380__auto__.push(arguments[i__8374__auto___10432]);
      var G__10433 = i__8374__auto___10432 + 1;
      i__8374__auto___10432 = G__10433;
      continue;
    } else {
    }
    break;
  }
  var argseq__8381__auto__ = 0 < args__8380__auto__.length ? new cljs.core.IndexedSeq(args__8380__auto__.slice(0), 0, null) : null;
  return cljs.core.list.cljs$core$IFn$_invoke$arity$variadic(argseq__8381__auto__);
};
cljs.core.list.cljs$core$IFn$_invoke$arity$variadic = function(xs) {
  var arr = xs instanceof cljs.core.IndexedSeq && xs.i === 0 ? xs.arr : function() {
    var arr = [];
    var xs__$1 = xs;
    while (true) {
      if (!(xs__$1 == null)) {
        arr.push(xs__$1.cljs$core$ISeq$_first$arity$1(null));
        var G__10434 = xs__$1.cljs$core$INext$_next$arity$1(null);
        xs__$1 = G__10434;
        continue;
      } else {
        return arr;
      }
      break;
    }
  }();
  var i = arr.length;
  var r = cljs.core.List.EMPTY;
  while (true) {
    if (i > 0) {
      var G__10435 = i - 1;
      var G__10436 = r.cljs$core$ICollection$_conj$arity$2(null, arr[i - 1]);
      i = G__10435;
      r = G__10436;
      continue;
    } else {
      return r;
    }
    break;
  }
};
cljs.core.list.cljs$lang$maxFixedArity = 0;
cljs.core.list.cljs$lang$applyTo = function(seq10430) {
  return cljs.core.list.cljs$core$IFn$_invoke$arity$variadic(cljs.core.seq(seq10430));
};
cljs.core.Cons = function(meta, first, rest, __hash) {
  this.meta = meta;
  this.first = first;
  this.rest = rest;
  this.__hash = __hash;
  this.cljs$lang$protocol_mask$partition0$ = 65929452;
  this.cljs$lang$protocol_mask$partition1$ = 8192;
};
cljs.core.Cons.prototype.toString = function() {
  var self__ = this;
  var coll = this;
  return cljs.core.pr_str_STAR_(coll);
};
cljs.core.Cons.prototype.equiv = function(other) {
  var self__ = this;
  var this$ = this;
  return this$.cljs$core$IEquiv$_equiv$arity$2(null, other);
};
cljs.core.Cons.prototype.indexOf = function() {
  var G__10437 = null;
  var G__10437__1 = function(x) {
    var self__ = this;
    var coll = this;
    return cljs.core._indexOf.cljs$core$IFn$_invoke$arity$3(coll, x, 0);
  };
  var G__10437__2 = function(x, start) {
    var self__ = this;
    var coll = this;
    return cljs.core._indexOf.cljs$core$IFn$_invoke$arity$3(coll, x, start);
  };
  G__10437 = function(x, start) {
    switch(arguments.length) {
      case 1:
        return G__10437__1.call(this, x);
      case 2:
        return G__10437__2.call(this, x, start);
    }
    throw new Error("Invalid arity: " + (arguments.length - 1));
  };
  G__10437.cljs$core$IFn$_invoke$arity$1 = G__10437__1;
  G__10437.cljs$core$IFn$_invoke$arity$2 = G__10437__2;
  return G__10437;
}();
cljs.core.Cons.prototype.lastIndexOf = function() {
  var G__10438 = null;
  var G__10438__1 = function(x) {
    var self__ = this;
    var coll = this;
    return cljs.core._lastIndexOf.cljs$core$IFn$_invoke$arity$3(coll, x, cljs.core.count(coll));
  };
  var G__10438__2 = function(x, start) {
    var self__ = this;
    var coll = this;
    return cljs.core._lastIndexOf.cljs$core$IFn$_invoke$arity$3(coll, x, start);
  };
  G__10438 = function(x, start) {
    switch(arguments.length) {
      case 1:
        return G__10438__1.call(this, x);
      case 2:
        return G__10438__2.call(this, x, start);
    }
    throw new Error("Invalid arity: " + (arguments.length - 1));
  };
  G__10438.cljs$core$IFn$_invoke$arity$1 = G__10438__1;
  G__10438.cljs$core$IFn$_invoke$arity$2 = G__10438__2;
  return G__10438;
}();
cljs.core.Cons.prototype.cljs$core$IMeta$_meta$arity$1 = function(coll) {
  var self__ = this;
  var coll__$1 = this;
  return self__.meta;
};
cljs.core.Cons.prototype.cljs$core$ICloneable$_clone$arity$1 = function(_) {
  var self__ = this;
  var ___$1 = this;
  return new cljs.core.Cons(self__.meta, self__.first, self__.rest, self__.__hash);
};
cljs.core.Cons.prototype.cljs$core$INext$_next$arity$1 = function(coll) {
  var self__ = this;
  var coll__$1 = this;
  if (self__.rest == null) {
    return null;
  } else {
    return cljs.core.seq(self__.rest);
  }
};
cljs.core.Cons.prototype.cljs$core$IHash$_hash$arity$1 = function(coll) {
  var self__ = this;
  var coll__$1 = this;
  var h__7700__auto__ = self__.__hash;
  if (!(h__7700__auto__ == null)) {
    return h__7700__auto__;
  } else {
    var h__7700__auto____$1 = cljs.core.hash_ordered_coll(coll__$1);
    self__.__hash = h__7700__auto____$1;
    return h__7700__auto____$1;
  }
};
cljs.core.Cons.prototype.cljs$core$IEquiv$_equiv$arity$2 = function(coll, other) {
  var self__ = this;
  var coll__$1 = this;
  return cljs.core.equiv_sequential(coll__$1, other);
};
cljs.core.Cons.prototype.cljs$core$IEmptyableCollection$_empty$arity$1 = function(coll) {
  var self__ = this;
  var coll__$1 = this;
  return cljs.core.with_meta(cljs.core.List.EMPTY, self__.meta);
};
cljs.core.Cons.prototype.cljs$core$IReduce$_reduce$arity$2 = function(coll, f) {
  var self__ = this;
  var coll__$1 = this;
  return cljs.core.seq_reduce.cljs$core$IFn$_invoke$arity$2(f, coll__$1);
};
cljs.core.Cons.prototype.cljs$core$IReduce$_reduce$arity$3 = function(coll, f, start) {
  var self__ = this;
  var coll__$1 = this;
  return cljs.core.seq_reduce.cljs$core$IFn$_invoke$arity$3(f, start, coll__$1);
};
cljs.core.Cons.prototype.cljs$core$ISeq$_first$arity$1 = function(coll) {
  var self__ = this;
  var coll__$1 = this;
  return self__.first;
};
cljs.core.Cons.prototype.cljs$core$ISeq$_rest$arity$1 = function(coll) {
  var self__ = this;
  var coll__$1 = this;
  if (self__.rest == null) {
    return cljs.core.List.EMPTY;
  } else {
    return self__.rest;
  }
};
cljs.core.Cons.prototype.cljs$core$ISeqable$_seq$arity$1 = function(coll) {
  var self__ = this;
  var coll__$1 = this;
  return coll__$1;
};
cljs.core.Cons.prototype.cljs$core$IWithMeta$_with_meta$arity$2 = function(coll, meta__$1) {
  var self__ = this;
  var coll__$1 = this;
  return new cljs.core.Cons(meta__$1, self__.first, self__.rest, self__.__hash);
};
cljs.core.Cons.prototype.cljs$core$ICollection$_conj$arity$2 = function(coll, o) {
  var self__ = this;
  var coll__$1 = this;
  return new cljs.core.Cons(null, o, coll__$1, null);
};
cljs.core.Cons.getBasis = function() {
  return new cljs.core.PersistentVector(null, 4, 5, cljs.core.PersistentVector.EMPTY_NODE, [new cljs.core.Symbol(null, "meta", "meta", -1154898805, null), new cljs.core.Symbol(null, "first", "first", 996428481, null), new cljs.core.Symbol(null, "rest", "rest", 398835108, null), cljs.core.with_meta(new cljs.core.Symbol(null, "__hash", "__hash", -1328796629, null), new cljs.core.PersistentArrayMap(null, 1, [new cljs.core.Keyword(null, "mutable", "mutable", 875778266), true], null))], null);
};
cljs.core.Cons.cljs$lang$type = true;
cljs.core.Cons.cljs$lang$ctorStr = "cljs.core/Cons";
cljs.core.Cons.cljs$lang$ctorPrWriter = function(this__7871__auto__, writer__7872__auto__, opt__7873__auto__) {
  return cljs.core._write(writer__7872__auto__, "cljs.core/Cons");
};
cljs.core.__GT_Cons = function cljs$core$__GT_Cons(meta, first, rest, __hash) {
  return new cljs.core.Cons(meta, first, rest, __hash);
};
cljs.core.Cons.prototype[cljs.core.ITER_SYMBOL] = function() {
  var this__8274__auto__ = this;
  return cljs.core.es6_iterator(this__8274__auto__);
};
cljs.core.cons = function cljs$core$cons(x, coll) {
  if (function() {
    var or__7260__auto__ = coll == null;
    if (or__7260__auto__) {
      return or__7260__auto__;
    } else {
      if (!(coll == null)) {
        if (coll.cljs$lang$protocol_mask$partition0$ & 64 || cljs.core.PROTOCOL_SENTINEL === coll.cljs$core$ISeq$) {
          return true;
        } else {
          return false;
        }
      } else {
        return false;
      }
    }
  }()) {
    return new cljs.core.Cons(null, x, coll, null);
  } else {
    return new cljs.core.Cons(null, x, cljs.core.seq(coll), null);
  }
};
cljs.core.hash_keyword = function cljs$core$hash_keyword(k) {
  return cljs.core.hash_symbol(k) + 2654435769 | 0;
};
cljs.core.compare_keywords = function cljs$core$compare_keywords(a, b) {
  if (a.fqn === b.fqn) {
    return 0;
  } else {
    if (cljs.core.truth_(function() {
      var and__7248__auto__ = cljs.core.not(a.ns);
      if (and__7248__auto__) {
        return b.ns;
      } else {
        return and__7248__auto__;
      }
    }())) {
      return -1;
    } else {
      if (cljs.core.truth_(a.ns)) {
        if (cljs.core.not(b.ns)) {
          return 1;
        } else {
          var nsc = function() {
            var G__10449 = a.ns;
            var G__10450 = b.ns;
            return goog.array.defaultCompare(G__10449, G__10450);
          }();
          if (0 === nsc) {
            var G__10451 = a.name;
            var G__10452 = b.name;
            return goog.array.defaultCompare(G__10451, G__10452);
          } else {
            return nsc;
          }
        }
      } else {
        var G__10453 = a.name;
        var G__10454 = b.name;
        return goog.array.defaultCompare(G__10453, G__10454);
      }
    }
  }
};
cljs.core.Keyword = function(ns, name, fqn, _hash) {
  this.ns = ns;
  this.name = name;
  this.fqn = fqn;
  this._hash = _hash;
  this.cljs$lang$protocol_mask$partition0$ = 2153775105;
  this.cljs$lang$protocol_mask$partition1$ = 4096;
};
cljs.core.Keyword.prototype.toString = function() {
  var self__ = this;
  var _ = this;
  return [cljs.core.str.cljs$core$IFn$_invoke$arity$1(":"), cljs.core.str.cljs$core$IFn$_invoke$arity$1(self__.fqn)].join("");
};
cljs.core.Keyword.prototype.equiv = function(other) {
  var self__ = this;
  var this$ = this;
  return this$.cljs$core$IEquiv$_equiv$arity$2(null, other);
};
cljs.core.Keyword.prototype.cljs$core$IEquiv$_equiv$arity$2 = function(_, other) {
  var self__ = this;
  var ___$1 = this;
  if (other instanceof cljs.core.Keyword) {
    return self__.fqn === other.fqn;
  } else {
    return false;
  }
};
cljs.core.Keyword.prototype.call = function() {
  var G__10456 = null;
  var G__10456__2 = function(self__, coll) {
    var self__ = this;
    var self____$1 = this;
    var kw = self____$1;
    return cljs.core.get.cljs$core$IFn$_invoke$arity$2(coll, kw);
  };
  var G__10456__3 = function(self__, coll, not_found) {
    var self__ = this;
    var self____$1 = this;
    var kw = self____$1;
    return cljs.core.get.cljs$core$IFn$_invoke$arity$3(coll, kw, not_found);
  };
  G__10456 = function(self__, coll, not_found) {
    switch(arguments.length) {
      case 2:
        return G__10456__2.call(this, self__, coll);
      case 3:
        return G__10456__3.call(this, self__, coll, not_found);
    }
    throw new Error("Invalid arity: " + (arguments.length - 1));
  };
  G__10456.cljs$core$IFn$_invoke$arity$2 = G__10456__2;
  G__10456.cljs$core$IFn$_invoke$arity$3 = G__10456__3;
  return G__10456;
}();
cljs.core.Keyword.prototype.apply = function(self__, args10455) {
  var self__ = this;
  var self____$1 = this;
  return self____$1.call.apply(self____$1, [self____$1].concat(cljs.core.aclone(args10455)));
};
cljs.core.Keyword.prototype.cljs$core$IFn$_invoke$arity$1 = function(coll) {
  var self__ = this;
  var kw = this;
  return cljs.core.get.cljs$core$IFn$_invoke$arity$2(coll, kw);
};
cljs.core.Keyword.prototype.cljs$core$IFn$_invoke$arity$2 = function(coll, not_found) {
  var self__ = this;
  var kw = this;
  return cljs.core.get.cljs$core$IFn$_invoke$arity$3(coll, kw, not_found);
};
cljs.core.Keyword.prototype.cljs$core$IHash$_hash$arity$1 = function(this$) {
  var self__ = this;
  var this$__$1 = this;
  var h__7700__auto__ = self__._hash;
  if (!(h__7700__auto__ == null)) {
    return h__7700__auto__;
  } else {
    var h__7700__auto____$1 = cljs.core.hash_keyword(this$__$1);
    self__._hash = h__7700__auto____$1;
    return h__7700__auto____$1;
  }
};
cljs.core.Keyword.prototype.cljs$core$INamed$_name$arity$1 = function(_) {
  var self__ = this;
  var ___$1 = this;
  return self__.name;
};
cljs.core.Keyword.prototype.cljs$core$INamed$_namespace$arity$1 = function(_) {
  var self__ = this;
  var ___$1 = this;
  return self__.ns;
};
cljs.core.Keyword.prototype.cljs$core$IPrintWithWriter$_pr_writer$arity$3 = function(o, writer, _) {
  var self__ = this;
  var o__$1 = this;
  return cljs.core._write(writer, [cljs.core.str.cljs$core$IFn$_invoke$arity$1(":"), cljs.core.str.cljs$core$IFn$_invoke$arity$1(self__.fqn)].join(""));
};
cljs.core.Keyword.getBasis = function() {
  return new cljs.core.PersistentVector(null, 4, 5, cljs.core.PersistentVector.EMPTY_NODE, [new cljs.core.Symbol(null, "ns", "ns", 2082130287, null), new cljs.core.Symbol(null, "name", "name", -810760592, null), new cljs.core.Symbol(null, "fqn", "fqn", -1749334463, null), cljs.core.with_meta(new cljs.core.Symbol(null, "_hash", "_hash", -2130838312, null), new cljs.core.PersistentArrayMap(null, 1, [new cljs.core.Keyword(null, "mutable", "mutable", 875778266), true], null))], null);
};
cljs.core.Keyword.cljs$lang$type = true;
cljs.core.Keyword.cljs$lang$ctorStr = "cljs.core/Keyword";
cljs.core.Keyword.cljs$lang$ctorPrWriter = function(this__7871__auto__, writer__7872__auto__, opt__7873__auto__) {
  return cljs.core._write(writer__7872__auto__, "cljs.core/Keyword");
};
cljs.core.__GT_Keyword = function cljs$core$__GT_Keyword(ns, name, fqn, _hash) {
  return new cljs.core.Keyword(ns, name, fqn, _hash);
};
cljs.core.keyword_QMARK_ = function cljs$core$keyword_QMARK_(x) {
  return x instanceof cljs.core.Keyword;
};
cljs.core.keyword_identical_QMARK_ = function cljs$core$keyword_identical_QMARK_(x, y) {
  if (x === y) {
    return true;
  } else {
    if (x instanceof cljs.core.Keyword && y instanceof cljs.core.Keyword) {
      return x.fqn === y.fqn;
    } else {
      return false;
    }
  }
};
cljs.core.symbol_identical_QMARK_ = function cljs$core$symbol_identical_QMARK_(x, y) {
  if (x === y) {
    return true;
  } else {
    if (x instanceof cljs.core.Symbol && y instanceof cljs.core.Symbol) {
      return x.str === y.str;
    } else {
      return false;
    }
  }
};
cljs.core.namespace = function cljs$core$namespace(x) {
  if (!(x == null) ? x.cljs$lang$protocol_mask$partition1$ & 4096 || cljs.core.PROTOCOL_SENTINEL === x.cljs$core$INamed$ ? true : false : false) {
    return x.cljs$core$INamed$_namespace$arity$1(null);
  } else {
    throw new Error([cljs.core.str.cljs$core$IFn$_invoke$arity$1("Doesn't support namespace: "), cljs.core.str.cljs$core$IFn$_invoke$arity$1(x)].join(""));
  }
};
cljs.core.ident_QMARK_ = function cljs$core$ident_QMARK_(x) {
  return x instanceof cljs.core.Keyword || x instanceof cljs.core.Symbol;
};
cljs.core.simple_ident_QMARK_ = function cljs$core$simple_ident_QMARK_(x) {
  return cljs.core.ident_QMARK_(x) && cljs.core.namespace(x) == null;
};
cljs.core.qualified_ident_QMARK_ = function cljs$core$qualified_ident_QMARK_(x) {
  var and__7248__auto__ = cljs.core.ident_QMARK_(x);
  if (and__7248__auto__) {
    var and__7248__auto____$1 = cljs.core.namespace(x);
    if (cljs.core.truth_(and__7248__auto____$1)) {
      return true;
    } else {
      return and__7248__auto____$1;
    }
  } else {
    return and__7248__auto__;
  }
};
cljs.core.simple_symbol_QMARK_ = function cljs$core$simple_symbol_QMARK_(x) {
  return x instanceof cljs.core.Symbol && cljs.core.namespace(x) == null;
};
cljs.core.qualified_symbol_QMARK_ = function cljs$core$qualified_symbol_QMARK_(x) {
  var and__7248__auto__ = x instanceof cljs.core.Symbol;
  if (and__7248__auto__) {
    var and__7248__auto____$1 = cljs.core.namespace(x);
    if (cljs.core.truth_(and__7248__auto____$1)) {
      return true;
    } else {
      return and__7248__auto____$1;
    }
  } else {
    return and__7248__auto__;
  }
};
cljs.core.simple_keyword_QMARK_ = function cljs$core$simple_keyword_QMARK_(x) {
  return x instanceof cljs.core.Keyword && cljs.core.namespace(x) == null;
};
cljs.core.qualified_keyword_QMARK_ = function cljs$core$qualified_keyword_QMARK_(x) {
  var and__7248__auto__ = x instanceof cljs.core.Keyword;
  if (and__7248__auto__) {
    var and__7248__auto____$1 = cljs.core.namespace(x);
    if (cljs.core.truth_(and__7248__auto____$1)) {
      return true;
    } else {
      return and__7248__auto____$1;
    }
  } else {
    return and__7248__auto__;
  }
};
cljs.core.keyword = function cljs$core$keyword(var_args) {
  var args10459 = [];
  var len__8373__auto___10462 = arguments.length;
  var i__8374__auto___10463 = 0;
  while (true) {
    if (i__8374__auto___10463 < len__8373__auto___10462) {
      args10459.push(arguments[i__8374__auto___10463]);
      var G__10464 = i__8374__auto___10463 + 1;
      i__8374__auto___10463 = G__10464;
      continue;
    } else {
    }
    break;
  }
  var G__10461 = args10459.length;
  switch(G__10461) {
    case 1:
      return cljs.core.keyword.cljs$core$IFn$_invoke$arity$1(arguments[0]);
      break;
    case 2:
      return cljs.core.keyword.cljs$core$IFn$_invoke$arity$2(arguments[0], arguments[1]);
      break;
    default:
      throw new Error([cljs.core.str.cljs$core$IFn$_invoke$arity$1("Invalid arity: "), cljs.core.str.cljs$core$IFn$_invoke$arity$1(args10459.length)].join(""));
  }
};
cljs.core.keyword.cljs$core$IFn$_invoke$arity$1 = function(name) {
  if (name instanceof cljs.core.Keyword) {
    return name;
  } else {
    if (name instanceof cljs.core.Symbol) {
      return new cljs.core.Keyword(cljs.core.namespace(name), cljs.core.name.cljs$core$IFn$_invoke$arity$1 ? cljs.core.name.cljs$core$IFn$_invoke$arity$1(name) : cljs.core.name.call(null, name), name.str, null);
    } else {
      if (typeof name === "string") {
        var parts = name.split("/");
        if (parts.length === 2) {
          return new cljs.core.Keyword(parts[0], parts[1], name, null);
        } else {
          return new cljs.core.Keyword(null, parts[0], name, null);
        }
      } else {
        return null;
      }
    }
  }
};
cljs.core.keyword.cljs$core$IFn$_invoke$arity$2 = function(ns, name) {
  var ns__$1 = ns instanceof cljs.core.Keyword ? cljs.core.name.cljs$core$IFn$_invoke$arity$1 ? cljs.core.name.cljs$core$IFn$_invoke$arity$1(ns) : cljs.core.name.call(null, ns) : ns instanceof cljs.core.Symbol ? cljs.core.name.cljs$core$IFn$_invoke$arity$1 ? cljs.core.name.cljs$core$IFn$_invoke$arity$1(ns) : cljs.core.name.call(null, ns) : ns;
  var name__$1 = name instanceof cljs.core.Keyword ? cljs.core.name.cljs$core$IFn$_invoke$arity$1 ? cljs.core.name.cljs$core$IFn$_invoke$arity$1(name) : cljs.core.name.call(null, name) : name instanceof cljs.core.Symbol ? cljs.core.name.cljs$core$IFn$_invoke$arity$1 ? cljs.core.name.cljs$core$IFn$_invoke$arity$1(name) : cljs.core.name.call(null, name) : name;
  return new cljs.core.Keyword(ns__$1, name__$1, [cljs.core.str.cljs$core$IFn$_invoke$arity$1(cljs.core.truth_(ns__$1) ? [cljs.core.str.cljs$core$IFn$_invoke$arity$1(ns__$1), cljs.core.str.cljs$core$IFn$_invoke$arity$1("/")].join("") : null), cljs.core.str.cljs$core$IFn$_invoke$arity$1(name__$1)].join(""), null);
};
cljs.core.keyword.cljs$lang$maxFixedArity = 2;
cljs.core.LazySeq = function(meta, fn, s, __hash) {
  this.meta = meta;
  this.fn = fn;
  this.s = s;
  this.__hash = __hash;
  this.cljs$lang$protocol_mask$partition0$ = 32374988;
  this.cljs$lang$protocol_mask$partition1$ = 1;
};
cljs.core.LazySeq.prototype.toString = function() {
  var self__ = this;
  var coll = this;
  return cljs.core.pr_str_STAR_(coll);
};
cljs.core.LazySeq.prototype.equiv = function(other) {
  var self__ = this;
  var this$ = this;
  return this$.cljs$core$IEquiv$_equiv$arity$2(null, other);
};
cljs.core.LazySeq.prototype.sval = function() {
  var self__ = this;
  var coll = this;
  if (self__.fn == null) {
    return self__.s;
  } else {
    self__.s = self__.fn.cljs$core$IFn$_invoke$arity$0 ? self__.fn.cljs$core$IFn$_invoke$arity$0() : self__.fn.call(null);
    self__.fn = null;
    return self__.s;
  }
};
cljs.core.LazySeq.prototype.indexOf = function() {
  var G__10466 = null;
  var G__10466__1 = function(x) {
    var self__ = this;
    var coll = this;
    return cljs.core._indexOf.cljs$core$IFn$_invoke$arity$3(coll, x, 0);
  };
  var G__10466__2 = function(x, start) {
    var self__ = this;
    var coll = this;
    return cljs.core._indexOf.cljs$core$IFn$_invoke$arity$3(coll, x, start);
  };
  G__10466 = function(x, start) {
    switch(arguments.length) {
      case 1:
        return G__10466__1.call(this, x);
      case 2:
        return G__10466__2.call(this, x, start);
    }
    throw new Error("Invalid arity: " + (arguments.length - 1));
  };
  G__10466.cljs$core$IFn$_invoke$arity$1 = G__10466__1;
  G__10466.cljs$core$IFn$_invoke$arity$2 = G__10466__2;
  return G__10466;
}();
cljs.core.LazySeq.prototype.lastIndexOf = function() {
  var G__10467 = null;
  var G__10467__1 = function(x) {
    var self__ = this;
    var coll = this;
    return cljs.core._lastIndexOf.cljs$core$IFn$_invoke$arity$3(coll, x, cljs.core.count(coll));
  };
  var G__10467__2 = function(x, start) {
    var self__ = this;
    var coll = this;
    return cljs.core._lastIndexOf.cljs$core$IFn$_invoke$arity$3(coll, x, start);
  };
  G__10467 = function(x, start) {
    switch(arguments.length) {
      case 1:
        return G__10467__1.call(this, x);
      case 2:
        return G__10467__2.call(this, x, start);
    }
    throw new Error("Invalid arity: " + (arguments.length - 1));
  };
  G__10467.cljs$core$IFn$_invoke$arity$1 = G__10467__1;
  G__10467.cljs$core$IFn$_invoke$arity$2 = G__10467__2;
  return G__10467;
}();
cljs.core.LazySeq.prototype.cljs$core$IMeta$_meta$arity$1 = function(coll) {
  var self__ = this;
  var coll__$1 = this;
  return self__.meta;
};
cljs.core.LazySeq.prototype.cljs$core$INext$_next$arity$1 = function(coll) {
  var self__ = this;
  var coll__$1 = this;
  coll__$1.cljs$core$ISeqable$_seq$arity$1(null);
  if (self__.s == null) {
    return null;
  } else {
    return cljs.core.next(self__.s);
  }
};
cljs.core.LazySeq.prototype.cljs$core$IHash$_hash$arity$1 = function(coll) {
  var self__ = this;
  var coll__$1 = this;
  var h__7700__auto__ = self__.__hash;
  if (!(h__7700__auto__ == null)) {
    return h__7700__auto__;
  } else {
    var h__7700__auto____$1 = cljs.core.hash_ordered_coll(coll__$1);
    self__.__hash = h__7700__auto____$1;
    return h__7700__auto____$1;
  }
};
cljs.core.LazySeq.prototype.cljs$core$IEquiv$_equiv$arity$2 = function(coll, other) {
  var self__ = this;
  var coll__$1 = this;
  return cljs.core.equiv_sequential(coll__$1, other);
};
cljs.core.LazySeq.prototype.cljs$core$IEmptyableCollection$_empty$arity$1 = function(coll) {
  var self__ = this;
  var coll__$1 = this;
  return cljs.core.with_meta(cljs.core.List.EMPTY, self__.meta);
};
cljs.core.LazySeq.prototype.cljs$core$IPending$_realized_QMARK_$arity$1 = function(coll) {
  var self__ = this;
  var coll__$1 = this;
  return cljs.core.not(self__.fn);
};
cljs.core.LazySeq.prototype.cljs$core$IReduce$_reduce$arity$2 = function(coll, f) {
  var self__ = this;
  var coll__$1 = this;
  return cljs.core.seq_reduce.cljs$core$IFn$_invoke$arity$2(f, coll__$1);
};
cljs.core.LazySeq.prototype.cljs$core$IReduce$_reduce$arity$3 = function(coll, f, start) {
  var self__ = this;
  var coll__$1 = this;
  return cljs.core.seq_reduce.cljs$core$IFn$_invoke$arity$3(f, start, coll__$1);
};
cljs.core.LazySeq.prototype.cljs$core$ISeq$_first$arity$1 = function(coll) {
  var self__ = this;
  var coll__$1 = this;
  coll__$1.cljs$core$ISeqable$_seq$arity$1(null);
  if (self__.s == null) {
    return null;
  } else {
    return cljs.core.first(self__.s);
  }
};
cljs.core.LazySeq.prototype.cljs$core$ISeq$_rest$arity$1 = function(coll) {
  var self__ = this;
  var coll__$1 = this;
  coll__$1.cljs$core$ISeqable$_seq$arity$1(null);
  if (!(self__.s == null)) {
    return cljs.core.rest(self__.s);
  } else {
    return cljs.core.List.EMPTY;
  }
};
cljs.core.LazySeq.prototype.cljs$core$ISeqable$_seq$arity$1 = function(coll) {
  var self__ = this;
  var coll__$1 = this;
  coll__$1.sval();
  if (self__.s == null) {
    return null;
  } else {
    var ls = self__.s;
    while (true) {
      if (ls instanceof cljs.core.LazySeq) {
        var G__10468 = ls.sval();
        ls = G__10468;
        continue;
      } else {
        self__.s = ls;
        return cljs.core.seq(self__.s);
      }
      break;
    }
  }
};
cljs.core.LazySeq.prototype.cljs$core$IWithMeta$_with_meta$arity$2 = function(coll, meta__$1) {
  var self__ = this;
  var coll__$1 = this;
  return new cljs.core.LazySeq(meta__$1, self__.fn, self__.s, self__.__hash);
};
cljs.core.LazySeq.prototype.cljs$core$ICollection$_conj$arity$2 = function(coll, o) {
  var self__ = this;
  var coll__$1 = this;
  return cljs.core.cons(o, coll__$1);
};
cljs.core.LazySeq.getBasis = function() {
  return new cljs.core.PersistentVector(null, 4, 5, cljs.core.PersistentVector.EMPTY_NODE, [new cljs.core.Symbol(null, "meta", "meta", -1154898805, null), cljs.core.with_meta(new cljs.core.Symbol(null, "fn", "fn", 465265323, null), new cljs.core.PersistentArrayMap(null, 1, [new cljs.core.Keyword(null, "mutable", "mutable", 875778266), true], null)), cljs.core.with_meta(new cljs.core.Symbol(null, "s", "s", -948495851, null), new cljs.core.PersistentArrayMap(null, 1, [new cljs.core.Keyword(null, "mutable", 
  "mutable", 875778266), true], null)), cljs.core.with_meta(new cljs.core.Symbol(null, "__hash", "__hash", -1328796629, null), new cljs.core.PersistentArrayMap(null, 1, [new cljs.core.Keyword(null, "mutable", "mutable", 875778266), true], null))], null);
};
cljs.core.LazySeq.cljs$lang$type = true;
cljs.core.LazySeq.cljs$lang$ctorStr = "cljs.core/LazySeq";
cljs.core.LazySeq.cljs$lang$ctorPrWriter = function(this__7871__auto__, writer__7872__auto__, opt__7873__auto__) {
  return cljs.core._write(writer__7872__auto__, "cljs.core/LazySeq");
};
cljs.core.__GT_LazySeq = function cljs$core$__GT_LazySeq(meta, fn, s, __hash) {
  return new cljs.core.LazySeq(meta, fn, s, __hash);
};
cljs.core.LazySeq.prototype[cljs.core.ITER_SYMBOL] = function() {
  var this__8274__auto__ = this;
  return cljs.core.es6_iterator(this__8274__auto__);
};
cljs.core.ChunkBuffer = function(buf, end) {
  this.buf = buf;
  this.end = end;
  this.cljs$lang$protocol_mask$partition0$ = 2;
  this.cljs$lang$protocol_mask$partition1$ = 0;
};
cljs.core.ChunkBuffer.prototype.add = function(o) {
  var self__ = this;
  var _ = this;
  self__.buf[self__.end] = o;
  return self__.end = self__.end + 1;
};
cljs.core.ChunkBuffer.prototype.chunk = function(o) {
  var self__ = this;
  var _ = this;
  var ret = new cljs.core.ArrayChunk(self__.buf, 0, self__.end);
  self__.buf = null;
  return ret;
};
cljs.core.ChunkBuffer.prototype.cljs$core$ICounted$_count$arity$1 = function(_) {
  var self__ = this;
  var ___$1 = this;
  return self__.end;
};
cljs.core.ChunkBuffer.getBasis = function() {
  return new cljs.core.PersistentVector(null, 2, 5, cljs.core.PersistentVector.EMPTY_NODE, [cljs.core.with_meta(new cljs.core.Symbol(null, "buf", "buf", 1426618187, null), new cljs.core.PersistentArrayMap(null, 1, [new cljs.core.Keyword(null, "mutable", "mutable", 875778266), true], null)), cljs.core.with_meta(new cljs.core.Symbol(null, "end", "end", 1372345569, null), new cljs.core.PersistentArrayMap(null, 1, [new cljs.core.Keyword(null, "mutable", "mutable", 875778266), true], null))], null);
};
cljs.core.ChunkBuffer.cljs$lang$type = true;
cljs.core.ChunkBuffer.cljs$lang$ctorStr = "cljs.core/ChunkBuffer";
cljs.core.ChunkBuffer.cljs$lang$ctorPrWriter = function(this__7871__auto__, writer__7872__auto__, opt__7873__auto__) {
  return cljs.core._write(writer__7872__auto__, "cljs.core/ChunkBuffer");
};
cljs.core.__GT_ChunkBuffer = function cljs$core$__GT_ChunkBuffer(buf, end) {
  return new cljs.core.ChunkBuffer(buf, end);
};
cljs.core.chunk_buffer = function cljs$core$chunk_buffer(capacity) {
  return new cljs.core.ChunkBuffer(new Array(capacity), 0);
};
cljs.core.ArrayChunk = function(arr, off, end) {
  this.arr = arr;
  this.off = off;
  this.end = end;
  this.cljs$lang$protocol_mask$partition0$ = 524306;
  this.cljs$lang$protocol_mask$partition1$ = 0;
};
cljs.core.ArrayChunk.prototype.cljs$core$ICounted$_count$arity$1 = function(_) {
  var self__ = this;
  var ___$1 = this;
  return self__.end - self__.off;
};
cljs.core.ArrayChunk.prototype.cljs$core$IIndexed$_nth$arity$2 = function(coll, i) {
  var self__ = this;
  var coll__$1 = this;
  return self__.arr[self__.off + i];
};
cljs.core.ArrayChunk.prototype.cljs$core$IIndexed$_nth$arity$3 = function(coll, i, not_found) {
  var self__ = this;
  var coll__$1 = this;
  if (i >= 0 && i < self__.end - self__.off) {
    return self__.arr[self__.off + i];
  } else {
    return not_found;
  }
};
cljs.core.ArrayChunk.prototype.cljs$core$IChunk$ = cljs.core.PROTOCOL_SENTINEL;
cljs.core.ArrayChunk.prototype.cljs$core$IChunk$_drop_first$arity$1 = function(coll) {
  var self__ = this;
  var coll__$1 = this;
  if (self__.off === self__.end) {
    throw new Error("-drop-first of empty chunk");
  } else {
    return new cljs.core.ArrayChunk(self__.arr, self__.off + 1, self__.end);
  }
};
cljs.core.ArrayChunk.prototype.cljs$core$IReduce$_reduce$arity$2 = function(coll, f) {
  var self__ = this;
  var coll__$1 = this;
  return cljs.core.array_reduce.cljs$core$IFn$_invoke$arity$4(self__.arr, f, self__.arr[self__.off], self__.off + 1);
};
cljs.core.ArrayChunk.prototype.cljs$core$IReduce$_reduce$arity$3 = function(coll, f, start) {
  var self__ = this;
  var coll__$1 = this;
  return cljs.core.array_reduce.cljs$core$IFn$_invoke$arity$4(self__.arr, f, start, self__.off);
};
cljs.core.ArrayChunk.getBasis = function() {
  return new cljs.core.PersistentVector(null, 3, 5, cljs.core.PersistentVector.EMPTY_NODE, [new cljs.core.Symbol(null, "arr", "arr", 2115492975, null), new cljs.core.Symbol(null, "off", "off", -2047994980, null), new cljs.core.Symbol(null, "end", "end", 1372345569, null)], null);
};
cljs.core.ArrayChunk.cljs$lang$type = true;
cljs.core.ArrayChunk.cljs$lang$ctorStr = "cljs.core/ArrayChunk";
cljs.core.ArrayChunk.cljs$lang$ctorPrWriter = function(this__7871__auto__, writer__7872__auto__, opt__7873__auto__) {
  return cljs.core._write(writer__7872__auto__, "cljs.core/ArrayChunk");
};
cljs.core.__GT_ArrayChunk = function cljs$core$__GT_ArrayChunk(arr, off, end) {
  return new cljs.core.ArrayChunk(arr, off, end);
};
cljs.core.array_chunk = function cljs$core$array_chunk(var_args) {
  var args10469 = [];
  var len__8373__auto___10472 = arguments.length;
  var i__8374__auto___10473 = 0;
  while (true) {
    if (i__8374__auto___10473 < len__8373__auto___10472) {
      args10469.push(arguments[i__8374__auto___10473]);
      var G__10474 = i__8374__auto___10473 + 1;
      i__8374__auto___10473 = G__10474;
      continue;
    } else {
    }
    break;
  }
  var G__10471 = args10469.length;
  switch(G__10471) {
    case 1:
      return cljs.core.array_chunk.cljs$core$IFn$_invoke$arity$1(arguments[0]);
      break;
    case 2:
      return cljs.core.array_chunk.cljs$core$IFn$_invoke$arity$2(arguments[0], arguments[1]);
      break;
    case 3:
      return cljs.core.array_chunk.cljs$core$IFn$_invoke$arity$3(arguments[0], arguments[1], arguments[2]);
      break;
    default:
      throw new Error([cljs.core.str.cljs$core$IFn$_invoke$arity$1("Invalid arity: "), cljs.core.str.cljs$core$IFn$_invoke$arity$1(args10469.length)].join(""));
  }
};
cljs.core.array_chunk.cljs$core$IFn$_invoke$arity$1 = function(arr) {
  return new cljs.core.ArrayChunk(arr, 0, arr.length);
};
cljs.core.array_chunk.cljs$core$IFn$_invoke$arity$2 = function(arr, off) {
  return new cljs.core.ArrayChunk(arr, off, arr.length);
};
cljs.core.array_chunk.cljs$core$IFn$_invoke$arity$3 = function(arr, off, end) {
  return new cljs.core.ArrayChunk(arr, off, end);
};
cljs.core.array_chunk.cljs$lang$maxFixedArity = 3;
cljs.core.ChunkedCons = function(chunk, more, meta, __hash) {
  this.chunk = chunk;
  this.more = more;
  this.meta = meta;
  this.__hash = __hash;
  this.cljs$lang$protocol_mask$partition0$ = 31850732;
  this.cljs$lang$protocol_mask$partition1$ = 1536;
};
cljs.core.ChunkedCons.prototype.toString = function() {
  var self__ = this;
  var coll = this;
  return cljs.core.pr_str_STAR_(coll);
};
cljs.core.ChunkedCons.prototype.equiv = function(other) {
  var self__ = this;
  var this$ = this;
  return this$.cljs$core$IEquiv$_equiv$arity$2(null, other);
};
cljs.core.ChunkedCons.prototype.indexOf = function() {
  var G__10476 = null;
  var G__10476__1 = function(x) {
    var self__ = this;
    var coll = this;
    return cljs.core._indexOf.cljs$core$IFn$_invoke$arity$3(coll, x, 0);
  };
  var G__10476__2 = function(x, start) {
    var self__ = this;
    var coll = this;
    return cljs.core._indexOf.cljs$core$IFn$_invoke$arity$3(coll, x, start);
  };
  G__10476 = function(x, start) {
    switch(arguments.length) {
      case 1:
        return G__10476__1.call(this, x);
      case 2:
        return G__10476__2.call(this, x, start);
    }
    throw new Error("Invalid arity: " + (arguments.length - 1));
  };
  G__10476.cljs$core$IFn$_invoke$arity$1 = G__10476__1;
  G__10476.cljs$core$IFn$_invoke$arity$2 = G__10476__2;
  return G__10476;
}();
cljs.core.ChunkedCons.prototype.lastIndexOf = function() {
  var G__10477 = null;
  var G__10477__1 = function(x) {
    var self__ = this;
    var coll = this;
    return cljs.core._lastIndexOf.cljs$core$IFn$_invoke$arity$3(coll, x, cljs.core.count(coll));
  };
  var G__10477__2 = function(x, start) {
    var self__ = this;
    var coll = this;
    return cljs.core._lastIndexOf.cljs$core$IFn$_invoke$arity$3(coll, x, start);
  };
  G__10477 = function(x, start) {
    switch(arguments.length) {
      case 1:
        return G__10477__1.call(this, x);
      case 2:
        return G__10477__2.call(this, x, start);
    }
    throw new Error("Invalid arity: " + (arguments.length - 1));
  };
  G__10477.cljs$core$IFn$_invoke$arity$1 = G__10477__1;
  G__10477.cljs$core$IFn$_invoke$arity$2 = G__10477__2;
  return G__10477;
}();
cljs.core.ChunkedCons.prototype.cljs$core$IMeta$_meta$arity$1 = function(coll) {
  var self__ = this;
  var coll__$1 = this;
  return self__.meta;
};
cljs.core.ChunkedCons.prototype.cljs$core$INext$_next$arity$1 = function(coll) {
  var self__ = this;
  var coll__$1 = this;
  if (cljs.core._count(self__.chunk) > 1) {
    return new cljs.core.ChunkedCons(cljs.core._drop_first(self__.chunk), self__.more, self__.meta, null);
  } else {
    var more__$1 = cljs.core._seq(self__.more);
    if (more__$1 == null) {
      return null;
    } else {
      return more__$1;
    }
  }
};
cljs.core.ChunkedCons.prototype.cljs$core$IHash$_hash$arity$1 = function(coll) {
  var self__ = this;
  var coll__$1 = this;
  var h__7700__auto__ = self__.__hash;
  if (!(h__7700__auto__ == null)) {
    return h__7700__auto__;
  } else {
    var h__7700__auto____$1 = cljs.core.hash_ordered_coll(coll__$1);
    self__.__hash = h__7700__auto____$1;
    return h__7700__auto____$1;
  }
};
cljs.core.ChunkedCons.prototype.cljs$core$IEquiv$_equiv$arity$2 = function(coll, other) {
  var self__ = this;
  var coll__$1 = this;
  return cljs.core.equiv_sequential(coll__$1, other);
};
cljs.core.ChunkedCons.prototype.cljs$core$IEmptyableCollection$_empty$arity$1 = function(coll) {
  var self__ = this;
  var coll__$1 = this;
  return cljs.core.with_meta(cljs.core.List.EMPTY, self__.meta);
};
cljs.core.ChunkedCons.prototype.cljs$core$ISeq$_first$arity$1 = function(coll) {
  var self__ = this;
  var coll__$1 = this;
  return cljs.core._nth.cljs$core$IFn$_invoke$arity$2(self__.chunk, 0);
};
cljs.core.ChunkedCons.prototype.cljs$core$ISeq$_rest$arity$1 = function(coll) {
  var self__ = this;
  var coll__$1 = this;
  if (cljs.core._count(self__.chunk) > 1) {
    return new cljs.core.ChunkedCons(cljs.core._drop_first(self__.chunk), self__.more, self__.meta, null);
  } else {
    if (self__.more == null) {
      return cljs.core.List.EMPTY;
    } else {
      return self__.more;
    }
  }
};
cljs.core.ChunkedCons.prototype.cljs$core$ISeqable$_seq$arity$1 = function(coll) {
  var self__ = this;
  var coll__$1 = this;
  return coll__$1;
};
cljs.core.ChunkedCons.prototype.cljs$core$IChunkedSeq$_chunked_first$arity$1 = function(coll) {
  var self__ = this;
  var coll__$1 = this;
  return self__.chunk;
};
cljs.core.ChunkedCons.prototype.cljs$core$IChunkedSeq$_chunked_rest$arity$1 = function(coll) {
  var self__ = this;
  var coll__$1 = this;
  if (self__.more == null) {
    return cljs.core.List.EMPTY;
  } else {
    return self__.more;
  }
};
cljs.core.ChunkedCons.prototype.cljs$core$IWithMeta$_with_meta$arity$2 = function(coll, m) {
  var self__ = this;
  var coll__$1 = this;
  return new cljs.core.ChunkedCons(self__.chunk, self__.more, m, self__.__hash);
};
cljs.core.ChunkedCons.prototype.cljs$core$ICollection$_conj$arity$2 = function(this$, o) {
  var self__ = this;
  var this$__$1 = this;
  return cljs.core.cons(o, this$__$1);
};
cljs.core.ChunkedCons.prototype.cljs$core$IChunkedNext$_chunked_next$arity$1 = function(coll) {
  var self__ = this;
  var coll__$1 = this;
  if (self__.more == null) {
    return null;
  } else {
    return self__.more;
  }
};
cljs.core.ChunkedCons.getBasis = function() {
  return new cljs.core.PersistentVector(null, 4, 5, cljs.core.PersistentVector.EMPTY_NODE, [new cljs.core.Symbol(null, "chunk", "chunk", 449371907, null), new cljs.core.Symbol(null, "more", "more", -418290273, null), new cljs.core.Symbol(null, "meta", "meta", -1154898805, null), cljs.core.with_meta(new cljs.core.Symbol(null, "__hash", "__hash", -1328796629, null), new cljs.core.PersistentArrayMap(null, 1, [new cljs.core.Keyword(null, "mutable", "mutable", 875778266), true], null))], null);
};
cljs.core.ChunkedCons.cljs$lang$type = true;
cljs.core.ChunkedCons.cljs$lang$ctorStr = "cljs.core/ChunkedCons";
cljs.core.ChunkedCons.cljs$lang$ctorPrWriter = function(this__7871__auto__, writer__7872__auto__, opt__7873__auto__) {
  return cljs.core._write(writer__7872__auto__, "cljs.core/ChunkedCons");
};
cljs.core.__GT_ChunkedCons = function cljs$core$__GT_ChunkedCons(chunk, more, meta, __hash) {
  return new cljs.core.ChunkedCons(chunk, more, meta, __hash);
};
cljs.core.ChunkedCons.prototype[cljs.core.ITER_SYMBOL] = function() {
  var this__8274__auto__ = this;
  return cljs.core.es6_iterator(this__8274__auto__);
};
cljs.core.chunk_cons = function cljs$core$chunk_cons(chunk, rest) {
  if (cljs.core._count(chunk) === 0) {
    return rest;
  } else {
    return new cljs.core.ChunkedCons(chunk, rest, null, null);
  }
};
cljs.core.chunk_append = function cljs$core$chunk_append(b, x) {
  return b.add(x);
};
cljs.core.chunk = function cljs$core$chunk(b) {
  return b.chunk();
};
cljs.core.chunk_first = function cljs$core$chunk_first(s) {
  return cljs.core._chunked_first(s);
};
cljs.core.chunk_rest = function cljs$core$chunk_rest(s) {
  return cljs.core._chunked_rest(s);
};
cljs.core.chunk_next = function cljs$core$chunk_next(s) {
  if (!(s == null) ? s.cljs$lang$protocol_mask$partition1$ & 1024 || cljs.core.PROTOCOL_SENTINEL === s.cljs$core$IChunkedNext$ ? true : false : false) {
    return cljs.core._chunked_next(s);
  } else {
    return cljs.core.seq(cljs.core._chunked_rest(s));
  }
};
cljs.core.to_array = function cljs$core$to_array(s) {
  var ary = [];
  var s__$1 = s;
  while (true) {
    if (cljs.core.seq(s__$1)) {
      ary.push(cljs.core.first(s__$1));
      var G__10480 = cljs.core.next(s__$1);
      s__$1 = G__10480;
      continue;
    } else {
      return ary;
    }
    break;
  }
};
cljs.core.to_array_2d = function cljs$core$to_array_2d(coll) {
  var ret = new Array(cljs.core.count(coll));
  var i_10481 = 0;
  var xs_10482 = cljs.core.seq(coll);
  while (true) {
    if (xs_10482) {
      ret[i_10481] = cljs.core.to_array(cljs.core.first(xs_10482));
      var G__10483 = i_10481 + 1;
      var G__10484 = cljs.core.next(xs_10482);
      i_10481 = G__10483;
      xs_10482 = G__10484;
      continue;
    } else {
    }
    break;
  }
  return ret;
};
cljs.core.int_array = function cljs$core$int_array(var_args) {
  var args10485 = [];
  var len__8373__auto___10488 = arguments.length;
  var i__8374__auto___10489 = 0;
  while (true) {
    if (i__8374__auto___10489 < len__8373__auto___10488) {
      args10485.push(arguments[i__8374__auto___10489]);
      var G__10490 = i__8374__auto___10489 + 1;
      i__8374__auto___10489 = G__10490;
      continue;
    } else {
    }
    break;
  }
  var G__10487 = args10485.length;
  switch(G__10487) {
    case 1:
      return cljs.core.int_array.cljs$core$IFn$_invoke$arity$1(arguments[0]);
      break;
    case 2:
      return cljs.core.int_array.cljs$core$IFn$_invoke$arity$2(arguments[0], arguments[1]);
      break;
    default:
      throw new Error([cljs.core.str.cljs$core$IFn$_invoke$arity$1("Invalid arity: "), cljs.core.str.cljs$core$IFn$_invoke$arity$1(args10485.length)].join(""));
  }
};
cljs.core.int_array.cljs$core$IFn$_invoke$arity$1 = function(size_or_seq) {
  if (typeof size_or_seq === "number") {
    return cljs.core.int_array.cljs$core$IFn$_invoke$arity$2(size_or_seq, null);
  } else {
    return cljs.core.into_array.cljs$core$IFn$_invoke$arity$1(size_or_seq);
  }
};
cljs.core.int_array.cljs$core$IFn$_invoke$arity$2 = function(size, init_val_or_seq) {
  var a = new Array(size);
  if (cljs.core.seq_QMARK_(init_val_or_seq)) {
    var s = cljs.core.seq(init_val_or_seq);
    var i = 0;
    var s__$1 = s;
    while (true) {
      if (s__$1 && i < size) {
        a[i] = cljs.core.first(s__$1);
        var G__10492 = i + 1;
        var G__10493 = cljs.core.next(s__$1);
        i = G__10492;
        s__$1 = G__10493;
        continue;
      } else {
        return a;
      }
      break;
    }
  } else {
    var n__8183__auto___10494 = size;
    var i_10495 = 0;
    while (true) {
      if (i_10495 < n__8183__auto___10494) {
        a[i_10495] = init_val_or_seq;
        var G__10496 = i_10495 + 1;
        i_10495 = G__10496;
        continue;
      } else {
      }
      break;
    }
    return a;
  }
};
cljs.core.int_array.cljs$lang$maxFixedArity = 2;
cljs.core.long_array = function cljs$core$long_array(var_args) {
  var args10497 = [];
  var len__8373__auto___10500 = arguments.length;
  var i__8374__auto___10501 = 0;
  while (true) {
    if (i__8374__auto___10501 < len__8373__auto___10500) {
      args10497.push(arguments[i__8374__auto___10501]);
      var G__10502 = i__8374__auto___10501 + 1;
      i__8374__auto___10501 = G__10502;
      continue;
    } else {
    }
    break;
  }
  var G__10499 = args10497.length;
  switch(G__10499) {
    case 1:
      return cljs.core.long_array.cljs$core$IFn$_invoke$arity$1(arguments[0]);
      break;
    case 2:
      return cljs.core.long_array.cljs$core$IFn$_invoke$arity$2(arguments[0], arguments[1]);
      break;
    default:
      throw new Error([cljs.core.str.cljs$core$IFn$_invoke$arity$1("Invalid arity: "), cljs.core.str.cljs$core$IFn$_invoke$arity$1(args10497.length)].join(""));
  }
};
cljs.core.long_array.cljs$core$IFn$_invoke$arity$1 = function(size_or_seq) {
  if (typeof size_or_seq === "number") {
    return cljs.core.long_array.cljs$core$IFn$_invoke$arity$2(size_or_seq, null);
  } else {
    return cljs.core.into_array.cljs$core$IFn$_invoke$arity$1(size_or_seq);
  }
};
cljs.core.long_array.cljs$core$IFn$_invoke$arity$2 = function(size, init_val_or_seq) {
  var a = new Array(size);
  if (cljs.core.seq_QMARK_(init_val_or_seq)) {
    var s = cljs.core.seq(init_val_or_seq);
    var i = 0;
    var s__$1 = s;
    while (true) {
      if (s__$1 && i < size) {
        a[i] = cljs.core.first(s__$1);
        var G__10504 = i + 1;
        var G__10505 = cljs.core.next(s__$1);
        i = G__10504;
        s__$1 = G__10505;
        continue;
      } else {
        return a;
      }
      break;
    }
  } else {
    var n__8183__auto___10506 = size;
    var i_10507 = 0;
    while (true) {
      if (i_10507 < n__8183__auto___10506) {
        a[i_10507] = init_val_or_seq;
        var G__10508 = i_10507 + 1;
        i_10507 = G__10508;
        continue;
      } else {
      }
      break;
    }
    return a;
  }
};
cljs.core.long_array.cljs$lang$maxFixedArity = 2;
cljs.core.double_array = function cljs$core$double_array(var_args) {
  var args10509 = [];
  var len__8373__auto___10512 = arguments.length;
  var i__8374__auto___10513 = 0;
  while (true) {
    if (i__8374__auto___10513 < len__8373__auto___10512) {
      args10509.push(arguments[i__8374__auto___10513]);
      var G__10514 = i__8374__auto___10513 + 1;
      i__8374__auto___10513 = G__10514;
      continue;
    } else {
    }
    break;
  }
  var G__10511 = args10509.length;
  switch(G__10511) {
    case 1:
      return cljs.core.double_array.cljs$core$IFn$_invoke$arity$1(arguments[0]);
      break;
    case 2:
      return cljs.core.double_array.cljs$core$IFn$_invoke$arity$2(arguments[0], arguments[1]);
      break;
    default:
      throw new Error([cljs.core.str.cljs$core$IFn$_invoke$arity$1("Invalid arity: "), cljs.core.str.cljs$core$IFn$_invoke$arity$1(args10509.length)].join(""));
  }
};
cljs.core.double_array.cljs$core$IFn$_invoke$arity$1 = function(size_or_seq) {
  if (typeof size_or_seq === "number") {
    return cljs.core.double_array.cljs$core$IFn$_invoke$arity$2(size_or_seq, null);
  } else {
    return cljs.core.into_array.cljs$core$IFn$_invoke$arity$1(size_or_seq);
  }
};
cljs.core.double_array.cljs$core$IFn$_invoke$arity$2 = function(size, init_val_or_seq) {
  var a = new Array(size);
  if (cljs.core.seq_QMARK_(init_val_or_seq)) {
    var s = cljs.core.seq(init_val_or_seq);
    var i = 0;
    var s__$1 = s;
    while (true) {
      if (s__$1 && i < size) {
        a[i] = cljs.core.first(s__$1);
        var G__10516 = i + 1;
        var G__10517 = cljs.core.next(s__$1);
        i = G__10516;
        s__$1 = G__10517;
        continue;
      } else {
        return a;
      }
      break;
    }
  } else {
    var n__8183__auto___10518 = size;
    var i_10519 = 0;
    while (true) {
      if (i_10519 < n__8183__auto___10518) {
        a[i_10519] = init_val_or_seq;
        var G__10520 = i_10519 + 1;
        i_10519 = G__10520;
        continue;
      } else {
      }
      break;
    }
    return a;
  }
};
cljs.core.double_array.cljs$lang$maxFixedArity = 2;
cljs.core.object_array = function cljs$core$object_array(var_args) {
  var args10521 = [];
  var len__8373__auto___10524 = arguments.length;
  var i__8374__auto___10525 = 0;
  while (true) {
    if (i__8374__auto___10525 < len__8373__auto___10524) {
      args10521.push(arguments[i__8374__auto___10525]);
      var G__10526 = i__8374__auto___10525 + 1;
      i__8374__auto___10525 = G__10526;
      continue;
    } else {
    }
    break;
  }
  var G__10523 = args10521.length;
  switch(G__10523) {
    case 1:
      return cljs.core.object_array.cljs$core$IFn$_invoke$arity$1(arguments[0]);
      break;
    case 2:
      return cljs.core.object_array.cljs$core$IFn$_invoke$arity$2(arguments[0], arguments[1]);
      break;
    default:
      throw new Error([cljs.core.str.cljs$core$IFn$_invoke$arity$1("Invalid arity: "), cljs.core.str.cljs$core$IFn$_invoke$arity$1(args10521.length)].join(""));
  }
};
cljs.core.object_array.cljs$core$IFn$_invoke$arity$1 = function(size_or_seq) {
  if (typeof size_or_seq === "number") {
    return cljs.core.object_array.cljs$core$IFn$_invoke$arity$2(size_or_seq, null);
  } else {
    return cljs.core.into_array.cljs$core$IFn$_invoke$arity$1(size_or_seq);
  }
};
cljs.core.object_array.cljs$core$IFn$_invoke$arity$2 = function(size, init_val_or_seq) {
  var a = new Array(size);
  if (cljs.core.seq_QMARK_(init_val_or_seq)) {
    var s = cljs.core.seq(init_val_or_seq);
    var i = 0;
    var s__$1 = s;
    while (true) {
      if (s__$1 && i < size) {
        a[i] = cljs.core.first(s__$1);
        var G__10528 = i + 1;
        var G__10529 = cljs.core.next(s__$1);
        i = G__10528;
        s__$1 = G__10529;
        continue;
      } else {
        return a;
      }
      break;
    }
  } else {
    var n__8183__auto___10530 = size;
    var i_10531 = 0;
    while (true) {
      if (i_10531 < n__8183__auto___10530) {
        a[i_10531] = init_val_or_seq;
        var G__10532 = i_10531 + 1;
        i_10531 = G__10532;
        continue;
      } else {
      }
      break;
    }
    return a;
  }
};
cljs.core.object_array.cljs$lang$maxFixedArity = 2;
cljs.core.bounded_count = function cljs$core$bounded_count(n, coll) {
  if (cljs.core.counted_QMARK_(coll)) {
    return cljs.core.count(coll);
  } else {
    var i = 0;
    var s = cljs.core.seq(coll);
    while (true) {
      if (!(s == null) && i < n) {
        var G__10533 = i + 1;
        var G__10534 = cljs.core.next(s);
        i = G__10533;
        s = G__10534;
        continue;
      } else {
        return i;
      }
      break;
    }
  }
};
cljs.core.spread = function cljs$core$spread(arglist) {
  if (arglist == null) {
    return null;
  } else {
    if (cljs.core.next(arglist) == null) {
      return cljs.core.seq(cljs.core.first(arglist));
    } else {
      return cljs.core.cons(cljs.core.first(arglist), function() {
        var G__10536 = cljs.core.next(arglist);
        return cljs.core.spread.cljs$core$IFn$_invoke$arity$1 ? cljs.core.spread.cljs$core$IFn$_invoke$arity$1(G__10536) : cljs.core.spread.call(null, G__10536);
      }());
    }
  }
};
cljs.core.concat = function cljs$core$concat(var_args) {
  var args10537 = [];
  var len__8373__auto___10549 = arguments.length;
  var i__8374__auto___10550 = 0;
  while (true) {
    if (i__8374__auto___10550 < len__8373__auto___10549) {
      args10537.push(arguments[i__8374__auto___10550]);
      var G__10551 = i__8374__auto___10550 + 1;
      i__8374__auto___10550 = G__10551;
      continue;
    } else {
    }
    break;
  }
  var G__10542 = args10537.length;
  switch(G__10542) {
    case 0:
      return cljs.core.concat.cljs$core$IFn$_invoke$arity$0();
      break;
    case 1:
      return cljs.core.concat.cljs$core$IFn$_invoke$arity$1(arguments[0]);
      break;
    case 2:
      return cljs.core.concat.cljs$core$IFn$_invoke$arity$2(arguments[0], arguments[1]);
      break;
    default:
      var argseq__8392__auto__ = new cljs.core.IndexedSeq(args10537.slice(2), 0, null);
      return cljs.core.concat.cljs$core$IFn$_invoke$arity$variadic(arguments[0], arguments[1], argseq__8392__auto__);
  }
};
cljs.core.concat.cljs$core$IFn$_invoke$arity$0 = function() {
  return new cljs.core.LazySeq(null, function() {
    return null;
  }, null, null);
};
cljs.core.concat.cljs$core$IFn$_invoke$arity$1 = function(x) {
  return new cljs.core.LazySeq(null, function() {
    return x;
  }, null, null);
};
cljs.core.concat.cljs$core$IFn$_invoke$arity$2 = function(x, y) {
  return new cljs.core.LazySeq(null, function() {
    var s = cljs.core.seq(x);
    if (s) {
      if (cljs.core.chunked_seq_QMARK_(s)) {
        return cljs.core.chunk_cons(cljs.core.chunk_first(s), cljs.core.concat.cljs$core$IFn$_invoke$arity$2(cljs.core.chunk_rest(s), y));
      } else {
        return cljs.core.cons(cljs.core.first(s), cljs.core.concat.cljs$core$IFn$_invoke$arity$2(cljs.core.rest(s), y));
      }
    } else {
      return y;
    }
  }, null, null);
};
cljs.core.concat.cljs$core$IFn$_invoke$arity$variadic = function(x, y, zs) {
  var cat = function cljs$core$cat(xys, zs__$1) {
    return new cljs.core.LazySeq(null, function() {
      var xys__$1 = cljs.core.seq(xys);
      if (xys__$1) {
        if (cljs.core.chunked_seq_QMARK_(xys__$1)) {
          return cljs.core.chunk_cons(cljs.core.chunk_first(xys__$1), cljs$core$cat(cljs.core.chunk_rest(xys__$1), zs__$1));
        } else {
          return cljs.core.cons(cljs.core.first(xys__$1), cljs$core$cat(cljs.core.rest(xys__$1), zs__$1));
        }
      } else {
        if (cljs.core.truth_(zs__$1)) {
          return cljs$core$cat(cljs.core.first(zs__$1), cljs.core.next(zs__$1));
        } else {
          return null;
        }
      }
    }, null, null);
  };
  return cat(cljs.core.concat.cljs$core$IFn$_invoke$arity$2(x, y), zs);
};
cljs.core.concat.cljs$lang$applyTo = function(seq10538) {
  var G__10539 = cljs.core.first(seq10538);
  var seq10538__$1 = cljs.core.next(seq10538);
  var G__10540 = cljs.core.first(seq10538__$1);
  var seq10538__$2 = cljs.core.next(seq10538__$1);
  return cljs.core.concat.cljs$core$IFn$_invoke$arity$variadic(G__10539, G__10540, seq10538__$2);
};
cljs.core.concat.cljs$lang$maxFixedArity = 2;
cljs.core.list_STAR_ = function cljs$core$list_STAR_(var_args) {
  var args10553 = [];
  var len__8373__auto___10561 = arguments.length;
  var i__8374__auto___10562 = 0;
  while (true) {
    if (i__8374__auto___10562 < len__8373__auto___10561) {
      args10553.push(arguments[i__8374__auto___10562]);
      var G__10563 = i__8374__auto___10562 + 1;
      i__8374__auto___10562 = G__10563;
      continue;
    } else {
    }
    break;
  }
  var G__10560 = args10553.length;
  switch(G__10560) {
    case 1:
      return cljs.core.list_STAR_.cljs$core$IFn$_invoke$arity$1(arguments[0]);
      break;
    case 2:
      return cljs.core.list_STAR_.cljs$core$IFn$_invoke$arity$2(arguments[0], arguments[1]);
      break;
    case 3:
      return cljs.core.list_STAR_.cljs$core$IFn$_invoke$arity$3(arguments[0], arguments[1], arguments[2]);
      break;
    case 4:
      return cljs.core.list_STAR_.cljs$core$IFn$_invoke$arity$4(arguments[0], arguments[1], arguments[2], arguments[3]);
      break;
    default:
      var argseq__8392__auto__ = new cljs.core.IndexedSeq(args10553.slice(4), 0, null);
      return cljs.core.list_STAR_.cljs$core$IFn$_invoke$arity$variadic(arguments[0], arguments[1], arguments[2], arguments[3], argseq__8392__auto__);
  }
};
cljs.core.list_STAR_.cljs$core$IFn$_invoke$arity$1 = function(args) {
  return cljs.core.seq(args);
};
cljs.core.list_STAR_.cljs$core$IFn$_invoke$arity$2 = function(a, args) {
  return cljs.core.cons(a, args);
};
cljs.core.list_STAR_.cljs$core$IFn$_invoke$arity$3 = function(a, b, args) {
  return cljs.core.cons(a, cljs.core.cons(b, args));
};
cljs.core.list_STAR_.cljs$core$IFn$_invoke$arity$4 = function(a, b, c, args) {
  return cljs.core.cons(a, cljs.core.cons(b, cljs.core.cons(c, args)));
};
cljs.core.list_STAR_.cljs$core$IFn$_invoke$arity$variadic = function(a, b, c, d, more) {
  return cljs.core.cons(a, cljs.core.cons(b, cljs.core.cons(c, cljs.core.cons(d, cljs.core.spread(more)))));
};
cljs.core.list_STAR_.cljs$lang$applyTo = function(seq10554) {
  var G__10555 = cljs.core.first(seq10554);
  var seq10554__$1 = cljs.core.next(seq10554);
  var G__10556 = cljs.core.first(seq10554__$1);
  var seq10554__$2 = cljs.core.next(seq10554__$1);
  var G__10557 = cljs.core.first(seq10554__$2);
  var seq10554__$3 = cljs.core.next(seq10554__$2);
  var G__10558 = cljs.core.first(seq10554__$3);
  var seq10554__$4 = cljs.core.next(seq10554__$3);
  return cljs.core.list_STAR_.cljs$core$IFn$_invoke$arity$variadic(G__10555, G__10556, G__10557, G__10558, seq10554__$4);
};
cljs.core.list_STAR_.cljs$lang$maxFixedArity = 4;
cljs.core.transient$ = function cljs$core$transient(coll) {
  return cljs.core._as_transient(coll);
};
cljs.core.persistent_BANG_ = function cljs$core$persistent_BANG_(tcoll) {
  return cljs.core._persistent_BANG_(tcoll);
};
cljs.core.conj_BANG_ = function cljs$core$conj_BANG_(var_args) {
  var args10565 = [];
  var len__8373__auto___10571 = arguments.length;
  var i__8374__auto___10572 = 0;
  while (true) {
    if (i__8374__auto___10572 < len__8373__auto___10571) {
      args10565.push(arguments[i__8374__auto___10572]);
      var G__10573 = i__8374__auto___10572 + 1;
      i__8374__auto___10572 = G__10573;
      continue;
    } else {
    }
    break;
  }
  var G__10570 = args10565.length;
  switch(G__10570) {
    case 0:
      return cljs.core.conj_BANG_.cljs$core$IFn$_invoke$arity$0();
      break;
    case 1:
      return cljs.core.conj_BANG_.cljs$core$IFn$_invoke$arity$1(arguments[0]);
      break;
    case 2:
      return cljs.core.conj_BANG_.cljs$core$IFn$_invoke$arity$2(arguments[0], arguments[1]);
      break;
    default:
      var argseq__8392__auto__ = new cljs.core.IndexedSeq(args10565.slice(2), 0, null);
      return cljs.core.conj_BANG_.cljs$core$IFn$_invoke$arity$variadic(arguments[0], arguments[1], argseq__8392__auto__);
  }
};
cljs.core.conj_BANG_.cljs$core$IFn$_invoke$arity$0 = function() {
  return cljs.core.transient$(cljs.core.PersistentVector.EMPTY);
};
cljs.core.conj_BANG_.cljs$core$IFn$_invoke$arity$1 = function(tcoll) {
  return tcoll;
};
cljs.core.conj_BANG_.cljs$core$IFn$_invoke$arity$2 = function(tcoll, val) {
  return cljs.core._conj_BANG_(tcoll, val);
};
cljs.core.conj_BANG_.cljs$core$IFn$_invoke$arity$variadic = function(tcoll, val, vals) {
  while (true) {
    var ntcoll = cljs.core._conj_BANG_(tcoll, val);
    if (cljs.core.truth_(vals)) {
      var G__10575 = ntcoll;
      var G__10576 = cljs.core.first(vals);
      var G__10577 = cljs.core.next(vals);
      tcoll = G__10575;
      val = G__10576;
      vals = G__10577;
      continue;
    } else {
      return ntcoll;
    }
    break;
  }
};
cljs.core.conj_BANG_.cljs$lang$applyTo = function(seq10566) {
  var G__10567 = cljs.core.first(seq10566);
  var seq10566__$1 = cljs.core.next(seq10566);
  var G__10568 = cljs.core.first(seq10566__$1);
  var seq10566__$2 = cljs.core.next(seq10566__$1);
  return cljs.core.conj_BANG_.cljs$core$IFn$_invoke$arity$variadic(G__10567, G__10568, seq10566__$2);
};
cljs.core.conj_BANG_.cljs$lang$maxFixedArity = 2;
cljs.core.assoc_BANG_ = function cljs$core$assoc_BANG_(var_args) {
  var args10578 = [];
  var len__8373__auto___10585 = arguments.length;
  var i__8374__auto___10586 = 0;
  while (true) {
    if (i__8374__auto___10586 < len__8373__auto___10585) {
      args10578.push(arguments[i__8374__auto___10586]);
      var G__10587 = i__8374__auto___10586 + 1;
      i__8374__auto___10586 = G__10587;
      continue;
    } else {
    }
    break;
  }
  var G__10584 = args10578.length;
  switch(G__10584) {
    case 3:
      return cljs.core.assoc_BANG_.cljs$core$IFn$_invoke$arity$3(arguments[0], arguments[1], arguments[2]);
      break;
    default:
      var argseq__8392__auto__ = new cljs.core.IndexedSeq(args10578.slice(3), 0, null);
      return cljs.core.assoc_BANG_.cljs$core$IFn$_invoke$arity$variadic(arguments[0], arguments[1], arguments[2], argseq__8392__auto__);
  }
};
cljs.core.assoc_BANG_.cljs$core$IFn$_invoke$arity$3 = function(tcoll, key, val) {
  return cljs.core._assoc_BANG_(tcoll, key, val);
};
cljs.core.assoc_BANG_.cljs$core$IFn$_invoke$arity$variadic = function(tcoll, key, val, kvs) {
  while (true) {
    var ntcoll = cljs.core._assoc_BANG_(tcoll, key, val);
    if (cljs.core.truth_(kvs)) {
      var G__10589 = ntcoll;
      var G__10590 = cljs.core.first(kvs);
      var G__10591 = cljs.core.second(kvs);
      var G__10592 = cljs.core.nnext(kvs);
      tcoll = G__10589;
      key = G__10590;
      val = G__10591;
      kvs = G__10592;
      continue;
    } else {
      return ntcoll;
    }
    break;
  }
};
cljs.core.assoc_BANG_.cljs$lang$applyTo = function(seq10579) {
  var G__10580 = cljs.core.first(seq10579);
  var seq10579__$1 = cljs.core.next(seq10579);
  var G__10581 = cljs.core.first(seq10579__$1);
  var seq10579__$2 = cljs.core.next(seq10579__$1);
  var G__10582 = cljs.core.first(seq10579__$2);
  var seq10579__$3 = cljs.core.next(seq10579__$2);
  return cljs.core.assoc_BANG_.cljs$core$IFn$_invoke$arity$variadic(G__10580, G__10581, G__10582, seq10579__$3);
};
cljs.core.assoc_BANG_.cljs$lang$maxFixedArity = 3;
cljs.core.dissoc_BANG_ = function cljs$core$dissoc_BANG_(var_args) {
  var args10593 = [];
  var len__8373__auto___10599 = arguments.length;
  var i__8374__auto___10600 = 0;
  while (true) {
    if (i__8374__auto___10600 < len__8373__auto___10599) {
      args10593.push(arguments[i__8374__auto___10600]);
      var G__10601 = i__8374__auto___10600 + 1;
      i__8374__auto___10600 = G__10601;
      continue;
    } else {
    }
    break;
  }
  var G__10598 = args10593.length;
  switch(G__10598) {
    case 2:
      return cljs.core.dissoc_BANG_.cljs$core$IFn$_invoke$arity$2(arguments[0], arguments[1]);
      break;
    default:
      var argseq__8392__auto__ = new cljs.core.IndexedSeq(args10593.slice(2), 0, null);
      return cljs.core.dissoc_BANG_.cljs$core$IFn$_invoke$arity$variadic(arguments[0], arguments[1], argseq__8392__auto__);
  }
};
cljs.core.dissoc_BANG_.cljs$core$IFn$_invoke$arity$2 = function(tcoll, key) {
  return cljs.core._dissoc_BANG_(tcoll, key);
};
cljs.core.dissoc_BANG_.cljs$core$IFn$_invoke$arity$variadic = function(tcoll, key, ks) {
  while (true) {
    var ntcoll = cljs.core._dissoc_BANG_(tcoll, key);
    if (cljs.core.truth_(ks)) {
      var G__10603 = ntcoll;
      var G__10604 = cljs.core.first(ks);
      var G__10605 = cljs.core.next(ks);
      tcoll = G__10603;
      key = G__10604;
      ks = G__10605;
      continue;
    } else {
      return ntcoll;
    }
    break;
  }
};
cljs.core.dissoc_BANG_.cljs$lang$applyTo = function(seq10594) {
  var G__10595 = cljs.core.first(seq10594);
  var seq10594__$1 = cljs.core.next(seq10594);
  var G__10596 = cljs.core.first(seq10594__$1);
  var seq10594__$2 = cljs.core.next(seq10594__$1);
  return cljs.core.dissoc_BANG_.cljs$core$IFn$_invoke$arity$variadic(G__10595, G__10596, seq10594__$2);
};
cljs.core.dissoc_BANG_.cljs$lang$maxFixedArity = 2;
cljs.core.pop_BANG_ = function cljs$core$pop_BANG_(tcoll) {
  return cljs.core._pop_BANG_(tcoll);
};
cljs.core.disj_BANG_ = function cljs$core$disj_BANG_(var_args) {
  var args10606 = [];
  var len__8373__auto___10612 = arguments.length;
  var i__8374__auto___10613 = 0;
  while (true) {
    if (i__8374__auto___10613 < len__8373__auto___10612) {
      args10606.push(arguments[i__8374__auto___10613]);
      var G__10614 = i__8374__auto___10613 + 1;
      i__8374__auto___10613 = G__10614;
      continue;
    } else {
    }
    break;
  }
  var G__10611 = args10606.length;
  switch(G__10611) {
    case 2:
      return cljs.core.disj_BANG_.cljs$core$IFn$_invoke$arity$2(arguments[0], arguments[1]);
      break;
    default:
      var argseq__8392__auto__ = new cljs.core.IndexedSeq(args10606.slice(2), 0, null);
      return cljs.core.disj_BANG_.cljs$core$IFn$_invoke$arity$variadic(arguments[0], arguments[1], argseq__8392__auto__);
  }
};
cljs.core.disj_BANG_.cljs$core$IFn$_invoke$arity$2 = function(tcoll, val) {
  return cljs.core._disjoin_BANG_(tcoll, val);
};
cljs.core.disj_BANG_.cljs$core$IFn$_invoke$arity$variadic = function(tcoll, val, vals) {
  while (true) {
    var ntcoll = cljs.core._disjoin_BANG_(tcoll, val);
    if (cljs.core.truth_(vals)) {
      var G__10616 = ntcoll;
      var G__10617 = cljs.core.first(vals);
      var G__10618 = cljs.core.next(vals);
      tcoll = G__10616;
      val = G__10617;
      vals = G__10618;
      continue;
    } else {
      return ntcoll;
    }
    break;
  }
};
cljs.core.disj_BANG_.cljs$lang$applyTo = function(seq10607) {
  var G__10608 = cljs.core.first(seq10607);
  var seq10607__$1 = cljs.core.next(seq10607);
  var G__10609 = cljs.core.first(seq10607__$1);
  var seq10607__$2 = cljs.core.next(seq10607__$1);
  return cljs.core.disj_BANG_.cljs$core$IFn$_invoke$arity$variadic(G__10608, G__10609, seq10607__$2);
};
cljs.core.disj_BANG_.cljs$lang$maxFixedArity = 2;
cljs.core.apply_to = function cljs$core$apply_to(f, argc, args) {
  var args__$1 = cljs.core.seq(args);
  if (argc === 0) {
    return f.cljs$core$IFn$_invoke$arity$0 ? f.cljs$core$IFn$_invoke$arity$0() : f.call(null);
  } else {
    var a8227 = cljs.core._first(args__$1);
    var args__$2 = cljs.core._rest(args__$1);
    if (argc === 1) {
      if (f.cljs$core$IFn$_invoke$arity$1) {
        return f.cljs$core$IFn$_invoke$arity$1(a8227);
      } else {
        return f.cljs$core$IFn$_invoke$arity$1 ? f.cljs$core$IFn$_invoke$arity$1(a8227) : f.call(null, a8227);
      }
    } else {
      var b8228 = cljs.core._first(args__$2);
      var args__$3 = cljs.core._rest(args__$2);
      if (argc === 2) {
        if (f.cljs$core$IFn$_invoke$arity$2) {
          return f.cljs$core$IFn$_invoke$arity$2(a8227, b8228);
        } else {
          return f.cljs$core$IFn$_invoke$arity$2 ? f.cljs$core$IFn$_invoke$arity$2(a8227, b8228) : f.call(null, a8227, b8228);
        }
      } else {
        var c8229 = cljs.core._first(args__$3);
        var args__$4 = cljs.core._rest(args__$3);
        if (argc === 3) {
          if (f.cljs$core$IFn$_invoke$arity$3) {
            return f.cljs$core$IFn$_invoke$arity$3(a8227, b8228, c8229);
          } else {
            return f.cljs$core$IFn$_invoke$arity$3 ? f.cljs$core$IFn$_invoke$arity$3(a8227, b8228, c8229) : f.call(null, a8227, b8228, c8229);
          }
        } else {
          var d8230 = cljs.core._first(args__$4);
          var args__$5 = cljs.core._rest(args__$4);
          if (argc === 4) {
            if (f.cljs$core$IFn$_invoke$arity$4) {
              return f.cljs$core$IFn$_invoke$arity$4(a8227, b8228, c8229, d8230);
            } else {
              return f.cljs$core$IFn$_invoke$arity$4 ? f.cljs$core$IFn$_invoke$arity$4(a8227, b8228, c8229, d8230) : f.call(null, a8227, b8228, c8229, d8230);
            }
          } else {
            var e8231 = cljs.core._first(args__$5);
            var args__$6 = cljs.core._rest(args__$5);
            if (argc === 5) {
              if (f.cljs$core$IFn$_invoke$arity$5) {
                return f.cljs$core$IFn$_invoke$arity$5(a8227, b8228, c8229, d8230, e8231);
              } else {
                return f.cljs$core$IFn$_invoke$arity$5 ? f.cljs$core$IFn$_invoke$arity$5(a8227, b8228, c8229, d8230, e8231) : f.call(null, a8227, b8228, c8229, d8230, e8231);
              }
            } else {
              var f8232 = cljs.core._first(args__$6);
              var args__$7 = cljs.core._rest(args__$6);
              if (argc === 6) {
                if (f.cljs$core$IFn$_invoke$arity$6) {
                  return f.cljs$core$IFn$_invoke$arity$6(a8227, b8228, c8229, d8230, e8231, f8232);
                } else {
                  return f.cljs$core$IFn$_invoke$arity$6 ? f.cljs$core$IFn$_invoke$arity$6(a8227, b8228, c8229, d8230, e8231, f8232) : f.call(null, a8227, b8228, c8229, d8230, e8231, f8232);
                }
              } else {
                var g8233 = cljs.core._first(args__$7);
                var args__$8 = cljs.core._rest(args__$7);
                if (argc === 7) {
                  if (f.cljs$core$IFn$_invoke$arity$7) {
                    return f.cljs$core$IFn$_invoke$arity$7(a8227, b8228, c8229, d8230, e8231, f8232, g8233);
                  } else {
                    return f.cljs$core$IFn$_invoke$arity$7 ? f.cljs$core$IFn$_invoke$arity$7(a8227, b8228, c8229, d8230, e8231, f8232, g8233) : f.call(null, a8227, b8228, c8229, d8230, e8231, f8232, g8233);
                  }
                } else {
                  var h8234 = cljs.core._first(args__$8);
                  var args__$9 = cljs.core._rest(args__$8);
                  if (argc === 8) {
                    if (f.cljs$core$IFn$_invoke$arity$8) {
                      return f.cljs$core$IFn$_invoke$arity$8(a8227, b8228, c8229, d8230, e8231, f8232, g8233, h8234);
                    } else {
                      return f.cljs$core$IFn$_invoke$arity$8 ? f.cljs$core$IFn$_invoke$arity$8(a8227, b8228, c8229, d8230, e8231, f8232, g8233, h8234) : f.call(null, a8227, b8228, c8229, d8230, e8231, f8232, g8233, h8234);
                    }
                  } else {
                    var i8235 = cljs.core._first(args__$9);
                    var args__$10 = cljs.core._rest(args__$9);
                    if (argc === 9) {
                      if (f.cljs$core$IFn$_invoke$arity$9) {
                        return f.cljs$core$IFn$_invoke$arity$9(a8227, b8228, c8229, d8230, e8231, f8232, g8233, h8234, i8235);
                      } else {
                        return f.cljs$core$IFn$_invoke$arity$9 ? f.cljs$core$IFn$_invoke$arity$9(a8227, b8228, c8229, d8230, e8231, f8232, g8233, h8234, i8235) : f.call(null, a8227, b8228, c8229, d8230, e8231, f8232, g8233, h8234, i8235);
                      }
                    } else {
                      var j8236 = cljs.core._first(args__$10);
                      var args__$11 = cljs.core._rest(args__$10);
                      if (argc === 10) {
                        if (f.cljs$core$IFn$_invoke$arity$10) {
                          return f.cljs$core$IFn$_invoke$arity$10(a8227, b8228, c8229, d8230, e8231, f8232, g8233, h8234, i8235, j8236);
                        } else {
                          return f.cljs$core$IFn$_invoke$arity$10 ? f.cljs$core$IFn$_invoke$arity$10(a8227, b8228, c8229, d8230, e8231, f8232, g8233, h8234, i8235, j8236) : f.call(null, a8227, b8228, c8229, d8230, e8231, f8232, g8233, h8234, i8235, j8236);
                        }
                      } else {
                        var k8237 = cljs.core._first(args__$11);
                        var args__$12 = cljs.core._rest(args__$11);
                        if (argc === 11) {
                          if (f.cljs$core$IFn$_invoke$arity$11) {
                            return f.cljs$core$IFn$_invoke$arity$11(a8227, b8228, c8229, d8230, e8231, f8232, g8233, h8234, i8235, j8236, k8237);
                          } else {
                            return f.cljs$core$IFn$_invoke$arity$11 ? f.cljs$core$IFn$_invoke$arity$11(a8227, b8228, c8229, d8230, e8231, f8232, g8233, h8234, i8235, j8236, k8237) : f.call(null, a8227, b8228, c8229, d8230, e8231, f8232, g8233, h8234, i8235, j8236, k8237);
                          }
                        } else {
                          var l8238 = cljs.core._first(args__$12);
                          var args__$13 = cljs.core._rest(args__$12);
                          if (argc === 12) {
                            if (f.cljs$core$IFn$_invoke$arity$12) {
                              return f.cljs$core$IFn$_invoke$arity$12(a8227, b8228, c8229, d8230, e8231, f8232, g8233, h8234, i8235, j8236, k8237, l8238);
                            } else {
                              return f.cljs$core$IFn$_invoke$arity$12 ? f.cljs$core$IFn$_invoke$arity$12(a8227, b8228, c8229, d8230, e8231, f8232, g8233, h8234, i8235, j8236, k8237, l8238) : f.call(null, a8227, b8228, c8229, d8230, e8231, f8232, g8233, h8234, i8235, j8236, k8237, l8238);
                            }
                          } else {
                            var m8239 = cljs.core._first(args__$13);
                            var args__$14 = cljs.core._rest(args__$13);
                            if (argc === 13) {
                              if (f.cljs$core$IFn$_invoke$arity$13) {
                                return f.cljs$core$IFn$_invoke$arity$13(a8227, b8228, c8229, d8230, e8231, f8232, g8233, h8234, i8235, j8236, k8237, l8238, m8239);
                              } else {
                                return f.cljs$core$IFn$_invoke$arity$13 ? f.cljs$core$IFn$_invoke$arity$13(a8227, b8228, c8229, d8230, e8231, f8232, g8233, h8234, i8235, j8236, k8237, l8238, m8239) : f.call(null, a8227, b8228, c8229, d8230, e8231, f8232, g8233, h8234, i8235, j8236, k8237, l8238, m8239);
                              }
                            } else {
                              var n8240 = cljs.core._first(args__$14);
                              var args__$15 = cljs.core._rest(args__$14);
                              if (argc === 14) {
                                if (f.cljs$core$IFn$_invoke$arity$14) {
                                  return f.cljs$core$IFn$_invoke$arity$14(a8227, b8228, c8229, d8230, e8231, f8232, g8233, h8234, i8235, j8236, k8237, l8238, m8239, n8240);
                                } else {
                                  return f.cljs$core$IFn$_invoke$arity$14 ? f.cljs$core$IFn$_invoke$arity$14(a8227, b8228, c8229, d8230, e8231, f8232, g8233, h8234, i8235, j8236, k8237, l8238, m8239, n8240) : f.call(null, a8227, b8228, c8229, d8230, e8231, f8232, g8233, h8234, i8235, j8236, k8237, l8238, m8239, n8240);
                                }
                              } else {
                                var o8241 = cljs.core._first(args__$15);
                                var args__$16 = cljs.core._rest(args__$15);
                                if (argc === 15) {
                                  if (f.cljs$core$IFn$_invoke$arity$15) {
                                    return f.cljs$core$IFn$_invoke$arity$15(a8227, b8228, c8229, d8230, e8231, f8232, g8233, h8234, i8235, j8236, k8237, l8238, m8239, n8240, o8241);
                                  } else {
                                    return f.cljs$core$IFn$_invoke$arity$15 ? f.cljs$core$IFn$_invoke$arity$15(a8227, b8228, c8229, d8230, e8231, f8232, g8233, h8234, i8235, j8236, k8237, l8238, m8239, n8240, o8241) : f.call(null, a8227, b8228, c8229, d8230, e8231, f8232, g8233, h8234, i8235, j8236, k8237, l8238, m8239, n8240, o8241);
                                  }
                                } else {
                                  var p8242 = cljs.core._first(args__$16);
                                  var args__$17 = cljs.core._rest(args__$16);
                                  if (argc === 16) {
                                    if (f.cljs$core$IFn$_invoke$arity$16) {
                                      return f.cljs$core$IFn$_invoke$arity$16(a8227, b8228, c8229, d8230, e8231, f8232, g8233, h8234, i8235, j8236, k8237, l8238, m8239, n8240, o8241, p8242);
                                    } else {
                                      return f.cljs$core$IFn$_invoke$arity$16 ? f.cljs$core$IFn$_invoke$arity$16(a8227, b8228, c8229, d8230, e8231, f8232, g8233, h8234, i8235, j8236, k8237, l8238, m8239, n8240, o8241, p8242) : f.call(null, a8227, b8228, c8229, d8230, e8231, f8232, g8233, h8234, i8235, j8236, k8237, l8238, m8239, n8240, o8241, p8242);
                                    }
                                  } else {
                                    var q8243 = cljs.core._first(args__$17);
                                    var args__$18 = cljs.core._rest(args__$17);
                                    if (argc === 17) {
                                      if (f.cljs$core$IFn$_invoke$arity$17) {
                                        return f.cljs$core$IFn$_invoke$arity$17(a8227, b8228, c8229, d8230, e8231, f8232, g8233, h8234, i8235, j8236, k8237, l8238, m8239, n8240, o8241, p8242, q8243);
                                      } else {
                                        return f.cljs$core$IFn$_invoke$arity$17 ? f.cljs$core$IFn$_invoke$arity$17(a8227, b8228, c8229, d8230, e8231, f8232, g8233, h8234, i8235, j8236, k8237, l8238, m8239, n8240, o8241, p8242, q8243) : f.call(null, a8227, b8228, c8229, d8230, e8231, f8232, g8233, h8234, i8235, j8236, k8237, l8238, m8239, n8240, o8241, p8242, q8243);
                                      }
                                    } else {
                                      var r8244 = cljs.core._first(args__$18);
                                      var args__$19 = cljs.core._rest(args__$18);
                                      if (argc === 18) {
                                        if (f.cljs$core$IFn$_invoke$arity$18) {
                                          return f.cljs$core$IFn$_invoke$arity$18(a8227, b8228, c8229, d8230, e8231, f8232, g8233, h8234, i8235, j8236, k8237, l8238, m8239, n8240, o8241, p8242, q8243, r8244);
                                        } else {
                                          return f.cljs$core$IFn$_invoke$arity$18 ? f.cljs$core$IFn$_invoke$arity$18(a8227, b8228, c8229, d8230, e8231, f8232, g8233, h8234, i8235, j8236, k8237, l8238, m8239, n8240, o8241, p8242, q8243, r8244) : f.call(null, a8227, b8228, c8229, d8230, e8231, f8232, g8233, h8234, i8235, j8236, k8237, l8238, m8239, n8240, o8241, p8242, q8243, r8244);
                                        }
                                      } else {
                                        var s8245 = cljs.core._first(args__$19);
                                        var args__$20 = cljs.core._rest(args__$19);
                                        if (argc === 19) {
                                          if (f.cljs$core$IFn$_invoke$arity$19) {
                                            return f.cljs$core$IFn$_invoke$arity$19(a8227, b8228, c8229, d8230, e8231, f8232, g8233, h8234, i8235, j8236, k8237, l8238, m8239, n8240, o8241, p8242, q8243, r8244, s8245);
                                          } else {
                                            return f.cljs$core$IFn$_invoke$arity$19 ? f.cljs$core$IFn$_invoke$arity$19(a8227, b8228, c8229, d8230, e8231, f8232, g8233, h8234, i8235, j8236, k8237, l8238, m8239, n8240, o8241, p8242, q8243, r8244, s8245) : f.call(null, a8227, b8228, c8229, d8230, e8231, f8232, g8233, h8234, i8235, j8236, k8237, l8238, m8239, n8240, o8241, p8242, q8243, r8244, s8245);
                                          }
                                        } else {
                                          var t8246 = cljs.core._first(args__$20);
                                          var args__$21 = cljs.core._rest(args__$20);
                                          if (argc === 20) {
                                            if (f.cljs$core$IFn$_invoke$arity$20) {
                                              return f.cljs$core$IFn$_invoke$arity$20(a8227, b8228, c8229, d8230, e8231, f8232, g8233, h8234, i8235, j8236, k8237, l8238, m8239, n8240, o8241, p8242, q8243, r8244, s8245, t8246);
                                            } else {
                                              return f.cljs$core$IFn$_invoke$arity$20 ? f.cljs$core$IFn$_invoke$arity$20(a8227, b8228, c8229, d8230, e8231, f8232, g8233, h8234, i8235, j8236, k8237, l8238, m8239, n8240, o8241, p8242, q8243, r8244, s8245, t8246) : f.call(null, a8227, b8228, c8229, d8230, e8231, f8232, g8233, h8234, i8235, j8236, k8237, l8238, m8239, n8240, o8241, p8242, q8243, r8244, s8245, t8246);
                                            }
                                          } else {
                                            throw new Error("Only up to 20 arguments supported on functions");
                                          }
                                        }
                                      }
                                    }
                                  }
                                }
                              }
                            }
                          }
                        }
                      }
                    }
                  }
                }
              }
            }
          }
        }
      }
    }
  }
};
cljs.core.apply = function cljs$core$apply(var_args) {
  var args10619 = [];
  var len__8373__auto___10628 = arguments.length;
  var i__8374__auto___10629 = 0;
  while (true) {
    if (i__8374__auto___10629 < len__8373__auto___10628) {
      args10619.push(arguments[i__8374__auto___10629]);
      var G__10630 = i__8374__auto___10629 + 1;
      i__8374__auto___10629 = G__10630;
      continue;
    } else {
    }
    break;
  }
  var G__10627 = args10619.length;
  switch(G__10627) {
    case 2:
      return cljs.core.apply.cljs$core$IFn$_invoke$arity$2(arguments[0], arguments[1]);
      break;
    case 3:
      return cljs.core.apply.cljs$core$IFn$_invoke$arity$3(arguments[0], arguments[1], arguments[2]);
      break;
    case 4:
      return cljs.core.apply.cljs$core$IFn$_invoke$arity$4(arguments[0], arguments[1], arguments[2], arguments[3]);
      break;
    case 5:
      return cljs.core.apply.cljs$core$IFn$_invoke$arity$5(arguments[0], arguments[1], arguments[2], arguments[3], arguments[4]);
      break;
    default:
      var argseq__8392__auto__ = new cljs.core.IndexedSeq(args10619.slice(5), 0, null);
      return cljs.core.apply.cljs$core$IFn$_invoke$arity$variadic(arguments[0], arguments[1], arguments[2], arguments[3], arguments[4], argseq__8392__auto__);
  }
};
cljs.core.apply.cljs$core$IFn$_invoke$arity$2 = function(f, args) {
  var fixed_arity = f.cljs$lang$maxFixedArity;
  if (f.cljs$lang$applyTo) {
    var bc = cljs.core.bounded_count(fixed_arity + 1, args);
    if (bc <= fixed_arity) {
      return cljs.core.apply_to(f, bc, args);
    } else {
      return f.cljs$lang$applyTo(args);
    }
  } else {
    return f.apply(f, cljs.core.to_array(args));
  }
};
cljs.core.apply.cljs$core$IFn$_invoke$arity$3 = function(f, x, args) {
  var arglist = cljs.core.list_STAR_.cljs$core$IFn$_invoke$arity$2(x, args);
  var fixed_arity = f.cljs$lang$maxFixedArity;
  if (f.cljs$lang$applyTo) {
    var bc = cljs.core.bounded_count(fixed_arity + 1, arglist);
    if (bc <= fixed_arity) {
      return cljs.core.apply_to(f, bc, arglist);
    } else {
      return f.cljs$lang$applyTo(arglist);
    }
  } else {
    return f.apply(f, cljs.core.to_array(arglist));
  }
};
cljs.core.apply.cljs$core$IFn$_invoke$arity$4 = function(f, x, y, args) {
  var arglist = cljs.core.list_STAR_.cljs$core$IFn$_invoke$arity$3(x, y, args);
  var fixed_arity = f.cljs$lang$maxFixedArity;
  if (f.cljs$lang$applyTo) {
    var bc = cljs.core.bounded_count(fixed_arity + 1, arglist);
    if (bc <= fixed_arity) {
      return cljs.core.apply_to(f, bc, arglist);
    } else {
      return f.cljs$lang$applyTo(arglist);
    }
  } else {
    return f.apply(f, cljs.core.to_array(arglist));
  }
};
cljs.core.apply.cljs$core$IFn$_invoke$arity$5 = function(f, x, y, z, args) {
  var arglist = cljs.core.list_STAR_.cljs$core$IFn$_invoke$arity$4(x, y, z, args);
  var fixed_arity = f.cljs$lang$maxFixedArity;
  if (f.cljs$lang$applyTo) {
    var bc = cljs.core.bounded_count(fixed_arity + 1, arglist);
    if (bc <= fixed_arity) {
      return cljs.core.apply_to(f, bc, arglist);
    } else {
      return f.cljs$lang$applyTo(arglist);
    }
  } else {
    return f.apply(f, cljs.core.to_array(arglist));
  }
};
cljs.core.apply.cljs$core$IFn$_invoke$arity$variadic = function(f, a, b, c, d, args) {
  var arglist = cljs.core.cons(a, cljs.core.cons(b, cljs.core.cons(c, cljs.core.cons(d, cljs.core.spread(args)))));
  var fixed_arity = f.cljs$lang$maxFixedArity;
  if (f.cljs$lang$applyTo) {
    var bc = cljs.core.bounded_count(fixed_arity + 1, arglist);
    if (bc <= fixed_arity) {
      return cljs.core.apply_to(f, bc, arglist);
    } else {
      return f.cljs$lang$applyTo(arglist);
    }
  } else {
    return f.apply(f, cljs.core.to_array(arglist));
  }
};
cljs.core.apply.cljs$lang$applyTo = function(seq10620) {
  var G__10621 = cljs.core.first(seq10620);
  var seq10620__$1 = cljs.core.next(seq10620);
  var G__10622 = cljs.core.first(seq10620__$1);
  var seq10620__$2 = cljs.core.next(seq10620__$1);
  var G__10623 = cljs.core.first(seq10620__$2);
  var seq10620__$3 = cljs.core.next(seq10620__$2);
  var G__10624 = cljs.core.first(seq10620__$3);
  var seq10620__$4 = cljs.core.next(seq10620__$3);
  var G__10625 = cljs.core.first(seq10620__$4);
  var seq10620__$5 = cljs.core.next(seq10620__$4);
  return cljs.core.apply.cljs$core$IFn$_invoke$arity$variadic(G__10621, G__10622, G__10623, G__10624, G__10625, seq10620__$5);
};
cljs.core.apply.cljs$lang$maxFixedArity = 5;
cljs.core.vary_meta = function cljs$core$vary_meta(var_args) {
  var args10632 = [];
  var len__8373__auto___10657 = arguments.length;
  var i__8374__auto___10658 = 0;
  while (true) {
    if (i__8374__auto___10658 < len__8373__auto___10657) {
      args10632.push(arguments[i__8374__auto___10658]);
      var G__10659 = i__8374__auto___10658 + 1;
      i__8374__auto___10658 = G__10659;
      continue;
    } else {
    }
    break;
  }
  var G__10641 = args10632.length;
  switch(G__10641) {
    case 2:
      return cljs.core.vary_meta.cljs$core$IFn$_invoke$arity$2(arguments[0], arguments[1]);
      break;
    case 3:
      return cljs.core.vary_meta.cljs$core$IFn$_invoke$arity$3(arguments[0], arguments[1], arguments[2]);
      break;
    case 4:
      return cljs.core.vary_meta.cljs$core$IFn$_invoke$arity$4(arguments[0], arguments[1], arguments[2], arguments[3]);
      break;
    case 5:
      return cljs.core.vary_meta.cljs$core$IFn$_invoke$arity$5(arguments[0], arguments[1], arguments[2], arguments[3], arguments[4]);
      break;
    case 6:
      return cljs.core.vary_meta.cljs$core$IFn$_invoke$arity$6(arguments[0], arguments[1], arguments[2], arguments[3], arguments[4], arguments[5]);
      break;
    default:
      var argseq__8392__auto__ = new cljs.core.IndexedSeq(args10632.slice(6), 0, null);
      return cljs.core.vary_meta.cljs$core$IFn$_invoke$arity$variadic(arguments[0], arguments[1], arguments[2], arguments[3], arguments[4], arguments[5], argseq__8392__auto__);
  }
};
cljs.core.vary_meta.cljs$core$IFn$_invoke$arity$2 = function(obj, f) {
  return cljs.core.with_meta(obj, function() {
    var G__10642 = cljs.core.meta(obj);
    return f.cljs$core$IFn$_invoke$arity$1 ? f.cljs$core$IFn$_invoke$arity$1(G__10642) : f.call(null, G__10642);
  }());
};
cljs.core.vary_meta.cljs$core$IFn$_invoke$arity$3 = function(obj, f, a) {
  return cljs.core.with_meta(obj, function() {
    var G__10643 = cljs.core.meta(obj);
    var G__10644 = a;
    return f.cljs$core$IFn$_invoke$arity$2 ? f.cljs$core$IFn$_invoke$arity$2(G__10643, G__10644) : f.call(null, G__10643, G__10644);
  }());
};
cljs.core.vary_meta.cljs$core$IFn$_invoke$arity$4 = function(obj, f, a, b) {
  return cljs.core.with_meta(obj, function() {
    var G__10645 = cljs.core.meta(obj);
    var G__10646 = a;
    var G__10647 = b;
    return f.cljs$core$IFn$_invoke$arity$3 ? f.cljs$core$IFn$_invoke$arity$3(G__10645, G__10646, G__10647) : f.call(null, G__10645, G__10646, G__10647);
  }());
};
cljs.core.vary_meta.cljs$core$IFn$_invoke$arity$5 = function(obj, f, a, b, c) {
  return cljs.core.with_meta(obj, function() {
    var G__10648 = cljs.core.meta(obj);
    var G__10649 = a;
    var G__10650 = b;
    var G__10651 = c;
    return f.cljs$core$IFn$_invoke$arity$4 ? f.cljs$core$IFn$_invoke$arity$4(G__10648, G__10649, G__10650, G__10651) : f.call(null, G__10648, G__10649, G__10650, G__10651);
  }());
};
cljs.core.vary_meta.cljs$core$IFn$_invoke$arity$6 = function(obj, f, a, b, c, d) {
  return cljs.core.with_meta(obj, function() {
    var G__10652 = cljs.core.meta(obj);
    var G__10653 = a;
    var G__10654 = b;
    var G__10655 = c;
    var G__10656 = d;
    return f.cljs$core$IFn$_invoke$arity$5 ? f.cljs$core$IFn$_invoke$arity$5(G__10652, G__10653, G__10654, G__10655, G__10656) : f.call(null, G__10652, G__10653, G__10654, G__10655, G__10656);
  }());
};
cljs.core.vary_meta.cljs$core$IFn$_invoke$arity$variadic = function(obj, f, a, b, c, d, args) {
  return cljs.core.with_meta(obj, cljs.core.apply.cljs$core$IFn$_invoke$arity$variadic(f, cljs.core.meta(obj), a, b, c, cljs.core.array_seq([d, args], 0)));
};
cljs.core.vary_meta.cljs$lang$applyTo = function(seq10633) {
  var G__10634 = cljs.core.first(seq10633);
  var seq10633__$1 = cljs.core.next(seq10633);
  var G__10635 = cljs.core.first(seq10633__$1);
  var seq10633__$2 = cljs.core.next(seq10633__$1);
  var G__10636 = cljs.core.first(seq10633__$2);
  var seq10633__$3 = cljs.core.next(seq10633__$2);
  var G__10637 = cljs.core.first(seq10633__$3);
  var seq10633__$4 = cljs.core.next(seq10633__$3);
  var G__10638 = cljs.core.first(seq10633__$4);
  var seq10633__$5 = cljs.core.next(seq10633__$4);
  var G__10639 = cljs.core.first(seq10633__$5);
  var seq10633__$6 = cljs.core.next(seq10633__$5);
  return cljs.core.vary_meta.cljs$core$IFn$_invoke$arity$variadic(G__10634, G__10635, G__10636, G__10637, G__10638, G__10639, seq10633__$6);
};
cljs.core.vary_meta.cljs$lang$maxFixedArity = 6;
cljs.core.not_EQ_ = function cljs$core$not_EQ_(var_args) {
  var args10661 = [];
  var len__8373__auto___10667 = arguments.length;
  var i__8374__auto___10668 = 0;
  while (true) {
    if (i__8374__auto___10668 < len__8373__auto___10667) {
      args10661.push(arguments[i__8374__auto___10668]);
      var G__10669 = i__8374__auto___10668 + 1;
      i__8374__auto___10668 = G__10669;
      continue;
    } else {
    }
    break;
  }
  var G__10666 = args10661.length;
  switch(G__10666) {
    case 1:
      return cljs.core.not_EQ_.cljs$core$IFn$_invoke$arity$1(arguments[0]);
      break;
    case 2:
      return cljs.core.not_EQ_.cljs$core$IFn$_invoke$arity$2(arguments[0], arguments[1]);
      break;
    default:
      var argseq__8392__auto__ = new cljs.core.IndexedSeq(args10661.slice(2), 0, null);
      return cljs.core.not_EQ_.cljs$core$IFn$_invoke$arity$variadic(arguments[0], arguments[1], argseq__8392__auto__);
  }
};
cljs.core.not_EQ_.cljs$core$IFn$_invoke$arity$1 = function(x) {
  return false;
};
cljs.core.not_EQ_.cljs$core$IFn$_invoke$arity$2 = function(x, y) {
  return !cljs.core._EQ_.cljs$core$IFn$_invoke$arity$2(x, y);
};
cljs.core.not_EQ_.cljs$core$IFn$_invoke$arity$variadic = function(x, y, more) {
  return cljs.core.not(cljs.core.apply.cljs$core$IFn$_invoke$arity$4(cljs.core._EQ_, x, y, more));
};
cljs.core.not_EQ_.cljs$lang$applyTo = function(seq10662) {
  var G__10663 = cljs.core.first(seq10662);
  var seq10662__$1 = cljs.core.next(seq10662);
  var G__10664 = cljs.core.first(seq10662__$1);
  var seq10662__$2 = cljs.core.next(seq10662__$1);
  return cljs.core.not_EQ_.cljs$core$IFn$_invoke$arity$variadic(G__10663, G__10664, seq10662__$2);
};
cljs.core.not_EQ_.cljs$lang$maxFixedArity = 2;
cljs.core.not_empty = function cljs$core$not_empty(coll) {
  if (cljs.core.seq(coll)) {
    return coll;
  } else {
    return null;
  }
};
cljs.core.nil_iter = function cljs$core$nil_iter() {
  if (typeof cljs.core.t_cljs$core10674 !== "undefined") {
  } else {
    cljs.core.t_cljs$core10674 = function(meta10675) {
      this.meta10675 = meta10675;
      this.cljs$lang$protocol_mask$partition0$ = 393216;
      this.cljs$lang$protocol_mask$partition1$ = 0;
    };
    cljs.core.t_cljs$core10674.prototype.cljs$core$IWithMeta$_with_meta$arity$2 = function(_10676, meta10675__$1) {
      var self__ = this;
      var _10676__$1 = this;
      return new cljs.core.t_cljs$core10674(meta10675__$1);
    };
    cljs.core.t_cljs$core10674.prototype.cljs$core$IMeta$_meta$arity$1 = function(_10676) {
      var self__ = this;
      var _10676__$1 = this;
      return self__.meta10675;
    };
    cljs.core.t_cljs$core10674.prototype.hasNext = function() {
      var self__ = this;
      var _ = this;
      return false;
    };
    cljs.core.t_cljs$core10674.prototype.next = function() {
      var self__ = this;
      var _ = this;
      return new Error("No such element");
    };
    cljs.core.t_cljs$core10674.prototype.remove = function() {
      var self__ = this;
      var _ = this;
      return new Error("Unsupported operation");
    };
    cljs.core.t_cljs$core10674.getBasis = function() {
      return new cljs.core.PersistentVector(null, 1, 5, cljs.core.PersistentVector.EMPTY_NODE, [new cljs.core.Symbol(null, "meta10675", "meta10675", -285927061, null)], null);
    };
    cljs.core.t_cljs$core10674.cljs$lang$type = true;
    cljs.core.t_cljs$core10674.cljs$lang$ctorStr = "cljs.core/t_cljs$core10674";
    cljs.core.t_cljs$core10674.cljs$lang$ctorPrWriter = function(this__7871__auto__, writer__7872__auto__, opt__7873__auto__) {
      return cljs.core._write(writer__7872__auto__, "cljs.core/t_cljs$core10674");
    };
    cljs.core.__GT_t_cljs$core10674 = function cljs$core$nil_iter_$___GT_t_cljs$core10674(meta10675) {
      return new cljs.core.t_cljs$core10674(meta10675);
    };
  }
  return new cljs.core.t_cljs$core10674(cljs.core.PersistentArrayMap.EMPTY);
};
cljs.core.StringIter = function(s, i) {
  this.s = s;
  this.i = i;
};
cljs.core.StringIter.prototype.hasNext = function() {
  var self__ = this;
  var _ = this;
  return self__.i < self__.s.length;
};
cljs.core.StringIter.prototype.next = function() {
  var self__ = this;
  var _ = this;
  var ret = self__.s.charAt(self__.i);
  self__.i = self__.i + 1;
  return ret;
};
cljs.core.StringIter.prototype.remove = function() {
  var self__ = this;
  var _ = this;
  return new Error("Unsupported operation");
};
cljs.core.StringIter.getBasis = function() {
  return new cljs.core.PersistentVector(null, 2, 5, cljs.core.PersistentVector.EMPTY_NODE, [new cljs.core.Symbol(null, "s", "s", -948495851, null), cljs.core.with_meta(new cljs.core.Symbol(null, "i", "i", 253690212, null), new cljs.core.PersistentArrayMap(null, 1, [new cljs.core.Keyword(null, "mutable", "mutable", 875778266), true], null))], null);
};
cljs.core.StringIter.cljs$lang$type = true;
cljs.core.StringIter.cljs$lang$ctorStr = "cljs.core/StringIter";
cljs.core.StringIter.cljs$lang$ctorPrWriter = function(this__7871__auto__, writer__7872__auto__, opt__7873__auto__) {
  return cljs.core._write(writer__7872__auto__, "cljs.core/StringIter");
};
cljs.core.__GT_StringIter = function cljs$core$__GT_StringIter(s, i) {
  return new cljs.core.StringIter(s, i);
};
cljs.core.string_iter = function cljs$core$string_iter(x) {
  return new cljs.core.StringIter(x, 0);
};
cljs.core.ArrayIter = function(arr, i) {
  this.arr = arr;
  this.i = i;
};
cljs.core.ArrayIter.prototype.hasNext = function() {
  var self__ = this;
  var _ = this;
  return self__.i < self__.arr.length;
};
cljs.core.ArrayIter.prototype.next = function() {
  var self__ = this;
  var _ = this;
  var ret = self__.arr[self__.i];
  self__.i = self__.i + 1;
  return ret;
};
cljs.core.ArrayIter.prototype.remove = function() {
  var self__ = this;
  var _ = this;
  return new Error("Unsupported operation");
};
cljs.core.ArrayIter.getBasis = function() {
  return new cljs.core.PersistentVector(null, 2, 5, cljs.core.PersistentVector.EMPTY_NODE, [new cljs.core.Symbol(null, "arr", "arr", 2115492975, null), cljs.core.with_meta(new cljs.core.Symbol(null, "i", "i", 253690212, null), new cljs.core.PersistentArrayMap(null, 1, [new cljs.core.Keyword(null, "mutable", "mutable", 875778266), true], null))], null);
};
cljs.core.ArrayIter.cljs$lang$type = true;
cljs.core.ArrayIter.cljs$lang$ctorStr = "cljs.core/ArrayIter";
cljs.core.ArrayIter.cljs$lang$ctorPrWriter = function(this__7871__auto__, writer__7872__auto__, opt__7873__auto__) {
  return cljs.core._write(writer__7872__auto__, "cljs.core/ArrayIter");
};
cljs.core.__GT_ArrayIter = function cljs$core$__GT_ArrayIter(arr, i) {
  return new cljs.core.ArrayIter(arr, i);
};
cljs.core.array_iter = function cljs$core$array_iter(x) {
  return new cljs.core.ArrayIter(x, 0);
};
cljs.core.INIT = {};
cljs.core.START = {};
cljs.core.SeqIter = function(_seq, _next) {
  this._seq = _seq;
  this._next = _next;
};
cljs.core.SeqIter.prototype.hasNext = function() {
  var self__ = this;
  var _ = this;
  if (self__._seq === cljs.core.INIT) {
    self__._seq = cljs.core.START;
    self__._next = cljs.core.seq(self__._next);
  } else {
    if (self__._seq === self__._next) {
      self__._next = cljs.core.next(self__._seq);
    } else {
    }
  }
  return !(self__._next == null);
};
cljs.core.SeqIter.prototype.next = function() {
  var self__ = this;
  var this$ = this;
  if (!this$.hasNext()) {
    throw new Error("No such element");
  } else {
    self__._seq = self__._next;
    return cljs.core.first(self__._next);
  }
};
cljs.core.SeqIter.prototype.remove = function() {
  var self__ = this;
  var _ = this;
  return new Error("Unsupported operation");
};
cljs.core.SeqIter.getBasis = function() {
  return new cljs.core.PersistentVector(null, 2, 5, cljs.core.PersistentVector.EMPTY_NODE, [cljs.core.with_meta(new cljs.core.Symbol(null, "_seq", "_seq", -449557847, null), new cljs.core.PersistentArrayMap(null, 1, [new cljs.core.Keyword(null, "mutable", "mutable", 875778266), true], null)), cljs.core.with_meta(new cljs.core.Symbol(null, "_next", "_next", 101877036, null), new cljs.core.PersistentArrayMap(null, 1, [new cljs.core.Keyword(null, "mutable", "mutable", 875778266), true], null))], null);
};
cljs.core.SeqIter.cljs$lang$type = true;
cljs.core.SeqIter.cljs$lang$ctorStr = "cljs.core/SeqIter";
cljs.core.SeqIter.cljs$lang$ctorPrWriter = function(this__7871__auto__, writer__7872__auto__, opt__7873__auto__) {
  return cljs.core._write(writer__7872__auto__, "cljs.core/SeqIter");
};
cljs.core.__GT_SeqIter = function cljs$core$__GT_SeqIter(_seq, _next) {
  return new cljs.core.SeqIter(_seq, _next);
};
cljs.core.seq_iter = function cljs$core$seq_iter(coll) {
  return new cljs.core.SeqIter(cljs.core.INIT, coll);
};
cljs.core.iter = function cljs$core$iter(coll) {
  if (coll == null) {
    return cljs.core.nil_iter();
  } else {
    if (typeof coll === "string") {
      return cljs.core.string_iter(coll);
    } else {
      if (cljs.core.array_QMARK_(coll)) {
        return cljs.core.array_iter(coll);
      } else {
        if (cljs.core.iterable_QMARK_(coll)) {
          return cljs.core._iterator(coll);
        } else {
          if (cljs.core.seqable_QMARK_(coll)) {
            return cljs.core.seq_iter(coll);
          } else {
            throw new Error([cljs.core.str.cljs$core$IFn$_invoke$arity$1("Cannot create iterator from "), cljs.core.str.cljs$core$IFn$_invoke$arity$1(coll)].join(""));
          }
        }
      }
    }
  }
};
cljs.core.lazy_transformer = function cljs$core$lazy_transformer(stepper) {
  return new cljs.core.LazyTransformer(stepper, null, null, null);
};
cljs.core.Stepper = function(xform, iter) {
  this.xform = xform;
  this.iter = iter;
};
cljs.core.Stepper.prototype.step = function(lt) {
  var self__ = this;
  var this$ = this;
  while (true) {
    if (cljs.core.truth_(function() {
      var and__7248__auto__ = !(lt.stepper == null);
      if (and__7248__auto__) {
        return self__.iter.hasNext();
      } else {
        return and__7248__auto__;
      }
    }())) {
      if (cljs.core.reduced_QMARK_(function() {
        var G__10677 = lt;
        var G__10678 = self__.iter.next();
        return self__.xform.cljs$core$IFn$_invoke$arity$2 ? self__.xform.cljs$core$IFn$_invoke$arity$2(G__10677, G__10678) : self__.xform.call(null, G__10677, G__10678);
      }())) {
        if (lt.rest == null) {
        } else {
          lt.rest.stepper = null;
        }
      } else {
        continue;
      }
    } else {
    }
    break;
  }
  if (lt.stepper == null) {
    return null;
  } else {
    return self__.xform.cljs$core$IFn$_invoke$arity$1 ? self__.xform.cljs$core$IFn$_invoke$arity$1(lt) : self__.xform.call(null, lt);
  }
};
cljs.core.Stepper.getBasis = function() {
  return new cljs.core.PersistentVector(null, 2, 5, cljs.core.PersistentVector.EMPTY_NODE, [new cljs.core.Symbol(null, "xform", "xform", -85179481, null), new cljs.core.Symbol(null, "iter", "iter", -1346195486, null)], null);
};
cljs.core.Stepper.cljs$lang$type = true;
cljs.core.Stepper.cljs$lang$ctorStr = "cljs.core/Stepper";
cljs.core.Stepper.cljs$lang$ctorPrWriter = function(this__7871__auto__, writer__7872__auto__, opt__7873__auto__) {
  return cljs.core._write(writer__7872__auto__, "cljs.core/Stepper");
};
cljs.core.__GT_Stepper = function cljs$core$__GT_Stepper(xform, iter) {
  return new cljs.core.Stepper(xform, iter);
};
cljs.core.stepper = function cljs$core$stepper(xform, iter) {
  var stepfn = function() {
    var cljs$core$stepper_$_stepfn = null;
    var cljs$core$stepper_$_stepfn__1 = function(result) {
      var lt = cljs.core.reduced_QMARK_(result) ? cljs.core.deref(result) : result;
      lt.stepper = null;
      return result;
    };
    var cljs$core$stepper_$_stepfn__2 = function(result, input) {
      var lt = result;
      lt.first = input;
      lt.rest = cljs.core.lazy_transformer(lt.stepper);
      lt.stepper = null;
      return lt.rest;
    };
    cljs$core$stepper_$_stepfn = function(result, input) {
      switch(arguments.length) {
        case 1:
          return cljs$core$stepper_$_stepfn__1.call(this, result);
        case 2:
          return cljs$core$stepper_$_stepfn__2.call(this, result, input);
      }
      throw new Error("Invalid arity: " + (arguments.length - 1));
    };
    cljs$core$stepper_$_stepfn.cljs$core$IFn$_invoke$arity$1 = cljs$core$stepper_$_stepfn__1;
    cljs$core$stepper_$_stepfn.cljs$core$IFn$_invoke$arity$2 = cljs$core$stepper_$_stepfn__2;
    return cljs$core$stepper_$_stepfn;
  }();
  return new cljs.core.Stepper(xform.cljs$core$IFn$_invoke$arity$1 ? xform.cljs$core$IFn$_invoke$arity$1(stepfn) : xform.call(null, stepfn), iter);
};
cljs.core.MultiStepper = function(xform, iters, nexts) {
  this.xform = xform;
  this.iters = iters;
  this.nexts = nexts;
};
cljs.core.MultiStepper.prototype.hasNext = function() {
  var self__ = this;
  var _ = this;
  var iters__$1 = cljs.core.seq(self__.iters);
  while (true) {
    if (!(iters__$1 == null)) {
      var iter = cljs.core.first(iters__$1);
      if (!iter.hasNext()) {
        return false;
      } else {
        var G__10679 = cljs.core.next(iters__$1);
        iters__$1 = G__10679;
        continue;
      }
    } else {
      return true;
    }
    break;
  }
};
cljs.core.MultiStepper.prototype.next = function() {
  var self__ = this;
  var _ = this;
  var n__8183__auto___10680 = self__.iters.length;
  var i_10681 = 0;
  while (true) {
    if (i_10681 < n__8183__auto___10680) {
      self__.nexts[i_10681] = self__.iters[i_10681].next();
      var G__10682 = i_10681 + 1;
      i_10681 = G__10682;
      continue;
    } else {
    }
    break;
  }
  return cljs.core.prim_seq.cljs$core$IFn$_invoke$arity$2(self__.nexts, 0);
};
cljs.core.MultiStepper.prototype.step = function(lt) {
  var self__ = this;
  var this$ = this;
  while (true) {
    if (cljs.core.truth_(function() {
      var and__7248__auto__ = !(lt.stepper == null);
      if (and__7248__auto__) {
        return this$.hasNext();
      } else {
        return and__7248__auto__;
      }
    }())) {
      if (cljs.core.reduced_QMARK_(cljs.core.apply.cljs$core$IFn$_invoke$arity$2(self__.xform, cljs.core.cons(lt, this$.next())))) {
        if (lt.rest == null) {
        } else {
          lt.rest.stepper = null;
        }
      } else {
        continue;
      }
    } else {
    }
    break;
  }
  if (lt.stepper == null) {
    return null;
  } else {
    return self__.xform.cljs$core$IFn$_invoke$arity$1 ? self__.xform.cljs$core$IFn$_invoke$arity$1(lt) : self__.xform.call(null, lt);
  }
};
cljs.core.MultiStepper.getBasis = function() {
  return new cljs.core.PersistentVector(null, 3, 5, cljs.core.PersistentVector.EMPTY_NODE, [new cljs.core.Symbol(null, "xform", "xform", -85179481, null), new cljs.core.Symbol(null, "iters", "iters", 719353031, null), new cljs.core.Symbol(null, "nexts", "nexts", -1683579407, null)], null);
};
cljs.core.MultiStepper.cljs$lang$type = true;
cljs.core.MultiStepper.cljs$lang$ctorStr = "cljs.core/MultiStepper";
cljs.core.MultiStepper.cljs$lang$ctorPrWriter = function(this__7871__auto__, writer__7872__auto__, opt__7873__auto__) {
  return cljs.core._write(writer__7872__auto__, "cljs.core/MultiStepper");
};
cljs.core.__GT_MultiStepper = function cljs$core$__GT_MultiStepper(xform, iters, nexts) {
  return new cljs.core.MultiStepper(xform, iters, nexts);
};
cljs.core.multi_stepper = function cljs$core$multi_stepper(var_args) {
  var args10683 = [];
  var len__8373__auto___10686 = arguments.length;
  var i__8374__auto___10687 = 0;
  while (true) {
    if (i__8374__auto___10687 < len__8373__auto___10686) {
      args10683.push(arguments[i__8374__auto___10687]);
      var G__10688 = i__8374__auto___10687 + 1;
      i__8374__auto___10687 = G__10688;
      continue;
    } else {
    }
    break;
  }
  var G__10685 = args10683.length;
  switch(G__10685) {
    case 2:
      return cljs.core.multi_stepper.cljs$core$IFn$_invoke$arity$2(arguments[0], arguments[1]);
      break;
    case 3:
      return cljs.core.multi_stepper.cljs$core$IFn$_invoke$arity$3(arguments[0], arguments[1], arguments[2]);
      break;
    default:
      throw new Error([cljs.core.str.cljs$core$IFn$_invoke$arity$1("Invalid arity: "), cljs.core.str.cljs$core$IFn$_invoke$arity$1(args10683.length)].join(""));
  }
};
cljs.core.multi_stepper.cljs$core$IFn$_invoke$arity$2 = function(xform, iters) {
  return cljs.core.multi_stepper.cljs$core$IFn$_invoke$arity$3(xform, iters, new Array(iters.length));
};
cljs.core.multi_stepper.cljs$core$IFn$_invoke$arity$3 = function(xform, iters, nexts) {
  var stepfn = function() {
    var cljs$core$stepfn = null;
    var cljs$core$stepfn__1 = function(result) {
      var lt = cljs.core.reduced_QMARK_(result) ? cljs.core.deref(result) : result;
      lt.stepper = null;
      return lt;
    };
    var cljs$core$stepfn__2 = function(result, input) {
      var lt = result;
      lt.first = input;
      lt.rest = cljs.core.lazy_transformer(lt.stepper);
      lt.stepper = null;
      return lt.rest;
    };
    cljs$core$stepfn = function(result, input) {
      switch(arguments.length) {
        case 1:
          return cljs$core$stepfn__1.call(this, result);
        case 2:
          return cljs$core$stepfn__2.call(this, result, input);
      }
      throw new Error("Invalid arity: " + (arguments.length - 1));
    };
    cljs$core$stepfn.cljs$core$IFn$_invoke$arity$1 = cljs$core$stepfn__1;
    cljs$core$stepfn.cljs$core$IFn$_invoke$arity$2 = cljs$core$stepfn__2;
    return cljs$core$stepfn;
  }();
  return new cljs.core.MultiStepper(xform.cljs$core$IFn$_invoke$arity$1 ? xform.cljs$core$IFn$_invoke$arity$1(stepfn) : xform.call(null, stepfn), iters, nexts);
};
cljs.core.multi_stepper.cljs$lang$maxFixedArity = 3;
cljs.core.LazyTransformer = function(stepper, first, rest, meta) {
  this.stepper = stepper;
  this.first = first;
  this.rest = rest;
  this.meta = meta;
  this.cljs$lang$protocol_mask$partition0$ = 31850700;
  this.cljs$lang$protocol_mask$partition1$ = 0;
};
cljs.core.LazyTransformer.prototype.indexOf = function() {
  var G__10690 = null;
  var G__10690__1 = function(x) {
    var self__ = this;
    var coll = this;
    return cljs.core._indexOf.cljs$core$IFn$_invoke$arity$3(coll, x, 0);
  };
  var G__10690__2 = function(x, start) {
    var self__ = this;
    var coll = this;
    return cljs.core._indexOf.cljs$core$IFn$_invoke$arity$3(coll, x, start);
  };
  G__10690 = function(x, start) {
    switch(arguments.length) {
      case 1:
        return G__10690__1.call(this, x);
      case 2:
        return G__10690__2.call(this, x, start);
    }
    throw new Error("Invalid arity: " + (arguments.length - 1));
  };
  G__10690.cljs$core$IFn$_invoke$arity$1 = G__10690__1;
  G__10690.cljs$core$IFn$_invoke$arity$2 = G__10690__2;
  return G__10690;
}();
cljs.core.LazyTransformer.prototype.lastIndexOf = function() {
  var G__10691 = null;
  var G__10691__1 = function(x) {
    var self__ = this;
    var coll = this;
    return cljs.core._lastIndexOf.cljs$core$IFn$_invoke$arity$3(coll, x, cljs.core.count(coll));
  };
  var G__10691__2 = function(x, start) {
    var self__ = this;
    var coll = this;
    return cljs.core._lastIndexOf.cljs$core$IFn$_invoke$arity$3(coll, x, start);
  };
  G__10691 = function(x, start) {
    switch(arguments.length) {
      case 1:
        return G__10691__1.call(this, x);
      case 2:
        return G__10691__2.call(this, x, start);
    }
    throw new Error("Invalid arity: " + (arguments.length - 1));
  };
  G__10691.cljs$core$IFn$_invoke$arity$1 = G__10691__1;
  G__10691.cljs$core$IFn$_invoke$arity$2 = G__10691__2;
  return G__10691;
}();
cljs.core.LazyTransformer.prototype.cljs$core$IMeta$_meta$arity$1 = function(this$) {
  var self__ = this;
  var this$__$1 = this;
  return self__.meta;
};
cljs.core.LazyTransformer.prototype.cljs$core$INext$_next$arity$1 = function(this$) {
  var self__ = this;
  var this$__$1 = this;
  if (self__.stepper == null) {
  } else {
    this$__$1.cljs$core$ISeqable$_seq$arity$1(null);
  }
  if (self__.rest == null) {
    return null;
  } else {
    return cljs.core._seq(self__.rest);
  }
};
cljs.core.LazyTransformer.prototype.cljs$core$IHash$_hash$arity$1 = function(this$) {
  var self__ = this;
  var this$__$1 = this;
  return cljs.core.hash_ordered_coll(this$__$1);
};
cljs.core.LazyTransformer.prototype.cljs$core$IEquiv$_equiv$arity$2 = function(this$, other) {
  var self__ = this;
  var this$__$1 = this;
  var s = this$__$1.cljs$core$ISeqable$_seq$arity$1(null);
  if (!(s == null)) {
    return cljs.core.equiv_sequential(this$__$1, other);
  } else {
    return cljs.core.sequential_QMARK_(other) && cljs.core.seq(other) == null;
  }
};
cljs.core.LazyTransformer.prototype.cljs$core$IEmptyableCollection$_empty$arity$1 = function(this$) {
  var self__ = this;
  var this$__$1 = this;
  return cljs.core.List.EMPTY;
};
cljs.core.LazyTransformer.prototype.cljs$core$ISeq$_first$arity$1 = function(this$) {
  var self__ = this;
  var this$__$1 = this;
  if (self__.stepper == null) {
  } else {
    this$__$1.cljs$core$ISeqable$_seq$arity$1(null);
  }
  if (self__.rest == null) {
    return null;
  } else {
    return self__.first;
  }
};
cljs.core.LazyTransformer.prototype.cljs$core$ISeq$_rest$arity$1 = function(this$) {
  var self__ = this;
  var this$__$1 = this;
  if (self__.stepper == null) {
  } else {
    this$__$1.cljs$core$ISeqable$_seq$arity$1(null);
  }
  if (self__.rest == null) {
    return cljs.core.List.EMPTY;
  } else {
    return self__.rest;
  }
};
cljs.core.LazyTransformer.prototype.cljs$core$ISeqable$_seq$arity$1 = function(this$) {
  var self__ = this;
  var this$__$1 = this;
  if (self__.stepper == null) {
  } else {
    self__.stepper.step(this$__$1);
  }
  if (self__.rest == null) {
    return null;
  } else {
    return this$__$1;
  }
};
cljs.core.LazyTransformer.prototype.cljs$core$IWithMeta$_with_meta$arity$2 = function(this$, new_meta) {
  var self__ = this;
  var this$__$1 = this;
  return new cljs.core.LazyTransformer(self__.stepper, self__.first, self__.rest, new_meta);
};
cljs.core.LazyTransformer.prototype.cljs$core$ICollection$_conj$arity$2 = function(this$, o) {
  var self__ = this;
  var this$__$1 = this;
  return cljs.core.cons(o, this$__$1.cljs$core$ISeqable$_seq$arity$1(null));
};
cljs.core.LazyTransformer.getBasis = function() {
  return new cljs.core.PersistentVector(null, 4, 5, cljs.core.PersistentVector.EMPTY_NODE, [cljs.core.with_meta(new cljs.core.Symbol(null, "stepper", "stepper", 1159163296, null), new cljs.core.PersistentArrayMap(null, 1, [new cljs.core.Keyword(null, "mutable", "mutable", 875778266), true], null)), cljs.core.with_meta(new cljs.core.Symbol(null, "first", "first", 996428481, null), new cljs.core.PersistentArrayMap(null, 1, [new cljs.core.Keyword(null, "mutable", "mutable", 875778266), true], null)), 
  cljs.core.with_meta(new cljs.core.Symbol(null, "rest", "rest", 398835108, null), new cljs.core.PersistentArrayMap(null, 1, [new cljs.core.Keyword(null, "mutable", "mutable", 875778266), true], null)), new cljs.core.Symbol(null, "meta", "meta", -1154898805, null)], null);
};
cljs.core.LazyTransformer.cljs$lang$type = true;
cljs.core.LazyTransformer.cljs$lang$ctorStr = "cljs.core/LazyTransformer";
cljs.core.LazyTransformer.cljs$lang$ctorPrWriter = function(this__7871__auto__, writer__7872__auto__, opt__7873__auto__) {
  return cljs.core._write(writer__7872__auto__, "cljs.core/LazyTransformer");
};
cljs.core.__GT_LazyTransformer = function cljs$core$__GT_LazyTransformer(stepper, first, rest, meta) {
  return new cljs.core.LazyTransformer(stepper, first, rest, meta);
};
cljs.core.LazyTransformer.prototype[cljs.core.ITER_SYMBOL] = function() {
  var this__8274__auto__ = this;
  return cljs.core.es6_iterator(this__8274__auto__);
};
cljs.core.LazyTransformer.create = function(xform, coll) {
  return new cljs.core.LazyTransformer(cljs.core.stepper(xform, cljs.core.iter(coll)), null, null, null);
};
cljs.core.LazyTransformer.createMulti = function(xform, colls) {
  var iters = [];
  var seq__10692_10696 = cljs.core.seq(colls);
  var chunk__10693_10697 = null;
  var count__10694_10698 = 0;
  var i__10695_10699 = 0;
  while (true) {
    if (i__10695_10699 < count__10694_10698) {
      var coll_10700 = chunk__10693_10697.cljs$core$IIndexed$_nth$arity$2(null, i__10695_10699);
      iters.push(cljs.core.iter(coll_10700));
      var G__10701 = seq__10692_10696;
      var G__10702 = chunk__10693_10697;
      var G__10703 = count__10694_10698;
      var G__10704 = i__10695_10699 + 1;
      seq__10692_10696 = G__10701;
      chunk__10693_10697 = G__10702;
      count__10694_10698 = G__10703;
      i__10695_10699 = G__10704;
      continue;
    } else {
      var temp__4657__auto___10705 = cljs.core.seq(seq__10692_10696);
      if (temp__4657__auto___10705) {
        var seq__10692_10706__$1 = temp__4657__auto___10705;
        if (cljs.core.chunked_seq_QMARK_(seq__10692_10706__$1)) {
          var c__8079__auto___10707 = cljs.core.chunk_first(seq__10692_10706__$1);
          var G__10708 = cljs.core.chunk_rest(seq__10692_10706__$1);
          var G__10709 = c__8079__auto___10707;
          var G__10710 = cljs.core.count(c__8079__auto___10707);
          var G__10711 = 0;
          seq__10692_10696 = G__10708;
          chunk__10693_10697 = G__10709;
          count__10694_10698 = G__10710;
          i__10695_10699 = G__10711;
          continue;
        } else {
          var coll_10712 = cljs.core.first(seq__10692_10706__$1);
          iters.push(cljs.core.iter(coll_10712));
          var G__10713 = cljs.core.next(seq__10692_10706__$1);
          var G__10714 = null;
          var G__10715 = 0;
          var G__10716 = 0;
          seq__10692_10696 = G__10713;
          chunk__10693_10697 = G__10714;
          count__10694_10698 = G__10715;
          i__10695_10699 = G__10716;
          continue;
        }
      } else {
      }
    }
    break;
  }
  return new cljs.core.LazyTransformer(cljs.core.multi_stepper.cljs$core$IFn$_invoke$arity$3(xform, iters, new Array(iters.length)), null, null, null);
};
cljs.core.sequence = function cljs$core$sequence(var_args) {
  var args10717 = [];
  var len__8373__auto___10723 = arguments.length;
  var i__8374__auto___10724 = 0;
  while (true) {
    if (i__8374__auto___10724 < len__8373__auto___10723) {
      args10717.push(arguments[i__8374__auto___10724]);
      var G__10725 = i__8374__auto___10724 + 1;
      i__8374__auto___10724 = G__10725;
      continue;
    } else {
    }
    break;
  }
  var G__10722 = args10717.length;
  switch(G__10722) {
    case 1:
      return cljs.core.sequence.cljs$core$IFn$_invoke$arity$1(arguments[0]);
      break;
    case 2:
      return cljs.core.sequence.cljs$core$IFn$_invoke$arity$2(arguments[0], arguments[1]);
      break;
    default:
      var argseq__8392__auto__ = new cljs.core.IndexedSeq(args10717.slice(2), 0, null);
      return cljs.core.sequence.cljs$core$IFn$_invoke$arity$variadic(arguments[0], arguments[1], argseq__8392__auto__);
  }
};
cljs.core.sequence.cljs$core$IFn$_invoke$arity$1 = function(coll) {
  if (cljs.core.seq_QMARK_(coll)) {
    return coll;
  } else {
    var or__7260__auto__ = cljs.core.seq(coll);
    if (or__7260__auto__) {
      return or__7260__auto__;
    } else {
      return cljs.core.List.EMPTY;
    }
  }
};
cljs.core.sequence.cljs$core$IFn$_invoke$arity$2 = function(xform, coll) {
  return cljs.core.LazyTransformer.create(xform, coll);
};
cljs.core.sequence.cljs$core$IFn$_invoke$arity$variadic = function(xform, coll, colls) {
  return cljs.core.LazyTransformer.createMulti(xform, cljs.core.to_array(cljs.core.cons(coll, colls)));
};
cljs.core.sequence.cljs$lang$applyTo = function(seq10718) {
  var G__10719 = cljs.core.first(seq10718);
  var seq10718__$1 = cljs.core.next(seq10718);
  var G__10720 = cljs.core.first(seq10718__$1);
  var seq10718__$2 = cljs.core.next(seq10718__$1);
  return cljs.core.sequence.cljs$core$IFn$_invoke$arity$variadic(G__10719, G__10720, seq10718__$2);
};
cljs.core.sequence.cljs$lang$maxFixedArity = 2;
cljs.core.every_QMARK_ = function cljs$core$every_QMARK_(pred, coll) {
  while (true) {
    if (cljs.core.seq(coll) == null) {
      return true;
    } else {
      if (cljs.core.truth_(function() {
        var G__10728 = cljs.core.first(coll);
        return pred.cljs$core$IFn$_invoke$arity$1 ? pred.cljs$core$IFn$_invoke$arity$1(G__10728) : pred.call(null, G__10728);
      }())) {
        var G__10729 = pred;
        var G__10730 = cljs.core.next(coll);
        pred = G__10729;
        coll = G__10730;
        continue;
      } else {
        return false;
      }
    }
    break;
  }
};
cljs.core.not_every_QMARK_ = function cljs$core$not_every_QMARK_(pred, coll) {
  return !cljs.core.every_QMARK_(pred, coll);
};
cljs.core.some = function cljs$core$some(pred, coll) {
  while (true) {
    if (cljs.core.seq(coll)) {
      var or__7260__auto__ = function() {
        var G__10734 = cljs.core.first(coll);
        return pred.cljs$core$IFn$_invoke$arity$1 ? pred.cljs$core$IFn$_invoke$arity$1(G__10734) : pred.call(null, G__10734);
      }();
      if (cljs.core.truth_(or__7260__auto__)) {
        return or__7260__auto__;
      } else {
        var G__10735 = pred;
        var G__10736 = cljs.core.next(coll);
        pred = G__10735;
        coll = G__10736;
        continue;
      }
    } else {
      return null;
    }
    break;
  }
};
cljs.core.not_any_QMARK_ = function cljs$core$not_any_QMARK_(pred, coll) {
  return cljs.core.not(cljs.core.some(pred, coll));
};
cljs.core.even_QMARK_ = function cljs$core$even_QMARK_(n) {
  if (cljs.core.integer_QMARK_(n)) {
    return (n & 1) === 0;
  } else {
    throw new Error([cljs.core.str.cljs$core$IFn$_invoke$arity$1("Argument must be an integer: "), cljs.core.str.cljs$core$IFn$_invoke$arity$1(n)].join(""));
  }
};
cljs.core.odd_QMARK_ = function cljs$core$odd_QMARK_(n) {
  return !cljs.core.even_QMARK_(n);
};
cljs.core.complement = function cljs$core$complement(f) {
  return function() {
    var G__10737 = null;
    var G__10737__0 = function() {
      return cljs.core.not(f.cljs$core$IFn$_invoke$arity$0 ? f.cljs$core$IFn$_invoke$arity$0() : f.call(null));
    };
    var G__10737__1 = function(x) {
      return cljs.core.not(f.cljs$core$IFn$_invoke$arity$1 ? f.cljs$core$IFn$_invoke$arity$1(x) : f.call(null, x));
    };
    var G__10737__2 = function(x, y) {
      return cljs.core.not(f.cljs$core$IFn$_invoke$arity$2 ? f.cljs$core$IFn$_invoke$arity$2(x, y) : f.call(null, x, y));
    };
    var G__10737__3 = function() {
      var G__10738__delegate = function(x, y, zs) {
        return cljs.core.not(cljs.core.apply.cljs$core$IFn$_invoke$arity$4(f, x, y, zs));
      };
      var G__10738 = function(x, y, var_args) {
        var zs = null;
        if (arguments.length > 2) {
          var G__10739__i = 0, G__10739__a = new Array(arguments.length - 2);
          while (G__10739__i < G__10739__a.length) {
            G__10739__a[G__10739__i] = arguments[G__10739__i + 2];
            ++G__10739__i;
          }
          zs = new cljs.core.IndexedSeq(G__10739__a, 0);
        }
        return G__10738__delegate.call(this, x, y, zs);
      };
      G__10738.cljs$lang$maxFixedArity = 2;
      G__10738.cljs$lang$applyTo = function(arglist__10740) {
        var x = cljs.core.first(arglist__10740);
        arglist__10740 = cljs.core.next(arglist__10740);
        var y = cljs.core.first(arglist__10740);
        var zs = cljs.core.rest(arglist__10740);
        return G__10738__delegate(x, y, zs);
      };
      G__10738.cljs$core$IFn$_invoke$arity$variadic = G__10738__delegate;
      return G__10738;
    }();
    G__10737 = function(x, y, var_args) {
      var zs = var_args;
      switch(arguments.length) {
        case 0:
          return G__10737__0.call(this);
        case 1:
          return G__10737__1.call(this, x);
        case 2:
          return G__10737__2.call(this, x, y);
        default:
          var G__10741 = null;
          if (arguments.length > 2) {
            var G__10742__i = 0, G__10742__a = new Array(arguments.length - 2);
            while (G__10742__i < G__10742__a.length) {
              G__10742__a[G__10742__i] = arguments[G__10742__i + 2];
              ++G__10742__i;
            }
            G__10741 = new cljs.core.IndexedSeq(G__10742__a, 0);
          }
          return G__10737__3.cljs$core$IFn$_invoke$arity$variadic(x, y, G__10741);
      }
      throw new Error("Invalid arity: " + (arguments.length - 1));
    };
    G__10737.cljs$lang$maxFixedArity = 2;
    G__10737.cljs$lang$applyTo = G__10737__3.cljs$lang$applyTo;
    G__10737.cljs$core$IFn$_invoke$arity$0 = G__10737__0;
    G__10737.cljs$core$IFn$_invoke$arity$1 = G__10737__1;
    G__10737.cljs$core$IFn$_invoke$arity$2 = G__10737__2;
    G__10737.cljs$core$IFn$_invoke$arity$variadic = G__10737__3.cljs$core$IFn$_invoke$arity$variadic;
    return G__10737;
  }();
};
cljs.core.constantly = function cljs$core$constantly(x) {
  return function() {
    var G__10743__delegate = function(args) {
      return x;
    };
    var G__10743 = function(var_args) {
      var args = null;
      if (arguments.length > 0) {
        var G__10744__i = 0, G__10744__a = new Array(arguments.length - 0);
        while (G__10744__i < G__10744__a.length) {
          G__10744__a[G__10744__i] = arguments[G__10744__i + 0];
          ++G__10744__i;
        }
        args = new cljs.core.IndexedSeq(G__10744__a, 0);
      }
      return G__10743__delegate.call(this, args);
    };
    G__10743.cljs$lang$maxFixedArity = 0;
    G__10743.cljs$lang$applyTo = function(arglist__10745) {
      var args = cljs.core.seq(arglist__10745);
      return G__10743__delegate(args);
    };
    G__10743.cljs$core$IFn$_invoke$arity$variadic = G__10743__delegate;
    return G__10743;
  }();
};
cljs.core.comp = function cljs$core$comp(var_args) {
  var args10746 = [];
  var len__8373__auto___10768 = arguments.length;
  var i__8374__auto___10769 = 0;
  while (true) {
    if (i__8374__auto___10769 < len__8373__auto___10768) {
      args10746.push(arguments[i__8374__auto___10769]);
      var G__10770 = i__8374__auto___10769 + 1;
      i__8374__auto___10769 = G__10770;
      continue;
    } else {
    }
    break;
  }
  var G__10752 = args10746.length;
  switch(G__10752) {
    case 0:
      return cljs.core.comp.cljs$core$IFn$_invoke$arity$0();
      break;
    case 1:
      return cljs.core.comp.cljs$core$IFn$_invoke$arity$1(arguments[0]);
      break;
    case 2:
      return cljs.core.comp.cljs$core$IFn$_invoke$arity$2(arguments[0], arguments[1]);
      break;
    case 3:
      return cljs.core.comp.cljs$core$IFn$_invoke$arity$3(arguments[0], arguments[1], arguments[2]);
      break;
    default:
      var argseq__8392__auto__ = new cljs.core.IndexedSeq(args10746.slice(3), 0, null);
      return cljs.core.comp.cljs$core$IFn$_invoke$arity$variadic(arguments[0], arguments[1], arguments[2], argseq__8392__auto__);
  }
};
cljs.core.comp.cljs$core$IFn$_invoke$arity$0 = function() {
  return cljs.core.identity;
};
cljs.core.comp.cljs$core$IFn$_invoke$arity$1 = function(f) {
  return f;
};
cljs.core.comp.cljs$core$IFn$_invoke$arity$2 = function(f, g) {
  return function() {
    var G__10772 = null;
    var G__10772__0 = function() {
      var G__10753 = g.cljs$core$IFn$_invoke$arity$0 ? g.cljs$core$IFn$_invoke$arity$0() : g.call(null);
      return f.cljs$core$IFn$_invoke$arity$1 ? f.cljs$core$IFn$_invoke$arity$1(G__10753) : f.call(null, G__10753);
    };
    var G__10772__1 = function(x) {
      var G__10754 = g.cljs$core$IFn$_invoke$arity$1 ? g.cljs$core$IFn$_invoke$arity$1(x) : g.call(null, x);
      return f.cljs$core$IFn$_invoke$arity$1 ? f.cljs$core$IFn$_invoke$arity$1(G__10754) : f.call(null, G__10754);
    };
    var G__10772__2 = function(x, y) {
      var G__10755 = g.cljs$core$IFn$_invoke$arity$2 ? g.cljs$core$IFn$_invoke$arity$2(x, y) : g.call(null, x, y);
      return f.cljs$core$IFn$_invoke$arity$1 ? f.cljs$core$IFn$_invoke$arity$1(G__10755) : f.call(null, G__10755);
    };
    var G__10772__3 = function(x, y, z) {
      var G__10756 = g.cljs$core$IFn$_invoke$arity$3 ? g.cljs$core$IFn$_invoke$arity$3(x, y, z) : g.call(null, x, y, z);
      return f.cljs$core$IFn$_invoke$arity$1 ? f.cljs$core$IFn$_invoke$arity$1(G__10756) : f.call(null, G__10756);
    };
    var G__10772__4 = function() {
      var G__10773__delegate = function(x, y, z, args) {
        var G__10757 = cljs.core.apply.cljs$core$IFn$_invoke$arity$5(g, x, y, z, args);
        return f.cljs$core$IFn$_invoke$arity$1 ? f.cljs$core$IFn$_invoke$arity$1(G__10757) : f.call(null, G__10757);
      };
      var G__10773 = function(x, y, z, var_args) {
        var args = null;
        if (arguments.length > 3) {
          var G__10774__i = 0, G__10774__a = new Array(arguments.length - 3);
          while (G__10774__i < G__10774__a.length) {
            G__10774__a[G__10774__i] = arguments[G__10774__i + 3];
            ++G__10774__i;
          }
          args = new cljs.core.IndexedSeq(G__10774__a, 0);
        }
        return G__10773__delegate.call(this, x, y, z, args);
      };
      G__10773.cljs$lang$maxFixedArity = 3;
      G__10773.cljs$lang$applyTo = function(arglist__10775) {
        var x = cljs.core.first(arglist__10775);
        arglist__10775 = cljs.core.next(arglist__10775);
        var y = cljs.core.first(arglist__10775);
        arglist__10775 = cljs.core.next(arglist__10775);
        var z = cljs.core.first(arglist__10775);
        var args = cljs.core.rest(arglist__10775);
        return G__10773__delegate(x, y, z, args);
      };
      G__10773.cljs$core$IFn$_invoke$arity$variadic = G__10773__delegate;
      return G__10773;
    }();
    G__10772 = function(x, y, z, var_args) {
      var args = var_args;
      switch(arguments.length) {
        case 0:
          return G__10772__0.call(this);
        case 1:
          return G__10772__1.call(this, x);
        case 2:
          return G__10772__2.call(this, x, y);
        case 3:
          return G__10772__3.call(this, x, y, z);
        default:
          var G__10776 = null;
          if (arguments.length > 3) {
            var G__10777__i = 0, G__10777__a = new Array(arguments.length - 3);
            while (G__10777__i < G__10777__a.length) {
              G__10777__a[G__10777__i] = arguments[G__10777__i + 3];
              ++G__10777__i;
            }
            G__10776 = new cljs.core.IndexedSeq(G__10777__a, 0);
          }
          return G__10772__4.cljs$core$IFn$_invoke$arity$variadic(x, y, z, G__10776);
      }
      throw new Error("Invalid arity: " + (arguments.length - 1));
    };
    G__10772.cljs$lang$maxFixedArity = 3;
    G__10772.cljs$lang$applyTo = G__10772__4.cljs$lang$applyTo;
    G__10772.cljs$core$IFn$_invoke$arity$0 = G__10772__0;
    G__10772.cljs$core$IFn$_invoke$arity$1 = G__10772__1;
    G__10772.cljs$core$IFn$_invoke$arity$2 = G__10772__2;
    G__10772.cljs$core$IFn$_invoke$arity$3 = G__10772__3;
    G__10772.cljs$core$IFn$_invoke$arity$variadic = G__10772__4.cljs$core$IFn$_invoke$arity$variadic;
    return G__10772;
  }();
};
cljs.core.comp.cljs$core$IFn$_invoke$arity$3 = function(f, g, h) {
  return function() {
    var G__10778 = null;
    var G__10778__0 = function() {
      var G__10758 = function() {
        var G__10759 = h.cljs$core$IFn$_invoke$arity$0 ? h.cljs$core$IFn$_invoke$arity$0() : h.call(null);
        return g.cljs$core$IFn$_invoke$arity$1 ? g.cljs$core$IFn$_invoke$arity$1(G__10759) : g.call(null, G__10759);
      }();
      return f.cljs$core$IFn$_invoke$arity$1 ? f.cljs$core$IFn$_invoke$arity$1(G__10758) : f.call(null, G__10758);
    };
    var G__10778__1 = function(x) {
      var G__10760 = function() {
        var G__10761 = h.cljs$core$IFn$_invoke$arity$1 ? h.cljs$core$IFn$_invoke$arity$1(x) : h.call(null, x);
        return g.cljs$core$IFn$_invoke$arity$1 ? g.cljs$core$IFn$_invoke$arity$1(G__10761) : g.call(null, G__10761);
      }();
      return f.cljs$core$IFn$_invoke$arity$1 ? f.cljs$core$IFn$_invoke$arity$1(G__10760) : f.call(null, G__10760);
    };
    var G__10778__2 = function(x, y) {
      var G__10762 = function() {
        var G__10763 = h.cljs$core$IFn$_invoke$arity$2 ? h.cljs$core$IFn$_invoke$arity$2(x, y) : h.call(null, x, y);
        return g.cljs$core$IFn$_invoke$arity$1 ? g.cljs$core$IFn$_invoke$arity$1(G__10763) : g.call(null, G__10763);
      }();
      return f.cljs$core$IFn$_invoke$arity$1 ? f.cljs$core$IFn$_invoke$arity$1(G__10762) : f.call(null, G__10762);
    };
    var G__10778__3 = function(x, y, z) {
      var G__10764 = function() {
        var G__10765 = h.cljs$core$IFn$_invoke$arity$3 ? h.cljs$core$IFn$_invoke$arity$3(x, y, z) : h.call(null, x, y, z);
        return g.cljs$core$IFn$_invoke$arity$1 ? g.cljs$core$IFn$_invoke$arity$1(G__10765) : g.call(null, G__10765);
      }();
      return f.cljs$core$IFn$_invoke$arity$1 ? f.cljs$core$IFn$_invoke$arity$1(G__10764) : f.call(null, G__10764);
    };
    var G__10778__4 = function() {
      var G__10779__delegate = function(x, y, z, args) {
        var G__10766 = function() {
          var G__10767 = cljs.core.apply.cljs$core$IFn$_invoke$arity$5(h, x, y, z, args);
          return g.cljs$core$IFn$_invoke$arity$1 ? g.cljs$core$IFn$_invoke$arity$1(G__10767) : g.call(null, G__10767);
        }();
        return f.cljs$core$IFn$_invoke$arity$1 ? f.cljs$core$IFn$_invoke$arity$1(G__10766) : f.call(null, G__10766);
      };
      var G__10779 = function(x, y, z, var_args) {
        var args = null;
        if (arguments.length > 3) {
          var G__10780__i = 0, G__10780__a = new Array(arguments.length - 3);
          while (G__10780__i < G__10780__a.length) {
            G__10780__a[G__10780__i] = arguments[G__10780__i + 3];
            ++G__10780__i;
          }
          args = new cljs.core.IndexedSeq(G__10780__a, 0);
        }
        return G__10779__delegate.call(this, x, y, z, args);
      };
      G__10779.cljs$lang$maxFixedArity = 3;
      G__10779.cljs$lang$applyTo = function(arglist__10781) {
        var x = cljs.core.first(arglist__10781);
        arglist__10781 = cljs.core.next(arglist__10781);
        var y = cljs.core.first(arglist__10781);
        arglist__10781 = cljs.core.next(arglist__10781);
        var z = cljs.core.first(arglist__10781);
        var args = cljs.core.rest(arglist__10781);
        return G__10779__delegate(x, y, z, args);
      };
      G__10779.cljs$core$IFn$_invoke$arity$variadic = G__10779__delegate;
      return G__10779;
    }();
    G__10778 = function(x, y, z, var_args) {
      var args = var_args;
      switch(arguments.length) {
        case 0:
          return G__10778__0.call(this);
        case 1:
          return G__10778__1.call(this, x);
        case 2:
          return G__10778__2.call(this, x, y);
        case 3:
          return G__10778__3.call(this, x, y, z);
        default:
          var G__10782 = null;
          if (arguments.length > 3) {
            var G__10783__i = 0, G__10783__a = new Array(arguments.length - 3);
            while (G__10783__i < G__10783__a.length) {
              G__10783__a[G__10783__i] = arguments[G__10783__i + 3];
              ++G__10783__i;
            }
            G__10782 = new cljs.core.IndexedSeq(G__10783__a, 0);
          }
          return G__10778__4.cljs$core$IFn$_invoke$arity$variadic(x, y, z, G__10782);
      }
      throw new Error("Invalid arity: " + (arguments.length - 1));
    };
    G__10778.cljs$lang$maxFixedArity = 3;
    G__10778.cljs$lang$applyTo = G__10778__4.cljs$lang$applyTo;
    G__10778.cljs$core$IFn$_invoke$arity$0 = G__10778__0;
    G__10778.cljs$core$IFn$_invoke$arity$1 = G__10778__1;
    G__10778.cljs$core$IFn$_invoke$arity$2 = G__10778__2;
    G__10778.cljs$core$IFn$_invoke$arity$3 = G__10778__3;
    G__10778.cljs$core$IFn$_invoke$arity$variadic = G__10778__4.cljs$core$IFn$_invoke$arity$variadic;
    return G__10778;
  }();
};
cljs.core.comp.cljs$core$IFn$_invoke$arity$variadic = function(f1, f2, f3, fs) {
  var fs__$1 = cljs.core.reverse(cljs.core.list_STAR_.cljs$core$IFn$_invoke$arity$4(f1, f2, f3, fs));
  return function(fs__$1) {
    return function() {
      var G__10784__delegate = function(args) {
        var ret = cljs.core.apply.cljs$core$IFn$_invoke$arity$2(cljs.core.first(fs__$1), args);
        var fs__$2 = cljs.core.next(fs__$1);
        while (true) {
          if (fs__$2) {
            var G__10785 = cljs.core.first(fs__$2).call(null, ret);
            var G__10786 = cljs.core.next(fs__$2);
            ret = G__10785;
            fs__$2 = G__10786;
            continue;
          } else {
            return ret;
          }
          break;
        }
      };
      var G__10784 = function(var_args) {
        var args = null;
        if (arguments.length > 0) {
          var G__10787__i = 0, G__10787__a = new Array(arguments.length - 0);
          while (G__10787__i < G__10787__a.length) {
            G__10787__a[G__10787__i] = arguments[G__10787__i + 0];
            ++G__10787__i;
          }
          args = new cljs.core.IndexedSeq(G__10787__a, 0);
        }
        return G__10784__delegate.call(this, args);
      };
      G__10784.cljs$lang$maxFixedArity = 0;
      G__10784.cljs$lang$applyTo = function(arglist__10788) {
        var args = cljs.core.seq(arglist__10788);
        return G__10784__delegate(args);
      };
      G__10784.cljs$core$IFn$_invoke$arity$variadic = G__10784__delegate;
      return G__10784;
    }();
  }(fs__$1);
};
cljs.core.comp.cljs$lang$applyTo = function(seq10747) {
  var G__10748 = cljs.core.first(seq10747);
  var seq10747__$1 = cljs.core.next(seq10747);
  var G__10749 = cljs.core.first(seq10747__$1);
  var seq10747__$2 = cljs.core.next(seq10747__$1);
  var G__10750 = cljs.core.first(seq10747__$2);
  var seq10747__$3 = cljs.core.next(seq10747__$2);
  return cljs.core.comp.cljs$core$IFn$_invoke$arity$variadic(G__10748, G__10749, G__10750, seq10747__$3);
};
cljs.core.comp.cljs$lang$maxFixedArity = 3;
cljs.core.partial = function cljs$core$partial(var_args) {
  var args10789 = [];
  var len__8373__auto___10797 = arguments.length;
  var i__8374__auto___10798 = 0;
  while (true) {
    if (i__8374__auto___10798 < len__8373__auto___10797) {
      args10789.push(arguments[i__8374__auto___10798]);
      var G__10799 = i__8374__auto___10798 + 1;
      i__8374__auto___10798 = G__10799;
      continue;
    } else {
    }
    break;
  }
  var G__10796 = args10789.length;
  switch(G__10796) {
    case 1:
      return cljs.core.partial.cljs$core$IFn$_invoke$arity$1(arguments[0]);
      break;
    case 2:
      return cljs.core.partial.cljs$core$IFn$_invoke$arity$2(arguments[0], arguments[1]);
      break;
    case 3:
      return cljs.core.partial.cljs$core$IFn$_invoke$arity$3(arguments[0], arguments[1], arguments[2]);
      break;
    case 4:
      return cljs.core.partial.cljs$core$IFn$_invoke$arity$4(arguments[0], arguments[1], arguments[2], arguments[3]);
      break;
    default:
      var argseq__8392__auto__ = new cljs.core.IndexedSeq(args10789.slice(4), 0, null);
      return cljs.core.partial.cljs$core$IFn$_invoke$arity$variadic(arguments[0], arguments[1], arguments[2], arguments[3], argseq__8392__auto__);
  }
};
cljs.core.partial.cljs$core$IFn$_invoke$arity$1 = function(f) {
  return f;
};
cljs.core.partial.cljs$core$IFn$_invoke$arity$2 = function(f, arg1) {
  return function() {
    var G__10801 = null;
    var G__10801__0 = function() {
      return f.cljs$core$IFn$_invoke$arity$1 ? f.cljs$core$IFn$_invoke$arity$1(arg1) : f.call(null, arg1);
    };
    var G__10801__1 = function(x) {
      return f.cljs$core$IFn$_invoke$arity$2 ? f.cljs$core$IFn$_invoke$arity$2(arg1, x) : f.call(null, arg1, x);
    };
    var G__10801__2 = function(x, y) {
      return f.cljs$core$IFn$_invoke$arity$3 ? f.cljs$core$IFn$_invoke$arity$3(arg1, x, y) : f.call(null, arg1, x, y);
    };
    var G__10801__3 = function(x, y, z) {
      return f.cljs$core$IFn$_invoke$arity$4 ? f.cljs$core$IFn$_invoke$arity$4(arg1, x, y, z) : f.call(null, arg1, x, y, z);
    };
    var G__10801__4 = function() {
      var G__10802__delegate = function(x, y, z, args) {
        return cljs.core.apply.cljs$core$IFn$_invoke$arity$variadic(f, arg1, x, y, z, cljs.core.array_seq([args], 0));
      };
      var G__10802 = function(x, y, z, var_args) {
        var args = null;
        if (arguments.length > 3) {
          var G__10803__i = 0, G__10803__a = new Array(arguments.length - 3);
          while (G__10803__i < G__10803__a.length) {
            G__10803__a[G__10803__i] = arguments[G__10803__i + 3];
            ++G__10803__i;
          }
          args = new cljs.core.IndexedSeq(G__10803__a, 0);
        }
        return G__10802__delegate.call(this, x, y, z, args);
      };
      G__10802.cljs$lang$maxFixedArity = 3;
      G__10802.cljs$lang$applyTo = function(arglist__10804) {
        var x = cljs.core.first(arglist__10804);
        arglist__10804 = cljs.core.next(arglist__10804);
        var y = cljs.core.first(arglist__10804);
        arglist__10804 = cljs.core.next(arglist__10804);
        var z = cljs.core.first(arglist__10804);
        var args = cljs.core.rest(arglist__10804);
        return G__10802__delegate(x, y, z, args);
      };
      G__10802.cljs$core$IFn$_invoke$arity$variadic = G__10802__delegate;
      return G__10802;
    }();
    G__10801 = function(x, y, z, var_args) {
      var args = var_args;
      switch(arguments.length) {
        case 0:
          return G__10801__0.call(this);
        case 1:
          return G__10801__1.call(this, x);
        case 2:
          return G__10801__2.call(this, x, y);
        case 3:
          return G__10801__3.call(this, x, y, z);
        default:
          var G__10805 = null;
          if (arguments.length > 3) {
            var G__10806__i = 0, G__10806__a = new Array(arguments.length - 3);
            while (G__10806__i < G__10806__a.length) {
              G__10806__a[G__10806__i] = arguments[G__10806__i + 3];
              ++G__10806__i;
            }
            G__10805 = new cljs.core.IndexedSeq(G__10806__a, 0);
          }
          return G__10801__4.cljs$core$IFn$_invoke$arity$variadic(x, y, z, G__10805);
      }
      throw new Error("Invalid arity: " + (arguments.length - 1));
    };
    G__10801.cljs$lang$maxFixedArity = 3;
    G__10801.cljs$lang$applyTo = G__10801__4.cljs$lang$applyTo;
    G__10801.cljs$core$IFn$_invoke$arity$0 = G__10801__0;
    G__10801.cljs$core$IFn$_invoke$arity$1 = G__10801__1;
    G__10801.cljs$core$IFn$_invoke$arity$2 = G__10801__2;
    G__10801.cljs$core$IFn$_invoke$arity$3 = G__10801__3;
    G__10801.cljs$core$IFn$_invoke$arity$variadic = G__10801__4.cljs$core$IFn$_invoke$arity$variadic;
    return G__10801;
  }();
};
cljs.core.partial.cljs$core$IFn$_invoke$arity$3 = function(f, arg1, arg2) {
  return function() {
    var G__10807 = null;
    var G__10807__0 = function() {
      return f.cljs$core$IFn$_invoke$arity$2 ? f.cljs$core$IFn$_invoke$arity$2(arg1, arg2) : f.call(null, arg1, arg2);
    };
    var G__10807__1 = function(x) {
      return f.cljs$core$IFn$_invoke$arity$3 ? f.cljs$core$IFn$_invoke$arity$3(arg1, arg2, x) : f.call(null, arg1, arg2, x);
    };
    var G__10807__2 = function(x, y) {
      return f.cljs$core$IFn$_invoke$arity$4 ? f.cljs$core$IFn$_invoke$arity$4(arg1, arg2, x, y) : f.call(null, arg1, arg2, x, y);
    };
    var G__10807__3 = function(x, y, z) {
      return f.cljs$core$IFn$_invoke$arity$5 ? f.cljs$core$IFn$_invoke$arity$5(arg1, arg2, x, y, z) : f.call(null, arg1, arg2, x, y, z);
    };
    var G__10807__4 = function() {
      var G__10808__delegate = function(x, y, z, args) {
        return cljs.core.apply.cljs$core$IFn$_invoke$arity$variadic(f, arg1, arg2, x, y, cljs.core.array_seq([z, args], 0));
      };
      var G__10808 = function(x, y, z, var_args) {
        var args = null;
        if (arguments.length > 3) {
          var G__10809__i = 0, G__10809__a = new Array(arguments.length - 3);
          while (G__10809__i < G__10809__a.length) {
            G__10809__a[G__10809__i] = arguments[G__10809__i + 3];
            ++G__10809__i;
          }
          args = new cljs.core.IndexedSeq(G__10809__a, 0);
        }
        return G__10808__delegate.call(this, x, y, z, args);
      };
      G__10808.cljs$lang$maxFixedArity = 3;
      G__10808.cljs$lang$applyTo = function(arglist__10810) {
        var x = cljs.core.first(arglist__10810);
        arglist__10810 = cljs.core.next(arglist__10810);
        var y = cljs.core.first(arglist__10810);
        arglist__10810 = cljs.core.next(arglist__10810);
        var z = cljs.core.first(arglist__10810);
        var args = cljs.core.rest(arglist__10810);
        return G__10808__delegate(x, y, z, args);
      };
      G__10808.cljs$core$IFn$_invoke$arity$variadic = G__10808__delegate;
      return G__10808;
    }();
    G__10807 = function(x, y, z, var_args) {
      var args = var_args;
      switch(arguments.length) {
        case 0:
          return G__10807__0.call(this);
        case 1:
          return G__10807__1.call(this, x);
        case 2:
          return G__10807__2.call(this, x, y);
        case 3:
          return G__10807__3.call(this, x, y, z);
        default:
          var G__10811 = null;
          if (arguments.length > 3) {
            var G__10812__i = 0, G__10812__a = new Array(arguments.length - 3);
            while (G__10812__i < G__10812__a.length) {
              G__10812__a[G__10812__i] = arguments[G__10812__i + 3];
              ++G__10812__i;
            }
            G__10811 = new cljs.core.IndexedSeq(G__10812__a, 0);
          }
          return G__10807__4.cljs$core$IFn$_invoke$arity$variadic(x, y, z, G__10811);
      }
      throw new Error("Invalid arity: " + (arguments.length - 1));
    };
    G__10807.cljs$lang$maxFixedArity = 3;
    G__10807.cljs$lang$applyTo = G__10807__4.cljs$lang$applyTo;
    G__10807.cljs$core$IFn$_invoke$arity$0 = G__10807__0;
    G__10807.cljs$core$IFn$_invoke$arity$1 = G__10807__1;
    G__10807.cljs$core$IFn$_invoke$arity$2 = G__10807__2;
    G__10807.cljs$core$IFn$_invoke$arity$3 = G__10807__3;
    G__10807.cljs$core$IFn$_invoke$arity$variadic = G__10807__4.cljs$core$IFn$_invoke$arity$variadic;
    return G__10807;
  }();
};
cljs.core.partial.cljs$core$IFn$_invoke$arity$4 = function(f, arg1, arg2, arg3) {
  return function() {
    var G__10813 = null;
    var G__10813__0 = function() {
      return f.cljs$core$IFn$_invoke$arity$3 ? f.cljs$core$IFn$_invoke$arity$3(arg1, arg2, arg3) : f.call(null, arg1, arg2, arg3);
    };
    var G__10813__1 = function(x) {
      return f.cljs$core$IFn$_invoke$arity$4 ? f.cljs$core$IFn$_invoke$arity$4(arg1, arg2, arg3, x) : f.call(null, arg1, arg2, arg3, x);
    };
    var G__10813__2 = function(x, y) {
      return f.cljs$core$IFn$_invoke$arity$5 ? f.cljs$core$IFn$_invoke$arity$5(arg1, arg2, arg3, x, y) : f.call(null, arg1, arg2, arg3, x, y);
    };
    var G__10813__3 = function(x, y, z) {
      return f.cljs$core$IFn$_invoke$arity$6 ? f.cljs$core$IFn$_invoke$arity$6(arg1, arg2, arg3, x, y, z) : f.call(null, arg1, arg2, arg3, x, y, z);
    };
    var G__10813__4 = function() {
      var G__10814__delegate = function(x, y, z, args) {
        return cljs.core.apply.cljs$core$IFn$_invoke$arity$variadic(f, arg1, arg2, arg3, x, cljs.core.array_seq([y, z, args], 0));
      };
      var G__10814 = function(x, y, z, var_args) {
        var args = null;
        if (arguments.length > 3) {
          var G__10815__i = 0, G__10815__a = new Array(arguments.length - 3);
          while (G__10815__i < G__10815__a.length) {
            G__10815__a[G__10815__i] = arguments[G__10815__i + 3];
            ++G__10815__i;
          }
          args = new cljs.core.IndexedSeq(G__10815__a, 0);
        }
        return G__10814__delegate.call(this, x, y, z, args);
      };
      G__10814.cljs$lang$maxFixedArity = 3;
      G__10814.cljs$lang$applyTo = function(arglist__10816) {
        var x = cljs.core.first(arglist__10816);
        arglist__10816 = cljs.core.next(arglist__10816);
        var y = cljs.core.first(arglist__10816);
        arglist__10816 = cljs.core.next(arglist__10816);
        var z = cljs.core.first(arglist__10816);
        var args = cljs.core.rest(arglist__10816);
        return G__10814__delegate(x, y, z, args);
      };
      G__10814.cljs$core$IFn$_invoke$arity$variadic = G__10814__delegate;
      return G__10814;
    }();
    G__10813 = function(x, y, z, var_args) {
      var args = var_args;
      switch(arguments.length) {
        case 0:
          return G__10813__0.call(this);
        case 1:
          return G__10813__1.call(this, x);
        case 2:
          return G__10813__2.call(this, x, y);
        case 3:
          return G__10813__3.call(this, x, y, z);
        default:
          var G__10817 = null;
          if (arguments.length > 3) {
            var G__10818__i = 0, G__10818__a = new Array(arguments.length - 3);
            while (G__10818__i < G__10818__a.length) {
              G__10818__a[G__10818__i] = arguments[G__10818__i + 3];
              ++G__10818__i;
            }
            G__10817 = new cljs.core.IndexedSeq(G__10818__a, 0);
          }
          return G__10813__4.cljs$core$IFn$_invoke$arity$variadic(x, y, z, G__10817);
      }
      throw new Error("Invalid arity: " + (arguments.length - 1));
    };
    G__10813.cljs$lang$maxFixedArity = 3;
    G__10813.cljs$lang$applyTo = G__10813__4.cljs$lang$applyTo;
    G__10813.cljs$core$IFn$_invoke$arity$0 = G__10813__0;
    G__10813.cljs$core$IFn$_invoke$arity$1 = G__10813__1;
    G__10813.cljs$core$IFn$_invoke$arity$2 = G__10813__2;
    G__10813.cljs$core$IFn$_invoke$arity$3 = G__10813__3;
    G__10813.cljs$core$IFn$_invoke$arity$variadic = G__10813__4.cljs$core$IFn$_invoke$arity$variadic;
    return G__10813;
  }();
};
cljs.core.partial.cljs$core$IFn$_invoke$arity$variadic = function(f, arg1, arg2, arg3, more) {
  return function() {
    var G__10819__delegate = function(args) {
      return cljs.core.apply.cljs$core$IFn$_invoke$arity$5(f, arg1, arg2, arg3, cljs.core.concat.cljs$core$IFn$_invoke$arity$2(more, args));
    };
    var G__10819 = function(var_args) {
      var args = null;
      if (arguments.length > 0) {
        var G__10820__i = 0, G__10820__a = new Array(arguments.length - 0);
        while (G__10820__i < G__10820__a.length) {
          G__10820__a[G__10820__i] = arguments[G__10820__i + 0];
          ++G__10820__i;
        }
        args = new cljs.core.IndexedSeq(G__10820__a, 0);
      }
      return G__10819__delegate.call(this, args);
    };
    G__10819.cljs$lang$maxFixedArity = 0;
    G__10819.cljs$lang$applyTo = function(arglist__10821) {
      var args = cljs.core.seq(arglist__10821);
      return G__10819__delegate(args);
    };
    G__10819.cljs$core$IFn$_invoke$arity$variadic = G__10819__delegate;
    return G__10819;
  }();
};
cljs.core.partial.cljs$lang$applyTo = function(seq10790) {
  var G__10791 = cljs.core.first(seq10790);
  var seq10790__$1 = cljs.core.next(seq10790);
  var G__10792 = cljs.core.first(seq10790__$1);
  var seq10790__$2 = cljs.core.next(seq10790__$1);
  var G__10793 = cljs.core.first(seq10790__$2);
  var seq10790__$3 = cljs.core.next(seq10790__$2);
  var G__10794 = cljs.core.first(seq10790__$3);
  var seq10790__$4 = cljs.core.next(seq10790__$3);
  return cljs.core.partial.cljs$core$IFn$_invoke$arity$variadic(G__10791, G__10792, G__10793, G__10794, seq10790__$4);
};
cljs.core.partial.cljs$lang$maxFixedArity = 4;
cljs.core.fnil = function cljs$core$fnil(var_args) {
  var args10822 = [];
  var len__8373__auto___10841 = arguments.length;
  var i__8374__auto___10842 = 0;
  while (true) {
    if (i__8374__auto___10842 < len__8373__auto___10841) {
      args10822.push(arguments[i__8374__auto___10842]);
      var G__10843 = i__8374__auto___10842 + 1;
      i__8374__auto___10842 = G__10843;
      continue;
    } else {
    }
    break;
  }
  var G__10824 = args10822.length;
  switch(G__10824) {
    case 2:
      return cljs.core.fnil.cljs$core$IFn$_invoke$arity$2(arguments[0], arguments[1]);
      break;
    case 3:
      return cljs.core.fnil.cljs$core$IFn$_invoke$arity$3(arguments[0], arguments[1], arguments[2]);
      break;
    case 4:
      return cljs.core.fnil.cljs$core$IFn$_invoke$arity$4(arguments[0], arguments[1], arguments[2], arguments[3]);
      break;
    default:
      throw new Error([cljs.core.str.cljs$core$IFn$_invoke$arity$1("Invalid arity: "), cljs.core.str.cljs$core$IFn$_invoke$arity$1(args10822.length)].join(""));
  }
};
cljs.core.fnil.cljs$core$IFn$_invoke$arity$2 = function(f, x) {
  return function() {
    var G__10845 = null;
    var G__10845__1 = function(a) {
      var G__10825 = a == null ? x : a;
      return f.cljs$core$IFn$_invoke$arity$1 ? f.cljs$core$IFn$_invoke$arity$1(G__10825) : f.call(null, G__10825);
    };
    var G__10845__2 = function(a, b) {
      var G__10826 = a == null ? x : a;
      var G__10827 = b;
      return f.cljs$core$IFn$_invoke$arity$2 ? f.cljs$core$IFn$_invoke$arity$2(G__10826, G__10827) : f.call(null, G__10826, G__10827);
    };
    var G__10845__3 = function(a, b, c) {
      var G__10828 = a == null ? x : a;
      var G__10829 = b;
      var G__10830 = c;
      return f.cljs$core$IFn$_invoke$arity$3 ? f.cljs$core$IFn$_invoke$arity$3(G__10828, G__10829, G__10830) : f.call(null, G__10828, G__10829, G__10830);
    };
    var G__10845__4 = function() {
      var G__10846__delegate = function(a, b, c, ds) {
        return cljs.core.apply.cljs$core$IFn$_invoke$arity$5(f, a == null ? x : a, b, c, ds);
      };
      var G__10846 = function(a, b, c, var_args) {
        var ds = null;
        if (arguments.length > 3) {
          var G__10847__i = 0, G__10847__a = new Array(arguments.length - 3);
          while (G__10847__i < G__10847__a.length) {
            G__10847__a[G__10847__i] = arguments[G__10847__i + 3];
            ++G__10847__i;
          }
          ds = new cljs.core.IndexedSeq(G__10847__a, 0);
        }
        return G__10846__delegate.call(this, a, b, c, ds);
      };
      G__10846.cljs$lang$maxFixedArity = 3;
      G__10846.cljs$lang$applyTo = function(arglist__10848) {
        var a = cljs.core.first(arglist__10848);
        arglist__10848 = cljs.core.next(arglist__10848);
        var b = cljs.core.first(arglist__10848);
        arglist__10848 = cljs.core.next(arglist__10848);
        var c = cljs.core.first(arglist__10848);
        var ds = cljs.core.rest(arglist__10848);
        return G__10846__delegate(a, b, c, ds);
      };
      G__10846.cljs$core$IFn$_invoke$arity$variadic = G__10846__delegate;
      return G__10846;
    }();
    G__10845 = function(a, b, c, var_args) {
      var ds = var_args;
      switch(arguments.length) {
        case 1:
          return G__10845__1.call(this, a);
        case 2:
          return G__10845__2.call(this, a, b);
        case 3:
          return G__10845__3.call(this, a, b, c);
        default:
          var G__10849 = null;
          if (arguments.length > 3) {
            var G__10850__i = 0, G__10850__a = new Array(arguments.length - 3);
            while (G__10850__i < G__10850__a.length) {
              G__10850__a[G__10850__i] = arguments[G__10850__i + 3];
              ++G__10850__i;
            }
            G__10849 = new cljs.core.IndexedSeq(G__10850__a, 0);
          }
          return G__10845__4.cljs$core$IFn$_invoke$arity$variadic(a, b, c, G__10849);
      }
      throw new Error("Invalid arity: " + (arguments.length - 1));
    };
    G__10845.cljs$lang$maxFixedArity = 3;
    G__10845.cljs$lang$applyTo = G__10845__4.cljs$lang$applyTo;
    G__10845.cljs$core$IFn$_invoke$arity$1 = G__10845__1;
    G__10845.cljs$core$IFn$_invoke$arity$2 = G__10845__2;
    G__10845.cljs$core$IFn$_invoke$arity$3 = G__10845__3;
    G__10845.cljs$core$IFn$_invoke$arity$variadic = G__10845__4.cljs$core$IFn$_invoke$arity$variadic;
    return G__10845;
  }();
};
cljs.core.fnil.cljs$core$IFn$_invoke$arity$3 = function(f, x, y) {
  return function() {
    var G__10851 = null;
    var G__10851__2 = function(a, b) {
      var G__10831 = a == null ? x : a;
      var G__10832 = b == null ? y : b;
      return f.cljs$core$IFn$_invoke$arity$2 ? f.cljs$core$IFn$_invoke$arity$2(G__10831, G__10832) : f.call(null, G__10831, G__10832);
    };
    var G__10851__3 = function(a, b, c) {
      var G__10833 = a == null ? x : a;
      var G__10834 = b == null ? y : b;
      var G__10835 = c;
      return f.cljs$core$IFn$_invoke$arity$3 ? f.cljs$core$IFn$_invoke$arity$3(G__10833, G__10834, G__10835) : f.call(null, G__10833, G__10834, G__10835);
    };
    var G__10851__4 = function() {
      var G__10852__delegate = function(a, b, c, ds) {
        return cljs.core.apply.cljs$core$IFn$_invoke$arity$5(f, a == null ? x : a, b == null ? y : b, c, ds);
      };
      var G__10852 = function(a, b, c, var_args) {
        var ds = null;
        if (arguments.length > 3) {
          var G__10853__i = 0, G__10853__a = new Array(arguments.length - 3);
          while (G__10853__i < G__10853__a.length) {
            G__10853__a[G__10853__i] = arguments[G__10853__i + 3];
            ++G__10853__i;
          }
          ds = new cljs.core.IndexedSeq(G__10853__a, 0);
        }
        return G__10852__delegate.call(this, a, b, c, ds);
      };
      G__10852.cljs$lang$maxFixedArity = 3;
      G__10852.cljs$lang$applyTo = function(arglist__10854) {
        var a = cljs.core.first(arglist__10854);
        arglist__10854 = cljs.core.next(arglist__10854);
        var b = cljs.core.first(arglist__10854);
        arglist__10854 = cljs.core.next(arglist__10854);
        var c = cljs.core.first(arglist__10854);
        var ds = cljs.core.rest(arglist__10854);
        return G__10852__delegate(a, b, c, ds);
      };
      G__10852.cljs$core$IFn$_invoke$arity$variadic = G__10852__delegate;
      return G__10852;
    }();
    G__10851 = function(a, b, c, var_args) {
      var ds = var_args;
      switch(arguments.length) {
        case 2:
          return G__10851__2.call(this, a, b);
        case 3:
          return G__10851__3.call(this, a, b, c);
        default:
          var G__10855 = null;
          if (arguments.length > 3) {
            var G__10856__i = 0, G__10856__a = new Array(arguments.length - 3);
            while (G__10856__i < G__10856__a.length) {
              G__10856__a[G__10856__i] = arguments[G__10856__i + 3];
              ++G__10856__i;
            }
            G__10855 = new cljs.core.IndexedSeq(G__10856__a, 0);
          }
          return G__10851__4.cljs$core$IFn$_invoke$arity$variadic(a, b, c, G__10855);
      }
      throw new Error("Invalid arity: " + (arguments.length - 1));
    };
    G__10851.cljs$lang$maxFixedArity = 3;
    G__10851.cljs$lang$applyTo = G__10851__4.cljs$lang$applyTo;
    G__10851.cljs$core$IFn$_invoke$arity$2 = G__10851__2;
    G__10851.cljs$core$IFn$_invoke$arity$3 = G__10851__3;
    G__10851.cljs$core$IFn$_invoke$arity$variadic = G__10851__4.cljs$core$IFn$_invoke$arity$variadic;
    return G__10851;
  }();
};
cljs.core.fnil.cljs$core$IFn$_invoke$arity$4 = function(f, x, y, z) {
  return function() {
    var G__10857 = null;
    var G__10857__2 = function(a, b) {
      var G__10836 = a == null ? x : a;
      var G__10837 = b == null ? y : b;
      return f.cljs$core$IFn$_invoke$arity$2 ? f.cljs$core$IFn$_invoke$arity$2(G__10836, G__10837) : f.call(null, G__10836, G__10837);
    };
    var G__10857__3 = function(a, b, c) {
      var G__10838 = a == null ? x : a;
      var G__10839 = b == null ? y : b;
      var G__10840 = c == null ? z : c;
      return f.cljs$core$IFn$_invoke$arity$3 ? f.cljs$core$IFn$_invoke$arity$3(G__10838, G__10839, G__10840) : f.call(null, G__10838, G__10839, G__10840);
    };
    var G__10857__4 = function() {
      var G__10858__delegate = function(a, b, c, ds) {
        return cljs.core.apply.cljs$core$IFn$_invoke$arity$5(f, a == null ? x : a, b == null ? y : b, c == null ? z : c, ds);
      };
      var G__10858 = function(a, b, c, var_args) {
        var ds = null;
        if (arguments.length > 3) {
          var G__10859__i = 0, G__10859__a = new Array(arguments.length - 3);
          while (G__10859__i < G__10859__a.length) {
            G__10859__a[G__10859__i] = arguments[G__10859__i + 3];
            ++G__10859__i;
          }
          ds = new cljs.core.IndexedSeq(G__10859__a, 0);
        }
        return G__10858__delegate.call(this, a, b, c, ds);
      };
      G__10858.cljs$lang$maxFixedArity = 3;
      G__10858.cljs$lang$applyTo = function(arglist__10860) {
        var a = cljs.core.first(arglist__10860);
        arglist__10860 = cljs.core.next(arglist__10860);
        var b = cljs.core.first(arglist__10860);
        arglist__10860 = cljs.core.next(arglist__10860);
        var c = cljs.core.first(arglist__10860);
        var ds = cljs.core.rest(arglist__10860);
        return G__10858__delegate(a, b, c, ds);
      };
      G__10858.cljs$core$IFn$_invoke$arity$variadic = G__10858__delegate;
      return G__10858;
    }();
    G__10857 = function(a, b, c, var_args) {
      var ds = var_args;
      switch(arguments.length) {
        case 2:
          return G__10857__2.call(this, a, b);
        case 3:
          return G__10857__3.call(this, a, b, c);
        default:
          var G__10861 = null;
          if (arguments.length > 3) {
            var G__10862__i = 0, G__10862__a = new Array(arguments.length - 3);
            while (G__10862__i < G__10862__a.length) {
              G__10862__a[G__10862__i] = arguments[G__10862__i + 3];
              ++G__10862__i;
            }
            G__10861 = new cljs.core.IndexedSeq(G__10862__a, 0);
          }
          return G__10857__4.cljs$core$IFn$_invoke$arity$variadic(a, b, c, G__10861);
      }
      throw new Error("Invalid arity: " + (arguments.length - 1));
    };
    G__10857.cljs$lang$maxFixedArity = 3;
    G__10857.cljs$lang$applyTo = G__10857__4.cljs$lang$applyTo;
    G__10857.cljs$core$IFn$_invoke$arity$2 = G__10857__2;
    G__10857.cljs$core$IFn$_invoke$arity$3 = G__10857__3;
    G__10857.cljs$core$IFn$_invoke$arity$variadic = G__10857__4.cljs$core$IFn$_invoke$arity$variadic;
    return G__10857;
  }();
};
cljs.core.fnil.cljs$lang$maxFixedArity = 4;
cljs.core.map_indexed = function cljs$core$map_indexed(var_args) {
  var args10863 = [];
  var len__8373__auto___10894 = arguments.length;
  var i__8374__auto___10895 = 0;
  while (true) {
    if (i__8374__auto___10895 < len__8373__auto___10894) {
      args10863.push(arguments[i__8374__auto___10895]);
      var G__10896 = i__8374__auto___10895 + 1;
      i__8374__auto___10895 = G__10896;
      continue;
    } else {
    }
    break;
  }
  var G__10865 = args10863.length;
  switch(G__10865) {
    case 1:
      return cljs.core.map_indexed.cljs$core$IFn$_invoke$arity$1(arguments[0]);
      break;
    case 2:
      return cljs.core.map_indexed.cljs$core$IFn$_invoke$arity$2(arguments[0], arguments[1]);
      break;
    default:
      throw new Error([cljs.core.str.cljs$core$IFn$_invoke$arity$1("Invalid arity: "), cljs.core.str.cljs$core$IFn$_invoke$arity$1(args10863.length)].join(""));
  }
};
cljs.core.map_indexed.cljs$core$IFn$_invoke$arity$1 = function(f) {
  return function(rf) {
    var i = cljs.core.volatile_BANG_.cljs$core$IFn$_invoke$arity$1 ? cljs.core.volatile_BANG_.cljs$core$IFn$_invoke$arity$1(-1) : cljs.core.volatile_BANG_.call(null, -1);
    return function(i) {
      return function() {
        var G__10898 = null;
        var G__10898__0 = function() {
          return rf.cljs$core$IFn$_invoke$arity$0 ? rf.cljs$core$IFn$_invoke$arity$0() : rf.call(null);
        };
        var G__10898__1 = function(result) {
          return rf.cljs$core$IFn$_invoke$arity$1 ? rf.cljs$core$IFn$_invoke$arity$1(result) : rf.call(null, result);
        };
        var G__10898__2 = function(result, input) {
          var G__10866 = result;
          var G__10867 = function() {
            var G__10868 = cljs.core._vreset_BANG_(i, cljs.core._deref(i) + 1);
            var G__10869 = input;
            return f.cljs$core$IFn$_invoke$arity$2 ? f.cljs$core$IFn$_invoke$arity$2(G__10868, G__10869) : f.call(null, G__10868, G__10869);
          }();
          return rf.cljs$core$IFn$_invoke$arity$2 ? rf.cljs$core$IFn$_invoke$arity$2(G__10866, G__10867) : rf.call(null, G__10866, G__10867);
        };
        G__10898 = function(result, input) {
          switch(arguments.length) {
            case 0:
              return G__10898__0.call(this);
            case 1:
              return G__10898__1.call(this, result);
            case 2:
              return G__10898__2.call(this, result, input);
          }
          throw new Error("Invalid arity: " + (arguments.length - 1));
        };
        G__10898.cljs$core$IFn$_invoke$arity$0 = G__10898__0;
        G__10898.cljs$core$IFn$_invoke$arity$1 = G__10898__1;
        G__10898.cljs$core$IFn$_invoke$arity$2 = G__10898__2;
        return G__10898;
      }();
    }(i);
  };
};
cljs.core.map_indexed.cljs$core$IFn$_invoke$arity$2 = function(f, coll) {
  var mapi = function cljs$core$mapi(idx, coll__$1) {
    return new cljs.core.LazySeq(null, function() {
      var temp__4657__auto__ = cljs.core.seq(coll__$1);
      if (temp__4657__auto__) {
        var s = temp__4657__auto__;
        if (cljs.core.chunked_seq_QMARK_(s)) {
          var c = cljs.core.chunk_first(s);
          var size = cljs.core.count(c);
          var b = cljs.core.chunk_buffer(size);
          var n__8183__auto___10899 = size;
          var i_10900 = 0;
          while (true) {
            if (i_10900 < n__8183__auto___10899) {
              cljs.core.chunk_append(b, function() {
                var G__10890 = idx + i_10900;
                var G__10891 = cljs.core._nth.cljs$core$IFn$_invoke$arity$2(c, i_10900);
                return f.cljs$core$IFn$_invoke$arity$2 ? f.cljs$core$IFn$_invoke$arity$2(G__10890, G__10891) : f.call(null, G__10890, G__10891);
              }());
              var G__10901 = i_10900 + 1;
              i_10900 = G__10901;
              continue;
            } else {
            }
            break;
          }
          return cljs.core.chunk_cons(cljs.core.chunk(b), cljs$core$mapi(idx + size, cljs.core.chunk_rest(s)));
        } else {
          return cljs.core.cons(function() {
            var G__10892 = idx;
            var G__10893 = cljs.core.first(s);
            return f.cljs$core$IFn$_invoke$arity$2 ? f.cljs$core$IFn$_invoke$arity$2(G__10892, G__10893) : f.call(null, G__10892, G__10893);
          }(), cljs$core$mapi(idx + 1, cljs.core.rest(s)));
        }
      } else {
        return null;
      }
    }, null, null);
  };
  return mapi(0, coll);
};
cljs.core.map_indexed.cljs$lang$maxFixedArity = 2;
cljs.core.keep = function cljs$core$keep(var_args) {
  var args10902 = [];
  var len__8373__auto___10907 = arguments.length;
  var i__8374__auto___10908 = 0;
  while (true) {
    if (i__8374__auto___10908 < len__8373__auto___10907) {
      args10902.push(arguments[i__8374__auto___10908]);
      var G__10909 = i__8374__auto___10908 + 1;
      i__8374__auto___10908 = G__10909;
      continue;
    } else {
    }
    break;
  }
  var G__10904 = args10902.length;
  switch(G__10904) {
    case 1:
      return cljs.core.keep.cljs$core$IFn$_invoke$arity$1(arguments[0]);
      break;
    case 2:
      return cljs.core.keep.cljs$core$IFn$_invoke$arity$2(arguments[0], arguments[1]);
      break;
    default:
      throw new Error([cljs.core.str.cljs$core$IFn$_invoke$arity$1("Invalid arity: "), cljs.core.str.cljs$core$IFn$_invoke$arity$1(args10902.length)].join(""));
  }
};
cljs.core.keep.cljs$core$IFn$_invoke$arity$1 = function(f) {
  return function(rf) {
    return function() {
      var G__10911 = null;
      var G__10911__0 = function() {
        return rf.cljs$core$IFn$_invoke$arity$0 ? rf.cljs$core$IFn$_invoke$arity$0() : rf.call(null);
      };
      var G__10911__1 = function(result) {
        return rf.cljs$core$IFn$_invoke$arity$1 ? rf.cljs$core$IFn$_invoke$arity$1(result) : rf.call(null, result);
      };
      var G__10911__2 = function(result, input) {
        var v = f.cljs$core$IFn$_invoke$arity$1 ? f.cljs$core$IFn$_invoke$arity$1(input) : f.call(null, input);
        if (v == null) {
          return result;
        } else {
          return rf.cljs$core$IFn$_invoke$arity$2 ? rf.cljs$core$IFn$_invoke$arity$2(result, v) : rf.call(null, result, v);
        }
      };
      G__10911 = function(result, input) {
        switch(arguments.length) {
          case 0:
            return G__10911__0.call(this);
          case 1:
            return G__10911__1.call(this, result);
          case 2:
            return G__10911__2.call(this, result, input);
        }
        throw new Error("Invalid arity: " + (arguments.length - 1));
      };
      G__10911.cljs$core$IFn$_invoke$arity$0 = G__10911__0;
      G__10911.cljs$core$IFn$_invoke$arity$1 = G__10911__1;
      G__10911.cljs$core$IFn$_invoke$arity$2 = G__10911__2;
      return G__10911;
    }();
  };
};
cljs.core.keep.cljs$core$IFn$_invoke$arity$2 = function(f, coll) {
  return new cljs.core.LazySeq(null, function() {
    var temp__4657__auto__ = cljs.core.seq(coll);
    if (temp__4657__auto__) {
      var s = temp__4657__auto__;
      if (cljs.core.chunked_seq_QMARK_(s)) {
        var c = cljs.core.chunk_first(s);
        var size = cljs.core.count(c);
        var b = cljs.core.chunk_buffer(size);
        var n__8183__auto___10912 = size;
        var i_10913 = 0;
        while (true) {
          if (i_10913 < n__8183__auto___10912) {
            var x_10914 = function() {
              var G__10905 = cljs.core._nth.cljs$core$IFn$_invoke$arity$2(c, i_10913);
              return f.cljs$core$IFn$_invoke$arity$1 ? f.cljs$core$IFn$_invoke$arity$1(G__10905) : f.call(null, G__10905);
            }();
            if (x_10914 == null) {
            } else {
              cljs.core.chunk_append(b, x_10914);
            }
            var G__10915 = i_10913 + 1;
            i_10913 = G__10915;
            continue;
          } else {
          }
          break;
        }
        return cljs.core.chunk_cons(cljs.core.chunk(b), cljs.core.keep.cljs$core$IFn$_invoke$arity$2(f, cljs.core.chunk_rest(s)));
      } else {
        var x = function() {
          var G__10906 = cljs.core.first(s);
          return f.cljs$core$IFn$_invoke$arity$1 ? f.cljs$core$IFn$_invoke$arity$1(G__10906) : f.call(null, G__10906);
        }();
        if (x == null) {
          return cljs.core.keep.cljs$core$IFn$_invoke$arity$2(f, cljs.core.rest(s));
        } else {
          return cljs.core.cons(x, cljs.core.keep.cljs$core$IFn$_invoke$arity$2(f, cljs.core.rest(s)));
        }
      }
    } else {
      return null;
    }
  }, null, null);
};
cljs.core.keep.cljs$lang$maxFixedArity = 2;
cljs.core.Atom = function(state, meta, validator, watches) {
  this.state = state;
  this.meta = meta;
  this.validator = validator;
  this.watches = watches;
  this.cljs$lang$protocol_mask$partition1$ = 16386;
  this.cljs$lang$protocol_mask$partition0$ = 6455296;
};
cljs.core.Atom.prototype.equiv = function(other) {
  var self__ = this;
  var this$ = this;
  return this$.cljs$core$IEquiv$_equiv$arity$2(null, other);
};
cljs.core.Atom.prototype.cljs$core$IEquiv$_equiv$arity$2 = function(o, other) {
  var self__ = this;
  var o__$1 = this;
  return o__$1 === other;
};
cljs.core.Atom.prototype.cljs$core$IDeref$_deref$arity$1 = function(_) {
  var self__ = this;
  var ___$1 = this;
  return self__.state;
};
cljs.core.Atom.prototype.cljs$core$IMeta$_meta$arity$1 = function(_) {
  var self__ = this;
  var ___$1 = this;
  return self__.meta;
};
cljs.core.Atom.prototype.cljs$core$IWatchable$_notify_watches$arity$3 = function(this$, oldval, newval) {
  var self__ = this;
  var this$__$1 = this;
  var seq__10916 = cljs.core.seq(self__.watches);
  var chunk__10917 = null;
  var count__10918 = 0;
  var i__10919 = 0;
  while (true) {
    if (i__10919 < count__10918) {
      var vec__10920 = chunk__10917.cljs$core$IIndexed$_nth$arity$2(null, i__10919);
      var key = cljs.core.nth.cljs$core$IFn$_invoke$arity$3(vec__10920, 0, null);
      var f = cljs.core.nth.cljs$core$IFn$_invoke$arity$3(vec__10920, 1, null);
      f.cljs$core$IFn$_invoke$arity$4 ? f.cljs$core$IFn$_invoke$arity$4(key, this$__$1, oldval, newval) : f.call(null, key, this$__$1, oldval, newval);
      var G__10926 = seq__10916;
      var G__10927 = chunk__10917;
      var G__10928 = count__10918;
      var G__10929 = i__10919 + 1;
      seq__10916 = G__10926;
      chunk__10917 = G__10927;
      count__10918 = G__10928;
      i__10919 = G__10929;
      continue;
    } else {
      var temp__4657__auto__ = cljs.core.seq(seq__10916);
      if (temp__4657__auto__) {
        var seq__10916__$1 = temp__4657__auto__;
        if (cljs.core.chunked_seq_QMARK_(seq__10916__$1)) {
          var c__8079__auto__ = cljs.core.chunk_first(seq__10916__$1);
          var G__10930 = cljs.core.chunk_rest(seq__10916__$1);
          var G__10931 = c__8079__auto__;
          var G__10932 = cljs.core.count(c__8079__auto__);
          var G__10933 = 0;
          seq__10916 = G__10930;
          chunk__10917 = G__10931;
          count__10918 = G__10932;
          i__10919 = G__10933;
          continue;
        } else {
          var vec__10923 = cljs.core.first(seq__10916__$1);
          var key = cljs.core.nth.cljs$core$IFn$_invoke$arity$3(vec__10923, 0, null);
          var f = cljs.core.nth.cljs$core$IFn$_invoke$arity$3(vec__10923, 1, null);
          f.cljs$core$IFn$_invoke$arity$4 ? f.cljs$core$IFn$_invoke$arity$4(key, this$__$1, oldval, newval) : f.call(null, key, this$__$1, oldval, newval);
          var G__10934 = cljs.core.next(seq__10916__$1);
          var G__10935 = null;
          var G__10936 = 0;
          var G__10937 = 0;
          seq__10916 = G__10934;
          chunk__10917 = G__10935;
          count__10918 = G__10936;
          i__10919 = G__10937;
          continue;
        }
      } else {
        return null;
      }
    }
    break;
  }
};
cljs.core.Atom.prototype.cljs$core$IWatchable$_add_watch$arity$3 = function(this$, key, f) {
  var self__ = this;
  var this$__$1 = this;
  this$__$1.watches = cljs.core.assoc.cljs$core$IFn$_invoke$arity$3(self__.watches, key, f);
  return this$__$1;
};
cljs.core.Atom.prototype.cljs$core$IWatchable$_remove_watch$arity$2 = function(this$, key) {
  var self__ = this;
  var this$__$1 = this;
  return this$__$1.watches = cljs.core.dissoc.cljs$core$IFn$_invoke$arity$2(self__.watches, key);
};
cljs.core.Atom.prototype.cljs$core$IHash$_hash$arity$1 = function(this$) {
  var self__ = this;
  var this$__$1 = this;
  return goog.getUid(this$__$1);
};
cljs.core.Atom.getBasis = function() {
  return new cljs.core.PersistentVector(null, 4, 5, cljs.core.PersistentVector.EMPTY_NODE, [new cljs.core.Symbol(null, "state", "state", -348086572, null), new cljs.core.Symbol(null, "meta", "meta", -1154898805, null), new cljs.core.Symbol(null, "validator", "validator", -325659154, null), new cljs.core.Symbol(null, "watches", "watches", 1367433992, null)], null);
};
cljs.core.Atom.cljs$lang$type = true;
cljs.core.Atom.cljs$lang$ctorStr = "cljs.core/Atom";
cljs.core.Atom.cljs$lang$ctorPrWriter = function(this__7871__auto__, writer__7872__auto__, opt__7873__auto__) {
  return cljs.core._write(writer__7872__auto__, "cljs.core/Atom");
};
cljs.core.__GT_Atom = function cljs$core$__GT_Atom(state, meta, validator, watches) {
  return new cljs.core.Atom(state, meta, validator, watches);
};
cljs.core.atom = function cljs$core$atom(var_args) {
  var args10938 = [];
  var len__8373__auto___10946 = arguments.length;
  var i__8374__auto___10947 = 0;
  while (true) {
    if (i__8374__auto___10947 < len__8373__auto___10946) {
      args10938.push(arguments[i__8374__auto___10947]);
      var G__10948 = i__8374__auto___10947 + 1;
      i__8374__auto___10947 = G__10948;
      continue;
    } else {
    }
    break;
  }
  var G__10942 = args10938.length;
  switch(G__10942) {
    case 1:
      return cljs.core.atom.cljs$core$IFn$_invoke$arity$1(arguments[0]);
      break;
    default:
      var argseq__8392__auto__ = new cljs.core.IndexedSeq(args10938.slice(1), 0, null);
      return cljs.core.atom.cljs$core$IFn$_invoke$arity$variadic(arguments[0], argseq__8392__auto__);
  }
};
cljs.core.atom.cljs$core$IFn$_invoke$arity$1 = function(x) {
  return new cljs.core.Atom(x, null, null, null);
};
cljs.core.atom.cljs$core$IFn$_invoke$arity$variadic = function(x, p__10943) {
  var map__10944 = p__10943;
  var map__10944__$1 = (!(map__10944 == null) ? map__10944.cljs$lang$protocol_mask$partition0$ & 64 || cljs.core.PROTOCOL_SENTINEL === map__10944.cljs$core$ISeq$ ? true : false : false) ? cljs.core.apply.cljs$core$IFn$_invoke$arity$2(cljs.core.hash_map, map__10944) : map__10944;
  var meta = cljs.core.get.cljs$core$IFn$_invoke$arity$2(map__10944__$1, new cljs.core.Keyword(null, "meta", "meta", 1499536964));
  var validator = cljs.core.get.cljs$core$IFn$_invoke$arity$2(map__10944__$1, new cljs.core.Keyword(null, "validator", "validator", -1966190681));
  return new cljs.core.Atom(x, meta, validator, null);
};
cljs.core.atom.cljs$lang$applyTo = function(seq10939) {
  var G__10940 = cljs.core.first(seq10939);
  var seq10939__$1 = cljs.core.next(seq10939);
  return cljs.core.atom.cljs$core$IFn$_invoke$arity$variadic(G__10940, seq10939__$1);
};
cljs.core.atom.cljs$lang$maxFixedArity = 1;
cljs.core.reset_BANG_ = function cljs$core$reset_BANG_(a, new_value) {
  if (a instanceof cljs.core.Atom) {
    var validate = a.validator;
    if (validate == null) {
    } else {
      if (cljs.core.truth_(validate.cljs$core$IFn$_invoke$arity$1 ? validate.cljs$core$IFn$_invoke$arity$1(new_value) : validate.call(null, new_value))) {
      } else {
        throw new Error("Validator rejected reference state");
      }
    }
    var old_value = a.state;
    a.state = new_value;
    if (a.watches == null) {
    } else {
      cljs.core._notify_watches(a, old_value, new_value);
    }
    return new_value;
  } else {
    return cljs.core._reset_BANG_(a, new_value);
  }
};
cljs.core.swap_BANG_ = function cljs$core$swap_BANG_(var_args) {
  var args10950 = [];
  var len__8373__auto___10964 = arguments.length;
  var i__8374__auto___10965 = 0;
  while (true) {
    if (i__8374__auto___10965 < len__8373__auto___10964) {
      args10950.push(arguments[i__8374__auto___10965]);
      var G__10966 = i__8374__auto___10965 + 1;
      i__8374__auto___10965 = G__10966;
      continue;
    } else {
    }
    break;
  }
  var G__10957 = args10950.length;
  switch(G__10957) {
    case 2:
      return cljs.core.swap_BANG_.cljs$core$IFn$_invoke$arity$2(arguments[0], arguments[1]);
      break;
    case 3:
      return cljs.core.swap_BANG_.cljs$core$IFn$_invoke$arity$3(arguments[0], arguments[1], arguments[2]);
      break;
    case 4:
      return cljs.core.swap_BANG_.cljs$core$IFn$_invoke$arity$4(arguments[0], arguments[1], arguments[2], arguments[3]);
      break;
    default:
      var argseq__8392__auto__ = new cljs.core.IndexedSeq(args10950.slice(4), 0, null);
      return cljs.core.swap_BANG_.cljs$core$IFn$_invoke$arity$variadic(arguments[0], arguments[1], arguments[2], arguments[3], argseq__8392__auto__);
  }
};
cljs.core.swap_BANG_.cljs$core$IFn$_invoke$arity$2 = function(a, f) {
  if (a instanceof cljs.core.Atom) {
    return cljs.core.reset_BANG_(a, function() {
      var G__10958 = a.state;
      return f.cljs$core$IFn$_invoke$arity$1 ? f.cljs$core$IFn$_invoke$arity$1(G__10958) : f.call(null, G__10958);
    }());
  } else {
    return cljs.core._swap_BANG_.cljs$core$IFn$_invoke$arity$2(a, f);
  }
};
cljs.core.swap_BANG_.cljs$core$IFn$_invoke$arity$3 = function(a, f, x) {
  if (a instanceof cljs.core.Atom) {
    return cljs.core.reset_BANG_(a, function() {
      var G__10959 = a.state;
      var G__10960 = x;
      return f.cljs$core$IFn$_invoke$arity$2 ? f.cljs$core$IFn$_invoke$arity$2(G__10959, G__10960) : f.call(null, G__10959, G__10960);
    }());
  } else {
    return cljs.core._swap_BANG_.cljs$core$IFn$_invoke$arity$3(a, f, x);
  }
};
cljs.core.swap_BANG_.cljs$core$IFn$_invoke$arity$4 = function(a, f, x, y) {
  if (a instanceof cljs.core.Atom) {
    return cljs.core.reset_BANG_(a, function() {
      var G__10961 = a.state;
      var G__10962 = x;
      var G__10963 = y;
      return f.cljs$core$IFn$_invoke$arity$3 ? f.cljs$core$IFn$_invoke$arity$3(G__10961, G__10962, G__10963) : f.call(null, G__10961, G__10962, G__10963);
    }());
  } else {
    return cljs.core._swap_BANG_.cljs$core$IFn$_invoke$arity$4(a, f, x, y);
  }
};
cljs.core.swap_BANG_.cljs$core$IFn$_invoke$arity$variadic = function(a, f, x, y, more) {
  if (a instanceof cljs.core.Atom) {
    return cljs.core.reset_BANG_(a, cljs.core.apply.cljs$core$IFn$_invoke$arity$5(f, a.state, x, y, more));
  } else {
    return cljs.core._swap_BANG_.cljs$core$IFn$_invoke$arity$5(a, f, x, y, more);
  }
};
cljs.core.swap_BANG_.cljs$lang$applyTo = function(seq10951) {
  var G__10952 = cljs.core.first(seq10951);
  var seq10951__$1 = cljs.core.next(seq10951);
  var G__10953 = cljs.core.first(seq10951__$1);
  var seq10951__$2 = cljs.core.next(seq10951__$1);
  var G__10954 = cljs.core.first(seq10951__$2);
  var seq10951__$3 = cljs.core.next(seq10951__$2);
  var G__10955 = cljs.core.first(seq10951__$3);
  var seq10951__$4 = cljs.core.next(seq10951__$3);
  return cljs.core.swap_BANG_.cljs$core$IFn$_invoke$arity$variadic(G__10952, G__10953, G__10954, G__10955, seq10951__$4);
};
cljs.core.swap_BANG_.cljs$lang$maxFixedArity = 4;
cljs.core.compare_and_set_BANG_ = function cljs$core$compare_and_set_BANG_(a, oldval, newval) {
  if (cljs.core._EQ_.cljs$core$IFn$_invoke$arity$2(a.cljs$core$IDeref$_deref$arity$1(null), oldval)) {
    cljs.core.reset_BANG_(a, newval);
    return true;
  } else {
    return false;
  }
};
cljs.core.set_validator_BANG_ = function cljs$core$set_validator_BANG_(iref, val) {
  return iref.validator = val;
};
cljs.core.get_validator = function cljs$core$get_validator(iref) {
  return iref.validator;
};
cljs.core.Volatile = function(state) {
  this.state = state;
  this.cljs$lang$protocol_mask$partition0$ = 32768;
  this.cljs$lang$protocol_mask$partition1$ = 0;
};
cljs.core.Volatile.prototype.cljs$core$IVolatile$ = cljs.core.PROTOCOL_SENTINEL;
cljs.core.Volatile.prototype.cljs$core$IVolatile$_vreset_BANG_$arity$2 = function(_, new_state) {
  var self__ = this;
  var ___$1 = this;
  return self__.state = new_state;
};
cljs.core.Volatile.prototype.cljs$core$IDeref$_deref$arity$1 = function(_) {
  var self__ = this;
  var ___$1 = this;
  return self__.state;
};
cljs.core.Volatile.getBasis = function() {
  return new cljs.core.PersistentVector(null, 1, 5, cljs.core.PersistentVector.EMPTY_NODE, [cljs.core.with_meta(new cljs.core.Symbol(null, "state", "state", -348086572, null), new cljs.core.PersistentArrayMap(null, 1, [new cljs.core.Keyword(null, "mutable", "mutable", 875778266), true], null))], null);
};
cljs.core.Volatile.cljs$lang$type = true;
cljs.core.Volatile.cljs$lang$ctorStr = "cljs.core/Volatile";
cljs.core.Volatile.cljs$lang$ctorPrWriter = function(this__7871__auto__, writer__7872__auto__, opt__7873__auto__) {
  return cljs.core._write(writer__7872__auto__, "cljs.core/Volatile");
};
cljs.core.__GT_Volatile = function cljs$core$__GT_Volatile(state) {
  return new cljs.core.Volatile(state);
};
cljs.core.volatile_BANG_ = function cljs$core$volatile_BANG_(val) {
  return new cljs.core.Volatile(val);
};
cljs.core.volatile_QMARK_ = function cljs$core$volatile_QMARK_(x) {
  return x instanceof cljs.core.Volatile;
};
cljs.core.vreset_BANG_ = function cljs$core$vreset_BANG_(vol, newval) {
  return cljs.core._vreset_BANG_(vol, newval);
};
cljs.core.keep_indexed = function cljs$core$keep_indexed(var_args) {
  var args10968 = [];
  var len__8373__auto___10999 = arguments.length;
  var i__8374__auto___11000 = 0;
  while (true) {
    if (i__8374__auto___11000 < len__8373__auto___10999) {
      args10968.push(arguments[i__8374__auto___11000]);
      var G__11001 = i__8374__auto___11000 + 1;
      i__8374__auto___11000 = G__11001;
      continue;
    } else {
    }
    break;
  }
  var G__10970 = args10968.length;
  switch(G__10970) {
    case 1:
      return cljs.core.keep_indexed.cljs$core$IFn$_invoke$arity$1(arguments[0]);
      break;
    case 2:
      return cljs.core.keep_indexed.cljs$core$IFn$_invoke$arity$2(arguments[0], arguments[1]);
      break;
    default:
      throw new Error([cljs.core.str.cljs$core$IFn$_invoke$arity$1("Invalid arity: "), cljs.core.str.cljs$core$IFn$_invoke$arity$1(args10968.length)].join(""));
  }
};
cljs.core.keep_indexed.cljs$core$IFn$_invoke$arity$1 = function(f) {
  return function(rf) {
    var ia = cljs.core.volatile_BANG_(-1);
    return function(ia) {
      return function() {
        var G__11003 = null;
        var G__11003__0 = function() {
          return rf.cljs$core$IFn$_invoke$arity$0 ? rf.cljs$core$IFn$_invoke$arity$0() : rf.call(null);
        };
        var G__11003__1 = function(result) {
          return rf.cljs$core$IFn$_invoke$arity$1 ? rf.cljs$core$IFn$_invoke$arity$1(result) : rf.call(null, result);
        };
        var G__11003__2 = function(result, input) {
          var i = cljs.core._vreset_BANG_(ia, cljs.core._deref(ia) + 1);
          var v = f.cljs$core$IFn$_invoke$arity$2 ? f.cljs$core$IFn$_invoke$arity$2(i, input) : f.call(null, i, input);
          if (v == null) {
            return result;
          } else {
            return rf.cljs$core$IFn$_invoke$arity$2 ? rf.cljs$core$IFn$_invoke$arity$2(result, v) : rf.call(null, result, v);
          }
        };
        G__11003 = function(result, input) {
          switch(arguments.length) {
            case 0:
              return G__11003__0.call(this);
            case 1:
              return G__11003__1.call(this, result);
            case 2:
              return G__11003__2.call(this, result, input);
          }
          throw new Error("Invalid arity: " + (arguments.length - 1));
        };
        G__11003.cljs$core$IFn$_invoke$arity$0 = G__11003__0;
        G__11003.cljs$core$IFn$_invoke$arity$1 = G__11003__1;
        G__11003.cljs$core$IFn$_invoke$arity$2 = G__11003__2;
        return G__11003;
      }();
    }(ia);
  };
};
cljs.core.keep_indexed.cljs$core$IFn$_invoke$arity$2 = function(f, coll) {
  var keepi = function cljs$core$keepi(idx, coll__$1) {
    return new cljs.core.LazySeq(null, function() {
      var temp__4657__auto__ = cljs.core.seq(coll__$1);
      if (temp__4657__auto__) {
        var s = temp__4657__auto__;
        if (cljs.core.chunked_seq_QMARK_(s)) {
          var c = cljs.core.chunk_first(s);
          var size = cljs.core.count(c);
          var b = cljs.core.chunk_buffer(size);
          var n__8183__auto___11004 = size;
          var i_11005 = 0;
          while (true) {
            if (i_11005 < n__8183__auto___11004) {
              var x_11006 = function() {
                var G__10995 = idx + i_11005;
                var G__10996 = cljs.core._nth.cljs$core$IFn$_invoke$arity$2(c, i_11005);
                return f.cljs$core$IFn$_invoke$arity$2 ? f.cljs$core$IFn$_invoke$arity$2(G__10995, G__10996) : f.call(null, G__10995, G__10996);
              }();
              if (x_11006 == null) {
              } else {
                cljs.core.chunk_append(b, x_11006);
              }
              var G__11007 = i_11005 + 1;
              i_11005 = G__11007;
              continue;
            } else {
            }
            break;
          }
          return cljs.core.chunk_cons(cljs.core.chunk(b), cljs$core$keepi(idx + size, cljs.core.chunk_rest(s)));
        } else {
          var x = function() {
            var G__10997 = idx;
            var G__10998 = cljs.core.first(s);
            return f.cljs$core$IFn$_invoke$arity$2 ? f.cljs$core$IFn$_invoke$arity$2(G__10997, G__10998) : f.call(null, G__10997, G__10998);
          }();
          if (x == null) {
            return cljs$core$keepi(idx + 1, cljs.core.rest(s));
          } else {
            return cljs.core.cons(x, cljs$core$keepi(idx + 1, cljs.core.rest(s)));
          }
        }
      } else {
        return null;
      }
    }, null, null);
  };
  return keepi(0, coll);
};
cljs.core.keep_indexed.cljs$lang$maxFixedArity = 2;
cljs.core.every_pred = function cljs$core$every_pred(var_args) {
  var args11014 = [];
  var len__8373__auto___11021 = arguments.length;
  var i__8374__auto___11022 = 0;
  while (true) {
    if (i__8374__auto___11022 < len__8373__auto___11021) {
      args11014.push(arguments[i__8374__auto___11022]);
      var G__11023 = i__8374__auto___11022 + 1;
      i__8374__auto___11022 = G__11023;
      continue;
    } else {
    }
    break;
  }
  var G__11020 = args11014.length;
  switch(G__11020) {
    case 1:
      return cljs.core.every_pred.cljs$core$IFn$_invoke$arity$1(arguments[0]);
      break;
    case 2:
      return cljs.core.every_pred.cljs$core$IFn$_invoke$arity$2(arguments[0], arguments[1]);
      break;
    case 3:
      return cljs.core.every_pred.cljs$core$IFn$_invoke$arity$3(arguments[0], arguments[1], arguments[2]);
      break;
    default:
      var argseq__8392__auto__ = new cljs.core.IndexedSeq(args11014.slice(3), 0, null);
      return cljs.core.every_pred.cljs$core$IFn$_invoke$arity$variadic(arguments[0], arguments[1], arguments[2], argseq__8392__auto__);
  }
};
cljs.core.every_pred.cljs$core$IFn$_invoke$arity$1 = function(p) {
  return function() {
    var cljs$core$ep1 = null;
    var cljs$core$ep1__0 = function() {
      return true;
    };
    var cljs$core$ep1__1 = function(x) {
      return cljs.core.boolean$(p.cljs$core$IFn$_invoke$arity$1 ? p.cljs$core$IFn$_invoke$arity$1(x) : p.call(null, x));
    };
    var cljs$core$ep1__2 = function(x, y) {
      return cljs.core.boolean$(function() {
        var and__7248__auto__ = p.cljs$core$IFn$_invoke$arity$1 ? p.cljs$core$IFn$_invoke$arity$1(x) : p.call(null, x);
        if (cljs.core.truth_(and__7248__auto__)) {
          return p.cljs$core$IFn$_invoke$arity$1 ? p.cljs$core$IFn$_invoke$arity$1(y) : p.call(null, y);
        } else {
          return and__7248__auto__;
        }
      }());
    };
    var cljs$core$ep1__3 = function(x, y, z) {
      return cljs.core.boolean$(function() {
        var and__7248__auto__ = p.cljs$core$IFn$_invoke$arity$1 ? p.cljs$core$IFn$_invoke$arity$1(x) : p.call(null, x);
        if (cljs.core.truth_(and__7248__auto__)) {
          var and__7248__auto____$1 = p.cljs$core$IFn$_invoke$arity$1 ? p.cljs$core$IFn$_invoke$arity$1(y) : p.call(null, y);
          if (cljs.core.truth_(and__7248__auto____$1)) {
            return p.cljs$core$IFn$_invoke$arity$1 ? p.cljs$core$IFn$_invoke$arity$1(z) : p.call(null, z);
          } else {
            return and__7248__auto____$1;
          }
        } else {
          return and__7248__auto__;
        }
      }());
    };
    var cljs$core$ep1__4 = function() {
      var G__11025__delegate = function(x, y, z, args) {
        return cljs.core.boolean$(function() {
          var and__7248__auto__ = cljs$core$ep1.cljs$core$IFn$_invoke$arity$3(x, y, z);
          if (cljs.core.truth_(and__7248__auto__)) {
            return cljs.core.every_QMARK_(p, args);
          } else {
            return and__7248__auto__;
          }
        }());
      };
      var G__11025 = function(x, y, z, var_args) {
        var args = null;
        if (arguments.length > 3) {
          var G__11026__i = 0, G__11026__a = new Array(arguments.length - 3);
          while (G__11026__i < G__11026__a.length) {
            G__11026__a[G__11026__i] = arguments[G__11026__i + 3];
            ++G__11026__i;
          }
          args = new cljs.core.IndexedSeq(G__11026__a, 0);
        }
        return G__11025__delegate.call(this, x, y, z, args);
      };
      G__11025.cljs$lang$maxFixedArity = 3;
      G__11025.cljs$lang$applyTo = function(arglist__11027) {
        var x = cljs.core.first(arglist__11027);
        arglist__11027 = cljs.core.next(arglist__11027);
        var y = cljs.core.first(arglist__11027);
        arglist__11027 = cljs.core.next(arglist__11027);
        var z = cljs.core.first(arglist__11027);
        var args = cljs.core.rest(arglist__11027);
        return G__11025__delegate(x, y, z, args);
      };
      G__11025.cljs$core$IFn$_invoke$arity$variadic = G__11025__delegate;
      return G__11025;
    }();
    cljs$core$ep1 = function(x, y, z, var_args) {
      var args = var_args;
      switch(arguments.length) {
        case 0:
          return cljs$core$ep1__0.call(this);
        case 1:
          return cljs$core$ep1__1.call(this, x);
        case 2:
          return cljs$core$ep1__2.call(this, x, y);
        case 3:
          return cljs$core$ep1__3.call(this, x, y, z);
        default:
          var G__11028 = null;
          if (arguments.length > 3) {
            var G__11029__i = 0, G__11029__a = new Array(arguments.length - 3);
            while (G__11029__i < G__11029__a.length) {
              G__11029__a[G__11029__i] = arguments[G__11029__i + 3];
              ++G__11029__i;
            }
            G__11028 = new cljs.core.IndexedSeq(G__11029__a, 0);
          }
          return cljs$core$ep1__4.cljs$core$IFn$_invoke$arity$variadic(x, y, z, G__11028);
      }
      throw new Error("Invalid arity: " + (arguments.length - 1));
    };
    cljs$core$ep1.cljs$lang$maxFixedArity = 3;
    cljs$core$ep1.cljs$lang$applyTo = cljs$core$ep1__4.cljs$lang$applyTo;
    cljs$core$ep1.cljs$core$IFn$_invoke$arity$0 = cljs$core$ep1__0;
    cljs$core$ep1.cljs$core$IFn$_invoke$arity$1 = cljs$core$ep1__1;
    cljs$core$ep1.cljs$core$IFn$_invoke$arity$2 = cljs$core$ep1__2;
    cljs$core$ep1.cljs$core$IFn$_invoke$arity$3 = cljs$core$ep1__3;
    cljs$core$ep1.cljs$core$IFn$_invoke$arity$variadic = cljs$core$ep1__4.cljs$core$IFn$_invoke$arity$variadic;
    return cljs$core$ep1;
  }();
};
cljs.core.every_pred.cljs$core$IFn$_invoke$arity$2 = function(p1, p2) {
  return function() {
    var cljs$core$ep2 = null;
    var cljs$core$ep2__0 = function() {
      return true;
    };
    var cljs$core$ep2__1 = function(x) {
      return cljs.core.boolean$(function() {
        var and__7248__auto__ = p1.cljs$core$IFn$_invoke$arity$1 ? p1.cljs$core$IFn$_invoke$arity$1(x) : p1.call(null, x);
        if (cljs.core.truth_(and__7248__auto__)) {
          return p2.cljs$core$IFn$_invoke$arity$1 ? p2.cljs$core$IFn$_invoke$arity$1(x) : p2.call(null, x);
        } else {
          return and__7248__auto__;
        }
      }());
    };
    var cljs$core$ep2__2 = function(x, y) {
      return cljs.core.boolean$(function() {
        var and__7248__auto__ = p1.cljs$core$IFn$_invoke$arity$1 ? p1.cljs$core$IFn$_invoke$arity$1(x) : p1.call(null, x);
        if (cljs.core.truth_(and__7248__auto__)) {
          var and__7248__auto____$1 = p1.cljs$core$IFn$_invoke$arity$1 ? p1.cljs$core$IFn$_invoke$arity$1(y) : p1.call(null, y);
          if (cljs.core.truth_(and__7248__auto____$1)) {
            var and__7248__auto____$2 = p2.cljs$core$IFn$_invoke$arity$1 ? p2.cljs$core$IFn$_invoke$arity$1(x) : p2.call(null, x);
            if (cljs.core.truth_(and__7248__auto____$2)) {
              return p2.cljs$core$IFn$_invoke$arity$1 ? p2.cljs$core$IFn$_invoke$arity$1(y) : p2.call(null, y);
            } else {
              return and__7248__auto____$2;
            }
          } else {
            return and__7248__auto____$1;
          }
        } else {
          return and__7248__auto__;
        }
      }());
    };
    var cljs$core$ep2__3 = function(x, y, z) {
      return cljs.core.boolean$(function() {
        var and__7248__auto__ = p1.cljs$core$IFn$_invoke$arity$1 ? p1.cljs$core$IFn$_invoke$arity$1(x) : p1.call(null, x);
        if (cljs.core.truth_(and__7248__auto__)) {
          var and__7248__auto____$1 = p1.cljs$core$IFn$_invoke$arity$1 ? p1.cljs$core$IFn$_invoke$arity$1(y) : p1.call(null, y);
          if (cljs.core.truth_(and__7248__auto____$1)) {
            var and__7248__auto____$2 = p1.cljs$core$IFn$_invoke$arity$1 ? p1.cljs$core$IFn$_invoke$arity$1(z) : p1.call(null, z);
            if (cljs.core.truth_(and__7248__auto____$2)) {
              var and__7248__auto____$3 = p2.cljs$core$IFn$_invoke$arity$1 ? p2.cljs$core$IFn$_invoke$arity$1(x) : p2.call(null, x);
              if (cljs.core.truth_(and__7248__auto____$3)) {
                var and__7248__auto____$4 = p2.cljs$core$IFn$_invoke$arity$1 ? p2.cljs$core$IFn$_invoke$arity$1(y) : p2.call(null, y);
                if (cljs.core.truth_(and__7248__auto____$4)) {
                  return p2.cljs$core$IFn$_invoke$arity$1 ? p2.cljs$core$IFn$_invoke$arity$1(z) : p2.call(null, z);
                } else {
                  return and__7248__auto____$4;
                }
              } else {
                return and__7248__auto____$3;
              }
            } else {
              return and__7248__auto____$2;
            }
          } else {
            return and__7248__auto____$1;
          }
        } else {
          return and__7248__auto__;
        }
      }());
    };
    var cljs$core$ep2__4 = function() {
      var G__11030__delegate = function(x, y, z, args) {
        return cljs.core.boolean$(function() {
          var and__7248__auto__ = cljs$core$ep2.cljs$core$IFn$_invoke$arity$3(x, y, z);
          if (cljs.core.truth_(and__7248__auto__)) {
            return cljs.core.every_QMARK_(function(and__7248__auto__) {
              return function(p1__11008_SHARP_) {
                var and__7248__auto____$1 = p1.cljs$core$IFn$_invoke$arity$1 ? p1.cljs$core$IFn$_invoke$arity$1(p1__11008_SHARP_) : p1.call(null, p1__11008_SHARP_);
                if (cljs.core.truth_(and__7248__auto____$1)) {
                  return p2.cljs$core$IFn$_invoke$arity$1 ? p2.cljs$core$IFn$_invoke$arity$1(p1__11008_SHARP_) : p2.call(null, p1__11008_SHARP_);
                } else {
                  return and__7248__auto____$1;
                }
              };
            }(and__7248__auto__), args);
          } else {
            return and__7248__auto__;
          }
        }());
      };
      var G__11030 = function(x, y, z, var_args) {
        var args = null;
        if (arguments.length > 3) {
          var G__11031__i = 0, G__11031__a = new Array(arguments.length - 3);
          while (G__11031__i < G__11031__a.length) {
            G__11031__a[G__11031__i] = arguments[G__11031__i + 3];
            ++G__11031__i;
          }
          args = new cljs.core.IndexedSeq(G__11031__a, 0);
        }
        return G__11030__delegate.call(this, x, y, z, args);
      };
      G__11030.cljs$lang$maxFixedArity = 3;
      G__11030.cljs$lang$applyTo = function(arglist__11032) {
        var x = cljs.core.first(arglist__11032);
        arglist__11032 = cljs.core.next(arglist__11032);
        var y = cljs.core.first(arglist__11032);
        arglist__11032 = cljs.core.next(arglist__11032);
        var z = cljs.core.first(arglist__11032);
        var args = cljs.core.rest(arglist__11032);
        return G__11030__delegate(x, y, z, args);
      };
      G__11030.cljs$core$IFn$_invoke$arity$variadic = G__11030__delegate;
      return G__11030;
    }();
    cljs$core$ep2 = function(x, y, z, var_args) {
      var args = var_args;
      switch(arguments.length) {
        case 0:
          return cljs$core$ep2__0.call(this);
        case 1:
          return cljs$core$ep2__1.call(this, x);
        case 2:
          return cljs$core$ep2__2.call(this, x, y);
        case 3:
          return cljs$core$ep2__3.call(this, x, y, z);
        default:
          var G__11033 = null;
          if (arguments.length > 3) {
            var G__11034__i = 0, G__11034__a = new Array(arguments.length - 3);
            while (G__11034__i < G__11034__a.length) {
              G__11034__a[G__11034__i] = arguments[G__11034__i + 3];
              ++G__11034__i;
            }
            G__11033 = new cljs.core.IndexedSeq(G__11034__a, 0);
          }
          return cljs$core$ep2__4.cljs$core$IFn$_invoke$arity$variadic(x, y, z, G__11033);
      }
      throw new Error("Invalid arity: " + (arguments.length - 1));
    };
    cljs$core$ep2.cljs$lang$maxFixedArity = 3;
    cljs$core$ep2.cljs$lang$applyTo = cljs$core$ep2__4.cljs$lang$applyTo;
    cljs$core$ep2.cljs$core$IFn$_invoke$arity$0 = cljs$core$ep2__0;
    cljs$core$ep2.cljs$core$IFn$_invoke$arity$1 = cljs$core$ep2__1;
    cljs$core$ep2.cljs$core$IFn$_invoke$arity$2 = cljs$core$ep2__2;
    cljs$core$ep2.cljs$core$IFn$_invoke$arity$3 = cljs$core$ep2__3;
    cljs$core$ep2.cljs$core$IFn$_invoke$arity$variadic = cljs$core$ep2__4.cljs$core$IFn$_invoke$arity$variadic;
    return cljs$core$ep2;
  }();
};
cljs.core.every_pred.cljs$core$IFn$_invoke$arity$3 = function(p1, p2, p3) {
  return function() {
    var cljs$core$ep3 = null;
    var cljs$core$ep3__0 = function() {
      return true;
    };
    var cljs$core$ep3__1 = function(x) {
      return cljs.core.boolean$(function() {
        var and__7248__auto__ = p1.cljs$core$IFn$_invoke$arity$1 ? p1.cljs$core$IFn$_invoke$arity$1(x) : p1.call(null, x);
        if (cljs.core.truth_(and__7248__auto__)) {
          var and__7248__auto____$1 = p2.cljs$core$IFn$_invoke$arity$1 ? p2.cljs$core$IFn$_invoke$arity$1(x) : p2.call(null, x);
          if (cljs.core.truth_(and__7248__auto____$1)) {
            return p3.cljs$core$IFn$_invoke$arity$1 ? p3.cljs$core$IFn$_invoke$arity$1(x) : p3.call(null, x);
          } else {
            return and__7248__auto____$1;
          }
        } else {
          return and__7248__auto__;
        }
      }());
    };
    var cljs$core$ep3__2 = function(x, y) {
      return cljs.core.boolean$(function() {
        var and__7248__auto__ = p1.cljs$core$IFn$_invoke$arity$1 ? p1.cljs$core$IFn$_invoke$arity$1(x) : p1.call(null, x);
        if (cljs.core.truth_(and__7248__auto__)) {
          var and__7248__auto____$1 = p2.cljs$core$IFn$_invoke$arity$1 ? p2.cljs$core$IFn$_invoke$arity$1(x) : p2.call(null, x);
          if (cljs.core.truth_(and__7248__auto____$1)) {
            var and__7248__auto____$2 = p3.cljs$core$IFn$_invoke$arity$1 ? p3.cljs$core$IFn$_invoke$arity$1(x) : p3.call(null, x);
            if (cljs.core.truth_(and__7248__auto____$2)) {
              var and__7248__auto____$3 = p1.cljs$core$IFn$_invoke$arity$1 ? p1.cljs$core$IFn$_invoke$arity$1(y) : p1.call(null, y);
              if (cljs.core.truth_(and__7248__auto____$3)) {
                var and__7248__auto____$4 = p2.cljs$core$IFn$_invoke$arity$1 ? p2.cljs$core$IFn$_invoke$arity$1(y) : p2.call(null, y);
                if (cljs.core.truth_(and__7248__auto____$4)) {
                  return p3.cljs$core$IFn$_invoke$arity$1 ? p3.cljs$core$IFn$_invoke$arity$1(y) : p3.call(null, y);
                } else {
                  return and__7248__auto____$4;
                }
              } else {
                return and__7248__auto____$3;
              }
            } else {
              return and__7248__auto____$2;
            }
          } else {
            return and__7248__auto____$1;
          }
        } else {
          return and__7248__auto__;
        }
      }());
    };
    var cljs$core$ep3__3 = function(x, y, z) {
      return cljs.core.boolean$(function() {
        var and__7248__auto__ = p1.cljs$core$IFn$_invoke$arity$1 ? p1.cljs$core$IFn$_invoke$arity$1(x) : p1.call(null, x);
        if (cljs.core.truth_(and__7248__auto__)) {
          var and__7248__auto____$1 = p2.cljs$core$IFn$_invoke$arity$1 ? p2.cljs$core$IFn$_invoke$arity$1(x) : p2.call(null, x);
          if (cljs.core.truth_(and__7248__auto____$1)) {
            var and__7248__auto____$2 = p3.cljs$core$IFn$_invoke$arity$1 ? p3.cljs$core$IFn$_invoke$arity$1(x) : p3.call(null, x);
            if (cljs.core.truth_(and__7248__auto____$2)) {
              var and__7248__auto____$3 = p1.cljs$core$IFn$_invoke$arity$1 ? p1.cljs$core$IFn$_invoke$arity$1(y) : p1.call(null, y);
              if (cljs.core.truth_(and__7248__auto____$3)) {
                var and__7248__auto____$4 = p2.cljs$core$IFn$_invoke$arity$1 ? p2.cljs$core$IFn$_invoke$arity$1(y) : p2.call(null, y);
                if (cljs.core.truth_(and__7248__auto____$4)) {
                  var and__7248__auto____$5 = p3.cljs$core$IFn$_invoke$arity$1 ? p3.cljs$core$IFn$_invoke$arity$1(y) : p3.call(null, y);
                  if (cljs.core.truth_(and__7248__auto____$5)) {
                    var and__7248__auto____$6 = p1.cljs$core$IFn$_invoke$arity$1 ? p1.cljs$core$IFn$_invoke$arity$1(z) : p1.call(null, z);
                    if (cljs.core.truth_(and__7248__auto____$6)) {
                      var and__7248__auto____$7 = p2.cljs$core$IFn$_invoke$arity$1 ? p2.cljs$core$IFn$_invoke$arity$1(z) : p2.call(null, z);
                      if (cljs.core.truth_(and__7248__auto____$7)) {
                        return p3.cljs$core$IFn$_invoke$arity$1 ? p3.cljs$core$IFn$_invoke$arity$1(z) : p3.call(null, z);
                      } else {
                        return and__7248__auto____$7;
                      }
                    } else {
                      return and__7248__auto____$6;
                    }
                  } else {
                    return and__7248__auto____$5;
                  }
                } else {
                  return and__7248__auto____$4;
                }
              } else {
                return and__7248__auto____$3;
              }
            } else {
              return and__7248__auto____$2;
            }
          } else {
            return and__7248__auto____$1;
          }
        } else {
          return and__7248__auto__;
        }
      }());
    };
    var cljs$core$ep3__4 = function() {
      var G__11035__delegate = function(x, y, z, args) {
        return cljs.core.boolean$(function() {
          var and__7248__auto__ = cljs$core$ep3.cljs$core$IFn$_invoke$arity$3(x, y, z);
          if (cljs.core.truth_(and__7248__auto__)) {
            return cljs.core.every_QMARK_(function(and__7248__auto__) {
              return function(p1__11009_SHARP_) {
                var and__7248__auto____$1 = p1.cljs$core$IFn$_invoke$arity$1 ? p1.cljs$core$IFn$_invoke$arity$1(p1__11009_SHARP_) : p1.call(null, p1__11009_SHARP_);
                if (cljs.core.truth_(and__7248__auto____$1)) {
                  var and__7248__auto____$2 = p2.cljs$core$IFn$_invoke$arity$1 ? p2.cljs$core$IFn$_invoke$arity$1(p1__11009_SHARP_) : p2.call(null, p1__11009_SHARP_);
                  if (cljs.core.truth_(and__7248__auto____$2)) {
                    return p3.cljs$core$IFn$_invoke$arity$1 ? p3.cljs$core$IFn$_invoke$arity$1(p1__11009_SHARP_) : p3.call(null, p1__11009_SHARP_);
                  } else {
                    return and__7248__auto____$2;
                  }
                } else {
                  return and__7248__auto____$1;
                }
              };
            }(and__7248__auto__), args);
          } else {
            return and__7248__auto__;
          }
        }());
      };
      var G__11035 = function(x, y, z, var_args) {
        var args = null;
        if (arguments.length > 3) {
          var G__11036__i = 0, G__11036__a = new Array(arguments.length - 3);
          while (G__11036__i < G__11036__a.length) {
            G__11036__a[G__11036__i] = arguments[G__11036__i + 3];
            ++G__11036__i;
          }
          args = new cljs.core.IndexedSeq(G__11036__a, 0);
        }
        return G__11035__delegate.call(this, x, y, z, args);
      };
      G__11035.cljs$lang$maxFixedArity = 3;
      G__11035.cljs$lang$applyTo = function(arglist__11037) {
        var x = cljs.core.first(arglist__11037);
        arglist__11037 = cljs.core.next(arglist__11037);
        var y = cljs.core.first(arglist__11037);
        arglist__11037 = cljs.core.next(arglist__11037);
        var z = cljs.core.first(arglist__11037);
        var args = cljs.core.rest(arglist__11037);
        return G__11035__delegate(x, y, z, args);
      };
      G__11035.cljs$core$IFn$_invoke$arity$variadic = G__11035__delegate;
      return G__11035;
    }();
    cljs$core$ep3 = function(x, y, z, var_args) {
      var args = var_args;
      switch(arguments.length) {
        case 0:
          return cljs$core$ep3__0.call(this);
        case 1:
          return cljs$core$ep3__1.call(this, x);
        case 2:
          return cljs$core$ep3__2.call(this, x, y);
        case 3:
          return cljs$core$ep3__3.call(this, x, y, z);
        default:
          var G__11038 = null;
          if (arguments.length > 3) {
            var G__11039__i = 0, G__11039__a = new Array(arguments.length - 3);
            while (G__11039__i < G__11039__a.length) {
              G__11039__a[G__11039__i] = arguments[G__1103