(ns org.knotation.api
  (:require [clojure.string :as string]

            [org.knotation.util :as util]
            [org.knotation.environment :as env]
            [org.knotation.state :as st]
            [org.knotation.link :as ln]
            [org.knotation.format :as fmt]

            org.knotation.kn
            org.knotation.ttl
            org.knotation.nq
            org.knotation.tsv))

;; Environments
(def add-base env/add-base)
(def add-prefix env/add-prefix)
(def add-label env/add-label)
(def set-datatype env/set-datatype)
(def set-language env/set-language)
(def set-template-content env/set-template-content)
(def default-env env/default-env)

(defn labels [env] (::env/label-seq env))
(defn prefixes [env] (::env/prefix-seq env))
(defn prefix-states [env]
  (map (fn [[prefix iri]]
         {:event :prefix
          :prefix prefix
          :iri iri})
       (::env/prefix-iri env)))

(def find-prefix ln/find-prefix)
(def ->iri ln/->iri)
(def iri->name ln/iri->name)
(def iri->curie ln/iri->curie)
(def iri->label ln/iri->label)

;; Hub format manipulation
;;; I'm calling it "hub", because this is the internal representation
;;; that each of the spokes eventually comes down to. I think we can
;;; simplify it a great deal once we figure out what we want the
;;; external interface to look like

;; Hub state queries
(defn graph-end? [s] (= :graph-end (:event s)))
(defn error? [s] (= :error (:event s)))

(defn error-message [s] (->> s :error :error-message))
(defn error-type [s] (->> s :error :error-type))

(defn line-num-in [s] (->> s :input :line-number))
(defn line-ct-in [s] (->> s :input :line-count))

(defn line-num-out [s] (->> s :output :line-number))
(defn line-ct-out [s] (->> s :output :line-count))

;; Hub collection queries
(defn env-of
  [h]
  (::env/env (last h)))

(defn errors-of
  [h]
  (map
   (fn [s]
     (when (error? s)
       [(error-type s) (error-message s)]))
   h))

(defn any-errors?
  [h]
  (->> h errors-of (remove nil?) empty? not))

;; Processing to/from hub
(defn read-lines
  ([format lines] (read-lines format default-env lines))
  ([format env lines] (fmt/read-lines format env lines)))

(defn read-from
  ([format thing] (read-from format default-env thing))
  ([format env thing]
   (cond
     (string? thing)
     (read-lines format env (util/split-lines thing))

     (coll? thing)
     (reduce
      (fn [prev s]
        (concat prev (read-lines format (or (env-of prev) env) (util/split-lines s))))
      nil thing)

     :else (util/throw-exception "Can't read from a thing of type" (type thing)))))

(defn render-to
  ([format h] (render-to format (env-of h) h))
  ([format env h]
   (fmt/render-output (fmt/render-states format env h))))

(defn collect-line-map
  [hub]
  (->> hub
       (map (fn [s]
              [[(get-in s [:input :line-number] 0) (get-in s [:input :line-count] 0)]
               [(get-in s [:output :line-number] 0) (get-in s [:output :line-count] 0)]]))
       (map (fn [[[ln-in ct-in] [ln-out ct-out]]]
              (let [out (set (take ct-out (drop ln-out (range))))]
                (map
                 (fn [in] [in out])
                 (take ct-in (drop ln-in (range)))))))
       dedupe
       (apply concat)))

(defn line-map-of
  ([format h] (line-map-of format (env-of h) h))
  ([format env h] (collect-line-map (fmt/render-states format env h))))
