;; Copyright 2008-2012 Red Hat, Inc, and individual contributors.
;; 
;; This is free software; you can redistribute it and/or modify it
;; under the terms of the GNU Lesser General Public License as
;; published by the Free Software Foundation; either version 2.1 of
;; the License, or (at your option) any later version.
;; 
;; This software is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
;; Lesser General Public License for more details.
;; 
;; You should have received a copy of the GNU Lesser General Public
;; License along with this software; if not, write to the Free
;; Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
;; 02110-1301 USA, or see the FSF site: http://www.fsf.org.

(ns immutant.web
  "Associate one or more Ring handlers with your application, mounted
   at unique context paths"
  (:require [immutant.registry     :as reg]
            [clojure.tools.logging :as log]
            [immutant.utilities :as util]
            [ring.middleware.reload :as ring])
  (:use immutant.web.internal))



(def  ^{:arglists '([context? handler & {:keys [reload]}])} start
  "Registers a Ring handler that will be called when requests are
   received on the given sub-context-path. If no sub-context-path is
   given, \"/\" is assumed.

   The following options are supported [default]:
     :reload    monitors the app's src/ dir for changes [false]"
  (fn
    [& args]
    (let [[sub-context-path args] (if (string? (first args))
                                    [(first args) (next args)]
                                    ["/" args])
          [handler & {:keys [reload]}] args
          handler (if reload
                    (ring/wrap-reload
                     handler
                     {:dirs [(util/app-relative "src")]})
                    handler)
          sub-context-path (normalize-subcontext-path sub-context-path)]
      (log/info "Registering ring handler at sub-context path:" sub-context-path)
      (store-servlet-info! (servlet-name sub-context-path)
                           {:wrapper (install-servlet "org.immutant.web.servlet.RingServlet"
                                                      sub-context-path)
                            :sub-context sub-context-path
                            :handler handler})
      nil)))

(defn stop
  "Deregisters the Ring handler attached to the given sub-context-path. If no sub-context-path is given,
\"/\" is assumed."
  ([]
     (stop "/"))
  ([sub-context-path]
     (let [sub-context-path (normalize-subcontext-path sub-context-path)]
       (if-let [info (remove-servlet-info! (servlet-name sub-context-path))]
         (do
           (log/info "Deregistering ring handler at sub-context path:" sub-context-path)
           (remove-servlet sub-context-path (:wrapper info)))
         (log/warn "Attempted to deregister ring handler at sub-context path:" sub-context-path ", but none found")))))

