(ns hypercrud.service.core
  (:require [clojure.core.match :refer [match]]
            [datomic.api :as d]
            [hypercrud.service.pedestal-util :as pedestal-util]
            [hypercrud.service.util :as util]
            [io.pedestal.http.body-params :as body-params]
            [ring.util.response :as ring-resp]))


(defn wrap-hypercrud [m]
  {:hypercrud m})


(defn query [datomic-conn
             security-predicate
             {{:keys [tx user-token]} :query-params
              [query params] :body-params}]
  (let [user user-token
        tx (Long/parseLong tx)
        db (d/filter (d/as-of (d/db datomic-conn) tx) (partial security-predicate user))]
    (->> (apply d/q query db params)
         (mapv (comp :db/id (partial d/entity db)))         ;resolve idents
         wrap-hypercrud
         ring-resp/response)))


(defn represent-ref-val [db v]
  ;; don't d/entity an entity
  (:db/id (if (keyword? v) (d/entity db v) v)))


(defn entity [datomic-conn
              schema
              security-predicate
              {{:keys [tx user-token]} :query-params
               {:keys [eid]} :path-params}]
  (let [user user-token
        tx (Long/parseLong tx)
        eid (Long/parseLong eid)
        db (d/filter (d/as-of (d/db datomic-conn) tx) (partial security-predicate user))]
    (->> (d/entity db eid)
         (map (fn [[attr val]]
                [attr (match ((juxt :db/valueType :db/cardinality) (attr schema))
                             [:db.type/ref :db.cardinality/one] (represent-ref-val db val)
                             [:db.type/ref :db.cardinality/many] (into #{} (map #(represent-ref-val db %) val))
                             [_ _] val)]))
         (into {:db/id eid})
         wrap-hypercrud
         ring-resp/response)))


(defn transact! [datomic-conn
                 validate-tx
                 {{:keys [user-token]} :query-params
                  user-tx :body-params}]
  (let [user user-token
        db (d/db datomic-conn)                              ; tx validation needs schema, so gets unfiltered db
        valid? (validate-tx db user user-tx)
        user-tx (map #(util/marshal-statement-types db %) user-tx)]
    (if-not valid?
      {:status 500 :headers {} :body "user tx failed validation"}

      (let [hc-tx [{:db/id (d/tempid :db.part/tx) :hypercrud/created-by user}]
            effect (d/transact datomic-conn (concat user-tx hc-tx))
            tx (-> @effect :db-after d/basis-t d/t->tx)]
        (ring-resp/response (wrap-hypercrud {:tx tx}))))))


(defn index [datomic-conn request]
  (let [tx (-> (d/db datomic-conn) d/basis-t d/t->tx)]
    (ring-resp/response (wrap-hypercrud {:tx tx}))))


(defn latest [datomic-conn request]
  (let [tx (-> (d/db datomic-conn) d/basis-t d/t->tx)]
    (ring-resp/response tx)))


(defn route [datomic-conn root schema validate-tx security-predicate]
  `[~root {:get [:index (partial index ~datomic-conn)]}
    ^:interceptors [~(body-params/body-params)
                    pedestal-util/combine-body-params
                    pedestal-util/auto-content-type]

    ["/query" {:post [:query (partial query ~datomic-conn ~security-predicate)]}]
    ["/entity/:eid" {:get [:entity (partial entity ~datomic-conn ~schema ~security-predicate)]}]
    ["/transact" {:post [:transact (partial transact! ~datomic-conn ~validate-tx)]}]
    ["/latest" {:get [:latest (partial latest ~datomic-conn)]}]])
