(ns hypercrud.service.core
  (:require [clojure.core.match :refer [match]]
            [datomic.api :as d]
            [datomico.db :as db]
            [hypercrud.service.datomic-util :as datomic-util :refer [with-db-as-of]]
            [hypercrud.service.pedestal-util :as pedestal-util]
            [hypercrud.service.root-renderer :as root-renderer]
            [io.pedestal.http.body-params :as body-params]))


;; Nice to render out vectors for the client since they are already associative and any
;; benefits of list e.g. lazy are lost due to serialization on the wire


(defn query [{{:keys [tx]} :query-params
              q ::pedestal-util/body-params}]
  (let [tx (Long/parseLong tx)
        resultset (datomic-util/datomic-simple-q q tx)]
    (with-db-as-of tx
      (mapv (comp :db/id (partial d/entity db/*db*)) resultset))))


(defn represent-ref-val [v]
  ;; don't d/entity an entity
  (:db/id (if (keyword? v) (d/entity db/*db* v) v)))


(defn entity [schema
              {{:keys [tx]} :query-params
               {:keys [eid]} :path-params}]
  (let [tx (Long/parseLong tx)
        eid (Long/parseLong eid)
        entity (with-db-as-of tx (d/entity db/*db* eid))]
    (with-db-as-of tx
      (->> entity
           (map (fn [[attr val]]
                  [attr (match ((juxt :db/valueType :db/cardinality) (attr schema))
                               [:db.type/ref :db.cardinality/one] (represent-ref-val val)
                               [:db.type/ref :db.cardinality/many] (into #{} (map represent-ref-val val))
                               [_ _] val)]))
           (into {})))))


(defn transact! [{:keys [::pedestal-util/body-params]}]
  (let [effect (d/transact db/*connection* body-params)
        tx (-> @effect :db-after d/basis-t d/t->tx)]
    {:tx tx}))


(defn index [request]
  {:tx (datomic-util/latest-tx)})


(defn route [root schema]
  `[~root {:get [:index index]}
    ^:interceptors [~(body-params/body-params)
                    pedestal-util/auto-content-type
                    pedestal-util/combine-body-params
                    pedestal-util/wrap-ring-response
                    root-renderer/wrap-hypercrud]

    ["/query" {:post [:query query]}]
    ["/entity/:eid" {:get [:entity (partial entity ~schema)]}]
    ["/transact" {:post [:transact transact!]}]])
