(ns hypercrud-ui.select
  (:require [hypercrud-client.core :as hypercrud]
            [hypercrud-ui.form :as form]
            [reagent.core :as reagent]
            [hypercrud-client.util :as util]))


(defn select-option
  ":option :value and :on-change is a string, since that's how the dom works"
  [client label-prop cj-option-item]
  ^{:key (:href cj-option-item)}
  [hypercrud/resolve client cj-option-item
   (fn [{:keys [href rel links data] :as cj-option-item}]
     ;; the dom will stringify the :value i believe
     [:option {:key (name rel)
               :value (util/transit-encode (select-keys cj-option-item [:href :rel]))}
      (name (get data label-prop))])
   (fn [hc-node] [:option])])


(defn select*
  [{:keys [client label-prop cj-options-list cur] :as props}]
  (assert (not (nil? cur)))
  (let [updating? (reagent/atom false)]
    (fn [{:keys [client label-prop cj-options-list cur] :as props}]
      [hypercrud/resolve client cj-options-list
       (fn [{:keys [data template] :as cj-options-list}]
         (let [props (-> props
                         (dissoc :client :label-prop :cj-options-list :cur)

                         ;; normalize value for the dom
                         ;; value is either nil, an :ident (keyword), or
                         ;; a cj item ref e.g. {:href ...}
                         (assoc :value (util/transit-encode
                                         (cond
                                           (nil? @cur) nil
                                           (util/tempid? (:rel @cur)) "create-new"
                                           :else (select-keys @cur [:href :rel]))))

                         ;; reconstruct the typed value (keyword or cj item ref)
                         (assoc :on-change
                                #(do
                                  (let [select-value (util/transit-decode (.-target.value %))
                                        hc-node (cond
                                                  (nil? select-value) nil
                                                  (= "create-new" select-value) (assoc (hypercrud/new client template) :data {label-prop ""})
                                                  :else-hc-select-option-node select-value)]
                                    (reset! cur hc-node)
                                    ;; and also add our new guy to the option list (for all combos)
                                    (reset! updating? false)))))

               create-new? (some-> @cur :rel util/tempid?)
               show-form? (or @updating? create-new?)]

           ^{:key (:href cj-options-list)}
           [:div.editable-select
            [:button {:on-click #(swap! updating? not)
                      :disabled (= nil @cur)}
             (if show-form? "Discard" "Edit")]
            [:span
             [:select props (-> (doall (map (fn [[rel hc-node]]
                                              (select-option client label-prop hc-node))
                                            data))
                                (concat [[:option {:key :create-new :value (util/transit-encode "create-new")} "Create New"]
                                         [:option {:key :blank :value (util/transit-encode nil)} "--"]]))]]
            (if show-form?
              [hypercrud/resolve client @cur
               (fn [hc-node]
                 [form/cj-form client cur hc-node template])])]))])))
