(ns hypercrud.service.pedestal-util
  (:require [clojure.string]
            [io.pedestal.http :as http]
            [io.pedestal.interceptor :as interceptor]
            [cognitect.transit :as transit]
            [ring.util.response :as ring-resp])
  (:import (java.io OutputStreamWriter
                    OutputStream)
           org.apache.commons.lang3.StringEscapeUtils))


(interceptor/defon-request combine-body-params
  [{:keys [json-params edn-params transit-params] :as request}]
  (assoc request ::body-params (or json-params
                                   edn-params
                                   transit-params)))

;; This is copied from the pedestal source code, it is private
(defn- print-fn
  [prn-fn]
  (fn [output-stream]
    (with-open [writer (OutputStreamWriter. output-stream)]
      (binding [*out* writer]
        (prn-fn))
      (.flush writer))))


;; Used for serializing the response body only; not used for parsing the request
(def content-types
  {"application/json;charset=UTF-8"
   (fn [body]
     (print-fn #(http/json-print body)))

   "application/edn;charset=UTF-8"
   (fn [body]
     (print-fn #(clojure.pprint/pprint body)))

   "application/transit+json;charset=UTF-8"
   (fn [body]
     (fn [^OutputStream output-stream]
       (transit/write (transit/writer output-stream :json-verbose) body)
       (.flush output-stream)))

   "application/transit+msgpack;charset=UTF-8"
   (fn [body]
     (fn [^OutputStream output-stream]
       (transit/write (transit/writer output-stream :msgpack) body)
       (.flush output-stream)))

   "text/html"
   (fn [body]
     (binding [clojure.pprint/*print-right-margin* 140]
       (let [body-str (as-> (with-out-str (clojure.pprint/pprint body)) $
                            (StringEscapeUtils/escapeHtml4 $)
                            (format "<html><body><pre>%s</pre></body></html>" $))]
         (print-fn #(.write *out* body-str)))))})




(interceptor/defafter auto-content-type
  "Use the Accept header to determine the content type and render the response body."
  [{:keys [request response] :as context}]
  (let [accepts (-> (get-in request [:headers "accept"])
                    (clojure.string/split #",")
                    (#(into #{} %)))
        accept (-> (some accepts (keys content-types))
                   #_ (or "application/transit+json;charset=UTF-8"))
        content-renderer (get content-types accept)]
    (assert content-renderer  (str "invalid Accept " (get-in request [:headers "accept"])))
    (-> context
        (update-in [:response :body] content-renderer)
        (update-in [:response :headers] assoc "Content-Type" accept))))


(interceptor/defafter wrap-ring-response
  "Wrap the response value in a ring response"
  [{:keys [request response] :as context}]
  (update-in context [:response] ring-resp/response))


(interceptor/defafter save-body-val
  [context]
  (let [servlet-response (:servlet-response context)
        body (get-in context [:response :body])
        body-val-atom (-> servlet-response meta :body-val)]
    (if body-val-atom (reset! body-val-atom body)))
  context)
