(ns hypercrud-service.rest.datomic-crud
  (:refer-clojure :exclude [set])
  (:require [datomic.api :as d]
            [datomico.db :as db]
            [hypercrud-service.datomic-util :as datomic-util :refer [with-db-as-of]]
            [hypercrud-service.url-resolver :refer [resolve-url]]
            [hypercrud-service.rest.crud :as crud]
            [hypercrud-service.collection-json :as cj]))



(defn remove-nil-vals [record]
  (into {} (remove (comp nil? second) record)))


(defn tempid? [keyword]
  (= (namespace keyword) "tempid"))


(defn resolve-entity
  "Uses pedestal to resolve the URL to the backing entity; returns the eid which
  can be used as a value in another entity."
  [{:keys [href rel] :as cj-item-ref}]
  (if (tempid? rel)
    (d/tempid :db.part/user)
    (-> (resolve-url (.toString href))
        :body
        :record
        :db/id)))


(defn build-updated-entity [typeinfo record tx]
  ;; since there can be refs to other entities (like combo values), this is
  ;; done recursively. Use the typeinfo as a guide, turn all :refs into entities.
  ;; Note we can do this by resolving the :href with pedestal whenever we see a :ref.
  (let [id (:db/id record)]                                 ;id is not tracked in typeinfo
    (->> typeinfo
         (filter #(contains? record (:name %)))
         (map (fn [{:keys [name datatype typetag set] :as fieldinfo}]
                (let [build-or-resolve-ent (fn [val]
                                             (if (:data val) ;we have data - create or update
                                               (let [typeinfo (cj/typeinfo typetag tx)
                                                     record (into {:db/id (resolve-entity val)} (:data val))]
                                                 (build-updated-entity typeinfo record tx))
                                               (resolve-entity val))) ; just a lookup ref by href
                      val (if-let [val (get record name)]   ;nil is a valid value if the key was specified as nil
                            (cond
                              (and (= datatype :ref) (not set)) (build-or-resolve-ent val)
                              ;; :ref :set comes up as a map :rel->{:rel :href}; datomic wants #{entity}
                              (and (= datatype :ref) set) (into #{} (map build-or-resolve-ent val))
                              :else-not-ref val))]
                  [name val])))
         (into {:db/id id}))))


(defmethod crud/entity-create :default [typetag form]
  "TODO: follow CJ spec on success.
   201 Created; Location: url..."
  (let [tx (datomic-util/latest-tx)]
    (with-db-as-of tx
      (let [typeinfo (cj/typeinfo typetag tx)
            record (:template form)                         ;no db/id since not created yet
            new-ent (build-updated-entity typeinfo record tx)
            transaction (datomic-util/data-with-dbid [new-ent])
            effect (d/transact db/*connection* transaction)]
        (-> @effect :db-after d/basis-t d/t->tx)))))


(defmethod crud/entity-read :default [typetag id tx]
  (with-db-as-of tx
    (d/entity db/*db* (Long/parseLong id))))


(defmethod crud/entity-update :default [typetag eid tx form]
  (with-db-as-of tx                                         ;do we need to check latest-tx for concurrent modification?
    (let [record (-> (:template form) (assoc :db/id (Long/parseLong eid)))
          typeinfo (cj/typeinfo typetag tx)
          next-ent (build-updated-entity typeinfo record tx)
          effect (d/transact db/*connection* [next-ent])]
      (-> @effect :db-after d/basis-t d/t->tx))))






(comment
  (def ents
    (db/with-bound-or-latest-database
      (->> (datomic-util/qes '[:find ?e :where [?e :community/name]] db/*db*))))



  (db/with-bound-or-latest-database
    (-> (d/basis-t db/*db*)
        (d/t->tx)))

  (db/with-bound-or-latest-database
    (d/basis-t db/*db*))

  (def e (first ents))

  (->> ents first d/touch keys)

  (datomic-util/entity-last-modified-tx (first ents))

  (db/with-bound-or-latest-database
    (->> (datomic-util/qes '[:find ?id
                             :where [?id :db/ident ?ident]
                             [(namespace ?ident) ?ns]
                             [(= ?ns "community.orgtype")]]
                           db/*db*)
         (mapv d/touch))))


