(ns hypercrud-service.cache
  (:require [hypercrud-service.url-resolver :as url-resolver]
            [hypercrud-service.response :as response]))


;; Return the representation of the successors.
;; All graph traversals are in terms of the representation - same as on the client.
(defmulti hc-succ :type)
(defmethod hc-succ :default [hc-node] nil)


(defmethod hc-succ :object [{:keys [href rel links type data queries template] :as hc-node}]
  ;; Filter for the :ref children, return an array of their representation.
  ;; Children might be :type :set. Child links aren't included here - those
  ;; are the next depth.
  (let [successor-fieldinfos (->> template :data (filter #(= (:datatype %) :ref)))]
    (mapcat (fn [{:keys [name set] :as fieldinfo}]
              (let [fielddata (get data name)]
                (cond (nil? fielddata) []
                      set fielddata
                      :else-not-set [fielddata])))
            successor-fieldinfos)))


(defmethod hc-succ :collection [{:keys [href rel links type data queries template] :as hc-node}]
  ;; Successors are both children and :links. E.g. combobox options need to be resolved too.
  ;; And return a list - links and data are indexed by :rel so lose the keys
  (concat (vals data) (vals links)))


(defn succ [{:keys [href] :as hc-node}]
  ;; We're using the final hc-node representation as the type that we bf-traverse against.
  ;; This is so the server graph traversal mirrors the client graph traversal, where resolving
  ;; a :href is an edge traversal. This gives us a chance to run business logic for particular routes
  ;; e.g. what :links should come with this representation or did it get transformed? Direct navigation
  ;; of the datomic entity graph would lose any special logic. For example combo options.
  (let [hc-node (-> (url-resolver/resolve-url (.toString href)) :body response/represent)]
    (hc-succ hc-node)))


(defn mk-hc-cache [hc-node]
  (as-> hc-node $
        (loom.alg-generic/bf-traverse
          succ $
          :when (fn [neighbor predecessor depth] (< depth 5))
          :f (fn [node predecessor-map depth]
               (-> (url-resolver/resolve-url (.toString (:href node)))
                   :body response/represent)))
        (into {} (map vector (map :href $) $))))


(comment
  (def routes seattle-service.routes/routes)
  (binding [io.pedestal.http.route/*url-for* (io.pedestal.http.route/url-for-routes routes)]
    (as-> (url-resolver/resolve-url "/api/communities?tx=13194139534333") $
          (:body $) (response/represent $)
          (mk-hc-cache $) (doall $))))
