(ns hypercrud-service.collection-json
  (:require [datomic.api :as d]
            [datomico.db :as db]
            [loom.alg-generic]
            [hypercrud-service.datomic-util :refer [latest-tx with-db-as-of]]
            [hypercrud-service.pedestal-util :refer [url-for rel-for]]
            [hypercrud-service.dependencies :as dependencies]
            [hypercrud-service.response :as response]
            [hypercrud-service.url-resolver :as url-resolver])
  (:import (hypercrud_service.response ReadItemResponse
                                       ReadCollectionResponse
                                       UpdateItemResponse
                                       CreateItemResponse)))


(defmulti url-for-entity (fn [typetag record] typetag))
(defmulti rel-for-entity (fn [typetag record] typetag))
(defmulti typeinfo (fn [typetag tx & [record]] typetag))
(defmulti route-for-entity identity)   ;route-for-typetag


(defmethod url-for-entity :default [typetag record]
  {:pre [(not (nil? (:db/id record)))]}
  (url-for (route-for-entity typetag)
           :params {:id (:db/id record)
                    :tx (latest-tx)}))


(defmethod rel-for-entity :default [typetag record]
  {:pre [(not (nil? (:db/id record)))]}
  (keyword (str (:db/id record))))


(defn lazy-cj-item
  "NB: lazy-cj-item generates a unique :rel (sufficient for object identity, and usable in
hc-collections). However if we are a hc-object, lazy linking to the hc-object not as a collection,
we want the :rel to be the fieldinfo.name, not the object identity, this is because :rels are
primarily used for traversing the hc graph."
  [typetag record]
  {:href (url-for-entity typetag record)
   :rel (rel-for-entity typetag record)})


(defn options-for-typeinfo [typeinfo tx]
  (->> typeinfo
       (mapv (fn [{:keys [options] :as fieldinfo}]
               (if options
                 (let [route (:route options)
                       href (url-for route :params {:tx tx})]
                   (-> fieldinfo
                       (update-in [:options] assoc :href href)
                       (update-in [:options] assoc :rel (rel-for route))
                       (update-in [:options] dissoc :typetag)
                       (update-in [:options] dissoc :route)))
                 fieldinfo)))))


(defn mk-template [typeinfo tx]
  (if typeinfo
    {:data (options-for-typeinfo typeinfo tx)}
    nil))


;; Must render out vectors, because when reconstructed on the client,
;; the cursors need collections to be associative

(extend-type ReadItemResponse
  response/HypercrudRepresentation
  (represent [{:keys [links record typeinfo typetag tx] :as this}]
    (with-db-as-of tx
      {:href (url-for-entity typetag record)
       :rel (rel-for-entity typetag record)
       :type :object
       :data

       ;; map the typeinfo instead of the ent directly
       ;; if the field is an entity, lazy-cj-item
       ;; if the field is a keyword, if it's a keyword, that's an ident and convert to lazy-cj-item
       (->> typeinfo
            (map (fn [{:keys [name datatype set options] :as fieldinfo}]
                   (let [v (get record name)
                         v (cond
                             (and (= datatype :ref) (not set) (not (nil? v)))
                             (lazy-cj-item (:typetag options)
                                           (cond
                                             (keyword? v) (d/entity db/*db* v)
                                             :just-a-primitive v))
                             (and (= datatype :ref) set (not (nil? v)))
                             (->> v
                                  (map #(lazy-cj-item
                                         (:typetag options)
                                         (cond
                                           (keyword? %) (d/entity db/*db* %)
                                           :just-a-primitive %)))
                                  (into #{}))

                             :just-a-primitive
                             v)]
                     [name v])))
            (into {}))

       :links links
       :template (mk-template typeinfo tx)})))


(extend-type ReadCollectionResponse
  response/HypercrudRepresentation
  (represent [{:keys [collection-route links ents typeinfo typetag tx] :as this}]
    {:href (url-for collection-route :params {:tx tx})
     :rel (rel-for collection-route)
     :links links
     :type :collection
     :data (into {} (let [hc-nodes (map #(lazy-cj-item typetag %) ents)]
                      (map vector (map :rel hc-nodes) hc-nodes)))
     :queries []
     :template (mk-template typeinfo tx)}))


(extend-type UpdateItemResponse
  response/HypercrudRepresentation
  (represent [this]
    {:t (:t this)}))


(extend-type CreateItemResponse
  response/HypercrudRepresentation
  (represent [this]
    {:t (:t this)}))
