(ns mcp-tasks.prompts
  "Task management prompts"
  (:require
    [babashka.fs :as fs]
    [clojure.java.io :as io]
    [clojure.string :as str]
    [mcp-clj.mcp-server.prompts :as prompts]))

(defn- parse-frontmatter
  "Parse simple 'field: value' frontmatter from markdown text.

  Expects frontmatter delimited by '---' at start and end.
  Format example:
    ---
    description: Task description
    author: John Doe
    ---
    Content here...

  Returns a map with :metadata (parsed key-value pairs)
  and :content (remaining text).  If no valid frontmatter is found,
  returns {:metadata nil :content <original-text>}."
  [text]
  (if-not (str/starts-with? text "---\n")
    {:metadata nil :content text}
    (let [lines (str/split-lines text)
          ;; Skip first "---" line
          after-start (rest lines)
          ;; Find closing "---"
          closing-idx (first (keep-indexed
                               (fn [idx line]
                                 (when (= "---" (str/trim line)) idx))
                               after-start))]
      (if-not closing-idx
        ;; No closing delimiter, treat as no frontmatter
        {:metadata nil :content text}
        (let [metadata-lines (take closing-idx after-start)
              content-lines (drop (inc closing-idx) after-start)
              ;; Parse "key: value" pairs
              metadata (reduce
                         (fn [acc line]
                           (if-let [[_ k v] (re-matches
                                              #"([^:]+):\s*(.*)"
                                              line)]
                             (assoc acc (str/trim k) (str/trim v))
                             acc))
                         {}
                         metadata-lines)
              content (str/join "\n" content-lines)]
          {:metadata (when (seq metadata) metadata)
           :content content})))))

(defn- discover-prompt-files
  "Discover .md prompt files in a directory.

  Takes a File object and returns a sorted vector of filenames without
  .md extension.  Returns empty vector if directory doesn't exist."
  [dir]
  (if (fs/exists? dir)
    (->> (fs/list-dir dir)
         (filter #(and (fs/regular-file? %)
                       (str/ends-with? (str (fs/file-name %)) ".md")))
         (map #(str/replace (str (fs/file-name %)) #"\.md$" ""))
         sort
         vec)
    []))

(defn discover-categories
  "Discover task categories by reading prompts subdirectory from resolved tasks dir.

  Takes config containing :resolved-tasks-dir. Returns a sorted vector of
  category names (filenames without .md extension) found in the prompts
  subdirectory."
  [config]
  (let [resolved-tasks-dir (:resolved-tasks-dir config)
        prompts-dir (str resolved-tasks-dir "/prompts")]
    (discover-prompt-files prompts-dir)))

(defn- read-task-prompt-text
  "Generate prompt text for reading the next task from a category.
  Config parameter included for API consistency but not currently used."
  [_config _category]
  "- Read the file .mcp-tasks/tasks.ednl

- Find the first incomplete task (marked with `- [ ]`) You can use the
  `select-tasks` tool with `:limit 1` to retrieve the next task without
  executing it.

- Show the task description
")

(defn- default-prompt-text
  "Generate default execution instructions for a category."
  []
  (slurp (io/resource "prompts/default-prompt-text.md")))

(defn- complete-task-prompt-text
  "Generate prompt text for completing and tracking a task.

  Conditionally includes git commit instructions based on
  config :use-git? value."
  [config _category]
  (let [base-text
        "- Mark the completed task as complete using the `complete-task` tool.
  This will update the task's status to :closed and move it from
  .mcp-tasks/tasks.ednl to .mcp-tasks/complete.ednl.

- Summarise any deviations in the execution of the task, compared to the task
  spec.
"
        git-text "\n- Commit the task tracking changes in the .mcp-tasks git repository\n"]
    (if (:use-git? config)
      (str base-text git-text)
      base-text)))

(defn- read-prompt-instructions
  "Read custom prompt instructions from prompts subdirectory in resolved tasks dir.

  Takes config containing :resolved-tasks-dir and category name.

  Returns a map with :metadata and :content keys if the file exists, or
  nil if it doesn't.

  The :metadata key contains parsed frontmatter (may be nil),
  and :content contains the prompt text with frontmatter stripped."
  [config category]
  (let [resolved-tasks-dir (:resolved-tasks-dir config)
        prompt-file (str resolved-tasks-dir "/prompts/" category ".md")]
    (when (fs/exists? prompt-file)
      (parse-frontmatter (slurp prompt-file)))))

(defn create-prompts
  "Generate MCP prompts for task categories.

  Reads prompt instructions from prompts/<category>.md files in resolved tasks
  directory if they exist, otherwise uses default prompt text. Each prompt
  provides complete workflow including task lookup, execution
  instructions, and completion steps.

  Returns a vector of prompt maps suitable for registration with the MCP
  server."
  [config categories]
  (vec
    (for [category categories]
      (let [prompt-data (read-prompt-instructions config category)
            metadata (:metadata prompt-data)
            custom-content (:content prompt-data)
            execution-instructions (or custom-content (default-prompt-text))
            prompt-text (str "Please complete the next "
                             category
                             " task following these steps:\n\n"
                             (read-task-prompt-text config category)
                             execution-instructions
                             (complete-task-prompt-text config category))
            description (or (get metadata "description")
                            (format
                              "Execute the next %s task from .mcp-tasks/tasks.ednl"
                              category))]
        (prompts/valid-prompt?
          {:name (str "next-" category)
           :description description
           :messages [{:role "user"
                       :content {:type "text"
                                 :text prompt-text}}]})))))

(defn category-descriptions
  "Get descriptions for all discovered categories.

  Returns a map of category name to description string. Categories
  without custom prompts or without description metadata will have a
  default description."
  [config]
  (let [categories (discover-categories config)]
    (into {}
          (for [category categories]
            (let [prompt-data (read-prompt-instructions config category)
                  metadata (:metadata prompt-data)
                  description (or (get metadata "description")
                                  (format "Tasks for %s category" category))]
              [category description])))))

(defn prompts
  "Generate all task prompts by discovering categories.

  Accepts config parameter to conditionally include git instructions in prompts.
  Uses :resolved-tasks-dir from config to locate prompts directory.

  Returns a map of prompt names to prompt definitions, suitable for registering
  with the MCP server."
  [config]
  (let [categories (discover-categories config)
        prompt-list (create-prompts config categories)]
    (into {} (map (fn [p] [(:name p) p]) prompt-list))))

;; Story prompt utilities

(defn- list-builtin-story-prompts
  "List all built-in story prompts available in resources.

  Returns a sequence of prompt names (without .md extension) found in
  resources/story/prompts directory."
  []
  (when-let [prompts-url (io/resource "story/prompts")]
    (discover-prompt-files (io/file (.toURI prompts-url)))))

(defn get-story-prompt
  "Get a story prompt by name, with file override support.

  Checks for override file at `.mcp-tasks/story/prompts/<name>.md` first.
  If not found, falls back to built-in prompt from resources/story/prompts.

  Returns a map with:
  - :name - the prompt name
  - :description - from frontmatter
  - :content - the prompt text (with frontmatter stripped)

  Returns nil if prompt is not found in either location."
  [prompt-name]
  (let [override-file (str ".mcp-tasks/story/prompts/" prompt-name ".md")]
    (if (fs/exists? override-file)
      (let [file-content (slurp override-file)
            {:keys [metadata content]} (parse-frontmatter file-content)]
        {:name prompt-name
         :description (get metadata "description")
         :content content})
      (when-let [resource-path (io/resource
                                 (str "story/prompts/" prompt-name ".md"))]
        (let [file-content (slurp resource-path)
              {:keys [metadata content]} (parse-frontmatter file-content)]
          {:name prompt-name
           :description (get metadata "description")
           :content content})))))

(defn list-story-prompts
  "List all available story prompts.

  Returns a sequence of maps with :name and :description for each available
  story prompt, including both built-in prompts and file overrides."
  []
  (let [builtin-prompts (for [prompt-name (list-builtin-story-prompts)
                              :let [prompt (get-story-prompt prompt-name)]
                              :when prompt]
                          {:name (:name prompt)
                           :description (:description prompt)})
        story-dir ".mcp-tasks/story/prompts"
        override-prompts (when (fs/exists? story-dir)
                           (for [file (fs/list-dir story-dir)
                                 :when (and (fs/regular-file? file)
                                            (str/ends-with?
                                              (str (fs/file-name file))
                                              ".md"))]
                             (let [name (str/replace
                                          (str (fs/file-name file))
                                          #"\.md$" "")
                                   {:keys [metadata]} (parse-frontmatter
                                                        (slurp (str file)))]
                               {:name name
                                :description (get metadata "description")})))
        all-prompts (concat override-prompts builtin-prompts)
        seen (atom #{})]
    (for [prompt all-prompts
          :when (not (contains? @seen (:name prompt)))]
      (do
        (swap! seen conj (:name prompt))
        prompt))))

(defn- parse-argument-hint
  "Parse argument-hint from frontmatter metadata.

  The argument-hint format uses angle brackets for required arguments and
  square brackets for optional arguments:
  - <arg-name> - required argument
  - [arg-name] - optional argument
  - [...] or [name...] - variadic/multiple values

  Example: '<story-name> [additional-context...]'

  Returns a vector of argument maps with :name, :description,
  and :required keys."
  [metadata]
  (when-let [hint (get metadata "argument-hint")]
    (vec
      (for [token (re-seq #"<([^>]+)>|\[([^\]]+)\]" hint)
            :let [[_ required optional] token
                  arg-name (or required optional)
                  is-required (some? required)
                  is-variadic (str/ends-with? arg-name "...")
                  clean-name (if is-variadic
                               (str/replace arg-name #"\.\.\.$" "")
                               arg-name)
                  description (cond
                                is-variadic (format
                                              "Optional additional %s (variadic)"
                                              clean-name)
                                is-required (format
                                              "The %s (required)"
                                              (str/replace clean-name "-" " "))
                                :else (format
                                        "Optional %s"
                                        (str/replace clean-name "-" " ")))]]
        {:name clean-name
         :description description
         :required is-required}))))

(defn- append-management-instructions
  "Append branch and worktree management instructions to prompt content.

  Conditionally appends management instruction files based on config flags.
  Only appends to prompts matching target-prompt-name.

  Parameters:
  - content: Base prompt content string
  - prompt-name: Name of the current prompt being processed
  - target-prompt-name: Prompt that should receive the instructions
  - config: Config map with :branch-management? and :worktree-management? flags

  Returns the content with instructions appended if conditions match."
  [content prompt-name target-prompt-name config]
  (cond-> content
    (and (= prompt-name target-prompt-name)
         (:branch-management? config))
    (str "\n\n" (slurp (io/resource "prompts/branch-management.md")))
    (and (= prompt-name target-prompt-name)
         (:worktree-management? config))
    (str "\n\n" (slurp (io/resource "prompts/worktree-management.md")))))

(defn story-prompts
  "Generate MCP prompts from story prompt vars in mcp-tasks.story-prompts.

  For execute-story-task prompt, tailors content based on
  config :branch-management?.

  Returns a map of prompt names to prompt definitions, suitable for registering
  with the MCP server."
  [config]
  (require 'mcp-tasks.story-prompts)
  (let [ns (find-ns 'mcp-tasks.story-prompts)
        prompt-vars (->> (ns-publics ns)
                         vals
                         (filter (fn [v] (string? @v))))]
    (into {}
          (for [v prompt-vars]
            (let [prompt-name (name (symbol v))
                  prompt-content @v
                  {:keys [metadata content]} (parse-frontmatter prompt-content)
                  ;; Tailor execute-story-task content based on config
                  tailored-content
                  (append-management-instructions
                    content
                    prompt-name
                    "execute-story-task"
                    config)
                  description (or (get metadata "description")
                                  (:doc (meta v))
                                  (format "Story prompt: %s" prompt-name))
                  arguments (parse-argument-hint metadata)]

              [prompt-name
               (prompts/valid-prompt?
                 (cond-> {:name prompt-name
                          :description description
                          :messages [{:role "user"
                                      :content {:type "text"
                                                :text tailored-content}}]}
                   (seq arguments) (assoc :arguments arguments)))])))))

(defn task-execution-prompts
  "Generate MCP prompts for general task execution workflows.

  Discovers prompt files from resources/prompts/ directory, excluding:
  - Category instruction files (simple.md, medium.md, etc.)
  - Internal files (default-prompt-text.md)

  Returns a map of prompt names to prompt definitions."
  [config]
  (when-let [prompts-url (io/resource "prompts")]
    (let [prompts-dir (io/file (.toURI prompts-url))
          all-prompts (discover-prompt-files prompts-dir)
          ;; Get category names to filter out
          categories (set (discover-categories config))
          ;; Filter out category instruction files and internal files
          excluded-names (conj categories "default-prompt-text")
          task-prompts (remove excluded-names all-prompts)
          prompts-data (for [prompt-name task-prompts
                             :let [resource-path (io/resource
                                                   (str
                                                     "prompts/"
                                                     prompt-name
                                                     ".md"))]
                             :when resource-path]
                         (let [file-content (slurp resource-path)
                               {:keys [metadata content]} (parse-frontmatter
                                                            file-content)
                               ;; Tailor execute-task content based on config
                               tailored-content
                               (append-management-instructions
                                 content
                                 prompt-name
                                 "execute-task"
                                 config)
                               description (or (get metadata "description")
                                               (format
                                                 "Task execution prompt: %s"
                                                 prompt-name))
                               arguments (parse-argument-hint metadata)]
                           [prompt-name
                            (prompts/valid-prompt?
                              (cond-> {:name prompt-name
                                       :description description
                                       :messages [{:role "user"
                                                   :content {:type "text"
                                                             :text tailored-content}}]}
                                (seq arguments) (assoc
                                                  :arguments
                                                  arguments)))]))]
      (into {} prompts-data))))

(defn category-prompt-resources
  "Generate MCP resources for category prompt files.

  Discovers all available categories and creates a resource for each category's
  prompt file found in prompts/<category>.md within resolved tasks directory.

  Each resource has:
  - :uri \"prompt://category-<category>\"
  - :name \"<category> category instructions\"
  - :description from frontmatter or default
  - :mimeType \"text/markdown\"
  - :text content with frontmatter preserved

  Missing files are gracefully skipped (not included in result).

  Returns a vector of resource maps."
  [config]
  (let [categories (discover-categories config)]
    (->> categories
         (keep (fn [category]
                 (when-let [prompt-data (read-prompt-instructions
                                          config
                                          category)]
                   (let [metadata (:metadata prompt-data)
                         content (:content prompt-data)
                         description (or (get metadata "description")
                                         (format
                                           "Execution instructions for %s category"
                                           category))
                         ;; Reconstruct frontmatter if metadata exists
                         frontmatter (when metadata
                                       (let [lines (keep (fn [[k v]]
                                                           (when v
                                                             (str k ": " v)))
                                                         metadata)]
                                         (when (seq lines)
                                           (str "---\n"
                                                (str/join "\n" lines)
                                                "\n---\n"))))
                         ;; Include frontmatter in text if it exists
                         text (str frontmatter content)]
                     {:uri (str "prompt://category-" category)
                      :name (str category " category instructions")
                      :description description
                      :mimeType "text/markdown"
                      :text text}))))
         vec)))
