(ns honeyeql-postgres.core
  (:require [honeyeql.meta-data :as heql-md]
            [next.jdbc.sql :as jdbc-sql]
            [honeyeql.core :as heql]
            [honeysql.format :as fmt]
            [honeysql.core :as sql]
            [honeyeql.debug :refer [trace>]]))

(defmethod ^{:private true} fmt/format-clause :left-join-lateral [[_ join-groups] _]
  (clojure.string/join
   " "
   (map (fn [[join-group alias]]
          (str "LEFT JOIN LATERAL " (fmt/to-sql join-group) " AS \"" (name alias) "\" ON TRUE"))
        join-groups)))

(fmt/register-clause! :left-join-lateral 135)

(defmethod heql-md/get-db-config "PostgreSQL" [_]
  {:schema             {:default "public"
                        :ignore  #{"information_schema" "pg_catalog" "pg_toast"}}
   :foreign-key-suffix "_id"})

; Ref: https://www.postgresql.org/docs/current/datatype.html
(defn- pg-type->col-type [type-name]
  (case type-name
    ("bigint" "int8" "bigserial" "serial8") :attr.type/big-integer
    ("bit" "bit varying")  :attr.type/string
    ("boolean" "bool")     :attr.type/boolean
    ("character" "char" "character varying" "varchar" "citext" "bpchar") :attr.type/string
    "date" :attr.type/date
    "decimal" :attr.type/decimal
    ("real" "float4" "float8") :attr.type/float
    "double precision" :attr.type/double
    "inet" :attr.type/ip-address
    ("integer" "int" "int2" "int4") :attr.type/integer
    "interval" :attr.type/time-span
    ("json" "jsonb") :attr.type/json
    ("macaddr" "macaddr8") :attr.type/string
    ("money" "numeric") :attr.type/decimal
    ("smallint" "smallserial" "serial" "serial2" "serial4") :attr.type/integer
    "text" :attr.type/string
    ; https://jdbc.postgresql.org/documentation/head/8-date-time.html
    ("time" "time without time zone") :attr.type/time
    ("timetz" "time with time zone") :attr.type/offset-time
    ("timestamp" "timestamp without time zone") :attr.type/data-time
    ("timestamptz" "timestamp with time zone") :attr.type/offset-date-time
    "uuid" :attr.type/uuid
    "xml" :attr.type/json
    :attr.type/unknown))

(defmethod heql-md/derive-attr-type "PostgreSQL" [_ column-meta-data]
  (pg-type->col-type (:type_name column-meta-data)))

(defn- result-set-hql [hql]
  {:with   [[:rs hql]]
   :from   [[{:select [:*]
              :from   [:rs]} :rs]]
   :select [(sql/raw "coalesce (json_agg(\"rs\"), '[]')::character varying as result")]})

(defrecord PostgresAdapter [db-spec heql-config heql-meta-data]
  heql/DbAdapter
  (db-spec [pg-adapter]
    (:db-spec pg-adapter))
  (meta-data [pg-adapter]
    (:heql-meta-data pg-adapter))
  (config [pg-adapter]
    (:heql-config pg-adapter))
  (merge-config [pg-adapter config-to-override]
    (update pg-adapter :heql-config merge config-to-override))
  (execute [pg-adapter hsql]
    (let [query (-> (result-set-hql hsql)
                    (sql/format :quoting :ansi)
                    (trace> :sql))]
      (-> (:db-spec pg-adapter)
          (jdbc-sql/query query)
          first
          :result))))

(defn initialize
  ([db-spec]
   (initialize db-spec heql/default-heql-config))
  ([db-spec heql-config]
   (let [heql-cfg (merge heql/default-heql-config heql-config)
         heql-meta-data (heql-md/fetch db-spec)]
     (map->PostgresAdapter {:db-spec db-spec
                            :heql-config heql-cfg
                            :heql-meta-data heql-meta-data}))))