(ns claspy.core
  (:require [clojure.edn :as edn]
            [clojure.walk :as walk]))

(defn list-like? [form]
  (or (list? form)
      (= clojure.lang.LazySeq (type form))))

(declare ast-expr)

(defn body-expr [body]
  (condp = (count body)
    0 '(Name "None" (Load))
    1 (ast-expr (first body))
    (list 'Subscript
          (list 'List (mapv ast-expr body) '(Load))
          '(Index (Num -1))
          '(Load))))

(defn call-expr [f args]
  (list 'Call (ast-expr f)
        (mapv ast-expr args)
        [] nil nil))

(defn compare-expr
  ([_ x] '(Name "True" (Load)))
  ([op x & xs]
   (list 'Compare
         (ast-expr x)
         (vec (repeat (count xs) (list op)))
         (mapv ast-expr xs))))

(defn def-expr [nm value]
  (with-meta
    (list 'Assign
          [(list 'Name (namespace-munge nm) '(Store))]
          (ast-expr value))
    {:statement true}))

(defn dot-expr [obj & args]
  (cond
    (and (= 1 (count args))
         (symbol? (first args))
         (= \- (-> args first name first)))
    (list 'Attribute (ast-expr obj) (subs (name (first args)) 1) '(Load))
    (list? (first args))
    (list 'Call
          (list 'Attribute (ast-expr obj) (name (ffirst args)) '(Load))
          (mapv ast-expr (rest (first args)))
          [] nil nil)
    :else
    (list 'Call
          (list 'Attribute (ast-expr obj) (name (first args))' (Load))
          (mapv ast-expr (rest args))
          [] nil nil)))

(defn fn-expr [[arg-names & body]]
  (list 'Lambda
        (list 'arguments (mapv #(list 'Name (name %) '(Param)) arg-names) nil nil [])
        (body-expr body)))

(defn if-expr
  ([condition then] (if-expr condition then nil))
  ([condition then else]
   (let [arg (name (gensym "if_arg"))]
     (list 'IfExp
           (list 'Call (list 'Lambda (list 'arguments [(list 'Name arg '(Param))] nil nil [])
                             (list 'BoolOp '(And) [(list 'Compare (list 'Name arg '(Load)) '[(IsNot)] '[(Name "None" (Load))])
                                                   (list 'Compare (list 'Name arg '(Load)) '[(IsNot)] '[(Name "False" (Load))])]))
                 [(ast-expr condition)] [] nil nil)
           (ast-expr then)
           (ast-expr else)))))

(defn get-expr
  ([obj k]
   (list 'Subscript
         (ast-expr obj)
         (list 'Index (ast-expr k))
         '(Load)))
  ([obj k not-found]
   (list 'IfExp
         (list 'Compare (ast-expr k) '[(In)] [(ast-expr obj)])
         (get-expr obj k)
         (ast-expr not-found))))

(defn infix-expr
  ([_ x] (ast-expr x))
  ([op x y]
   (list 'BinOp (ast-expr x) (list op) (ast-expr y)))
  ([op x y & xs]
   (list 'BinOp
         (apply infix-expr op x y (butlast xs))
         (list op)
         (ast-expr (last xs)))))

(defn let-expr [bindings & body]
  (condp = (count bindings)
    0 (body-expr body)
    1 (throw (Exception. (str "let* binding requires even number of forms")))
    (let [[sym value] (take 2 bindings)]
      (list 'Call
            (list 'Lambda
                  (list 'arguments [(list 'Name (name sym) '(Param))] nil nil [])
                  (apply let-expr (drop 2 bindings) body))
            [(ast-expr value)]
            [] nil nil))))

(defn nil?-expr [x]
  (list 'Compare (ast-expr x) '[(Is)] '[(Name "None" (Load))]))

(defn not-expr [x]
  (list 'UnaryOp '(Not) (ast-expr x)))

(defn require-expr [& imports]
  (with-meta
    (sequence cat
      (for [im imports
            :let [[nm & opts] (if (vector? im) im [im])
                  opts (apply hash-map opts)]]
        (cond-> []
          (opts :as)        (conj (list 'Import [(list 'alias (namespace-munge nm) (namespace-munge (opts :as)))]))
          (nil? (opts :as)) (conj (list 'Import [(list 'alias (namespace-munge nm) nil)]))
          (opts :refer)     (conj (list 'ImportFrom (namespace-munge nm)
                                        (mapv #(list 'alias (namespace-munge %) nil) (opts :refer)))))))
    {:multi-statement true}))

(defn set-expr [target value]
  (with-meta
    (cond
      (symbol? target)
      (list 'Assign
            [(list 'Name (name target) '(Store))]
            (ast-expr value))
      (and (list-like? target) (= 'slice (first target)))
      (let [[_ obj low high step] target
            build #(if (nil? %) nil (ast-expr %))]
        (list 'Assign
              [(list 'Subscript (ast-expr obj)
                     (list 'Slice (build low) (build high) (build step))
                     '(Store))]
              (ast-expr value)))
      :else
      (throw (Exception. (str "don't know how to set " target))))
    {:statement true}))

(defn sub-expr
  ([x]
   (list 'UnaryOp '(USub) (ast-expr x)))
  ([x & xs]
   (apply infix-expr 'Sub x xs)))

(def special-forms
  {'if #'if-expr
   'dec #(infix-expr 'Sub % 1)
   'def #'def-expr
   'get #'get-expr
   'fn* #'fn-expr
   'inc #(infix-expr 'Add % 1)
   'let* #'let-expr
   'nil? #'nil?-expr
   'not #'not-expr
   'require #'require-expr
   'set! #'set-expr
   '. #'dot-expr
   '+ (partial infix-expr 'Add)
   '- #'sub-expr
   '* (partial infix-expr 'Mult)
   '/ (partial infix-expr 'Div)
   '> (partial compare-expr 'Gt)
   '>= (partial compare-expr 'GtE)
   '< (partial compare-expr 'Lt)
   '<= (partial compare-expr 'LtE)})

(defn expand-list [[f & args]]
  (if-let [expr-fn (special-forms f)]
    (apply expr-fn args)
    (call-expr f args)))

(defmulti ast-expr type)

(defmethod ast-expr nil [_]
  '(Name "None" (Load)))

(defmethod ast-expr java.lang.Boolean [b]
  (if b
    '(Name "True" (Load))
    '(Name "False" (Load))))

(defmethod ast-expr java.lang.String [s]
  (list 'Str s))

(defmethod ast-expr java.lang.Long [form]
  (list 'Num form))

(defmethod ast-expr clojure.lang.Keyword [k]
  (list 'Str (name k)))

(defmethod ast-expr clojure.lang.Symbol [sym]
  (if-let [nmsp (namespace sym)]
    (list 'Attribute (list 'Name nmsp '(Load)) (name sym) '(Load))
    (list 'Name (name sym) '(Load))))

(defmethod ast-expr clojure.lang.LazySeq [lst]
  (expand-list lst))

(defmethod ast-expr clojure.lang.PersistentList [lst]
  (expand-list lst))

(defmethod ast-expr clojure.lang.PersistentVector [values]
  (list 'List (mapv ast-expr values) '(Load)))

(defmethod ast-expr clojure.lang.PersistentArrayMap [m]
  (list 'Dict
        (mapv ast-expr (keys m))
        (mapv ast-expr (vals m))))

(defn ast [form]
  (let [form (walk/macroexpand-all form)
        body (ast-expr form)
        {:keys [multi-statement statement]} (meta body)]
    (cond
      multi-statement (list 'Module (vec body))
      statement (list 'Module [body])
      :else (list 'Expression body))))

(comment

  (def code '(not true))

  #spy/d (ast code)

  #spy/d (walk/macroexpand-all code)

  )
