(ns astrometrics.orbit
  (:require [math.core :as math]))

(defn polar->cartesian [r theta]
  [(* r (math/cos theta))
   (* r (math/sin theta))
   0])

(defn mean-anomaly [period mean-anomaly-at-epoch time-since-epoch]
  (-> time-since-epoch
    (* math/tau)
    (/ period)
    (+ mean-anomaly-at-epoch)))

(defn newton [f f' x0 tolerance]
  (loop [x x0]
    (let [x-next (- x (/ (f x) (f' x)))]
      (if (< (diff x x-next) tolerance)
        x-next
        (recur x-next)))))

(defn eccentric-anomaly [mean-anomaly eccentricity]
  (let [f #(- % (* eccentricity (math/sin %)) mean-anomaly)
        f' #(- 1 (* eccentricity (math/cos %)))]
    (newton f f' mean-anomaly 1e-7)))

(defn true-anomaly [eccentric-anamoly eccentricity]
  (* 2 (math/atan (* (math/sqrt (+ 1 eccentricity)) (math/sin (/ eccentric-anamoly 2)))
                  (* (math/sqrt (- 1 eccentricity)) (math/cos (/ eccentric-anamoly 2))))))

(defn distance-from-sun [semi-major-axis eccentricity true-anomaly]
  (/ (* semi-major-axis (- 1 (math/sqr eccentricity)))
     (+ 1 (* eccentricity (math/cos true-anomaly)))))

(defn position
  "Calculates the Cartesian position of a celestial body given its orbital
   elements and a time. All times should be given in seconds, distances in
   meters, and angles in radians."
  [{:keys [period
           mean-anomaly-at-epoch
           time-of-epoch
           eccentricity
           semi-major-axis
           longitude-of-ascending-node
           argument-of-perihelion]}
   time-of-observation]
  (let [mean-anom (mean-anomaly period mean-anomaly-at-epoch (- time-of-observation time-of-epoch))
        eccentric-anom (eccentric-anomaly mean-anom eccentricity)
        true-anom (true-anomaly eccentric-anom eccentricity)
        dist-from-sun (distance-from-sun semi-major-axis eccentricity true-anom)]
    (reduce
      (fn [v m]
        (matrix/matrix-transform m v))
      (polar->cartesian dist-from-sun true-anom)
      [(matrix/rotate-z argument-of-perihelion)
       (matrix/rotate-x inclination)
       (matrix/rotate-z longitude-of-ascending-node)])))
