(ns org.euandreh.http.interceptors
  (:require [buddy.auth.middleware :as auth.middleware]
            [clojure.spec.alpha :as s]
            [io.pedestal.http.body-params :as body-params]
            [io.pedestal.interceptor :as i]
            [io.pedestal.interceptor.chain :as i.chain]
            [io.pedestal.log :as log]
            [org.euandreh.http.exception :as exception]
            [org.euandreh.http.protocols.crypto :as protocols.crypto]
            [org.euandreh.misc.core :as misc]))

(defn- valid-query?-fn
  [{{:keys [transit-params edn-params]} :request :as context}]
  (let [params (or transit-params edn-params)]
    (if-let [error-reason (cond
                            (not (map? params))         "Body isn't an edn map."
                            (not (vector? (:q params))) "[:q] in body isn't an edn vector.")]
      (exception/invalid-input! #:exception{:reason     ::bad-query
                                            :reason-str error-reason
                                            :body       params})
      (assoc-in context [:request :query] params))))

(def valid-query?
  "Interceptor for validating the body payload as a Fulcro query.

   The body must come as either edn or transit/json, and have the minimal structure as follows:
   {:q [...]}

   The actual query must be under the `:q` key."
  (i/interceptor
   {:name  ::valid-query?
    :enter valid-query?-fn}))

(def body-params
  "edn and transit/json parsing capable interceptor. Custom edn tags (present in `*data-readers*`) are supported."
  (body-params/body-params
   (body-params/default-parser-map
     :edn-options {:readers (merge default-data-readers *data-readers*)})))

(defn- terminate-with
  "Helper function for always terminating and giving some response to the client."
  [context response]
  (i.chain/terminate (assoc context :response response)))

(defn- catch-fn
  [context exception]
  (let [{:exception/keys [status-code reason-str reason]} (ex-data exception)]
    (log/error :exception exception)
    (terminate-with context
                    {:status (or status-code 500)
                     :body   (misc/assoc-if {:error (or (.getMessage exception)
                                                        "Server Error")}
                                            :reason reason
                                            :reason-str reason-str)})))

(def catch
  "Pedestal terminating interceptor for handling `:error` cases.

   It looks for `ex-data` information inside the received exception to decide what to print/log and to pick the proper HTTP status code (defaults to 500 when missing)."
  (i/interceptor
   {:name  ::catch
    :error catch-fn}))

(defn- authentication-interceptor-fn
  [context]
  (let [crypto (get-in context [:request :components :crypto])]
    (update context :request auth.middleware/authentication-request (protocols.crypto/backend crypto))))

(def authentication-interceptor
  "Port of buddy-auth's wrap-authentication middleware."
  (i/interceptor
   {:name  ::authenticate
    :enter authentication-interceptor-fn}))

;; Specs

(s/def ::simple-query-join
  (s/map-of (s/alt :simple-join keyword?
                   :parameterized-join (s/tuple keyword? any?))
            ::simple-query
            :min-count 1
            :max-count 1))

(s/def ::simple-query-fragment
  (s/alt :keyword keyword?
         :join ::simple-query-join))

(s/def ::simple-query
  (s/alt ::toplevel-query (s/* ::simple-query-fragment)))
