(ns org.euandre.om-auth.db.datomic.user
  (:require [org.euandre.misc.exception :as exception]))

(defn lookup-by-id [user-id db]
  (datomic.api/entity db [:user/id user-id]))

(defn lookup-by-id! [user-id db]
  (if-let [user (lookup-by-id user-id db)]
    user
    (exception/not-found! {::exception/reason     ::user-by-id-not-found
                           ::exception/reason-str "User not found."})))

(defn lookup-by-email [email db]
  (datomic.api/entity db [:user/emails email]))

(defn lookup-by-email! [email db]
  (if-let [user (lookup-by-email (str email) db)]
    user
    (exception/not-found! {::exception/reason     ::user-by-email-not-found
                           ::exception/reason-str "User not found"})))

(defn create-refresh-token-for-user! [user-id token-id hashed-refresh-token refresh-before datomic]
  (let [add-refresh-token-transaction {:user/id user-id
                                       :user/refresh-tokens
                                       {:refresh-token/id token-id
                                        :refresh-token/refresh-before refresh-before
                                        :refresh-token/token-hash     hashed-refresh-token}}]
    @(datomic.api/transact datomic [add-refresh-token-transaction])))

(defn revoke-refresh-token-by-hash! [hashed-refresh-token datomic]
  (let [retract-token-transaction [:db.fn/retractEntity [:refresh-token/token-hash hashed-refresh-token]]]
    @(datomic.api/transact datomic [retract-token-transaction])))

(defn dissoc-email! [user-id email datomic]
  (let [retract-email-transaction [:db/retract [:user/id user-id] :user/emails email]]
    @(datomic.api/transact datomic [retract-email-transaction])))

(defn assoc-email! [user-id email datomic]
  (let [user-with-new-email {:user/id user-id
                             :user/emails email}]
    @(datomic.api/transact datomic [user-with-new-email])))

(defn create-user! [user datomic]
  @(datomic.api/transact datomic [user]))

(defn change-password! [user-id old-password-hash new-password-hash datomic]
  (let [add-password-transaction {:user/id            user-id
                                  :user/password-hash new-password-hash}]
    @(datomic.api/transact datomic [add-password-transaction])))

(defn revoke-refresh-tokens-by-id! [ids datomic]
  (let [transactions (mapcat (fn [id]
                               [[:db.fn/retractEntity [:refresh-token/id id]]
                                {:blacklist/token id}])
                             ids)]
    @(datomic.api/transact datomic transactions)))

(defn lookup-blacklisted-token [token-id db]
  (datomic.api/entity db [:blacklist/token token-id]))

(defn tokens-id-by-user-id [user-id db]
  (let [result-tuples (datomic.api/q '{:find [?token-id]
                                       :in   [$ ?user-id]
                                       :where
                                       [[?eid :user/id ?user-id]
                                        [?eid :user/refresh-tokens ?token]
                                        [?token :refresh-token/id ?token-id]]}
                                     db
                                     user-id)]
    (into #{} (map first result-tuples))))

(defn lookup-confirmation-id [confirmation-id db]
  (datomic.api/entity db [:user/confirmation-ids confirmation-id]))

(defn confirm-user! [user-id datomic]
  (let [confirmation-transaction {:user/id user-id
                                  :user/confirmation-status :confirmation/confirmed}]
    @(datomic.api/transact datomic [confirmation-transaction])))

(defn register-reset-password-token! [user-id reset-password-id datomic]
  (let [reset-password-transaction {:user/reset-password-ids reset-password-id
                                    :user/id                 user-id}]
    @(datomic.api/transact datomic [reset-password-transaction])))

(defn lookup-reset-password-id! [reset-password-id db]
  (if-let [user (datomic.api/entity db [:user/reset-password-ids reset-password-id])]
    user
    (exception/not-found! {::exception/reason     ::user-by-reset-password-id-not-found
                           ::exception/reason-str "Bad reset password link."})))

(defn update-password-hash! [id hashed-password datomic]
  (let [update-password-transaction {:user/id            id
                                     :user/password-hash hashed-password}]
    @(datomic.api/transact datomic [update-password-transaction])))
