(ns org.euandre.om-auth.logic.auth
  (:require [clj-time.core :as time]
            [org.euandre.om-auth.db.datomic.user :as db.user]
            [org.euandre.misc.edn.email :as email]
            [buddy.auth.backends :as auth.backends]
            [buddy.hashers :as hashers]
            [org.euandre.misc.exception :as exception]))

(def ^:private +jwt-signing-alg+
  "The algorithm used for signing JWT tokens."
  :hs512)

(def ^:private +trusted-hashing-alg+
  "Whitelisted hashing algorithm to be used in verification."
  :pbkdf2+sha512)

(def ^:private default-scopes
  "The default scopes that a newly-registered user has."
  #{:scope/guest})

(defn hash-password
  "Simple (pure) password hashing."
  [password]
  (hashers/derive password {:alg +trusted-hashing-alg+}))

(defn verify-password
  "Verify that a password matches it's hash."
  [password hash]
  (hashers/check password hash {:limit #{+trusted-hashing-alg+}}))

(defn now->refresh-token-expiration
  "Add 30 days to =now=.

  A refresh token is valid for 30 days from the day it was issued."
  [now]
  (.toDate (time/plus now (time/days 30))))

(defn new-wrapped-jwt-payload [user-id scopes refresh-token now token-id {:keys [jwt-signing-secret issuer-url]}]
  (let [refresh-before (now->refresh-token-expiration now)
        expiration     (.toDate (time/plus now (time/days 7)))
        ;; Reference: https://tools.ietf.org/html/rfc7519#section-4.1
        claims         {:iss    issuer-url    ;; Issuer
                        :sub    (str user-id) ;; Subject
                        :exp    expiration    ;; Expiration Time
                        :iat    (.toDate now) ;; Issued At
                        :jti    token-id      ;; JWT ID
                        :scopes scopes}]
    {:user/refresh-before refresh-before
     :user/refresh-token  refresh-token
     :user/access-token   (buddy.sign.jwt/sign claims
                                               jwt-signing-secret
                                               {:alg +jwt-signing-alg+})}))

(defn assert-good-email [email db]
  (let [{:email.validation/keys [valid?]} (email/validate-email email)]
    (if (and valid? (not (db.user/lookup-by-email email db)))
      (str email)
      (exception/conflict! {::exception/reason ::email-already-in-use}))))

(defn validate-unique-email [email db]
  (let [{:email.validation/keys [valid?] :as email-validation-map} (email/validate-email email)]
    (if (and valid? (not (db.user/lookup-by-email email db)))
      email-validation-map
      {:email.validation/valid?     false
       :email.validation/reason     :email.validation/already-in-use
       :email.validation/reason-str (str "Email '" email "' already registered.")})))

(defn new-user [email password db user-id confirmation-id]
  (let [email-str                                           (str email)
        {:email.validation/keys [valid? reason reason-str]} (validate-unique-email email-str db)]
    (when-not valid?
      (exception/invalid-input! {::exception/reason     reason
                                 ::exception/reason-str reason-str}))
    {:user/emails              #{email-str}
     :user/password-hash       (hash-password password)
     :user/confirmation-ids    confirmation-id
     :user/confirmation-status :confirmation/pending
     :user/id                  user-id
     :user/scopes              default-scopes}))

(defn epoch-seconds->date [n]
  (java.util.Date. (* 1000 n)))

(defn jws-backend [jwt-signing-secret]
  (auth.backends/jws {:secret     jwt-signing-secret
                      :token-name "Bearer"
                      :options    {:alg +jwt-signing-alg+}
                      :authfn     (fn [{:keys [exp iat scopes jti sub] :as claims}]
                                    (merge claims
                                           {:exp    (epoch-seconds->date exp)
                                            :iat    (epoch-seconds->date iat)
                                            :jti    (java.util.UUID/fromString jti)
                                            :sub    (java.util.UUID/fromString sub)
                                            :scopes (set (map keyword scopes))}))}))

(defn assert-not-blacklisted! [{:keys [jti]} db]
  (when (db.user/lookup-blacklisted-token jti db)
    (exception/unauthorized! {::exception/reason ::unauthorized
                              ::exception/reason-str "Unauthorized"})))

(defn should-confirm-user [user]
  (cond
    (not user)
    [false "Link de confirmação inválido."]

    (= :confirmation/confirmed (:user/confirmation-status user))
    [false "Email já confirmado."]

    :otherwise
    [true "Email confirmado :)"]))
