(ns org.euandre.om-auth.db.schema
  (:require [clojure.spec.alpha :as s]
            [clojure.spec.gen.alpha :as gen]
            [org.euandre.misc.edn.urx :as url]))

(def refresh-tokens-schema
  [{:db/ident       :refresh-token/id
    :db/unique      :db.unique/value
    :db/valueType   :db.type/uuid
    :db/cardinality :db.cardinality/one
    :db/doc         "The JWT ID sent to the client in JWT claims payload."}
   {:db/ident       :refresh-token/token-hash
    :db/unique      :db.unique/value
    :db/valueType   :db.type/string
    :db/cardinality :db.cardinality/one
    :db/doc         "The hash of the actual refresh-token."}
   {:db/ident       :refresh-token/refresh-before
    :db/valueType   :db.type/instant
    :db/cardinality :db.cardinality/one
    :db/doc         "Until when the refresh token is valid."}])

(def blacklist-schema
  [{:db/ident       :blacklist/token
    :db/unique      :db.unique/value
    :db/valueType   :db.type/uuid
    :db/cardinality :db.cardinality/many
    :db/doc         "Set of all (within :refresh-token/refresh-berofe time range) revoked tokens."}])

(s/def :person/id uuid?)
(s/def :person/first-name string?)
(s/def :person/last-name string?)

(s/def :person/birthdate
  (s/with-gen (partial instance? java.util.Date)
    (fn [] (gen/fmap (fn [[offset add?]]
                       (let [op         (if add? + -)
                             big-offset (* 1000 offset)]
                         (-> (java.util.Date.)
                             (.toInstant)
                             (.toEpochMilli)
                             (op big-offset)
                             (java.util.Date.))))
                     (gen/tuple gen/int gen/boolean)))))

(s/def :person/picture-url
  (s/with-gen url/url?
    (fn [] (gen/fmap #(url/string->url (str "https://pires.arrobaponto.org/static/" %)) (s/gen uuid?)))))

(s/def :person/role #{:roles/esde-student
                      :roles/esde-teacher
                      :roles/child-evangelism-student
                      :roles/child-evangelism-teacher
                      :roles/youth-evangelism-student
                      :roles/youth-evangelism-teacher})

(defn either-role [r1 r2 roles]
  (or (not (:roles/esde-teacher roles))
      (and (:roles/esde-teacher roles) (not (:roles/esde-student roles)))))

(defn non-conflicting-roles? [roles]
  (and (either-role :roles/esde-teacher
                    :roles/esde-student
                    roles)
       (either-role :roles/child-evangelism-teacher
                    :roles/child-evangelism-student
                    roles)
       (either-role :roles/youth-evangelism-teacher
                    :roles/youth-evangelism-student
                    roles)))

(s/def :person/roles (s/and (s/coll-of :person/role :into #{} :distinct true)
                            non-conflicting-roles?))

(s/def :pires.schema/person
  (s/keys :opt [:person/id
                :person/first-name
                :person/last-name
                :person/birthdate
                :person/roles
                :person/picture-url]))

(def person-schema
  [{:db/ident       :person/id
    :db/unique      :db.unique/identity
    :db/valueType   :db.type/uuid
    :db/cardinality :db.cardinality/one
    :db/doc         "The UUID of the person."}
   {:db/ident       :person/first-name
    :db/valueType   :db.type/string
    :db/cardinality :db.cardinality/one
    :db/doc         "The first name of the person."}
   {:db/ident       :person/last-name
    :db/valueType   :db.type/string
    :db/cardinality :db.cardinality/one
    :db/doc         "The family name of the person."}
   {:db/ident       :person/birthdate
    :db/valueType   :db.type/instant
    :db/cardinality :db.cardinality/one
    :db/doc         "The (optional) birthdate of the person. Useful for children and teenagers."}
   {:db/ident       :person/picture-url
    :db/valueType   :db.type/uri
    :db/cardinality :db.cardinality/one
    :db/doc         "The current profile picture of the person."}
   {:db/ident       :person/roles
    :db/valueType   :db.type/ref
    :db/cardinality :db.cardinality/many
    :db/doc         "All the roles that the person exercises. Not authorization-related."}
   {:db/ident :roles/esde-student}
   {:db/ident :roles/esde-teacher}
   {:db/ident :roles/child-evangelism-student}
   {:db/ident :roles/child-evangelism-teacher}
   {:db/ident :roles/youth-evangelism-student}
   {:db/ident :roles/youth-evangelism-teacher}])

(def user-schema
  [{:db/ident       :user/id
    :db/unique      :db.unique/identity
    :db/valueType   :db.type/uuid
    :db/cardinality :db.cardinality/one
    :db/doc         "Registered user internal identifier."}
   {:db/ident       :user/password-hash
    :db/valueType   :db.type/string
    :db/cardinality :db.cardinality/one
    :db/doc         "The hash of the current password."}
   {:db/ident       :user/confirmation-ids
    :db/unique      :db.unique/value
    :db/valueType   :db.type/uuid
    :db/cardinality :db.cardinality/many
    :db/doc         "The IDs confirmation sent to the email."}
   {:db/ident       :user/confirmation-status
    :db/valueType   :db.type/ref
    :db/cardinality :db.cardinality/one
    :db/doc         "The current confirmation status of a given user."}
   {:db/ident :confirmation/pending}
   {:db/ident :confirmation/confirmed}

   {:db/ident       :user/emails
    :db/unique      :db.unique/identity
    :db/valueType   :db.type/string
    :db/cardinality :db.cardinality/many
    :db/doc         "Registered user email."}
   {:db/ident       :user/scopes
    :db/valueType   :db.type/ref
    :db/cardinality :db.cardinality/many
    :db/doc         "The scopes associated with a given token."}
   {:db/ident :scope/admin}
   {:db/ident :scope/teacher}
   {:db/ident :scope/guest}

   {:db/ident       :user/reset-password-ids
    :db/unique      :db.unique/value
    :db/valueType   :db.type/uuid
    :db/cardinality :db.cardinality/many
    :db/doc         "The IDs of the reset password requests sent by email."}

   {:db/ident       :user/refresh-tokens
    :db/valueType   :db.type/ref
    :db/cardinality :db.cardinality/many
    :db/isComponent true
    :db/doc         "Registered refresh tokens."}
   {:db/ident       :user/person
    :db/valueType   :db.type/ref
    :db/cardinality :db.cardinality/one
    :db/doc         "The Person profile of the authenticated user"}])

(def course-schema
  [{:db/ident       :course/id
    :db/unique      :db.unique/identity
    :db/valueType   :db.type/uuid
    :db/cardinality :db.cardinality/one
    :db/doc         "The UUID of the course."}
   {:db/ident       :course/title
    :db/valueType   :db.type/string
    :db/cardinality :db.cardinality/one
    :db/doc         "The title of the course."}
   {:db/ident       :couse/classes
    :db/valueType   :db.type/ref
    :db/cardinality :db.cardinality/many
    :db/doc         "The editions of the course."}

   {:db/ident       :course/category
    :db/valueType   :db.type/ref
    :db/cardinality :db.cardinality/one
    :db/doc         "The kind of the course."}
   {:db/ident :course.category/esde}
   {:db/ident :course.category/child-evangelism}
   {:db/ident :course.category/youth-evangelism}])

(def period-schema
  [{:db/ident :period/id
    :db/unique :db.unique/identity
    :db/valueType :db.type/uuid
    :db/cardinality :db.cardinality/one
    :db/doc "The UUID of the period."}
   {:db/ident :period/from
    :db/valueType :db.type/instant
    :db/cardinality :db.cardinality/one
    :db/doc "When the period starts."}
   {:db/ident :period/to
    :db/valueType :db.type/instant
    :db/cardinality :db.cardinality/one
    :db/doc "When the period ends."}
   {:db/ident :period/title
    :db/valueType :db.type/string
    :db/cardinality :db.cardinality/one
    :db/doc "The name of the period."}])

(def class-schema
  [{:db/ident       :class/id
    :db/unique      :db.unique/identity
    :db/valueType   :db.type/uuid
    :db/cardinality :db.cardinality/one
    :db/doc         "The UUID of the class."}
   {:db/ident       :class/course
    :db/valueType   :db.type/ref
    :db/cardinality :db.cardinality/one
    :db/doc         "The course which this class is an instance of."}
   {:db/ident       :class/period
    :db/valueType   :db.type/ref
    :db/cardinality :db.cardinality/one
    :db/doc         "The period when this class occurs."}
   {:db/ident       :class/teachers
    :db/valueType   :db.type/ref
    :db/cardinality :db.cardinality/many
    :db/doc         "The teachers of this class."}
   {:db/ident       :class/students
    :db/valueType   :db.type/ref
    :db/cardinality :db.cardinality/many
    :db/doc         "The students of this class."}])

(def schemas
  ((comp vec concat)
   refresh-tokens-schema
   blacklist-schema
   person-schema
   period-schema
   course-schema
   class-schema
   user-schema))
